﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using Nintendo.Foundation.IO;
using System.Diagnostics;

namespace LayoutEditor
{
    /// <summary>
    /// コマンドラインオプションです。
    /// </summary>
    public class CommandLineOptions
    {
        /// <summary>
        /// カレントフレームです。
        /// </summary>
        private int currentFrame;

        /// <summary>
        /// 入力ファイルパスを取得または設定します。
        /// </summary>
        [Nintendo.Foundation.IO.CommandLineValue(0,ValueName = "inputFilePath", Description = "COMMANDLINE_DESC_INPUT_FILE_PATH", DescriptionConverterName = "LocalizeDescription", IsRequired=false)]
        public string InputFilePath { get; set; }

        /// <summary>
        /// 中間ファイルの出力ファイルパスを取得または設定します。
        /// </summary>
        [CommandLineOption('o', "output", ValueName = "filePath", Description = "COMMANDLINE_DESC_OUTPUT_FILE_PATH", DescriptionConverterName = "LocalizeDescription")]
        public string OutputFilePath { get; set; }

        /// <summary>
        /// 読み込み済みのファイルを閉じるかどうか取得または設定します。
        /// </summary>
        [CommandLineOption("close-if-loaded", Description = "COMMANDLINE_DESC_CLOSE_IF_LOADED", DescriptionConverterName = "LocalizeDescription")]
        public bool CloseIfLoaded { get; set; }

        /// <summary>
        /// 部品ルートパスを取得または設定します。
        /// </summary>
        [CommandLineOption("parts-root-path", ValueName = "rootPath", Description = "COMMANDLINE_DESC_PARTS_ROOT_PATH", DescriptionConverterName = "LocalizeDescription")]
        public string PartsRootPath { get; set; }

        /// <summary>
        /// カレントフレームを取得または設定します。
        /// </summary>
        [CommandLineOption("current-frame", ValueName = "frame", Description = "COMMANDLINE_DESC_CURRENT_FRAME", DescriptionConverterName = "LocalizeDescription")]
        public int CurrentFrame
        {
            get
            {
                return this.currentFrame;
            }

            set
            {
                this.currentFrame = value;
                this.CurrentFrameEnabled = true;
            }
        }

        /// <summary>
        /// カレントフレームが指定されたかどうか取得します。
        /// </summary>
        public bool CurrentFrameEnabled { get; private set; }

        /// <summary>
        /// キャプチャサイズを取得または設定します。
        /// </summary>
        [CommandLineOption("capture-size", ValueName = "width,height", ConverterName = "ConvertSize", Description = "COMMANDLINE_DESC_CAPTURE_SIZE", DescriptionConverterName = "LocalizeDescription")]
        public Size CaptureSize { get; set; }

        /// <summary>
        /// キャプチャを保存するかどうか取得または設定します。
        /// </summary>
        [CommandLineOption("save-screen-capture", Description = "COMMANDLINE_DESC_SAVE_SCREEN_CAPTURE", DescriptionConverterName = "LocalizeDescription")]
        public bool SaveScreenCapture { get; set; }

        /// <summary>
        /// キャプチャファイルパスを取得または設定します。
        /// </summary>
        [CommandLineOption("capture-file-path", ValueName = "filePath", Description = "COMMANDLINE_DESC_CAPTURE_FILE_PATH", DescriptionConverterName = "LocalizeDescription")]
        public string CaptureFilePath { get; set; }

        /// <summary>
        /// ウインドウを隠すかどうか取得または設定します。
        /// </summary>
        [Obsolete]
        [CommandLineOption("hide-window", Description = "COMMANDLINE_DESC_HIDE_WINDOW", DescriptionConverterName = "LocalizeDescription", IsHidden = true)]
        public bool HideWindow { get; set; }

        /// <summary>
        /// ループキーを焼き付けるかどうか取得または設定します。
        /// </summary>
        [CommandLineOption("bake-infinity", Description = "COMMANDLINE_DESC_BAKE_INFINITY", DescriptionConverterName = "LocalizeDescription")]
        public bool BakeInfinity { get; set; }

        /// <summary>
        /// 部品レイアウトを一括同期するかどうか取得または設定します。
        /// </summary>
        [CommandLineOption("update-parts-layout", Description = "COMMANDLINE_DESC_UPDATE_PARTS_LAYOUT", DescriptionConverterName = "LocalizeDescription")]
        public bool UpdatePartsLayout { get; set; }

        /// <summary>
        /// ビューアへのプレビュー転送を行います。
        /// </summary>
        [CommandLineOption("send-viewer", Description = "COMMANDLINE_DESC_SEND_VIEWER", DescriptionConverterName = "LocalizeDescription")]
        public bool SendViewer { get; set; }

        /// <summary>
        /// ビューアへのコマンド通知を行います。(内部利用のため、隠しオプション)
        /// </summary>
        [CommandLineOption("viewer-cmd", Description = "COMMANDLINE_DESC_VIEWER_CMD", DescriptionConverterName = "LocalizeDescription", IsHidden = true)]
        public string ViewerCommand { get; set; }

        /// <summary>
        /// ログを詳細表示するかどうか取得または設定します。
        /// </summary>
        [CommandLineOption('v', "verbose", Description = "COMMANDLINE_DESC_VERBOSE", DescriptionConverterName = "LocalizeDescription")]
        public bool Verbose { get; set; }

        /// <summary>
        /// 多重起動を認めます。
        /// </summary>
        [CommandLineOption("multiprocess", IsHidden = true)]
        public bool Multiprocess { get; set; }

        /// <summary>
        /// バイナリコンバータをプロセスとして起動します。
        /// </summary>
        [CommandLineOption("debug-converter", IsHidden = true)]
        public bool DebugConverter { get; set; }

        public bool IsViewerCommandEnabled { get { return !string.IsNullOrEmpty(this.ViewerCommand); } }

        /// <summary>
        /// コマンドライン引数値をSize型の値に変換します。
        /// </summary>
        /// <param name="value">コマンドライン引数値</param>
        /// <param name="name">コマンドライン引数名</param>
        /// <returns>Size型の値を返します。</returns>
        public static object ConvertSize(string value, string name)
        {
            // "[幅],[高さ]" をSizeに変換
            string[] wh = value.Split(',');
            if (wh.Length != 2)
            {
                return Size.Empty;
            }

            int w, h;

            if (!int.TryParse(wh[0], out w))
            {
                return Size.Empty;
            }

            if (!int.TryParse(wh[1], out h))
            {
                return Size.Empty;
            }

            return new Size(w, h);
        }

        /// <summary>
        /// ローカライズしたオプションの説明を取得します。
        /// </summary>
        /// <param name="description">Descriptionプロパティの値</param>
        /// <param name="valueName">引数名またはオプション名</param>
        /// <returns>ローカライズされたコマンドラインオプションの説明を返します。</returns>
        public static string LocalizeDescription(string description, string valueName)
        {
            string result;

            result = StringResMgr.Get(description);
            System.Diagnostics.Debug.Assert(result != null, "コマンドオプションの Description が null");

            return result;
        }

        /// <summary>
        /// 複数プロセス間でやり取りするオプション、
        /// バージョンが異なることもあるので既存のものを変えるときは注意
        /// </summary>
        [Flags]
        private enum Flags : ulong
        {
            CloseIfLoaded = 1ul << 0,
            SendViewer = 1ul << 1,
            SendViewerCommand = 1ul << 2,
        }

        static bool HasAnythingToEncodeForWindowMessage_(CommandLineOptions option)
        {
            return
                option.CloseIfLoaded ||
                option.SendViewer ||
                !string.IsNullOrEmpty(option.InputFilePath) ||
                option.IsViewerCommandEnabled;
        }

        const string MessageVersionPrefix = "_layout_editor_msgver";
        const int MessageVersion0 = 0;
        const int MessageVersion1 = 1;



        /// <summary>
        /// ウインドウメッセージで送るため文字列にエンコードします。
        /// </summary>
        /// <param name="option">エンコードするオプション</param>
        /// <returns>エンコードした文字列データ</returns>
        public static string EncodeForWindowMessage(CommandLineOptions option)
        {
            if(!HasAnythingToEncodeForWindowMessage_(option))
            {
                return string.Empty;
            }

            // "[CloseIfLoadedフラグ]?[入力ファイルパス]?[ビューアーコマンド]_layout_editor_msgverN" にエンコード
            // パス文字として不正な ? をデリミタとして使います。
            Flags flags =
                (option.CloseIfLoaded ? Flags.CloseIfLoaded : 0) |
                (option.SendViewer ? Flags.SendViewer : 0);

            string message = string.Format("{0}?{1}?{2}_layout_editor_msgver{3}",
                (ulong)flags,
                option.InputFilePath ?? string.Empty,
                option.ViewerCommand ?? string.Empty,
                MessageVersion1);

            return message;
        }

        /// <summary>
        /// メッセージバージョンを抽出する
        /// </summary>
        static int ExtractMessageVersion_(string message)
        {
            // バージョン管理する以前の、古いバージョン
            if (!message.Contains(MessageVersionPrefix))
            {
                return MessageVersion0;
            }
            else
            {
                string versionNumStr = message.Substring(message.IndexOf(MessageVersionPrefix) + MessageVersionPrefix.Length);
                return int.Parse(versionNumStr);
            }
        }

        /// <summary>
        /// メッセージバージョンを取り除く
        /// </summary>
        static string TrimMessage_(string message)
        {
            int index = message.IndexOf(MessageVersionPrefix);
            return index != -1 ? message.Substring(0, index) : message;
        }

        /// <summary>
        /// Ver0
        /// </summary>
        static CommandLineOptions DecodeForWindowMessageVer0_(string message)
        {
            // "[CloseIfLoadedフラグ]:[入力ファイルパス]" をデコード
            int index = message.IndexOf(':');
            var flags = (Flags)ulong.Parse(message.Substring(0, index));
            string path = message.Substring(index + 1);

            CommandLineOptions option = new CommandLineOptions();
            option.CloseIfLoaded = flags.HasFlag(Flags.CloseIfLoaded);
            option.SendViewer = flags.HasFlag(Flags.SendViewer);
            option.InputFilePath = path;
            option.ViewerCommand = string.Empty;

            return option;
        }

        /// <summary>
        /// Ver1
        /// </summary>
        static CommandLineOptions DecodeForWindowMessageVer1_(string message)
        {
            // "[CloseIfLoadedフラグ]?[入力ファイルパス]?[ビューアーコマンド]" をデコード
            var messages = message.Split('?');
            Debug.Assert(messages.Length == 3);

            var flags = (Flags)ulong.Parse(messages[0]);
            string path = messages[1];
            string command = messages[2];

            CommandLineOptions option = new CommandLineOptions();
            option.CloseIfLoaded = flags.HasFlag(Flags.CloseIfLoaded);
            option.SendViewer = flags.HasFlag(Flags.SendViewer);
            option.InputFilePath = path;
            option.ViewerCommand = command;

            return option;
        }

        /// <summary>
        /// ウインドウメッセージで送られた文字列データをデコードします。
        /// </summary>
        /// <param name="message">デコードする文字列データ</param>
        /// <returns>デコードしたオプション</returns>
        public static CommandLineOptions DecodeForWindowMessage(string message)
        {
            int version = ExtractMessageVersion_(message);

            message = TrimMessage_(message);
            switch (version)
            {
                case MessageVersion0: return DecodeForWindowMessageVer0_(message);
                case MessageVersion1: return DecodeForWindowMessageVer1_(message);
                default:
                    Trace.WriteLine(string.Format("Unknown WindowMessage ver={0}", version));
                    return new CommandLineOptions();
            }
        }
    }
}
