﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Windows.Forms;
using System.Linq;
using LECore.Structures;
using LECore.Structures.Core;

namespace LECore
{
    /// <summary>
    /// ビューアパラメータです。
    /// コマンド文字列の取得も本クラスから行います。
    /// </summary>
    public class ViewerPreviewParam
    {
        public const int WaitTimeForExit = 3600 * 1000;  // プロセスが終了するまで待つ時間(ミリセカンド)

        //----------------------------------------------------------
        public enum ViewerKind
        {
            PC,
            Target,
        }

        public enum TransferDestination
        {
            NoSpecify,
            Upper,
            Lower,
        }

        public enum PreviewMode
        {
            NoSpecify,
            Animation,
            Control,
            PCViewerScreenCapture
        }

        //----------------------------------------------------------
        public string InputDir { get; set; }
        public string TempDir { get; set; }
        public string LayoutFileNameWithoutExt { get; set; }
        public string PartsLayoutFileNameWithoutExt { get; set; }
        public int ConvertTimeOutMiliSec { get; set; }
        public Color BackgroundColor { get; set; }
        public string BackgroundImagePath { get; set; }

        public int PreviewDisplayWidth { get; set; }
        public int PreviewDisplayHeight { get; set; }

        public bool DrawHiddenPanesAsNull { get; set; }
        public bool PreviewConverterOptionOmitNoKeyAllEnabled { get; set; }
        public bool AllowNoGroupAnimationTag { get; set; }

        public bool ConvertTextboxControlCode { get; set; }
        public bool FixInfinityCurve { get; set; }

        public ViewerKind Kind { get; set; }
        public TransferDestination DestinationScreen { get; set; }
        public PreviewMode Mode { get; set; }
        public bool IsAutoReload { get; set; }
        public string CustomPCViewerWindowTitleHint { get; set; }

        // デバッグ用
        public bool SearchPCViewerByTitle { get; set; }

        public string ViewMode { get; set; }

        public bool IsPerspectiveProjection { get; set; }
        public float PerspectiveFOV { get; set; }
        public float PerspectiveNear { get; set; }
        public float PerspectiveFar { get; set; }

        public bool Is30Fps { get; set; }

        public bool SRGBWriteEnabled { get; set; }
        public bool DegammaTexture { get; set; }
        public bool DegammaParameter { get; set; }

        public string CaptureFilePath { get; set; }
        public int CaptureFrameCount { get; set; }
        public Size CaptureSize { get; set; }

        public IWin32Window ErrorDialogOwner { get; set; }

        // TODO: マルチスレッドとの相性がよくないので必要なら代替手段を提供する
        public Form ProgressDialog { get; set; }

        public Size LayoutSize { get; set; }

        public bool AutoFilteredFontGenerationEnabled { get; set; }
        public bool AutoFilteredFontGenerationFromSameNameFfntEnabled { get; set; }
        internal List<AutoFilteredFontUtil.FontCharUsageData> FontUsages { get; set; }
        internal string[] ConvertedImageFileNames { get; set; }

        public string PreArchiverCommandPath { get; set; }

        public string PreviewTileOptimize { get; set; }
        public int PreviewTileSizeThreshold { get; set; }

        public string IsPreviewGpuEncodingEnabled { get; set; }

        public bool PushedToViewerThread { get; set; }

        // 接続の再試行をする期間 ms
        public int ConnectionRetryDuration { get; set; } = 1000;

        // 転送処理の完了イベント (必ずしも送信の成功やビューアでの受信の完了を意味しない)
        public event EventHandler Finished;

        // バイナリコンバータをプロセスとして起動する
        public bool ExecuteConverterAsProcess { get; set; } = false;

        #region InitializeParemeterDependingOnPlatform で初期化されるパラメータ
        public string BinaryConverterOptionsString { get; private set; }
        public bool MultiScreenEnabled { get; private set; }
        public string PcViewerPath { get; private set; }
        public PlatformPreference PlatformPreference { get; private set; }
        #endregion

        /// <summary>
        /// プレビュー中に行った警告をユーザに通知するためのアクションです。
        /// </summary>
        public Action<string> OnPreviewWarningAction { get; set; }

        public List<string> CombinerBlockDefinitionDirectories { get; set; }
        public string CombinerEditorPath { get; set; }

        public string CombinerReplaceFile { get; set; }

        public string CombinerReceivedTempFile { get; set; }

        public string ViewerCommand { get; set; }

        //----------------------------------------------------------
        // UI スレッドで他のプロパティが設定された後で呼ぶこと
        // TODO: ちょっと無理があるので修正したい
        public void InitializeParametersDependingOnPlatform()
        {
            BinaryConverterOptionsString = GetBinaryConverterOptionsString();
            MultiScreenEnabled = LayoutEditorCore.PlatformDetail.MultiScreenEnabled;

            // プラグインが読み込んだプラットフォーム設定を適用します。
            PcViewerPath = Environment.ExpandEnvironmentVariables(AppConstants.PCLayoutViewerPath);

            var prefPf = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
            if (prefPf != null && prefPf.PlatformPreference != null)
            {
                PlatformPreference = prefPf.PlatformPreference;
                PcViewerPath = Path.Combine(AppConstants.NwToolsRootPath, prefPf.PlatformPreference.PcViewerPath);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void InitTempPath(string tempDirStr)
        {
            TempDir = tempDirStr;
            InputDir = Path.Combine(TempDir, "src");
        }

        /// <summary>
        ///
        /// </summary>
        public void InitPreviewMode(ISubScene targetSubScene, PreviewMode mode)
        {
            if (mode == ViewerPreviewParam.PreviewMode.NoSpecify)
            {
                this.Mode =
                    targetSubScene.ContainsPartsPane() || targetSubScene.IsPartsScene() ?
                    ViewerPreviewParam.PreviewMode.Control : ViewerPreviewParam.PreviewMode.Animation;
            }
            else
            {
                this.Mode = mode;
            }

            // キャプチャモードなら...
            if (mode == ViewerPreviewParam.PreviewMode.PCViewerScreenCapture)
            {
                // PCビューア
                this.Kind = ViewerKind.PC;

                int frameCount = GlobalTime.Inst.Time;

                // ファイル名が指定されていなければデフォルト設定
                if (!this.CaptureFilePath.EndsWith(".bmp") &&
                    !this.CaptureFilePath.EndsWith(".BMP"))
                {
                    string defaultFileName = string.Format("{0}_{1}_{2}.bmp", DateTime.Now.ToString("yyMMdd"), frameCount, LayoutFileNameWithoutExt);
                    this.CaptureFilePath = Path.Combine(this.CaptureFilePath, defaultFileName);
                }
            }
        }

        /// <summary>
        /// フォルダ掃除
        /// </summary>
        public void CleanupTemporaryPath(bool create)
        {
            // inputDataDir に 以前のデータが残っている場合は、消去します。
            if (Directory.Exists(TempDir))
            {
                DirectoryInfo di = new DirectoryInfo(TempDir);
                di.Delete(true);
            }

            if (create)
            {
                // inputDataDir に layoutFileNameWithoutExt
                // でプレビュー用のデータを一時的に保存します。
                Directory.CreateDirectory(TempDir);
                Directory.CreateDirectory(InputDir);
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// プレビューモードを指定するオプション文字を取得します。
        /// </summary>
        public string GetModeOptionString()
        {
            return (this.Mode == ViewerPreviewParam.PreviewMode.Control) ? "ctrl" : "anim";
        }

        /// <summary>
        /// コンバータオプション文字列を取得します。
        /// </summary>
        private string GetBinaryConverterOptionsString()
        {
            string argStr = string.Empty;

            // bake-infinity は必ず指定します。
            argStr = " --bake-infinity";

            // -i ”隠す”状態のファイルをNullペインに変換
            if (this.DrawHiddenPanesAsNull)
            {
                argStr += " -i";
            }

            // 別に指定するので不要
            //argStr += " --logfile"; // ログファイルを出力(nw4r_lytcvtr.log)

            if (!this.ConvertTextboxControlCode)
            {
                // 制御文字の変換
                argStr += " --no-convert-cvtrchar";
            }

            if (this.PreviewConverterOptionOmitNoKeyAllEnabled)
            {
                // 区間タグで指定されたフレーム区間内にキーが無い場合は、アニメーションを出力しない。
                argStr += " --omit-nokey-all";
            }

            // 対象グループが指定されていない アニメーション区間タグを許容する。
            if(this.AllowNoGroupAnimationTag)
            {
                argStr += " --allow-nogroup-animtag";
            }

            // キャプチャモードの時は、タグ情報を無視して一続きのアニメーションと解釈します。
            if (this.Mode != PreviewMode.PCViewerScreenCapture)
            {
                argStr += " --tag-split-and-whole";
            }

            // テクスチャのガンマ逆補正
            if (this.DegammaTexture)
            {
                argStr += " --degamma-texture";
            }

            // パラメータ（頂点カラー、補間カラーなど）のガンマ逆補正
            if (this.DegammaParameter)
            {
                argStr += " --degamma-parameter";
            }


            if (AppConstants.IsGfxMode)
            {
                // プラットフォームを指定する
                var prefPf = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
                PlatformPreference platformPreference = prefPf != null ? prefPf.PlatformPreference: null;
                switch (this.Kind)
                {
                    case ViewerKind.PC:
                        argStr += " " + (platformPreference != null ? platformPreference.PcConverterAdditionalArguments: string.Empty);
                        break;
                    case ViewerKind.Target:
                        argStr += " " + (platformPreference != null ? platformPreference.TargetConverterAdditionalArguments: string.Empty);
                        break;
                }

                // ffnt のコピーはしない
                argStr += " --no-copy-ffnt";

                // キャッシュディレクトリの設定
                var dir = LECore.Util.TemporaryFileUtil.FtxbCacheDir;
                argStr += " --ftxb-cache-directory=\"" + dir + "\"";

                // gpuエンコーディングの有効・無効
                if(!string.IsNullOrEmpty(this.IsPreviewGpuEncodingEnabled))
                {
                    argStr += " --gpu-encoding=\"" + this.IsPreviewGpuEncodingEnabled + "\"";
                }
            }

            // コンバイナユーザーシェーダ用のディレクトリを設定
            if (CombinerBlockDefinitionDirectories != null && CombinerBlockDefinitionDirectories.Any())
            {
                argStr += " --shader-env-directories ";
                int index = 0;
                foreach(var str in CombinerBlockDefinitionDirectories)
                {
                    if (index > 0)
                    {
                        argStr += ";";
                    }
                    argStr += "\"" + str + "\"";
                    index++;
                }
            }

            // コンバイナエディタのファイルパスを設定
            if (!string.IsNullOrEmpty(CombinerEditorPath))
            {
                argStr += " --combiner-editor-path ";
                argStr += "\"" + CombinerEditorPath + "\"";
            }

            return argStr;
        }

        /// <summary>
        /// 一時中間ファイル出力ディレクトリ名を取得します。
        /// </summary>
        public string GetTemporaryLayoutDirName()
        {
            return Path.Combine(this.InputDir, this.LayoutFileNameWithoutExt);
        }

        /// <summary>
        /// 一時中間ファイル出力ディレクトリ名を取得します。
        /// </summary>
        public string GetTemporaryPartsLayoutDirName()
        {
            Debug.Assert(!string.IsNullOrEmpty(this.PartsLayoutFileNameWithoutExt));

            return Path.Combine(this.InputDir, this.PartsLayoutFileNameWithoutExt);
        }

        /// <summary>
        /// バイナリ出力ディレクトリ名を取得します。
        /// </summary>
        public string GetBinaryOutputDirName()
        {
            return GetBinaryOutputDirName(this.TempDir);
        }

        /// <summary>
        /// バイナリ出力ディレクトリ名を取得します。
        /// </summary>
        public static string GetBinaryOutputDirName(string tempDir)
        {
            return Path.Combine(tempDir, "arc");
        }

        /// <summary>
        /// アーカイブファイルのパスを取得します。
        /// </summary>
        public string GetArcFilePath()
        {
            return Path.Combine(TempDir, "data\\content");
        }

        /// <summary>
        /// バイナリ出力ディレクトリ名を取得します。
        /// </summary>
        public string GetPreviewTargetBinaryDirName()
        {
            if (!AppConstants.IsGfxMode)
            {
                return Kind == ViewerKind.PC ? GetBinaryOutputDirName(this.TempDir) : "layout.arc";
            }

            return Kind == ViewerKind.PC ? GetBinaryOutputDirName(this.TempDir) : Path.Combine(GetArcFilePath(), "layout.arc").Replace('\\', '/');
        }

        /// <summary>
        /// リニア色空間（sRGB書き込みをデフォルトにするかどうか）モードのオプション文字列を取得します。
        /// </summary>
        public string GetGammaOptionString()
        {
            return this.SRGBWriteEnabled ? "-g " : "";
        }

        /// <summary>
        /// 表示対象画面を指定するオプション文字列。
        /// </summary>
        public string GetDestinationOptionString()
        {
            if (!MultiScreenEnabled)
            {
                return string.Empty;
            }

            if (this.Kind == ViewerKind.PC)
            {
                return string.Empty;
            }

            if (this.DestinationScreen == TransferDestination.NoSpecify)
            {
                // レイアウトサイズで自動判別する
                return (this.LayoutSize.Width <= 320) ? "-btm " : string.Empty;
            }
            else
            {
                return (this.DestinationScreen == TransferDestination.Lower) ? "-btm " : string.Empty;
            }
        }

        /// <summary>
        /// プロジェクション関連のオプション文字列を取得します。
        /// </summary>
        public string GetProjectionParamatersOptionString()
        {
            if (this.IsPerspectiveProjection)
            {
                return string.Format("-projection pers -fov {0} -near {1} -far {2} ", this.PerspectiveFOV, this.PerspectiveNear, this.PerspectiveFar);
            }
            else
            {
                return "-projection ortho ";
            }
        }

        /// <summary>
        /// プレビュー時の動作モード関連のオプション文字列を取得します。
        /// </summary>
        public string GetViewSize()
        {
            return string.Format("-viewmode {0} ", this.ViewMode);
        }

        /// <summary>
        /// プレビュー時の背景色のオプション文字列を取得します。
        /// </summary>
        public string GetBackgroundColor()
        {
            return string.Format("-bg_color 0x{0}{1}{2}{3} ",
                                BackgroundColor.R.ToString("X2"),
                                BackgroundColor.G.ToString("X2"),
                                BackgroundColor.B.ToString("X2"),
                                BackgroundColor.A.ToString("X2"));
        }

        /// <summary>
        /// プレビュー時の背景画像のオプション文字列を取得します。
        /// </summary>
        public string GetBackgroundImage()
        {
            if (BackgroundImagePath != null)
            {
                return string.Format("-bg_image __Combined.bntx ");
            }
            else
            {
                return string.Format("");
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string GetPreviewNotifyCommand()
        {
            if (AppConstants.IsGfxMode)
            {
                return string.Format("{3}{4}{5}{6}{7}{8}{9}{10}-m {0} -n {1} {2}",
                    this.GetModeOptionString(),
                    this.LayoutFileNameWithoutExt,
                    this.GetPreviewTargetBinaryDirName(),
                    this.GetGammaOptionString(),
                    this.GetDestinationOptionString(),
                    this.GetProjectionParamatersOptionString(),
                    this.Is30Fps ? "-is30fps " : "-is60fps ",
                    this.GetViewSize(),
                    this.IsAutoReload ? "-continue-animation " : "",
                    this.GetBackgroundColor(),
                    this.GetBackgroundImage());
            }
            else
            {
                return string.Format("{3}{4}-m {0} -n {1} {2}",
                    this.GetModeOptionString(),
                    this.LayoutFileNameWithoutExt,
                    this.GetPreviewTargetBinaryDirName(),
                    this.GetGammaOptionString(),
                    this.GetDestinationOptionString());
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string GetPreviewNotifyCommandForCapture()
        {
            // キャプチャ用
            // 複数のレイアウトの中から、表示するレイアウト名を指定します。
            // キャプチャサイズ、PCビューアが異常終了することがあるので最小サイズ制限を行います。
            Size captureSize = this.CaptureSize == Size.Empty ? this.LayoutSize : this.CaptureSize;
            int captureW = Math.Max(captureSize.Width, 400);
            int captureH = Math.Max(captureSize.Height, 240);

            // ビューアには、アニメーション再生開始フレームを基点(0)としたフレーム番号を指定します。
            // 負の値は非対応なので、0 でクランプします。
            int viewerFrameCount = Math.Max(this.CaptureFrameCount - GlobalTime.Inst.AnimPlayStartTime, 0);
            return string.Format(
                "-s {0} {5}{4}{6}{7}-f {1} -n {2} {3} ",
                this.CaptureFilePath,
                viewerFrameCount,
                this.LayoutFileNameWithoutExt,
                this.GetPreviewTargetBinaryDirName(),
                string.Format("-w {0} -h {1} ", captureW, captureH),
                GetGammaOptionString(),
                GetBackgroundColor(),
                GetBackgroundImage());
        }

        /// <summary>
        /// レイアウトコンバータのパスを取得します。
        /// </summary>
        public string GetLayoutConverterPath()
        {
            return this.Kind == ViewerKind.PC ? AppConstants.LayoutConverterForPCViewerPath : AppConstants.LayoutConverterPath;
        }

        /// <summary>
        ///
        /// </summary>
        public string GetLaunchPCViewerCommand()
        {
            if (LECore.LayoutEditorCore.PlatformDetail.PCViewerScreenSizeLimitEnabled)
            {
                // PV起動
                return string.Format("-w {0} -h {1}",
                    Math.Min(1280, this.LayoutSize.Width * 2),
                    Math.Min(720, this.LayoutSize.Height * 2));
            }
            else
            {
                return string.Format("-dw {0} -dh {1} ", this.PreviewDisplayWidth, this.PreviewDisplayHeight);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public string GetLaunchTargetViewerCommand()
        {
            return string.Format("-dw {0} -dh {1} {2}", this.PreviewDisplayWidth, this.PreviewDisplayHeight, GetGammaOptionString());
        }

        /// <summary>
        ///
        /// </summary>
        public string GetTargetViewerStarterPath()
        {
            return AppConstants.ViewerStarterPath;
        }

        /// <summary>
        /// 転送処理の完了 (必ずしも送信の成功やビューアでの受信の完了を意味しない)
        /// </summary>
        public void OnFinished()
        {
            Finished?.Invoke(this, new EventArgs());
        }
    }
}
