﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Diagnostics;
using System.Text.RegularExpressions;

namespace LECore.Structures
{
    using Core;
    using Core.Command;
    using Util;
    using System.Collections.Generic;
    using LECore.Structures.SerializableObject.Lyt;

    #region BackGround
    /// <summary>
    /// SubSceneの背景を表現するクラスです。
    ///
    /// </summary>
    public class BackGround
    {
        // 以下はアプリケーションの設定となりました。
        Color _gridColor = Color.FromArgb(Color.Gray.R, Color.Gray.R, Color.Gray.R, 64);
        float _gridSize = 40.0f;
        uint _gridDivisionNum = 4;       // 非ゼロの値
        bool _gridVisible = true;
        bool _gridSnapEnalbed = false;

        Color _backGroundColor = Color.DarkGray;
        FVec2 _screenSize = new FVec2(AppConstants.DefaultScreenSize);

        ITextureImage _backgroundImage = null;

        void NotifyChangeToScene()
        {
            if (OnBackGroundModify != null)
            {
                SceneModifyEventArgs args = new SceneModifyEventArgs(SceneModifyEventArgs.Kind.SubScenePropertyModify, this);
                OnBackGroundModify(this, args);
            }
        }

        #region -------------- プロパティ --------------

        /// <summary>
        /// 背景の色
        /// </summary>
        public Color BackGroundColor
        {
            get { return _backGroundColor; }
            set {
                if (_backGroundColor != value)
                {
                    _backGroundColor = value;
                    NotifyChangeToScene();
                }
            }
        }

        /// <summary>
        /// グリッド線の色
        /// </summary>
        public Color GridColor
        {
            get { return _gridColor; }
            set {
                if (_gridColor != value)
                {
                    _gridColor = value;
                    NotifyChangeToScene();
                }
            }
        }

        /// <summary>
        /// グリッドサイズ
        /// </summary>
        public float GridSize
        {
            get { return _gridSize; }
            set
            {
                if (_gridSize != value)
                {
                    _gridSize = value;
                    NotifyChangeToScene();
                }
            }
        }

        /// <summary>
        /// 指定した整数値がグリッド分割数として適切か判定します。
        /// </summary>
        static public bool IsValidForGridDivisionNum(uint intVal)
        {
            return (intVal > 0 && intVal < 16);
        }

        /// <summary>
        /// グリッド分割数
        /// </summary>
        public uint GridDivisionNum
        {
            get { return _gridDivisionNum; }
            set
            {
                if (IsValidForGridDivisionNum(value))
                {
                    if (_gridDivisionNum != value)
                    {
                        _gridDivisionNum = value;
                        NotifyChangeToScene();
                    }
                }
            }
        }

        /// <summary>
        /// グリッドが見えるかどうか
        /// </summary>
        public bool GridVisible
        {
            get { return _gridVisible; }
            set
            {
                if (_gridVisible != value)
                {
                    _gridVisible = value;
                    NotifyChangeToScene();
                }
            }
        }

        /// <summary>
        /// グリッド吸着が有効か？
        /// </summary>
        public bool GridSnapEnalbed
        {
            get { return _gridSnapEnalbed; }
            set {
                if (_gridSnapEnalbed != value)
                {
                    _gridSnapEnalbed = value;
                    NotifyChangeToScene();
                }
            }
        }

        /// <summary>
        /// スクリーンサイズ
        /// </summary>
        public FVec2 ScreenSize
        {
            get { return new FVec2(_screenSize); }
            set
            {
                if (_screenSize != value)
                {
                    _screenSize.Set(value);
                    NotifyChangeToScene();
                }
            }
        }

        /// <summary>
        /// 背景画像
        /// 使用しない場合は、nullを設定します。
        /// </summary>
        public ITextureImage BackgroundImage
        {
            get { return _backgroundImage; }
            set {
                if (_backgroundImage != value)
                {
                    _backgroundImage = value;
                    NotifyChangeToScene();
                }
            }
        }

        #endregion -------------- プロパティ --------------

        #region  -------------- 公開イベント --------------
        /// <summary>
        /// ペイン更新時に呼び出されるイベントです。
        /// </summary>
        public delegate void OnBackGroundModifyHandler(BackGround sender, SceneModifyEventArgs args);
        public event OnBackGroundModifyHandler OnBackGroundModify;

        #endregion // -------------- 公開イベント --------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public BackGround()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public void Set(BackGround backGround)
        {
            _gridColor = backGround.GridColor;
            _gridSize = backGround.GridSize;
            _gridDivisionNum = backGround.GridDivisionNum;
            _gridVisible = backGround.GridVisible;
            _gridSnapEnalbed = backGround.GridSnapEnalbed;

            _backGroundColor = backGround.BackGroundColor;
            _screenSize = backGround.ScreenSize;

            _backgroundImage = backGround.BackgroundImage;
        }

        public static BackGround Default = new BackGround();

        /// <summary>
        /// 情報をコピーします。
        /// </summary>
        public void CopyFrom(BackGround srcBackGround)
        {
            this.GridColor = srcBackGround.GridColor;
            this.GridSize = srcBackGround.GridSize;
            this.GridDivisionNum = srcBackGround.GridDivisionNum;
            this.GridVisible = srcBackGround.GridVisible;
            this.GridSnapEnalbed = srcBackGround.GridSnapEnalbed;

            this.BackGroundColor = srcBackGround.BackGroundColor;
            this.ScreenSize = srcBackGround.ScreenSize;

            this.BackgroundImage = srcBackGround.BackgroundImage;
        }
    }
    #endregion BackGround

    #region SceneModifyEventAggregater
    /// <summary>
    /// 複数のシーン変更イベントを1つのイベントに集約するクラス。
    ///
    /// 変更イベントを集約することによって、再描画処理の回数を減らし、パフォーマンスアップすることを
    /// ねらって作成されたクラスです。
    ///
    /// TODO：集約機能の実装は簡単なものです。
    /// 同一種類のイベント登録は破棄されます（最初のイベントのみが保存されます）。
    /// </summary>
    internal class SceneModifyEventAggregater
    {
        readonly Stack _cmdStack = new Stack();


        /// <summary>
        /// 現在のコマンドバッファを取得します。
        /// </summary>
        ArrayList _CmdBuffer
        {
            get
            {
                Debug.Assert(Active);
                return _cmdStack.Peek() as ArrayList;
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SceneModifyEventAggregater() { }

        /// <summary>
        /// アクティブ状態か
        /// </summary>
        public bool Active
        {
            get { return _cmdStack.Count != 0; }
        }

        /// <summary>
        /// コマンド登録を開始します。
        /// 入れ子呼び出しに対応しています(変更になりました)。
        /// </summary>
        /// <param name="cmd"></param>
        public void BeginRegister()
        {
            _cmdStack.Push(new ArrayList());

        }
        /// <summary>
        /// コマンドを登録します。
        /// </summary>
        /// <param name="cmd"></param>
        public void AddCommand(SceneModifyEventArgs newCmd)
        {
            Debug.Assert(Active);
            Debug.Assert(newCmd != null);

            for (int i = 0; i < _CmdBuffer.Count; i++)
            {
                SceneModifyEventArgs cmd = _CmdBuffer[i] as SceneModifyEventArgs;
                Debug.Assert(cmd != null);

                // 同じ種類のコマンドがすでに登録されている場合、
                // 新規のコマンドに、過去のコマンドがマージされ
                // 新規コマンドが追加されます。
                if (newCmd.IsSameType(cmd))
                {
                    newCmd.MergeCmd(cmd);
                    _CmdBuffer.Remove(cmd);
                }
            }
            _CmdBuffer.Add(newCmd);
        }

        /// <summary>
        /// コマンド登録を完了し、集約されたコマンドを取得します。
        /// </summary>
        /// <returns></returns>
        public SceneModifyEventArgs[] EndRegister()
        {
            Debug.Assert(Active);

            SceneModifyEventArgs[] result =
                _CmdBuffer.ToArray(typeof(SceneModifyEventArgs)) as SceneModifyEventArgs[];

            _CmdBuffer.Clear();
            _cmdStack.Pop();

            return result;
        }
    }
    #endregion SceneModifyEventAggregater

    /// <summary>
    /// キャプチャテクスチャの更新モードです。
    /// </summary>
    public enum CaptureTextureUpdateMode
    {
        All,
        Sequential,
    };

    /// <summary>
    /// SubScene システムコア外部公開用インタフェース
    /// </summary>
    public interface ISubScene :
        IDrawable         // レンダラによって描画可能です。
    {
        /// <summary>
        /// ドキュメントのヘッダーを取得します。
        /// </summary>
        IDocumentHeader DocumentHeader { get; set; }

        IPane CurrentPane { get; set; }

        /// <summary>
        /// ペインセットを取得します。
        /// </summary>
        IEnumerable<IPane> IPaneArray { get; }

        /// <summary>
        /// ルートペインを取得します。
        /// </summary>
        IPane RootIPane { get; }

        /// <summary>
        /// ユニークなペイン名を取得します。
        /// </summary>
        string GetUniqueNewPaneName(IPane pane, string currentName, string newName);

        /// <summary>
        /// 大量の変更を開始することをシーンに通知します。
        /// </summary>
        void BeginMassiveModify();

        /// <summary>
        /// 大量の変更が終了したことをシーンに通知します。
        /// </summary>
        void EndMassiveModify();

        /// <summary>
        /// 大量の変更が終了したことをシーンに通知します。
        /// コマンドを破棄します。
        /// </summary>
        void EndMassiveModify(bool forceDropCommand, bool forceDropEvent);

        /// <summary>
        /// 描画準備
        /// </summary>
        void PrepareDraw(IRenderer renderer);

        #region キャプチャテクスチャ関連

        /// <summary>
        /// キャプチャテクスチャの更新処理
        /// </summary>
        /// <param name="renderer">キャプチャテクスチャのレンダリングに使用するレンダラー</param>
        void RenderCaptureTexture(IRenderer renderer);

        /// <summary>
        /// キャプチャテクスチャの更新リクエスト
        /// </summary>
        /// <param name="mode">キャプチャテクスチャの更新モードです</param>
        void RequestRenderCaptureTexture(CaptureTextureUpdateMode mode);

        /// <summary>
        /// キャプチャテクスチャの状態を更新します。
        /// </summary>
        void UpdateCaptureTextureState();

        #endregion キャプチャテクスチャ関連

        /// <summary>
        /// 背景設定情報関連
        /// </summary>
        BackGround BackGround { get; }

        //
        // SelectedSetModify は外部に公開せずに、
        // 代わりに SelectedSetModify に対して行っていた処理をシーンが提供することにします。
        // （レイヤ境界付近のクラス数を減らしたいので）
        // 選択セットの変更、また、選択セットに対する操作に関するメソッドは、
        // SubSceneが提供することとします。
        // "選択セットに対する操作"の数が増加してきた場合には、SubSceneSelectedSetManipulator
        // として別クラスに分離しようと考えていますが、現状ではその必要がないと判断しています。
        //
        // ==> だんだん複雑になってきたので、方針を変更し、公開したいと思います。
        /// <summary>
        /// 選択セットを取得します。
        /// </summary>
        ISelectedSet ISelectedSet { get; }

        /// <summary>
        /// グループマネージャを取得します。
        /// </summary>
        ILEGroupMgr ILEGroupMgr { get; }

        /// <summary>
        /// テクスチャマネージャ
        /// </summary>
        ITextureMgr ITextureMgr { get; }

        /// <summary>
        /// フォントマネージャ
        /// </summary>
        ILEFontManager ILEFontManager { get; }

        /// <summary>
        /// アニメーション共有情報マネージャ
        /// </summary>
        IAnimShareInfomationManager IAnimShareInfomationManager { get; }

        /// <summary>
        /// アニメーションフレーム区間
        /// </summary>
        IAnimFrameSectionSet IAnimFrameSectionSet { get; }

        /// <summary>
        /// 部品設定
        /// </summary>
        IPartsSettings IPartsSettings { get; }

        /// <summary>
        /// 部品レイアウト
        /// </summary>
        IPartsLayout IPartsLayout { get; }

        /// <summary>
        /// コントロール設定
        /// </summary>
        IEnumerable<IControlSettings> IControlSettings { get; }

        /// <summary>
        /// コントロール設定
        /// </summary>
        IControlSettingsSet IControlSettingsSet { get; }

        /// <summary>
        /// 拡張ユーザデータ
        /// </summary>
        IUserDataHolder IUserDataHolder { get; }

        /// <summary>
        /// アニメーション範囲(開始)
        /// </summary>
        int AnimStartTime { get; }
        /// <summary>
        /// アニメーション範囲(終了)
        /// </summary>
        int AnimEndTime { get; }

        /// <summary>
        /// アニメーション再生範囲(開始)
        /// </summary>
        int AnimPlayStartTime { get; }
        /// <summary>
        /// アニメーション再生範囲(終了)
        /// </summary>
        int AnimPlayEndTime { get; }

        /// <summary>
        /// 選択中の区間タグ名
        /// </summary>
        string CurrentTagName { get; }

        /// <summary>
        /// 分割モード時のアニメーション再評価中かどうか
        /// </summary>
        bool EvaluateMultiAnimation { get; }

        /// <summary>
        /// ファイルパスや更新日時などの情報
        /// </summary>
        LayoutDocument OwnerDocument { get; }

        /// <summary>
        /// ステート関連
        /// </summary>
        IStateMachine IStateMachine { get; }

        #region Undo/Redo コマンド関連

        /// <summary>
        /// Undo可能か?
        /// </summary>
        bool UndoEnable { get; }
        /// <summary>
        /// Redo可能か?
        /// </summary>
        bool RedoEnable { get; }

        /// <summary>
        /// コマンドキューの状態を取得します。
        /// シーン更新の判定に利用します。
        /// </summary>
        SubSceneCmdQueue.InternalState CmdQueueInternalState { get; }
        ISubSceneCmdQueue ISubSceneCmdQueue { get; }
        #endregion Undo/Redo コマンド関連
    }
}
