﻿// --------------------------------------------------------------------------------
// <copyright>
//  Copyright (C)Nintendo/HAL Laboratory, Inc.  All rights reserved.
//
//  These coded instructions, statements, and computer programs contain proprietary
//  information of Nintendo and/or its licensed developers and are protected by
//  national and international copyright laws. They may not be disclosed to third
//  parties or copied or duplicated in any form, in whole or in part, without the
//  prior written consent of Nintendo.
//
//  The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using LECore.Structures.LECoreInterface;
using System.Collections.Generic;
using LECore.Structures.Nsrif.Attributes;
using System.Xml.Serialization;

namespace LECore.Structures
{
    /// <summary>
    /// コンバイナ情報
    /// </summary>
    public class TextureCombineModeDesc
    {
        public string Descriptions { get; private set; }
        public AttrCombineMode CombineMode { get; private set; }

        public TextureCombineModeDesc(AttrCombineMode combineMode, string descriptions)
        {
            this.CombineMode = combineMode;
            this.Descriptions = descriptions;
        }
    }

    /// <summary>
    /// プリファレンス
    /// </summary>
    public class PlatformPreference
    {
        /// <summary>
        /// プラットフォーム名です。
        /// </summary>
        public string PlatformName { get; set; }

        /// <summary>
        /// UI上で表示するプラットフォーム名です。
        /// </summary>
        public string UiPlatformName { get; set; }

        /// <summary>
        /// アーカイバーに指定するプラットフォーム名です。
        /// </summary>
        public string ArchiverPlatformName { get; set; }

        /// <summary>
        /// プラットフォーム
        /// </summary>
        [XmlArrayItem("AcceptableBntxPlatformName")]
        public List<string> AcceptableBntxPlatformNames { get; set; }

        /// <summary>
        /// PCビューアの実行ファイル名です。
        /// </summary>
        public string PcViewerPath { get; set; }

        /// <summary>
        /// 実機ビューアの実行バッチファイル名です。
        /// </summary>
        public string TargetViewerRunnnerPath { get; set; }

        /// <summary>
        /// プレビュースターター名です。
        /// </summary>
        public string PreviewStarterPath { get; set; }

        /// <summary>
        /// PCビューアへの転送プロトコル名です。
        /// </summary>
        public string PcTransferProtocol { get; set; }

        /// <summary>
        /// PCビューアの PeerType です。
        /// </summary>
        public string PcPeerType { get; set; }

        /// <summary>
        /// 実機ビューアへの転送プロトコル名です。
        /// </summary>
        public string TargetTransferProtocol { get; set; }

        /// <summary>
        /// バイナリコンバーターパスです。
        /// </summary>
        public string TargetConverterPath { get; set; }

        /// <summary>
        /// バイナリコンバーターパスです。
        /// </summary>
        public string PCConverterPath { get; set; }

        /// <summary>
        /// バイナリコンバーター追加引数です。
        /// </summary>
        public string PcConverterAdditionalArguments { get; set; }

        /// <summary>
        /// バイナリコンバーター追加引数です。
        /// </summary>
        public string TargetConverterAdditionalArguments { get; set; }

        /// <summary>
        /// アーカイバーパスです。
        /// </summary>
        public string ArchiverPath { get; set; }

        /// <summary>
        /// フォントコンバーターコンソールパスです。
        /// </summary>
        public string FontConverterConsolePath { get; set; }

        /// <summary>
        /// フォントコンバーター追加引数です。
        /// </summary>
        public string FontConverterAdditionalArguments { get; set; }

        /// <summary>
        /// 背景用テクスチャコンバーター追加引数です。
        /// </summary>
        public string BgTextureConverterAdditionalArguments { get; set; }

        /// <summary>
        /// アーカイバ追加引数です。
        /// </summary>
        public string ArchiverAdditionalArguments { get; set; }

        /// <summary>
        /// アーカイバ追加引数です。
        /// </summary>
        public string PreArchiverCommandAdditionalArguments { get; set; }

        /// <summary>
        /// プレビュー用のファイルの保存ディレクトリです。
        /// </summary>
        public string PreviewTemporaryPath { get; set; }

        /// <summary>
        /// NW4F モードで動作するかどうか？（プレビュー時のバイナリコンバーターやビューアーの起動方法に影響します）
        /// </summary>
        public bool IsNW4FMode { get; set; }

        /// <summary>
        /// 実機ビューアの PeerType です。
        /// </summary>
        public string TargetPeerType { get; set; }

        public bool SearchPCViewerByTitle { get; set; }

        public PlatformPreference()
        {
            AcceptableBntxPlatformNames = new List<string>();
        }
    }

    /// <summary>
    /// プラットフォーム固有の情報を提供するインタフェース。
    /// </summary>
    public interface IPlatformDetail
    {
        /// <summary>
        /// プレビュー前(バイナリコンバート前)の準備
        /// </summary>
        bool PrepareBeforeBinaryConvert(bool forPCViewer, string tempFilePath);

        /// <summary>
        /// プレビュー前のフォント変換
        /// </summary>
        bool PrepareBinaryFontForPreview(bool forPCViewer, string tempFilePath);

        /// <summary>
        /// PCビューアの画面サイズ制限が有効か
        /// </summary>
        bool PCViewerScreenSizeLimitEnabled { get; }

        /// <summary>
        // テクスチャサイズが小さすぎる場合に問題が発生する。(ランタイム側で正しくテクスチャを復元できない問題がある)
        // 出力されたフォントのサイズがかなり小さいことを検出して、シートサイズを指定して再度フォント出力をおこなう。
        // ランタイム側の問題が解消したら、この処理は不要になるので削除する。
        /// </summary>
        bool SmallSizeFontProblemWorkaroundEnabled { get; }

        /// <summary>
        /// リニア編集環境が有効かどうか
        /// </summary>
        bool IsLinearGammaSettingEnabled { get; }

        /// <summary>
        /// ツールルートパス
        /// </summary>
        string NwToolsRootPath { get; }

        /// <summary>
        /// レイアウトツールルートパス
        /// </summary>
        string LayoutToolsRootPath { get; }

        /// <summary>
        /// フォントツールルートパス
        /// </summary>
        string NwFontToolsRootPath { get; }

        /// <summary>
        /// バイナリコンバータのパス
        /// </summary>
        string LayoutConverterPath { get; }

        /// <summary>
        /// バイナリコンバータのパス
        /// </summary>
        string LayoutConverterForPCViewerPath { get; }

        /// <summary>
        /// フォントコンバータのパス
        /// </summary>
        string FontConverterConsolePath { get; }

        /// <summary>
        /// PCビューアのパス
        /// </summary>
        string PCLayoutViewerPath { get; }

        /// <summary>
        /// 実機ビューア起動バッチのパス
        /// </summary>
        string RunCafeViewerBatPath { get; }

        /// <summary>
        /// Cafeビューアスタータのパス
        /// </summary>
        string ViewerStarterPath { get; }

        /// <summary>
        /// UI に表示するプラットフォーム名を取得します。
        /// </summary>
        string UIPlatformName { get; }

        /// <summary>
        /// プラットフォーム名を取得します。
        /// </summary>
        string PlatformName { get; }

        /// <summary>
        /// サンプルデータパスです。
        /// </summary>
        string NwDefaultSampleRoot { get; }

        /// <summary>
        /// 実機通信に必要な環境変数設定
        /// </summary>
        IEnumerable<string> GetTargetComEnvList();

        /// <summary>
        /// アイコンを取得します。
        /// </summary>
        System.Drawing.Icon GetIcon();

        /// <summary>
        /// 実機テクスチャでの、ビット当たりのバイト数を取得します。
        /// </summary>
        float GetBPP(TexImagePixelFmt PixelFmt);

        /// <summary>
        /// UIに表示する文字列を取得します。
        /// </summary>
        string GetPixcelFormatString(TexImagePixelFmt fmt);

        /// <summary>
        /// NX用テクスチャDLLを読み込みます。
        /// </summary>
        void LoadNXDll();

        /// <summary>
        /// 実機テクスチャサイズを計算します。(Generic用)
        /// </summary>
        int CalcNativeTextureSize(int width, int height, TexImagePixelFmt fmt);

        /// <summary>
        /// 実機テクスチャサイズを計算します。(NX用)
        /// </summary>
        int CalcNativeTextureSizeNX(int width, int height, TexImagePixelFmt fmt, uint tilingFlags, int sizeThreshold);

        /// <summary>
        /// GPU によるテクスチャのエンコードが可能かどうかを取得します。
        /// </summary>
        bool IsGpuEncodingAvailable { get; }

        /// <summary>
        /// 条件にそった、デフォルトフォーマットを取得する。
        /// </summary>
        TexImagePixelFmt GetDefaultTextureFormat(bool isGrayScale, bool intendedToBeAlphaFormat, TexImagePixelFmt nw4cTgaFormat);

        /// <summary>
        /// フォーマットリストを取得します。
        /// </summary>
        IEnumerable<TexImagePixelFmt> GetFormatList(bool hasAlpha, bool subFormat);

        /// <summary>
        /// フォーマットリストを取得します。
        /// </summary>
        IEnumerable<TexImagePixelFmt> GetGrayScaleFormatList(bool hasAlpha);

        /// <summary>
        /// テクスチャコンバイナ設定の候補を取得します。
        /// </summary>
        IEnumerable<TextureCombineModeDesc> GetTextreuCombinerInfoColor(int stageNo);

        /// <summary>
        /// テクスチャコンバイナ設定の候補を取得します。
        /// </summary>
        IEnumerable<TextureCombineModeDesc> GetTextreuCombinerInfoAlpha();

        /// <summary>
        /// 利用可能なステージ数を取得します。
        /// </summary>
        int GetNumAvailableStages(ITevStages tevStages);

        /// <summary>
        /// 非２べき乗テクスチャでミラーやリピートが可能か？
        /// </summary>
        bool CheckTextureMirrorRepeatEnabled(TexImagePixelFmt nw4cTgaFormat, int size);

        /// <summary>
        /// エディタでのテクスチャフォーマット変更を許すか？
        /// </summary>
        bool TextureFormatEditEnabled { get; }

        /// <summary>
        /// ウインドウペインのカスタムフレームサイズ設定が有効かどうか
        /// </summary>
        bool CustomWindowFrameSizeEnabled { get; }

        /// <summary>
        /// インダイレクトテクスチャが有効か
        /// </summary>
        bool IndirectTextureEnabled { get; }

        /// <summary>
        /// 複数画面が存在するか
        /// </summary>
        bool MultiScreenEnabled { get; }

        /// <summary>
        /// 詳細なマテリアルの編集は可能か？
        /// </summary>
        bool LowLevelCombinerSettingsEnabled { get; }

        /// <summary>
        /// 最大TEVステージ数を取得します。
        /// </summary>
        int MaxTevStageCount { get; }

        /// <summary>
        /// 手部コンスタントカラーの数を取得します。
        /// </summary>
        int TevConstantColorCount { get; }

        /// <summary>
        /// Bntx 中のプラットフォームを受け付けるかを判定します。
        /// </summary>
        bool IsAcceptableBntxPlatform(string platform);

        /// <summary>
        /// プレビュー一時ディレクトリパスを取得します。
        /// </summary>
        string PreviewTemporaryPath { get; }

        /// <summary>
        /// プラットフォーム設定です。
        /// </summary>
        PlatformPreference PlatformPreference { get; }

        /// <summary>
        /// NX用テクスチャDLLの読み込み状態です。
        /// </summary>
        bool IsNXDllLoaded { get; }

    }

    /// <summary>
    /// プラットフォーム設定を読み込むプラグイン専用のインタフェースです。
    /// </summary>
    public interface IPlatformPreferenceOwner
    {
        /// <summary>
        /// 現在のプラットフォーム設定です。
        /// </summary>
        PlatformPreference PlatformPreference { get; set; }

        /// <summary>
        /// プラットフォーム設定の一覧です。
        /// </summary>
        List<PlatformPreference> PlatformPreferences { get; }
    }
}
