﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;

namespace LECore.Structures
{
    /// <summary>
    /// ヘルパー
    /// </summary>
    public static class ParamaterPaneHelper
    {
        public static bool IsValid(this IParamaterPane pane, ISubScene subScene)
        {
            if (string.IsNullOrEmpty(pane.PaneName))
            {
                if (pane.IsRequired)
                {
                    return false;
                }
            }
            else
            {
                // なにか指定されていた場合、必須かどうかにかかわらず、対象が発見できなければ不正
                if (subScene.FindPaneByName(pane.PaneName) == null)
                {
                    return false;
                }
            }

            return true;
        }
    }

    /// <summary>
    /// ペインパラメータです。
    /// </summary>
    public interface IParamaterPane
    {
        /// <summary>
        /// パラメータの名前です。
        /// </summary>
        string Name { get; }

        /// <summary>
        /// ペイン名です。
        /// </summary>
        string PaneName { get; set; }

        /// <summary>
        /// 設定が必須かどうかです。
        /// </summary>
        bool IsRequired { get; }
    }

    /// <summary>
    /// ヘルパー
    /// </summary>
    public static class ParamaterAnimaitonHelper
    {
        public static bool IsValid(this IParamaterAnimaiton anim, IAnimFrameSectionSet animFrameSectionSet)
        {
            if(string.IsNullOrEmpty(anim.TagName))
            {
                if(anim.IsRequired)
                {
                    return false;
                }
            }
            else
            {
                // なにか指定されていた場合、必須かどうかにかかわらず、対象が発見できなければ不正
                if(animFrameSectionSet.FindAnimFrameSectionByName(anim.TagName) == null)
                {
                    return false;
                }
            }

            return true;
        }
    }

    /// <summary>
    /// アニメーションパラメータです。
    /// </summary>
    public interface IParamaterAnimaiton
    {
        /// <summary>
        /// パラメータの名前です。
        /// </summary>
        string Name { get; }

        /// <summary>
        /// アニメーション区間タグの名前です。
        /// </summary>
        string TagName { get; set; }

        /// <summary>
        /// 設定が必須かどうかです。
        /// </summary>
        bool IsRequired { get; }
    }

    /// <summary>
    /// 部品上書きプロパティの設定です。
    /// </summary>
    public interface IPartsPropaertySettings
    {
        /// <summary>
        /// 種類です。
        /// </summary>
        PaneKind PaneKind { get; }

        /// <summary>
        /// 対象ペイン名です。
        /// </summary>
        string TargetName { get; }

        /// <summary>
        /// 説明です。
        /// </summary>
        string Description { get; }
    }

    /// <summary>
    /// 部品プリセット設定です。
    /// </summary>
    public interface IDerivativePartsSettings
    {
        /// <summary>
        /// 名前
        /// </summary>
        string Name { get; }

        /// <summary>
        /// 説明
        /// </summary>
        string Descriptions { get; }

        /// <summary>
        /// 部品のレイアウトです。
        /// </summary>
        ISubScene PartsSubScene { get; }

        /// <summary>
        /// 上書きパラメータ列です。
        /// </summary>
        IEnumerable<IPartsPropaerty> PartsPropaerties { get; }
    }

    /// <summary>
    ///
    /// </summary>
    public interface IControlSettings
    {
        /// <summary>
        /// 名前です。
        /// </summary>
        string Name { get; }

        /// <summary>
        /// UI表示名です。
        /// </summary>
        string UIName { get; }

        /// <summary>
        /// 短い説明名称です。
        /// </summary>
        string DescriptionName { get; }

        /// <summary>
        /// 部品の説明文字列です。
        /// </summary>
        string Description { get; }

        /// <summary>
        /// 部品の説明文字列です。
        /// </summary>
        IControlSettingsSet Owner { get; }

        /// <summary>
        /// Readonlyのペインパラメータ列です。
        /// </summary>
        IEnumerable<IParamaterPane> Panes { get; }

        /// <summary>
        /// Readonlyのアニメーションパラメータ列です。
        /// </summary>
        IEnumerable<IParamaterAnimaiton> Animations { get; }

        /// <summary>
        /// 拡張ユーザデータ
        /// </summary>
        IUserDataHolder IUserDataHolder { get; }
    }

    /// <summary>
    ///
    /// </summary>
    public interface IControlSettingsSet
    {
        void CloneAdd(IControlSettings newItem);
        void Remove(IControlSettings newItem);
        void Clear();

        ISubScene Owner { get; }

        IEnumerable<IControlSettings> Items { get; }
    }

    /// <summary>
    /// パーツ設定です。
    /// <remarks>
    /// 中間ファイrのControlタグに対応します。
    /// 一度初期化したら、のちの更新は
    /// PaneName、TagNameに対してのみとなります。
    /// </remarks>
    /// </summary>
    public interface IPartsSettings
    {
        /// <summary>
        /// 名前です。
        /// </summary>
        string Name { get; }

        /// <summary>
        /// 短い説明名称です。
        /// </summary>
        string DescriptionName { get; }

        /// <summary>
        /// 部品の説明文字列です。
        /// </summary>
        string Description { get; }

        /// <summary>
        /// 上書きプロパティです。
        /// </summary>
        IEnumerable<IPartsPropaertySettings> OverridePorperties { get; }

        /// <summary>
        /// 部品の説明文字列です。
        /// </summary>
        ISubScene Owner { get; }

        /// <summary>
        /// 初期状態かどうかです。
        /// </summary>
        bool IsInitialState { get; }

        /// <summary>
        /// 基底レイアウトの名前です。
        /// <remarks>
        /// 派生部品の場合だけ設定されます。
        /// </remarks>
        /// </summary>
        string BasePartsName { get; }

        /// <summary>
        /// 基底レイアウトの最終変更時間です。
        /// </summary>
        DateTime BasePartsLastModify { get; }

        /// <summary>
        /// 既定レイアウトファイルのハッシュ値 (SHA1) です。
        /// </summary>
        string BasePartsHashValue { get; }

        /// <summary>
        /// 元のサイズです。
        /// </summary>
        FVec3 OriginalSize { get; }
    }
}
