﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Diagnostics;
using LECore.Structures.Core;
using System.Linq;

namespace LECore.Structures
{
    /// <summary>
    /// ユーザ拡張データ(要素種類)
    /// </summary>
    public enum UserDataKind
    {
        None = 0,
        Float,
        Int,
        String,
        AnmInt,
        AnmFloat,
        AnmIntVec2,
        AnmFloatVec2,
        AnmFloatVec3,
        AnmByteRGBA4,
        AnmByteRGBA4Degamma,
    }

    /// <summary>
    /// ユーザ拡張データ(要素)
    /// 公開インタフェース
    /// </summary>
    public interface IUserDataElement
    {
        UserDataKind UserDataKind { get; }
        int Size { get; }
        string Name { get; }
        bool Overwrite { get; }
        object Value { get; }
        IAnmAttribute IAnmAttribute { get; }
    }

    /// <summary>
    /// ユーザ拡張データ(要素)
    /// ヘルパ
    /// </summary>
    public static class UserDataElementHelper
    {
        public const string BlankSeparatorStr = " ";

        /// <summary>
        /// 種類をあらわす文字列を取得します。
        /// </summary>
        public static string GetUserDataKindUIString( UserDataKind kind )
        {
            switch( kind )
            {
                case UserDataKind.None: return "None";
                case UserDataKind.Int: return "Int";
                case UserDataKind.Float: return "Float";
                case UserDataKind.String: return "String";
                case UserDataKind.AnmInt: return "AnmInt";
                case UserDataKind.AnmFloat: return "AnmFloat";
                case UserDataKind.AnmIntVec2: return "AnmIntVec2";
                case UserDataKind.AnmFloatVec2: return "AnmFloatVec2";
                case UserDataKind.AnmFloatVec3: return "AnmFloatVec3";
                case UserDataKind.AnmByteRGBA4: return "AnmByteRGBA4";
                case UserDataKind.AnmByteRGBA4Degamma: return "AnmByteRGBA4Degamma";
                default: Debug.Assert( false ); return "";
            }
        }

        /// <summary>
        /// 拡張ユーザ情報を一行で表示
        /// </summary>
        public static string ToOneLineString(IUserDataElement[] userData)
        {
            var sb = new StringBuilder();
            var sp = "";
            foreach (var item in userData)
            {
                sb.Append(sp);
                sb.Append(item.Name);
                sb.Append("(" + GetUserDataKindUIString(item.UserDataKind) + ")");
                sb.Append(":");
                sb.Append(ToString(item.UserDataKind, item.Value, false, " ", true));
                sp = ", ";
            }
            return sb.ToString();
        }

        /// <summary>
        /// 名前から要素を検索します。
        /// </summary>
        public static IUserDataElement FindByName( string name, System.Collections.IList elementSet )
        {
            Debug.Assert( elementSet != null );

            foreach( IUserDataElement element in elementSet )
            {
                if( element.Name == name )
                {
                    return element;
                }
            }
            return null;
        }

        /// <summary>
        /// 文字列に変換します。
        /// </summary>
        public static string ToString( IUserDataElement userData )
        {
            return ToString( userData.UserDataKind, userData.Value, false, " ", false);
        }

        /// <summary>
        /// 文字列に変換します。
        /// </summary>
        public static string ToString( UserDataKind kind, object value, bool bAsHex )
        {
            return ToString( kind, value, bAsHex, " ", false );
        }

        /// <summary>
        /// 文字列に変換します。
        /// </summary>
        public static string ToString( UserDataKind kind, object value, bool bAsHex, string separatorStr, bool replaceNewLine )
        {
            value = ConvertAnmValueToArray(kind, value);

            if ( kind == UserDataKind.String )
            {
                if (replaceNewLine)
                {
                    // 改行で区切る形にする。
                    // 特に separatorStr が \r\n のときは \n が \r\n に置き換わる。
                    return (value as string).Replace("\n", separatorStr);
                }
                else
                {
                    return (value as string);
                }
            }
            else if( value is int[] )
            {
                int[] intArray = value as int[];

                StringBuilder sb = new StringBuilder();
                for( int i = 0 ; i < intArray.Length ; i++ )
                {
                    bool isNextElementExists = i + 1 < intArray.Length;
                    string actualSeparatorStr = isNextElementExists ? separatorStr : "";
                    if( bAsHex )
                    {
                        sb.AppendFormat("{0:x08}{1}", intArray[i], actualSeparatorStr);
                    }else{
                        sb.AppendFormat("{0:d}{1}", intArray[i], actualSeparatorStr);
                    }
                }
                return sb.ToString();
            }
            else if ( value is float[] )
            {
                float[] floatArray = value as float[];

                StringBuilder sb = new StringBuilder();
                for( int i = 0 ; i < floatArray.Length ; i++ )
                {
                    bool isNextElementExists = i + 1 < floatArray.Length;
                    string actualSeparatorStr = isNextElementExists ? separatorStr : "";

                    sb.AppendFormat("{0:f7}{1}", floatArray[i], actualSeparatorStr);
                }
                return sb.ToString();
            }

            return null;
        }

        /// <summary>
        /// アニメーションの値を配列に変換します。
        /// </summary>
        internal static object ConvertAnmValueToArray(UserDataKind kind, object value)
        {
            switch (kind)
            {
                case UserDataKind.AnmInt:
                    return new int[1] { (int)value };
                case UserDataKind.AnmIntVec2:
                    return new int[2] { ((IVec2)value).X, ((IVec2)value).Y };
                case UserDataKind.AnmByteRGBA4:
                case UserDataKind.AnmByteRGBA4Degamma:
                    return new int[4] { ((RGBAColor)value).R, ((RGBAColor)value).G, ((RGBAColor)value).B, ((RGBAColor)value).A };
                case UserDataKind.AnmFloat:
                    return new float[1] { (float)value };
                case UserDataKind.AnmFloatVec2:
                    return new float[2] { ((FVec2)value).X, ((FVec2)value).Y };
                case UserDataKind.AnmFloatVec3:
                    return new float[3] { ((FVec3)value).X, ((FVec3)value).Y, ((FVec3)value).Z };
            }

            return value;
        }

        /// <summary>
        ///
        /// </summary>
        public static string GetValifiedStringForValue( UserDataKind kind, string valueString )
        {
            if( kind == UserDataKind.String )
            {
                return AppConstants.RegexInvalidCharForUserDataExString.Replace( valueString, "" );
            }
            else
            {
                Regex regexInvalidForNumber = new Regex( @"[^0-9.+\n\r -]" );
                return regexInvalidForNumber.Replace( valueString, "" );
            }
        }

        /// <summary>
        /// ２つの値文字列が同じ値かどうかを判定します。
        /// </summary>
        public static bool IsStringValueSame(UserDataKind kind, string valueStringL, string valueStringR, string separatorStr)
        {
            // 両方とも値に戻した後、再度文字列化して比較する。
            object objL = ParseString(kind, valueStringL, separatorStr);
            object objR = ParseString(kind, valueStringR, separatorStr);

            return (objL != null && objR != null) ?
                ToString(kind, objL, false) == ToString(kind, objR, false) :
                objL == objR;
        }

        /// <summary>
        /// 文字列を解釈します。正常に解釈できない場合は null を返します。
        /// </summary>
        public static object ParseString( UserDataKind kind, string valueString, string separatorStr)
        {
            if( kind == UserDataKind.String )
            {
                if (separatorStr == Environment.NewLine)
                {
                    return valueString.Replace(Environment.NewLine, "\n");
                }
                else
                {
                    return valueString;
                }
            }
            else
            {
                if( IsIntKind( kind ) )
                {
                    if (string.IsNullOrEmpty(valueString))
                    {
                        return null;
                    }

                    string[] valueSet = valueString.Trim().Split(separatorStr.ToCharArray(),StringSplitOptions.RemoveEmptyEntries);

                    int[] intSet = new int[valueSet.Length];
                    for( int i = 0 ; i < valueSet.Length ; i++ )
                    {
                        float fval;
                        float.TryParse(valueSet[i], out fval);
                        try
                        {
                            intSet[i] = (int)fval;
                        }
                        catch
                        {
                            intSet[i] = 0;
                        }
                    }

                    switch (kind)
                    {
                        case UserDataKind.Int:
                            return intSet;
                        case UserDataKind.AnmInt:
                            if (intSet.Length == 1)
                            {
                                return intSet[0];
                            }
                            break;
                        case UserDataKind.AnmIntVec2:
                            if (intSet.Length == 2)
                            {
                                return new IVec2(intSet[0], intSet[1]);
                            }
                            break;
                        case UserDataKind.AnmByteRGBA4:
                        case UserDataKind.AnmByteRGBA4Degamma:
                            if (intSet.Length == 4)
                            {
                                return new RGBAColor(intSet[0], intSet[1], intSet[2], intSet[3]);
                            }
                            break;
                    }
                }
                else if( IsFloatKind( kind ) )
                {
                    if (string.IsNullOrEmpty(valueString))
                    {
                        return null;
                    }

                    string[] valueSet = valueString.Trim().Split(separatorStr.ToCharArray(), StringSplitOptions.RemoveEmptyEntries);

                    float[] floatSet = new float[valueSet.Length];
                    for( int i = 0 ; i < valueSet.Length ; i++ )
                    {
                        Single.TryParse( valueSet[i], out floatSet[i] );
                    }

                    switch (kind)
                    {
                        case UserDataKind.Float:
                            return floatSet;
                        case UserDataKind.AnmFloat:
                            if (floatSet.Length == 1)
                            {
                                return floatSet[0];
                            }
                            break;
                        case UserDataKind.AnmFloatVec2:
                            if (floatSet.Length == 2)
                            {
                                return new FVec2(floatSet[0], floatSet[1]);
                            }
                            break;
                        case UserDataKind.AnmFloatVec3:
                            if (floatSet.Length == 3)
                            {
                                return new FVec3(floatSet[0], floatSet[1], floatSet[2]);
                            }
                            break;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// アニメーション型がどうかを取得します。
        /// </summary>
        public static bool IsAnimationKind(UserDataKind kind)
        {
            switch(kind)
            {
                case UserDataKind.None:
                case UserDataKind.Float:
                case UserDataKind.Int:
                case UserDataKind.String:
                    return false;
                default:
                    return true;
            }
        }

        /// <summary>
        /// 編集操作が可能なユーザーデータ型かどうかを取得します。
        /// </summary>
        public static bool IsInvalidModificationOfKind(UserDataKind beforeKind, UserDataKind afterKind)
        {
            // 変更がない
            if (beforeKind == afterKind)
            {
                return false;
            }

            return IsAnimationKind(beforeKind) || IsAnimationKind(afterKind);
        }

        /// <summary>
        /// 要素型がint配列かどうか取得します。
        /// </summary>
        /// <returns>要素型がint配列のときtrueを返します。</returns>
        public static bool IsIntKind( UserDataKind kind)
        {
            if ( kind == UserDataKind.Int || kind == UserDataKind.AnmInt || kind == UserDataKind.AnmIntVec2 || kind == UserDataKind.AnmByteRGBA4 || kind == UserDataKind.AnmByteRGBA4Degamma )
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// 要素型がfloat配列かどうか取得します。
        /// </summary>
        /// <returns>要素型がfloat配列のときtrueを返します。</returns>
        public static bool IsFloatKind( UserDataKind kind )
        {
            if ( kind == UserDataKind.Float || kind == UserDataKind.AnmFloat || kind == UserDataKind.AnmFloatVec2 || kind == UserDataKind.AnmFloatVec3 )
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// パラメータが妥当か判定します。
        /// </summary>
        public static bool CheckParamatersValid( string name, UserDataKind kind, object value )
        {
            // name について...
            if( name == null || name.Length <= 0 )
            {
                return false;
            }
            else
            {
                if( AppConstants.RegexInvalidCharForUserDataExName.IsMatch( name ) )
                {
                    return false;
                }
            }

            // kindについて...
            if( kind == UserDataKind.None )
            {
                return true;
            }


            // value について...
            if( value == null )
            {
                return false;
            }
            else
            {
                if ( CheckParameterValueValid( kind, value ) == false )
                {
                    return false;
                }
            }

            // 有効である。
            return true;
        }

        /// <summary>
        /// パラメータの値が妥当か判定します。
        /// </summary>
        public static bool CheckParameterValueValid( UserDataKind kind, object value )
        {
            switch (kind)
            {
                case UserDataKind.String:
                    return !string.IsNullOrEmpty(value as string) && !AppConstants.RegexInvalidCharForUserDataExString.IsMatch((string)value);
                case UserDataKind.Int:
                    return value is int[] && ((int[])value).Length > 0;
                case UserDataKind.Float:
                    return value is float[] && ((float[])value).Length > 0;
                case UserDataKind.AnmInt:
                    return value is int;
                case UserDataKind.AnmIntVec2:
                    return value is IVec2;
                case UserDataKind.AnmByteRGBA4:
                case UserDataKind.AnmByteRGBA4Degamma:
                    return value is RGBAColor;
                case UserDataKind.AnmFloat:
                    return value is float;
                case UserDataKind.AnmFloatVec2:
                    return value is FVec2;
                case UserDataKind.AnmFloatVec3:
                    return value is FVec3;
            }
            return false;
        }
    }

    /// <summary>
    /// ユーザ拡張データ(要素)
    /// </summary>
    internal class UserDataElement : IUserDataElement
    {
        //----------------------------------------------------------

        //----------------------------------------------------------
        #region フィールド・プロパティ
        string _name;
        UserDataKind _userDataKind;
        bool _overwrite;
        object _value;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public UserDataElement( string name, UserDataKind kind, bool overwrite, object value )
        {
            _name = name;
            _userDataKind = kind;
            _overwrite = overwrite;

            AttributeType? type = null;
            // サブアトリビュート記述子を取得
            switch (kind)
            {
                case UserDataKind.AnmInt:
                    type = AttributeType.Int;
                    break;

                case UserDataKind.AnmIntVec2:
                    type = AttributeType.IntVec2;
                    break;

                case UserDataKind.AnmFloat:
                    type = AttributeType.Float;
                    break;

                case UserDataKind.AnmFloatVec2:
                    type = AttributeType.FloatVec2;
                    break;

                case UserDataKind.AnmFloatVec3:
                    type = AttributeType.FloatVec3;
                    break;

                case UserDataKind.AnmByteRGBA4:
                case UserDataKind.AnmByteRGBA4Degamma:
                    type = AttributeType.ByteRGBA4;
                    break;
            }

            if (type.HasValue)
            {
                var anmAttribute = new AnmAttribute(null, null, new AnmAttrDescripter(type.Value, name, null, value))
                {
                    EnableLocalize = false,
                };

                _value = anmAttribute;
            }
            else
            {
                _value = value;
            }
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        public void Set( string name, UserDataKind kind, bool overwrite, object value, object baseValue )
        {
            if( UserDataElementHelper.CheckParamatersValid( name, kind, value ) )
            {
                _name = name;
                _userDataKind = kind;
                _overwrite = overwrite;

                if (_value is AnmAttribute)
                {
                    AnmAttribute userDataAttribute = (AnmAttribute)_value;
                    userDataAttribute.SetValue(value);
                    userDataAttribute.SetBaseValue(baseValue);
                    userDataAttribute.Name = name;
                }
                else
                {
                    _value = value;
                }
            }
        }

        /// <summary>
        /// ユーザーデータアトリビュートを取得します。
        /// </summary>
        public AnmAttribute UserDataAttribute
        {
            get { return _value as AnmAttribute; }
        }

        #endregion

        //----------------------------------------------------------
        #region IUserDataElement メンバ

        /// <summary>
        /// データ種類
        /// </summary>
        public UserDataKind UserDataKind
        {
            get { return _userDataKind; }
        }

        /// <summary>
        /// 文字列の値を取得します。
        /// </summary>
        string ValueAsString_
        {
            get
            {
                if( _value == null )
                {
                    return string.Empty;
                }
                else
                {
                    return _value.ToString();
                }
            }
        }

        /// <summary>
        /// データサイズを取得します。
        /// </summary>
        public int Size
        {
            get
            {
                switch( _userDataKind )
                {
                    case UserDataKind.None: return 0;
                    case UserDataKind.Float: return ( _value as float[] ).Length;
                    case UserDataKind.Int: return ( _value as int[]).Length;
                    case UserDataKind.String: return ValueAsString_.Length;
                    case UserDataKind.AnmInt: return 1;
                    case UserDataKind.AnmIntVec2: return 2;
                    case UserDataKind.AnmFloat: return 1;
                    case UserDataKind.AnmFloatVec2: return 2;
                    case UserDataKind.AnmFloatVec3: return 3;
                    case UserDataKind.AnmByteRGBA4:
                    case UserDataKind.AnmByteRGBA4Degamma:
                        return 4;
                    default: Debug.Assert( false ); return 0;
                }
            }
        }

        /// <summary>
        /// データ項目名を取得します。
        /// </summary>
        public string Name
        {
            get { return _name; }
        }

        /// <summary>
        /// データを上書きするか取得します。
        /// </summary>
        public bool Overwrite
        {
            get { return _overwrite; }
        }

        /// <summary>
        /// 値を取得します。
        /// </summary>
        public object Value
        {
            get
            {
                if( _value is AnmAttribute )
                {
                    AnmAttribute userDataAttribute = (AnmAttribute)_value;
                    object value;
                    userDataAttribute.GetValue(out value);
                    return value;
                }
                else
                {
                    return _value;
                }
            }
        }

        /// <summary>
        /// アニメーションアトリビュートを取得します。
        /// </summary>
        public IAnmAttribute IAnmAttribute
        {
            get { return this._value as IAnmAttribute; }
        }

        #endregion
    }
}
