﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures
{
    using Core;

    /// <summary>
    /// 矩形 のアニメーションアトリビュート
    /// </summary>
    internal class RectAttribute
        : AnmAttribute
    {
        bool _stopModifyEvent = true;

        static readonly AnmAttrDescripter[] SubAttrDescs =
        {
            new  AnmAttrDescripter( AttributeType.Float, "Top", null, 0.0f ),
            new  AnmAttrDescripter( AttributeType.Float, "Bottom", null, 0.0f ),
            new  AnmAttrDescripter( AttributeType.Float, "Left", null, 0.0f ),
            new  AnmAttrDescripter( AttributeType.Float, "Right", null, 0.0f )
        };

        #region ------------- プロパティ -------------

        // 主にセーブロードで使用
        public AnmAttribute TopAnmAttr { get { return FindAttributeByIdx(0); } }
        public AnmAttribute BottomAnmAttr { get { return FindAttributeByIdx(1); } }
        public AnmAttribute LeftAnmAttr { get { return FindAttributeByIdx(2); } }
        public AnmAttribute RightAnmAttr { get { return FindAttributeByIdx(3); } }


        public float Top
        {
            get { return TopAnmAttr.GetAsFloat(); }
            set { TopAnmAttr.SetValue(value); }
        }

        public float Bottom
        {
            get { return BottomAnmAttr.GetAsFloat(); }
            set { BottomAnmAttr.SetValue(value); }
        }

        public float Left
        {
            get { return LeftAnmAttr.GetAsFloat(); }
            set { LeftAnmAttr.SetValue(value); }
        }

        public float Right
        {
            get { return RightAnmAttr.GetAsFloat(); }
            set { RightAnmAttr.SetValue(value); }
        }

        public RectangleF AsRectangle
        {
            get { return RectangleF.FromLTRB(this.Left, this.Top, this.Right, this.Bottom); }
        }

        public bool IsSame(LEWindowFrameSize frameSize)
        {
            return
                this.Top == frameSize.Top &&
                this.Bottom == frameSize.Bottom &&
                this.Left == frameSize.Left &&
                this.Right == frameSize.Right;
        }

        public void Set(LEWindowFrameSize frameSize)
        {
            _stopModifyEvent = true;

            this.Top = frameSize.Top;
            this.Bottom = frameSize.Bottom;
            this.Left = frameSize.Left;
            if (!frameSize.IsSpecified)
            {
                // 自動設定のときは、基本値も更新する
                TopAnmAttr.SetBaseValue(frameSize.Top);
                BottomAnmAttr.SetBaseValue(frameSize.Bottom);
                LeftAnmAttr.SetBaseValue(frameSize.Left);
                RightAnmAttr.SetBaseValue(frameSize.Right);
            }

            _stopModifyEvent = false;

            this.Right = frameSize.Right;
        }

        #endregion ------------- プロパティ -------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RectAttribute(LEDataNode owner, string name)
            : base(
            owner,
            SubAttrDescs,
            new AnmAttrDescripter(AttributeType.Combined, name, null, null))
        {
            this.TopAnmAttr.MinValue = 0.0f;
            this.BottomAnmAttr.MinValue = 0.0f;
            this.LeftAnmAttr.MinValue = 0.0f;
            this.RightAnmAttr.MinValue = 0.0f;

            _stopModifyEvent = false;
        }

        public override void NotifyModifyEvent(LEDataNode sender, int kind)
        {
            if (_stopModifyEvent)
            {
                return;
            }

            base.NotifyModifyEvent(this, kind);
        }
    }
}
