﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using LECore.Structures.LECoreInterface;

namespace LECore.Structures
{
    using Core;
    using LECore.Structures.Nsrif.Attributes;
    using LECore.Manipulator;

    internal class ProceduralShape :
        LEDataNode,
        IProceduralShape,
        IDisposable
    {
        public const string ProceduralShapeNodeName = "ProceduralShape";

        readonly ProceduralShapeAttribute _attribute = null;

        readonly Pane _ownerPane = null;
        bool _isProceduralShapeEnabled = false;
        bool _isInnerStrokeEnabled = false;
        bool _isInnerShadowEnabled = false;
        bool _isColorOverlayEnabled = false;
        bool _isGradationOverlayEnabled = false;
        bool _isOuterShadowEnabled = false;
        ProceduralShapeEffectBlendMode _strokeBlendMode = ProceduralShapeEffectBlendMode.Normal;
        ProceduralShapeEffectBlendMode _colorOverlayBlendMode = ProceduralShapeEffectBlendMode.Normal;
        ProceduralShapeEffectBlendMode _gradationOverlayBlendMode = ProceduralShapeEffectBlendMode.Normal;
        ProceduralShapeEffectBlendMode _innerShadowBlendMode = ProceduralShapeEffectBlendMode.Multiply;
        ProceduralShapeEffectBlendMode _outerShadowBlendMode = ProceduralShapeEffectBlendMode.Multiply;
        ProceduralShapeShadowType _innerShadowType = ProceduralShapeShadowType.Normal;
        ProceduralShapeShadowType _outerShadowType = ProceduralShapeShadowType.Normal;
        int _gradationOverlayControlPointCount = 2;

        public ProceduralShape(Pane ownerPane)
            : base(ownerPane, ProceduralShapeNodeName)
        {
            System.Diagnostics.Debug.Assert(ownerPane != null);

            _ownerPane = ownerPane;

            _attribute = new Structures.ProceduralShape.ProceduralShapeAttribute(this);
        }

        public void Dispose()
        {
            _attribute.Dispose();
        }

        /// <summary>
        /// 変更をシーンに通知します。
        /// </summary>
        void NotifyChangeToScene_()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        #region IProceduralShape


        public IPane OwnerPane
        {
            get
            {
                return _ownerPane;
            }
        }

        #region 角丸

        /// <summary>
        /// 角丸が有効かどうか
        /// </summary>
        public bool IsProceduralShapeEnabled
        {
            set
            {
                if (_isProceduralShapeEnabled != value)
                {
                    _isProceduralShapeEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isProceduralShapeEnabled;
            }
        }

        /// <summary>
        /// 任意形状の形状設定パラメータ。
        /// x ^ Exp + y ^ Exp = 1 の曲線で形状が決定する。
        /// </summary>
        public float Exp
        {
            set
            {
                _attribute.ExpAttr.SetValue(value);
            }

            get
            {
                return _attribute.ExpAttr.GetAsFloat();
            }
        }

        /// <summary>
        /// 角丸半径
        /// </summary>
        public float Radius
        {
            set
            {
                _attribute.RadiusAttr.SetValue(value);
            }

            get
            {
                return _attribute.RadiusAttr.GetAsFloat();
            }
        }

        #endregion

        #region 境界線(内側)

        /// <summary>
        /// 境界線(内側)機能が有効かどうか
        /// </summary>
        public bool IsInnerStrokeEnabled
        {
            set
            {
                if (_isInnerStrokeEnabled != value)
                {
                    _isInnerStrokeEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isInnerStrokeEnabled;
            }
        }

        /// <summary>
        /// 境界線(内側)の太さ
        /// </summary>
        public float InnerStrokeSize
        {
            set
            {
                _attribute.InnerStrokeSizeAttr.SetValue(value);
            }

            get
            {
                return _attribute.InnerStrokeSizeAttr.GetAsFloat();
            }
        }

        /// <summary>
        /// 境界線(内側)のブレンド方法
        /// </summary>
        public ProceduralShapeEffectBlendMode InnerStrokeBlendMode
        {
            set
            {
                if (_strokeBlendMode != value)
                {
                    _strokeBlendMode = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _strokeBlendMode;
            }
        }

        /// <summary>
        /// 境界線(内側)の色
        /// </summary>
        public RGBAColor InnerStrokeColor
        {
            set
            {
                _attribute.InnerStrokeColorAttr.SetValue(value);
            }

            get
            {
                return _attribute.InnerStrokeColorAttr.GetAsRBGA();
            }
        }

        #endregion

        #region ドロップシャドウ(内側)

        /// <summary>
        /// ドロップシャドウ(内側)機能が有効かどうか
        /// </summary>
        public bool IsInnerShadowEnabled
        {
            set
            {
                if (_isInnerShadowEnabled != value)
                {
                    _isInnerShadowEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isInnerShadowEnabled;
            }
        }

        /// <summary>
        /// ドロップシャドウ(内側)
        /// </summary>
        public ProceduralShapeEffectBlendMode InnerShadowBlendMode
        {
            set
            {
                if (_innerShadowBlendMode != value)
                {
                    _innerShadowBlendMode = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _innerShadowBlendMode;
            }

        }

        /// <summary>
        /// ドロップシャドウ(内側)の色
        /// </summary>
        public RGBAColor InnerShadowColor
        {
            set
            {
                _attribute.InnerShadowColorAttr.SetValue(value);
            }

            get
            {
                return _attribute.InnerShadowColorAttr.GetAsRBGA();
            }
        }

        /// <summary>
        /// ドロップシャドウ(内側)の角度
        /// </summary>
        public float InnerShadowAngle
        {
            set
            {
                _attribute.InnerShadowAngleAttr.SetValue(value);
            }

            get
            {
                return _attribute.InnerShadowAngleAttr.GetAsFloat();
            }
        }

        /// <summary>
        /// ドロップシャドウ(内側)の距離
        /// </summary>
        public float InnerShadowDistance
        {
            set
            {
                _attribute.InnerShadowDistanceAttr.SetValue(value);
            }

            get
            {
                return _attribute.InnerShadowDistanceAttr.GetAsFloat();
            }
        }

        /// <summary>
        /// ドロップシャドウ(内側)のサイズ
        /// </summary>
        public float InnerShadowSize
        {
            set
            {
                _attribute.InnerShadowSizeAttr.SetValue(value);
            }

            get
            {
                return _attribute.InnerShadowSizeAttr.GetAsFloat();
            }
        }

        /// <summary>
        /// ドロップシャドウ(内側)の影の減衰タイプ
        /// </summary>
        public ProceduralShapeShadowType InnerShadowType
        {
            set
            {
                if (_innerShadowType != value)
                {
                    _innerShadowType = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _innerShadowType;
            }
        }

        #endregion

        #region カラーオーバーレイ

        /// <summary>
        /// カラーオーバーレイが有効かどうか
        /// </summary>
        public bool IsColorOverayEnabled
        {
            set
            {
                if (_isColorOverlayEnabled != value)
                {
                    _isColorOverlayEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isColorOverlayEnabled;
            }
        }

        /// <summary>
        /// カラーオーバーレイのブレンドモード
        /// </summary>
        public ProceduralShapeEffectBlendMode ColorOverlayBlendMode
        {
            set
            {
                if (_colorOverlayBlendMode != value)
                {
                    _colorOverlayBlendMode = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _colorOverlayBlendMode;
            }

        }

        /// <summary>
        /// カラーオーバーレイの色
        /// </summary>
        public RGBAColor ColorOverlayColor
        {
            set
            {
                _attribute.ColorOverlayColorAttr.SetValue(value);
            }

            get
            {
                return _attribute.ColorOverlayColorAttr.GetAsRBGA();
            }
        }

        #endregion

        #region グラデーションオーバーレイ

        /// <summary>
        /// グラデーションオーバーレイが有効かどうか
        /// </summary>
        public bool IsGradationOverayEnabled
        {
            set
            {
                if (_isGradationOverlayEnabled != value)
                {
                    _isGradationOverlayEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isGradationOverlayEnabled;
            }
        }

        /// <summary>
        /// グラデーションオーバーレイのブレンド方法
        /// </summary>
        public ProceduralShapeEffectBlendMode GradationOverlayBlendMode
        {
            set
            {
                if (_gradationOverlayBlendMode != value)
                {
                    _gradationOverlayBlendMode = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _gradationOverlayBlendMode;
            }
        }

        public int GradationOverlayControlPointCount
        {
            set
            {
                if (_gradationOverlayControlPointCount != value)
                {
                    _gradationOverlayControlPointCount = value;

                    // コントロールポイント数が変わったら数に応じて表示するカーブも切り替えます。
                    for (int i = 0; i < 4; ++i)
                    {
                        bool attrEnabled = i < _gradationOverlayControlPointCount;
                        GradationOverlayControlPointAttrArray[i].IsActiveAttribute = attrEnabled;
                        GradationOverlayColorAttrArray[i].IsActiveAttribute = attrEnabled;
                    }
                    NotifyChangeToScene_();
                    // カーブエディタのアトリビュート表示状態更新のためのイベントを発生させる
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneAttrModify);
                }
            }

            get
            {
                return _gradationOverlayControlPointCount;
            }
        }

        /// <summary>
        /// 指定したインデックスのグラデーションオーバーレイの色を取得します。
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public RGBAColor GetGradationOverlayColor(int index)
        {
            return _attribute.GradationOverlayColorAttrArray[index].GetAsRBGA();
        }

        /// <summary>
        /// 指定したインデックスのグラデーションオーバーレイの色を設定します。
        /// </summary>
        /// <param name="index"></param>
        /// <param name="color"></param>
        public void SetGradationOverlayColor(int index, RGBAColor color)
        {
            if (_attribute.GradationOverlayColorAttrArray[index].GetAsRBGA() != color)
            {
                _attribute.GradationOverlayColorAttrArray[index].SetValue(color);
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// 指定したインデックスのグラデーションオーバーレイののコントロールポイントを取得します。
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public float GetGradationOverlayControlPoint(int index)
        {
            return _attribute.GradationOverlayControlPointAttrArray[index].GetAsFloat();
        }

        /// <summary>
        /// 指定したインデックスのグラデーションオーバーレイののコントロールポイントを設定します。
        /// </summary>
        /// <param name="index"></param>
        /// <param name="value"></param>
        public void SetGradationOverlayControlPoint(int index, float value)
        {
            if (_attribute.GradationOverlayControlPointAttrArray[index].GetAsFloat() != value)
            {
                _attribute.GradationOverlayControlPointAttrArray[index].SetValue(value);
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// グラデーションオーバーレイの角度
        /// </summary>
        public float GradationOverlayAngle
        {
            set
            {
                _attribute.GradationOverlayAngleAttr.SetValue(value);
            }

            get
            {
                return _attribute.GradationOverlayAngleAttr.GetAsFloat();
            }
        }

        #endregion

        #region ドロップシャドウ(外側)

        /// <summary>
        /// ドロップシャドウ(外側)が有効かどうか
        /// </summary>
        public bool IsOuterShadowEnabled
        {
            set
            {
                if (_isOuterShadowEnabled != value)
                {
                    _isOuterShadowEnabled = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _isOuterShadowEnabled;
            }
        }

        /// <summary>
        /// ドロップシャドウ(外側)のブレンドモード
        /// </summary>
        public ProceduralShapeEffectBlendMode OuterShadowBlendMode
        {
            set
            {
                if (_outerShadowBlendMode != value)
                {
                    _outerShadowBlendMode = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _outerShadowBlendMode;
            }

        }

        /// <summary>
        /// ドロップシャドウ(外側)の色
        /// </summary>
        public RGBAColor OuterShadowColor
        {
            set
            {
                _attribute.OuterShadowColorAttr.SetValue(value);
            }

            get
            {
                return _attribute.OuterShadowColorAttr.GetAsRBGA();
            }

        }

        /// <summary>
        /// ドロップシャドウ(外側)の角度
        /// </summary>
        public float OuterShadowAngle
        {
            set
            {
                _attribute.OuterShadowAngleAttr.SetValue(value);
            }

            get
            {
                return _attribute.OuterShadowAngleAttr.GetAsFloat();
            }

        }

        /// <summary>
        /// ドロップシャドウ(外側)の距離
        /// </summary>
        public float OuterShadowDistance
        {
            set
            {
                _attribute.OuterShadowDistanceAttr.SetValue(value);
            }

            get
            {
                return _attribute.OuterShadowDistanceAttr.GetAsFloat();
            }
        }

        /// <summary>
        /// ドロップシャドウ(外側)のサイズ
        /// </summary>
        public float OuterShadowSize
        {
            set
            {
                _attribute.OuterShadowSizeAttr.SetValue(value);
            }

            get
            {
                return _attribute.OuterShadowSizeAttr.GetAsFloat();
            }
        }

        /// <summary>
        /// ドロップシャドウ(外側)の影の減衰タイプ
        /// </summary>
        public ProceduralShapeShadowType OuterShadowType
        {
            set
            {
                if (_outerShadowType != value)
                {
                    _outerShadowType = value;
                    NotifyChangeToScene_();
                }
            }

            get
            {
                return _outerShadowType;
            }
        }

        #endregion

        #region IAnmAttribute インターフェイス

        public IAnmAttribute IExpAttr
        {
            get
            {
                return _attribute.ExpAttr;
            }

        }

        public IAnmAttribute IRadiusAttr
        {
            get
            {
                return _attribute.RadiusAttr;
            }

        }

        public IAnmAttribute IInnerStrokeSizeAttr
        {
            get
            {
                return _attribute.InnerStrokeSizeAttr;
            }
        }

        public IAnmAttribute IInnerStrokeColorAttr
        {
            get
            {
                return _attribute.InnerStrokeColorAttr;
            }
        }

        public IAnmAttribute IInnerShadowColorAttr
        {
            get
            {
                return _attribute.InnerShadowColorAttr;
            }
        }

        public IAnmAttribute IInnerShadowAngleAttr
        {
            get
            {
                return _attribute.InnerShadowAngleAttr;
            }
        }

        public IAnmAttribute IInnerShadowDistanceAttr
        {
            get
            {
                return _attribute.InnerShadowDistanceAttr;
            }
        }

        public IAnmAttribute IInnerShadowSizeAttr
        {
            get
            {
                return _attribute.InnerShadowSizeAttr;
            }
        }

        public IAnmAttribute IColorOverlayColorAttr
        {
            get
            {
                return _attribute.ColorOverlayColorAttr;
            }
        }

        public IAnmAttribute[] IGradationOverlayColorAttrArray
        {
            get
            {
                return _attribute.GradationOverlayColorAttrArray;
            }
        }

        public IAnmAttribute[] IGradationOverlayControlPointAttrArray
        {
            get
            {
                return _attribute.GradationOverlayControlPointAttrArray;
            }
        }


        public IAnmAttribute IGradationOverlayAngleAttr
        {
            get
            {
                return _attribute.GradationOverlayAngleAttr;
            }
        }

        public IAnmAttribute IOuterShadowColorAttr
        {
            get
            {
                return _attribute.OuterShadowColorAttr;
            }
        }

        public IAnmAttribute IOuterShadowAngleAttr
        {
            get
            {
                return _attribute.OuterShadowAngleAttr;
            }
        }

        public IAnmAttribute IOuterShadowDistanceAttr
        {
            get
            {
                return _attribute.OuterShadowDistanceAttr;
            }
        }

        public IAnmAttribute IOuterShadowSizeAttr
        {
            get
            {
                return _attribute.OuterShadowSizeAttr;
            }
        }

        #endregion

        #endregion

        #region AnmAttribute インターフェイス

        public AnmAttribute ExpAttr
        {
            get
            {
                return _attribute.ExpAttr;
            }

        }

        public AnmAttribute RadiusAttr
        {
            get
            {
                return _attribute.RadiusAttr;
            }

        }

        public AnmAttribute InnerStrokeSizeAttr
        {
            get
            {
                return _attribute.InnerStrokeSizeAttr;
            }
        }

        public AnmAttribute InnerStrokeColorAttr
        {
            get
            {
                return _attribute.InnerStrokeColorAttr;
            }
        }

        public AnmAttribute InnerShadowColorAttr
        {
            get
            {
                return _attribute.InnerShadowColorAttr;
            }
        }

        public AnmAttribute InnerShadowAngleAttr
        {
            get
            {
                return _attribute.InnerShadowAngleAttr;
            }
        }

        public AnmAttribute InnerShadowDistanceAttr
        {
            get
            {
                return _attribute.InnerShadowDistanceAttr;
            }
        }

        public AnmAttribute InnerShadowSizeAttr
        {
            get
            {
                return _attribute.InnerShadowSizeAttr;
            }
        }

        public AnmAttribute ColorOverlayColorAttr
        {
            get
            {
                return _attribute.ColorOverlayColorAttr;
            }
        }

        public AnmAttribute[] GradationOverlayColorAttrArray
        {
            get
            {
                return _attribute.GradationOverlayColorAttrArray;
            }
        }

        public AnmAttribute[] GradationOverlayControlPointAttrArray
        {
            get
            {
                return _attribute.GradationOverlayControlPointAttrArray;
            }
        }

        public AnmAttribute GradationOverlayAngleAttr
        {
            get
            {
                return _attribute.GradationOverlayAngleAttr;
            }
        }

        public AnmAttribute OuterShadowColorAttr
        {
            get
            {
                return _attribute.OuterShadowColorAttr;
            }
        }

        public AnmAttribute OuterShadowAngleAttr
        {
            get
            {
                return _attribute.OuterShadowAngleAttr;
            }
        }

        public AnmAttribute OuterShadowDistanceAttr
        {
            get
            {
                return _attribute.OuterShadowDistanceAttr;
            }
        }

        public AnmAttribute OuterShadowSizeAttr
        {
            get
            {
                return _attribute.OuterShadowSizeAttr;
            }
        }

        #endregion

        internal class ProceduralShapeAttribute :
            AnmAttribute
        {
            static readonly AnmAttrDescripter[] SubAttrDescs =
            {
                new AnmAttrDescripter(AttributeType.Float, "exp", null, 2.0f),
                new AnmAttrDescripter(AttributeType.Float, "radius", null, 16.0f),
                new AnmAttrDescripter(AttributeType.Float, "innerStrokeSize", null, 3.0f),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "innerStrokeColor", null, RGBAColor.Black),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "innerShadowColor", null, RGBAColor.Black),
                new AnmAttrDescripter(AttributeType.Float, "innerShadowAngle", null, 120.0f),
                new AnmAttrDescripter(AttributeType.Float, "innerShadowDistance", null, 0.0f),
                new AnmAttrDescripter(AttributeType.Float, "innerShadowSize", null, 5.0f),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "colorOverlayColor", null, RGBAColor.Black),
                new AnmAttrDescripter(AttributeType.Float, "gradationOverlayControlPoint_1", null, 0.0f),
                new AnmAttrDescripter(AttributeType.Float, "gradationOverlayControlPoint_2", null, 1.0f),
                new AnmAttrDescripter(AttributeType.Float, "gradationOverlayControlPoint_3", null, 1.0f),
                new AnmAttrDescripter(AttributeType.Float, "gradationOverlayControlPoint_4", null, 1.0f),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "gradationOverlayColor_1", null, RGBAColor.Black),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "gradationOverlayColor_2", null, RGBAColor.White),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "gradationOverlayColor_3", null, RGBAColor.White),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "gradationOverlayColor_4", null, RGBAColor.White),
                new AnmAttrDescripter(AttributeType.Float, "gradationOverlayAngle", null, 0.0f),
                new AnmAttrDescripter(AttributeType.ByteRGBA4, "outerShadowColor", null, RGBAColor.Black),
                new AnmAttrDescripter(AttributeType.Float, "outerShadowAngle", null, 120.0f),
                new AnmAttrDescripter(AttributeType.Float, "outerShadowDistance", null, 5.0f),
                new AnmAttrDescripter(AttributeType.Float, "outerShadowSize", null, 5.0f),
            };

            public ProceduralShapeAttribute(ProceduralShape ownerProceduralShape)
                : base(ownerProceduralShape, SubAttrDescs,
                      new AnmAttrDescripter(AttributeType.Combined, "ProceduralShape", null, null))
            {
            }

            public AnmAttribute ExpAttr
            {
                get
                {
                    return FindAttributeByIdx(0);
                }
            }

            public AnmAttribute RadiusAttr
            {
                get
                {
                    return FindAttributeByIdx(1);
                }
            }
            public AnmAttribute InnerStrokeSizeAttr
            {
                get
                {
                    return FindAttributeByIdx(2);
                }
            }

            public AnmAttribute InnerStrokeColorAttr
            {
                get
                {
                    return FindAttributeByIdx(3);
                }
            }

            public AnmAttribute InnerShadowColorAttr
            {
                get
                {
                    return FindAttributeByIdx(4);
                }
            }

            public AnmAttribute InnerShadowAngleAttr
            {
                get
                {
                    return FindAttributeByIdx(5);
                }
            }

            public AnmAttribute InnerShadowDistanceAttr
            {
                get
                {
                    return FindAttributeByIdx(6);
                }
            }

            public AnmAttribute InnerShadowSizeAttr
            {
                get
                {
                    return FindAttributeByIdx(7);
                }
            }

            public AnmAttribute ColorOverlayColorAttr
            {
                get
                {
                    return FindAttributeByIdx(8);
                }
            }

            public AnmAttribute[] GradationOverlayControlPointAttrArray
            {
                get
                {
                    List<AnmAttribute> result = new List<AnmAttribute>();
                    for (int i = 0; i < ProceduralShapeConstants.GradationControlPointCount; ++i)
                    {
                        result.Add(FindAttributeByIdx(9 + i));
                    }

                    return result.ToArray();
                }
            }


            public AnmAttribute[] GradationOverlayColorAttrArray
            {
                get
                {
                    List<AnmAttribute> result = new List<AnmAttribute>();
                    for (int i = 0; i < ProceduralShapeConstants.GradationControlPointCount; ++i)
                    {
                        result.Add(FindAttributeByIdx(13 + i));
                    }

                    return result.ToArray();
                }
            }

            public AnmAttribute GradationOverlayAngleAttr
            {
                get
                {
                    return FindAttributeByIdx(17);
                }
            }

            public AnmAttribute OuterShadowColorAttr
            {
                get
                {
                    return FindAttributeByIdx(18);
                }
            }

            public AnmAttribute OuterShadowAngleAttr
            {
                get
                {
                    return FindAttributeByIdx(19);
                }
            }

            public AnmAttribute OuterShadowDistanceAttr
            {
                get
                {
                    return FindAttributeByIdx(20);
                }
            }

            public AnmAttribute OuterShadowSizeAttr
            {
                get
                {
                    return FindAttributeByIdx(21);
                }
            }

        }
    }
}
