﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using LECore.Structures.Core;

namespace LECore.Structures
{
    /// <summary>
    /// 部品ペインが持つ部品レイアウト情報です。
    /// </summary>
    internal class PartsLayout : LEDataNode,
        IPartsLayout, IPaneExParamater
    {
        private SubScene subScene = null;
        private string _partsLayoutName;
        private FVec3 _magnify;
        private FVec3 _sizeConstraint;
        private readonly FVec3 _initialPartsSize;
        private RectangleF _partsRect;

        //----------------------------------------------------------

        /// <summary>
        /// 静的コンストラクタ
        /// </summary>
        static PartsLayout()
        {
        }

        //static int _count = 0;
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsLayout(IPane owner, ISubScene partsSubScene, string partsLayoutName)
            : base(owner as Pane, "Parts")
        {
            //DbgConsole.WriteLine("PartsLayout[{0}]", _count++);
            Ensure.Argument.NotNull(partsSubScene);

            this._ParentMagnify = FVec3.One;
            this.OwnerPane = owner;

            (owner as Pane).InfluenceChildrenTransparency = true;
            (owner as Pane).BindPaneExData(this);

            this.PartsLayoutName = partsLayoutName;
            this.subScene = CloneSubScene(partsSubScene);
            this.subScene.SetPartsLayout(this);

            this.RawPartsPropaerties = new List<IPartsPropaerty>();

            // シーン内のすべてのペインを含む範囲をペインサイズとする。
            UpdatePartsPaneSize_();
            _initialPartsSize = this.OwnerPane.Size;

            this.Magnify = FVec3.One;
            this.SizeConstraint = FVec3.Empty;

            // 部品ペインのサイズ変更は禁止されます。
            (owner as Pane).PaneAttrRef.SizeAttr.IsActiveAttribute = false;

            foreach (var propertySource in this.subScene.IPartsSettings.OverridePorperties)
            {
                IPane propertyTargetPane = this.subScene.FindPaneByName(propertySource.TargetName);

                // 初期状態では無効なプロパティとして登録します。
                this.RawPartsPropaerties.Add(new PartsPropaerty(
                    propertySource.PaneKind, propertySource.TargetName, null, PartsPropertyUsageOptions.None));
            }

            CheckPartsPaneNameValidity();
        }

        /// <summary>
        /// 破棄
        /// </summary>
        public void Dispose()
        {
            //DbgConsole.WriteLine("PartsLayout[{0}]", _count--);
            if (this.subScene != null)
            {
                this.subScene.Dispose();
            }
        }

        //----------------------------------------------------------

        /// <summary>
        /// 部品レイアウト名です。
        /// </summary>
        public string PartsLayoutName
        {
            get
            {
                return _partsLayoutName;
            }

            internal set
            {
                if (_partsLayoutName != value)
                {
                    _partsLayoutName = value;
                }
            }
        }

        /// <summary>
        /// 上書きパラメータ列です。
        /// </summary>
        public IEnumerable<IPartsPropaerty> PartsPropaerties { get { return this.RawPartsPropaerties; } }

        /// <summary>
        /// 上書きパラメータ列の内部リストです。
        /// </summary>
        internal List<IPartsPropaerty> RawPartsPropaerties { get; set; }

        /// <summary>
        /// 部品サブシーンです。
        /// </summary>
        public ISubScene PartsSubScene
        {
            get { return this.subScene; }
        }

        /// <summary>
        /// 部品の倍率です。
        /// </summary>
        public FVec3 Magnify
        {
            get
            {
                return _magnify;
            }

            internal set
            {
                if (_magnify != value)
                {
                    _magnify = value;

                    FVec3 adjustmentMagnify = GetAdjustmentMagnify_();
                    adjustmentMagnify.Scale(this._ParentMagnify);

                    AdjustSize_(adjustmentMagnify);
                }
            }
        }

        /// <summary>
        /// 指定のサイズです。
        /// </summary>
        public FVec3 SizeConstraint
        {
            get
            {
                return _sizeConstraint;
            }

            internal set
            {
                if (_sizeConstraint != value)
                {
                    _sizeConstraint = value;

                    FVec3 adjustmentMagnify = GetAdjustmentMagnify_();
                    AdjustSize_(adjustmentMagnify);
                }
            }
        }

        /// <summary>
        /// 部品の元サイズです。
        /// </summary>
        public FVec3 InitialSize
        {
            get
            {
                return subScene.IPartsSettings.OriginalSize != FVec3.Empty ?
                    subScene.IPartsSettings.OriginalSize : _initialPartsSize;
            }
        }

        /// <summary>
        /// UV座標を列挙します。
        /// </summary>
        public IEnumerable<TexCoord4> TexCoords
        {
            get
            {
                yield break;
            }
        }

        /// <summary>
        /// 調整前のサイズ倍率です。
        /// </summary>
        private FVec3 _ParentMagnify { get; set; }

        /// <summary>
        /// 境界を書くかどうか
        /// </summary>
        public bool PaneBoundDrawEnabled
        {
            get { return false; }
        }

        //----------------------------------------------------------

        /// <summary>
        /// サイズ調整を行います。
        /// </summary>
        private void AdjustSize_(FVec3 magnify)
        {
            // すべての部品配下ペインについてサイズ調整を行います。
            foreach (Pane childPane in this.subScene.PaneSet)
            {
                FVec3 childMagnify = new FVec3(magnify);

                // 部品倍率モードが、境界から固定値を維持するモードならchildMagnifyを別に計算
                if (childPane.PartsMagnifyInfluence == PartsMagnifyInfluence.AdjustToPartsBound)
                {
                    var originalPartsSize = new FVec3(InitialSize);
                    var expectPartsSize = new FVec3(InitialSize);
                    expectPartsSize.Scale(childMagnify);

                    // 位置調整処理は廃止しました。以下の理由によります。
                    //
                    // 1.位置調整を行わない場合の挙動にも便利な利用ケースがあった。
                    //  - ペイン位置左寄せ、右寄せでしたペインを...
                    //  - 部品倍率に影響を受けない絵（★）にあわせて配置をする。
                    //  - 拡大した際に、位置調整が行われると、（★）との位置関係がずれる！
                    // 2.ランタイム側で対応が難しい。また、データ製作で必須となる状況が発覚しなかった。

                    // 中央位置以外は、増加サイズの半分を調整量とする。
                    // (片辺が固定となるので)
                    float wH = childPane.BasePosTypeH == HorizontalLocation.Center ? 1.0f : 0.5f;
                    float wV = childPane.BasePosTypeV == VerticalLocation.Center ? 1.0f : 0.5f;
                    var expW = (expectPartsSize.X - originalPartsSize.X) * wH;
                    var expH = (expectPartsSize.Y - originalPartsSize.Y) * wV;

                    // 差分が維持されるようなサイズになるようにスケール値を定める。
                    {
                        // 現在の部品サイズと、ペインのサイズ差分をもとめる
                        // 親ノードのスケールなどの影響を考慮したサイズで計算する。
                        // 現状、回転の影響は正しく扱えない。
                        SizeF localBounding;

                        // 部品ペインの場合は、初期サイズをもとに倍率を算出します。
                        if (childPane.PaneKind == PaneKind.Parts)
                        {
                            var childPartsLayout = childPane.IPartsLayout as PartsLayout;

                            bool isSizeConstraintEnabled = childPartsLayout.SizeConstraint != FVec3.Empty;

                            // 初期サイズ もしくは、サイズ制約で指定されたサイズをベースに、部品倍率をかけて変更前のサイズとします。
                            var sizeChildParts = isSizeConstraintEnabled ? childPartsLayout.SizeConstraint : childPartsLayout.InitialSize;
                            sizeChildParts.Scale(childPartsLayout.Magnify);
                            localBounding = new SizeF(sizeChildParts.X, sizeChildParts.Y);
                        }
                        else
                        {
                            // それ以外の場合は、ペインのワールド空間での大きさを計算します。
                            var rect = childPane.GetBoundingVolumeInWorld(false);
                            localBounding = new SizeF(rect.Width, rect.Height);
                        }

                        // 目標とする大きさになるように、スケールを計算します。
                        childMagnify = new FVec3(
                            (localBounding.Width + expW) / localBounding.Width,
                            (localBounding.Height + expH) / localBounding.Height,
                            magnify.Z);
                    }
                }

                // サイズ倍率を設定します。
                childPane.SizeMagnify = childMagnify;

                // 配下の部品ペインについても再帰的に処理
                if (childPane.PaneKind == PaneKind.Parts)
                {
                    if (childPane.PartsMagnifyInfluence != PartsMagnifyInfluence.None)
                    {
                        (childPane.IPartsLayout as PartsLayout)._ParentMagnify = childMagnify;

                        childMagnify.Scale((childPane.IPartsLayout as PartsLayout).GetAdjustmentMagnify_());
                        (childPane.IPartsLayout as PartsLayout).AdjustSize_(childMagnify);

                        childPane.SizeMagnify = FVec3.One;
                    }
                }
            }

            // 部品ペインのサイズを再計算する。
            UpdatePartsPaneSize_();

            this.NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        /// <summary>
        /// 部品ペインサイズを更新します。
        /// </summary>
        private void UpdatePartsPaneSize_()
        {
            this._partsRect = PartsLayoutHelper.CalcPartsPaneRectanbleFromPanes(subScene.IPaneArray);
            (this.OwnerPane as Pane).Size = new FVec3(PartsLayoutHelper.CalcPartsPaneSizeFromRectangle(this._partsRect));
        }

        /// <summary>
        /// 実際に補正に使われる部品倍率を取得します。
        /// </summary>
        private FVec3 GetAdjustmentMagnify_()
        {
            // サイズ調整倍率を求めます。
            // サイズ制約が有効なら、サイズ制約から算出されます。
            FVec3 adjustmentMagnify = this.Magnify;
            if (!SizeConstraint.IsZero)
            {
                // あきらめて適当な数値を返す。
                if (InitialSize == FVec3.Empty)
                {
                    return FVec3.One;
                }

                adjustmentMagnify = new FVec3(
                    this.SizeConstraint.X / InitialSize.X,
                    this.SizeConstraint.Y / InitialSize.Y,
                    this.SizeConstraint.Z);
            }

            return adjustmentMagnify;
        }

        /// <summary>
        /// ファイル保存向けに部品倍率を取得します。
        /// </summary>
        internal FVec3 GetAdjustmentMagnifyForFileSave()
        {
            try
            {
                this.OwnerPane.OwnerSubScene.BeginMassiveModify();
                return GetAdjustmentMagnify_();
            }
            finally
            {
                this.OwnerPane.OwnerSubScene.EndMassiveModify(true, false);
            }
        }

        /// <summary>
        /// ペイン拡張情報の上書きのための、コピーオプションフラグを設定する。
        /// </summary>
        static void SetParamaterCopyOptionForPaneEx_(PartsPropertyUsageOptions propertyUsage, PaneParamaterCopyOption dstCopyOption)
        {
            // ペイン拡張情報が上書きされているかどうか
            if ((propertyUsage & PartsPropertyUsageOptions.UsePaneExParamater) != 0)
            {
                // 部分上書きが指定されているかどうか
                bool useTextOnly = (propertyUsage & PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText) != 0;
                bool useColorBlendOnly = (propertyUsage & PartsPropertyUsageOptions.UsePaneExParamaterColorBlend) != 0;
                bool useTextureOnly = (propertyUsage & PartsPropertyUsageOptions.UsePaneExParameterTexture) != 0;
                if (useTextOnly || useColorBlendOnly || useTextureOnly)
                {
                    if (useTextOnly) { dstCopyOption.EnableOption(ParamaterKind.ContentsText); }
                    if (useColorBlendOnly) { dstCopyOption.EnableOption(ParamaterKind.Material_ColorBlend); }
                    if (useTextureOnly) { dstCopyOption.EnableOption(ParamaterKind.Material_Textures); }
                }
                else
                {
                    // 全部上書きする
                    dstCopyOption.EnableOption(
                        ParamaterKind.PictureAll |
                        ParamaterKind.TextBoxAll |
                        ParamaterKind.WindowAll |
                        ParamaterKind.MaterialAll |
                        ParamaterKind.MaterialDetailsAll |
                        ParamaterKind.PartsAll |
                        ParamaterKind.AllVtxColor);
                }
            }
            else
            {
                // 要修正：仕様調整によって、UseTextBoxText と扱いを統合すればこの記述は消去できる。
                // (今は、UsePaneExParamater を指定せずに、UsePaneExParamaterColorBlend だけ単独で指定できる)
                dstCopyOption.SetOption(ParamaterKind.Material_ColorBlend, (propertyUsage & PartsPropertyUsageOptions.UsePaneExParamaterColorBlend) != 0);
                dstCopyOption.SetOption(ParamaterKind.Material_Textures, (propertyUsage & PartsPropertyUsageOptions.UsePaneExParameterTexture) != 0);
            }
        }

        /// <summary>
        /// ペイン基本情報の上書きのための、コピーオプションフラグを設定する。
        /// </summary>
        static void SetParamaterCopyOptionForPaneBasic_(PartsPropertyUsageOptions propertyUsage, PaneParamaterCopyOption dstCopyOption)
        {
            dstCopyOption.SetOption(ParamaterKind.Position, (propertyUsage & PartsPropertyUsageOptions.UsePanePosition) != 0);
            dstCopyOption.SetOption(ParamaterKind.Size, (propertyUsage & PartsPropertyUsageOptions.UsePaneSize) != 0);
            dstCopyOption.SetOption(ParamaterKind.Scale, (propertyUsage & PartsPropertyUsageOptions.UsePaneScale) != 0);
            dstCopyOption.SetOption(ParamaterKind.Rotate, (propertyUsage & PartsPropertyUsageOptions.UsePaneRotate) != 0);
            dstCopyOption.SetOption(ParamaterKind.Transparency, (propertyUsage & PartsPropertyUsageOptions.UsePaneTransparency) != 0);
            dstCopyOption.SetOption(ParamaterKind.PaneBasicFlags, (propertyUsage & PartsPropertyUsageOptions.UsePaneVisibility) != 0);
            dstCopyOption.SetOption(ParamaterKind.UserData, (propertyUsage & PartsPropertyUsageOptions.UsePaneBasicUserData) != 0);
            dstCopyOption.SetOption(ParamaterKind.UserDataEx, (propertyUsage & PartsPropertyUsageOptions.UsePaneExUserData) != 0);
        }

        /// <summary>
        /// コピーオプションを取得します。
        /// </summary>
        static PaneParamaterCopyOption GetParamaterCopyOption_(PartsPropertyUsageOptions propertyUsage)
        {
            PaneParamaterCopyOption dstCopyOption = new PaneParamaterCopyOption();
            dstCopyOption.DisableOption(ParamaterKind.All);

            SetParamaterCopyOptionForPaneEx_(propertyUsage, dstCopyOption);
            SetParamaterCopyOptionForPaneBasic_(propertyUsage, dstCopyOption);

            return dstCopyOption;
        }

        /// <summary>
        /// サブシーンにハンドラを設定。
        /// </summary>
        private void InitSubSceneHandler_(SubScene subScene)
        {
            subScene.OnModified += this.OnPartsSubSceneModified;
        }

        /// <summary>
        /// 部品ペイン名の有効性をチェックする。
        /// </summary>
        public void CheckPartsPaneNameValidity()
        {
            this.WarningMsg = string.Empty;
            if (SubSceneHelper.FindPaneByName(this.PartsSubScene, this.OwnerPane.PaneName) != null)
            {
                string msg = LECoreStringResMgr.Get("LECORE_ERROR_PARTSPANE_NAMECONFLICT", this.OwnerName, this.PartsLayoutName);
                this.WarningMsg = string.Format(msg);
            }
        }

        /// <summary>
        /// 他の部品ペインで復元します。
        /// </summary>
        /// <param name="source"></param>
        public void Restore(PartsLayout source, bool loadTexure = true)
        {
            this.PartsLayoutName = source.PartsLayoutName;

            if (this.subScene != null)
            {
                this.subScene.OnModified -= this.OnPartsSubSceneModified;
                this.subScene.Dispose();
            }

            if (source.subScene != null)
            {
                source.subScene.OnModified -= source.OnPartsSubSceneModified;
            }

            this.subScene = source.subScene;
            this.subScene.SetPartsLayout(this);
            this.InitSubSceneHandler_(this.subScene);

            // プロパティの内容をコピー
            {
                RestoreProperties_(this, source);

                List<IPartsPropaerty> currentEnabledProperties = new List<IPartsPropaerty>(this.RawPartsPropaerties.Where((prop) => prop.Paramater != null));

                this.RawPartsPropaerties.Clear();
                this.RawPartsPropaerties.AddRange(source.RawPartsPropaerties);
                this.SetEnabledProperties(currentEnabledProperties, false);
            }

            // サイズ再計算
            UpdatePartsPaneSize_();

            this.SizeConstraint = source.SizeConstraint;
            this.Magnify = source.Magnify;
        }

        /// <summary>
        /// プロパティの上書き部分以外をソースで更新します。
        /// </summary>
        private void RestoreProperties_(PartsLayout dstParts, PartsLayout srcParts)
        {
            foreach (var dstProp in dstParts.PartsPropaerties.Where((prop) => prop.Option != PartsPropertyUsageOptions.None))
            {
                IPane dstPane = dstProp.Paramater.OwnerPane;
                if (dstPane == null)
                {
                    continue;
                }

                var srcPane = srcParts.PartsSubScene.FindPaneByName(dstProp.TargetName);
                if (srcPane == null)
                {
                    continue;
                }

                // 上書きするユーザーデータをコピー
                List<UserDataElement> overwriteUserDataSet = new List<UserDataElement>();
                foreach (UserDataElement element in dstPane.IUserDataHolder.UserDataElementSet)
                {
                    if (element.Overwrite)
                    {
                        overwriteUserDataSet.Add(element);
                    }
                }

                // 上書きオプションを取得
                ParamaterKind overwriteOption = GetParamaterCopyOption_(dstProp.Option).Option;

                // 上書きする部分以外をコピーするオプションでソースの値に復元
                PaneParamaterCopyOption option = new PaneParamaterCopyOption();
                option.DisableOption(overwriteOption);
                option.EnableOption(ParamaterKind.UserDataEx);
                PaneParamaterPaster.PasteParamaters(option, dstPane, srcPane);

                // 拡張ユーザー情報を上書き
                if ((overwriteOption & ParamaterKind.UserDataEx) != 0)
                {
                    OverwriteExtUserData_((Pane)dstPane, overwriteUserDataSet);
                }

                // 部品ペインの上書きは再帰的に処理を行う
                if (dstProp.PaneKind == PaneKind.Parts)
                {
                    RestoreProperties_(dstPane.IPartsLayout as PartsLayout, srcPane.IPartsLayout as PartsLayout);
                }
            }
        }

        /// <summary>
        /// 拡張ユーザー情報を上書きします。
        /// </summary>
        /// <param name="dstPane">上書き先のペイン</param>
        /// <param name="overwriteUserDataSet">上書くユーザーデータ</param>
        private void OverwriteExtUserData_(Pane dstPane, List<UserDataElement> overwriteUserDataSet)
        {
            UserDataHolder dstUserDataHolder = dstPane.UserDataHolder;

            foreach (IUserDataElement srcElement in overwriteUserDataSet)
            {
                if (srcElement.Overwrite)
                {
                    IUserDataElement dstElement = UserDataElementHelper.FindByName(srcElement.Name, dstUserDataHolder.UserDataElementSet);

                    if (dstElement == null)
                    {
                        // ユーザーデータ項目を追加する
                        // TODO: アニメーション対応
                        dstUserDataHolder.AddUserDataElement(srcElement.Name, srcElement.UserDataKind, true, srcElement.Value);
                    }
                    else
                    {
                        // 置き換え前の項目番号を取得
                        int elementIndex = Array.IndexOf(dstUserDataHolder.UserDataElementSet, dstElement);
                        int attributeIndex = dstUserDataHolder.SubAttrArray.IndexOf(dstElement.IAnmAttribute);

                        // ユーザーデータ項目を置き換える
                        // TODO: アニメーション対応
                        dstUserDataHolder.RemoveUserDataElement(srcElement.Name);
                        dstUserDataHolder.InsertUserDataElement(srcElement.Name, srcElement.UserDataKind, true, srcElement.Value, elementIndex, attributeIndex);
                    }
                }
            }
        }

        /// <summary>
        /// 部品プロパティに変更があったことを通知する。
        /// </summary>
        public void NotifyPartsPropertyChanged()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PartsLayout);
        }

        /// <summary>
        /// 有効なプロパティだけを設定します。
        /// </summary>
        public void SetEnabledProperties(IEnumerable<IPartsPropaerty> currentEnabledProperties, bool loadTexure = true)
        {
            // プロパティの状態をコピーする。
            SetPropertiesByCondition_(currentEnabledProperties, (prop) => prop.Option != PartsPropertyUsageOptions.None, loadTexure);
        }

        /// <summary>
        /// すべてのプロパティを設定します。
        /// </summary>
        public void SetAllProperties(IEnumerable<IPartsPropaerty> currentEnabledProperties, bool loadTexure = true)
        {
            // プロパティの状態をコピーする。
            SetPropertiesByCondition_(currentEnabledProperties, (prop) => true, loadTexure);
        }

        /// <summary>
        /// プロパティを設定します。
        /// </summary>
        void SetPropertiesByCondition_(IEnumerable<IPartsPropaerty> currentEnabledProperties, Predicate<IPartsPropaerty> condition, bool loadTexure = true)
        {
            // プロパティの状態をコピーする。
            foreach (var property in currentEnabledProperties.Where((prop) => condition(prop)))
            {
                // まず、新しいフラグ状態にリセットします。（順番が重要です。新たに上書きが有効になる場合、リセット後に値をコピーする必要があります）
                // loadTexureがfalseの時は、上書きされたテクスチャを自動ロードしないようにするため、参照テクスチャを自動的に読み込みません。
                ModifyOverwriteParamaters(property, property.Option, loadTexure);

                // 次に、値をコピーします。
                IPane propertyTargetPane = this.subScene.FindPaneByName(property.TargetName);
                if (propertyTargetPane != null && property.Paramater != null)
                {
                    IPane srcPane = property.Paramater.OwnerPane;
                    PaneParamaterPaster.PasteParamaters(GetParamaterCopyOption_(property.Option), propertyTargetPane, srcPane);
                    // loadTexureがfalseの時は上書きされたテクスチャを読み込みます(おそらく不要だが念のため)
                    if (!loadTexure && property.PaneKind == PaneKind.Picture)
                    {
                        var picture = (Picture)property.Paramater;
                        foreach (var texMap in picture.IMaterialTexMapSet)
                        {
                            var tex = this.OwnerPane.OwnerSubScene.ITextureMgr.FindITextureImageByName(texMap.TexImgName);
                            if (tex != null)
                            {
                                propertyTargetPane.OwnerSubScene.ITextureMgr.TryRegisterTextureImage(tex);
                            }
                        }
                        NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PartsLayoutTexResAdjusted);
                    }
                }
            }
        }

        /// <summary>
        /// 親部品が上書きしているプロパティを探します。
        /// </summary>
        public IPartsPropaerty GetParentOverwriteProperty(IPartsLayout parentPartsLayout, string targetName)
        {
            // 親部品があるか
            if (parentPartsLayout == null)
            {
                return null;
            }

            // 親の部品ペインを取ってくる
            IPane originalPartsPane = parentPartsLayout.GetOriginalPane(this.OwnerPane.PaneName);
            if (originalPartsPane == null)
            {
                return null;
            }

            // 親が自分自身(targetName)を上書きしていないか探す
            IPartsPropaerty originalProp = originalPartsPane.IPartsLayout.FindPartsPropertyByName(targetName);
            return originalProp;
        }

        /// <summary>
        /// プロパティを設定します。
        /// </summary>
        public void SetPropertyToPane(IPane targetPane, IPartsPropaerty parentOriginalProp, PartsPropertyUsageOptions option)
        {
            // 上書き前のパラメータが元々存在しないか、発見できない場合は処理をしません。
            if (parentOriginalProp == null ||
                parentOriginalProp.Paramater == null ||
                parentOriginalProp.Paramater.OwnerPane == null)
            {
                return;
            }

            // 上書きしていた場合、本当の初期状態ではなく、親が上書きしている初期状態に戻したい！
            // 今回新規に設定された項目でかつ親が上書きしている項目
            option = option & parentOriginalProp.Option;

            // 親の上書き値をペインに貼り付けてリセット
            PaneParamaterPaster.PasteParamaters(GetParamaterCopyOption_(option), targetPane, parentOriginalProp.Paramater.OwnerPane);
        }

        /// <summary>
        /// 部品ペインを上書きします。
        /// </summary>
        public void OverrideChildePartsPane(
            string paneName, IPaneExParamater newPartsLayoutName)
        {
            var partsProp = this.RawPartsPropaerties.Find(
                (prop) =>
                    prop.PaneKind == PaneKind.Parts &&
                    prop.TargetName == paneName) as PartsPropaerty;

            if (partsProp == null)
            {
                return;
            }

            if (!object.ReferenceEquals(partsProp.Paramater, newPartsLayoutName))
            {
                partsProp.Paramater = newPartsLayoutName;
                NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PartsLayout);
            }
        }

        /// <summary>
        /// テクスチャだけ上書きの際に TexMapNum が変わっていたら上書きを解除します。
        /// </summary>
        private PartsPropertyUsageOptions ResetTexMapNumOfTextureOverwriting(IPartsPropaerty property, PartsPropertyUsageOptions option)
        {
            if (CheckTexMapChangedWhenTextureOverwriting(property))
            {
                option &= ~PartsPropertyUsageOptions.UsePaneExParamater;
                option &= ~PartsPropertyUsageOptions.UsePaneExParameterTexture;
                string msg = LECoreStringResMgr.Get("LECORE_ERROR_TEXTURE_OVERWRITING_FAILED", this.PartsLayoutName, property.TargetName);
                LayoutEditorCore.MsgReporter.ReportError("", msg);
            }
            return option;
        }

        /// <summary>
        /// テクスチャだけ上書きの際に TexMapNum が変わっているかどうかをチェックします。
        /// </summary>
        private bool CheckTexMapChangedWhenTextureOverwriting(IPartsPropaerty property)
        {
            IPane targetPane = this.GetOriginalPane(property.TargetName);
            if (targetPane.PaneKind == PaneKind.Picture)
            {
                return IsTexMapNumChanged_(property.Paramater, targetPane.IPicture);
            }
            else if (targetPane.PaneKind == PaneKind.Window)
            {
                return IsTexMapNumChanged_(property.Paramater, targetPane.ILEWindow);
            }
            return false;
        }

        /// <summary>
        /// 上書きプロパティを変更します。
        /// </summary>
        public void ModifyOverwriteParamaters(IPartsPropaerty property, PartsPropertyUsageOptions usageOption, bool loadTexure = true)
        {
            var targetSubScene = this.OwnerPane != null ? this.OwnerPane.OwnerSubScene as SubScene : null;
            try
            {
                if (targetSubScene != null) { targetSubScene.BeginMassiveModify(); }

                // テクスチャだけ上書きの際に TexMapNum が変わっていた場合に、ペイン固有の上書きをリセットします。
                if ((usageOption & PartsPropertyUsageOptions.UsePaneExParameterTexture) != 0)
                {
                    usageOption = ResetTexMapNumOfTextureOverwriting(property, usageOption);
                }

                DoModifyOverwriteParamaters_(property, usageOption, loadTexure);
            }
            finally
            {
                if (targetSubScene != null) { targetSubScene.EndMassiveModify(); }
            }
        }

        /// <summary>
        /// 新たにフラグが On になったか
        /// </summary>
        private bool IsFlagTrigOn_(PartsPropertyUsageOptions currentVal, PartsPropertyUsageOptions newValueVal, PartsPropertyUsageOptions flag)
        {
            return (currentVal & flag) == 0 && (newValueVal & flag) != 0;
        }

        /// <summary>
        /// 新たにフラグが Off になったか
        /// </summary>
        private bool FlagTrigOff_(PartsPropertyUsageOptions currentVal, PartsPropertyUsageOptions newValueVal, PartsPropertyUsageOptions flag)
        {
            return (currentVal & flag) != 0 && (newValueVal & flag) == 0;
        }

        /// <summary>
        /// テクスチャだけ上書きの際に TexMapNum が変わっているかどうかを取得します。
        /// </summary>
        private bool IsTexMapNumChanged_(IPaneExParamater parameter, IPaneExParamater targetParameter)
        {
            IRevHWMaterialHolder materialHolder = parameter as IRevHWMaterialHolder;
            IRevHWMaterialHolder targetMaterialHolder = targetParameter as IRevHWMaterialHolder;
            if (materialHolder == null || targetMaterialHolder == null)
            {
                return false;
            }
            IRevHWMaterial[] materials = materialHolder.IRevHWMaterial;
            IRevHWMaterial[] targetMaterials = targetMaterialHolder.IRevHWMaterial;
            if (materials == null || targetMaterials == null || materials.Length != targetMaterials.Length)
            {
                return false;
            }
            for (int i = 0; i < materials.Length; i++)
            {
                if (materials[i].IMaterialTexMapSet.Length != targetMaterials[i].IMaterialTexMapSet.Length)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 上書きプロパティを変更します。
        /// </summary>
        private void DoModifyOverwriteParamaters_(IPartsPropaerty property, PartsPropertyUsageOptions usageOption, bool loadTexure = true)
        {
            PaneKind paneKind = property.PaneKind;
            string targetName = property.TargetName;
            var targetProperty = this.PartsPropaerties.FirstOrDefault((prop) => prop.TargetName == targetName && prop.PaneKind == paneKind);
            if (targetProperty == null)
            {
                return;
            }

            var pane = this.PartsSubScene.FindPaneByName(targetName);
            if (pane == null || pane.PaneKind != paneKind)
            {
                return;
            }

            // 設定ずみなら、リセット処理を試行します。
            if (targetProperty.Paramater != null)
            {
                // 上書き対象を元の値にリセットします。
                if ((usageOption & PartsPropertyUsageOptions.UsePaneExParamater) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneExParamater) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.WindowAll | ParamaterKind.TextBoxAll | ParamaterKind.PictureAll | ParamaterKind.MaterialDetailsAll | ParamaterKind.PartsAll);
                }

                // 新たに PaneExParamater の部分的な上書きが有効になった場合、
                if ((targetProperty.Option & PartsPropertyUsageOptions.UsePaneExParamater) != 0)
                {
                    if (IsFlagTrigOn_(targetProperty.Option, usageOption, PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText) ||
                        IsFlagTrigOn_(targetProperty.Option, usageOption, PartsPropertyUsageOptions.UsePaneExParamaterColorBlend) ||
                        IsFlagTrigOn_(targetProperty.Option, usageOption, PartsPropertyUsageOptions.UsePaneExParameterTexture))
                    {
                        // 部分的な更新が有効となっている部分以外をリセットする。
                        ParamaterKind textResetParam = ParamaterKind.TextBoxAll;
                        if((usageOption & PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText) != 0)
                        {
                            textResetParam &= ~ParamaterKind.ContentsText;
                        }

                        ParamaterKind matResetParam = ParamaterKind.MaterialDetailsAll;
                        if ((usageOption & PartsPropertyUsageOptions.UsePaneExParamaterColorBlend) != 0)
                        {
                            matResetParam &= ~ParamaterKind.Material_ColorBlend;
                        }
                        if ((usageOption & PartsPropertyUsageOptions.UsePaneExParameterTexture) != 0)
                        {
                            matResetParam &= ~ParamaterKind.Material_Textures;
                        }

                        TryRestoreOriginalValue_(paneKind, targetProperty, textResetParam | ParamaterKind.WindowAll| ParamaterKind.PictureAll | matResetParam);
                    }
                }

                // テキスト、IDだけ OFF
                if ((usageOption & PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.ContentsText);
                }

                // カラー補間
                if ((usageOption & PartsPropertyUsageOptions.UsePaneExParamaterColorBlend) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneExParamaterColorBlend) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.Material_ColorBlend);
                }

                // テクスチャだけ上書き
                if ((usageOption & PartsPropertyUsageOptions.UsePaneExParameterTexture) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneExParameterTexture) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.Material_Textures);
                }

                // pos
                if ((usageOption & PartsPropertyUsageOptions.UsePanePosition) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePanePosition) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.Position);
                }

                // size
                if ((usageOption & PartsPropertyUsageOptions.UsePaneSize) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneSize) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.Size);
                }

                // scale
                if ((usageOption & PartsPropertyUsageOptions.UsePaneScale) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneScale) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.Scale);
                }

                // rot
                if ((usageOption & PartsPropertyUsageOptions.UsePaneRotate) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneRotate) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.Rotate);
                }

                // ペイン基本
                if ((usageOption & PartsPropertyUsageOptions.UsePaneTransparency) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneTransparency) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.Transparency);
                }

                // ペイン基本
                if ((usageOption & PartsPropertyUsageOptions.UsePaneVisibility) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneVisibility) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.PaneBasicFlags);
                }

                // ペイン基本
                if ((usageOption & PartsPropertyUsageOptions.UsePaneBasicUserData) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneBasicUserData) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.UserData);
                }

                // ユーザデータ
                if ((usageOption & PartsPropertyUsageOptions.UsePaneExUserData) == 0 &&
                    (targetProperty.Option & PartsPropertyUsageOptions.UsePaneExUserData) != 0)
                {
                    TryRestoreOriginalValue_(paneKind, targetProperty, ParamaterKind.UserDataEx);
                }
            }

            // パラメータを設定します。
            (targetProperty as PartsPropaerty).Paramater = (usageOption == PartsPropertyUsageOptions.None) ? null : pane.IPaneExParamater;
            (targetProperty as PartsPropaerty).Option = usageOption;

            // 使っているリソースを持ち主シーンにも登録する。
            ISubScene ownerSubScene = this.GetOwnerSubScene();
            if (PartsPropertyUsageOptionsHelper.IsPaneExDataOverwrote(usageOption) && ownerSubScene != null)
            {
                // Font登録
                try
                {
                    ownerSubScene.BeginMassiveModify();

                    foreach (var usedFont in this.PartsSubScene.ILEFontManager.ILEFontSet.Where((font) => pane.UsesFont(font)))
                    {
                        ownerSubScene.ILEFontManager.TryCreateFont(usedFont);
                    }
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PartsLayoutFntResAdjusted);

                    // Texture登録
                    if (loadTexure)
                    {
                        foreach (var usedTex in this.PartsSubScene.ITextureMgr.ITextureImageSet.Where((tex) => tex.SourceType == LECoreInterface.TextureSourceType.Dynamic ? false : pane.UsesTextureImage(tex.Name)))
                        {
                            ownerSubScene.ITextureMgr.TryRegisterTextureImage(usedTex);
                        }

                        NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PartsLayoutTexResAdjusted);
                    }
                }
                finally
                {
                    ownerSubScene.EndMassiveModify();
                }
            }

            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PartsLayout);
        }

        /// <summary>
        /// 元の値への復元を試行します。
        /// </summary>
        private void TryRestoreOriginalValue_(
            PaneKind paneKind, IPartsPropaerty targetProperty, ParamaterKind copyOption)
        {
            if (targetProperty.Paramater == null)
            {
                return;
            }

            IPane targetPane = targetProperty.Paramater.OwnerPane;
            var originalPane = this.GetOriginalPane(targetProperty.TargetName);
            if (originalPane != null && originalPane.PaneKind == paneKind)
            {
                PaneParamaterCopyOption option = new PaneParamaterCopyOption();
                option.DisableOption(ParamaterKind.All);
                option.EnableOption(copyOption);
                PaneParamaterPaster.PasteParamaters(option, targetPane, originalPane);
            }
        }

        /// <summary>
        /// 部品レイアウトを部品ペイン毎に複製します。
        /// </summary>
        private SubScene CloneSubScene(ISubScene sourceSubScene)
        {
            SubScene dstSubScene = SubSceneHelper.CloneSubSceneWithoutAnim(sourceSubScene);

            // マテリアル名の変更。
            DoUpdateMatarialName(dstSubScene);

            this.InitSubSceneHandler_(dstSubScene);

            return dstSubScene;
        }

        /// <summary>
        /// </summary>
        static private bool IsFontAdjustedEvent_(SceneModifyEventArgs.Kind kind)
        {
            return
                kind == SceneModifyEventArgs.Kind.FontManager ||
                kind == SceneModifyEventArgs.Kind.PartsLayoutFntResAdjusted;
        }

        /// <summary>
        /// </summary>
        static private bool IsTextureAdjustedEvent_(SceneModifyEventArgs.Kind kind)
        {
            return
                kind == SceneModifyEventArgs.Kind.TextureManager ||
                kind == SceneModifyEventArgs.Kind.PartsLayoutTexResAdjusted;
        }

        /// <summary>
        /// 配下のペイン変更通知ハンドラです。
        /// </summary>
        private void OnPartsSubSceneModified(ISubScene sender, SceneModifyEventArgs.Kind kind)
        {
            if (IsFontAdjustedEvent_(kind))
            {
                var partsFontMgr = this.subScene?.ILEFontManager;
                var selfFontMgr = this.OwnerPane?.OwnerSubScene?.ILEFontManager;

                if(partsFontMgr != null && selfFontMgr != null)
                {
                    bool isModified = false;
                    // 利用されているフォントについて自分にも登録します。
                    foreach (var partsFont in partsFontMgr.ILEFontSet)
                    {
                        if (PaneHelper.UsesFont(this.OwnerPane, partsFont))
                        {
                            (selfFontMgr as FontManager).CreateLEFont(partsFont.FontName, partsFont.FontPath, partsFont.FontSettings);
                            isModified |= true;
                        }
                    }

                    // 変更イベントを発生します。
                    // 変更イベント は 親部品の OnPartsSubSceneModified でキャッチされ階層を伝播していきます。
                    if (isModified)
                    {
                        (selfFontMgr as FontManager).RaiseChangeEvent();
                    }
                }
            }
            else if (IsTextureAdjustedEvent_(kind))
            {
                var partsTextureMgr = this.subScene?.ITextureMgr;
                var selfTextureMgr = this.OwnerPane?.OwnerSubScene?.ITextureMgr;

                // 利用されているフォントについて自分にも登録します。
                if (partsTextureMgr != null && selfTextureMgr != null)
                {
                    bool isModified = false;
                    foreach (var textureImage in partsTextureMgr.ITextureImageSet)
                    {
                        if (PaneHelper.UsesTextureImage(this.OwnerPane, textureImage.Name))
                        {
                            (selfTextureMgr as TextureMgr).TryRegisterTextureImage(textureImage);
                            isModified |= true;
                        }
                    }

                    // 変更イベントを発生します。
                    // 変更イベント は 親部品の OnPartsSubSceneModified でキャッチされ階層を伝播していきます。
                    if (isModified)
                    {
                        (selfTextureMgr as TextureMgr).RaiseChangeEvent();
                    }
                }
            }
            else if (kind == SceneModifyEventArgs.Kind.PartsLayout)
            {
                // 配下の部品ペインで、上書き設定が変更された場合など。
                // OverridePartsPaneイベントを再送して親シーンに伝搬します。
                this.NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PartsLayout);
            }
            else if (kind == SceneModifyEventArgs.Kind.MaterialEditMode)
            {
                // 配下の部品ペインで、マテリアル（テクスチャコンバイナ）の上書き設定が変更された場合など。
                this.NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.MaterialEditMode);
            }
            else
            {
                this.NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
            }
        }

        /// <summary>
        /// マテリアル名が重複しないように適当に修正します。
        /// </summary>
        private void DoUpdateMatarialName(ISubScene dstSubScene)
        {
            // マテリアル名が重複しないように適当に修正する
            // 暫定対処が不要になったので、修正してある名前をもとの名前に修復します。
            foreach (var pane in dstSubScene.IPaneArray)
            {
                if (pane.IPaneExParamater != null)
                {
                    pane.IPaneExParamater.UpdateMatarialName();
                }
            }

            if (this.PartsPropaerties != null)
            {
                foreach (IPartsPropaerty partsPropaerty in this.PartsPropaerties)
                {
                    if (partsPropaerty.Paramater != null)
                    {
                        partsPropaerty.Paramater.UpdateMatarialName();
                    }
                }
            }
        }

        //----------------------------------------------------------
        // IPaneExParamater
        //----------------------------------------------------------

        /// <summary>
        ///
        /// </summary>
        public IPane OwnerPane { get; internal set; }

        /// <summary>
        /// マテリアル名が重複しないように適当に修正します。
        /// </summary>
        public void UpdateMatarialName()
        {
            CheckPartsPaneNameValidity();
            this.DoUpdateMatarialName(this.subScene);
        }

        /// <summary>
        /// フォントを登録します。
        /// </summary>
        private void RegisterFontToPartsSubScene_(ILEFont newFont)
        {
            if (subScene.ILEFontManager.FindFontByName(newFont.FontName) == null)
            {
                (subScene.ILEFontManager as FontManager).CreateLEFont(newFont.FontName, newFont.FontPath, newFont.FontSettings);

                // 配下のシーンにも再帰的に登録します。
                foreach (IPane childScenePane in this.subScene.PaneSet)
                {
                    if (childScenePane.PaneKind == PaneKind.Parts)
                    {
                        (childScenePane.IPartsLayout as PartsLayout).RegisterFontToPartsSubScene_(newFont);
                    }
                }
            }

            // フォント参照をリセットしておく。
            foreach (var textPorp in this.PartsPropaerties.Where((prop) =>
                prop.PaneKind == PaneKind.Textbox && prop.Paramater is TextBox))
            {
                TextBox targetTextBox = textPorp.Paramater as TextBox;
                if (targetTextBox.FontName == newFont.FontName)
                {
                    targetTextBox.UpdateFontReference();
                }
            }
        }

        /// <summary>
        /// テクスチャを登録します。
        /// </summary>
        private void RegisterTextureToPartsSubScene_(ITextureImage newTexture)
        {
            if (subScene.ILEFontManager.FindFontByName(newTexture.Name) == null)
            {
                subScene.ITextureMgr.SyncTextureImageNoEvent(newTexture);

                // 配下のシーンにも再帰的に登録します。
                foreach (IPane childScenePane in this.subScene.PaneSet)
                {
                    if (childScenePane.PaneKind == PaneKind.Parts)
                    {
                        (childScenePane.IPartsLayout as PartsLayout).RegisterTextureToPartsSubScene_(newTexture);
                    }
                }
            }
        }

        /// <summary>
        /// シーン登録時の初期化
        /// </summary>
        public void OnJoinSceneInitialize()
        {
            UpdateMatarialName();

            ISubScene ownerSubScene = this.GetOwnerSubScene();
            if (ownerSubScene != null)
            {
                subScene.BeginMassiveModify();

                // 部品ペインの親シーンが上書きのために持っているフォントも登録しておく
                {
                    foreach (var font in ownerSubScene.ILEFontManager.ILEFontSet)
                    {
                        if (subScene.ILEFontManager.FindFontByName(font.FontName) == null)
                        {
                            RegisterFontToPartsSubScene_(font);
                        }
                    }
                }

                // 部品ペインの親シーンのテクスチャを登録しておく。
                try
                {
                    (subScene.ITextureMgr as TextureMgr).BeginUpdate();
                    foreach (var texture in ownerSubScene.ITextureMgr.ITextureImageSet)
                    {
                        if (subScene.ITextureMgr.FindITextureImageByName(texture.Name) == null)
                        {
                            RegisterTextureToPartsSubScene_(texture);
                        }
                    }
                }
                finally
                {
                    (subScene.ITextureMgr as TextureMgr).EndUpdate();
                }

                subScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// 編集対象に設定される直前の初期化処理
        /// </summary>
        public void FirstTimeInitialize()
        {
            // 何もしません。
        }

        /// <summary>
        /// 警告文字列を取得します。UIの各所で表示します。
        /// </summary>
        public string WarningMsg
        {
            get;
            private set;
        }

        /// <summary>
        /// 描画します。
        /// </summary>
        public void Draw(IRenderer renderer, DrawableOption option)
        {
            DrawImpl_(renderer, option);
        }

        /// <summary>
        ///
        /// </summary>
        private void DrawImpl_(IRenderer renderer, DrawableOption option)
        {
            if (subScene != null)
            {
                renderer.Trans(-this.OwnerPane.DrawOffsetX, -this.OwnerPane.DrawOffsetY);
                renderer.Scale(1.0f, -1.0f);

                if (!option.CheckActive(DrawableOptionFlag.IgnorePartsPaneRectDrawing) &&
                    (!option.CheckActive(DrawableOptionFlag.IgnorePaneImageDrawing)))
                {
                    // 枠の描画
                    RectangleF toltalRect = PaneHelper.GetBoundingVolume(this.OwnerPane);
                    SizeF bgSize = toltalRect.Size;
                    int origTransparency = renderer.Transparency;

                    renderer.Color = option.SystemBlue;
                    renderer.Transparency = (int)(origTransparency * (32.0f / 255.0f));
                    renderer.FillRectangle(this._partsRect.X, this._partsRect.Y, 0, this._partsRect.Width, this._partsRect.Height);

                    renderer.Transparency = (int)(origTransparency * (255.0f / 255.0f));

                    DrawPartsSubScene_(renderer, option);

                    // 枠の描画
                    renderer.Color = option.SystemBlue;
                    renderer.Transparency = (int)(origTransparency * (32.0f / 255.0f));
                    bgSize.Width = Math.Max(bgSize.Width - 4, 0);
                    bgSize.Height = Math.Max(bgSize.Height - 4, 0);
                    renderer.Transparency = (int)(origTransparency * (255.0f / 255.0f));

                    if (subScene.IStateMachine.IsEnabled)
                    {
                        subScene.IStateMachine.Draw(renderer);
                    }
                }
                else
                {
                    DrawPartsSubScene_(renderer, option);
                }

                renderer.Scale(1.0f, -1.0f);
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void DrawPartsSubScene_(IRenderer renderer, DrawableOption option)
        {
            if ((renderer.OptionFlag & IRenderOptionFlag.DrawOutline) != 0)
            {
                return;
            }

            DrawableOptionFlag originalFlag = option.DrawableOptionFlag;
            option.EnableFlag(DrawableOptionFlag.IgnoreBVDrawing | DrawableOptionFlag.IgnoreDummyObjectDrawing);

            subScene.Draw(renderer, option);

            // 選択されたペインでは、アクティブな上書きプロパティの外周に四角を描画します。
            if (this.OwnerPane.IsSelected || option.CheckActive(DrawableOptionFlag.DrawPartsPaneActiveProperty))
            {
                // 以降、子供階層での描画では、アクティブなプロパティの描画を行う
                option.EnableFlag(DrawableOptionFlag.DrawPartsPaneActiveProperty);

                Func<IPane, bool> paneDrawCondition = (drawPane) =>
                {
                    if(drawPane.PaneKind == PaneKind.Null ||
                       drawPane.PaneKind == PaneKind.Textbox ||
                       drawPane.PaneKind == PaneKind.Parts)
                    {
                        return this.PartsPropaerties.Any((prop) => prop.IsEnabled() && prop.TargetName == drawPane.PaneName);
                    }else{
                        return false;
                    }
                };

                // 境界の四角だけ書いて、中身は書かない。
                option.DisableFlag(DrawableOptionFlag.IgnoreBVDrawing);
                option.EnableFlag(DrawableOptionFlag.IgnorePaneImageDrawing);

                subScene.Draw(renderer, option, paneDrawCondition);
            }

            option.DrawableOptionFlag = originalFlag;
        }
    }

    //----------------------------------------------------------

    /// <summary>
    /// 部品が持つ上書きパラメータです。
    /// </summary>
    internal class PartsPropaerty : IPartsPropaerty
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsPropaerty(PaneKind paneKind, string TargetName, IPaneExParamater paramater, PartsPropertyUsageOptions option)
        {
            this.PaneKind = paneKind;
            this.TargetName = TargetName;
            this.Paramater = paramater;
            this.Option = option;
        }

        /// <summary>
        /// 種類です。
        /// </summary>
        public PaneKind PaneKind { get; private set; }

        /// <summary>
        /// 対象ペイン名です。
        /// </summary>
        public string TargetName { get; private set; }

        /// <summary>
        /// 上書きする実データです。
        /// </summary>
        public IPaneExParamater Paramater { get; internal set; }

        /// <summary>
        /// 部品ペインプロパティのオプションです。
        /// </summary>
        public PartsPropertyUsageOptions Option { get; internal set; }
    }
}
