﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;

namespace LECore.Structures
{
    using Core;
    using LECore.Structures;
    using LECore.Manipulator;

    class ParamaterPasterCommon
    {


        #region 基本型コピー関数
        /// <summary>
        /// 2次元 ベクトル値のコピー
        /// </summary>
        /// <param name="option"></param>
        /// <param name="vDst"></param>
        /// <param name="vSrc"></param>
        static public FVec2 CopyFVec2(
            PaneParamaterCopyOption option,
            FVec2 vDst,
            FVec2 vSrc )
        {
            FVec2 result = new FVec2( vDst );

            if(option.IsVectorElementEnable( VectorElement.X ) )
            {
                result.X = vSrc.X;
            }
            if(option.IsVectorElementEnable( VectorElement.Y ) )
            {
                result.Y = vSrc.Y;
            }
            return result;
        }

        /// <summary>
        /// 3次元 ベクトル値のコピー
        /// </summary>
        /// <param name="option"></param>
        /// <param name="vDst"></param>
        /// <param name="vSrc"></param>
        static public FVec3 CopyFVec3(
            PaneParamaterCopyOption option,
            FVec3 vDst,
            FVec3 vSrc )
        {
            FVec3 result = new FVec3( vDst );

            if(option.IsVectorElementEnable( VectorElement.X ) )
            {
                result.X = vSrc.X;
            }
            if(option.IsVectorElementEnable( VectorElement.Y ) )
            {
                result.Y = vSrc.Y;
            }
            if(option.IsVectorElementEnable( VectorElement.Z ) )
            {
                result.Z = vSrc.Z;
            }
            return result;
        }

        /// <summary>
        /// RGBAカラーのコピー
        /// </summary>
        /// <param name="option"></param>
        /// <param name="vDst"></param>
        /// <param name="vSrc"></param>
        static public RGBAColor CopyRGBA(
            PaneParamaterCopyOption option,
            RGBAColor cDst,
            RGBAColor cSrc )
        {
            RGBAColor result = new RGBAColor( cDst );

            if(option.IsVectorElementEnable( VectorElement.R ) )
            {
                result.R = cSrc.R;
            }
            if(option.IsVectorElementEnable( VectorElement.G ) )
            {
                result.G = cSrc.G;
            }
            if(option.IsVectorElementEnable( VectorElement.B ) )
            {
                result.B = cSrc.B;
            }
            if(option.IsVectorElementEnable( VectorElement.A ) )
            {
                result.A = cSrc.A;
            }

            return result;
        }

        /// <summary>
        /// 浮動小数点カラーのコピー
        /// </summary>
        /// <param name="option"></param>
        /// <param name="vDst"></param>
        /// <param name="vSrc"></param>
        static public FloatColor CopyRGBA(
            PaneParamaterCopyOption option,
            FloatColor cDst,
            FloatColor cSrc)
        {
            FloatColor result = new FloatColor(cDst);

            if (option.IsVectorElementEnable(VectorElement.R))
            {
                result.R = cSrc.R;
            }
            if (option.IsVectorElementEnable(VectorElement.G))
            {
                result.G = cSrc.G;
            }
            if (option.IsVectorElementEnable(VectorElement.B))
            {
                result.B = cSrc.B;
            }
            if (option.IsVectorElementEnable(VectorElement.A))
            {
                result.A = cSrc.A;
            }

            result.ByteConvert = cSrc.ByteConvert;

            return result;
        }
        #endregion 基本型コピー関数

        /// <summary>
        /// アトリビュートのキー情報をコピーします。
        /// </summary>
        static public void CopyAllAnimAttribute(
            IAnmAttribute dstAttr,
            IAnmAttribute srcAttr,
            ISubScene ownerSubScene,
            bool bOverwriteAnimation,
            bool copyTargetTag,
            string targetTagName)
        {
            if (!srcAttr.IsActiveAttribute || !dstAttr.IsActiveAttribute)
            {
                return;
            }

            if (copyTargetTag)
            {
                // 選択中のタグのみをコピー
                var currentTag = ownerSubScene.CurrentTagName;
                CopyAnimation_(
                    (srcAttr as AnmAttribute).GetAnmCurve(targetTagName),
                    (dstAttr as AnmAttribute).GetAnmCurve(currentTag),
                    bOverwriteAnimation
                );
            }
            else
            {
                // すべてのタグについてコピー
                var tags = SubSceneHelper.GetTags(ownerSubScene);
                foreach (string tag in tags)
                {
                    CopyAnimation_(
                        (srcAttr as AnmAttribute).GetAnmCurve(tag),
                        (dstAttr as AnmAttribute).GetAnmCurve(tag),
                        bOverwriteAnimation
                    );
                }
            }

            if ( srcAttr.NumSubAttribute == dstAttr.NumSubAttribute )
            {
                for( int i = 0 ; i < srcAttr.NumSubAttribute ; i++ )
                {
                    CopyAllAnimAttribute(
                        dstAttr.FindSubAttributeByIdx( i ),
                        srcAttr.FindSubAttributeByIdx( i ),
                        ownerSubScene,
                        bOverwriteAnimation,
                        copyTargetTag,
                        targetTagName);
                }
            }
        }

        private static void CopyAnimation_(AnmCurve src, AnmCurve dst, bool bOverwriteAnimation)
        {
            if (src != null && dst != null)
            {
                IAnmKeyFrame[] srcKeys = src.IKeyFrameSet;
                AnmCurve srcCurve = src as AnmCurve;
                AnmCurve dstCurve = dst as AnmCurve;

                dstCurve.PostInfinityType = src.PostInfinityType;
                dstCurve.PreInfinityType = src.PreInfinityType;

                // コピー先のキーは全て消去します。
                if (bOverwriteAnimation)
                {
                    dstCurve.ClearKeyFrames();
                }

                if (srcKeys.Length != 0)
                {
                    for (int i = 0; i < srcKeys.Length; i++)
                    {
                        IAnmKeyFrame srcKey = srcKeys[i];
                        AnmKeyFrame dstKey = dstCurve.FindKeyExistAt(srcKey.Time) as AnmKeyFrame;
                        if (dstKey == null)
                        {
                            dstCurve.AddNewKeyFrame(
                                srcKey.Time,
                                srcKey.ValueAsFloat,
                                srcKey.InTangent == srcKey.OutTangent,
                                srcKey.InTangent,
                                srcKey.OutTangent,
                                srcKey.InterporationType,
                                srcCurve.ViewScale);
                        }
                        else
                        {
                            dstKey.SetParamaters(srcKey as AnmKeyFrame);
                        }
                    }
                    dstCurve.Update(true);
                }
            }
        }
    }
}
