﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Text;

namespace LECore.Structures
{
    using Core;

    using Nsrif.Attributes;

    #region TevDirectStage

    ///
    public sealed class DirectStageTevArg
    {
        private AttrTevSource    _source = AttrTevSource.Texture0;
        public AttrTevOp        Op = AttrTevOp.Rgb;

        public AttrTevSource Source
        {
            get { return _source; }
            set { _source = value; }
        }

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set( DirectStageTevArg src)
        {
            if (object.ReferenceEquals(this, src)) return;

            this.Source = src.Source;
            this.Op     = src.Op;
        }
        #endregion

        #region 比較
        /// <summary>
        /// 比較。
        /// </summary>
        public bool IsSame( DirectStageTevArg src )
        {
            if( object.ReferenceEquals( this, src ) ) return true;

            return ( this.Source == src.Source &&
                     this.Op     == src.Op );
        }
        #endregion
    }

    /// <summary>
    /// TevDirectStage
    ///
    /// ほとんどそのまま、3Dエディタからコピーしています。
    /// ただし、XMLドキュメントに依存する部分を消去しています。
    /// </summary>
    public sealed class TevDirectStage
    {
        private const int MaxTevDirectStageCount = 3;

        private AttrCombineMode _combineMode = AttrCombineMode.Replace;
        private AttrTevKonst    _konst = AttrTevKonst.K0;
        private AttrTevScale    _scale  = AttrTevScale.Scale1;
        private bool            _copyReg = false;
        private readonly ITexMtx _indirectTexMtx;

        private DirectStageTevArg[] _tevArgs = new DirectStageTevArg[ MaxTevDirectStageCount ] {
            new DirectStageTevArg(),
            new DirectStageTevArg(),
            new DirectStageTevArg(),
        };

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private TevDirectStage(bool alphaStage)
        {
            // カラーステージ用とアルファステージ用で選択項目が異なるので
            // デフォルトコンストラクタは持たせない

            // アルファステージ規定値
            if (alphaStage)
            {
                for( int i = 0; i < this.TevArgs.Length; i++ ) {
                    this.TevArgs[ i ].Op = AttrTevOp.Alpha;
                }
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevDirectStage(IRevHWMaterial ownerMaterial, int stageNo, bool alphaStage) : this( alphaStage)
        {
            // インダイレクト行列は一つのみという仕様になっているので、先頭の行列を使います。
            var indMtxIndex = 0;
            _indirectTexMtx = !alphaStage && ownerMaterial != null ?
                ownerMaterial.IIndirectMaterialTexMtxSet[indMtxIndex] : new TexMtx(null, 0);

            for( int index = 0; index < this.TevArgs.Length; index++ ) {
                this.TevArgs[ index ].Source =
                    stageNo == 0 ? AttrTevSource.Constant : AttrTevSource.Previous;
            }
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevDirectStage(TevDirectStage src)
        {
            Set(src);
        }
        #endregion

        #region アトリビュート

        /// <summary>
        /// コンバインモード
        /// </summary>
        public AttrCombineMode CombineMode
        {
            get { return _combineMode; }
            set { _combineMode = value; }
        }

        /// <summary>
        /// スケール。
        /// </summary>
        public AttrTevScale Scale
        {
            get { return _scale;  }
            set { _scale = value; }
        }

        /// <summary>
        /// コンスタントレジスタ。
        /// </summary>
        public AttrTevKonst Konst
        {
            get { return _konst;  }
            set { _konst = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool CopyReg
        {
            get { return _copyReg;  }
            set { _copyReg = value; }
        }

        /// <summary>
        ///
        /// </summary>
        public DirectStageTevArg[] TevArgs
        {
            get { return _tevArgs; }
        }

        /// <summary>
        ///
        /// </summary>
        public FVec2 IndirectScale
        {
            get { return _indirectTexMtx.Scale; }
            set
            {
                if (_indirectTexMtx.Scale != value)
                {
                    (_indirectTexMtx as TexMtx).Scale = value;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public float IndirectRotate
        {
            get { return _indirectTexMtx.Rotate; }
            set
            {
                if (_indirectTexMtx.Rotate != value)
                {
                    (_indirectTexMtx as TexMtx).Rotate = value;
                }
            }
        }

        internal ITexMtx IndirectTexMtx
        {
            get { return _indirectTexMtx; }
        }

        /// <summary>
        ///
        /// </summary>
        public bool IsIndirectStage
        {
            get { return
                this.CombineMode == AttrCombineMode.Indirect ||
                this.CombineMode == AttrCombineMode.BlendIndirect ||
                this.CombineMode == AttrCombineMode.EachIndirect;
            }
        }

        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TevDirectStage src)
        {
            if (object.ReferenceEquals(this, src)) return;

            this.CombineMode = src.CombineMode;
            this.Konst       = src.Konst;
            this.Scale       = src.Scale;
            this.CopyReg     = src.CopyReg;

            this.IndirectScale = src.IndirectScale;
            this.IndirectRotate = src.IndirectRotate;

            for( int i = 0; i < this.TevArgs.Length; i++ ) {
                this.TevArgs[ i ].Set( src.TevArgs[ i ]);
            }
        }
        #endregion

        #region 比較
        /// <summary>
        /// 比較。
        /// </summary>
        public bool IsSame( TevDirectStage tds )
        {
            if( object.ReferenceEquals( this, tds ) ) return true;

            for( int i = 0; i < this.TevArgs.Length; i++ ) {
                if( this.TevArgs[ i ].IsSame( tds.TevArgs[ i ]) == false ) {
                    return false;
                }
            }

            return ( this.CombineMode == tds.CombineMode &&
                     this.Konst       == tds.Konst &&
                     this.Scale       == tds.Scale &&
                     this.CopyReg     == tds.CopyReg &&
                     this.IndirectScale == tds.IndirectScale &&
                     this.IndirectRotate == tds.IndirectRotate);
        }
        #endregion
    }
    #endregion TevDirectStage

    /// <summary>
    /// TevStage 外部公開インタフェース
    /// </summary>
    public interface ITevStage
    {
        /// <summary>
        /// ラスタライズカラーオーダー。
        /// </summary>
        AttrRasOrder RasOrder { get; }
        /// <summary>
        /// テクスチャマップオーダー。
        /// </summary>
        int TexMapOrder { get; }
        /// <summary>
        /// テクスチャ座標オーダー。
        /// </summary>
        int TexCoordOrder { get; }
        /// <summary>
        /// ラスタライズドカラースワップ。
        /// </summary>
        int RasSwap { get; }
        /// <summary>
        /// テクスチャマップスワップ。
        /// </summary>
        int TexMapSwap { get; }
        /// <summary>
        /// カラーステージ。
        /// </summary>
        TevDirectStage ColorStage { get; }
        /// <summary>
        /// アルファステージ。
        /// </summary>
        TevDirectStage AlphaStage { get; }
    }

    /// <summary>
    /// TevStage 外部公開インタフェース
    /// </summary>
    public interface ITevStages
    {
        IRevHWMaterial Owner { get; }
        int NumStages { get; }

        /// <summary>
        /// ステージ（インデクサ）。
        /// </summary>
        ITevStage GetTevStage(int stageNo);
    }

    /// <summary>
    /// TevStageアニメーションアトリビュート
    /// </summary>
    internal class TevStage :
        LEDataNode,
        ITevStage
    {
        public const string TevStageNodeName = "TevStage";

        #region ------------- フィールド -------------

        AttrRasOrder _rasOrder      = AttrRasOrder.Color0A0;
        int          _texMapOrder   = -1;
        int          _texCoordOrder = -1;

        // スワップ
        int _rasSwap    = 0;
        int _texMapSwap = 0;

        // ステージ
        readonly TevDirectStage   _colorStage;
        readonly TevDirectStage   _alphaStage;

        #endregion ------------- フィールド -------------

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ。
        /// </summary>
        private TevStage()
            :this( 0, null )
        {
        }

        /// <summary>
        /// コンストラクタ。
        /// </summary>
        public TevStage( int stageNo, LEDataNode owner )
            :base( owner, TevStageNodeName )
        {
            IAnmAttribute ownerMat = owner;
            while (ownerMat != null)
            {
                if (ownerMat is IRevHWMaterial)
                {
                    break;
                }

                ownerMat = ownerMat.OwnerNode;
            }

            _colorStage = new TevDirectStage(ownerMat as IRevHWMaterial, stageNo, false);
            _alphaStage = new TevDirectStage(ownerMat as IRevHWMaterial, stageNo, true);
        }
        #endregion コンストラクタ

        #region アトリビュート
        /// <summary>
        /// ラスタライズカラーオーダー。
        /// </summary>
        public AttrRasOrder RasOrder
        {
            get { return _rasOrder;  }
            set { _rasOrder = value; NotifyChage_();}
        }

        /// <summary>
        /// テクスチャマップオーダー。
        /// </summary>
        public int TexMapOrder
        {
            get { return _texMapOrder;  }
            set { _texMapOrder = value; NotifyChage_();}
        }

        /// <summary>
        /// テクスチャ座標オーダー。
        /// </summary>
        public int TexCoordOrder
        {
            get { return _texCoordOrder;  }
            set { _texCoordOrder = value; NotifyChage_(); }
        }

        /// <summary>
        /// ラスタライズドカラースワップ。
        /// </summary>
        public int RasSwap
        {
            get { return _rasSwap;  }
            set { _rasSwap = value; NotifyChage_(); }
        }

        /// <summary>
        /// テクスチャマップスワップ。
        /// </summary>
        public int TexMapSwap
        {
            get { return _texMapSwap;  }
            set { _texMapSwap = value; NotifyChage_();}
        }

        /// <summary>
        /// カラーステージ。
        /// </summary>
        public TevDirectStage ColorStage
        {
            get { return _colorStage; }
        }

        /// <summary>
        /// アルファステージ。
        /// </summary>
        public TevDirectStage AlphaStage
        {
            get { return _alphaStage; }
        }
        #endregion

        #region 設定
        /// <summary>
        /// 設定。
        /// </summary>
        public void Set(TevStage src)
        {
            if (object.ReferenceEquals(this, src)) return;

            _rasOrder      = src._rasOrder;
            _texMapOrder   = src._texMapOrder;
            _texCoordOrder = src._texCoordOrder;
            _rasSwap       = src._rasSwap;
            _texMapSwap    = src._texMapSwap;

            _colorStage.Set(src._colorStage);
            _alphaStage.Set(src._alphaStage);

            NotifyChage_();
        }

        /// <summary>
        /// 更新を通知します。
        /// </summary>
        void NotifyChage_()
        {
            base.NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        #endregion

    }

    /// <summary>
    /// TevStage列
    /// </summary>
    internal class TevStages
        : LEDataNode, ITevStages
    {
        public const string TevStagesNodeName = "TevStages";
        public static readonly int NumMaxStages = LayoutEditorCore.PlatformDetail.MaxTevStageCount;

        int                         _numStages = 0;
        TevStage[] _stages = new TevStage[NumMaxStages];
        readonly RevHWMaterial _owner;

        #region アトリビュート
        /// <summary>
        /// ステージ数。
        /// </summary>
        public int NumStages
        {
            get
            {
                return Math.Min(NumAvailableStages, _numStages);
            }

            set
            {
                _numStages = value; NotifyChage_();
            }
        }

        /// <summary>
        /// 利用可能なステージ数
        /// </summary>
        public int NumAvailableStages
        {
            get { return LECore.LayoutEditorCore.PlatformDetail.GetNumAvailableStages(this); }
        }

        public IRevHWMaterial Owner
        {
            get { return _owner; }
        }

        public ITevStage GetTevStage(int stageNo)
        {
            return _stages[stageNo];
        }

        public ITevStage[] GetTevStages()
        {
            return _stages;
        }

        /// <summary>
        /// ステージ（インデクサ）。
        /// </summary>
        public TevStage this[int stageNo]
        {
            get { return _stages[stageNo]; }
        }
        #endregion

        #region コンストラクタ
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public TevStages(LEDataNode ownerNode, RevHWMaterial revHWMaterial)
            :base( ownerNode, TevStagesNodeName )
        {
            for (int i = 0; i < _stages.Length; i++)
            {
                _stages[i] = new TevStage( i, this );
            }

            _owner = revHWMaterial;
        }
        #endregion コンストラクタ

        /// <summary>
        /// 更新を通知します。
        /// </summary>
        void NotifyChage_()
        {
            base.NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }
    }
}
