﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Structures
{
    using System.Drawing;
    using Core;

    /// <summary>
    /// ＴＥＶKカラーレジスタアトリビュート。
    /// </summary>
    [Obsolete]
    internal class TevKColor :
        AnmAttribute
    {
        #region RGBA カラーとしてのプロパティ


        /// <summary>
        /// RGBA カラーとしてのプロパティ
        /// </summary>
        public RGBAColor     RGBAColor
        {
            get
            {
                return  new RGBAColor(
                    FindAttributeByIdx( 0 ).GetAsByte(),
                    FindAttributeByIdx( 1 ).GetAsByte(),
                    FindAttributeByIdx( 2 ).GetAsByte(),
                    FindAttributeByIdx( 3 ).GetAsByte() );

            }
            set
            {
                FindAttributeByIdx( 0 ).SetValue( value.R );
                FindAttributeByIdx( 1 ).SetValue( value.G );
                FindAttributeByIdx( 2 ).SetValue( value.B );
                FindAttributeByIdx( 3 ).SetValue( value.A );
            }
        }
        #endregion RGBA カラーとしてのプロパティ

        /// <summary>
        /// コンストラクタ：複数のアトリビュートをサブアトリビュートに持つ、
        /// 混合アトリビュートとして初期化します。
        /// </summary>
        public TevKColor(
            LEDataNode owner,
            AnmAttrDescripter selfAttrDesc,
            AnmAttribute refColAttr0,
            AnmAttribute refAlpAttr0,
            AnmAttribute refColAttr1,
            AnmAttribute refAlpAttr1
        )
            : base( owner, null, selfAttrDesc )
        {
            // サブアトリビュートを初期化します。
            // 参照アトリビュートとして初期化を行っています。
            AnmAttrDescripter[]   descs =
            {
                new AnmAttrDescripter( AttributeType.RefAttribute, "R", null, refColAttr0 ),
                new AnmAttrDescripter( AttributeType.RefAttribute, "G", null, refAlpAttr0 ),
                new AnmAttrDescripter( AttributeType.RefAttribute, "B", null, refColAttr1 ),
                new AnmAttrDescripter( AttributeType.RefAttribute, "A", null, refAlpAttr1 )
            };

            foreach( AnmAttrDescripter desc in descs )
            {
                new AnmAttribute( this, null, desc );
            }
        }
    }

    public class TevColorsConstants
    {
        public static readonly int NumTevKColorRegs = LayoutEditorCore.PlatformDetail.TevConstantColorCount;
        public static readonly int ObsoleteNumTevKColorRegs = 6;
    }

    /// <summary>
    /// ＴＥＶKカラー管理クラス。
    ///
    /// 実際には、マテリアルパラメータの、
    /// テクスチャ混合比率アトリビュートを参照するアトリビュートです。
    /// </summary>
    internal class TevKColors : AnmAttribute
    {
        /// <summary>
        /// アニメーションアトリビュートの取得
        /// </summary>
        internal AnmAttribute GetTevKColorAttr( int index )
        {
            Debug.Assert(index >= 0 && index <= TevColorsConstants.NumTevKColorRegs);
            return FindAttributeByIdx( index );
        }

        /// <summary>
        /// コンストラクタ(非公開)
        /// </summary>
        internal TevKColors(LEDataNode owner, AnmAttrDescripter selfAttrDesc)
            : base(owner, null, selfAttrDesc)
        {
            for (int i = 0; i < LayoutEditorCore.PlatformDetail.TevConstantColorCount; i++)
            {
                AnmAttrDescripter desc = new AnmAttrDescripter(AttributeType.FloatRGBA4, "TEV_color" + i.ToString(), null, null);
                new AnmAttribute(this, null, desc);
            }
        }
    }

    /// <summary>
    /// TevData 外部公開インタフェース
    /// </summary>
    public interface ITevData
    {
        ITevStage     GetITevStage( int index );
        int           NumStages { get; }

        int NumTevColors { get; }
        FloatColor GetTevColor(int index);
    }

    public static class ITevDataHelper
    {
        public static bool IsIndirectState(this ITevData tevData, int index)
        {
            ITevStage stage = tevData.GetITevStage(index);
            if (stage.ColorStage.IsIndirectStage)
            {
                return true;
            }

            // 前段が BlendIndirect なら後段もインダイレクト用途
            int prevIndex = index - 1;
            if (prevIndex >= 0)
            {
                ITevStage prevStage = tevData.GetITevStage(prevIndex);
                return prevStage.ColorStage.CombineMode == Nsrif.Attributes.AttrCombineMode.BlendIndirect;
            }

            return false;
        }
    }

    /// <summary>
    /// TevData
    /// </summary>
    internal class TevData :
        AnmAttribute,
        ITevData
    {
        #region ------------- 定数 -------------
        static readonly AnmAttrDescripter  TevColAttrDescs =
            new  AnmAttrDescripter( AttributeType.Combined, "TevColors", null, null );
        static readonly AnmAttrDescripter  TevKColAttrDescs =
            new  AnmAttrDescripter( AttributeType.Combined, "TevKColors", null, null );


        #endregion ------------- 定数 -------------

        /// <summary>
        /// TEV ステージ情報
        /// </summary>
        readonly TevStages              _tevStages = null;
        readonly TevKColors _tevKColorAnims = null;
        bool[] _tevKColorsByteConvert = null;
        #region ------------- プロパティ -------------

        public TevStages TevStages
        {
            get{ return _tevStages;}
        }

        #endregion ------------- プロパティ -------------

        #region ------------- ITevDataメンバ -------------

        /// <summary>
        /// TEVステージ情報を取得します。
        /// </summary>
        public ITevStage     GetITevStage( int stageNo )
        {
            if (stageNo >= 0 && stageNo < NumStages)
            {
                return TevStages[stageNo];
            }
            return null;
        }

        /// <summary>
        /// TEVステージ数を取得します。
        /// </summary>
        public int           NumStages { get{ return TevStages.NumStages;} }
        public int NumTevColors { get { return TevColorsConstants.NumTevKColorRegs; } }

        public FloatColor GetTevColor(int index)
        {
            FloatColor color = GetTevColorAttr(index).GetAsFloatColor();
            color.ByteConvert = _tevKColorsByteConvert[index];
            return color;
        }

        public void SetTevColor(FloatColor color, int index)
        {
            _tevKColorsByteConvert[index] = color.ByteConvert;

            AnmAttribute tecColorAttr = GetTevColorAttr(index);
            if (tecColorAttr != null)
            {
                tecColorAttr.SetValue(new FVec4(color.R, color.G, color.B, color.A));
            }

            Material.SetColorAttributeCurveMode(_tevKColorsByteConvert[index], tecColorAttr);
        }

        public bool GetTevColorByteConvert(int index)
        {
            return _tevKColorsByteConvert[index];
        }

        public void SetTevColorByteConvert(bool byteConvert, int index)
        {
            _tevKColorsByteConvert[index] = byteConvert;
            var attr = GetTevColorAttr(index);
            Material.SetColorAttributeCurveMode(byteConvert, attr);
        }

        public AnmAttribute GetTevColorAttr(int index)
        {
            return _tevKColorAnims.FindSubAttributeByIdx(index) as AnmAttribute;
        }

        #endregion ------------- ITevDataメンバ -------------

        /// <summary>
        /// コンストラクタ(非公開)
        /// </summary>
        /// <param name="owner"></param>
        public TevData(LEDataNode owner, AnmAttrDescripter selfAttrDesc, RevHWMaterial revHWMaterial)
            :base( owner, null, selfAttrDesc )
        {
            // サブアトリビュートの初期化
            _tevKColorAnims = new TevKColors(this, TevKColAttrDescs);

            // TevStages初期化
            _tevStages = new TevStages(this, revHWMaterial);

            _tevKColorsByteConvert = new bool[TevColorsConstants.NumTevKColorRegs];
            for (int i = 0; i < TevColorsConstants.NumTevKColorRegs; i++)
            {
                _tevKColorsByteConvert[i] = true;

                AnmAttribute attr = GetTevColorAttr(i);

                // カーブ状態の初期設定
                Material.SetColorAttributeCurveMode(true, attr);
            }
        }

        /// <summary>
        /// 値を設定します。
        /// </summary>
        public void Set( ITevData src )
        {
            // TevStage
            this.TevStages.NumStages = src.NumStages;
            for( int i = 0;i < this.TevStages.NumStages; i++ )
            {
                ITevStage       srcTevStage = src.GetITevStage( i );
                this.TevStages[i].Set( srcTevStage as TevStage );
            }

            for (int i = 0;i < src.NumTevColors;i++)
            {
                this.SetTevColor(src.GetTevColor(i), i);
            }
        }
    }
}
