﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;

namespace LECore.Structures
{
    using Core;
    using LECoreInterface;
    using Nsrif.Attributes;
    using System.Collections.Generic;

    /// <summary>
    /// テクスチャの利用種類(追加)
    /// </summary>
    public enum LETexMapKind
    {
        Normal,
        IndirectWarpTexture,
        IndirectBumpMap
    };

    /// <summary>
    /// MaterialTexMap 用のキャッシュ
    /// </summary>
    internal class MaterialTexMapImageCache
    {
        private ITextureImage _textureImage { get; set; }
        private DateTime _updateTime { get; set; }

        /// <summary>
        /// キャッシュを設定する。
        /// </summary>
        public void Set(ITextureImage textureImage)
        {
            this._textureImage = textureImage;
            this._updateTime = textureImage != null ? textureImage.UpdateTime : DateTime.MinValue;
        }

        /// <summary>
        /// リセット
        /// </summary>
        public void Reset()
        {
            this.Set(null);
        }

        /// <summary>
        /// 状態を更新する。
        /// </summary>
        public ITextureImage GetTextureImage()
        {
            if (this._textureImage == null)
            {
                return null;
            }

            if (this._updateTime > this._textureImage.UpdateTime)
            {
                return null;
            }


            // ビットマップがDisposeされてるときなどをチェック
            if (this._textureImage.ColorGDIBitmap == null || this._textureImage.GDIBitmap == null || !this._textureImage.Valid)
            {
                return null;
            }

            return this._textureImage;
        }
    }

    /// <summary>
    /// MaterialTexMap 外部モジュール公開インタフェース
    /// </summary>
    public interface IMaterialTexMap
    {
        string           TexImgName        { get; }
        string           PaletteName       { get; }
        int              SlotIdx           { get; }

        ITextureChanger  ITextureChanger   { get; }

        AttrTexWrap      WrapS             { get; }
        AttrTexWrap      WrapT             { get; }

        AttrTexFilterMin MinFilter         { get; }
        AttrTexFilterMag MagFilter         { get; }

        ITexGen      ITexGen                  { get; }

        bool         IsActive                 { get; }
        LETexMapKind TexMapKind               { get; }

        AttrTextureResourceType ResourceType  { get; }
    }

    /// <summary>
    /// ヘルパクラス
    /// </summary>
    public static class MaterialTexMapHelper
    {
        public static ITextureImage TryFindTextureImage(this IMaterialTexMap materialTexMap, ISubScene subScene)
        {
            MaterialTexMap matTexMap = materialTexMap as MaterialTexMap;
            ITextureImage texImg = matTexMap.ImageCache.GetTextureImage();
            if (texImg == null)
            {
                if (matTexMap.ResourceType == AttrTextureResourceType.LocalFile)
                {
                    texImg = subScene.ITextureMgr.FindITextureImageByName(matTexMap.TexImgName);
                }
                else
                {
                    // 動的キャプチャテクスチャはファイルリソースと違い SubScene ごとにテクスチャを保持しているため
                    // パーツから親をたどって所属している可能性のある TextureMgr をすべて探す。
                    texImg = SubSceneHelper.FindCaptureTextureRecursively(subScene, matTexMap.TexImgName);
                }
                matTexMap.ImageCache.Set(texImg);
            }

            return texImg;
        }

        public static IEnumerable<IMaterialTexMap> EnumerateCaptureRelatedTexMap(this IEnumerable<IMaterialTexMap> materialTexMaps)
        {
            return materialTexMaps.Where((map) => map.ResourceType != AttrTextureResourceType.LocalFile);
        }

        public static AttrTextureResourceType GetAttrTextureResourceType(TextureSourceType type)
        {
            return type == TextureSourceType.Dynamic ?
                Structures.Nsrif.Attributes.AttrTextureResourceType.LocalCaptured :
                Structures.Nsrif.Attributes.AttrTextureResourceType.LocalFile;
        }

        public static AttrTextureResourceType GetAttrTextureResourceType(IMaterial ownerMaterial, TextureSourceType type)
        {
            AttrTextureResourceType resType = AttrTextureResourceType.LocalFile;

            if (type == LECore.Structures.LECoreInterface.TextureSourceType.Dynamic)
            {
                resType = AttrTextureResourceType.LocalCaptured;

                if (ownerMaterial.OwnerPane.OwnerSubScene.IPartsLayout != null)
                {
                    resType = AttrTextureResourceType.OverrideCaptured;
                }
            }

            return resType;
        }
    }

    /// <summary>
    /// 4頂点のUV座標
    /// </summary>
    public class TexCoord4
    {
        public delegate void TexCoord4ModifyFunc( TexCoord4 modifyData );

        FVec2            _LT = new FVec2( 0.0f, 0.0f );
        FVec2            _RT = new FVec2( 1.0f, 0.0f );
        FVec2            _LB = new FVec2( 0.0f, 1.0f );
        FVec2            _RB = new FVec2( 1.0f, 1.0f );

        public FVec2 LT { get{ return _LT;} set{ _LT = value;} }
        public FVec2 RT { get{ return _RT;} set{ _RT = value;} }
        public FVec2 LB { get{ return _LB;} set{ _LB = value;} }
        public FVec2 RB { get{ return _RB;} set{ _RB = value;} }

        /// <summary>
        ///
        /// </summary>
        public void Set( TexCoord4 src )
        {
            _LT = new FVec2( src.LT );
            _RT = new FVec2( src.RT );
            _LB = new FVec2( src.LB );
            _RB = new FVec2( src.RB );
        }

        /// <summary>
        ///
        /// </summary>
        public TexCoord4()
        {

        }

        static void ScaleFVec2_( ref FVec2 vec, float sx, float sy, float ox, float oy )
        {
            vec.X = ( vec.X - ox ) * sx + ox;
            vec.Y = ( vec.Y - oy ) * sy + oy;
        }

        /// <summary>
        ///
        /// </summary>
        public void Scale( float sx, float sy, float ox, float oy )
        {
            ScaleFVec2_( ref _LT, sx, sy, ox, oy );
            ScaleFVec2_( ref _RT, sx, sy, ox, oy );
            ScaleFVec2_( ref _LB, sx, sy, ox, oy );
            ScaleFVec2_( ref _RB, sx, sy, ox, oy );
        }

        /// <summary>
        ///
        /// </summary>
        public TexCoord4( TexCoord4 src )
        {
            this.Set( src );
        }

        /// <summary>
        ///
        /// </summary>
        static public bool CheckSame( TexCoord4 lhs, TexCoord4 rhs )
        {
            return
                lhs._LT == rhs._LT &&
                lhs._RT == rhs._RT &&
                lhs._LB == rhs._LB &&
                lhs._RB == rhs._RB;
        }

        // 水平フリップ
        public void FlipHorizontal()
        {
            FVec2 temp = new FVec2();

            temp.Set( _LT );
            _LT = _RT;
            _RT = temp;

            temp.Set( _LB );
            _LB = _RB;
            _RB = temp;
        }

        // 垂直フリップ
        public void FlipVertical()
        {
            FVec2 temp = new FVec2();

            temp.Set( _LT );
            _LT = _LB;
            _LB = temp;

            temp.Set( _RT );
            _RT = _RB;
            _RB = temp;
        }

        // 90度時計回転
        public void Rotate90()
        {
            FVec2 temp = new FVec2();

            temp.Set( _LT );
            _LT = _LB;
            _LB = _RB;
            _RB = _RT;
            _RT = temp;
        }

        // 180度時計回転
        public void Rotate180()
        {
            FVec2 temp = new FVec2();

            temp.Set( _LT );
            _LT = _RB;
            _RB = temp;

            temp.Set( _LB );
            _LB = _RT;
            _RT = temp;
        }

        // 270度時計回転
        public void Rotate270()
        {
            FVec2 temp = new FVec2();

            temp.Set( _LT );

            _LT = _RT;
            _RT = _RB;
            _RB = _LB;
            _LB = temp;
        }

    }

    /// <summary>
    /// マテリアルが保持している、テクスチャマップ
    /// </summary>
    internal class MaterialTexMap :
        LEDataNode,
        IMaterialTexMap,
        IDisposable
    {
        public delegate ITextureImage TextureImageLoadFunc( string fileName );

        public const string NoneTextureName        = "none";
        public const string MaterialTexMapNodeName = "MaterialTexMap_{0}";

#region ------------- フィールド -------------
        /// <summary>
        /// テクスチャ名
        /// </summary>
        string         _texName = "DummyTexName";

        /// <summary>
        /// ラップモード
        /// </summary>
        AttrTexWrap      _wrapS     = AttrTexWrap.Clamp;
        AttrTexWrap      _wrapT     = AttrTexWrap.Clamp;

        /// <summary>
        /// フィルタ(拡大、縮小)
        /// </summary>
        AttrTexFilterMin _minFilter = AttrTexFilterMin.Linear;
        AttrTexFilterMag _magFilter = AttrTexFilterMag.Linear;

        /// <summary>
        /// 参照しているテクスチャのリソースタイプ
        /// </summary>
        AttrTextureResourceType _resType = AttrTextureResourceType.LocalFile;

        /// <summary>
        /// 4隅のテクスチャ座標
        /// </summary>
        // TexCoord4        _texCoord  = new TexCoord4();
        // LETexGenMethod _texCoordSrc = LETexGenMethod.UV0;

        /// <summary>
        /// テクスチャ座標生成関連パラメータ
        /// </summary>
        readonly TexGen             _texGen;

        /// <summary>
        /// テクスチャパターンアニメーション
        /// </summary>
        readonly TextureChanger     _textureChanger;

        /// <summary>
        /// テクスチャ番号
        /// </summary>
        readonly int                _slotIdx;

        /// <summary>
        /// 持ち主マテリアル
        /// </summary>
        readonly Material           _owner;

        /// <summary>
        /// MaterialTexMap が持つ
        /// アニメーション アトリビュートを束ねるアトリビュート
        /// </summary>
        readonly LEDataNode  _ownerMaterialNode;

        readonly LETexMapKind           _texMapKind = LETexMapKind.Normal;

        readonly MaterialTexMapImageCache _imageCache = new MaterialTexMapImageCache();

#endregion ------------- フィールド -------------

#region ---------------- プロパティ ----------------

        /// <summary>
        /// MaterialTexMap が持つ
        /// アニメーションアトリビュートを取得します。
        /// </summary>
        public LEDataNode OwnerMaterialNode
        {
            get { return _ownerMaterialNode; }
        }

        /// <summary>
        /// テクスチャ座標生成パラメータを取得します。
        /// </summary>
        public TexGen           TexGen
        {
            get{ return _texGen;}
        }

        /// <summary>
        /// 持ち主マテリアルを取得します。
        /// </summary>
        public Material           OwnerMaterial
        {
            get{ return _owner;}
        }

#endregion ---------------- プロパティ ----------------

#region IMaterialTexMap メンバ

        /// <summary>
        /// 繰り返し方法 S
        /// </summary>
        public AttrTexWrap      WrapS
        {
            get{ return _wrapS;}
            set{ _wrapS = value; NotifyModifyEvent_();}
        }

        /// <summary>
        /// 繰り返し方法 T
        /// </summary>
        public AttrTexWrap      WrapT
        {
            get{ return _wrapT;}
            set{ _wrapT = value; NotifyModifyEvent_();}
        }

        /// <summary>
        /// 縮小フィルタ
        /// </summary>
        public AttrTexFilterMin MinFilter
        {
            get{ return _minFilter; }
            set{ _minFilter = value; NotifyModifyEvent_();}
        }

        /// <summary>
        /// 拡大フィルタ
        /// </summary>
        public AttrTexFilterMag MagFilter
        {
            get{ return _magFilter; }
            set{ _magFilter = value; NotifyModifyEvent_();}
        }

        /// <summary>
        /// テクスチャパターンアニメータ
        /// </summary>
        public ITextureChanger   ITextureChanger
        {
            get{ return _textureChanger;}
        }

        /// <summary>
        /// テクスチャパターンアニメータ
        /// </summary>
        public TextureChanger   TextureChanger
        {
            get{ return _textureChanger;}
        }

        /// <summary>
        /// テクスチャ画像名
        /// </summary>
        public string TexImgName
        {
            get { return _texName; }
            set
            {
                if( _texName != value )
                {
                    _texName = value;
                    this.ImageCache.Reset();

                    if (_textureChanger != null)
                    {
                        _textureChanger.IsActive = IsActive;
                    }

                    NotifyModifyEvent_();
                }
            }
        }

        public MaterialTexMapImageCache ImageCache { get { return _imageCache; } }

        /// <summary>
        /// テクスチャパレット名
        /// </summary>
        public string PaletteName
        {
            get{ return "TODO:"; }
            set{ ; }
        }

        /// <summary>
        /// スロット番号
        /// </summary>
        public int SlotIdx
        {
            get{ return _slotIdx;}
        }

        /// <summary>
        /// テクスチャ座標生成パラメータ
        /// </summary>
        public ITexGen   ITexGen
        {
            get{ return _texGen;}
        }

        /// <summary>
        /// 有効な状態か取得します。
        /// </summary>
        public bool      IsActive
        {
            get { return ( TexImgName != NoneTextureName ); }
        }

        /// <summary>
        /// テクスチャの種類
        /// </summary>
        public LETexMapKind TexMapKind
        {
            get{ return _texMapKind;}
        }

        /// <summary>
        /// 参照しているテクスチャのリソースタイプ
        /// </summary>
        public AttrTextureResourceType ResourceType
        {
            get { return _resType; }
            set { _resType = value; }
        }

#endregion IMaterialTexMap メンバ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public MaterialTexMap(
            string    texName,
            Material  ownerMaterial,
            int       slotIdx
            ):base( ownerMaterial, string.Format( MaterialTexMapNodeName, slotIdx ) )
        {
            _slotIdx = slotIdx;
            _owner           = ownerMaterial;
            _ownerMaterialNode = ownerMaterial;

            // テクスチャアニメーションを初期化します。
            _textureChanger = new TextureChanger( this, slotIdx );

            // テクスチャ座標生成を初期化します。
            _texGen = new TexGen(this, ownerMaterial);

            this.Reset();

            // リセット後に、名前を設定する
            this.TexImgName = texName;
        }

        /// <summary>
        /// リソースを破棄します。
        /// </summary>
        public void Dispose()
        {
            _texGen.Dispose();
            _textureChanger.Dispose();
        }

        /// <summary>
        /// テクスチャをリセットします。
        /// リセット後は、有効なテクスチャではありません。
        /// </summary>
        public void Reset()
        {
            TexImgName = NoneTextureName;
            _textureChanger.Reset();
        }

        void NotifyModifyEvent_()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }

        /// <summary>
        /// 上位階層のキャプチャテクスチャで上書きされたパーツ内のマテリアルで参照しているテクスチャ名を、自身に配置された構造に応じて名前を適切に設定します。
        /// 上書きで設定されたキャプチャテクスチャは、設定された時点のボディのレイアウトをルートとした名前で保存されているため、実際に配置されたパーツ階層に応じて名前を修飾します。
        /// </summary>
        public void ResolveOverrideCaptureTextureName(string namePrefix)
        {
            // Prefix が空の場合は修飾されていないので何もしない。
            if (namePrefix.Length == 0)
            {
                return;
            }

            if (this.ResourceType != AttrTextureResourceType.OverrideCaptured)
            {
                return;
            }

            // すでに全く同じ Prefix が設定されている場合は無視する。
            if (this.TexImgName.IndexOf(namePrefix) >= 0)
            {
                return;
            }

            this.TexImgName = PaneHelper.MakeCaptureTextureName(namePrefix, this.TexImgName);
        }

        public void ResolveTexturename()
        {
            if(this.ResourceType == AttrTextureResourceType.LocalFile)
            {
                return;
            }

            int dotPos = this.TexImgName.LastIndexOf(PaneHelper.CaptureTextureNameSeparator);
            string localName = dotPos >= 0 ? this.TexImgName.Substring(dotPos + 1) : this.TexImgName;

            // pane インスタンスの階層構造をたどって Prefix を決定します。
            // そのため pane インスタンスが最終的なインスタンスの階層構造に組み込まれた状態でないと意図した Prefix が取得できません。
            switch (this.ResourceType)
            {
                case AttrTextureResourceType.LocalCaptured:
                    IPane ownerPane = this.OwnerMaterial?.OwnerPane;
                    Debug.Assert(ownerPane != null);

                    this.TexImgName = PaneHelper.MakeCaptureTextureName(PaneHelper.GetCaptureTexturePrefix(ownerPane), localName);
                    break;
                case AttrTextureResourceType.OverrideCaptured:
                    // 上書きキャプチャテクスチャを使用している場合は PartsPane を処理するさいに ResolveOverrideCaptureTextureName で必要に応じて解決される。
                    break;
            }
        }

        public void ResetCache()
        {
            if (this.ResourceType != Nsrif.Attributes.AttrTextureResourceType.LocalFile)
            {
                this.ImageCache.Reset();
            }
        }

        public bool IsSelfReferenceCapture(string paneName)
        {
            return
                this.ResourceType != Nsrif.Attributes.AttrTextureResourceType.LocalFile &&
                this.TexImgName == paneName;
        }
    }

    /// <summary>
    /// 外部モジュール公開インタフェース
    /// </summary>
    public interface ITextureStage
    {
        int TextureCoordGenIndex { get; }
        int TextureMapIndex      { get; }
    }

    /// <summary>
    /// マテリアルのテクスチャ情報
    /// 画像とテクスチャ座標を関連付けます。
    /// </summary>
    internal class TextureStage :
        ITextureStage
    {
        int       _textureCoordGenIndex = 0;
        int       _textureMapIndex      = 0;

#region ITextureStage メンバ

        /// <summary>
        /// テクスチャ座標番号
        /// </summary>
        public int TextureCoordGenIndex
        {
            get{ return _textureCoordGenIndex; }
            set{ _textureCoordGenIndex = value; }
        }

        /// <summary>
        /// テクスチャマップ番号
        /// </summary>
        public int TextureMapIndex
        {
            get{ return _textureMapIndex; }
            set{ _textureMapIndex = value; }
        }

#endregion ITextureStage メンバ
    }

}
