﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Reflection;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures
{
    using Core;
    using Nsrif.Attributes;

    #region ------------- マテリアル -------------

    public interface IColorBlendHolder
    {
        bool IsThresholdingAlphaInterpolationEnabled { get; }

        FloatColor BlackColor { get; }
        FloatColor WhiteColor { get; }

        IAnmAttribute BlackColorIAnmAttr { get; }
        IAnmAttribute WhiteColorIAnmAttr { get; }
    }

    /// <summary>
    /// マテリアル外部公開インタフェース
    /// </summary>
    public interface IMaterial : IColorBlendHolder
    {
        /// <summary>
        /// アニメーションアトリビュートを取得します。
        /// </summary>
        string                  MaterialName    {get;}
        IMaterialTexMap[]       IMaterialTexMapSet{get;}
        ITexMtx[]               ITexMtxSet{get;}
        int                     UserMaxNumTexCoord{get;}

        float HOffset { get; }
        float SScale { get; }
        float BScale { get; }

        IPane OwnerPane { get; }
    }

    public class MaterialDefinition
    {
        public const int MaxNumTexCoord   = 3;
        public const int MaxNumMatTexture = 3;
        public const int MaxNumMatTexMtx  = 3;
    }

    /// <summary>
    /// マテリアル
    /// </summary>
    internal class Material :
        AnmAttribute,
        IMaterial,                // 外部モジュール公開インタフェース
        IDisposable
    {
        public const int MaxNumTexCoord   = MaterialDefinition.MaxNumTexCoord;
        public const int MaxNumMatTexture = MaterialDefinition.MaxNumMatTexture;
        public const int MaxNumMatTexMtx  = MaterialDefinition.MaxNumMatTexMtx;
        public const string _MaterialNodeName = "Material";

        static readonly AnmAttrDescripter[] SubAttrDescs =
        {
            new  AnmAttrDescripter( AttributeType.FloatRGBA4, "BlackColor",null, new FVec4(0.0f, 0.0f, 0.0f, 0.0f)),
            new  AnmAttrDescripter( AttributeType.FloatRGBA4, "WhiteColor",null, new FVec4(1.0f, 1.0f, 1.0f, 1.0f))
        };

        static AnmAttrDescripter _MatAnmAttrDesc =
            new AnmAttrDescripter( AttributeType.Combined, "material", null, null );

        string                      _name = null;

        /// <summary>
        /// マテリアルが保持するテクスチャマップ
        /// </summary>
        MaterialTexMap[]            _materialTexMaps = new MaterialTexMap[MaxNumMatTexture];
        TexMtx[]                    _textureMtxSet   = new TexMtx[MaxNumMatTexMtx];

        /// <summary>
        /// 状況に応じて設定可能である、テクスチャ関連パラメータの最大値
        /// </summary>
        int _userMaxNumTexCoord   = 1;

        bool _isThresholdingAlphaInterpolationEnabled;

        /// <summary>
        /// テクスチャ数変更時イベントです。
        /// テクスチャステージ数の更新のために利用しています。
        /// </summary>
        public event Action OnTextureCountChanged;

        #region ------------ プロパティ ------------

        public TexMtx[]            TexMtxSet          {get{return _textureMtxSet;}}

        public float _hOffset = 0.0f;
        public float _sScale = 1.0f;
        public float _bScale = 1.0f;

        public float HOffset
        {
            get { return _hOffset; }
            set { if (_hOffset != value) { _hOffset = value; NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify); } }
        }

        public float SScale
        {
            get { return _sScale; }
            set { if (_sScale != value) { _sScale = value; NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify); } }
        }

        public float BScale
        {
            get { return _bScale; }
            set { if (_bScale != value) { _bScale = value; NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify); } }
        }

        #endregion ------------ プロパティ ------------

        #region -------------- アクセサ類 --------------

        /// <summary>
        /// マテリアルテクスチャを番号を指定して取得します。
        /// </summary>
        public IMaterialTexMap GetIMaterialTexMapByIndex( int index )
        {
            return GetMaterialTexMapByIndex( index ) as IMaterialTexMap;
        }

        /// <summary>
        /// マテリアルテクスチャを番号を指定して取得します。
        /// </summary>
        public MaterialTexMap GetMaterialTexMapByIndex( int index )
        {
            if( index < MaxNumMatTexture )
            {
                return _materialTexMaps[index];
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// テクスチャリストが指定されたテクスチャを含むか調査します。
        /// </summary>
        bool MatTexListContains_( MaterialTexMap matTex )
        {
            foreach( MaterialTexMap mat in _materialTexMaps )
            {
                if( matTex == mat )
                {
                    // 発見
                    return true;
                }
            }
            return false;
        }

        public MaterialTexMap RegisterMatTexture(string texImageName, int index)
        {
            return RegisterMatTexture(texImageName, index, AttrTextureResourceType.LocalFile);
        }

        /// <summary>
        /// マテリアルの使用するテクスチャを追加します。
        /// texImageName は 拡張子無しで指定します。
        /// </summary>
        public MaterialTexMap RegisterMatTexture( string texImageName, int index, AttrTextureResourceType resType)
        {
            Debug.Assert( !texImageName.Contains( ".tga" ) );

            if( index < MaxNumMatTexture )
            {
                Debug.Assert( _materialTexMaps[index] != null );

                _materialTexMaps[index].TexImgName = texImageName;
                _materialTexMaps[index].ResourceType = resType;
                _textureMtxSet[index].IsActiveAttribute = true;

                if (OnTextureCountChanged != null)
                {
                    OnTextureCountChanged();
                }

                // 変更を通知します。
                NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
                NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneAttrModify);
                return _materialTexMaps[index];
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// マテリアルの使用するテクスチャを削除します。
        /// </summary>
        private bool RemoveMatTexture_(int index)
        {
            if (index < MaxNumMatTexture)
            {
                Debug.Assert(_materialTexMaps[index] != null);
                if (_materialTexMaps[index].IsActive)
                {
                    _materialTexMaps[index].Reset();
                    _textureMtxSet[index].IsActiveAttribute = false;

                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// マテリアルの使用するテクスチャを削除します。
        /// </summary>
        public void RemoveMatTexture(int index)
        {
            if (RemoveMatTexture_(index))
            {
                if (OnTextureCountChanged != null)
                {
                    OnTextureCountChanged();
                }

                NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
                NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneAttrModify);
            }
        }

        /// <summary>
        /// すべてのマテリアルテクスチャを削除します。
        /// </summary>
        public void RemoveMatTextureAll()
        {
            for( int i = 0 ; i < MaxNumMatTexture ; i++ )
            {
                RemoveMatTexture(i);
            }
        }

        /// <summary>
        /// すべてのマテリアルテクスチャを削除します。
        /// </summary>
        public void RemoveMatTextureAllNoEvent()
        {
            for (int i = 0; i < MaxNumMatTexture; i++)
            {
                RemoveMatTexture_(i);
            }
        }

        /// <summary>
        /// テクスチャマトリクスを取得します。
        /// </summary>
        public TexMtx GetTextureMtx( int texMtxIdx )
        {
            if( texMtxIdx >= MaxNumMatTexMtx )
            {
                return null;
            }

            Debug.Assert( texMtxIdx >= -1 && texMtxIdx < MaxNumMatTexMtx );

            if( texMtxIdx == -1 )
            {
                return null;
            }

            Debug.Assert( _textureMtxSet[texMtxIdx] != null );

            return _textureMtxSet[texMtxIdx];
        }

        /// <summary>
        /// 指定テクスチャ名のテクスチャが使用されているか調査します。
        /// </summary>
        public bool UsesTextureImage( string texName )
        {
            string[] nameSet = MaterialHelper.GetAllTextureName( this );
            return Array.IndexOf( nameSet, texName ) != -1;
        }

        #endregion -------------- アクセサ類 --------------

        #region ---------------- コンストラクタ ----------------

        /// <summary>
        /// テスト用ダミーデータの生成。
        /// </summary>
        void MakeDummyTexMapForTesting_()
        {
            MaterialTexMap newTex = null;
            newTex = RegisterMatTexture( "Tex0", 0);
            newTex = RegisterMatTexture( "Tex1", 1);
            newTex = RegisterMatTexture( "Tex2", 2);
        }

        /// <summary>
        /// マテリアルの持ち主ペインを取得します。
        /// </summary>
        public IPane OwnerPane
        {
            get
            {
                IPaneExParamater paneExParamater = this.OwnerNode as IPaneExParamater;
                if( paneExParamater != null )
                {
                    return paneExParamater.OwnerPane;
                }
                else
                {
                    return null;
                }
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Material(LEDataNode ownerAnmAttr, string name)
            : base(ownerAnmAttr, SubAttrDescs, _MatAnmAttrDesc)
        {
            _name = name;

            // テクスチャ行列の初期化
            for (int i = 0; i < _textureMtxSet.Length; i++)
            {
                Debug.Assert(_textureMtxSet[i] == null);
                _textureMtxSet[i] = new TexMtx(this, i);
                _textureMtxSet[i].IsActiveAttribute = false;
            }

            // マテリアルテクスチャを初期化します。
            for (int texIdx = 0; texIdx < MaxNumMatTexture; texIdx++)
            {
                var newTex = new MaterialTexMap(MaterialTexMap.NoneTextureName, this, texIdx);
                _materialTexMaps[texIdx] = newTex;
            }

            // カーブ状態の初期設定
            SetColorAttributeCurveMode(WhiteColorByteConvert, WhiteColorAnmAttr);
            SetColorAttributeCurveMode(BlackColorByteConvert, BlackColorAnmAttr);
        }

        /// <summary>
        /// リソースを開放します。
        /// </summary>
        public override void Dispose()
        {
            for( int texIdx = 0; texIdx < MaxNumMatTexture; texIdx++ )
            {
                Debug.Assert( _materialTexMaps[texIdx] != null );
                _materialTexMaps[texIdx].Dispose();
            }

            base.Dispose();
        }

        #endregion ---------------- コンストラクタ ----------------

        #region ------------ IMaterial メンバ ------------

        private bool _blackColorByteConvert = true;
        public bool BlackColorByteConvert
        {
            get
            {
                return _blackColorByteConvert;
            }
            set
            {
                _blackColorByteConvert = value;
                SetColorAttributeCurveMode(value, BlackColorAnmAttr);
            }
        }

        private bool _whiteColorByteConvert = true;
        public bool WhiteColorByteConvert
        {
            get
            {
                return _whiteColorByteConvert;
            }
            set
            {
                _whiteColorByteConvert = value;
                SetColorAttributeCurveMode(value, WhiteColorAnmAttr);
            }
        }

        /// <summary>
        /// 黒カラー
        /// </summary>
        public FloatColor BlackColor
        {
            get
            {
                FloatColor  color = BlackColorAnmAttr.GetAsFloatColor();
                color.ByteConvert = BlackColorByteConvert;

                return color;
            }
            set
            {
                BlackColorAnmAttr.SetValue(new FVec4(value.R, value.G, value.B, value.A));
                BlackColorByteConvert = value.ByteConvert;
            }
        }

        /// <summary>
        /// 白カラー
        /// </summary>
        public FloatColor WhiteColor
        {
            get
            {
                FloatColor  color = WhiteColorAnmAttr.GetAsFloatColor();
                color.ByteConvert = WhiteColorByteConvert;

                return color;
            }
            set
            {
                WhiteColorAnmAttr.SetValue(new FVec4(value.R, value.G, value.B, value.A));
                WhiteColorByteConvert = value.ByteConvert;
            }
        }

        /// <summary>
        /// カラーアトリビュートのモード切替処理
        /// </summary>
        static public void SetColorAttributeCurveMode(bool byteMode, AnmAttribute attr)
        {
            if (byteMode)
            {
                for (int i = 0; i < 4; ++i)
                {
                    if (attr.FindAttributeByIdx(i).HasAnimationCurve)
                    {
                        attr.FindAttributeByIdx(i).ICurrentAnimationCurve.ViewScale = 255.0f;
                        attr.FindAttributeByIdx(i).MinValue = 0.0f;
                        attr.FindAttributeByIdx(i).MaxValue = 1.0f;
                    }
                }
            }
            else
            {
                for (int i = 0; i < 4;++i)
                {
                    if (attr.FindAttributeByIdx(i).HasAnimationCurve)
                    {
                        attr.FindAttributeByIdx(i).ICurrentAnimationCurve.ViewScale = 1.0f;
                        attr.FindAttributeByIdx(i).MinValue = 0.0f;
                        attr.FindAttributeByIdx(i).MaxValue = System.Single.MaxValue;
                    }
                }
            }
        }


        public bool IsThresholdingAlphaInterpolationEnabled
        {
            get
            {
                return _isThresholdingAlphaInterpolationEnabled;
            }

            set
            {
                if (_isThresholdingAlphaInterpolationEnabled != value)
                {
                    _isThresholdingAlphaInterpolationEnabled = value;
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
                }
            }
        }

        /// <summary>
        /// テクスチャマップ配列を取得します。
        /// 前詰めで、有効なテクスチャのみの配列です。
        /// 以前は有効ではないテクスチャを含む配列を返していました。
        /// </summary>
        public IMaterialTexMap[] IMaterialTexMapSet
        {
            get
            {
                List<IMaterialTexMap> texs = new List<IMaterialTexMap>();
                foreach( IMaterialTexMap tex in _materialTexMaps )
                {
                    if( tex.IsActive )
                    {
                        texs.Add( tex );
                    }
                    else
                    {
                        break;
                    }
                }
                return texs.ToArray();
            }
        }

        /// <summary>
        /// テクスチャマップ配列を取得します。
        /// </summary>
        public MaterialTexMap[] MaterialTexMapSet {get{return  _materialTexMaps;}}

        /// <summary>
        /// テクスチャ行列配列を取得します。
        /// </summary>
        public ITexMtx[] ITexMtxSet
        {
            get
            {
                ITexMtx[]  result = new ITexMtx[MaxNumMatTexMtx];
                _textureMtxSet.CopyTo( result, 0 );
                return result;
            }
        }

        /// <summary>
        /// マテリアル名
        /// </summary>
        public string MaterialName
        {
            get
            {
                return _name;
            }

            set
            {
                if (_name != value)
                {
                    _name = value;
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
                }
            }
        }

        /// <summary>
        /// ユーザが指定した制限を満たさない、テクスチャ座標指定を調整します。
        /// </summary>
        void AdujustMaterialTexMapTexCoord_()
        {
            foreach( MaterialTexMap texMap in _materialTexMaps )
            {
                if( (int)texMap.TexGen.Method >= _userMaxNumTexCoord )
                {
                    texMap.TexGen.Method = LETexGenMethod.UV0;
                }
            }
        }

        /// <summary>
        /// 設定可能なテクスチャ座標の数
        /// </summary>
        public int UserMaxNumTexCoord
        {
            get{ return _userMaxNumTexCoord;}
            set
            {
                if( value <= MaxNumTexCoord )
                {
                    if( _userMaxNumTexCoord != value )
                    {
                        _userMaxNumTexCoord = value;

                        AdujustMaterialTexMapTexCoord_();

                        NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
                    }
                }
            }
        }

        public AnmAttribute BlackColorAnmAttr
        {
            get { return FindAttributeByIdx(0); }
        }

        public AnmAttribute WhiteColorAnmAttr
        {
            get { return FindAttributeByIdx(1); }
        }

        public IAnmAttribute BlackColorIAnmAttr
        {
            get{ return BlackColorAnmAttr;}
        }

        public IAnmAttribute WhiteColorIAnmAttr
        {
            get{ return WhiteColorAnmAttr;}
        }

        #endregion ------------ IMaterial メンバ ------------
    }

    #endregion ------------- マテリアル -------------
}
