﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace LECore.Structures
{
    /// <summary>
    /// レイアウトファイル(アプリレイヤー固有のルールで、LECoreDocumentを関連付けるクラス)
    /// </summary>
    public class LayoutDocument
    {
        private readonly ISubScene _subScene;

        private readonly LECoreDocument _document = null;
        private LECoreDocument _animDocument = null;

        /// <summary>
        /// サブシーン
        /// </summary>
        public ISubScene ISubScene
        {
            get { return _subScene; }
        }

        /// <summary>
        /// サブシーンが更新されたか？
        /// </summary>
        public bool IsUpdated
        {
            get { return _document.CheckUpdated(_subScene); }
        }

        /// <summary>
        /// 名前付けされたか？
        /// </summary>
        public bool IsFileSaved { get { return _document.IsFileSaved; } }

        /// <summary>
        /// 最後に保存されたファイルパスを取得します。
        /// </summary>
        public string LastSavedFilePath { get { return _document.FilePath; } }

        /// <summary>
        /// 最後の保存時間
        /// </summary>
        public DateTime LastSavedTime { get { return _document.LastSavedTime; } }

        /// <summary>
        /// 最後に読み込まれた時間
        /// </summary>
        public DateTime LastReloadedTime { get { return _document.LastReloadedTime; } }

        /// <summary>
        /// ファイル名を取得します。
        /// </summary>
        public string LastSavedFileNameWithoutExtension
        {
            get { return Path.GetFileNameWithoutExtension(LastSavedFilePath); }
        }

        /// <summary>
        /// ファイル名を取得します。
        /// </summary>
        public string LastSavedFileName
        {
            get { return LastSavedFileNameWithoutExtension + AppConstants.LayoutFileExt; }
        }

        /// <summary>
        /// コピーコンストラクタ
        /// </summary>
        public LayoutDocument(LayoutDocument other, DateTime time)
            : this(other._subScene, other._document, time)
        {
        }

        /// <summary>
        /// コンストラクタ（LECore 内部でだけ利用される）
        /// </summary
        internal LayoutDocument(ISubScene subScene, LECoreDocument document, DateTime currentTime)
        {
            _subScene = subScene;
            ((SubScene)_subScene).OwnerDocument = this;
            _document = document;

            _document.SetStateReloaded(currentTime);

            this.UpdateSaveFileState_(document.FilePath, currentTime);

            this.ResetModifyState();
        }

        /// <summary>
        /// 最新状態になったことを設定します。
        /// </summary>
        public void SetStateReloaded()
        {
            _document.SetStateReloaded(DateTime.Now);
        }

        /// <summary>
        /// セーブされた状態に設定します。
        /// saveFileName は絶対パスを指定する必要があります。
        /// </summary>
        public void SetStateSaved(string saveFileName)
        {
            // 変更まえに、実在するファイルを読み込んでいる状態で別名で保存する場合、
            // 前に読んでいたファイルは ツール内で Unload 状態とする。
            if (_document.IsFileSaved && _document.FilePath != saveFileName)
            {
                LayoutEditorCore.NotifyUnloadLECoreDocument(_document);
                if (_animDocument != null)
                {
                    LayoutEditorCore.NotifyUnloadLECoreDocument(_animDocument);
                }
            }

            // ファイルの状態を更新する。
            DateTime dateTimeNow = DateTime.Now;
            UpdateSaveFileState_(saveFileName, dateTimeNow);
        }

        /// <summary>
        /// 更新する。
        /// </summary>
        void UpdateSaveFileState_(string saveFileName, DateTime dateTimeNow)
        {
            _document.SetStateSaved(_subScene, saveFileName, dateTimeNow);

            string animFilePath = Path.ChangeExtension(_document.FilePath, AppConstants.AnimationFileExt);

            if (_animDocument == null && File.Exists(animFilePath))
            {
                _animDocument = LayoutEditorCore.MakeLECoreDocument(animFilePath);
            }

            if (_animDocument != null)
            {
                _animDocument.SetStateSaved(_subScene, animFilePath, dateTimeNow);
            }
        }

        /// <summary>
        ///
        /// </summary>
        public void ResetModifyState()
        {
            _document.ResetModifyState(_subScene);
        }

        /// <summary>
        ///
        /// </summary>
        public void ForceModifyStateDirty()
        {
            _document.ForceModifyStateDirty();
        }

        /// <summary>
        ///
        /// </summary>
        public bool TryUnloadDocument()
        {
            // アンロードは不要だった。
            if (!File.Exists(_document.FilePath))
            {
                return true;
            }

            bool success = LayoutEditorCore.NotifyUnloadLECoreDocument(_document);
            if (_animDocument != null)
            {
                if (!File.Exists(_animDocument.FilePath))
                {
                    return success;
                }

                success |= LayoutEditorCore.NotifyUnloadLECoreDocument(_animDocument);
            }

            return success;
        }
    }
}
