﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures
{
    using System.Collections.Generic;
    using Core;
    using LECore.Structures.SerializableObject.Lyt;

    /// <summary>
    /// ウインドウペインです。
    /// </summary>
    internal class LEWindow :
        LEDataNode,
        IPaneExParamater,
        IDrawable,
        ILEWindow,
        IVertexColor4Settable,
        IRevHWMaterialHolder,
        IDisposable
    {
        #region 定数、静的メソッド
        const int PartsCount = 9;
        public const string WindowNodeName = "Window";

        /// <summary>
        /// マテリアル名の接尾子
        /// </summary>
        static string[] _WindowMaterialNamePostFix = new string[]
          {
              "C ", "LT", "RT", "LB", "RB", "T ", "B ", "L ", "R "
          };

        /// <summary>
        /// ウインドウ部位描画のテクスチャ座標で、左上マテリアルを利用するかどうか
        /// </summary>
        enum UseLTForTexCoord
        {
            None,
            Enabled,
        }

        /// <summary>
        /// ウインドウ部位描画のテクスチャコンバイナ（ブレンド）で、左上マテリアルを利用するかどうか
        /// </summary>
        enum UseLTForBlend
        {
            None,
            Enabled,
        }

        /// <summary>
        /// マテリアル名を取得します。
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        public static string GetMaterialName_(IPane ownerPane, LEWindowPartsID id)
        {
            Debug.Assert(ownerPane != null);

            return string.Format("{0}{1}",
                                  ownerPane.PaneName,
                                  _WindowMaterialNamePostFix[(int)id]);
        }

        /// <summary>
        /// マテリアル名からLEWindowPartsIDを取得します。
        /// </summary>
        public static LEWindowPartsID GetLEWindowPartsIDByName(string materialName)
        {
            for (int i = 0; i < LEWindow._WindowMaterialNamePostFix.Length; i++)
            {
                if (materialName.EndsWith(_WindowMaterialNamePostFix[i]))
                {
                    return (LEWindowPartsID)i;
                }
            }
            return LEWindowPartsID.Content;
        }
        #endregion 定数

        #region ------------ フィールド

        // ウインドウ種類
        WindowKind _windowKind = WindowKind.Around;

        // 頂点カラーをペイン全体に適用するか
        bool _useVtxColorForAllPane = true;

        // ウインドウが持つ画像の種類
        int _numTexture = 2;

        // ウインドウが詳細なマテリアルを使用するかどうか
        bool[] _useDetailedMaterials = new bool[PartsCount];
        // ウインドウが持つ画像のマテリアル
        Material[] _materials = new Material[9];
        // ウインドウが持つ画像のマテリアル(Revolution固有パラメータ)
        RevHWMaterial[] _revMaterials = new RevHWMaterial[PartsCount];
        // ウインドウが持つ画像のフリップ・回転状態
        // (0番要素はコンテンツ領域用のため、使用されません。)
        LEWindowFrameFlipRotate[] _flipRotateState = new LEWindowFrameFlipRotate[PartsCount];

        // ------------ コンテンツ領域関連
        // コンテンツ領域拡大量
        ILEWindowContentsInflation _contentsInflation = new ILEWindowContentsInflation(0.0f, 0.0f, 0.0f, 0.0f);

        // コンテンツ領域で使用するテクスチャ座標数
        int _numContentsTexCoord = 0;

        // コンテンツ領域で使用するテクスチャ座標
        TexCoord4[] _contetnsTexCoord4Set = new TexCoord4[MaterialDefinition.MaxNumTexCoord];

        // コンテンツの4隅の色
        readonly VtxColor4Attribute _cornerColorsAttr = null;
        readonly RectAttribute _frameSizeAttr = null;
        bool _isFrameSizeSpecified = false;

        // 既定の頂点色
        static readonly Color[] _defaultVtxColor = new Color[] { Color.White, Color.White, Color.White, Color.White };

        // ウインドウペイン本体への参照
        readonly Pane _ownerPane = null;

        //
        private bool useLTMaterial = true;

        private bool notDrawContent = false;

        #endregion ------------ フィールド

        #region ------------ プロパティ

        /// <summary>
        /// int => Enum(不正な入力は Content となります。)
        /// </summary>
        public LEWindowPartsID IntToPartsID(int id)
        {
            try
            {
                return (LEWindowPartsID)id;
            }
            catch
            {
                Debug.Assert(false);
                return LEWindowPartsID.Content;
            }
        }

        /// <summary>
        /// Enum => int(不正な入力は 0 となります。)
        /// </summary>
        public int PartsIDToInt(LEWindowPartsID id)
        {
            try
            {
                return (int)id;
            }
            catch
            {
                Debug.Assert(false);
                return 0;
            }
        }

        /// <summary>
        /// コンテンツ領域で使用するテクスチャ枚数を取得します。
        /// </summary>
        int _NumContentsTextures
        {
            get
            {
                IMaterialTexMap[] texMaps = GetMaterial(LEWindowPartsID.Content).IMaterialTexMapSet;
                return texMaps.Length;
            }
        }

        /// <summary>
        /// テクスチャマネージャ
        /// </summary>
        ITextureMgr _ITextureMgr
        {
            get { return this.GetOwnerSubScene() != null ? this.GetOwnerSubScene().ITextureMgr : null; }
        }

        /// <summary>
        /// コンテンツを描画しないフラグ
        /// </summary>
        public bool NotDrawContent
        {
            get
            {
                return notDrawContent;
            }

            set
            {
                if (notDrawContent != value)
                {
                    notDrawContent = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public bool IsFrameSizeOrFrameSizeCurveSpecified
        {
            get
            {
                return this._frameSizeAttr.CheckHaveKeyRecursiveInAllTag() || _isFrameSizeSpecified;
            }
        }

        internal bool IsFrameSizeSpecified
        {
            get
            {
                return _isFrameSizeSpecified;
            }
            set
            {
                _isFrameSizeSpecified = value;
            }
        }

        public int NumTexCoord
        {
            get { return this.NumContentsTexCoord; }
        }

        public TexCoord4 GetTexCoord4(int index)
        {
            return GetContentsTexCoord4(index);
        }

        public IMaterialTexMap[] IMaterialTexMapSet
        {
            get { return GetCornerTextureMaterial_(LEWindowPartsID.Content).IMaterialTexMapSet; }
        }

        #endregion ------------ プロパティ

        /// <summary>
        /// アトリビュートの有効状態を更新します。
        /// </summary>
        void UpdateAttributeActiveState_()
        {
            for (int i = 0; i < _numTexture; i++)
            {
                _materials[i].IsActiveAttribute = true;
                _revMaterials[i].IsActiveAttribute = true;
            }
            for (int i = _numTexture; i < _materials.Length; i++)
            {
                _materials[i].IsActiveAttribute = false;
                _revMaterials[i].IsActiveAttribute = false;
            }

            // コンテント領域
            {
                int index = (int)LEWindowPartsID.Content;
                _materials[index].IsActiveAttribute = _windowKind != WindowKind.HorizontalNoContent;
                _revMaterials[index].IsActiveAttribute = _windowKind != WindowKind.HorizontalNoContent;
            }

            // アニメーションの有効無効
            AnmAttribute sizeYAnmAttr = (this.OwnerPane.IPaneAttribute.SizeIAttr as AnmAttribute).FindAttributeByIdx(1);
            sizeYAnmAttr.IsActiveAttribute = _windowKind == WindowKind.Around;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEWindow(Pane ownerPane)
            : base(ownerPane, WindowNodeName)
        {
            Debug.Assert(ownerPane != null);

            // ウインドウペインの初期化
            _ownerPane = ownerPane;
            _ownerPane.BindPaneExData(this);

            _cornerColorsAttr = new VtxColor4Attribute(this);
            _frameSizeAttr = new RectAttribute(this, "FrameSize");

            // マテリアルを初期化します。
            foreach (LEWindowPartsID matID in Enum.GetValues(typeof(LEWindowPartsID)))
            {
                int i = (int)matID;

                _useDetailedMaterials[i] = false;
                _materials[i] = new Material(this, GetMaterialName_(_ownerPane, matID));
                _revMaterials[i] = new RevHWMaterial(this, _materials[i]);

                _materials[i].Description = _WindowMaterialNamePostFix[i];
                _revMaterials[i].Description = _WindowMaterialNamePostFix[i];

                _flipRotateState[i] = LEWindowFrameFlipRotate.None;
            }

            // コンテンツ領域が使用するテクスチャ座標を初期化します。
            for (int i = 0; i < _contetnsTexCoord4Set.Length; i++)
            {
                _contetnsTexCoord4Set[i] = new TexCoord4();
            }

            UpdateAttributeActiveState_();
        }

        /// <summary>
        /// リソースの開放をします。
        /// </summary>
        public void Dispose()
        {
            _cornerColorsAttr.Dispose();

            foreach (LEWindowPartsID matID in Enum.GetValues(typeof(LEWindowPartsID)))
            {
                int i = (int)matID;
                _materials[i].Dispose();
                _revMaterials[i].Dispose();
            }
        }

        #region IDrawable

        #region 描画処理を行う関数群

        /// <summary>
        /// 参照マテリアル変換テーブル
        /// マテリアル使用数２用
        ///
        /// すべてのフレーム用マテリアルは、
        /// CornerLT を参照するように変換されます。
        /// </summary>
        static readonly LEWindowPartsID[] _tbl_TexNum2_ =
        {
            LEWindowPartsID.Content,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerLT
        };

        /// <summary>
        /// 参照マテリアル変換テーブル
        /// マテリアル使用数5用
        ///
        /// フレーム用マテリアルの辺の部分は、
        /// 角のマテリアルを参照するように変換されます。
        /// </summary>
        static readonly LEWindowPartsID[] _tbl_TexNum5_ =
        {
            LEWindowPartsID.Content,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerRT,
            LEWindowPartsID.CornerLB,
            LEWindowPartsID.CornerRB,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerRB,
            LEWindowPartsID.CornerLB,
            LEWindowPartsID.CornerRT
        };

        /// <summary>
        /// 参照マテリアル変換テーブル
        /// マテリアル使用数5(水平)用
        ///
        /// 左角か右角を参照するように変換されます。
        /// </summary>
        static readonly LEWindowPartsID[] _tbl_TexNum5H_ =
        {
            LEWindowPartsID.Content,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerRT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerRT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerRT,
            LEWindowPartsID.CornerLT,
            LEWindowPartsID.CornerRT
        };

        static public int GetInternalWindowPartsID(ILEWindow window, LEWindowPartsID partsID)
        {
            switch (window.NumTexture)
            {
                case 2: partsID = _tbl_TexNum2_[(int)partsID]; break;
                case 3: partsID = _tbl_TexNum5H_[(int)partsID]; break;
                case 5: partsID = _tbl_TexNum5_[(int)partsID]; break;
                case 9: break;  // 変換テーブルは使用されません。
            }

            return (int)partsID;
        }

        /// <summary>
        /// マテリアルを取得します。
        /// </summary>
        Material GetCornerTextureMaterial_(LEWindowPartsID partsID)
        {

            return _materials[GetInternalWindowPartsID(this, partsID)];
        }

        /// <summary>
        /// マテリアルを取得します。
        /// </summary>
        RevHWMaterial GetCornerTextureMaterialDetail_(LEWindowPartsID partsID)
        {
            return _revMaterials[GetInternalWindowPartsID(this, partsID)];
        }

        /// <summary>
        /// ウインドウパーツの大きさを取得します。
        /// </summary>
        FVec2 GetPartsSize_(LEWindowPartsID partsID)
        {
            float contT = GetPartsTextureSize_(LEWindowPartsID.CornerLT).Y;
            float contB = GetPartsPosition_(LEWindowPartsID.CornerRB).Y;

            float contL = GetPartsTextureSize_(LEWindowPartsID.CornerLB).X;
            float contR = GetPartsPosition_(LEWindowPartsID.CornerRT).X;
            float contW = Math.Max(contR - contL, 0.0f);
            float contH = Math.Max(contB - contT, 0.0f);
            //

            switch (partsID)
            {
                case LEWindowPartsID.Content:
                    FVec2 result = new FVec2(contR - contL, contB - contT);
                    result.X += ContentsInflation.Left + ContentsInflation.Right;
                    result.Y += ContentsInflation.Top + ContentsInflation.Bottom;
                    return result;

                case LEWindowPartsID.FrameT:
                    {
                        FVec2 lt = GetPartsTextureSize_(LEWindowPartsID.CornerLT);
                        return new FVec2(contW, lt.Y);
                    }

                case LEWindowPartsID.FrameB:
                    {
                        FVec2 rb = GetPartsTextureSize_(LEWindowPartsID.CornerRB);
                        return new FVec2(contW, rb.Y);
                    }

                case LEWindowPartsID.CornerLB:
                    {
                        FVec2 lb = GetPartsTextureSize_(LEWindowPartsID.CornerLB);
                        FVec2 rb = GetPartsTextureSize_(LEWindowPartsID.CornerRB);
                        return new FVec2(lb.X, rb.Y);
                    }

                case LEWindowPartsID.CornerLT:
                    {
                        FVec2 lt = GetPartsTextureSize_(LEWindowPartsID.CornerLT);
                        FVec2 lb = GetPartsTextureSize_(LEWindowPartsID.CornerLB);
                        return new FVec2(lb.X, lt.Y);
                    }

                case LEWindowPartsID.FrameL:
                    {
                        FVec2 lb = GetPartsTextureSize_(LEWindowPartsID.CornerLB);
                        return new FVec2(lb.X, contH);
                    }

                case LEWindowPartsID.FrameR:
                    {
                        FVec2 rt = GetPartsTextureSize_(LEWindowPartsID.CornerRT);
                        return new FVec2(rt.X, contH);
                    }

                case LEWindowPartsID.CornerRT:
                    {
                        FVec2 lt = GetPartsTextureSize_(LEWindowPartsID.CornerLT);
                        FVec2 rt = GetPartsTextureSize_(LEWindowPartsID.CornerRT);
                        return new FVec2(rt.X, lt.Y);
                    }

                case LEWindowPartsID.CornerRB:
                    {
                        FVec2 rt = GetPartsTextureSize_(LEWindowPartsID.CornerRT);
                        FVec2 rb = GetPartsTextureSize_(LEWindowPartsID.CornerRB);
                        return new FVec2(rt.X, rb.Y);
                    }

                default:
                    return new FVec2(FVec2.Empty);
            }
        }

        /// <summary>
        /// ウインドウパーツのテクスチャサイズを取得します。
        ///
        /// 1枚目のテクスチャの大きさをパーツの大きさとして扱うこととします。
        /// </summary>
        FVec2 GetPartsTextureSize_(LEWindowPartsID partsID)
        {
            if (this.IsFrameSizeOrFrameSizeCurveSpecified)
            {
                var fs = this.FrameSize;
                switch (partsID)
                {
                    case LEWindowPartsID.CornerLT: return new FVec2(fs.Left, fs.Top);
                    case LEWindowPartsID.CornerRT: return new FVec2(fs.Right, fs.Top);
                    case LEWindowPartsID.CornerLB: return new FVec2(fs.Left, fs.Bottom);
                    case LEWindowPartsID.CornerRB: return new FVec2(fs.Right, fs.Bottom);
                    case LEWindowPartsID.FrameT: return new FVec2(fs.Left, fs.Top);
                    case LEWindowPartsID.FrameB: return new FVec2(fs.Left, fs.Bottom);
                    case LEWindowPartsID.FrameL: return new FVec2(fs.Left, fs.Top);
                    case LEWindowPartsID.FrameR: return new FVec2(fs.Right, fs.Bottom);
                }
            }

            if (_ITextureMgr != null)
            {
                // テクスチャが設定済みで...
                IMaterial mtl = GetCornerTextureMaterial_(partsID);
                if (mtl != null && mtl.IMaterialTexMapSet.Length != 0)
                {
                    // 正しくテクスチャ情報が取得できたら...
                    MaterialTexMap texMap = mtl.IMaterialTexMapSet[0] as MaterialTexMap;
                    ITextureImage texImg = texMap.TryFindTextureImage(OwnerPane.OwnerSubScene);
                    if (texImg != null)
                    {
                        // テクスチャの大きさをパーツの大きさとして返します。
                        FVec2 result = new FVec2(texImg.Size);
                        return result;
                    }
                }
            }

            // ペインの20%をサイズとします。
            int w = (int)Math.Max(_ownerPane.Width / 5.0f, ILEWindowHelper.InitialMinFrameSize);
            int h = (int)Math.Max(_ownerPane.Height / 5.0f, ILEWindowHelper.InitialMinFrameSize);

            return new FVec2(w, h);
        }

        /// <summary>
        /// ウインドウパーツの位置を取得します。
        /// // TODO:　FVec2と一緒に、アンカー位置を返すようにします。
        /// </summary>
        FVec2 GetPartsPosition_(LEWindowPartsID partsID)
        {
            FVec2 result = new FVec2();

            // Xについて
            switch (partsID)
            {
                case LEWindowPartsID.CornerLT:
                case LEWindowPartsID.FrameL:
                case LEWindowPartsID.CornerLB:
                default:
                    result.X = 0.0f;
                    break;
                case LEWindowPartsID.FrameT:
                case LEWindowPartsID.FrameB:
                    result.X = Math.Min(
                        GetPartsTextureSize_(LEWindowPartsID.CornerLB).X,
                        _ownerPane.RenderingWidth - GetPartsTextureSize_(LEWindowPartsID.CornerRB).X);

                    break;

                case LEWindowPartsID.Content:
                    result.X = GetPartsTextureSize_(LEWindowPartsID.CornerLB).X - ContentsInflation.Left;
                    break;

                case LEWindowPartsID.CornerRT:
                case LEWindowPartsID.FrameR:
                case LEWindowPartsID.CornerRB:
                    result.X = _ownerPane.RenderingWidth - GetPartsTextureSize_(LEWindowPartsID.CornerRT).X;
                    break;
            }

            // Yについて
            switch (partsID)
            {
                case LEWindowPartsID.CornerLT:
                case LEWindowPartsID.FrameT:
                case LEWindowPartsID.CornerRT:
                default:
                    result.Y = 0.0f;
                    break;

                case LEWindowPartsID.FrameL:
                case LEWindowPartsID.FrameR:
                    result.Y = Math.Min(
                        GetPartsTextureSize_(LEWindowPartsID.CornerLT).Y,
                        _ownerPane.RenderingHeight - GetPartsTextureSize_(LEWindowPartsID.CornerLB).Y);

                    break;

                case LEWindowPartsID.Content:
                    result.Y = GetPartsTextureSize_(LEWindowPartsID.CornerLT).Y - ContentsInflation.Top;
                    break;

                case LEWindowPartsID.CornerLB:
                case LEWindowPartsID.FrameB:
                case LEWindowPartsID.CornerRB:
                    result.Y = _ownerPane.RenderingHeight - GetPartsTextureSize_(LEWindowPartsID.CornerRB).Y;
                    break;
            }

            return result;
        }

        /// <summary>
        ///
        /// </summary>
        private int GetTexCoordIndex(LETexGenMethod method)
        {
            if (method == LETexGenMethod.UV0) { return 0; }
            if (method == LETexGenMethod.UV1) { return 1; }
            if (method == LETexGenMethod.UV2) { return 2; }

            Ensure.Operation.True(false, "[LETexGenMethod] Must be UVMapping.");

            return -1;
        }

        /// <summary>
        /// ウインドウパーツのUV座標を計算します。
        /// </summary>
        TexCoord4 GetWindowPartsTexCoord_(
            int numTexture, FVec2 texImgSize, IMaterialTexMap texMap, FVec2 size, LEWindowPartsID partsID)
        {
            // コンテンツ領域は固定値を返します。
            if (partsID == LEWindowPartsID.Content)
            {
                return texMap.ITexGen.IsUVMethod ?
                    GetContentsTexCoord4(GetTexCoordIndex(texMap.ITexGen.Method)) : null;
            }

            TexCoord4 texCoord4 = new TexCoord4();

            // フリップ状態に応じて、UV値を変更します。
            LEWindowFrameFlipRotate flipState = GetMaterialFlipRotateState(partsID);

            switch (flipState)
            {
                case LEWindowFrameFlipRotate.FlipH: texCoord4.FlipHorizontal(); break;
                case LEWindowFrameFlipRotate.FlipV: texCoord4.FlipVertical(); break;
                case LEWindowFrameFlipRotate.Rotate90: texCoord4.Rotate90(); break;
                case LEWindowFrameFlipRotate.Rotate180: texCoord4.Rotate180(); break;
                case LEWindowFrameFlipRotate.Rotate270: texCoord4.Rotate270(); break;
            }

            // パーツ種類に応じてスケール原点を計算します。
            FVec2 uvOrigin = GetWindoePartUVScalingOrigin_(partsID, texCoord4);

            // スケールを計算します。フリップ状態に応じて、スケール軸をスワップします。
            FVec2 uvScale;
            switch (flipState)
            {
                default:
                case LEWindowFrameFlipRotate.None:
                case LEWindowFrameFlipRotate.Rotate180:
                case LEWindowFrameFlipRotate.FlipH:
                case LEWindowFrameFlipRotate.FlipV:
                    uvScale = new FVec2(size.X / texImgSize.X, size.Y / texImgSize.Y);
                    break;

                case LEWindowFrameFlipRotate.Rotate90:
                case LEWindowFrameFlipRotate.Rotate270:
                    uvScale = new FVec2(size.Y / texImgSize.X, size.X / texImgSize.Y);
                    break;
            }

            texCoord4.Scale(uvScale.X, uvScale.Y, uvOrigin.X, uvOrigin.Y);
            return texCoord4;
        }

        #endregion

        /// <summary>
        /// フリップ状況を考慮した、テクスチャサイズを取得します。
        /// </summary>
        FVec2 GetPartsTextureSizeFlip_(LEWindowPartsID partsID)
        {
            FVec2 size = GetPartsTextureSize_(partsID);
            switch (GetMaterialFlipRotateState(partsID))
            {
                case LEWindowFrameFlipRotate.Rotate90:
                case LEWindowFrameFlipRotate.Rotate270:
                    return new FVec2(size.Y, size.X);
                default: return size;
            }
        }

        /// <summary>
        /// パーツテクスチャのサイズが正しいか調べます。
        /// </summary>
        public bool IsFrameTextureSizeValid()
        {
            // 初期状態で不正なサイズ設定は無い
            if (IsMaterialInitialState_())
            {
                return true;
            }

            // 縦方向のサイズについて

            // 上横一列のサイズが完全に一致しない...
            if (GetPartsTextureSizeFlip_(LEWindowPartsID.CornerLT).Y != GetPartsTextureSizeFlip_(LEWindowPartsID.FrameT).Y ||
                GetPartsTextureSizeFlip_(LEWindowPartsID.FrameT).Y != GetPartsTextureSizeFlip_(LEWindowPartsID.CornerRT).Y)
            {
                return false;
            }

            // 下横一列
            if (GetPartsTextureSizeFlip_(LEWindowPartsID.CornerLB).Y != GetPartsTextureSizeFlip_(LEWindowPartsID.FrameB).Y ||
                GetPartsTextureSizeFlip_(LEWindowPartsID.FrameB).Y != GetPartsTextureSizeFlip_(LEWindowPartsID.CornerRB).Y)
            {
                return false;
            }

            // 横方向のサイズについて
            // 上横一列のサイズが完全に一致しない...
            if (GetPartsTextureSizeFlip_(LEWindowPartsID.CornerLT).X != GetPartsTextureSizeFlip_(LEWindowPartsID.FrameL).X ||
                GetPartsTextureSizeFlip_(LEWindowPartsID.FrameL).X != GetPartsTextureSizeFlip_(LEWindowPartsID.CornerLB).X)
            {
                return false;
            }

            // 下横一列
            if (GetPartsTextureSizeFlip_(LEWindowPartsID.CornerRT).X != GetPartsTextureSizeFlip_(LEWindowPartsID.FrameR).X ||
                GetPartsTextureSizeFlip_(LEWindowPartsID.FrameR).X != GetPartsTextureSizeFlip_(LEWindowPartsID.CornerRB).X)
            {
                return false;
            }

            // 問題なし
            return true;
        }

        /// <summary>
        /// ウインドウパーツの矩形を計算します。
        /// </summary>
        void GetWindowPartsRectangle_(
            int numTexture, LEWindowPartsID partsType, out FVec2 pos, out FVec2 size)
        {
            // サイズを取得
            pos = new FVec2();
            size = new FVec2();

            // テクスチャ枚数 9枚の場合
            if (numTexture == 9)
            {
                // 9枚の場合は、単純な計算です。
                pos = GetPartsPosition_(partsType);
                size = GetPartsSize_(partsType);
                return;
            }

            // テクスチャ枚数 9枚の場合以外
            // パーツごとに
            Debug.Assert(numTexture == 2 || numTexture == 3 || numTexture == 5);
            switch (partsType)
            {
                case LEWindowPartsID.Content:
                    {
                        pos = GetPartsPosition_(partsType);
                        size = GetPartsSize_(partsType);
                        break;
                    }
                case LEWindowPartsID.CornerLT:
                    {
                        pos = GetPartsPosition_(partsType);

                        float frameW = GetPartsSize_(LEWindowPartsID.FrameT).X;

                        size = GetPartsSize_(LEWindowPartsID.CornerLT);
                        size = new FVec2(size.X + frameW, size.Y);
                        break;
                    }
                case LEWindowPartsID.CornerRB:
                    {
                        pos = GetPartsPosition_(LEWindowPartsID.FrameB);

                        float frameW = GetPartsSize_(LEWindowPartsID.FrameB).X;

                        size = GetPartsSize_(LEWindowPartsID.CornerRB);
                        size = new FVec2(size.X + frameW, size.Y);
                        break;
                    }
                case LEWindowPartsID.CornerRT:
                    {
                        pos = GetPartsPosition_(partsType);

                        float frameH = GetPartsSize_(LEWindowPartsID.FrameR).Y;

                        size = GetPartsSize_(LEWindowPartsID.CornerRT);
                        size = new FVec2(size.X, size.Y + frameH);
                        break;
                    }
                case LEWindowPartsID.CornerLB:
                    {
                        pos = GetPartsPosition_(LEWindowPartsID.FrameL);

                        float frameH = GetPartsSize_(LEWindowPartsID.FrameL).Y;

                        size = GetPartsSize_(LEWindowPartsID.CornerLB);
                        size = new FVec2(size.X, size.Y + frameH);
                        break;
                    }
                default:
                    {
                        Debug.Assert(false);
                        break;
                    }
            }
        }

        /// <summary>
        /// ウインドウパーツを描画します。
        /// </summary>
        void DrawWindowParts_(
            IRenderer renderer,
            LEWindowPartsID partsType,
            int numTexture,
            DrawableOption option,
            UseLTForTexCoord useLTMaterialForTexCoord,
            UseLTForBlend useLTMaterialForBlend)
        {
            DrawWindowParts_(renderer, partsType, partsType, numTexture, option, useLTMaterialForTexCoord, useLTMaterialForBlend);
        }

        /// <summary>
        /// 頂点カラーを算出します。
        /// </summary>
        private static Color GetInternalPointColor_(Color[] cols, FVec2 subRectPoint)
        {
            Color colorL = ColorHelper.Blend(cols[0], cols[3], subRectPoint.Y);
            Color colorR = ColorHelper.Blend(cols[1], cols[2], subRectPoint.Y);

            return ColorHelper.Blend(colorL, colorR, subRectPoint.X);
        }

        /// <summary>
        /// 頂点カラーを算出します。
        /// </summary>
        private Color[] GetWindowPartsVertexColor_(FVec2 paneSize, FVec2 pos, FVec2 size)
        {
            if (paneSize.X == 0.0f || paneSize.Y == 0.0f)
            {
                return _cornerColorsAttr.ColorSet;
            }

            // 0 - 1 に正規化
            FVec2 nrmPos = new FVec2(pos.X / paneSize.X, pos.Y / paneSize.Y);
            FVec2 nrmSize = new FVec2(size.X / paneSize.X, size.Y / paneSize.Y);

            Color[] cols = _cornerColorsAttr.ColorSet;
            Color[] resultCols = new Color[4];

            resultCols[0] = GetInternalPointColor_(cols, nrmPos);
            resultCols[1] = GetInternalPointColor_(cols, new FVec2(nrmPos.X + nrmSize.X, nrmPos.Y));
            resultCols[2] = GetInternalPointColor_(cols, new FVec2(nrmPos.X + nrmSize.X, nrmPos.Y + nrmSize.Y));
            resultCols[3] = GetInternalPointColor_(cols, new FVec2(nrmPos.X, nrmPos.Y + nrmSize.Y));

            return resultCols;
        }

        /// <summary>
        /// ウインドウパーツを描画します。
        /// </summary>
        void DrawWindowParts_(
            IRenderer renderer,
            LEWindowPartsID texPartsType,
            LEWindowPartsID partsType,
            int numTexture,
            DrawableOption option,
            UseLTForTexCoord useLTMaterialForTexCoord,
            UseLTForBlend useLTMaterialForBlend)
        {
            // 位置とサイズを取得
            FVec2 pos;
            FVec2 size;
            GetWindowPartsRectangle_(numTexture, partsType, out pos, out size);

            // 小さい場合は書かない
            if (size.X <= float.Epsilon || size.Y <= float.Epsilon)
            {
                return;
            }

            // マテリアルカラーの設定
            // HSV補正処理を施して、白黒補間を設定します。
            {
                LEWindowPartsID colrBlendType = !this.GetMaterialMatColorEditable(texPartsType) ? LEWindowPartsID.CornerLT : texPartsType;
                IMaterial colorBlendMaterial = GetCornerTextureMaterial_(colrBlendType);
                renderer.SetMatrialColorBlend(colorBlendMaterial.WhiteColor, colorBlendMaterial.BlackColor, colorBlendMaterial.IsThresholdingAlphaInterpolationEnabled);
            }

            // マテリアルを取得
            IMaterial texMaterial = GetCornerTextureMaterial_(texPartsType);

            // UseLTMaterial のフレームでのマテリアル
            IMaterial material = GetCornerTextureMaterial_(useLTMaterialForTexCoord == UseLTForTexCoord.Enabled ? LEWindowPartsID.CornerLT : texPartsType);
            RevHWMaterial materialDetail = GetCornerTextureMaterialDetail_(useLTMaterialForBlend == UseLTForBlend.Enabled ? LEWindowPartsID.CornerLT : texPartsType);

            // ブレンド設定
            {
                var pe = materialDetail.PEData;
                PEBlend colBlend = pe.UseDefaultBlendSettings ? PEBlend.DefaultBlendColor : pe.Blend;
                PEBlend alpBlend = pe.UseColorBlendSettingsForAlpha ? PEBlend.DefaultBlendAlpha : pe.BlendAlpha;

                renderer.SetBlendMode(
                    colBlend.Type,
                    colBlend.SrcFactor, colBlend.DstFactor, colBlend.BlendOp,
                    alpBlend.SrcFactor, alpBlend.DstFactor, alpBlend.BlendOp);

                renderer.SetAlphaCompare(!pe.UseDefaultAlphaTestSettings, pe.ACompare.Comp, pe.ACompare.Ref);
            }

            // 頂点カラー設定
            Color[] color = null;
            if (this.UseVtxColorForAllWindow)
            {
                color = GetWindowPartsVertexColor_(this.OwnerPane.Size.FVec2, pos, size);
            }
            else
            {
                color = texPartsType == LEWindowPartsID.Content ? _cornerColorsAttr.ColorSet : _defaultVtxColor;
            }

            // すべてのマテリアルテクスチャについてテクスチャ毎の設定を行います。
            IMaterialTexMap[] texMapSet = texMaterial.IMaterialTexMapSet;
            TexCoord4[] texCooors = texMapSet.Length != 0 ? new TexCoord4[Material.MaxNumMatTexture] : null;

            // フレームについて、テクスチャが存在しない場合は一切の描画を行わない。
            // （コンテンツ領域は、頂点カラーだけの描画を行う）
            if (partsType != LEWindowPartsID.Content && material.IMaterialTexMapSet.Length == 0)
            {
                return;
            }

            ITextureImage texImg1st = null;
            foreach (IMaterialTexMap texMap in texMapSet)
            {
                Bitmap bitmap = null;
                TexCoord4 texCoord4 = null;

                // bitmap, texCoord4
                ITextureImage texImg = texMap.TryFindTextureImage(OwnerPane.OwnerSubScene);
                if (texImg != null)
                {
                    // 一枚目のテクスチャはUV計算に利用するので覚えておきます。
                    if (texImg1st == null) { texImg1st = texImg; Debug.Assert(texImg1st != null); }

                    // テクスチャ行列をリセットします。
                    int texGenSlotIdx = texMap.SlotIdx;
                    if (texGenSlotIdx < 0 || texGenSlotIdx >= material.IMaterialTexMapSet.Length)
                    {
                        continue;
                    }

                    var uvSettingTexMapm = material.IMaterialTexMapSet[texGenSlotIdx];
                    ITexGen texgen = uvSettingTexMapm.ITexGen;
                    ITexMtx texMtx = uvSettingTexMapm.ITexGen.ITexMtx;
                    FVec2 sca = TextureImageHelper.GetModifiedScale(texImg, texMtx.Scale);
                    FVec2 tra = TextureImageHelper.GetModifiedTranslate(texImg, texMtx.Trans);
                    renderer.SetTextureMtx((int)uvSettingTexMapm.ITexGen.MtxID, tra, sca, texMtx.Rotate);

                    // UV座標を計算します。
                    if (uvSettingTexMapm.ITexGen.IsUVMethod)
                    {
                        // 一枚目のテクスチャを元にUVを算出します。
                        FVec2 imgSize = new FVec2(texImg1st.Size);
                        if(this.IsFrameSizeOrFrameSizeCurveSpecified)
                        {
                            AdjustFrameUVStretch_(this.FrameSize, partsType, numTexture, ref size, ref imgSize);
                        }

                        texCoord4 = GetWindowPartsTexCoord_(numTexture, imgSize, uvSettingTexMapm, size, texPartsType);
                    }
                    else
                    {
                        texCoord4 = new TexCoord4();
                        SetupProjectionTexCoord_(renderer, uvSettingTexMapm, texImg);
                    }

                    texCooors[texMap.SlotIdx] = texCoord4;
                    bitmap = texImg.GDIBitmap;

                    // TextureStage の設定
                    if (!materialDetail.LowLevelCombinerSettingsEnabled)
                    {
                        int stageIndex = texMap.SlotIdx - 1;
                        if (materialDetail.ITevData.NumStages > stageIndex)
                        {
                            RendererTextrureFormat renderTexFmt = RendererTextrureFormatHelper.GetRendererTextrureFormat(texImg.PixelFmt, texImg.SrcImageHasAlphaBit);
                            ITevStage tevStage = materialDetail.ITevData.GetITevStage(stageIndex);
                            if (tevStage != null)
                            {
                                // テクスチャサンプリング方法の設定
                                bool isIndirectStage = materialDetail.ITevData.IsIndirectState(stageIndex);
                                renderer.SetTextureSamplingState((int)texgen.MtxID, texMap.WrapS, texMap.WrapT, texMap.MinFilter, texMap.MagFilter, !isIndirectStage);
                                renderer.SetTextureState(
                                    texImg.GDIBitmap,
                                    texMap.SlotIdx,
                                    tevStage.ColorStage.CombineMode,
                                    tevStage.AlphaStage.CombineMode,
                                    tevStage.ColorStage.IndirectScale,
                                    tevStage.ColorStage.IndirectRotate,
                                    isIndirectStage ? TextureImageHelper.GetChandelNum(texImg.PixelFmt) : -1,
                                    renderTexFmt);
                            }
                            else
                            {
                                renderer.SetTextureSamplingState((int)texgen.MtxID, texMap.WrapS, texMap.WrapT, texMap.MinFilter, texMap.MagFilter, true);
                                renderer.SetTextureState(texImg.GDIBitmap, texMap.SlotIdx, renderTexFmt);
                            }
                        }
                    }
                    else
                    {
                        RendererTextrureFormat renderTexFmt = RendererTextrureFormatHelper.GetRendererTextrureFormat(texImg.PixelFmt, texImg.SrcImageHasAlphaBit);
                        renderer.SetTextureSamplingState((int)texgen.MtxID, texMap.WrapS, texMap.WrapT, texMap.MinFilter, texMap.MagFilter, true);
                        renderer.SetTextureState(texImg.GDIBitmap, texMap.SlotIdx, renderTexFmt);
                    }
                }
            }

            // 詳細マテリアル
            if (materialDetail.LowLevelCombinerSettingsEnabled)
            {
                for (int j = 0; j < materialDetail.TevData.NumStages; j++)
                {
                    foreach (var args in materialDetail.TevData.TevStages.GetTevStage(j).ColorStage.TevArgs)
                    {
                        LECore.Structures.Nsrif.Attributes.AttrTevSource attrTevSource = args.Source;
                    }
                }

                // コンスタントカラー
                List<Color> colorList = new List<Color>();

                for (int i = 0; i < materialDetail.TevData.NumTevColors; i++)
                {
                    colorList.Add(materialDetail.TevData.GetTevColor(i).ToSystemColor());
                }

                // 黒カラー
                colorList.Add(material.BlackColor.ToSystemColor());

                // 白カラー
                colorList.Add(material.WhiteColor.ToSystemColor());

                renderer.SetDetailedCombinerState(
                    colorList.ToArray(),
                    materialDetail.TevData.TevStages.GetTevStages(),
                    materialDetail.TevData.NumStages);
            }

            // 描画
            {
                FVec3 rectPos = new FVec3(pos.X, pos.Y, 0.0f);

                renderer.DrawImage(rectPos, size, texCooors, color);
                if (!option.CheckActive(DrawableOptionFlag.IgnoreBVDrawing))
                {
                    renderer.DrawRectangle(rectPos.X, rectPos.Y, rectPos.Z, size.X, size.Y);
                }
            }
        }

        /// <summary>
        /// フレームサイズが指定されている場合は
        /// 特定部位以外は、画像サイズ=ポリゴンサイズに調整していっぱいに貼る。
        /// </summary>
        private void AdjustFrameUVStretch_(LEWindowFrameSize frameSize, LEWindowPartsID partsType, int numTexture, ref FVec2 size, ref FVec2 imgSize)
        {
            if (numTexture == 9)
            {
                if (partsType != LEWindowPartsID.FrameT && partsType != LEWindowPartsID.FrameB)
                {
                    imgSize.X = size.X;
                }

                if (partsType != LEWindowPartsID.FrameL && partsType != LEWindowPartsID.FrameR)
                {
                    imgSize.Y = size.Y;
                }
            }
            else
            {
                if (partsType != LEWindowPartsID.CornerLT && partsType != LEWindowPartsID.CornerRB)
                {
                    imgSize.X = size.X;
                }

                if (partsType != LEWindowPartsID.CornerRT && partsType != LEWindowPartsID.CornerLB)
                {
                    imgSize.Y = size.Y;
                }
            }
        }

        /// <summary>
        /// 投影テクスチャ設定
        /// </summary>
        private void SetupProjectionTexCoord_(IRenderer renderer, IMaterialTexMap texMap, ITextureImage texImg)
        {
            // 投影マッピング
            Matrix34 tempMtx = renderer.CurrentMtx34;
            renderer.PopMtx();

            Matrix34 paneCenterMtx = renderer.CurrentMtx34;
            renderer.PushMtx();
            renderer.CurrentMtx34 = tempMtx;

            Matrix34 projMtx34 = TexGenHelper.CalcTextureProjectionMtx(paneCenterMtx, this, texMap.ITexGen, texImg);

            renderer.SetTextureProjectionState(texMap.SlotIdx, new FVec2(texImg.Size), projMtx34);
        }

        /// <summary>
        /// ウインドウのマテリアルが初期状態か判定します。
        /// </summary>
        /// <returns></returns>
        bool IsMaterialInitialState_()
        {
            foreach (LEWindowPartsID partsID in Enum.GetValues(typeof(LEWindowPartsID)))
            {
                IMaterial material = GetCornerTextureMaterial_(partsID);
                if (material == null)
                {
                    return true;
                }

                if (material.IMaterialTexMapSet.Length != 0)
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// ウインドウを描画します。
        /// </summary>>
        public void Draw(IRenderer renderer, DrawableOption option)
        {
            if (option.CheckActive(DrawableOptionFlag.IgnorePaneImageDrawing))
            {
                return;
            }

            // ----------- ウインドウフレームを構成する9つの部分を描画します。
            if (this.WindowKind == WindowKind.Around)
            {
                bool useLTMaterial = this.UseLTMaterial && this.NumTexture != 2;

                UseLTForTexCoord useLTForTexCoord = useLTMaterial ? UseLTForTexCoord.Enabled : UseLTForTexCoord.None;
                UseLTForBlend useLTForBlend = useLTMaterial ? UseLTForBlend.Enabled : UseLTForBlend.None;

                if (NumTexture == 2 || NumTexture == 3 || NumTexture == 5)
                {

                    if (!this.NotDrawContent)
                    {
                        DrawWindowParts_(renderer, LEWindowPartsID.Content, NumTexture, option, UseLTForTexCoord.None, UseLTForBlend.None);
                    }

                    DrawWindowParts_(renderer, LEWindowPartsID.CornerLT, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerRT, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerRB, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerLB, NumTexture, option, useLTForTexCoord, useLTForBlend);
                }
                else if (NumTexture == 9)
                {
                    if (!this.NotDrawContent)
                    {
                        DrawWindowParts_(renderer, LEWindowPartsID.Content, NumTexture, option, UseLTForTexCoord.None, UseLTForBlend.None);
                    }

                    DrawWindowParts_(renderer, LEWindowPartsID.CornerLT, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerRT, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerRB, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerLB, NumTexture, option, useLTForTexCoord, useLTForBlend);

                    DrawWindowParts_(renderer, LEWindowPartsID.FrameT, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.FrameR, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.FrameB, NumTexture, option, useLTForTexCoord, useLTForBlend);
                    DrawWindowParts_(renderer, LEWindowPartsID.FrameL, NumTexture, option, useLTForTexCoord, useLTForBlend);
                }
            }
            else if (this.WindowKind == WindowKind.HorizontalNoContent)
            {
                DrawWindowParts_(renderer, LEWindowPartsID.CornerLT, NumTexture, option, UseLTForTexCoord.Enabled, UseLTForBlend.Enabled);
                DrawWindowParts_(renderer, LEWindowPartsID.CornerRT, 9, option, UseLTForTexCoord.Enabled, UseLTForBlend.Enabled);
            }
            else if (this.WindowKind == WindowKind.Horizontal)
            {
                if (!this.NotDrawContent)
                {
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerLT, LEWindowPartsID.CornerLT, 9, option, UseLTForTexCoord.Enabled, UseLTForBlend.Enabled);
                    DrawWindowParts_(renderer, LEWindowPartsID.Content, LEWindowPartsID.FrameT, 9, option, UseLTForTexCoord.None, UseLTForBlend.Enabled);
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerRT, 9, option, UseLTForTexCoord.Enabled, UseLTForBlend.Enabled);
                }
                else
                {
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerLT, LEWindowPartsID.CornerLT, 9, option, UseLTForTexCoord.Enabled, UseLTForBlend.Enabled);
                    DrawWindowParts_(renderer, LEWindowPartsID.CornerRT, 9, option, UseLTForTexCoord.Enabled, UseLTForBlend.Enabled);
                }
            }

            if (!IsFrameTextureSizeValid())
            {
                // 設定画像が不正であることを示す×印を描画します。
                float oldLineWidth = renderer.LineWidth;

                renderer.LineWidth = 4.0f;

                renderer.DrawLine(PointF.Empty, new PointF(OwnerPane.Width, OwnerPane.Height));
                renderer.DrawLine(new PointF(OwnerPane.Width, 0), new PointF(0, OwnerPane.Height));

                renderer.LineWidth = oldLineWidth;
            }
        }

        #endregion IDrawable

        #region ILEWindow

        /// <summary>
        ///
        /// </summary>
        public WindowKind WindowKind
        {
            get { return _windowKind; }
            set
            {
                if (_windowKind != value)
                {
                    _windowKind = value;

                    UpdateAttributeActiveState_();
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// ウインドウ種類変更時の更新処理です。
        /// <remarks>
        /// サイズ再計算を避けるために、
        /// UIからの WindowKind 変更時に明示的に読んでもらっています。
        /// </remarks>
        /// </summary>
        public void OnWindowKindChangeUpdate()
        {
            // 水平モード
            if (_windowKind != WindowKind.Around)
            {
                // 水平モード設定時は、必ずサイズ調整する。
                this.AdjustPaneSize();

                // 有効なテクスチャ枚数に
                this.NumTexture = Math.Min(this.NumTexture, 3);
                // 左上マテリアル共有ON
                this.UseLTMaterial = true;

                if (_windowKind == WindowKind.HorizontalNoContent)
                {
                    this.UseVtxColorForAllWindow = true;
                }
            }
            else
            {
                Debug.Assert(_windowKind == WindowKind.Around);

                // Horizontal から Around モードに戻る場合、テクスチャ枚数を適当な値に設定する。(Around モード なら、2, 5, 9 が妥当な数)
                bool isInvalidAroundFrameCount = this.NumTexture == 3;
                if (isInvalidAroundFrameCount)
                {
                    // テクスチャが設定されているパーツの数を数え上げる。
                    int numTextureFrame = 0;
                    for (int i = 0; i < PartsCount; i++)
                    {
                        if (this.GetPartsMultiTextureCount((LEWindowPartsID)i) != 0)
                        {
                            numTextureFrame++;
                        }
                    }

                    this.NumTexture = Math.Max(numTextureFrame, 2);
                }
            }
        }

        /// <summary>
        /// サイズ調整します。
        /// </summary>
        public void AdjustPaneSize()
        {
            if (_windowKind != WindowKind.Around)
            {
                AdjustPaneSizeHorizontal_();
            }
            else
            {
                float minHeight = this.GetPartsMultiTextureCount(LEWindowPartsID.CornerLT) == 0 ?
                    ILEWindowHelper.InitialMinFrameSize * 3 : GetPartsSize_(LEWindowPartsID.CornerLT).Y * 3;
                float newHeight = Math.Max(this.OwnerPane.Y, minHeight);

                (this.OwnerPane as Pane).Size = new FVec3(this.OwnerPane.Size.X, newHeight, this.OwnerPane.Size.Z);
            }
        }

        /// <summary>
        /// 水平モードのサイズ調整
        /// </summary>
        private void AdjustPaneSizeHorizontal_()
        {
            if (_windowKind != WindowKind.Around)
            {
                float newHeight = this.GetPartsMultiTextureCount(LEWindowPartsID.CornerLT) == 0 ?
                    ILEWindowHelper.InitialMinFrameSize : GetPartsSize_(LEWindowPartsID.CornerLT).Y;
                (this.OwnerPane as Pane).Size = new FVec3(this.OwnerPane.Size.X, newHeight, this.OwnerPane.Size.Z);

                // 基本値を更新します
                ((this.OwnerPane as IPane).IPaneAttribute.SizeIAttr as AnmAttribute).SetBaseValueRecursive();
            }
        }

        /// <summary>
        /// フレームサイズを調整
        /// </summary>
        private void AdjustFrameSizeIfNotSpecified_()
        {
            if (!this.IsFrameSizeOrFrameSizeCurveSpecified)
            {
                FVec2 lt = GetPartsTextureSize_(LEWindowPartsID.CornerLT);
                FVec2 rb = GetPartsTextureSize_(LEWindowPartsID.CornerRB);

                this.FrameSize = new LEWindowFrameSize(lt.Y, rb.Y, lt.X, rb.X, false);

                // 基本値を更新します
                this.FrameSizeAnmAttr.SetBaseValueRecursive();
            }
        }

        /// <summary>
        /// 頂点カラーをペイン全体に適用するかを指定します。
        /// </summary>
        public bool UseVtxColorForAllWindow
        {
            get
            {
                return (_windowKind == WindowKind.HorizontalNoContent) ? true : _useVtxColorForAllPane;
            }

            set
            {
                if (_windowKind == WindowKind.HorizontalNoContent)
                {
                    return;
                }

                if (_useVtxColorForAllPane != value)
                {
                    _useVtxColorForAllPane = value;
                    NotifyChangeToScene_();
                }
            }
        }

        public int NumTexture
        {
            get { return _numTexture; }
            set
            {
                if (_numTexture != value)
                {
                    _numTexture = value;
                    UpdateAttributeActiveState_();
                    NotifyChangeToScene_();
                    NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneAttrModify); // アトリビュート変更イベントを発行してノードツリーを更新します
                }
            }
        }

        /// <summary>
        /// @@@
        /// </summary>
        public bool UseLTMaterial
        {
            get { return this.useLTMaterial; }
            set
            {
                if (this.useLTMaterial != value)
                {
                    this.useLTMaterial = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// フレームサイズ
        /// </summary>
        public LEWindowFrameSize FrameSize
        {
            get
            {
                return new LEWindowFrameSize(this._frameSizeAttr.Top, this._frameSizeAttr.Bottom, this._frameSizeAttr.Left, this._frameSizeAttr.Right, _isFrameSizeSpecified);
            }

            set
            {
                if(!this._frameSizeAttr.IsSame(value) || _isFrameSizeSpecified != value.IsSpecified)
                {
                    if (this.GetOwnerSubScene() != null)
                    {
                        this.GetOwnerSubScene().BeginMassiveModify();
                    }

                    _isFrameSizeSpecified = value.IsSpecified;

                    this._frameSizeAttr.Set(value);
                    NotifyChangeToScene_();

                    if (this.GetOwnerSubScene() != null)
                    {
                        this.GetOwnerSubScene().EndMassiveModify();
                    }
                }
            }
        }

        /// <summary>
        /// フレームサイズを取得します。
        /// </summary>
        public ILEWindowContentsInflation ContentsInflation
        {
            get { return _contentsInflation; }
            set
            {
                if (!_contentsInflation.IsSame(value))
                {
                    _contentsInflation = value;
                    NotifyChangeToScene_();
                }
            }
        }

        /// <summary>
        /// 詳細なマテリアルを使用するかどうかのフラグを取得します。
        /// </summary>
        public bool GetUseDetailedMaterialFlag(LEWindowPartsID id)
        {
            int index = PartsIDToInt(id);
            if (index < _useDetailedMaterials.Length)
            {
                return _useDetailedMaterials[index];
            }
            else
            {
                Debug.Assert(false);
                return false;
            }
        }

        /// <summary>
        /// 詳細なマテリアルを使用するかどうかのフラグを設定します。
        /// </summary>
        public void SetUseDetailedMaterialFlag(LEWindowPartsID id, bool flag)
        {
            int index = PartsIDToInt(id);
            if (index < _useDetailedMaterials.Length)
            {
                if (_useDetailedMaterials[index] != flag)
                {
                    _useDetailedMaterials[index] = flag;
                    NotifyChangeToScene_();
                }
            }
            else
            {
                Debug.Assert(false);
            }
        }

        /// <summary>
        /// マテリアルを取得します。
        /// </summary>
        public IMaterial GetMaterial(LEWindowPartsID id)
        {
            int index = PartsIDToInt(id);
            if (index < _materials.Length)
            {
                return _materials[index];
            }
            else
            {
                Debug.Assert(false);
                return null;
            }
        }

        /// <summary>
        /// フリップ状態を取得します。
        /// </summary>
        public LEWindowFrameFlipRotate
            GetMaterialFlipRotateState(LEWindowPartsID id)
        {
            int index = PartsIDToInt(id);

            if (index < _flipRotateState.Length)
            {
                if (NumTexture == 2)
                {
                    switch (id)
                    {
                        case LEWindowPartsID.CornerLT: return LEWindowFrameFlipRotate.None;
                        case LEWindowPartsID.CornerRT: return LEWindowFrameFlipRotate.FlipH;
                        case LEWindowPartsID.CornerRB: return LEWindowFrameFlipRotate.Rotate180;
                        case LEWindowPartsID.CornerLB: return LEWindowFrameFlipRotate.FlipV;
                        default: return _flipRotateState[index];
                    }
                }
                else
                {
                    return _flipRotateState[index];
                }

            }
            else
            {
                Debug.Assert(false);
                return LEWindowFrameFlipRotate.None;
            }
        }

        /// <summary>
        /// UV座標スケーリングの原点を取得します。
        /// </summary>
        FVec2 GetWindoePartUVScalingOrigin_(LEWindowPartsID partsID, TexCoord4 texCoord4)
        {
            switch (partsID)
            {
                case LEWindowPartsID.CornerLT:
                case LEWindowPartsID.FrameT:
                    {
                        return texCoord4.LT;
                    }

                case LEWindowPartsID.CornerLB:
                case LEWindowPartsID.FrameL:
                    {
                        return texCoord4.LB;
                    }

                case LEWindowPartsID.CornerRT:
                case LEWindowPartsID.FrameR:
                    {
                        return texCoord4.RT;
                    }

                case LEWindowPartsID.CornerRB:
                case LEWindowPartsID.FrameB:
                    {
                        return texCoord4.RB;
                    }

                case LEWindowPartsID.Content:
                    {
                        return FVec2.Empty;
                    }

                default:
                    {
                        Debug.Assert(false);
                        return FVec2.Empty;
                    }
            }
        }


        #region コンテンツ領域関連
        /// <summary>
        /// コンテンツ領域で使用するテクスチャ座標の数
        /// コンテンツ領域で使用するテクスチャ枚数以下を指定する必要があります。
        /// </summary>
        public int NumContentsTexCoord
        {
            get
            {
                if (_numContentsTexCoord >= _NumContentsTextures)
                {
                    _numContentsTexCoord = _NumContentsTextures;
                }
                return _numContentsTexCoord;
            }
            set
            {
                if (_NumContentsTextures >= value)
                {
                    if (_numContentsTexCoord != value)
                    {
                        _numContentsTexCoord = value;

                        Material mat = GetMaterial(LEWindowPartsID.Content) as Material;
                        mat.UserMaxNumTexCoord = value;

                        NotifyChangeToScene_();
                    }
                }
            }
        }

        /// <summary>
        /// コンテンツ領域のテクスチャ座標を取得します。
        /// </summary>
        /// <param name="texIndex"></param>
        /// <returns></returns>
        public TexCoord4 GetContentsTexCoord4(int texIndex)
        {
            if (NumContentsTexCoord > texIndex)
            {
                return _contetnsTexCoord4Set[texIndex];
            }
            else
            {
                return null;
            }
        }

        #endregion コンテンツ領域関連

        /// <summary>
        /// ウインドウパーツの矩形を取得します。
        /// </summary>
        public void GetWindowPartsRectangle(
            int numTexture, LEWindowPartsID partsType, out FVec2 pos, out FVec2 size)
        {
            GetWindowPartsRectangle_(numTexture, partsType, out pos, out size);
        }

        #endregion ILEWindow

        #region アニメーションアトリビュート取得関連

        public IAnmAttribute VtxCol_LTIAnmAttr { get { return VtxCol_LTAnmAttr; } }
        public IAnmAttribute VtxCol_RTIAnmAttr { get { return VtxCol_RTAnmAttr; } }
        public IAnmAttribute VtxCol_LBIAnmAttr { get { return VtxCol_LBAnmAttr; } }
        public IAnmAttribute VtxCol_RBIAnmAttr { get { return VtxCol_RBAnmAttr; } }

        public AnmAttribute VtxCol_LTAnmAttr { get { return _cornerColorsAttr.VtxCol_LTAnmAttr; } }
        public AnmAttribute VtxCol_RTAnmAttr { get { return _cornerColorsAttr.VtxCol_RTAnmAttr; } }
        public AnmAttribute VtxCol_LBAnmAttr { get { return _cornerColorsAttr.VtxCol_LBAnmAttr; } }
        public AnmAttribute VtxCol_RBAnmAttr { get { return _cornerColorsAttr.VtxCol_RBAnmAttr; } }


        public IAnmAttribute FrameSizeIAnmAttr { get { return _frameSizeAttr; } }
        public RectAttribute FrameSizeAnmAttr { get { return _frameSizeAttr; } }
        #endregion アニメーションアトリビュート取得関連

        #region IRevHWLEWindow
        /// <summary>
        /// Revolution固有マテリアルを取得します。
        /// </summary>
        public IRevHWMaterial[] IRevHWMaterial
        {
            get { return _revMaterials; }
        }
        #endregion IRevHWLEWindow

        #region IPaneExParamater
        /// <summary>
        /// 自身を持っているペインの参照です。
        /// </summary>
        public IPane OwnerPane
        {
            get { return _ownerPane; }
        }

        /// <summary>
        /// マテリアル名を最新の状態に更新します。
        /// </summary>
        public void UpdateMatarialName()
        {
            foreach (LEWindowPartsID matID in Enum.GetValues(typeof(LEWindowPartsID)))
            {
                // マテリアル名を更新します。
                _materials[(int)matID].MaterialName =
                    GetMaterialName_(this.OwnerPane, matID);
            }
        }

        /// <summary>
        /// シーン登録時の初期化
        /// </summary>
        public void OnJoinSceneInitialize()
        {
            OnWindowKindChangeUpdate();

            AdjustFrameSizeIfNotSpecified_();

            // エラーメッセージの更新。
            this.WarningMsg = !IsFrameTextureSizeValid() ? LECoreStringResMgr.Get("LECORE_WINDOWPANE_TEXSIZE_WARN") : string.Empty;
        }

        /// <summary>
        /// 編集対象に設定される直前の初期化処理
        /// </summary>
        public void FirstTimeInitialize()
        {
            // 何もしません。
        }

        /// <summary>
        /// UV座標を列挙します。
        /// </summary>
        public IEnumerable<TexCoord4> TexCoords
        {
            get
            {
                int count = 0;
                foreach (var texCoord in this._contetnsTexCoord4Set)
                {
                    if (count >= _numContentsTexCoord)
                    {
                        break;
                    }

                    yield return texCoord;
                    count++;
                }
            }
        }

        /// <summary>
        /// 警告文字列を取得します。UIの各所で表示します。
        /// </summary>
        public string WarningMsg
        {
            get;
            private set;
        }

        /// <summary>
        /// 境界を書くかどうか
        /// </summary>
        public bool PaneBoundDrawEnabled
        {
            get { return true; }
        }
        #endregion IPaneExParamater

        #region Setメソッド群

        /// <summary>
        /// マテリアルのフリップ状態を設定します。
        /// </summary>
        public void
            SetMaterialFlipState(LEWindowPartsID id, LEWindowFrameFlipRotate state)
        {
            int index = PartsIDToInt(id);
            if (index < _flipRotateState.Length)
            {
                _flipRotateState[index] = state;
                NotifyChangeToScene_();
            }
            else
            {
                Debug.Assert(false);
            }
        }

        public RGBAColor VtxCol_LT
        {
            get { return _cornerColorsAttr.VtxCol_LT; }
            set { if (_cornerColorsAttr.VtxCol_LT != value) { _cornerColorsAttr.VtxCol_LT = value; } }
        }

        public RGBAColor VtxCol_RT
        {
            get { return _cornerColorsAttr.VtxCol_RT; }
            set { if (_cornerColorsAttr.VtxCol_RT != value) { _cornerColorsAttr.VtxCol_RT = value; } }
        }

        public RGBAColor VtxCol_LB
        {
            get { return _cornerColorsAttr.VtxCol_LB; }
            set { if (_cornerColorsAttr.VtxCol_LB != value) { _cornerColorsAttr.VtxCol_LB = value; } }
        }

        public RGBAColor VtxCol_RB
        {
            get { return _cornerColorsAttr.VtxCol_RB; }
            set { if (_cornerColorsAttr.VtxCol_RB != value) { _cornerColorsAttr.VtxCol_RB = value; } }
        }

        /// <summary>
        /// コンテンツ領域のテクスチャ座標を設定します。
        /// </summary>
        public void SetContentsTexCood4(int texIndex, TexCoord4 texCoord)
        {
            if (NumContentsTexCoord > texIndex)
            {
                _contetnsTexCoord4Set[texIndex].Set(texCoord);
                NotifyChangeToScene_();
            }
        }

        #endregion Setメソッド群

        #region イベント通知

        /// <summary>
        /// オーバーライド：更新イベント通知をします。
        /// </summary>
        public override void NotifyModifyEvent(LEDataNode sender, int kind)
        {
            // 配下でフレームサイズの変更があった場合はサイズ調整を実行
            if (sender is RectAttribute)
            {
                AdjustFrameSizeIfNotSpecified_();

                if (kind == (int)EventKind.Modify)
                {
                    if (_windowKind != WindowKind.Around)
                    {
                        // 水平モード設定時は、必ずサイズ調整する。
                        this.AdjustPaneSize();
                    }
                }
            }

            // 配下でテクスチャ変更があった場合はサイズ調整を実行
            if (sender is MaterialTexMap)
            {
                AdjustFrameSizeIfNotSpecified_();

                this.AdjustPaneSizeHorizontal_();
            }

            // 不正なら警告メッセージを設定
            this.WarningMsg = !IsFrameTextureSizeValid() ? LECoreStringResMgr.Get("LECORE_WINDOWPANE_TEXSIZE_WARN") : string.Empty;

            // マテリアルが変更された場合は、
            // テクスチャ数が変更されたか確認します。
            // テクスチャ数を更新します。
            Material contentsMat = this.GetCornerTextureMaterial_(LEWindowPartsID.Content);
            if (sender == contentsMat)
            {
                // テクスチャが設定されているか確認し、
                if (contentsMat.MaterialTexMapSet.Length != 0)
                {
                    // テクスチャ座標が最低1種類設定されていることを保障します。
                    if (this._numContentsTexCoord == 0)
                    {
                        this._numContentsTexCoord = 1;
                    }
                }
            }

            base.NotifyModifyEvent(sender, kind);
        }

        /// <summary>
        /// 変更を通知します。
        /// アニメーションアトリビュート以外の更新を通知します。
        /// </summary>
        void NotifyChangeToScene_()
        {
            NotifyModifyEvent(this, (int)SceneModifyEventArgs.Kind.PaneModify);
        }
        #endregion イベント通知
    }
}
