﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures
{
    using Core;
    using LECore.Structures.SerializableObject.Lyt;

    public static class ILEWindowHelper
    {
        public const int NumTextureMax = 9;
        public const int NumTextureContentAndCorners = 5;


        static private RectangleF GetWindowPartsRect_(this ILEWindow window, int numWndTex, LEWindowPartsID partsID)
        {
            FVec2 pos;
            FVec2 size;
            window.GetWindowPartsRectangle(numWndTex, partsID, out pos, out size);

            return new RectangleF(pos.AsPointF, size.AsSizeF);
        }

        /// <summary>
        /// 位置に対応する、ウインドウ部位を取得します。
        /// </summary>
        static public LEWindowPartsID GetPartIDByPosition( ILEWindow window, PointF posInPaneLocal )
        {
            LEWindowPartsID result = LEWindowPartsID.Content;

            if (window.WindowKind == WindowKind.Around)
            {
                int numRect = (window.NumTexture <= NumTextureContentAndCorners) ? NumTextureContentAndCorners : window.NumTexture;
                for (int i = 0; i < numRect; i++)
                {
                    LEWindowPartsID partsID = window.IntToPartsID(i);
                    RectangleF rect = window.GetWindowPartsRect_(numRect, partsID);
                    if (rect.Contains(posInPaneLocal))
                    {
                        result = partsID;
                        break;
                    }
                }
            }
            else
            {
                // 水平ウインドウ
                // 左端ヒットチェック
                RectangleF rectL = window.GetWindowPartsRect_(NumTextureMax, LEWindowPartsID.CornerLT);
                if (rectL.Contains(posInPaneLocal))
                {
                    result = LEWindowPartsID.CornerLT;
                }
                else
                {
                    // 右端ヒットチェック
                    RectangleF rectR = window.GetWindowPartsRect_(NumTextureMax, LEWindowPartsID.CornerRT);
                    result = rectR.Contains(posInPaneLocal) ? LEWindowPartsID.CornerRT : LEWindowPartsID.Content;
                }
            }

            return window.IntToPartsID( Math.Min( window.PartsIDToInt( result ), window.NumTexture - 1 ) );
        }

        /// <summary>
        ///
        /// </summary>
        public static LEWindowPartsID GetLEWindowPartsIDByName( string materialName)
        {
            return LEWindow.GetLEWindowPartsIDByName( materialName);
        }

        /// <summary>
        ///
        /// </summary>
        public static bool IsEnabledUseLTMaterial( LEWindowPartsID partsID, ILEWindow targetWindow)
        {
            if( partsID == LEWindowPartsID.Content ||
                partsID == LEWindowPartsID.CornerLT )
            {
                return false;
            }

            if (targetWindow.NumTexture == 2 || targetWindow.NumTexture == 3)
            {
                return false;
            }

            return targetWindow.UseLTMaterial;
        }

        public const float InitialMinFrameSize = 16.0f;

        /// <summary>
        /// フレームテクスチャサイズが有効か判定します。
        /// </summary>
        public static bool IsFrameTextureSizeValid(this ILEWindow window)
        {
            return (window as LEWindow).IsFrameTextureSizeValid();
        }

        /// <summary>
        /// マルチテクスチャの枚数が適当か判定します。
        /// </summary>
        public static bool IsMultiTextureCountValid(this ILEWindow window)
        {
            if (!window.UseLTMaterial)
            {
                return true;
            }

            int matLTMultiTexCount = GetPartsMultiTextureCount(window, LEWindowPartsID.CornerLT);
            if (window.WindowKind == WindowKind.Around)
            {
                for (int i = 0; i < window.NumTexture; i++)
                {
                    // コンテンツ領域は別
                    if (LEWindowPartsID.Content == (LEWindowPartsID)i)
                    {
                        continue;
                    }

                    if (matLTMultiTexCount != GetPartsMultiTextureCount(window, (LEWindowPartsID)i))
                    {
                        return false;
                    }
                }

                return true;
            }
            else if (window.WindowKind == WindowKind.HorizontalNoContent)
            {
                return (matLTMultiTexCount == GetPartsMultiTextureCount(window, (LEWindowPartsID)LEWindow.GetInternalWindowPartsID(window, LEWindowPartsID.CornerRT)));//
            }
            else if (window.WindowKind == WindowKind.Horizontal)
            {
                return
                    (matLTMultiTexCount == GetPartsMultiTextureCount(window, (LEWindowPartsID)LEWindow.GetInternalWindowPartsID(window, LEWindowPartsID.CornerRT))) &&
                    (matLTMultiTexCount == GetPartsMultiTextureCount(window, (LEWindowPartsID)LEWindow.GetInternalWindowPartsID(window, LEWindowPartsID.Content)));
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// 指定部位のマルチテクスチャ枚数を取得します。
        /// </summary>
        public static int GetPartsMultiTextureCount(this ILEWindow window, LEWindowPartsID id)
        {
            var matLT = window.GetMaterial(id);
            return matLT != null ? matLT.IMaterialTexMapSet.Length : 0;
        }

        /// <summary>
        /// マテリアルカラーが個別に編集可能か？
        /// </summary>
        public static bool GetMaterialMatColorEditable(this ILEWindow wnd, LEWindowPartsID partsID)
        {
            bool IsTextureMappingEditable;
            bool IsMaterialColorEditable;
            bool IsBlendEditable;
            bool IsTextureStageEditable;

            ILEWindowHelper.GetMaterialEditableCondition(wnd.WindowKind, partsID, wnd.UseLTMaterial,
                out IsTextureMappingEditable,
                out IsMaterialColorEditable,
                out IsBlendEditable,
                out IsTextureStageEditable);

            return IsMaterialColorEditable;
        }

        /// <summary>
        /// 編集可能状態を取得する
        /// </summary>
        public static void GetMaterialEditableCondition(
            WindowKind kind,
            LEWindowPartsID partsID,
            bool useLTMaterial,
            out bool IsTextureMappingEditable,
            out bool IsMaterialColorEditable,
            out bool IsBlendEditable,
            out bool IsTextureStageEditable)
        {
            // 左上は常に編集可能
            if (partsID == LEWindowPartsID.CornerLT)
            {
                IsTextureMappingEditable = true;
                IsMaterialColorEditable = true;
                IsBlendEditable = true;
                IsTextureStageEditable = true;
            }
            else
            {
                // 左上以外の場合...ウインドウ種類毎に
                switch (kind)
                {
                    case WindowKind.Around:
                        if (partsID == LEWindowPartsID.Content)
                        {
                            IsTextureMappingEditable = true;
                            IsMaterialColorEditable = true;
                            IsBlendEditable = true;
                            IsTextureStageEditable = true;
                        }
                        else
                        {
                            if (useLTMaterial)
                            {
                                IsTextureMappingEditable = false;
                                IsMaterialColorEditable = false;
                                IsBlendEditable = false;
                                IsTextureStageEditable = false;
                            }
                            else
                            {
                                IsTextureMappingEditable = true;
                                IsMaterialColorEditable = true;
                                IsBlendEditable = true;
                                IsTextureStageEditable = true;
                            }
                        }
                        break;
                    case WindowKind.Horizontal:
                    case WindowKind.HorizontalNoContent:
                        IsTextureMappingEditable = partsID != LEWindowPartsID.CornerRT;
                        IsMaterialColorEditable = false;
                        IsBlendEditable = false;
                        IsTextureStageEditable = false;
                        break;
                    default:
                        IsTextureMappingEditable = true;
                        IsMaterialColorEditable = true;
                        IsBlendEditable = true;
                        IsTextureStageEditable = true;
                        break;
                }
            }
        }
    }
}
