﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Text.RegularExpressions;
using System.Linq;
using LECore.Manipulator;
using LECore.Structures.Core;
using LECore.Structures.SerializableObject.Lyt;
using System.IO;

namespace LECore.Structures
{
    /// <summary>
    /// 集計情報
    /// </summary>
    public class SubSceneMetrics
    {
        public int NullPaneCount { get; set; }
        public int BoundingPaneCount { get; set; }
        public int PicturePaneCount { get; set; }
        public int TextBoxPaneCount { get; set; }
        public int WindowPaneCount { get; set; }
        public int CapturePaneCount { get; set; }
        public int AligmentPaneCount { get; set; }
        public int ScissorPaneCount { get; set; }
        public int PartsPaneCount { get; set; }

        public int TotalPixcel0Tex { get; set; }
        public int TotalPixcel1Tex { get; set; }
        public int TotalPixcel2Tex { get; set; }
        public int TotalPixcel3Tex { get; set; }

        public int ScreenW { get; set; }
        public int ScreenH { get; set; }

        // 総ペイン数
        public int GetTotalPaneCount()
        {
            return NullPaneCount + BoundingPaneCount + PicturePaneCount + TextBoxPaneCount + WindowPaneCount + CapturePaneCount + AligmentPaneCount + ScissorPaneCount + PartsPaneCount;
        }

        /// <summary>
        /// 総面積、Null, bounding, Parts の面積は除外しています。(GPU負荷)
        /// </summary>
        public int GetTotalPixcelCount()
        {
            return TotalPixcel0Tex + TotalPixcel1Tex + TotalPixcel2Tex + TotalPixcel3Tex;
        }

        public void Add(SubSceneMetrics metrics)
        {
            this.NullPaneCount += metrics.NullPaneCount;
            this.BoundingPaneCount += metrics.BoundingPaneCount;
            this.PicturePaneCount += metrics.PicturePaneCount;
            this.TextBoxPaneCount += metrics.TextBoxPaneCount;
            this.WindowPaneCount += metrics.WindowPaneCount;
            this.CapturePaneCount += metrics.CapturePaneCount;
            this.AligmentPaneCount += metrics.AligmentPaneCount;
            this.ScissorPaneCount += metrics.ScissorPaneCount;
            this.PartsPaneCount += metrics.PartsPaneCount;

            this.TotalPixcel0Tex += metrics.TotalPixcel0Tex;
            this.TotalPixcel1Tex += metrics.TotalPixcel1Tex;
            this.TotalPixcel2Tex += metrics.TotalPixcel2Tex;
            this.TotalPixcel3Tex += metrics.TotalPixcel3Tex;
        }
    }

    /// <summary>
    /// SubSceneHelper の概要の説明です。
    /// </summary>
    public static class SubSceneHelper
    {
        const bool           SelectLockedEnable   = true;
        const bool           SelectLockedDisable  = false;
        const int MinNumDigit = 2;
        const int _sectionMargin = 10;
        static bool _isPartsReloading = false;

        static public bool IsPartsReloading { get { return _isPartsReloading; } }

        #region 自動命名処理

        /// <summary>
        /// シーン内に特定の名前のデータが存在するか判定する関数。
        /// </summary>
        public delegate object FindTargetByName(string targetName);


        /// <summary>
        /// baseName_XXX ( XXX = count ) の ような名前を取得します
        /// </summary>
        static string GetNumberdString_(string baseName, int maxNameLength, int count, int countOriginalDigit)
        {
            string resultName;

            // 最大文字数から、_XXX の分の文字数を消去します。XXX は 最小でMinNumDigit桁表示になります。
            int lenNumberDigit = Math.Max(Math.Max(count.ToString().Length, MinNumDigit), countOriginalDigit);
            int lenSubName = Math.Min(baseName.Length, maxNameLength - (lenNumberDigit + 1));

            resultName = baseName.Substring(0, lenSubName);

            string formatString = "{0}_{1:d" + lenNumberDigit + "}";
            return string.Format(formatString, resultName, count);
        }

        /// <summary>
        /// 数字サフィックスを取り去った、名前を取得します。
        /// </summary>
        static public string GetBaseName(string originalName)
        {
            Regex numberedName = new Regex(@"_[0-9]+$");
            return numberedName.Replace(originalName, "");
        }

        /// <summary>
        /// 入力された名前を元に、シーン内で重複しない名前を取得します。
        /// </summary>
        static public string GetValidName(
            FindTargetByName findFunc,
            int maxNameLength,
            string originalName)
        {
            // 制限文字数に切り捨てる
            originalName = originalName.Substring(0, Math.Min(originalName.Length, maxNameLength));

            // 同名のペインが存在する場合は、名前を適切に変更します。
            if (findFunc(originalName) != null)
            {
                // 番号サフィックスがあれば取り去る。
                string name = null;
                string baseName = GetBaseName(originalName);

                int countBase = baseName.Length + 1; // "baseName_" の長さ
                int countOriginalDigit = originalName.Length > countBase ? originalName.Length - countBase : 0;

                // 重複するペインが存在しなくなるまで数字を増やしながら命名
                int countTrial = 0;
                name = GetNumberdString_(baseName, maxNameLength, countTrial, countOriginalDigit);
                while (findFunc(name) != null)
                {
                    countTrial++;
                    name = GetNumberdString_(baseName, maxNameLength, countTrial, countOriginalDigit);
                }
                return name;
            }
            else
            {
                // 問題なければ、そのまま返します。
                return originalName;
            }
        }
        #endregion

        #region シーン検索

        /// <summary>
        /// 未使用フォントセットを取得します。
        /// </summary>
        static public ILEFont[] GetUnuseFontSet(this ISubScene subScene)
        {
            List<ILEFont> unusedFontSet = new List<ILEFont>();
            foreach (ILEFont font in subScene.ILEFontManager.ILEFontSet)
            {
                bool fFontUsed = subScene.IPaneArray.FirstOrDefault((pane) => pane.UsesFont(font)) != null;
                if (!fFontUsed)
                {
                    unusedFontSet.Add(font);
                }
            }
            return unusedFontSet.ToArray();
        }

        /// <summary>
        /// 指定したフォント名のフォントが使われているかチェックします。
        /// </summary>
        static public bool CheckFontUsedByName(this ISubScene subScene, string fontName)
        {
            return !GetUnuseFontSet(subScene).Any((font) => font.FontName == fontName);
        }

        /// <summary>
        /// 指定テクスチャを使用しているペインセットを収集します。
        /// </summary>
        static public IPane[] GetUseTexturePaneSet(this ISubScene subScene, string texImgName)
        {
            ArrayList  paneSet = new ArrayList();
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.UsesTextureImage(texImgName))
                {
                    paneSet.Add(pane);
                }
            }
            return paneSet.ToArray(typeof(IPane)) as IPane[];
        }

        /// <summary>
        /// 指定テクスチャを使用しているかキャプチャもとになっているペインセットを収集します。
        /// </summary>
        static public IPane[] GetUseTextureAndCapturePaneSet(this ISubScene subScene, string texImgName)
        {
            ArrayList  paneSet = new ArrayList();
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.UsesTextureImage(texImgName) ||
                    pane.IsCapturedSource(texImgName))
                {
                    paneSet.Add(pane);
                }
            }
            return paneSet.ToArray(typeof(IPane)) as IPane[];
        }



        /// <summary>
        /// 隠された状態のペインセットを収集します。
        /// </summary>
        static public IPane[] GetHiddenPaneSet(ISubScene subScene)
        {
            ArrayList  paneSet = new ArrayList();
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.Hidden)
                {
                    paneSet.Add(pane);
                }
            }
            return paneSet.ToArray(typeof(IPane)) as IPane[];
        }

        /// <summary>
        /// ロックされた状態のペインセットを収集します。
        /// </summary>
        static public IPane[] GetLockedPaneSet(ISubScene subScene)
        {
            ArrayList  paneSet = new ArrayList();
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.Locked)
                {
                    paneSet.Add(pane);
                }
            }
            return paneSet.ToArray(typeof(IPane)) as IPane[];
        }

        /// <summary>
        /// ロック且つ隠されていない状態のペインセットを収集します。
        /// </summary>
        static public IPane[] GetLockedAndAppearPaneSet(ISubScene subScene)
        {
            ArrayList paneSet = new ArrayList();
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.Locked && !pane.Hidden)
                {
                    paneSet.Add(pane);
                }
            }
            return paneSet.ToArray(typeof(IPane)) as IPane[];
        }

        /// <summary>
        /// 隠された且つロックされていない状態のペインセットを収集します。
        /// </summary>
        static public IPane[] GetHiddenAndUnLockedPaneSet(ISubScene subScene)
        {
            ArrayList paneSet = new ArrayList();
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.Hidden && !pane.Locked)
                {
                    paneSet.Add(pane);
                }
            }
            return paneSet.ToArray(typeof(IPane)) as IPane[];
        }

        /// <summary>
        /// ペインの数を数えます。
        /// </summary>
        static public IPane[] GetPaneSet(ISubScene subScene, PaneKind kind, bool onlyActive)
        {
            ArrayList  paneSet = new ArrayList();
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.PaneKind == kind)
                {
                    if (!onlyActive || (!pane.Locked && !pane.Hidden))
                    {
                        paneSet.Add(pane);
                    }
                }
            }
            return paneSet.ToArray(typeof(IPane)) as IPane[];
        }

        /// <summary>
        /// 条件にあったペインを列挙します。
        /// </summary>
        static public IEnumerable<IPane> GetPaneSet(
            this ISubScene subScene, Func<IPane, bool> condition)
        {
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (condition(pane))
                {
                    yield return pane;
                }
            }
        }

        /// <summary>
        /// ペイン数を部品以下も含めて再帰的に数えます。
        /// </summary>
        static public void CalcSubsceneMetrics(this ISubScene subScene, SubSceneMetrics metrics)
        {
            if(metrics.ScreenH <= 0)
            {
                metrics.ScreenW = (int)subScene.BackGround.ScreenSize.X;
                metrics.ScreenH = (int)subScene.BackGround.ScreenSize.Y;
            }

            foreach (Pane pane in subScene.IPaneArray)
            {
                switch (pane.PaneKind)
                {
                    case PaneKind.Null:
                        metrics.NullPaneCount++;
                        break;
                    case PaneKind.Bounding:
                        metrics.BoundingPaneCount++;
                        break;
                    case PaneKind.Picture:
                        metrics.PicturePaneCount++;
                        switch (pane.IPicture.IMaterial.IMaterialTexMapSet.Length)
                        {
                            case 0: metrics.TotalPixcel0Tex += (int)(pane.Size.X * pane.Size.Y); break;
                            case 1: metrics.TotalPixcel1Tex += (int)(pane.Size.X * pane.Size.Y); break;
                            case 2: metrics.TotalPixcel2Tex += (int)(pane.Size.X * pane.Size.Y); break;
                            case 3: metrics.TotalPixcel3Tex += (int)(pane.Size.X * pane.Size.Y); break;
                        }
                        break;
                    case PaneKind.Textbox:
                        metrics.TextBoxPaneCount++;
                        metrics.TotalPixcel1Tex += (int)(pane.Size.X * pane.Size.Y);
                        break;
                    case PaneKind.Window:
                        metrics.WindowPaneCount++;
                        pane.ILEWindow.GetWindowPaneMetrics_(ref metrics);
                        break;
                    case PaneKind.Capture:
                        metrics.CapturePaneCount++;
                        // TODO:
                        // TotalPixel に加算すべきかもしれない
                        break;
                    case PaneKind.Alignment:
                        metrics.AligmentPaneCount++;
                        break;
                    case PaneKind.Scissor:
                        metrics.ScissorPaneCount++;
                        break;
                    case PaneKind.Parts:
                        metrics.PartsPaneCount++;
                        if (pane.IPartsLayout != null)
                        {
                            CalcSubsceneMetrics(pane.IPartsLayout.PartsSubScene, metrics);
                        }
                        break;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        static void GetWindowPartsMetrics_(this ILEWindow window, int numTexture, LEWindowPartsID id, ref SubSceneMetrics metrics)
        {
            FVec2 pos;
            FVec2 size;

            window.GetWindowPartsRectangle(numTexture, id, out pos, out size);

            switch (window.GetPartsMultiTextureCount(window.IntToPartsID(LEWindow.GetInternalWindowPartsID(window, id))))
            {
                case 0: metrics.TotalPixcel0Tex += (int)(size.X * size.Y); break;
                case 1: metrics.TotalPixcel1Tex += (int)(size.X * size.Y); break;
                case 2: metrics.TotalPixcel2Tex += (int)(size.X * size.Y); break;
                case 3: metrics.TotalPixcel3Tex += (int)(size.X * size.Y); break;
            }
        }

        /// <summary>
        ///
        /// </summary>
        static void GetWindowPaneMetrics_(this ILEWindow window, ref SubSceneMetrics metrics)
        {
            if (window.WindowKind == WindowKind.Around)
            {
                if (window.NumTexture == 2 || window.NumTexture == 3 || window.NumTexture == 5)
                {
                    if (!window.NotDrawContent)
                    {
                        GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.Content, ref metrics);
                    }

                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerLT, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerRT, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerRB, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerLB, ref metrics);
                }
                else if (window.NumTexture == 9)
                {
                    if (!window.NotDrawContent)
                    {
                        GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.Content, ref metrics);
                    }

                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerLT, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerRT, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerRB, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerLB, ref metrics);

                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.FrameT, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.FrameR, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.FrameB, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.FrameL, ref metrics);
                }
            }
            else if (window.WindowKind == WindowKind.HorizontalNoContent)
            {
                GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerLT, ref metrics);
                GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerRT, ref metrics);
            }
            else if (window.WindowKind == WindowKind.Horizontal)
            {
                if (!window.NotDrawContent)
                {
                    GetWindowPartsMetrics_(window, 9, LEWindowPartsID.CornerLT, ref metrics);
                    GetWindowPartsMetrics_(window, 9, LEWindowPartsID.FrameT, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerRT, ref metrics);
                }
                else
                {
                    GetWindowPartsMetrics_(window, 9, LEWindowPartsID.CornerLT, ref metrics);
                    GetWindowPartsMetrics_(window, window.NumTexture, LEWindowPartsID.CornerRT, ref metrics);
                }
            }
        }

        #region ペインの検索

        /// <summary>
        /// 種類を指定してペインを列挙します。
        /// </summary>
        static public IEnumerable<IPane> FindPanesByKind(this ISubScene subScene, PaneKind kind)
        {
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.PaneKind == kind)
                {
                    yield return pane;
                }
            }
        }

        /// <summary>
        /// 種類を指定してペインを列挙します。
        /// </summary>
        static public IEnumerable<IPane> FindPanesByKindRecursively(this ISubScene subScene, PaneKind kind)
        {
            if(kind == PaneKind.Parts)
            {
                foreach (IPane pane in subScene.IPaneArray)
                {
                    if (pane.PaneKind == kind)
                    {
                        yield return pane;

                        // 再帰的に列挙する。
                        var partsResults = FindPanesByKindRecursively(pane.IPartsLayout.PartsSubScene, kind);
                        foreach (var partsResult in partsResults)
                        {
                            yield return partsResult;
                        }
                    }
                }
            }
            else
            {
                foreach (var result in FindPanesByKind(subScene, kind))
                {
                    yield return result;
                }
            }
        }

        /// <summary>
        /// ペイン名からペインを検索します。
        /// </summary>
        /// <param name="name"></param>
        /// <returns></returns>
        static public IPane FindPaneByName(this ISubScene subScene, string name)
        {
            if (name == subScene.RootIPane.PaneName)
            {
                return subScene.RootIPane;
            }

            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.PaneName == name)
                {
                    return pane;
                }
            }
            // 発見できなかった
            return null;
        }

        /// <summary>
        /// 名前を指定してペインを列挙します。
        /// </summary>
        static public IPane FindPaneByNameRecursively(this ISubScene subScene, string name)
        {
            IPane result = FindPaneByName(subScene, name);

            if (result == null)
            {
                foreach (IPane pane in subScene.IPaneArray)
                {
                    if (pane.PaneKind == PaneKind.Parts)
                    {
                        // 再帰的に列挙する。
                        result = FindPaneByNameRecursively(pane.IPartsLayout.PartsSubScene, name);
                        if (result != null)
                        {
                            break;
                        }
                    }
                }
            }

            return result;
        }

        /// <summary>
        /// マテリアル名からペインを検索します。
        /// </summary>
        static public IPane FindPaneByMaterialName(this ISubScene subScene, Type dataType, string name)
        {
            foreach (Pane pane in subScene.IPaneArray)
            {
                Material result = pane.FindMaterialByName(name);
                if (result != null)
                {
                    // ペインを返します。
                    return pane;
                }
            }
            return null;
        }

        /// <summary>
        /// マテリアル名を指定して、マテリアルを検索します。
        /// </summary>
        static public IMaterial FindMaterialByName(this ISubScene subScene, string name)
        {
            foreach (Pane pane in subScene.IPaneArray)
            {
                Material result = pane.FindMaterialByName(name);
                if (result != null)
                {
                    return result;
                }
            }
            return null;
        }

        /// <summary>
        /// 指定したタグ区間にキーを持つペインを列挙します
        /// </summary>
        static public IEnumerable<IPane> FindPanesByKeyFrame(this ISubScene subScene, int startFrame, int endFrame, string tagName)
        {
            List<IPane> paneSet = new List<IPane>();
            foreach (IPane pane in subScene.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    if (curveAttr.CheckHaveKeyRecursiveInTargetTag(startFrame, endFrame, tagName, subScene.IsAnimEditSeparateMode()))
                    {
                        paneSet.Add(pane);
                        break;
                    }
                }
            }

            return paneSet;
        }

        #endregion ペインの検索

        #endregion シーン検索

        #region ヒットテスト

        #region シーン内のペインの取得

        #region 非公開関数
        /// <summary>
        /// 指定点における衝突判定を描画順におこないます。
        ///
        /// Hidden Locked 状態のペイン以下の階層では、処理を中断し、
        /// 判定を行いません。
        /// </summary>
        static IPane GetPanesByPointHierarchyOrder_(IPane pane, FVec2 posInScene, Func<IPane, bool> terminateTraverseCondition)
        {
            IPane hitPane = null;

            // BVが点を含むか判定します。
            if (pane.BoundingVolumeInWorld.Contains(posInScene.AsPointF))
            {
                if (!pane.Hidden && !pane.Locked)
                {
                    hitPane = pane;
                }
            }

            if(terminateTraverseCondition(pane))
            {
                // 子供について再帰的に処理を行います。
                foreach (Pane childPane in pane.Children)
                {
                    // 衝突が検出された場合は、衝突ペインを更新します。
                    //
                    // 同じ子供同士で衝突が複数起きた場合には、最後に描画される
                    // (一番上に表示される)ペインが判定結果になります。
                    IPane result = GetPanesByPointHierarchyOrder_(childPane, posInScene, terminateTraverseCondition);
                    if (result != null)
                    {
                        hitPane = result;
                    }
                }
            }

            return hitPane;
        }

        /// <summary>
        /// 指定矩形における衝突判定を描画順におこないます。
        /// </summary>summary>
        static void GetPanesByRectHierarchyOrder_(IPane pane, RectangleF rectInScene, ArrayList hitPaneSet, Func<IPane, bool> terminateTraverseCondition)
        {
            // BVが点を含むか判定します。
            RectangleF paneBV = pane.BoundingVolumeInWorld;
            if (RectangleF.Intersect(paneBV, rectInScene) != RectangleF.Empty)
            {
                if (!pane.Hidden && !pane.Locked)
                {
                    hitPaneSet.Add(pane);
                }
            }

            if(terminateTraverseCondition(pane))
            {
                // 子供について再帰的に処理を行います。
                foreach (Pane childPane in pane.Children)
                {
                    GetPanesByRectHierarchyOrder_(childPane, rectInScene, hitPaneSet, terminateTraverseCondition);
                }
            }
        }

        static IPane GetPanesByPoint_(ISubScene subScene, FVec2 posInScene, bool bSelectLocked, Func<IPane, bool> terminateTraverseCondition)
        {
            IPane hitPane = GetPanesByPointHierarchyOrder_(subScene.RootIPane, posInScene, terminateTraverseCondition);

            if (hitPane != null)
            {
                if (bSelectLocked || !hitPane.Locked)
                {
                    if (!hitPane.Hidden)
                    {
                        return hitPane;
                    }
                }
            }
            return null;
        }

        /// <summary>
        /// 指定矩形内における、衝突ペインを返します。
        ///
        /// </summary>
        static ArrayList GetPanesByRect_(ISubScene subScene, RectangleF rectInScene, bool bSelectLocked, Func<IPane, bool> terminateTraverseCondition)
        {
            ArrayList    selPanes = new ArrayList();

            // すべてのペインについて、選択矩形が接触するなら選択セットに登録します。
            GetPanesByRectHierarchyOrder_(subScene.RootIPane, rectInScene, selPanes, terminateTraverseCondition);

            // 必要があれば、ロックされたペインを取り除く
            IPane[] paneSet = selPanes.ToArray(typeof(IPane)) as IPane[];
            foreach (IPane pane in paneSet)
            {
                if (pane.Hidden)
                {
                    selPanes.Remove(pane);
                }
                else
                {
                    if (!bSelectLocked && pane.Locked)
                    {
                        selPanes.Remove(pane);
                    }
                }
            }


            return selPanes;
        }
        #endregion 非公開関数

        static public IPane GetPanesByPoint(ISubScene subScene, FVec2 posInScene)
        {
            return GetPanesByPoint(subScene, posInScene, (pane) => true);
        }

        /// <summary>
        /// 指定点における最初に検出された、衝突ペインを返します。
        ///
        /// </summary>
        static public IPane GetPanesByPoint(ISubScene subScene, FVec2 posInScene, Func<IPane, bool> terminateTraverseCondition)
        {
            return GetPanesByPoint_(subScene, posInScene, SelectLockedDisable, terminateTraverseCondition);
        }

        /// <summary>
        /// 指定矩形内における、衝突ペインを返します。
        ///
        /// </summary>
        static public ArrayList GetPanesByRect(ISubScene subScene, RectangleF rectInScene)
        {
            return GetPanesByRect(subScene, rectInScene, (pane) => true);
        }

        /// <summary>
        /// 指定矩形内における、衝突ペインを返します。
        ///
        /// </summary>
        static public ArrayList GetPanesByRect(ISubScene subScene, RectangleF rectInScene, Func<IPane, bool> terminateTraverseCondition)
        {
            return GetPanesByRect_(subScene, rectInScene, SelectLockedDisable, terminateTraverseCondition);
        }
        #endregion シーン内のペインの取得

        #endregion ヒットテスト

        #region パーツ

        /// <summary>
        /// パーツ用のサブシーンかどうかを調べます。
        /// </summary>
        /// <param name="subScene">サブシーンです。</param>
        /// <returns>パーツ用のサブシーンなら true を返します。</returns>
        public static bool IsPartsScene(this ISubScene subScene)
        {
            return subScene.IPartsSettings != null;
        }

        /// <summary>
        /// 部品ペインを持っているか調べます。
        /// </summary>
        public static bool ContainsPartsPane(this ISubScene subScene)
        {
            return subScene.GetPaneSet((pane) => pane.PaneKind == PaneKind.Parts).Count() > 0;
        }

        /// <summary>
        /// 部品リロード
        /// </summary>
        public static void ReloadAllPartsPanes(this ISubScene subScene)
        {
            if(_isPartsReloading)
            {
                return;
            }

            try
            {
                _isPartsReloading = true;
                subScene.BeginMassiveModify();

                List<string> reloadedFiles = new List<string>();
                LayoutEditorCore.MsgReporter.BeginPacking(LECoreStringResMgr.Get("ERROR_FAILED_TO_RELOAD_PARTSPANE_TITILE"));
                foreach (var partsPane in subScene.GetPaneSet((pane) => pane.PaneKind == PaneKind.Parts).ToArray())
                {
                    // 参照ファイルをすべて再読み込みする。
                    {
                        // 参照レイアウト
                        List<string> relatedFiles = new List<string>(partsPane.IPartsLayout.EnumlateReferencingLayoutFiles());

                        // 基底部品
                        IPartsSubScene partsSubScene = Scene.Instance.FindPartsSubSceneByFileName(partsPane.IPartsLayout.PartsLayoutName);
                        if (partsSubScene != null && partsSubScene.IsDerivativeParts())
                        {
                            relatedFiles.Add(partsSubScene.BasePartsName);
                        }

                        foreach (var referencingLayoutFile in relatedFiles)
                        {
                            if (!reloadedFiles.Contains(referencingLayoutFile))
                            {
                                Scene.Instance.ReloadOnePartsSubScene(referencingLayoutFile);
                                reloadedFiles.Add(referencingLayoutFile);
                            }
                        }
                    }

                    // 部品ペインをリロードします。
                    PartsLayoutManipulator.ReloadPartsPane(partsPane);
                }
            }
            finally
            {
                LayoutEditorCore.MsgReporter.EndPacking();
                subScene.EndMassiveModify();
                _isPartsReloading = false;
            }
        }

        /// <summary>
        /// 部品リロード
        /// </summary>
        public static void ReloadAllPartsPanesWithoutEventNotifications(this ISubScene subScene)
        {
            try
            {
                LECore.Structures.Scene.Instance.ForceDisableEventNotifications = true;
                LayoutEditorCore.MsgReporter.ForceStopErrorReport = true;
                ReloadAllPartsPanes(subScene);
            }
            finally
            {
                LECore.Structures.Scene.Instance.ForceDisableEventNotifications = false;
                LayoutEditorCore.MsgReporter.ForceStopErrorReport = false;
            }
        }

        #endregion

        #region アニメーションモード切り替え

        /// <summary>
        /// セクション間のマージン値。
        /// </summary>
        public static int SectionMargin
        {
            get { return _sectionMargin; }
        }

        /// <summary>
        /// UserDataの新規作成時に適切なAnmCurveを生成します。
        /// </summary>
        public static void AddAllAnimTagsToAttribute(ISubScene target, IAnmAttribute targetAttr)
        {
            if (!target.IsAnimEditSeparateMode())
            {
                Debug.Assert(false);
                return;
            }

            foreach (IAnimFrameSection section in target.IAnimFrameSectionSet.IAnimFrameSectionSet)
            {
                foreach (AnmAttribute curveAttr in targetAttr.EnumCurveAttribute())
                {
                    curveAttr.AddAnmTag(section.Name);
                }
            }
        }

        /// <summary>
        /// アニメーション区間タグのパラメータを算出します。
        /// </summary>
        public static Dictionary<string, Tuple<int, int>> CalcSectionParam(
            IAnimFrameSectionSet sectionSet,
            Dictionary<string, int> rangeList,
            Dictionary<string, Tuple<int, int>> marginList)
        {
            Dictionary<string, Tuple<int, int>> ret = new Dictionary<string, Tuple<int, int>>();
            int startFrame = 0;
            int endFrame = 0;
            foreach (IAnimFrameSection section in sectionSet.IAnimFrameSectionSet)
            {
                if (rangeList.ContainsKey(section.Name) && marginList.ContainsKey(section.Name))
                {
                    int sectionRange = (rangeList[section.Name] <= 1) ? rangeList[section.Name] : section.EndFrame - section.StartFrame;

                    startFrame += marginList[section.Name].Item1;
                    endFrame = startFrame + sectionRange;

                    ret.Add(section.Name, new Tuple<int, int>(startFrame, endFrame));

                    startFrame = endFrame + marginList[section.Name].Item2 + _sectionMargin;
                }
            }

            return ret;
        }

        /// <summary>
        /// タグ区間ごとの最大時間幅を算出します。
        /// </summary>
        public static Dictionary<string, int> CalcSectionMaxRange(IEnumerable<IPane> paneArray, IAnimFrameSectionSet sectionSet)
        {
            Dictionary<string, int> maxRangeList = new Dictionary<string, int>();
            Dictionary<string, Tuple<int, int>> keyInfoList = new Dictionary<string, Tuple<int, int>>();

            foreach (IPane pane in paneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    // rangeの算出
                    var rangeList = CalcSectionRange_(sectionSet, curveAttr);

                    // 最大最小値を求める
                    foreach (KeyValuePair<string, Tuple<int, int>> pair in rangeList)
                    {
                        if (keyInfoList.ContainsKey(pair.Key))
                        {
                            var compList = keyInfoList[pair.Key];

                            int min = (compList.Item1 > pair.Value.Item1) ? pair.Value.Item1 : compList.Item1;
                            int max = (compList.Item2 < pair.Value.Item2) ? pair.Value.Item2 : compList.Item2;

                            keyInfoList[pair.Key] = new Tuple<int, int>(min, max);
                        }
                        else
                        {
                            keyInfoList[pair.Key] = pair.Value;
                        }
                    }

                    // 最大最小値からrangeを求める
                    foreach (KeyValuePair<string, Tuple<int, int>> pair in keyInfoList)
                    {
                        int range = pair.Value.Item2 - pair.Value.Item1;
                        maxRangeList[pair.Key] = range;
                    }
                }

            }

            return maxRangeList;
        }

        /// <summary>
        /// タグ区間ごとの最大時間幅を算出します。
        /// </summary>
        private static Dictionary<string, Tuple<int, int>> CalcSectionRange_(
            IAnimFrameSectionSet sectionSet,
            AnmAttribute attr)
        {
            Dictionary<string, Tuple<int, int>> ret = new Dictionary<string, Tuple<int, int>>();

            foreach (IAnimFrameSection section in sectionSet.IAnimFrameSectionSet)
            {
                AnmCurve curve = attr.GetAnmCurve(section.Name);
                if (curve != null)
                {
                    if (curve.NumKeyFrame <= 0)
                    {
                        ret[section.Name] = new Tuple<int, int>(section.StartFrame, section.EndFrame);
                        continue;
                    }

                    IAnmKeyFrame firstKey = curve.GetFirstKey();
                    IAnmKeyFrame lastKey = curve.GetLastKey();

                    ret[section.Name] = new Tuple<int, int>(firstKey.TimeAsInt, lastKey.TimeAsInt);
                }
                else
                {
                    ret[section.Name] = new Tuple<int, int>(section.StartFrame, section.EndFrame);
                }
            }

            return ret;
        }

        /// <summary>
        /// タグ区間ごとの最大マージンを算出します。
        /// </summary>
        public static Dictionary<string, Tuple<int, int>> CalcSectionMaxMargin(IEnumerable<IPane> paneArray, IAnimFrameSectionSet sectionSet)
        {
            Dictionary<string, Tuple<int, int>> maxMarginList = new Dictionary<string, Tuple<int, int>>();

            foreach (IPane pane in paneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    // marginの算出
                    var marginList = CalcSectionMargin_(sectionSet, curveAttr);

                    // 最大最小値を求める
                    foreach (KeyValuePair<string, Tuple<int, int>> pair in marginList)
                    {
                        if (maxMarginList.ContainsKey(pair.Key))
                        {
                            if (maxMarginList[pair.Key].Item1 < pair.Value.Item1)
                            {
                                maxMarginList[pair.Key] = new Tuple<int, int>(pair.Value.Item1, maxMarginList[pair.Key].Item2);
                            }

                            if (maxMarginList[pair.Key].Item2 < pair.Value.Item2)
                            {
                                maxMarginList[pair.Key] = new Tuple<int, int>(maxMarginList[pair.Key].Item1, pair.Value.Item2);
                            }
                        }
                        else
                        {
                            maxMarginList[pair.Key] = pair.Value;
                        }
                    }
                }
            }

            return maxMarginList;
        }

        /// <summary>
        /// タグ区間ごとの最大マージンを算出します。
        /// </summary>
        private static Dictionary<string, Tuple<int, int>> CalcSectionMargin_(
            IAnimFrameSectionSet sectionSet,
            AnmAttribute attr)
        {
            Dictionary<string, Tuple<int, int>> ret = new Dictionary<string, Tuple<int, int>>();

            foreach (AnimFrameSection animFramesection in sectionSet.IAnimFrameSectionSet)
            {
                // タグ名に対応するアニメーションが無い場合はスキップ
                AnmCurve anmCurve = attr.GetAnmCurve(animFramesection.Name); ;
                if (anmCurve == null)
                {
                    ret.Add(animFramesection.Name, new Tuple<int, int>(0, 0));
                    continue;
                }

                // 左右のマージンを算出する
                int leftMargin = 0;
                int rightMargin = 0;
                IAnmKeyFrame first_key = anmCurve.GetFirstKey();
                IAnmKeyFrame last_key = anmCurve.GetLastKey();
                if (first_key != null && last_key != null)
                {
                    leftMargin = first_key.TimeAsInt < 0 ? (animFramesection.StartFrame - first_key.TimeAsInt) : 0;
                    rightMargin = last_key.TimeAsInt > animFramesection.EndFrame ? (last_key.TimeAsInt - animFramesection.EndFrame) : 0;
                }

                ret.Add(animFramesection.Name, new Tuple<int, int>(leftMargin, rightMargin));
            }

            return ret;
        }

        /// <summary>
        /// アトリビュートに区間タグアニメーションを追加します。
        /// この操作はコマンドに登録されません。
        /// </summary>
        public static void AddAnimSectionDirect(ISubScene target, string tagName)
        {
            SubScene _target = target as SubScene;

            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            foreach (IPane pane in _target.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    // value set
                    curveAttr.AddAnmTag(tagName);
                }
            }
        }

        /// <summary>
        /// アニメーションカーブを参照登録します。
        /// </summary>
        public static void MakeReferenceCurves(ISubScene target)
        {
            SubScene _target = target as SubScene;

            if (_target == null || !_target.IsAnimEditSeparate)
            {
                return;
            }

            foreach (IPane pane in _target.IPaneArray)
            {
                foreach (AnmAttribute curveAttr in (pane as IAnmAttribute).EnumCurveAttribute())
                {
                    if (curveAttr.IsReferenceCurves)
                    {
                        if (curveAttr.CurveNum > 1)
                        {
                            var tags = GetTags(_target);
                            string srcTag = tags[1];
                            foreach (AnimFrameSection animFramesection in _target.IAnimFrameSectionSet.IAnimFrameSectionSet)
                            {
                                curveAttr.ReferenceAnmCurve(animFramesection.Name);
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// アニメーションを持つすべてのペインアトリビュートをbaseValueに戻します。
        /// </summary>
        public static void ResetValueAsBaseValue(ISubScene target)
        {
            SubScene subScene = target as SubScene;

            subScene.BeginMassiveModify();

            foreach (IPane pane in subScene.IPaneArray)
            {
                PaneHelper.ResetValueAsBaseValue(pane);
            }

            subScene.EndMassiveModify();
        }

        /// <summary>
        /// アニメーションの再評価を行ないます。
        /// </summary>
        public static void EvaluateAnimation(ISubScene target, int time)
        {
            if(target == null)
            {
                return;
            }

            SubScene subScene = target as SubScene;

            subScene.EvaluateMultiAnimation = true; // イベントを抑制する

            EvaluateAnimationForce(subScene, time);

            subScene.EvaluateMultiAnimation = false;

            subScene.RaisePaneModifyEvent(subScene.IPaneArray);
        }

        /// <summary>
        /// アニメーションの再評価を行ないます。
        /// </summary>
        public static void EvaluateAnimationForce(ISubScene target, int time)
        {
            if (target == null) return;

            foreach (Pane pane in target.IPaneArray)
            {
                PaneAnimationHelper.EvaluateAnimationForce(pane, time);
            }
        }


        /// <summary>
        /// アニメーション区間タグの一覧を取得します。
        /// 先頭には結合モード用カーブの参照用に空文字がセットされます。
        /// </summary>
        public static string[] GetTags(ISubScene target)
        {
            List<string> ret = new List<string>();

            if (target != null)
            {
                ret.Add(String.Empty);

                foreach (AnimFrameSection section in target.IAnimFrameSectionSet.IAnimFrameSectionSet)
                {
                    ret.Add(section.Name);
                }
            }

            return ret.ToArray();
        }

        /// <summary>
        /// 指定したタグ区間の前後のタグ区間を取得します。
        /// </summary>
        public static void GetBeforeAfterAnimSection(this ISubScene target, out IAnimFrameSection beforeSection, out IAnimFrameSection afterSection)
        {
            beforeSection = null;
            afterSection = null;

            if (target.IAnimFrameSectionSet.TargetIAnimFrameSection == null) return;

            bool bCurrentFound = false;
            foreach (IAnimFrameSection section in target.IAnimFrameSectionSet.IAnimFrameSectionSet)
            {
                if (section.Name == target.CurrentTagName)
                {
                    bCurrentFound = true;
                    continue;
                }

                if (!bCurrentFound)
                {
                    beforeSection = section;
                }
                else
                {
                    afterSection = section;
                    break;
                }
            }
        }

        public static bool IsAnimEditSeparateMode(this ISubScene target)
        {
            return target != null && (target as SubScene).IsAnimEditSeparate;
        }

        /// <summary>
        /// アニメーションが編集可能な状態か判定します。
        /// </summary>
        public static bool IsAnimationEditable(this ISubScene target)
        {
            // 対象が無い場合は不可
            if (target == null)
            {
                return false;
            }

            // 分割モードで
            if (target.IsAnimEditSeparateMode())
            {
                // タグが非選択の場合は、アニメーション編集が不可です。
                if (target.IAnimFrameSectionSet == null || target.IAnimFrameSectionSet.TargetIAnimFrameSection == null)
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 基本値が編集可能な状態か判定します。
        /// </summary>
        public static bool IsBaseValueEditable(this ISubScene target)
        {
            // 対象が無い場合は不可
            if (target == null)
            {
                return false;
            }

            // 分割モードで
            if (target.IsAnimEditSeparateMode())
            {
                // タグが選択されている場合は、基本値編集が不可能です。
                if (target.IAnimFrameSectionSet != null && target.IAnimFrameSectionSet.TargetIAnimFrameSection != null)
                {
                    return false;
                }
            }

            return true;
        }


        #endregion

        #region テクスチャパス変更

        /// <summary>
        /// エラーの種類です。
        /// </summary>
        public enum ChangeTexturePathError
        {
            None,
            FailToChange,
            InvalidSameNameTexture
        }

        /// <summary>
        /// ペイン内のテクスチャパスを変更します。
        /// </summary>
        private static bool ChangeTexturePathForPane_(IPane pane, string newFilePath, string oldTexName, ITextureImage texImage, out ChangeTexturePathError errorMessageId)
        {
            bool bResult = true;
            errorMessageId = ChangeTexturePathError.None;

            // 部品の場合は、関与するペインを列挙して再帰的に処理を実行します。
            if (pane.PaneKind == PaneKind.Parts)
            {
                foreach (var overwritePane in pane.IPartsLayout.EnumeratePanesOverwritingTextures())
                {
                    // 一つでもエラーになったらエラー
                    ChangeTexturePathError subErrorMessageId;
                    if(!ChangeTexturePathForPane_(overwritePane, newFilePath, oldTexName, texImage, out subErrorMessageId))
                    {
                        errorMessageId = ChangeTexturePathError.FailToChange;
                        bResult = false;
                    }
                }
            }

            // 関係するマテリアルについて更新処理を実施する。
            var materialMnp = new RevMaterialManipulator();
            foreach (var mtl in pane.EnumrateDetailMaterial().Where(m => MaterialHelper.UsesTextureImage(m.IMaterial, oldTexName)))
            {
                materialMnp.BindTarget(mtl);
                for (var i = 0; i < mtl.IMaterialTexMapSet.Length; i++)
                {
                    IMaterialTexMap texMap = mtl.IMaterialTexMapSet[i];

                    // テクスチャーパターンリストの変更
                    {
                        bool isTextureListModified = false;
                        var textureSubSetNameSet = new List<string>(texMap.ITextureChanger.ITextureSubSet.TextureSubSetNameSet);
                        for (var texIdx = 0; texIdx < textureSubSetNameSet.Count; texIdx++)
                        {
                            if (textureSubSetNameSet[texIdx] == oldTexName)
                            {
                                textureSubSetNameSet[texIdx] = texImage.Name;
                                isTextureListModified = true;
                            }
                        }

                        if (isTextureListModified)
                        {
                            TextureChangerManipulator texChangerMnp = new TextureChangerManipulator();
                            texChangerMnp.BindTarget(texMap.ITextureChanger);
                            texChangerMnp.SetAll(textureSubSetNameSet.ToArray());
                        }
                    }

                    if (texMap.TexImgName == oldTexName)
                    {
                        materialMnp.RegisterMatTexture(texImage.Name, i);
                    }
                }
            }

            return bResult;
        }

        public static bool ChangeTexturePath(ISubScene dstSubScene, string newFilePath, ITextureImage texImage, out ChangeTexturePathError errorMessageId)
        {
            bool bResult;
            errorMessageId = ChangeTexturePathError.None;

            // ファイル以外のタイプのテクスチャはパスの設定はできない。
            if (texImage.SourceType != LECoreInterface.TextureSourceType.File)
            {
                return false;
            }

            TextureMgrManipulator _TextureMgrManipulator = new TextureMgrManipulator();
            _TextureMgrManipulator.BindTarget(dstSubScene.ITextureMgr);

            dstSubScene.BeginMassiveModify();
            _TextureMgrManipulator.BeginUpdate();

            // ファイル名が同じかどうかで処理を分ける
            if (string.Compare(Path.GetFileNameWithoutExtension(newFilePath), texImage.Name, StringComparison.OrdinalIgnoreCase) == 0)
            {
                // 同名別パスファイル
                bResult = _TextureMgrManipulator.UpdateTextureImage(texImage, newFilePath);
                if(!bResult)
                {
                    errorMessageId = ChangeTexturePathError.FailToChange;
                }
            }
            else
            {
                // 別名ファイル
                // 新しいファイル名と同名テクスチャが以前に登録されているときは変更不可
                var sameNameImage = dstSubScene.ITextureMgr.FindITextureImageByName(Path.GetFileNameWithoutExtension(newFilePath));
                if (sameNameImage != null)
                {
                    bResult = false;
                    errorMessageId = ChangeTexturePathError.InvalidSameNameTexture;
                }
                else
                {
                    // テクスチャを使用しているペインのマテリアルテクスチャを更新します。
                    var oldTexName = texImage.Name;
                    bResult = _TextureMgrManipulator.UpdateTextureImage(texImage, newFilePath);
                    if (bResult)
                    {
                        // テクスチャを使用しているペインを取得します。
                        var paneSet = dstSubScene.GetUseTexturePaneSet(oldTexName);
                        foreach(var pane in paneSet)
                        {
                            bResult &= ChangeTexturePathForPane_(pane, newFilePath, oldTexName, texImage, out errorMessageId);
                        }
                    }
                    else
                    {
                        errorMessageId = ChangeTexturePathError.FailToChange;
                    }
                }
            }

            _TextureMgrManipulator.EndUpdate();
            dstSubScene.EndMassiveModify();

            return bResult;
        }

        #endregion

        #region 複製

        //----------------------------------------------------------
        // 複製
        //----------------------------------------------------------

        internal static void ClonePanes(SubScene dstSubScene, ISubScene srcSubScene)
        {
            ClonePanes_(dstSubScene, srcSubScene, true);
        }

        internal static void ClonePanesWithoutAnimation(SubScene dstSubScene, ISubScene srcSubScene)
        {
            ClonePanes_(dstSubScene, srcSubScene, false);
        }

        internal static void ClonePanesWithoutAnimationOnceBindAnim(SubScene dstSubScene, ISubScene srcSubScene)
        {
            ClonePanes_(dstSubScene, srcSubScene, true);

            foreach (Pane pane in dstSubScene.PaneSet)
            {
                // アニメーションは、一度コピーして0フレームの値で評価したあとを全削除する
                AnmAttributeHelper.EvaluateAnimAll(pane, 0);
                AnmAttributeHelper.RemoveKeyFrameRecursive(pane, (attr) => (attr is AnmAttribute) && !(attr as AnmAttribute).IsReferenceCurves);
            }
        }

        private static void ClonePanes_(SubScene dstSubScene, ISubScene srcSubScene, bool copyAnimaitons)
        {
            Ensure.Operation.True(dstSubScene.IPaneArray.Count() == 0);

            // ペインを複製して登録。
            foreach (var pane in srcSubScene.IPaneArray)
            {
                Pane newTempPane = copyAnimaitons ? pane.Clone() : pane.CloneWithoutAnimation();//
                dstSubScene.AddPane(newTempPane);
            }

            // 親子設定を行う。
            SubSceneHelper.CloneHierarcy(dstSubScene, srcSubScene, srcSubScene.RootIPane as IPane);
        }

        /// <summary>
        /// 親子設定を複製します。
        /// </summary>
        internal static void CloneHierarcy(SubScene dstSubScene, ISubScene srcSubScene, IPane pane)
        {
            var parentPane = pane.Parent as IPane;
            if (parentPane != null)
            {
                var dstParentPane = dstSubScene.FindPaneByName(parentPane.PaneName) as Pane;
                if (dstParentPane != null)
                {
                    var dstPane = dstSubScene.FindPaneByName(pane.PaneName) as Pane;
                    Ensure.Operation.ObjectNotNull(dstPane);
                    dstParentPane.AddChildNodeByLoaclCoordinate(dstPane);
                }
            }

            foreach (IPane childPane in pane.Children)
            {
                CloneHierarcy(dstSubScene, srcSubScene, childPane);
            }
        }

        /// <summary>
        ///
        /// </summary>
        internal static SubScene CloneSubSceneWithoutAnim(ISubScene sourceSubScene)
        {
            return CloneSubScene_(sourceSubScene, false);
        }

        /// <summary>
        ///
        /// </summary>
        internal static SubScene CloneSubScene(ISubScene sourceSubScene)
        {
            return CloneSubScene_(sourceSubScene, true);
        }

        /// <summary>
        /// フォントやテクスチャ類をコピーする
        /// </summary>
        public static void CopySubSceneTextureAndFont(ISubScene sourceSubScene, ISubScene dstSubScene)
        {
            foreach (var font in sourceSubScene.ILEFontManager.ILEFontSet)
            {
                (dstSubScene.ILEFontManager as FontManager).CreateLEFont(font.FontName, font.FontPath, font.FontSettings);
            }

            try
            {
                (dstSubScene.ITextureMgr as TextureMgr).BeginUpdate();
                foreach (var texture in sourceSubScene.ITextureMgr.ITextureImageSet)
                {
                    TextureMgr texMgr = dstSubScene.ITextureMgr as TextureMgr;

                    ITextureImage newImg = null;
                    switch (texture.SourceType)
                    {
                        case LECoreInterface.TextureSourceType.File:
                            newImg = texMgr.RegisterTextureImageFromFile(texture.FilePath);
                            break;
                        case LECoreInterface.TextureSourceType.Dynamic:
                            // CopySubSceneCaptureTexture で処理している。
                            break;
                    }

                    if (newImg != null &&
                        !newImg.PixelFmtIsFixed)
                    {
                        newImg.PixelFmt = texture.PixelFmt;
                    }
                }
            }
            finally
            {
                (dstSubScene.ITextureMgr as TextureMgr).EndUpdate();
            }
        }

        /// <summary>
        /// キャプチャテクスチャをコピーする
        /// </summary>
        public static void CopySubSceneCaptureTexture(ISubScene sourceSubScene, ISubScene dstSubScene)
        {
            try
            {
                (dstSubScene.ITextureMgr as TextureMgr).BeginUpdate();
                foreach (var texture in sourceSubScene.ITextureMgr.ITextureImageSet)
                {
                    TextureMgr texMgr = dstSubScene.ITextureMgr as TextureMgr;

                    ITextureImage newImg = null;
                    switch (texture.SourceType)
                    {
                        case LECoreInterface.TextureSourceType.File:
                            // CopySubSceneTextureAndFont で処理している。
                            break;
                        case LECoreInterface.TextureSourceType.Dynamic:
                            ITextureImage sourceCaptureTexture = TextureMgrHelper.FindCaptureTextureByName(sourceSubScene.ITextureMgr, texture.Name);
                            IPane pane = null;

                            if (sourceCaptureTexture.ICaptureTexture.OwnerPane != null)
                            {
                                pane = dstSubScene.FindPaneByName(sourceCaptureTexture.ICaptureTexture.OwnerPane.PaneName);
                            }
                            bool isHidingFromList = sourceCaptureTexture.IsHidingFromList;
                            newImg = texMgr.RegisterCaptureTexture(pane, texture.Name, isHidingFromList, sourceCaptureTexture.ICaptureTexture.Usage);

                            (newImg.ICaptureTexture as CaptureTexture).CopyCaptureParams(sourceCaptureTexture.ICaptureTexture);
                            break;
                    }

                    if (newImg != null &&
                        !newImg.PixelFmtIsFixed)
                    {
                        newImg.PixelFmt = texture.PixelFmt;
                    }
                }
            }
            finally
            {
                (dstSubScene.ITextureMgr as TextureMgr).EndUpdate();
            }
        }

        /// <summary>
        /// アニメーション情報を持っているか
        /// </summary>
        /// <param name="subScene"></param>
        public static bool HasAnyAnimationOrTexturePatternTexMap(ISubScene subScene)
        {
            bool subSceneHasAnimation = subScene.IPaneArray.Any((pane) => pane.HasAnyAnimation() || pane.UsedMaterials.Any((mat) => mat.HasAnyAnimationOrTexturePatternTexMap()));
            if (subSceneHasAnimation)
            {
                return true;
            }

            if (subScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Count() > 0)
            {
                return true;
            }

            if (subScene.IsAnimEditSeparateMode())
            {
                return true;
            }

            return false;
        }

        /// <summary>
        /// 分割モードを設定します。テスト用途でのみ利用するようにしてください。
        /// </summary>
        public static void SetAnimEditSeparateForTest(ISubScene subScene, bool isSeparate)
        {
            (subScene as SubScene).IsAnimEditSeparate = isSeparate;
        }

        /// <summary>
        /// 部品レイアウトを部品ペイン毎に複製します。
        /// </summary>
        private static SubScene CloneSubScene_(ISubScene sourceSubScene, bool cloneAnim)
        {
            SubScene dstSubScene = Scene.Instance.CreateSubSceneInstance();

            dstSubScene.BackGround.Set(sourceSubScene.BackGround);

            // 部品設定
            dstSubScene.IPartsSettings.Set(dstSubScene, sourceSubScene.IPartsSettings);

            // コントロール設定
            dstSubScene.ControlSettingsSet.Set(sourceSubScene.IControlSettings);

            // フォントやテクスチャ類をコピーする
            CopySubSceneTextureAndFont(sourceSubScene, dstSubScene);

            // ペインを複製して登録。
            // 親子設定を行う。
            if (cloneAnim)
            {
                SubSceneHelper.ClonePanes(dstSubScene, sourceSubScene);
            }
            else
            {
                SubSceneHelper.ClonePanesWithoutAnimationOnceBindAnim(dstSubScene, sourceSubScene);
            }

            // キャプチャテクスチャをコピーする
            CopySubSceneCaptureTexture(sourceSubScene, dstSubScene);

            // ステート設定をコピーする
            (dstSubScene.IStateMachine as StateMachineData).Set(sourceSubScene.IStateMachine);

            return dstSubScene;
        }

        #endregion //複製

        #region キャプチャペイン

        public static IPane[] GetSelectedPaneArrayWithChildren(ISubScene scene)
        {
            return (scene as SubScene).SelectedPaneSet .SelectedPaneArrayWithChildren;
        }

        /// <summary>
        /// ペインの名前が変わった際に TexMap のキャプチャテクスチャ参照を更新します。キャプチャしているペインとパーツペインの名前が変わった際に呼び出す必要があります。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="renamedPane"></param>
        /// <param name="oldName"></param>
        /// <param name="newName"></param>
        public static void UpdateTexMapReferencedCaptureTexture(ISubScene subScene, IPane renamedPane, string oldName, string newName)
        {
            foreach (IPane pane in subScene.IPaneArray)
            {
                // パーツの場合は再帰処理
                if (pane.PaneKind == PaneKind.Parts)
                {
                    UpdateTexMapReferencedCaptureTexture(pane.IPartsLayout.PartsSubScene, renamedPane, oldName, newName);
                }
                else
                {
                    IRevHWMaterial[] revMats = PaneHelper.GetRevHWMatFromPane(pane);

                    foreach (var mat in revMats)
                    {
                        int texMapIndex = 0;
                        foreach (var itexMap in mat.IMaterialTexMapSet)
                        {
                            MaterialTexMap texMap = itexMap as MaterialTexMap;

                            // キャプチャテクスチャを参照しているものは名前を新しいものに置き換える

                            if (texMap.ResourceType != Nsrif.Attributes.AttrTextureResourceType.LocalFile)
                            {
                                // 子パーツ内のペインはリネームできないため、ルートレベルのレイアウトに含まれているペインの名前が変わったことを想定
                                if (renamedPane.PaneKind == PaneKind.Parts)
                                {
                                    // パーツのリネーム
                                    // パーツ中に含まれるキャプチャテクスチャの絶対パスが変わるためマテリアルの参照名を変更する
                                    string[] separatedPath = texMap.TexImgName.Split(PaneHelper.CaptureTextureNameSeparator.ToCharArray());
                                    if (separatedPath.Any((name) => name == oldName))
                                    {
                                        var mnp = new RevMaterialManipulator();
                                        mnp.BindTarget(mat);

                                        string newPath = "";
                                        foreach (var name in separatedPath)
                                        {
                                            if (newPath.Length != 0)
                                            {
                                                newPath += PaneHelper.CaptureTextureNameSeparator;
                                                newPath += name;
                                            }
                                            else
                                            {
                                                // ルートレベルのパーツペインの名前が変わっているはずなので先頭に新しい名前を設定する
                                                newPath += newName;
                                            }
                                        }

                                        mnp.RenameMatReferencedTexture(newPath, texMapIndex, texMap.ResourceType);
                                    }
                                }
                                else
                                {
                                    // キャプチャテクスチャを作成しているペインのリネーム
                                    // キャプチャテクスチャを参照しているマテリアルの参照名を変更する
                                    if (texMap.TexImgName == oldName)
                                    {
                                        var mnp = new RevMaterialManipulator();
                                        mnp.BindTarget(mat);

                                        mnp.RenameMatReferencedTexture(newName, texMapIndex, texMap.ResourceType);
                                    }
                                }
                            }
                            ++texMapIndex;
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 上位階層のキャプチャテクスチャで上書きされたパーツ内のマテリアルで参照しているテクスチャ名を、自身に配置された構造に応じて名前を適切に設定します。
        /// 上書きで設定されたキャプチャテクスチャは、設定された時点のボディのレイアウトをルートとした名前で保存されているため、実際に配置されたパーツ階層に応じて名前を修飾します。
        /// </summary>
        /// <param name="pane">名前解決を行うペイン</param>
        /// <param name="overrideData">上書き状態を調べるための上書きデータ</param>
        /// <param name="namePrefix">修飾する際に設定されるプリフィックス文字列</param>
        private static void ResolveOverrideCaptureTextureName(IPane pane, IPane overrideData, string namePrefix)
        {
            // Prefix が空の場合は修飾されていないので何もしない。
            if (namePrefix.Length == 0)
            {
                return;
            }

            int matIndex = 0;
            foreach (var mat in overrideData.IMaterial)
            {
                int texMapIndex = 0;
                foreach (MaterialTexMap texMap in mat.IMaterialTexMapSet)
                {
                    if (matIndex < pane.IMaterial.Length)
                    {
                        IMaterial targetMat = pane.IMaterial[matIndex];
                        if (texMapIndex < targetMat.IMaterialTexMapSet.Length)
                        {
                            MaterialTexMap writeTexMap = targetMat.IMaterialTexMapSet[texMapIndex] as MaterialTexMap;

                            writeTexMap.ResolveOverrideCaptureTextureName(namePrefix);
                        }
                    }
                    ++texMapIndex;
                }
                ++matIndex;
            }
        }

        /// <summary>
        /// 通常のキャプチャテクスチャの参照解決処理です。
        /// 同一レイアウト内での解決のため、自身の所属するレイアウトのパーツ階層を Prefix として使用します。
        /// </summary>
        /// <param name="pane">名前解決を行うペイン</param>
        private static void ResolveLocalCaptureTexturename(IPane pane)
        {
            foreach (var mat in pane.IMaterial)
            {
                foreach (var itexMap in mat.IMaterialTexMapSet)
                {
                    MaterialTexMap texMap = itexMap as MaterialTexMap;
                    texMap.ResolveTexturename();
                }
            }
        }

        /// <summary>
        /// パーツペイン内で使用されているキャプチャテクスチャの動的な参照情報を現在の階層構造を考慮して更新します。
        /// </summary>
        /// <param name="partsPane"></param>
        public static void ResolvePartsLayoutCaptureTextureReference(IPane partsPane)
        {
            if (partsPane != null &&
                partsPane.PaneKind != PaneKind.Parts ||
                partsPane.IPartsLayout == null)
            {
                return;
            }

            // 最上位のパーツペインから名前の解決処理を行う。
            IPane rootPartsPane = partsPane;

            while (rootPartsPane.OwnerSubScene != null &&
                   rootPartsPane.OwnerSubScene.IPartsLayout != null)
            {
                rootPartsPane = rootPartsPane.OwnerSubScene.IPartsLayout.OwnerPane;
            }

            ResolvePartsLayoutCaptureTextureReference_(rootPartsPane, "");

            // パーツをレイアウト内へ配置されたタイミングで一度だけ実行される処理のため、合わせて指定されたパーツペイン以下の SubScene のキャプチャテクスチャを更新フラグを立てておく。
            rootPartsPane.OwnerSubScene.RequestRenderCaptureTexture(CaptureTextureUpdateMode.All);
        }

        /// <summary>
        /// ResolvePartsLayoutCaptureTextureReference の内部実装です。
        /// </summary>
        /// <param name="partsPane"></param>
        /// <param name="newParentName"></param>
        static void ResolvePartsLayoutCaptureTextureReference_(IPane partsPane, string newParentName)
        {
            // パーツレイアウトが設定されることによって親レイアウトと接続して階層構造が決定する。
            // キャプチャテクスチャはパーツの階層構造を考慮した名前で指定する必要があるため、確定した階層構造で参照するテクスチャイメージを更新する。
            string modifiedParentName = newParentName;
            bool validOverrideProperty = false;

            // 上書きキャプチャテクスチャの参照解決処理
            foreach (var property in partsPane.IPartsLayout.PartsPropaerties)
            {
                if (property.Option != PartsPropertyUsageOptions.None)
                {
                    IPane targetPane = FindPaneByNameRecursively(partsPane.IPartsLayout.PartsSubScene, property.TargetName);

                    // 上書き設定されているキャプチャテクスチャ参照の名前を解決します。
                    ResolveOverrideCaptureTextureName(targetPane, property.Paramater.OwnerPane, newParentName);

                    validOverrideProperty = true;
                }
            }

            // 一番末尾で有効な上書きプロパティが設定されているところがパーツ内キャプチャテクスチャの上書きの基点となる。
            // 有効な上書きプロパティが見つかるまで prefix へ経由したパーツ名を連結していく。
            if (!validOverrideProperty)
            {
                modifiedParentName = PaneHelper.MakeCaptureTextureName(newParentName, partsPane.PaneName);
            }

            foreach (var partsChildPane in partsPane.IPartsLayout.PartsSubScene.IPaneArray)
            {
                if (partsChildPane.PaneKind == PaneKind.Parts)
                {
                    ResolvePartsLayoutCaptureTextureReference_(partsChildPane, modifiedParentName);
                }
                else
                {
                    // パーツ以外の場合は同一レイアウト内の上書きキャプチャテクスチャの名前解決処理を行う。
                    // シーンの階層関係が確定した状態でターゲットのペインから親をたどって名前を自動的に決定する。
                    ResolveLocalCaptureTexturename(partsChildPane);
                }
            }
        }

        /// <summary>
        /// 指定されたサブシーンから上位方向へパーツをたどって指定された名前のキャプチャテクスチャを取得します。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="texImgName"></param>
        /// <returns></returns>
        public static ITextureImage FindCaptureTextureRecursively(ISubScene subScene, string texImgName)
        {
            ISubScene currentSubScene = subScene;

            while (currentSubScene != null)
            {
                ITextureImage texImg = TextureMgrHelper.FindCaptureTextureByName(currentSubScene.ITextureMgr, texImgName);
                if (texImg != null)
                {
                    return texImg;
                }

                if (currentSubScene.IPartsLayout != null)
                {
                    currentSubScene = currentSubScene.IPartsLayout.GetOwnerSubScene();
                }
                else
                {
                    currentSubScene = null;
                }
            }

            return null;
        }

        /// <summary>
        /// 指定された名前のキャプチャテクスチャを管理しているサブシーンを取得します。
        /// 渡されたサブシーンをルートとして、サブシーンが持っているパーツから再帰的に検索します。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="texImgName"></param>
        /// <returns></returns>
        public static ISubScene FindCaptureTextureManagedSubScene(ISubScene subScene, string texImgName)
        {
            ITextureImage tex = TextureMgrHelper.FindCaptureTextureByName(subScene.ITextureMgr, texImgName);

            // 渡されたサブシーン内で見つかればそのまま返す
            if (tex != null)
            {
                return subScene;
            }

            // 子ペインのパーツをたどれないかチェックする
            foreach (var pane in subScene.IPaneArray)
            {
                if (pane.IPartsLayout != null)
                {
                    ISubScene result = FindCaptureTextureManagedSubScene(pane.IPartsLayout.PartsSubScene, texImgName);
                    if (result != null)
                    {
                        return result;
                    }
                }
            }

            return null;
        }


        /// <summary>
        /// 指定されたペインセットをサブシーンから削除します。
        /// </summary>
        public static void DeletePaneSetFromSubScene(ISubScene subScene, IPane[] paneSet)
        {
            // ペインを消去したのち、
            {
                SubSceneManipulator subSceneMnp = new SubSceneManipulator();
                subSceneMnp.BindTarget(subScene);
                subSceneMnp.ISubScene.BeginMassiveModify();

                subSceneMnp.BeginSelectSetChange();
                subSceneMnp.ResetSelectedSet();

                foreach (IPane pane in paneSet)
                {
                    subSceneMnp.SelectPanesByPaneRef(pane);
                }
                subSceneMnp.EndSelectSetChange();

                subSceneMnp.DeleteSelectedPanes();
                subSceneMnp.ISubScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// キャプチャ元ペイン削除時のキャプチャテクスチャ側の対応処理です。
        /// </summary>
        /// <param name="subScene">削除処理を行うサブシーンです</param>
        /// <param name="selectedPanes">削除されるペインの配列です</param>
        public static void TryToDeleteCaptureTexture(ISubScene subScene, IPane[] selectedPanes)
        {
            // 最後に残ったリンク切れのキャプチャテクスチャの削除処理
            DeleteLastInvalidCaptureTexture_(subScene, selectedPanes);

            // キャプチャ関連物の列挙
            Hashtable texToPanesTable = new Hashtable();
            foreach (IPane selectedPane in selectedPanes)
            {
                // キャプチャしているペインのチェック
                ITextureImage generatedTexture = PaneHelper.GetCapturedTexture(selectedPane);

                if (generatedTexture != null &&
                    !texToPanesTable.ContainsKey(generatedTexture))
                {
                    // キャプチャテクスチャを使用しているペインを取得します。
                    texToPanesTable.Add(generatedTexture, SubSceneHelper.GetUseTexturePaneSet(subScene, generatedTexture.Name));
                }
            }

            // 関連テクスチャが存在しないためなにも行わない
            if (texToPanesTable.Keys.Count <= 0)
            {
                return;
            }

            // キャプチャ元ペイン削除に伴うキャプチャテクスチャの削除処理
            DeleteCaptureTextureRelatedTextures_(subScene, selectedPanes, texToPanesTable);
        }

        /// <summary>
        /// 選択ペイン内に最後に残ったリンク切れのキャプチャテクスチャを参照しているものがあった場合、そのキャプチャテクスチャを削除します。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="selectedPanes"></param>
        static void DeleteLastInvalidCaptureTexture_(ISubScene subScene, IPane[] selectedPanes)
        {
            // キャプチャ関連物の列挙
            Hashtable texToPanesTable = new Hashtable();
            foreach (IPane selectedPane in selectedPanes)
            {
                var textures = PaneHelper.GetUsedCaptureTexturesWithPartsTree(selectedPane);

                foreach (var texture in textures)
                {
                    if (texture.ICaptureTexture.OwnerPane == null)
                    {
                        List<IPane> usedPanes = new List<IPane>();
                        CollectCaptureTextureUsedPanes(usedPanes, subScene, texture);

                        if (usedPanes.Count() == 1)
                        {
                            TextureMgrManipulator textureMgrMnp = new TextureMgrManipulator();
                            textureMgrMnp.BindTarget(subScene.ITextureMgr);

                            textureMgrMnp.RemoveTexture(texture as ITextureImage);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 指定されたキャプチャテクスチャを使用しているペインをパーツ内も含めて再帰的に収集します。
        /// </summary>
        /// <param name="panes"></param>
        /// <param name="subScene"></param>
        /// <param name="captureTexture"></param>
        public static void CollectCaptureTextureUsedPanes(List<IPane> panes, ISubScene subScene, ITextureImage captureTexture)
        {
            foreach (IPane pane in subScene.IPaneArray)
            {
                if (pane.PaneKind == PaneKind.Parts &&
                    pane.IPartsLayout != null)
                {
                    CollectCaptureTextureUsedPanes(panes, pane.IPartsLayout.PartsSubScene, captureTexture);
                }
                else
                {
                    ITextureImage[] usedCaptureTextures = PaneHelper.GetUsedCaptureTextures(pane);
                    foreach (ITextureImage usedTexture in usedCaptureTextures)
                    {
                        if (usedTexture.Name == captureTexture.Name)
                        {
                            panes.Add(pane);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// キャプチャ元ペイン削除時のキャプチャテクスチャ側の対応処理です。
        /// マテリアルから参照が残っているキャプチャテクスチャはオーナーペイン不在の不正な状態に設定し、どこからも参照されていないキャプチャテクスチャは削除します。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="selectedPanes"></param>
        /// <param name="relatedTextureSet"></param>
        /// <param name="relatedPaneSet"></param>
        private static void DeleteCaptureTextureRelatedTextures_(ISubScene subScene, IPane[] selectedPanes, Hashtable texToPanesTable)
        {
            subScene.BeginMassiveModify();

            List<ITextureImage> deleteTextureList = new List<ITextureImage>();

            foreach (ITextureImage tex in texToPanesTable.Keys)
            {
                IPane[] panes = texToPanesTable[tex] as IPane[];
                bool unlinkPane = false;

                if (panes != null)
                {
                    // 選択ペインと使用ペインの積が使用ペインの数と同じならば
                    // キャプチャテクスチャを使用するペインがすべて削除されるためテクスチャ自体を削除する。
                    // 逆の場合はまだ使用されるため削除せずにペインのリンクを外すだけにする。
                    var intersect = panes.Intersect(selectedPanes);
                    if (intersect.Count() != panes.Length)
                    {
                        unlinkPane = true;
                    }
                }

                if (unlinkPane)
                {
                    // 参照しているペインが残っているためキャプチャテクスチャは削除せず、ペイントのリンクを切るだけにする。
                    CaptureTexture captureTexture = TextureMgrHelper.FindCaptureTextureByName(subScene.ITextureMgr, tex.Name).ICaptureTexture as CaptureTexture;

                    CaptureTextureManipulator captureTextureMnp = new CaptureTextureManipulator();
                    captureTextureMnp.BindTarget(captureTexture);

                    captureTextureMnp.OwnerPane = null;
                }
                else
                {
                    // 完全に使用されなくなるため削除する
                    deleteTextureList.Add(tex);
                }
            }

            // 完全に不要になったテクスチャを削除します。
            if (deleteTextureList.Count > 0)
            {
                TextureMgrManipulator textureMgrMnp = new TextureMgrManipulator();
                textureMgrMnp.BindTarget(subScene.ITextureMgr);

                textureMgrMnp.RemoveTextureSet(deleteTextureList.ToArray());
            }

            subScene.EndMassiveModify();
        }

        /// <summary>
        /// 渡されたサブシーンで選択されているペインを子供としてキャプチャペインを作成します。
        /// </summary>
        /// <param name="subScene"></param>
        /// <param name="newName"></param>
        /// <param name="errorDialog"></param>
        public static void AddCapturePaneWithSelectedPanesAsChild(ISubScene subScene, string newName, Action<List<IPane>> errorDialog)
        {
            // 選択された子供にキャプチャ関連のペインが存在するとキャプチャテクスチャ間に依存関係が発生して
            // 再現できないデータとなるため作成前にチェックしてはじく。
            var errorPaneList = new List<IPane>();
            foreach (var selected in subScene.ISelectedSet.IPaneArray)
            {
                IPane result = PaneHelper.CheckHaveCaptureRelativePaneInSubTree(selected);
                if (result != null)
                {
                    errorPaneList.Add(result);
                }
            }

            if (errorPaneList.Count > 0)
            {
                errorDialog.Invoke(errorPaneList);

                return;
            }

            // 変更開始
            subScene.BeginMassiveModify();

            IPane[] selectedPanes = subScene.ISelectedSet.IPaneArray;

            // キャプチャペインを作成したときに選択状態が変わるため、作成直前の状態でサイズを計算します。
            RectangleF globalRect = subScene.ISelectedSet.BoundingVolumeInWorld;

            var subSceneMnpl = new SubSceneManipulator();
            subSceneMnpl.BindTarget(subScene);
            IPane pane = subSceneMnpl.AddCapturePane(newName);

            // 選択されたペインを新たに作成したキャプチャペインの子供としてぶら下げる。
            // 選択されているキャプチャペイン全体を覆うサイズになるように位置と大きさを調整する。
            var mnpl = new PaneManipulator();

            mnpl.BindTarget(pane);
            mnpl.Size = new FVec3(globalRect.Width, globalRect.Height, 0);
            mnpl.Trans = new FVec3(globalRect.X + globalRect.Width / 2, globalRect.Y + globalRect.Height / 2, 0);

            foreach (var selected in selectedPanes)
            {
                if (!subScene.ISelectedSet.IPaneArray.Any((pane_) => selected.Parent == pane_))
                {
                    RectangleF selectedGlobalRect = selected.BoundingVolumeInWorld;
                    HierarchyManipulator.SetHierarchyToPanePair(pane, selected);
                    mnpl.BindTarget(selected);
                    mnpl.Trans = new FVec3(
                        selectedGlobalRect.X + selectedGlobalRect.Width / 2 - pane.Trans.X,
                        selectedGlobalRect.Y + selectedGlobalRect.Height / 2 - pane.Trans.Y,
                        0);
                }
            }

            subScene.EndMassiveModify();
        }

        /// <summary>
        /// クリップボードにコピーされたペインのデータの中にテクスチャをキャプチャするペインが含まれているかどうかをチェックします。
        /// </summary>
        /// <returns>クリップボードにコピーされたデータの中にテクスチャをキャプチャしているペインが含まれていれば true、それ以外は false を返します。</returns>
        public static bool DoesClipboardHaveCaptureTextureRelativePane()
        {
            // クリップボードの中身をいったんペーストして内容をチェックする。
            object pasteResult = LEClipboard.Instance.Paste(typeof(ClipBoardPane));
            if (pasteResult != null)
            {
                ClipBoardPane clipBoardPane = pasteResult as ClipBoardPane;

                foreach (IPane pane in clipBoardPane.PaneSet)
                {
                    if (PaneHelper.IsTextureCapturedPane(pane) ||
                        PaneHelper.IsCaptureTextureUsed(pane))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

#endregion //キャプチャペイン
    }
}
