﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;

namespace LECore.Structures
{
    /// <summary>
    /// ヘルパ
    /// </summary>
    public static class PartsLayoutHelper
    {
        /// <summary>
        /// 部品ペインサイズを計算します。
        /// </summary>
        static public RectangleF CalcPartsPaneRectanbleFromPanes(IEnumerable<IPane> panes)
        {
            List<IPane> childPanes = new List<IPane>(panes);

            // 境界ペインは部品サイズに含めません。
            childPanes.RemoveAll((pane) => pane.PaneKind == PaneKind.Bounding);

            RectangleF totalRect = RectangleF.Empty;
            foreach (IPane pane in childPanes)
            {
                totalRect = RectangleF.Union(totalRect, PaneHelper.GetBoundingVolumeInWorld(pane, true));
            }

            return totalRect;
        }

        /// <summary>
        /// 部品ペインサイズを計算します。
        /// </summary>
        static public FVec2 CalcPartsPaneSizeFromRectangle(RectangleF rect)
        {
            float sW = Math.Max(Math.Abs(rect.Left), Math.Abs(rect.Right)) * 2.0f;
            float sH = Math.Max(Math.Abs(rect.Top), Math.Abs(rect.Bottom)) * 2.0f;

            return new FVec2(sW, sH);
        }

        /// <summary>
        ///
        /// </summary>
        static public IPartsPropaerty FindPartsPropertyByName(this IPartsLayout parts, string name)
        {
            return parts.PartsPropaerties.FirstOrDefault((prop) => prop.TargetName == name);
        }

        /// <summary>
        ///
        /// </summary>
        static public bool IsEnabled(this IPartsPropaerty prop)
        {
            return prop != null && prop.Option != PartsPropertyUsageOptions.None;
        }

        /// <summary>
        /// 指定したレイアウトを参照しているか？
        /// </summary>
        public static bool IsReferencingLayout(this IPartsLayout parts, string layoutName)
        {
            return EnumlateReferencingLayoutFiles(parts).Any(
                (referencingFileName) => referencingFileName == layoutName);
        }

        /// <summary>
        /// 参照しているレイアウトファイルを列挙します。
        /// </summary>
        public static IEnumerable<string> EnumlateReferencingLayoutFiles(this IPartsLayout parts)
        {
            // 注意：リロードする際に都合がよいように、子供が参照しているものから返す。

            // シーン内の孫部品を再帰的に列挙
            foreach (IPane partsPane in parts.PartsSubScene.FindPanesByKind(PaneKind.Parts))
            {
                foreach (var fileName in EnumlateReferencingLayoutFiles(partsPane.IPartsLayout))
                {
                    yield return fileName;
                }
            }

            // 親ファイルを処理するには、子供ファイルが処理されている必要があるため。
            foreach (var prop in parts.PartsPropaerties)
            {
                if (prop.PaneKind == PaneKind.Parts && prop.Paramater is IPartsLayout)
                {
                    foreach (var fileName in EnumlateReferencingLayoutFiles(prop.Paramater as IPartsLayout))
                    {
                        yield return fileName;
                    }
                }
            }

            yield return parts.PartsLayoutName;
        }

        /// <summary>
        /// 変更前ペインを取得します。
        /// </summary>
        public static IPane GetOriginalPane(this IPartsLayout parts, string paneName)
        {
            var originalSubScene =
                LayoutEditorCore.Scene.FindPartsSubSceneByFileName(parts.PartsLayoutName);
            if (originalSubScene == null || !originalSubScene.IsLoaded)
            {
                return null;
            }

            return originalSubScene.SubScene.FindPaneByName(paneName);
        }

        /// <summary>
        /// 上書きペインの設定情報を取得します。
        /// </summary>
        public static IPartsPropaertySettings GetPartsPropaertySettings(this IPartsLayout parts, string targetName)
        {
            var originalSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(parts.PartsLayoutName);
            if (originalSubScene == null || !originalSubScene.IsLoaded)
            {
                return null;
            }

            return originalSubScene.SubScene.IPartsSettings.OverridePorperties.FirstOrDefault((prop) => prop.TargetName == targetName);
        }
    }
}
