﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing.Drawing2D;
using LECore.Util;
using LECore.Manipulator;
using System.Diagnostics;
using LECore.Structures.Core;

namespace LECore.Structures
{
    /// <summary>
    /// ペインツリー構造を圧縮するクラス
    /// </summary>
    public class PaneTreeCompressionHelper
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PaneTreeCompressionHelper()
        {
        }

        //----------------------------------------------------------

        /// <summary>
        /// 圧縮します。
        /// iSubScene は 一つのレイアウトに対応する概念です。
        /// </summary>
        public List<string> Compress(ISubScene iSubScene)
        {
            // まず操作対処がどれだけ存在するか把握して記録しておく。

            List<IPane> compressedPanes = new List<IPane>();
            var subScene = iSubScene as SubScene;
            try
            {
                // 大規模な編集開始
                subScene.BeginMassiveModify();

                // 深さ優先でツリーを走査し...
                EnumPaneByTreeOrder_(subScene.RootIPane).ToList().ForEach((pane) =>
                {
                    // 圧縮条件を満たすペインがあれば...
                    if (IsPaneTreeCompressionTarget_(pane, iSubScene))
                    {
                        // 圧縮する
                        CompressPane_(pane);

                        // 圧縮したペインは後始末があるので記録しておく。
                        compressedPanes.Add(pane);
                    }
                });

                // 圧縮処理の後始末を行います。
                foreach (Pane removedPane in compressedPanes)
                {
                    // 以前に自分があった位置に、子供をすべて移植する
                    Pane newParentPane = removedPane.Parent as Pane;
                    int origIndex = Array.IndexOf(newParentPane.Children, removedPane);
                    foreach (Pane childPane in removedPane.Children)
                    {
                        newParentPane.AddChildPane(childPane);
                        newParentPane.ChangeChildOrder(childPane, origIndex);
                        origIndex++;
                    }

                    // いろいろリセットしてから消す(グループメンバーから消すなど...)
                    (subScene.RootIPane as Pane).AddChildPane(removedPane);
                    (subScene.ILEGroupMgr as LEGroupMgr).RemoveMemberFromAll(removedPane);
                    subScene.RemovePain(removedPane);
                }

                // シーンを更新状態にするため、ダミーのコマンドを積む。
                Debug.Assert(subScene.RootIPane.Children.Length > 0);
                subScene.CommandFactory.MakeEditHierarchyOrderCmd(
                    subScene,
                    subScene.RootIPane as Pane,
                    subScene.RootIPane.Children[0] as Pane, 0);
            }
            finally
            {
                // 大規模な編集完了
                subScene.EndMassiveModify();
            }

            // 処理ペイン名のリストを返す。
            return compressedPanes.ConvertAll<string>((pane) => pane.PaneName);
        }

        //----------------------------------------------------------

        static public bool CheckPaneTreeCompressionTarget(IPane pane, bool isPaneHasAnyAnimation)
        {
            return IsPaneTreeCompressionTarget_(pane, isPaneHasAnyAnimation);
        }

        private bool IsPaneTreeCompressionTarget_(IPane pane, ISubScene subScene)
        {
            return IsPaneTreeCompressionTarget_(pane, pane.HasAnyAnimation());
        }

        /// <summary>
        /// 圧縮対象か判定します。
        /// TODO :自分が 非正規スケールを持っている場合は圧縮を
        /// 禁止しないと表示結果が等しくならないので自動処理を行う場合は圧縮を禁止したほうが良いです。
        /// </summary>
        private static bool IsPaneTreeCompressionTarget_(IPane pane, bool isPaneHasAnyAnimation)
        {
            return
                !pane.IsRootPane() &&
                pane.PaneKind == PaneKind.Null &&
                !pane.AvoidPaneTreeCompression &&
                !isPaneHasAnyAnimation &&
                CheckNoChildChangesParentOrigin_(pane);
        }

        /// <summary>
        /// 子供ペインが親の中心を変更していないか？
        /// </summary>
        private static bool CheckNoChildChangesParentOrigin_(IPane pane)
        {
            return pane.Children.All(
                (child) =>
                    (child as IPane).ParentBasePosTypeH == HorizontalLocation.Center &&
                    (child as IPane).ParentBasePosTypeV == VerticalLocation.Center);
        }

        /// <summary>
        /// Matrix34 に変換
        /// </summary>
        Matrix34 Create34Mtx_(Matrix mtx)
        {
            Matrix34 mtx34 = new Matrix34();
            mtx34[0, 0] = mtx.Elements[0];
            mtx34[0, 1] = mtx.Elements[2];
            mtx34[0, 3] = mtx.Elements[4];

            mtx34[1, 0] = mtx.Elements[1];
            mtx34[1, 1] = mtx.Elements[3];
            mtx34[1, 3] = mtx.Elements[5];

            mtx34[2, 0] = 0;
            mtx34[2, 1] = 0;
            mtx34[2, 2] = 1;
            mtx34[2, 3] = 0;

            return mtx34;
        }

        /// <summary>
        /// ペインの圧縮
        /// </summary>
        private void CompressPane_(IPane pane)
        {
            // SRT の合成
            Matrix34 parentMtx = Create34Mtx_(pane.LocalMtx);
            foreach (IPane ichildPane in pane.Children)
            {
                var childPane = ichildPane as Pane;

                // SRT
                {
                    Matrix34 childMtx = Create34Mtx_(childPane.LocalMtx);
                    Matrix34 mergedMtx = parentMtx * childMtx;

                    FVec3 s;
                    FVec3 r;
                    FVec3 t;
                    GetTransform_(mergedMtx, out s, out r, out t);
                    r = MathUtil.RadToDeg(r);

                    childPane.Scale = s;
                    childPane.RotAng = r;
                    childPane.Trans = t;
                }

                // SRTアニメーション
                {
                    // 特定アトリビュート以下を評価して最新に
                    CompressVec33AttrKeies_(parentMtx, childPane, (s, r, t) => s, childPane.PaneAttrRef.ScaleAttr);
                    CompressVec33AttrKeies_(parentMtx, childPane, (s, r, t) => MathUtil.RadToDeg(r), childPane.PaneAttrRef.RotAngAttr);
                    CompressVec33AttrKeies_(parentMtx, childPane, (s, r, t) => t, childPane.PaneAttrRef.TransAttr);
                }
            }

            // 透明度
            if (pane.InfluenceChildrenTransparency)
            {
                float alphaScale = pane.Transparency / 255.0f;
                if (alphaScale != 1.0f)
                {
                    foreach (Pane childPane in pane.Children)
                    {
                        ScaleTransparencyAll_(childPane, alphaScale);
                    }
                }
            }

            // 圧縮が終わったらリセット
            (pane as Pane).Scale = FVec3.One;
            (pane as Pane).RotAng = FVec3.Empty;
            (pane as Pane).Trans = FVec3.Empty;
            (pane as Pane).Transparency = 255;
        }

        /// <summary>
        /// SRTを指定時間の値に更新する。
        /// </summary>
        private void EvaluateCurrentSRT_(Pane pane, float time)
        {
            pane.PaneAttrRef.ScaleAttr.EvaluateAnimAll(time);
            pane.PaneAttrRef.RotAngAttr.EvaluateAnimAll(time);
            pane.PaneAttrRef.TransAttr.EvaluateAnimAll(time);
        }

        /// <summary>
        /// Vec3アニメーションを圧縮します。
        ///
        /// 以下では、キーのある位置だけで行列の再評価を行っています。
        /// より正確な結果を得る場合は、フレーム間隔 0.1 ~ 1.0 位で値をすべて計算（ベイク処理）、その後
        /// 特定の許容誤差を満たしている場合キーを間引く処理をしてキーを最小限に減らす処理を行います。
        /// </summary>
        private void CompressVec33AttrKeies_(Matrix34 parentMtx, Pane pane, Func<FVec3, FVec3, FVec3, FVec3> selectFunc, AnmAttribute vec3Attr)
        {
            // すべての~キーについて
            foreach (AnmKeyFrame key in vec3Attr.FindSubAttributeByIdx(0).ICurrentAnimationCurve.IKeyFrameSet)
            {
                // キーのある時間でSRTを更新。親との合成行列を得る。
                EvaluateCurrentSRT_(pane, key.Time);
                Matrix34 currentMtx = parentMtx * Create34Mtx_(pane.LocalMtx);

                // SRT 表現に変換。
                FVec3 s, r, t;
                GetTransform_(currentMtx, out s, out r, out t);

                // SRT から適切な値を選んでキーのX値とする。
                key.Value = selectFunc(s,r,t).X;
            }

            foreach (AnmKeyFrame key in vec3Attr.FindSubAttributeByIdx(1).ICurrentAnimationCurve.IKeyFrameSet)
            {
                EvaluateCurrentSRT_(pane, key.Time);
                Matrix34 currentMtx = parentMtx * Create34Mtx_(pane.LocalMtx);

                FVec3 s, r, t;
                GetTransform_(currentMtx, out s, out r, out t);

                key.Value = selectFunc(s, r, t).Y;
            }

            if (vec3Attr.FindSubAttributeByIdx(2) != null)
            {
                foreach (AnmKeyFrame key in vec3Attr.FindSubAttributeByIdx(2).ICurrentAnimationCurve.IKeyFrameSet)
                {
                    EvaluateCurrentSRT_(pane, key.Time);
                    Matrix34 currentMtx = parentMtx * Create34Mtx_(pane.LocalMtx);

                    FVec3 s, r, t;
                    GetTransform_(currentMtx, out s, out r, out t);

                    key.Value = selectFunc(s, r, t).Z;
                }
            }
        }

        /// <summary>
        /// 透明度をスケールします。
        /// </summary>
        private void ScaleTransparencyAll_(Pane pane, float alphaScale)
        {
            byte newTransparency = (byte)((float)pane.Transparency * alphaScale);
            pane.Transparency = newTransparency;

            // 透明度アニメーション
            foreach (AnmKeyFrame key in pane.PaneAttrRef.TransparencyAttr.ICurrentAnimationCurve.IKeyFrameSet)
            {
                float newValue = AnmAttribute.GetValueAsFloat(AttributeType.Byte, key.Value) * alphaScale;
                object temp = null;
                AnmAttribute.SetValueAsFloat(AttributeType.Byte, ref temp, newValue);
                key.Value = temp;
            }

            foreach (Pane child in pane.Children)
            {
                ScaleTransparencyAll_(child, alphaScale);
            }
        }

        /// <summary>
        /// 深さ優先でペインツリーを列挙します。
        /// </summary>
        private IEnumerable<IPane> EnumPaneByTreeOrder_(IPane pane)
        {
            foreach (IPane child in pane.Children)
            {
                yield return child;
                foreach (IPane childChilde in EnumPaneByTreeOrder_(child))
                {
                    yield return childChilde;
                }
            }
        }

        /// <summary>
        /// 行列からSRTを取得します。
        /// </summary>
        /// <param name="source">行列です。</param>
        /// <param name="trfm">設定するSRTです。</param>
        private static void GetTransform_(Matrix34 source, out FVec3 s, out FVec3 r, out FVec3 t)
        {
            Ensure.Argument.NotNull(source);

            s = new FVec3();
            r = new FVec3();
            t = new FVec3();

            Matrix34 mtx = new Matrix34();
            mtx.Set(source);
            for (int iaxis = 0; iaxis < 3; ++iaxis)
            {
                FVec3 f = new FVec3(mtx[0, iaxis], mtx[1, iaxis], mtx[2, iaxis]);
                s[iaxis] = MathUtil.GetVecLength(f);
                if (s[iaxis] != 0)
                {
                    if (iaxis == 2)
                    {
                        FVec3 ax = new FVec3(mtx[0, 0], mtx[1, 0], mtx[2, 0]);
                        FVec3 ay = new FVec3(mtx[0, 1], mtx[1, 1], mtx[2, 1]);
                        FVec3 cz = MathUtil.CrossProduct(ax, ay);
                        if (MathUtil.DotProduct(cz, f) < 0.0F)
                        {
                            s[iaxis] = -s[iaxis];
                        }
                    }

                    float d = 1.0F / s[iaxis];
                    mtx[0, iaxis] *= d;
                    mtx[1, iaxis] *= d;
                    mtx[2, iaxis] *= d;
                }
            }

            GetRotateEuler_(mtx, ref r);

            t[0] = mtx[0, 3];
            t[1] = mtx[1, 3];
            t[2] = mtx[2, 3];
        }

        /// <summary>
        /// 回転行列のオイラー角を取得します。
        /// オイラー角の表現法はxyzの順です。
        /// </summary>
        /// <param name="mtx">回転行列です。</param>
        /// <param name="rotate">設定するオイラー角（ラジアン単位)のベクトルです。</param>
        private static void GetRotateEuler_(Matrix34 mtx, ref FVec3 rotate)
        {
            Ensure.Argument.NotNull(mtx);

            double cosEpsilon = 0.00001;
            double syr = -mtx[2, 0];
            double cyr2 = 1.0 - (syr * syr);
            if (cyr2 < 0.0)
            {
                cyr2 = 0.0;
            }

            double cyr = Math.Sqrt(cyr2);
            if (cyr < cosEpsilon)
            {
                rotate.X = (float)Math.Atan2(-mtx[1, 2], mtx[1, 1]);
                rotate.Y = (syr > 0.0) ? (float)Math.PI / 2 : (float)-Math.PI / 2;
                rotate.Z = 0.0F;
            }
            else
            {
                rotate.X = (float)Math.Atan2(mtx[2, 1], mtx[2, 2]);
                rotate.Y = (float)Math.Atan2(syr, cyr);
                rotate.Z = (float)Math.Atan2(mtx[1, 0], mtx[0, 0]);
            }
        }
    }
}
