﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.Linq;
using System.Drawing.Drawing2D;
using System.Text.RegularExpressions;
using System.Numerics;

namespace LECore.Structures
{
    using Core;
    using LECore.Util;
    using LECore.Manipulator;

    /// <summary>
    /// PaneHelper
    /// </summary>
    public static class PaneHelper
    {
        //---------------------------------------------------------------------
        #region 定数
        public const string PaneNodeName = "";
        public const string RootPaneName = "RootPane";
        public const int MaxTransparency = 255;
        #endregion 定数

        //---------------------------------------------------------------------
        #region 詳細マテリアルの取得
        static public IRevHWMaterial[] GetRevHWMatFromPane( IPane pane )
        {
            switch( pane.PaneKind )
            {
                case PaneKind.Picture:
                return ( pane.IPicture as IRevHWMaterialHolder ).IRevHWMaterial;
                case PaneKind.Textbox:
                return ( pane.ITextBox as IRevHWMaterialHolder ).IRevHWMaterial;
                case PaneKind.Window:
                return ( pane.ILEWindow as IRevHWMaterialHolder ).IRevHWMaterial;
                case PaneKind.Parts:
                case PaneKind.Null:
                case PaneKind.Bounding:
                case PaneKind.Capture:
                case PaneKind.Alignment:
                case PaneKind.Scissor:
                    return new IRevHWMaterial[0];
                default:
                // 想定しない型
                Debug.Assert( false );
                return new IRevHWMaterial[0];

            }
        }
        #endregion

        //---------------------------------------------------------------------
        #region 有効性のチェック
        /// <summary>
        /// ペイン名として適切か判定します。
        /// </summary>
        static public bool CheckPaneNameValid( string name )
        {
            if( name != PaneHelper.RootPaneName )
            {
                if( name.Length > 0 && name.Length <= LECore.AppConstants.MaxPaneNameLength )
                {
                    Match m = LECore.AppConstants.RegexInvalidCharForObjName.Match( name );
                    if( m.Length == 0 )
                    {
                        return true;
                    }
                }
            }
            return false;
        }

        /// <summary>
        /// ユーザデータ文字列として適切か判定します。
        /// </summary>
        static public bool CheckUserDataValid( string name )
        {
            if( name.Length <= LECore.AppConstants.MaxUserdataStringLength )
            {
                Match m = LECore.AppConstants.RegexInvalidCharForUserDataStr.Match( name );
                if( m.Length == 0 )
                {
                    return true;
                }
            }
            return false;
        }
        #endregion 有効性のチェック

        //---------------------------------------------------------------------
        #region 階層関連
        /// <summary>
        /// ルートペインか判定します。
        /// </summary>
        static public bool IsRootPane( this IPane pane )
        {
            return pane != null && pane.PaneName == RootPaneName;
        }

        /// <summary>
        /// ペイン列をツリー階層順に列挙する。
        /// </summary>
        static public IEnumerable<IPane> EnumlatePaneByTreeOrder(this IPane pane)
        {
            yield return pane;
            foreach (IPane childPane in pane.Children)
            {
                foreach (var resut in EnumlatePaneByTreeOrder(childPane))
                {
                    yield return resut;
                }
            }
        }

        /// <summary>
        /// 一つ前面のペインを取得します。
        /// </summary>
        static public IPane GetFrontPane( IPane pane )
        {
            if( pane == null )
            {
                return null;
            }

            // 子供がいれば先頭の子供を返す
            if( pane.Children.Length > 0 )
            {
                return pane.Children[0] as IPane;
            }
            IHierarchyNode parent = pane.Parent;
            while( parent != null )
            {
                IHierarchyNode[] children = parent.Children;
                int index = Array.IndexOf( children, pane );
                if( index != -1 && index != children.Length - 1 )
                {
                    return children[index + 1] as IPane;
                }
                pane = parent as IPane;
                parent = parent.Parent;
            }

            return null;
        }

        /// <summary>
        /// 一つ背面のペインを取得します。
        /// </summary>
        static public IPane GetBackPane( IPane pane )
        {
            if( pane == null )
            {
                return null;
            }

            if( pane.Parent != null )
            {
                IHierarchyNode[] children = pane.Parent.Children;
                int index = Array.IndexOf( children, pane );
                if( index != -1 && index != 0 )
                {
                    // 一つ前の兄弟
                    return children[index - 1] as IPane;
                }
                else
                {
                    // 自分が子供の先頭なら、親を返す。
                    return pane.Parent as IPane;
                }
            }

            return null;
        }

        /// <summary>
        /// 引数ペインの、階層レベルを取得します。
        /// </summary>
        static public int GetPaneHierarchyLevel( IPane pane )
        {
            int level = 0;
            while( pane.Parent != null )
            {
                level++;
                pane = pane.Parent as IPane;
            }
            return level;
        }

        /// <summary>
        /// 引数ペインの、兄弟ペイン配列での順番を取得します。
        /// </summary>
        static public int GetPaneHierarchyChildIndex( IPane pane )
        {
            int index = -1;
            if( pane.Parent != null )
            {
                index = Array.IndexOf( pane.Parent.Children, pane );
            }

            return index;
        }

        /// <summary>
        /// ペイン配列を描画順にソートする
        /// xが大きければ正の値を、yが大きければ負の値を、xとyが等しければ0を返すようにします。
        /// </summary>
        /// <param name="paneSet"></param>
        static public void SortPaneSetByDrawOrder( IPane[] paneSet )
        {
            Array.Sort(paneSet, (lhs, rhs) => ComparePaneByTreeOrder_(lhs, rhs));
        }

        /// <summary>
        /// 階層順にソートする比較関数
        /// </summary>
        static public int ComparePaneByTreeOrder(IPane lhs, IPane rhs)
        {
            return ComparePaneByTreeOrder_(lhs, rhs);
        }

        /// <summary>
        /// 階層順にソートする比較関数
        /// （軽くはない処理なので頻繁に呼び出す場合は要注意です。）
        /// </summary>
        static private int ComparePaneByTreeOrder_(IPane lhs, IPane rhs)
        {
            if (lhs == rhs)
            {
                return 0;
            }

            if (lhs == null)
            {
                return Int16.MaxValue;
            }
            else if (rhs == null)
            {
                return Int16.MinValue;
            }

            // 階層の深さを比較する
            if (lhs.OwnerSubScene == null || lhs.OwnerSubScene.RootIPane == null) { return Int16.MaxValue; }

            List<IPane> treeOrderedPaneList = EnumlatePaneByTreeOrder(lhs.OwnerSubScene.RootIPane).ToList();
            return treeOrderedPaneList.IndexOf(lhs) - treeOrderedPaneList.IndexOf(rhs);
        }

        /// <summary>
        /// 2つのペインが親子関係にあるか調査します。
        /// </summary>
        static public bool CheckPaneIsParentOf( IPane parentPane, IPane childPane )
        {
            Debug.Assert( parentPane != null );
            Debug.Assert( childPane != null );

            IHierarchyNode pane = childPane.Parent as IPane;　

            while( pane == parentPane )
            {
                // parentPaneを発見できないまま、シーンルートに到達してしまった。
                if( pane == null )
                {
                    return false;
                }

                pane = pane.Parent;
            }

            return true;
        }

        /// <summary>
        /// 回転、スケール中心
        /// </summary>
        static public float GetParentDrawOffsetX(this IPane self)
        {
            return GetParentDrawOffsetX(self, self.Parent as IPane);
        }

        /// <summary>
        /// 回転、スケール中心
        /// </summary>
        static public float GetParentDrawOffsetY(this IPane self)
        {
            return GetParentDrawOffsetY(self, self.Parent as IPane);
        }

        /// <summary>
        /// 回転、スケール中心
        /// </summary>
        static public float GetParentDrawOffsetX(this IPane self, IPane parent)
        {
            float parentOffsetX = 0.0f;
            if (parent != null)
            {
                switch (self.ParentBasePosTypeH)
                {
                    case HorizontalLocation.Left: parentOffsetX = parent.RenderingWidth * -0.5f; break;
                    case HorizontalLocation.Right: parentOffsetX = parent.RenderingWidth * 0.5f; break;
                }
            }
            return parentOffsetX;
        }

        /// <summary>
        /// 回転、スケール中心
        /// </summary>
        static public float GetParentDrawOffsetY(this IPane self, IPane parent)
        {
            float parentOffsetY = 0.0f;
            if (parent != null)
            {
                switch (self.ParentBasePosTypeV)
                {
                    case VerticalLocation.Top: parentOffsetY = parent.RenderingHeight * 0.5f; break;
                    case VerticalLocation.Bottom: parentOffsetY = parent.RenderingHeight * -0.5f; break;
                }
            }

            return parentOffsetY;
        }

        #endregion 階層関連

        //---------------------------------------------------------------------
        #region ヒットテスト関連

        /// <summary>
        /// ペインの矩形を求める。
        /// </summary>
        static public RectangleF GetBoundingVolume( this IPane pane )
        {
            return MathUtil.MakePositiveSizeRectangle( new RectangleF( pane.DrawOffsetX, -pane.DrawOffsetY - pane.Height, pane.Width, pane.Height ) );
        }

        /// <summary>
        ///
        /// </summary>
        static public RectangleF GetBoundingVolumeInWorld(this IPane pane)
        {
            return GetBoundingVolumeInWorld(pane, false);
        }

        /// <summary>
        ///
        /// </summary>
        static public RectangleF GetBoundingVolumeInWorld(this IPane pane, bool magnifySize)
        {
            return GetBoundingVolume(pane, (pane as Pane).ParentMtx, magnifySize);
        }

        /// <summary>
        ///
        /// </summary>
        static public RectangleF GetBoundingVolumeInWorldWithPartsHierarchy(this IPane pane, bool magnifySize)
        {
            return GetBoundingVolume(pane, (pane as Pane).ParentMtxOverPartsHierarchy, magnifySize);
        }

        /// <summary>
        ///
        /// </summary>
        static public RectangleF GetBoundingVolume(this IPane pane, Matrix parentMtx, bool magnifySize)
        {
            float w = magnifySize ? pane.RenderingWidth : pane.Width;
            float h = magnifySize ? pane.RenderingHeight : pane.Height;

            float halfW = w * 0.5f;
            float halfH = h * 0.5f;

            // ワールド行列とローカル変換行列を乗算します。
            Matrix mtx = parentMtx;
            mtx.Multiply(pane.LocalMtx);

            // 座標系の変更
            mtx.Scale(1.0f, -1.0f);

            // 描画されるペイン右上位置への変換
            mtx.Translate(pane.DrawOffsetX, pane.DrawOffsetY);

            // ペイン中心への変換
            mtx.Translate(halfW, halfH);


            // ローカル系におけるペイン頂点
            PointF[] corners = {
                new PointF( -halfW,  halfH ),
                new PointF(  halfW,  halfH ),
                new PointF(  halfW, -halfH ),
                new PointF( -halfW, -halfH ) };

            // 変換してAABBを計算します。
            mtx.TransformPoints(corners);

            float maxX = float.MinValue;
            float minX = float.MaxValue;
            float maxY = float.MinValue;
            float minY = float.MaxValue;

            foreach (PointF p in corners)
            {
                maxX = Math.Max(p.X, maxX);
                maxY = Math.Max(p.Y, maxY);

                minX = Math.Min(p.X, minX);
                minY = Math.Min(p.Y, minY);
            }

            // ゼロスケールの対策
            if (float.IsNaN(minX) || float.IsNaN(minY) || float.IsNaN(maxX) || float.IsNaN(maxY))
            {
                return RectangleF.Empty;
            }

            return RectangleF.FromLTRB(minX, minY, maxX, maxY);
        }

        /// <summary>
        /// ローカル行列を計算します。
        /// </summary>
        static public Matrix CalcLocalMatrix( PointF pos, float rotZ, PointF scale )
        {
            Matrix mtxLocal = new Matrix();

            mtxLocal.Translate( pos.X, pos.Y );
            mtxLocal.Rotate( rotZ );
            mtxLocal.Scale( scale.X, scale.Y );

            return mtxLocal;
        }

        /// <summary>
        /// ローカル行列を計算します。
        /// </summary>
        static public Matrix4x4 CalcLocalMatrix4x4(Matrix4x4 trans, Matrix4x4 rotateX, Matrix4x4 rotateY, Matrix4x4 rotateZ, Matrix4x4 scale)
        {
            Matrix4x4 mtxLocal = Matrix4x4.Identity;

            mtxLocal = Matrix4x4.Multiply( mtxLocal, scale);
            mtxLocal = Matrix4x4.Multiply( mtxLocal, rotateX );
            mtxLocal = Matrix4x4.Multiply( mtxLocal, rotateY );
            mtxLocal = Matrix4x4.Multiply( mtxLocal, rotateZ );
            mtxLocal = Matrix4x4.Multiply( mtxLocal, trans);

            return mtxLocal;
        }

        static public Matrix CalcLeftTopCoordinateWorldMtx( IPane pane )
        {
            Matrix mtx = pane.WorldMtx;

            mtx.Scale( 1.0f, -1.0f );
            mtx.Translate( pane.DrawOffsetX, pane.DrawOffsetY );

            return mtx;
        }

        #endregion ヒットテスト関連

        //---------------------------------------------------------------------

        #region 参照チェック

        /// <summary>
        /// 使用しているテクスチャ画像を取得します。
        /// </summary>
        public static IEnumerable<string> EnumrateAllTextureName(this IPane pane)
        {
            // 部品ペインの上書きプロパティが利用しているか調査します。
            List<string> texNames = new List<string>();
            if (pane.PaneKind == PaneKind.Parts)
            {
                foreach(var name in IPartsLayoutHelper.EnumerateTexturesForOverwrite(pane.IPartsLayout))
                {
                    if (!texNames.Contains(name))
                    {
                        yield return name;
                    }
                }
            }

            // マスク機能を使っていればテクスチャを調査します。
            if (pane.IMask.IsMaskEnabled)
            {
                IMaterial material = pane.IMask.InternalMaterial;
                foreach (var name in material.GetAllTextureName())
                {
                    if (!texNames.Contains(name))
                    {
                        yield return name;
                    }
                }
            }

            // マテリアルを持っていれば、マテリアル内のテクスチャを調査します。
            foreach (IMaterial material in pane.UsedMaterials)
            {
                foreach (var name in material.GetAllTextureName())
                {
                    if (!texNames.Contains(name))
                    {
                        yield return name;
                    }
                }
            }
        }

        /// <summary>
        /// 指定のテクスチャ画像を使用しているか調査します。
        /// </summary>
        public static bool UsesTextureImage(this IPane pane, string textureImageName)
        {
            var allTextureName = EnumrateAllTextureName(pane);
            return allTextureName.Any((texName) => texName == textureImageName);
        }

        /// <summary>
        /// 指定のテクスチャのキャプチャもとになっているか調査します。
        /// </summary>
        public static bool IsCapturedSource(this IPane pane, string texImageName)
        {
            return pane.PaneName == texImageName;
        }

        /// <summary>
        /// 指定のフォントを使用しているか調査します。
        /// </summary>
        public static bool UsesFont(this IPane pane, ILEFont font)
        {
            if (pane.PaneKind == PaneKind.Parts)
            {
                // 上書きプロパティで使っているフォント
                foreach (var partsPropaerty in pane.IPartsLayout.PartsPropaerties)
                {
                    if (partsPropaerty.Paramater != null)
                    {
                        if (UsesFont(partsPropaerty.Paramater.OwnerPane, font))
                        {
                            return true;
                        }
                    }
                }

                // 部品レイアウトシーンで使っているフォント
                foreach (IPane childPane in pane.IPartsLayout.PartsSubScene.IPaneArray)
                {
                    if (UsesFont(childPane, font))
                    {
                        return true;
                    }
                }
            }
            else if (pane.PaneKind == PaneKind.Textbox)
            {
                Ensure.Operation.ObjectNotNull(pane.ITextBox);
                if (pane.ITextBox.FontName == font.FontName)
                {
                    if (pane.ITextBox.ILEFont != null)
                    {
                        return pane.ITextBox.ILEFont.FontPath == font.FontPath;
                    }

                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// 指定のフォントを使用しているテキストボックスを列挙します。
        /// </summary>
        public static void CollectTextBoxesUsingSpecificFont(this IPane pane, ILEFont font, List<ITextBox> textBoxList)
        {
            if (pane.PaneKind == PaneKind.Parts)
            {
                // 上書きプロパティで使っているフォント
                if (pane.IPartsLayout.PartsPropaerties != null)
                {
                    foreach (var partsPropaerty in pane.IPartsLayout.PartsPropaerties)
                    {
                        if (partsPropaerty.Paramater != null)
                        {
                            CollectTextBoxesUsingSpecificFont(partsPropaerty.Paramater.OwnerPane, font, textBoxList);
                        }
                    }
                }

                // 部品レイアウトシーンで使っているフォント
                if (pane.IPartsLayout.PartsSubScene != null)
                {
                    foreach (IPane childPane in pane.IPartsLayout.PartsSubScene.IPaneArray)
                    {
                        CollectTextBoxesUsingSpecificFont(childPane, font, textBoxList);
                    }
                }
            }
            else if (pane.PaneKind == PaneKind.Textbox)
            {
                Ensure.Operation.ObjectNotNull(pane.ITextBox);
                if (pane.ITextBox.FontName == font.FontName)
                {
                    if (pane.ITextBox.ILEFont != null)
                    {
                        if (pane.ITextBox.ILEFont.FontPath == font.FontPath)
                        {
                            textBoxList.Add(pane.ITextBox);
                        }
                        return;
                    }
                    textBoxList.Add(pane.ITextBox);
                }
            }
        }

        /// <summary>
        /// マテリアルの参照をチェックします。
        /// </summary>
        public static bool HasMaterial(this IPane pane, IMaterial material)
        {
            if (pane.IMaterial == null)
            {
                return false;
            }

            return Array.IndexOf(pane.IMaterial, material) != -1;
        }

        /// <summary>
        /// 引数で指定されたペインの配下にテクスチャをキャプチャしているペインが存在するかチェックします。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        public static bool HasCapturingPaneUnderTree(IPane pane)
        {
            if (pane == null)
            {
                return false;
            }

            // キャプチャペインの場合
            if (IsTextureCapturedPane(pane))
            {
                return true;
            }

            foreach (var child in pane.Children)
            {
                if (HasCapturingPaneUnderTree(child as IPane))
                {
                    return true;
                }
            }

            return false;
        }

        #endregion

        //---------------------------------------------------------------------
        #region 複製

        /// <summary>
        /// 複製します。
        /// </summary>
        internal static Pane CloneWithoutAnimation(this IPane source)
        {
            return Clone_(source, false);
        }

        internal static Pane Clone(this IPane source)
        {
            return Clone_(source, true);
        }

        /// <summary>
        /// 複製します。
        /// </summary>
        private static Pane Clone_(this IPane source, bool copyAnimations)
        {
            Pane newTempPane = new Pane();
            newTempPane.PaneName = source.PaneName;
            if (source.IPaneExParamater is ITextBox)
            {
                var srcText = source.IPaneExParamater as ITextBox;
                new TextBox(newTempPane, srcText.ContentsText, srcText.FontName);
            }
            else if (source.IPaneExParamater is IPicture)
            {
                new Picture(newTempPane);
            }
            else if (source.IPaneExParamater is ILEWindow)
            {
                new LEWindow(newTempPane);
            }
            else if (source.IPaneExParamater is Bounding)
            {
                new Bounding(newTempPane);
            }
            else if (source.IPaneExParamater is Capture)
            {
                new Capture(newTempPane);
            }
            else if (source.IPaneExParamater is PartsLayout)
            {
                PartsLayout sourcePartsLayout = source.IPartsLayout as PartsLayout;
                new PartsLayout(
                    newTempPane,
                    sourcePartsLayout.PartsSubScene,
                    sourcePartsLayout.PartsLayoutName);
            }
            else if (source.IPaneExParamater is IAlignment)
            {
                new Alignment(newTempPane);
            }
            else if (source.IPaneExParamater is IScissor)
            {
                new Scissor(newTempPane);
            }

            PaneParamaterCopyOption option = new PaneParamaterCopyOption();
            option.EnableOption(ParamaterKind.All);
            if (!copyAnimations) { option.DisableOption(ParamaterKind.Animation_All); }
            option.CopyBaseValue = true;

            PaneParamaterPaster.PasteParamaters(option, newTempPane, source);

            return newTempPane;
        }

        #endregion

        //---------------------------------------------------------------------
        #region 整列
        /// <summary>
        /// 整列
        /// </summary>
        public static void AlignPanes(IPane[] paneSet, Func<IPane, IPane, FVec3> getDiffFunc)
        {
            if (paneSet.Length <= 0)
            {
                return;
            }

            IPane basePane = paneSet[0];

            try
            {
                basePane.OwnerSubScene.BeginMassiveModify();

                PaneManipulator paneMnp = new PaneManipulator();
                foreach (var modPane in paneSet)
                {
                    paneMnp.BindTarget(modPane);

                    FVec3 diff = getDiffFunc(basePane, modPane);

                    paneMnp.TranslateInWorld(diff);
                }
            }
            finally
            {
                basePane.OwnerSubScene.EndMassiveModify();
            }
        }

        private static int DetectVecSign_(FVec3 v)
        {
            return Math.Sign(v.X) + Math.Sign(v.Y) + Math.Sign(v.Z);
        }

        /// <summary>
        /// 分布処理
        /// </summary>
        public static void DistributePanes(IPane[] paneSet, Func<IPane, IPane, FVec3> getDiffFunc)
        {
            if (paneSet.Length <= 1)
            {
                return;
            }

            IPane basePane = paneSet[0];

            // 差異ベクトルのリストを作り、差異の小さい順にソートします。
            List<KeyValuePair<IPane, FVec3>> diffSet = new List<KeyValuePair<IPane, FVec3>>();
            {
                foreach (var modPane in paneSet)
                {
                    FVec3 diff = getDiffFunc(basePane, modPane);
                    diffSet.Add(new KeyValuePair<IPane, FVec3>(modPane, diff));
                }

                Comparison<KeyValuePair<IPane, FVec3>> comparison = (
                    KeyValuePair<IPane, FVec3> lhs, KeyValuePair<IPane, FVec3> rhs) =>
                {
                    float diffL = MathUtil.GetVecLength(lhs.Value) * DetectVecSign_(lhs.Value);
                    float diffR = MathUtil.GetVecLength(rhs.Value) * DetectVecSign_(rhs.Value);
                    return (int)(diffL - diffR);
                };
                diffSet.Sort(comparison);
            }

            FVec3 baseDiff = new FVec3(diffSet.First().Value);
            FVec3 aveDiffStep = (diffSet.Last().Value - baseDiff).Scale(1.0f / (diffSet.Count - 1));

            try
            {
                basePane.OwnerSubScene.BeginMassiveModify();

                PaneManipulator paneMnp = new PaneManipulator();
                for (int i = 0; i < diffSet.Count; i++)
                {
                    FVec3 newDiff = baseDiff + new FVec3(aveDiffStep).Scale((float)i);

                    paneMnp.BindTarget(diffSet[i].Key);
                    paneMnp.TranslateInWorld(diffSet[i].Value - newDiff);
                }
            }
            finally
            {
                basePane.OwnerSubScene.EndMassiveModify();
            }
        }
        /// <summary>
        /// リサイズ
        /// </summary>
        public static void ResizePanesToTexutureSize(IPane[] paneSet)
        {
            var paneMnp = new PaneManipulator();
            var subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
            subScene.BeginMassiveModify();
            foreach (var pane in paneSet.Where(x => x.PaneKind == PaneKind.Picture))
            {
                var textureNames = pane.EnumrateAllTextureName();
                // 使用しているテクスチャが一枚の時だけリサイズ
                if (textureNames.Count() != 1)
                {
                    continue;
                }
                var texMgr = LECore.LayoutEditorCore.Scene.CurrentISubScene.ITextureMgr;
                var tex = texMgr.FindITextureImageByName(textureNames.First());
                //                pane.Height = tex.Size.Y;
                paneMnp.BindTarget(pane);
                paneMnp.Width = tex.Size.X;
                paneMnp.Height= tex.Size.Y;
            }
            subScene.EndMassiveModify();
        }

        #endregion 整列

        //---------------------------------------------------------------------
        #region その他

        /// <summary>
        /// ペインが、分割モードに属しているのかどうかを判定します。
        /// </summary>
        static public bool IsInSeparateMode(this IPane pane)
        {
            if(pane == null)
            {
                return false;
            }

            if(pane.OwnerSubScene == null)
            {
                // OwnerSubScene をたどることに失敗しています。
                // クローンしたペインに対して実行している。。。
                // ファイルロード時、シーン登録前のペインについて実行している。。。
                // 回避策として、以下があります。
                //  ・クローン前のペインで実行して結果を保存する。
                //  ・CurrentSubScene(編集対象)からモードを取得する(ただしそれで意味的に正しいかはよく考える必要があります)。
                //  ・IsInSeparateModeWithoutOwner を使う。
                Debug.Assert(false);
                return false;
            }

            return pane.OwnerSubScene.IsAnimEditSeparateMode();
        }

        /// <summary>
        /// IsInSeparateMode() の pane.OwnerSubScene が未設定の状態でも呼び出すコードで利用する版です。
        /// UserDataElement の追加周りで呼ばれています。
        /// </summary>
        static public bool IsInSeparateModeWithoutOwner(this IPane pane)
        {
            if (pane == null)
            {
                return false;
            }

            if (pane.OwnerSubScene == null)
            {
                // ファイルロード時など、null になります。
                return false;
            }

            return pane.OwnerSubScene.IsAnimEditSeparateMode();
        }

        static public bool CheckPanesInSameAnimationSeparateMode(bool isSeperate, IEnumerable<IPane> panes)
        {
            return panes.All((p) => IsInSeparateMode(p) == isSeperate);
        }

        /// <summary>
        /// 0 - 1 に正規化された透明度を取得します。
        /// </summary>
        static public float GetNormalizedTransparency( int transparency )
        {
            return (float)transparency / (float)PaneHelper.MaxTransparency;
        }

        /// <summary>
        /// ペインセットが同一種類のペインで構成されているか判定します。
        ///
        /// 長さゼロのペインセットでは、trueが返ります。
        /// </summary>
        static public bool CheckPaneSetConsistOfSameKindPane( IPane[] paneSet )
        {
            Debug.Assert( paneSet != null );

            if( paneSet.Length != 0 )
            {
                PaneKind kind = paneSet[0].PaneKind;
                for( int i = 1 ; i < paneSet.Length ; i++ )
                {
                    if( paneSet[i].PaneKind != kind )
                    {
                        return false;
                    }
                }
            }
            return true;
        }

        /// <summary>
        /// 詳細なマテリアルを列挙します。
        /// </summary>
        static public IEnumerable<IRevHWMaterial> EnumrateDetailMaterial(this IPane pane)
        {
            var holder = pane.IPaneExParamater as IRevHWMaterialHolder;
            if (holder != null)
            {
                foreach (var mat in holder.IRevHWMaterial)
                {
                    yield return mat;
                }
            }
        }

        /// <summary>
        /// アニメーションを持っているか調べる
        /// </summary>
        static public bool HasAnyAnimation(this IPane pane, bool allTags = true)
        {
            foreach (var animKind in ParamaterKindHelper.AnimationKindSet)
            {
                foreach (var attribute in pane.GetAnimationTargetAttributeSet(animKind))
                {
                    if (allTags)
                    {
                        // すべてのタグ区間を調べます
                        if (attribute.CheckHaveKeyRecursiveInAllTag() || attribute.CheckParameterizedAnimRecursive())
                        {
                            return true;
                        }
                    }
                    else
                    {
                        // 選択中のタグ区間を調べます
                        if (attribute.CheckHaveKeyRecursive() || attribute.CheckParameterizedAnimRecursive())
                        {
                            return true;
                        }
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// アクティブでないアニメーションを取得します。
        /// </summary>
        static public IEnumerable<IAnmAttribute> GatherInactiveAnimations(this IPane pane)
        {
            foreach (var animKind in ParamaterKindHelper.AnimationKindSet)
            {
                foreach (var attribute in pane.GetAnimationTargetAttributeSet(animKind))
                {
                    if (attribute.CheckHaveKeyRecursiveInAllTag() && !attribute.CheckActiveRecursive())
                    {
                        yield return attribute;
                    }
                }
            }
        }

        /// <summary>
        /// baseValueを初期化します
        /// </summary>
        static public void StoreCurrentValueToBaseValue(this IPane target)
        {
            Pane pane = target as Pane;

            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                IAnmAttribute[] attrSet = pane.GetAnimationTargetAttributeSet(kind);

                foreach (AnmAttribute attr in attrSet)
                {
                    attr.SetBaseValueRecursive();
                }
            }
        }

        /// <summary>
        /// 現在値をbaseValueに戻します
        /// </summary>
        static public void ResetValueAsBaseValue(this IPane target)
        {
            Pane pane = target as Pane;

            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                IAnmAttribute[] attrSet = pane.GetAnimationTargetAttributeSet(kind);

                foreach (AnmAttribute attr in attrSet)
                {
                    attr.ResetValueAsBaseValueRecursive();
                }
            }
        }

        #endregion

        #region キャプチャペイン

        /// <summary>
        /// 指定されたペインがキャプチャテクスチャのソースになる可能性があるかどうか判定します。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns>キャプチャテクスチャを作成する可能性があれば true を、それ以外は false を返します。</returns>
        static public bool CanPaneBeCaptureTextureSource(IPane pane)
        {
            if (pane.PaneKind == PaneKind.Capture ||
                pane.PaneKind == PaneKind.Picture ||
                pane.PaneKind == PaneKind.Window)
            {
                return true;
            }

            return false;
        }

        /// テクスチャをキャプチャしているペインかどうかを判定します。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        static public bool IsTextureCapturedPane(IPane pane)
        {
            // キャプチャペイン
            if (pane.PaneKind == PaneKind.Capture)
            {
                return true;
            }

            // 自己参照型のキャプチャテクスチャの場合
            foreach (var mat in pane.IMaterial)
            {
                foreach (MaterialTexMap texMap in mat.IMaterialTexMapSet)
                {
                    if(texMap.IsSelfReferenceCapture(pane.PaneName))
                    {
                        return true;
                    }
                }
            }

            return false;
        }

        /// <summary>
        /// 指定したペインがキャプチャテクスチャを作成するペインだった場合、作成されたテクスチャのインスタンスを取得します。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        static public ITextureImage GetCapturedTexture(IPane pane)
        {
            ITextureImage texture = null;
            if (pane.ICapture != null)
            {
                if (pane.ICapture.OwnTexture != null)
                {
                    texture = pane.ICapture.OwnTexture as ITextureImage;
                }
            }
            else
            {
                texture = PaneHelper.GetCaptureTextureFromPane(pane);
            }

            return texture;
        }

        /// <summary>
        /// ペインがキャプチャテクスチャを作成するペインだった場合、作成された ITextureImage を取得します。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        static public ITextureImage GetCaptureTextureFromPane(IPane pane)
        {
            if (pane != null &&
                pane.OwnerSubScene != null)
            {
                TextureMgr mgr = pane.OwnerSubScene.ITextureMgr as TextureMgr;

                if (mgr != null)
                {
                    return TextureMgrHelper.FindCaptureTextureByName(mgr, PaneHelper.GetFullyQualifiedCaptureTextureName(pane));
                }
            }

            return null;
        }

        /// <summary>
        /// キャプチャしたテクスチャを使用しているペインかどうかを判定します。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        static public bool IsCaptureTextureUsed(IPane pane)
        {
            return GetUsedCaptureTextures(pane).Count() != 0;
        }

        /// <summary>
        /// キャプチャしたテクスチャを使用しているペインの場合は使用しているキャプチャテクスチャを収集します。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        static public ITextureImage[] GetUsedCaptureTextures(IPane pane)
        {
            Pane checkPane = pane as Pane;
            List<ITextureImage> textureList = new List<ITextureImage>();

            foreach (var mat in checkPane.IMaterial)
            {
                foreach (var texMap in mat.IMaterialTexMapSet.EnumerateCaptureRelatedTexMap())
                {
                    ITextureImage tex = SubSceneHelper.FindCaptureTextureRecursively(pane.OwnerSubScene, texMap.TexImgName);
                    if (tex != null)
                    {
                        textureList.Add(tex);
                    }
                }
            }

            return textureList.ToArray();
        }


        /// <summary>
        /// キャプチャしたテクスチャを使用しているペインの場合は使用しているキャプチャテクスチャを収集します。
        /// また、パーツペインの場合はパーツのツリーをたどってパーツ内で使用されているキャプチャテクスチャも収集します。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        static public List<ITextureImage> GetUsedCaptureTexturesWithPartsTree(IPane pane)
        {
            var textures = new List<ITextureImage>();

            if (pane.PaneKind == PaneKind.Parts)
            {
                if (pane.IPartsLayout != null &&
                    pane.IPartsLayout.PartsSubScene != null)
                {
                    foreach (var paneInParts in pane.IPartsLayout.PartsSubScene.IPaneArray)
                    {
                        var result = GetUsedCaptureTexturesWithPartsTree(paneInParts);
                        textures.AddRange(result);
                    }
                }
            }
            else
            {
                textures.AddRange(GetUsedCaptureTextures(pane));
            }

            return textures;
        }

        /// <summary>
        /// 渡されたペインがパーツペインの場合は、内部のキャプチャテクスチャを参照している TexMap のイメージキャッシュをリセットします。
        /// パーツ内にパーツが含まれている場合は再帰的に処理されます。
        /// </summary>
        /// <param name="pane"></param>
        static public void ResetCaptureTextureImageCacheInPartsTree(IPane pane)
        {
            // 入力ペインがパーツでなければ何もしない。
            if (pane.PaneKind != PaneKind.Parts || pane.IPartsLayout == null || pane.IPartsLayout.PartsSubScene == null)
            {
                return;
            }

            // パーツ中のペインに対して、キャプチャテクスチャを参照しているものをリセットしていく。
            foreach (var paneInParts in pane.IPartsLayout.PartsSubScene.IPaneArray)
            {
                if (paneInParts.PaneKind == PaneKind.Parts)
                {
                    // パーツの場合は再帰的に処理する
                    ResetCaptureTextureImageCacheInPartsTree(paneInParts);
                }
                else
                {
                    foreach (var mat in paneInParts.IMaterial)
                    {
                        foreach (MaterialTexMap texMap in mat.IMaterialTexMapSet)
                        {
                            texMap.ResetCache();
                        }
                    }
                }
            }
        }

        /// <summary>
        /// 入力されたテクスチャ名がキャプチャテクスチャだった場合、指定されたペインに設定可能かどうかチェックします。
        /// </summary>
        /// <param name="pane">テクスチャを設定されたマテリアルを参照しているペイン</param>
        /// <param name="textureName">入力テクスチャ名</param>
        /// <returns>設定できる場合は null を、できない場合は障害となっている IPane を返します。</returns>
        static public IPane FindInvalidPaneForMaterialCaptureTextureInput(IPane pane, string textureName)
        {
            if (pane == null ||
                pane.OwnerSubScene == null)
            {
                return null;
            }

            // 入力された名前が「FramebufferCaptureTexture」か、キャプチャテクスチャとして登録されている場合はチェックします。
            if (textureName == PaneHelper.FramebufferCaptureDummyTextureName ||
                TextureMgrHelper.FindCaptureTextureByName(pane.OwnerSubScene.ITextureMgr, PaneHelper.GetCaptureTexturePrefix(pane) + textureName) != null)
            {
                // 指定されたペインの親階層にキャプチャテクスチャを作成しているペインが存在しなければ設定可能です。
                return PaneHelper.CheckHaveCapturedPaneInParentTree(pane.Parent as IPane);
            }

            return null;
        }

        /// <summary>
        /// 親からルートまでたどるペインツリー内にテクスチャをキャプチャするペインが存在するかどうかを調べます。
        /// </summary>
        /// <param name="pane">調査対象のペイン。このペインから調べられます。</param>
        /// <returns></returns>
        static public IPane CheckHaveCapturedPaneInParentTree(IPane pane)
        {
            IHierarchyNode checkNode = pane;

            while (checkNode != null)
            {
                IPane checkPane = checkNode as IPane;
                if (IsTextureCapturedPane(checkPane))
                {
                    return checkPane;
                }

                if (checkNode.Parent == null)
                {
                    // 親が null になっていたらパーツペイン経由で親のサブシーンのペインをたどる
                    if (checkPane.OwnerSubScene.IPartsLayout != null)
                    {
                        checkNode = checkPane.OwnerSubScene.IPartsLayout.OwnerPane;
                    }
                    else
                    {
                        checkNode = null;
                    }
                }
                else
                {
                    checkNode = checkNode.Parent;
                }
            }

            return null;
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        static public IPane CheckHaveCaptureRelativePaneInSubTree(IPane pane)
        {
            if (PaneHelper.IsTextureCapturedPane(pane) ||
                PaneHelper.IsCaptureTextureUsed(pane))
            {
                return pane;
            }

            // パーツレイアウトの場合
            if (pane.IPartsLayout != null)
            {
                IPane result = CheckHaveCaptureRelativePaneInSubTree(pane.IPartsLayout.PartsSubScene.RootIPane);
                if (result != null)
                {
                    return result;
                }
            }

            foreach (var child in pane.Children)
            {
                IPane result = CheckHaveCaptureRelativePaneInSubTree(child as IPane);
                if (result != null)
                {
                    return result;
                }
            }

            return null;
        }

        /// <summary>
        /// キャプチャテクスチャの名称が変わることによるテクスチャ名重複を防ぐためのチェックを行います。
        /// 渡された pane が初期化途中など、必要なデータにアクセスできない場合はチェックを行いません。
        /// </summary>
        /// <param name="pane">改名対象ペイン</param>
        /// <param name="newName">新しい名前</param>
        /// <returns></returns>
        static public bool IsCaptureTextureNameDuplicated(ISubScene subScene, IPane pane, string newName)
        {
            if (pane == null ||
                subScene == null)
            {
                return false;
            }

            // キャプチャソースになりうるペインのみがチェック対象
            if (CanPaneBeCaptureTextureSource(pane) &&
                GetCapturedTexture(pane) != null)
            {
                // 同名テクスチャが存在している場合は基本的に NG だが、リンク切れのキャプチャテクスチャだけは
                // ペインの改名でキャプチャテクスチャのソースになることができるため許可する。
                ITextureImage tex = subScene.ITextureMgr.FindITextureImageByName(newName);
                if (tex != null)
                {
                    if (tex.SourceType == LECoreInterface.TextureSourceType.Dynamic)
                    {
                        ITextureImage captureTexture = TextureMgrHelper.FindCaptureTextureByName(subScene.ITextureMgr, newName);
                        if (captureTexture != null &&
                            captureTexture.ICaptureTexture.OwnerPane == null)
                        {
                            return false;
                        }
                    }

                    return true;
                }
            }

            return false;

        }

        /// <summary>
        /// 指定したペインが既存のリンク切れ状態のキャプチャテクスチャのソースとして復帰できないかチェックし、可能な場合は復帰します。
        /// </summary>
        /// <param name="pane"></param>
        static public void ReconnectInvalidCaptureTexture(IPane pane)
        {
            // キャプチャテクスチャのもとになる可能性のあるペインは、作成時に既存のリンク切れのキャプチャテクスチャのソースになれないかチェックします。
            if (PaneHelper.CanPaneBeCaptureTextureSource(pane))
            {
                ITextureImage tex = PaneHelper.GetCaptureTextureFromPane(pane);

                if (tex != null)
                {
                    // キャプチャテクスチャのインスタンスは存在して、参照しているペインがない時はリンク切れ状態のため
                    // 同名のキャプチャテクスチャとして復帰します。
                    if (tex.ICaptureTexture.OwnerPane == null)
                    {
                        CaptureTextureManipulator mnp = new CaptureTextureManipulator();
                        mnp.BindTarget(tex.ICaptureTexture);
                        mnp.OwnerPane = pane;
                    }
                }
            }
        }

        /// <summary>
        /// マテリアルでフレームバッファキャプチャを指定する際のダミーテクスチャ名
        /// </summary>
        public const string FramebufferCaptureDummyTextureName = "FramebufferCaptureTexture";

        /// <summary>
        /// キャプチャテクスチャ設定時の一時的なペインを作成します。
        /// </summary>
        /// <returns></returns>
        static public IPane CreateCaptureDummyPane()
        {
            Pane pane = new Pane();
            pane.PaneName = FramebufferCaptureDummyTextureName;

            return pane;
        }

        /// <summary>
        /// キャプチャテクスチャ名のパス区切り文字
        /// </summary>
        public const string CaptureTextureNameSeparator = "%";

        /// <summary>
        /// 所属するレイアウトを考慮して修飾されたキャプチャテクスチャのプリフィックスを取得します。
        /// </summary>
        /// <param name="pane">キャプチャテクスチャのソースとなっているペイン</param>
        /// <returns></returns>
        static public string GetCaptureTexturePrefix(IPane pane)
        {
            string name = "";

            IPane checkPane = pane;

            // パーツペインをを上にたどっていき、名前で階層構造を作成する。
            while (checkPane.OwnerSubScene != null &&
                   checkPane.OwnerSubScene.IPartsLayout != null)
            {
                name = CaptureTextureNameSeparator + checkPane.OwnerSubScene.IPartsLayout.OwnerPane.PaneName + name;
                checkPane = checkPane.OwnerSubScene.IPartsLayout.OwnerPane;
            }

            return name.Length > 0 ? name.Remove(0, 1) : "";
        }

        /// <summary>
        /// 所属するレイアウトを考慮して修飾されたキャプチャテクスチャの名前を取得します。
        /// </summary>
        /// <param name="pane">キャプチャテクスチャのソースとなっているペイン</param>
        /// <returns></returns>
        static public string GetFullyQualifiedCaptureTextureName(IPane pane)
        {
            string prefix = GetCaptureTexturePrefix(pane);
            return MakeCaptureTextureName(GetCaptureTexturePrefix(pane), pane.PaneName);
        }

        /// <summary>
        /// キャプチャテクスチャの名前を作成します。
        /// </summary>
        /// <param name="prefix"></param>
        /// <param name="name"></param>
        /// <returns></returns>
        static public string MakeCaptureTextureName(string prefix, string name)
        {
            return prefix.Length > 0 ? prefix + CaptureTextureNameSeparator + name : name;
        }

        #endregion

        #region StateMachine

        public static IStateMachine GetPartsPaneStateMachine(this IPane pane)
        {
            return pane?.IPartsLayout?.PartsSubScene?.IStateMachine;
        }

        public static bool IsStateMachinePartsPane(this IPane pane)
        {
            var stateMachine = pane.GetPartsPaneStateMachine();
            if(stateMachine == null)
            {
                return false;
            }

            return stateMachine.IsEnabled;
        }

        #endregion
    }
}
