﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;

namespace LECore.Structures
{
    using GroupMgrParamaters = LEGroupMgrModifyEventArgs.GroupMgrParamaters;

    /// <summary>
    /// LEGroup の概要の説明です。
    /// </summary>
    internal class LEGroup : ILEGroup
    {
        /// <summary>
        /// 変更イベントデリゲート
        /// </summary>
        public delegate void OnChnageHandler( LEGroupMgrModifyEventArgs args );

        readonly ArrayList _groupMember = new ArrayList();
        string _name = string.Empty;
        string _userComment = string.Empty;
        readonly LEGroupMgr _ownerManager;
        /// <summary>
        /// グループ変更イベント
        /// </summary>
        readonly OnChnageHandler _onChnageHandler;




        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEGroup( string name, LEGroupMgr ownerManager, OnChnageHandler onChnageHandler )
        {
            Debug.Assert( onChnageHandler != null );
            Debug.Assert( ownerManager != null );

            _name = name;
            _ownerManager = ownerManager;
            _onChnageHandler = onChnageHandler;
        }

        /// <summary>
        /// 新規登録
        /// 重複登録は無視されます。
        /// </summary>
        public void AddMember( IPane newMember, bool enableEvent )
        {
            if (this.IsReadOnlyLocked)
            {
                return;
            }

            // すでに含まれていないか？
            if( FindMemberByName( newMember.PaneName ) == null )
            {
                // 登録
                _groupMember.Add( newMember );

                if( enableEvent != false )
                {
                    //// マネージャに通知
                    LEGroupMgrModifyEventArgs args = LEGroupMgrModifyEventArgs.MakePaneAddEvent(
                                  this._ownerManager,
                                  this,
                                  newMember );
                    _onChnageHandler( args );
                }
            }
        }


        /// <summary>
        /// 新規登録
        /// 重複登録は無視されます。
        /// </summary>
        public void AddMember( IPane newMember )
        {
            AddMember( newMember, true);
        }

        /// <summary>
        /// 対象グループから、指定のメンバを消去します。
        /// 子供要素からの削除は行いません。
        /// </summary>
        static int RemoveMember_( LEGroup group, IPane deleteMember )
        {
            if (group.IsReadOnlyLocked)
            {
                return -1;
            }

            // bool bResult = false;
            int idx = group._groupMember.IndexOf( deleteMember );

            group._groupMember.Remove( deleteMember );

            return idx;
        }

        /// <summary>
        /// メンバの削除
        /// </summary>
        public void RemoveMember( IPane deleteMember )
        {
            int idx = RemoveMember_( this, deleteMember );
            if( idx != -1 )
            {
                // 削除が起こった。
                LEGroupMgrModifyEventArgs args = LEGroupMgrModifyEventArgs.MakePaneRemoveEvent(
                    this._ownerManager,
                    this,
                    deleteMember,
                    idx );

                _onChnageHandler( args );
            }
        }

        /// <summary>
        /// 子供ペインの順番を変更します。
        /// 描画順に対応しています。
        /// </summary>
        public int ChangeMemberOrder( IPane member, int newIndex )
        {
            Debug.Assert( this.Contains( member ) );

            if (this.IsReadOnlyLocked)
            {
                return -1;
            }

            // 挿入し、現在のインスタンスを削除します。
            int currentIndex = _groupMember.IndexOf(member);
            if( currentIndex != newIndex )
            {
                _groupMember.Insert( newIndex, member );

                if( currentIndex > newIndex )
                {
                    _groupMember.RemoveAt( currentIndex + 1 );
                }
                else
                {
                    _groupMember.RemoveAt( currentIndex );
                }

                // メッセージを送信します。
                LEGroupMgrModifyEventArgs args
                    = LEGroupMgrModifyEventArgs.MakeEditPaneOrderEvent(
                        _ownerManager,
                        this,
                        member,
                        currentIndex,
                        newIndex );

                _onChnageHandler( args );

            }

            return ( currentIndex > newIndex ) ? newIndex + 1: newIndex;
        }

        /// <summary>
        /// 名前でメンバを検索する。
        /// メンバの子供は調べない。
        /// </summary>
        public IPane FindMemberByName( string name )
        {
            foreach( IPane subMember in this.Member )
            {
                if( name == subMember.PaneName )
                {
                    return subMember;
                }
            }
            return null;
        }

        /// <summary>
        /// メンバの番号を取得します。
        /// </summary>
        public int GetMemberIndex( IPane member )
        {
            return _groupMember.IndexOf( member );
        }

        /// <summary>
        /// グループが同一の内容であるか調べます。
        /// </summary>
        public bool IsSame( LEGroup another )
        {
            return this.IsSame( another.Member );
        }

        /// <summary>
        /// グループが同一の内容であるか調べます。
        /// </summary>
        public bool IsSame( IPane[] members )
        {
            if( members.Length == this.Member.Length )
            {
                // 比較対照のすべてのメンバが自己のグループから
                // 発見されるか確認します。
                foreach( IPane member in members )
                {
                    if( FindMemberByName( member.PaneName ) == null )
                    {
                        // 発見できないメンバが存在した。
                        return false;
                    }
                }
                return true;
            }
            return false;
        }



        #region ILEGroup メンバ

        /// <summary>
        /// グループメンバを取得する
        /// </summary>
        public IPane[] Member
        {
            get { return _groupMember.ToArray( typeof( IPane ) ) as IPane[]; }
        }

        /// <summary>
        /// グループメンバを持つか？
        /// </summary>
        public bool HasMember
        {
            get { return _groupMember.Count != 0; }
        }

        /// <summary>
        /// グループ要素の名前
        /// (グループ要素間での名前の重複は不正です。)
        /// </summary>
        public string GrouprName
        {
            get { return _name; }
            set
            {
                if( _name != value )
                {
                    _name = value;

                    // メッセージを送信します。
                    LEGroupMgrModifyEventArgs args =
                        LEGroupMgrModifyEventArgs.MakeEditGroupEvent( _ownerManager, this );

                    _onChnageHandler( args );
                }
            }
        }

        /// <summary>
        /// ユーザコメント
        /// </summary>
        public string UserCommentString
        {
            get { return _userComment; }
            set
            {
                if( _userComment != value )
                {
                    _userComment = value;

                    // メッセージを送信します。
                    LEGroupMgrModifyEventArgs args =
                        LEGroupMgrModifyEventArgs.MakeEditGroupEvent( _ownerManager, this );

                    _onChnageHandler( args );
                }
            }
        }

        /// <summary>
        /// ペインがメンバか調べます
        /// </summary>
        public bool Contains( IPane pane )
        {
            return _groupMember.Contains( pane );
        }

        /// <summary>
        /// 読み取り専用か？
        /// </summary>
        public bool IsReadOnlyLocked
        {
            get { return _ownerManager != null ? _ownerManager.CheckReadOnlyByName(this.GrouprName) : false; }
        }

        #endregion
    }
}
