﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;

namespace LECore.Structures.Core
{
    using LECore.Util;// to use Range

    public delegate void OnTimeChangedHandler(int time, TimeChageEventType type);

    /// <summary>
    /// 時間イベントの種類
    /// </summary>
    public enum TimeChageEventType
    {
        Play,
        RangeChanged,
        Tick,
        Stop
    }

    /// <summary>
    /// 時間イベントをリレーするクラス
    /// </summary>
    internal class TimeReleyProvider : ITimeChangeEventProvider
    {
        public int Time { get; private set; }

        // 時間変更にともなう処理
        // OnTimeChanged の前に行う
        public HashSet<ITimeChageEventListener> TimeChangingHandlers { get; private set; } = new HashSet<ITimeChageEventListener>();

        public event OnTimeChangedHandler OnTimeChanged;

        public void Reley(int time, TimeChageEventType type)
        {
            this.Time = time;

            foreach (var handler in TimeChangingHandlers)
            {
                handler.OnTimeChangedHandler(time, type);
            }
            this.OnTimeChanged?.Invoke(time, type);
        }
    }

    /// <summary>
    /// 時間変更イベントを監視する機能を基底するインタフェース。
    ///
    /// GrobalTime.OnTimeChanged イベントを利用するクラスは継承してください。
    /// </summary>
    public interface ITimeChageEventListener
    {
        void OnTimeChangedHandler( int time, TimeChageEventType type );
    }

    /// <summary>
    /// 時間変更イベントを提供するもの
    /// </summary>
    public interface ITimeChangeEventProvider
    {
        // 時間変更にともなう処理
        // OnTimeChanged の前に行う
        // イベントハンドラを多く登録すると削除に時間がかかるので Dictionary で管理する。
        HashSet<ITimeChageEventListener> TimeChangingHandlers { get; }

        // 時間変更後の処理
        event OnTimeChangedHandler OnTimeChanged;
    }

    /// <summary>
    /// インタフェース
    /// </summary>
    public interface ITime
    {
        void Play();
        void Stop();

        bool IsPlaying { get; set; }

        int Time { get; set; }

        int AnimPlayStartTime { get; }

        int AnimPlayEndTime { get; }

        void PostTickEvent();
    }

    static public class ITimeHelper
    {
        static public float GetNormalizedTime(int startTime, int endTime, int time)
        {
            int duration = endTime - startTime;
            if (duration <= 0)
            {
                return 0.0f;
            }

            time -= startTime;

            return (float)time / duration;
        }
    }

    /// <summary>
    /// シーンの時間を表す、シングルトンクラスです。
    ///
    /// 時間に関連するパラメータを管理します。
    /// 変更の通知イベントを送信する役割を果たします。
    /// </summary>
    public class GlobalTime : ITimeChangeEventProvider, ITime
    {
        // 時間変更にともなう処理
        // OnTimeChanged の前に行う
        public HashSet<ITimeChageEventListener> TimeChangingHandlers { get; private set; } = new HashSet<ITimeChageEventListener>();

        // ---------------- 時間が変更されたことを通知するイベント
        public event OnTimeChangedHandler       OnTimeChanged;

        // ---------------- 再生制御
        /// <summary>
        /// シーンのアニメーションを再生します。
        /// </summary>
        public void Play()
        {
            _isplaying = true;
            OnTimeChanged_( _time, TimeChageEventType.Play );
        }
        /// <summary>
        /// シーンのアニメーションを停止します。
        /// </summary>
        public void Stop()
        {
            _isplaying = false;
            OnTimeChanged_( _time, TimeChageEventType.Stop );
        }

        public void PostTickEvent()
        {
            NotifyTimeChangesToAll_();
        }

        /// <summary>
        /// 時間パラメータを設定します。
        /// </summary>
        public bool Set( int start, int end, int playStart, int playEnd )
        {
            // 開始と終了が入れ替わっている場合、
            // 警告を表示して、設定を行いません。
            if( ( start > end ) || ( playStart > playEnd ) )
            {
                return false;
            }

            if (Scene.CurrentSubScene == null)
            {
                return false;
            }

            Scene.CurrentSubScene.AnimRange.Max = Math.Max(end, playEnd);
            Scene.CurrentSubScene.AnimRange.Min = Math.Min(start, playStart);

            Scene.CurrentSubScene.AnimPlayRange.Max = playEnd;
            Scene.CurrentSubScene.AnimPlayRange.Min = playStart;

            NotifyRangeChangesToAll_();

            return true;
        }

        private bool _useTargetFrameSectionRange = false;

        /// <summary>
        /// 選択された区間タグの範囲をアニメーション再生範囲にします。
        /// </summary>
        public bool UseTargetFrameSectionRangeAsPlayRange
        {
            get
            {
                return _useTargetFrameSectionRange;
            }
            set
            {
                if (_useTargetFrameSectionRange != value)
                {
                    _useTargetFrameSectionRange = value;
                    FitAndNotifyAfterTargetFrameSectionChanged();
                }
            }
        }

        private bool _autoAdjustPlayRange = false;

        /// <summary>
        /// ビューア転送時に再生範囲を自動的に調整します。
        /// </summary>
        public bool AutoAdjustPlayRange
        {
            get
            {
                return _autoAdjustPlayRange;
            }
            set
            {
                if (_autoAdjustPlayRange != value)
                {
                    _autoAdjustPlayRange = value;
                    FitAndNotifyAfterTargetFrameSectionChanged();
                }
            }
        }

        /// <summary>
        /// 選択区間タグの変更に伴い再生範囲の変更に伴う調整と通知を行います。
        /// </summary>
        public void FitAndNotifyAfterTargetFrameSectionChanged()
        {
            bool isPlaying = IsPlaying;
            try
            {
                // 再生中にしないと処理が重い
                // 再生中にすることで、UI更新処理が ViewManager の OnSceneModifyHandler で フィルタアウトされ、Stop() 時に一度だけ行われるようになる。
                if (!isPlaying)
                {
                    Play();
                }

                if (Scene.CurrentSubScene == null)
                {
                    NotifyRangeChangesToAll_();
                    return;
                }

                if (AnimPlayStartTime < Scene.CurrentSubScene.AnimRange.Min)
                {
                    Scene.CurrentSubScene.AnimRange.Min = AnimPlayStartTime;
                }

                if (Scene.CurrentSubScene.AnimRange.Max < AnimPlayEndTime)
                {
                    Scene.CurrentSubScene.AnimRange.Max = AnimPlayEndTime;
                }

                NotifyRangeChangesToAll_();

                if (_useTargetFrameSectionRange)
                {
                    if (IsTargetFrameSectionSelected && AnimPlayStartTime != Time)
                    {
                        Time = AnimPlayStartTime;
                    }
                }
            }
            finally
            {
                if (!isPlaying)
                {
                    Stop();
                }
            }
        }

        /// <summary>
        /// 区間タグが選択されているか？
        /// </summary>
        public bool IsTargetFrameSectionSelected
        {
            get
            {
                return Scene.CurrentSubScene?.IAnimFrameSectionSet?.IAnimFrameSectionSet != null;
            }
        }

        /// <summary>
        /// 再生レンジ内か判定します。
        /// </summary>
        public bool CheckInPlayRange( float time )
        {
            return (Inst.AnimPlayStartTime <= time) &&
                    (time <= Inst.AnimPlayEndTime);
        }

        /// <summary>
        /// イベントの通知を中断する
        /// </summary>
        public void SuspendEventNotification()
        {
            if( _timeChangeEventSuspended <= 0 )
            {
                _suspendedEventMap.Clear();
                _timeChangeEventSuspended = 0;
            }

            _timeChangeEventSuspended++;
        }

        /// <summary>
        /// イベントの通知を再開する
        /// </summary>
        public void ResumeEventNotification(bool bNotifyIfNeeded)
        {
            _timeChangeEventSuspended--;

            if (_timeChangeEventSuspended <= 0)
            {
                if (bNotifyIfNeeded)
                {
                    try
                    {
                        CallingOnTimeChanged = true;

                        // イベント発行によって _suspendedEventMap の内容が変更される可能性があるので
                        // 一旦リストに退避してから実行します。
                        List<TimeChageEventType> eventTypes = new List<TimeChageEventType>(_suspendedEventMap.Keys);
                        foreach (TimeChageEventType type in eventTypes)
                        {
                            int time = _suspendedEventMap[type];
                            foreach (var handler in TimeChangingHandlers)
                            {
                                handler.OnTimeChangedHandler(time, type);
                            }

                            OnTimeChanged?.Invoke(time, type);
                        }
                    }
                    finally
                    {
                        CallingOnTimeChanged = false;
                    }
                }
                _suspendedEventMap.Clear();
                _timeChangeEventSuspended = 0;
            }
        }

        #region フィールド

        // シングルトン実体
        static GlobalTime       _theInst = null;

        // グローバル時間
        int          _time = 0;

        bool         _isplaying     = false;

        int		_timeChangeEventSuspended = 0;
        Dictionary<TimeChageEventType, int> _suspendedEventMap = new Dictionary<TimeChageEventType, int>();

        #endregion

        #region ---------------- 非公開メソッド ----------------

        void OnTimeChanged_( int time, TimeChageEventType type )
        {
            if (_timeChangeEventSuspended <= 0)
            {
                Debug.Assert(!CallingOnTimeChanged);
                CallingOnTimeChanged = true;
                foreach (var handler in TimeChangingHandlers)
                {
                    handler.OnTimeChangedHandler(time, type);
                }
                OnTimeChanged?.Invoke(time, type);
                CallingOnTimeChanged = false;
            }
            else
            {
                // サスペンド中は、
                // 各種類ごとに最新のイベントのみ記録しておきます。
                if (_suspendedEventMap.ContainsKey(type))
                {
                    _suspendedEventMap[type] = time;
                }
                else
                {
                    _suspendedEventMap.Add(type, time);
                }
            }
        }

        /// <summary>
        /// 時間変更イベントを通知します。
        /// </summary>
        void NotifyTimeChangesToAll_()
        {
            Debug.Assert( IsTimeParamsValid_() );
            OnTimeChanged_(_time, TimeChageEventType.Tick);

            //DbgConsole.WriteLine("TimeEventDelgs = {0}", OnTimeChanged.GetInvocationList().Length);
        }

        /// <summary>
        /// 範囲変更を通知します。
        /// </summary>
        void NotifyRangeChangesToAll_()
        {
            Debug.Assert(IsTimeParamsValid_());
            OnTimeChanged_(_time, TimeChageEventType.RangeChanged);
        }
        #endregion // ---------------- 非公開メソッド ----------------

        #region ------------------ プロパティ ------------------

        /// <summary>
        /// アニメーションが再生中か取得または設定します。
        /// </summary>
        public bool IsPlaying
        {
            get{ return _isplaying;}
            set { _isplaying = value; }
        }

        public bool CallingOnTimeChanged
        {
            get;
            private set;
        }

        // UI 側で管理するほうが適切かもしれない
        /// <summary>
        /// SetTimeForcibly 実行中かどうか
        /// </summary>
        public bool SettingTimeForcibly
        {
            get;
            set;
        }

        /// <summary>
        /// アニメーション範囲(開始)
        /// </summary>
        public int          AnimStartTime
        {
            get
            {
                return Scene.CurrentSubScene != null ? Scene.CurrentSubScene.AnimRange.Min : 0;
            }
        }
        /// <summary>
        /// アニメーション範囲(終了)
        /// </summary>
        public int          AnimEndTime
        {
            get
            {
                return Scene.CurrentSubScene != null ? Scene.CurrentSubScene.AnimRange.Max : 100;
            }
        }

        /// <summary>
        /// アニメーション再生範囲(開始)
        /// </summary>
        public int        AnimPlayStartTime
        {
            get
            {
                if (Scene.CurrentSubScene == null)
                {
                    return 0;
                }

                if (UseTargetFrameSectionRangeAsPlayRange)
                {
                    var targetIAnimFrameSection = Scene.CurrentSubScene?.IAnimFrameSectionSet?.TargetIAnimFrameSection;
                    if (targetIAnimFrameSection != null)
                    {
                        return targetIAnimFrameSection.StartFrame;
                    }
                }

                return Scene.CurrentSubScene.AnimPlayRange.Min;
            }
        }
        /// <summary>
        /// アニメーション再生範囲(終了)
        /// </summary>
        public int        AnimPlayEndTime
        {
            get
            {
                if (Scene.CurrentSubScene == null)
                {
                    return 100;
                }

                if (UseTargetFrameSectionRangeAsPlayRange)
                {
                    var targetIAnimFrameSection = Scene.CurrentSubScene?.IAnimFrameSectionSet?.TargetIAnimFrameSection;
                    if (targetIAnimFrameSection != null)
                    {
                        return targetIAnimFrameSection.EndFrame;
                    }
                }

                return Scene.CurrentSubScene.AnimPlayRange.Max;
            }
        }

        /// <summary>
        /// グローバル時間
        /// </summary>
        public int Time
        {
            get{ return  _time;}
            set
            {
                if( _time != value )
                {
                    _time = value;
                    NotifyTimeChangesToAll_();
                }
            }
        }

        /// <summary>
        /// 時間を設定し、強制的に更新します。
        /// </summary>
        public void SetTimeForcibly( int value)
        {
            SettingTimeForcibly = true;
            _time = value;
            Debug.Assert(!CallingOnTimeChanged);
            CallingOnTimeChanged = true;

            foreach (var handler in TimeChangingHandlers)
            {
                handler.OnTimeChangedHandler(_time, TimeChageEventType.Tick);
            }
            OnTimeChanged?.Invoke(_time, TimeChageEventType.Tick);
            CallingOnTimeChanged = false;
            SettingTimeForcibly = false;
        }

        /// <summary>
        /// シングルトン実体を取得します。
        /// </summary>
        static public GlobalTime Inst
        {
            get
            {
                if( _theInst == null )
                {
                    _theInst = new GlobalTime();
                }

                return _theInst;
            }
        }

        #endregion // ------------------ プロパティ ------------------

        #region ------------ 検証 ------------

        /// <summary>
        /// 値が現在の時間として適切か判定します。
        /// </summary>
        public bool CheckValidtyForCurrentTime( int currentTime )
        {
            return (currentTime >= AnimStartTime && currentTime <= AnimEndTime);

        }

        /// <summary>
        /// 値がアニメーション範囲として適切か判定します。
        /// </summary>
        public bool CheckValidtyForAnimRange( int anmStart, int anmEnd )
        {
            if( anmStart <= anmEnd )
            {
                if (anmStart <= AnimPlayStartTime && AnimPlayEndTime <= anmEnd)
                {
                    return true;
                }
            }
            return false;
        }


        /// <summary>
        /// 値がアニメーション再生範囲として適切か判定します。
        /// </summary>
        public bool CheckValidtyForAnimPlayRange( int anmStart, int anmEnd )
        {
            if( anmStart <= anmEnd )
            {
                if (AnimStartTime <= anmStart && anmEnd <= AnimEndTime)
                {
                    return true;
                }
            }
            return false;
        }

        /// <summary>
        /// 現在の値が適切か判定します。
        /// </summary>
        bool IsTimeParamsValid_()
        {
            return ( CheckValidtyForCurrentTime( _time ) &&
                CheckValidtyForAnimRange(AnimStartTime, AnimEndTime) &&
                CheckValidtyForAnimPlayRange(AnimPlayStartTime, AnimPlayEndTime));
        }

        #endregion // ------------ 検証 ------------

    }
}
