﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Diagnostics;
using System.Drawing;

namespace LECore.Structures.Core
{
    /// <summary>
    /// ソートに使用される、比較処理ファンクタクラス。
    /// キーを時間順に並べます。
    /// </summary>
    public class AnmKeyFrameComparer : IComparer
    {
        public int Compare( object x, object y )
        {
            AnmKeyFrame frmL = x as AnmKeyFrame;
            AnmKeyFrame frmR = y as AnmKeyFrame;
            Trace.Assert( frmL != null && frmR != null );

            if( frmL.Time < frmR.Time )
            {
                return -1;
            }
            else if( frmL.Time == frmR.Time )
            {
                return 0;
            }
            else
            {
                return 1;
            }
        }
    }

    /// <summary>
    /// キーフレームクラス
    /// 時間 time での アトリビュート値 を持っているが、
    /// このクラスにとっては、そのアトリビュート値がどんな型なのか、関係がない。
    ///
    /// 本クラスに関する変更は、シーンに通知されない。
    /// 変更があった場合、持ち主クラスの更新メソッドを呼ぶ。
    /// </summary>
    internal class AnmKeyFrame : ICloneable, IAnmKeyFrame
    {
        #region ------------- フィールド -------------
        // 時間
        float _time;

        // 値(実際の解釈はカーブが行う)
        object _value;

        // 傾き情報
        float _inTangent = 0.0f;  // 傾き(左方向)
        float _outTangent = 0.0f;  // 傾き(右方向)

        bool _unifyTangents = true;

        // 補間種類 (傾き自動計算時の振る舞い)
        InterporationType _interporationType = InterporationType.Spline;


        // 自分が所属しているアニメーションカーブ
        readonly AnmCurve _anmCurve = null;


        #endregion ------------- フィールド -------------


        #region ------------- プロパティ -------------

        /// <summary>
        /// キーを保持しているアニメーションカーブ
        /// </summary>
        public AnmCurve OwnerAnmCurve
        {
            get { return _anmCurve; }
        }

        public IAnmCurve OwnerIAnmCurve
        {
            get { return _anmCurve; }
        }

        /// <summary>
        /// 時間
        /// </summary>
        public float Time
        {
            get { return _time; }
            set
            {
                if( _time != value )
                {
                    _time = value;
                    OwnerAnmCurve.SetCurveStateDirty();
                }
            }
        }

        /// <summary>
        /// 時間
        /// </summary>
        public int TimeAsInt
        {
            get { return (int)Math.Ceiling(_time); }
            set
            {
                if (_time != value)
                {
                    _time = value;
                    OwnerAnmCurve.SetCurveStateDirty();
                }
            }
        }

        /// <summary>
        /// 左側接線
        ///
        /// 接線情報も内部的には ViewScale = 1.0 の時の情報を保持しておき、キーの値と整合性をとるため ViewScale を適用します。
        /// </summary>
        public float InTangent
        {
            get { return _inTangent * ViewScale; }
            set
            {
                Trace.Assert( value != float.NaN );
                if( _inTangent != value && TangentModifyEnabled )
                {
                    _inTangent = value / ViewScale;
                    if( _unifyTangents )
                    {
                        _outTangent = _inTangent;
                    }
                }
            }
        }

        /// <summary>
        /// 左側接線
        ///
        /// 内部で保持している ViewScale = 1.0 の時のデータをそのまま取得します。
        /// </summary>
        public float InTangentNoEffect
        {
            get { return _inTangent; }
        }

        /// <summary>
        /// 右側接線
        ///
        /// 接線情報も内部的には ViewScale = 1.0 の時の情報を保持しておき、キーの値と整合性をとるため ViewScale を適用します。
        /// </summary>
        public float OutTangent
        {
            get { return _outTangent * ViewScale; }
            set
            {
                Trace.Assert( value != float.NaN );

                if( _outTangent != value && TangentModifyEnabled )
                {
                    _outTangent = value / ViewScale;
                    if( _unifyTangents )
                    {
                        _inTangent = _outTangent;
                    }
                }
            }
        }

        /// <summary>
        /// 右側接線
        ///
        /// 内部で保持している ViewScale = 1.0 の時のデータをそのまま取得します。
        /// </summary>
        public float OutTangentNoEffect
        {
            get { return _outTangent; }
        }


        /// <summary>
        /// 見かけ上のスケール
        /// </summary>
        public float ViewScale
        {
            get
            {
                if (_anmCurve != null &&
                    _anmCurve.TargetAttribute != null)
                {
                    return _anmCurve.TargetAttribute.ViewScale;
                }
                return 1.0f;
            }
        }

        /// <summary>
        /// 補間種類
        /// </summary>
        public InterporationType InterporationType
        {
            get { return _interporationType; }
            set
            {
                if( _interporationType != value &&
                    !_anmCurve.LockInterpType &&
                    AnmAttribute.CheckInterpTypeValid( _anmCurve.TargetAttrType, value ) )
                {
                    // スロープ同期をOnにします。
                    if( value == InterporationType.Step ||
                        value == InterporationType.Spline ||
                        value == InterporationType.Zero )
                    {
                        this.UnifyTangents = true;
                    }
                    // スロープをゼロに設定します。
                    if( value == InterporationType.Step ||
                        value == InterporationType.Zero )
                    {
                        this.InTangent = 0.0f;
                        this.OutTangent = 0.0f;
                    }
                    _interporationType = value;
                    // OwnerAnmCurve.SetCurveSlopeDirty();
                }
            }
        }

        /// <summary>
        /// 値(方情報はOwnerAnmCurveから参照します。)
        /// </summary>
        public object Value
        {
            get { return _value; }
            set
            {
                if( _value != value )
                {
                    _value = value;
                    // OwnerAnmCurve.SetCurveSlopeDirty();
                }
            }
        }

        #endregion // ------------- プロパティ -------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmKeyFrame( AnmCurve anmCurve, float time, object value )
            : this( anmCurve, time, value, true, 0.0f, 0.0f, anmCurve.DefaultInterpType )
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmKeyFrame( AnmCurve anmCurve, float time, object value, float viewScale )
            : this( anmCurve, time, value, true, 0.0f, 0.0f, anmCurve.DefaultInterpType, viewScale, true )
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmKeyFrame
            (
            AnmCurve anmCurve,
            float time,
            object value,
            bool unifyTangents,
            float inTangent,
            float outTangent,
            InterporationType interporationType
            )
            : this(anmCurve, time, value, unifyTangents, inTangent, outTangent, interporationType, 1.0f, true)
        {
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmKeyFrame
            (
            AnmCurve anmCurve,
            float time,
            object value,
            bool unifyTangents,
            float inTangent,
            float outTangent,
            InterporationType interporationType,
            float viewScale,
            bool normalize
            )
        {
            _anmCurve = anmCurve;
            _time = time;

            // 値を、妥当な値域にクランプするため、一旦Floatに変換します。
            float valueAsFloat = AnmAttribute.GetValueAsFloat(OwnerAnmCurve.TargetAttrType, value);
            if (normalize)
            {
                valueAsFloat /= ViewScale;
            }
            valueAsFloat = ClampByTargetRange_(valueAsFloat);
            AnmAttribute.SetValueAsFloat(OwnerAnmCurve.TargetAttrType, ref _value, valueAsFloat);

            _unifyTangents = unifyTangents;

            _inTangent = inTangent / ViewScale;
            _outTangent = outTangent / ViewScale;

            _interporationType = interporationType;
        }


        /// <summary>
        /// パラメータコピー
        ///
        /// 同じアニメーションカーブに属する、
        /// キー同士である必要があります。
        /// </summary>
        /// <param name="src"></param>
        public void SetParamaters( AnmKeyFrame src )
        {
            Debug.Assert( this.OwnerAnmCurve.TargetAttrType ==
                           src.OwnerAnmCurve.TargetAttrType );

            _time = src.Time;
            // 値を、妥当な値域にクランプするため、一旦Floatに変換します。
            // _value = src.Value;
            AnmAttribute.SetValueAsFloat( OwnerAnmCurve.TargetAttrType, ref _value, src.ValueAsFloat / ViewScale );

            _unifyTangents = src.UnifyTangents;

            _inTangent = src.InTangentNoEffect;
            _outTangent = src.OutTangentNoEffect;

            _interporationType = src.InterporationType;
        }

        /// <summary>
        /// 対象の値域でクランプします。
        /// </summary>
        private float ClampByTargetRange_(float value)
        {
            return Math.Max(Math.Min(value, OwnerAnmCurve.TargetAttributeMaxValue), OwnerAnmCurve.TargetAttributeMinValue);
        }

        #region スロープ再計算

        /// <summary>
        /// 2つのキーの間のスロープを計算します。
        /// </summary>
        static float GetTangentBetweenTwoKeies_( IAnmKeyFrame kf0, IAnmKeyFrame kf1 )
        {
            float v0 = kf0.ValueAsFloat;
            float v1 = kf1.ValueAsFloat;
            float dt = kf1.Time - kf0.Time;

            if( Math.Abs( dt ) > float.Epsilon )
            {
                return ( v1 - v0 ) / ( dt );
            }
            else
            {
                return 0.0f;
            }
        }

        /// <summary>
        /// 線形補間キーのスロープ再計算を行います。
        /// </summary>
        static void UpdateLinerKeySlope_(
            AnmKeyFrame key,
            IAnmKeyFrame prevKey,
            IAnmKeyFrame nextKey )
        {
            // 線形補間設定をします。
            if( nextKey == null && prevKey == null )
            {
                return;
            }

            // スロープ同期は解除します。
            key.UnifyTangents = false;

            // OutTangent( 右スロープ )
            if( nextKey != null )
            {
                float slope = GetTangentBetweenTwoKeies_( key, nextKey );
                key.OutTangent = slope;
            }

            // InTangent( 左スロープ )
            if( prevKey != null )
            {
                float slope = GetTangentBetweenTwoKeies_( prevKey, key );
                key.InTangent = slope;
            }
        }



        /// <summary>
        /// スプライン補間キーのスロープ再計算を行います。
        /// </summary>
        static void UpdateSplineKeySlope_(
            AnmKeyFrame key,
            IAnmKeyFrame prevKey,
            IAnmKeyFrame nextKey )
        {
            IAnmKeyFrame p0 = ( prevKey != null ) ? prevKey : key;
            IAnmKeyFrame p1 = ( nextKey != null ) ? nextKey : key;

            float tangent = GetTangentBetweenTwoKeies_( p0, p1 );

            key.InTangent = tangent;
            key.OutTangent = tangent;
        }

        /// <summary>
        /// スロープの再計算をします。
        /// 前後のキーのスロープ値の計算も付随して行われます。
        ///
        /// クラス外部から明示的に呼び出してもらいます。
        /// 値変更時、種類変更時、などにクラス内部から暗黙的に呼び出すことはしません。
        /// </summary>
        public void UpdateSlope( bool changeNeighbor )
        {
            IAnmKeyFrame nextKey = OwnerIAnmCurve.GetNextKeyFrame( this );
            IAnmKeyFrame prevKey = OwnerIAnmCurve.GetPrevKeyFrame( this );
            // 両端のスロープを更新するか？
            if( changeNeighbor )
            {
                if( nextKey != null )
                {
                    ( nextKey as AnmKeyFrame ).UpdateSlope( false );
                }
                if( prevKey != null )
                {
                    ( prevKey as AnmKeyFrame ).UpdateSlope( false );
                }
            }

            switch( _interporationType )
            {
                case InterporationType.None:
                case InterporationType.Step:
                case InterporationType.Fixed:
                case InterporationType.Zero:
                {
                    // 何もしません。
                }
                break;
                case InterporationType.Liner:
                {
                    // 線形補間設定をします。
                    UpdateLinerKeySlope_( this, prevKey, nextKey );
                }
                break;
                case InterporationType.Spline:
                {
                    // スプライン設定をします。
                    UpdateSplineKeySlope_( this, prevKey, nextKey );
                }
                break;
            }
        }
        #endregion スロープ再計算

        #region ICloneable メンバ

        /// <summary>
        /// クローンコピーを生成します。
        /// </summary>
        public object Clone()
        {
            return new AnmKeyFrame( this.OwnerAnmCurve, this.Time, this.Value, this.UnifyTangents, this._inTangent, this._outTangent, this.InterporationType, this.ViewScale, false );
        }

        #endregion

        #region IAnmKeyFrame メンバ

        /// <summary>
        /// 選択されているか判定します。
        /// </summary>
        public bool IsSelected
        {
            get { return _anmCurve.IsSelectedKey( this ); }
        }

        /// <summary>
        /// フロート値として、キーの値を取得します。
        /// 主に、アニメーションモジュールから使用されます。
        /// </summary>
        public float ValueAsFloat
        {
            get
            {
                return ClampByTargetRange_(AnmAttribute.GetValueAsFloat( OwnerAnmCurve.TargetAttrType, this._value )) * ViewScale;
            }

            set
            {
                value = ClampByTargetRange_(value / ViewScale);
                object dstVal = this._value;
                AnmAttribute.SetValueAsFloat( OwnerAnmCurve.TargetAttrType, ref dstVal, value );
                this._value = dstVal;
            }
        }

        /// <summary>
        /// フロート値として、加工されていない生のキーの値を取得します。
        /// </summary>
        public float ValueAsFloatNoEffect
        {
            get
            {
                return ClampByTargetRange_(AnmAttribute.GetValueAsFloat(OwnerAnmCurve.TargetAttrType, this._value));
            }
        }


        /// <summary>
        /// In Out Tangentを同期編集するか取得します。
        /// </summary>
        public bool UnifyTangents
        {
            get { return _unifyTangents; }
            set
            {
                if( _unifyTangents != value && TangentModifyEnabled )
                {
                    _unifyTangents = value;
                }
            }
        }

        /// <summary>
        /// スロープが変更可能かどうか取得します。
        /// InterporationType.Zeroは変更可能です。
        /// </summary>
        public bool TangentModifyEnabled
        {
            get { return this.InterporationType != InterporationType.Step; }
        }
        #endregion

    }
}
