﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Drawing.Drawing2D;


namespace LECore.Structures
{
    using Save_Load;
    using SerializableFmt = LECore.Structures.SerializableObject.Lyt;
    using SerializableAnmFmt = LECore.Structures.SerializableObject.Lan;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    /// <summary>
    /// アニメーションデータ配列セット
    /// </summary>
    public class AnimContentsArraySet
    {
        readonly Dictionary<string, Dictionary<ParamaterKind, List<SerializableAnmFmt.AnimContent>>> _animContentsArraySet =
            new Dictionary<string, Dictionary<ParamaterKind, List<SerializableAnmFmt.AnimContent>>>();

        /// <summary>
        /// サブシーン
        /// </summary>
        private ISubScene CurrentSubScene
        {
            get
            {
                return LECore.LayoutEditorCore.Scene.CurrentISubScene;
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimContentsArraySet()
        {
        }

        /// <summary>
        /// 設定します。
        /// </summary>
        public void SetAnimation(IPane pane)
        {
            if (!CurrentSubScene.IsAnimEditSeparateMode())
            {
                foreach (var animationKind in ParamaterKindHelper.AnimationKindSet)
                {
                    this.SetAnimContentArray_(animationKind, RlanHelper.SavePaneAnim(animationKind, pane, AnimCurvesParam.__NoSelected__), AnimCurvesParam.__NoSelected__);
                }
            }
            else
            {
                // カレントタグを切り替えながらすべてのカーブを走査する
                foreach (IAnimFrameSection animFrameSection in CurrentSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet)
                {
                    foreach (var animationKind in ParamaterKindHelper.AnimationKindSet)
                    {
                        this.SetAnimContentArray_(animationKind, RlanHelper.SavePaneAnim(animationKind, pane, animFrameSection.Name), animFrameSection.Name);
                    }
                }

                // テクスチャパターンをコピーする
                SetAnimContentArray_(
                    ParamaterKind.Animation_TexturePattern,
                    RlanHelper.SavePaneAnim(ParamaterKind.Animation_TexturePattern, pane, AnimCurvesParam.__NoSelected__),
                    AnimCurvesParam.__NoSelected__);
            }
        }

        /// <summary>
        /// アニメーションを持っているか取得します。
        /// </summary>
        private bool HasAnimationOf_(ParamaterKind kind, string tag)
        {
            if (_animContentsArraySet.ContainsKey(tag))
            {
                var set = _animContentsArraySet[tag];
                if (set.ContainsKey(kind))
                {
                    return set[kind].Count > 0;
                }
            }

            return false;
        }

        /// <summary>
        /// 追加します。
        /// </summary>
        private void SetAnimContentArray_(ParamaterKind kind, SerializableAnmFmt.AnimContent[] anims, string tag)
        {
            var set = new Dictionary<ParamaterKind, List<SerializableAnmFmt.AnimContent>>();
            if (_animContentsArraySet.ContainsKey(tag))
            {
                set = _animContentsArraySet[tag];
            }

            if (!set.ContainsKey(kind))
            {
                set.Add(kind, new List<SerializableAnmFmt.AnimContent>());
            }

            set[kind].AddRange(anims);
            _animContentsArraySet[tag] = set;
        }

        /// <summary>
        /// アニメーションデータを取得します。
        /// </summary>
        public SerializableAnmFmt.AnimContent[] GetAnimContentArray(ParamaterKind kind, string tag)
        {
            if (this.HasAnimationOf_(kind, tag))
            {
                var set = _animContentsArraySet[tag];
                return set[kind].ToArray();
            }
            else
            {
                return new SerializableAnmFmt.AnimContent[0];
            }
        }
    }

    /// <summary>
    /// ClipBoardPane
    /// クリップボードに保存される際の、ペインの型です。
    /// </summary>
    public class ClipBoardPane
    {
        public struct FontResource
        {
            public string FontName { get; set; }
            public string FontPath { get; set; }
        }

        public struct CaptureTextureResource
        {
            public string Name { get; set; }
            public bool FrameBufferCaptureEnabled { get; set; }
            public bool CaptureOnlyFirstFrame { get; set; }
            public FloatColor ClearColor { get; set; }
            public TexImagePixelFmt Format { get; set; }
            public float TextureScale { get; set; }
            public bool IsHidingFromList { get; set; }
            public CaptureTextureUsage Usage { get; set; }
        }

        readonly IPane[] _paneSet;
        readonly PaneSetDupulicator.Option _option;
        readonly string[] _textureResourcePathSet;
        readonly FontResource[] _fontResourceSet;
        readonly CaptureTextureResource[] _captureTextureResourceSet;

        /// <summary>
        /// オプション設定フラグを取得します。
        /// </summary>
        public bool CheckOption( PaneSetDupulicator.Option option )
        {
            return ( _option & option ) != 0;
        }

        /// <summary>
        /// コピーオプション
        /// </summary>
        public PaneSetDupulicator.Option Option
        {
            get { return _option; }
        }

        /// <summary>
        /// ペイン配列を取得します。
        /// アニメーションデータは
        /// AnimContentsArraySetに分離されて格納されています。
        /// </summary>
        public IPane[] PaneSet { get { return _paneSet; } }

        /// <summary>
        /// テクスチャリソースのパス
        /// </summary>
        public string[] TextureResourcePathSet { get { return _textureResourcePathSet; } }

        /// <summary>
        /// フォーマットマップ
        /// </summary>
        public Dictionary<string, TexImagePixelFmt> TextureFormatMap { get; private set; }

        /// <summary>
        /// フォントリソースのパス
        /// </summary>
        public FontResource[] FontResourcePathSet { get { return _fontResourceSet; } }

        /// <summary>
        /// キャプチャテクスチャリソースの情報
        /// </summary>
        public CaptureTextureResource[] CaptureTextureResourceSet { get { return _captureTextureResourceSet; } }


        /// <summary>
        /// テクスチャリソースのパスを列挙します。
        /// </summary>
        static string[] InitialzeTextureResourcePathSet_( IPane[] paneSet )
        {
            List<string> textureFilePathSet = new List<string>();
            ITextureMgr texMgr = LECore.LayoutEditorCore.Scene.CurrentISubScene.ITextureMgr;

            foreach (IPane pane in paneSet)
            {
                foreach (string texName in pane.EnumrateAllTextureName())
                {
                    ITextureImage texImg = texMgr.FindITextureImageByName(texName);
                    // ダイナミックテクスチャはファイルリソースは持っていないためリストに含めない
                    if (texImg != null &&
                        texImg.SourceType == TextureSourceType.File)
                    {
                        if (!textureFilePathSet.Contains(texImg.FilePath))
                        {
                            textureFilePathSet.Add(texImg.FilePath);
                        }
                    }
                }
            }

            return textureFilePathSet.ToArray();
        }

        /// <summary>
        /// フォントリソースのパスを列挙します。
        /// </summary>
        static FontResource[] InitialzeFontResourcePathSet_( IPane[] paneSet )
        {
            List<FontResource> fontFilePathSet = new List<FontResource>();
            ILEFontManager fontMgr = LECore.LayoutEditorCore.Scene.CurrentISubScene.ILEFontManager;

            foreach( IPane pane in paneSet )
            {
                if( pane.ITextBox != null )
                {
                    ITextBox textBox = pane.ITextBox;
                    ILEFont font = fontMgr.FindFontByName( textBox.FontName );
                    if( font != null )
                    {
                        if (!fontFilePathSet.Exists((fontRes) => fontRes.FontName == font.FontName && fontRes.FontPath == font.FontPath))
                        {
                            FontResource newRest =new FontResource{ FontName = font.FontName, FontPath = font.FontPath };
                            fontFilePathSet.Add(newRest);
                        }
                    }
                }
            }

            return fontFilePathSet.ToArray();
        }

        /// <summary>
        /// キャプチャテクスチャリソースの情報を列挙します。
        /// </summary>
        static CaptureTextureResource[] InitialzeCaptureTextureResourceSet_( IPane[] paneSet )
        {
            List<CaptureTextureResource> captureTextureResourceSet = new List<CaptureTextureResource>();
            ITextureMgr mgr = LECore.LayoutEditorCore.Scene.CurrentISubScene.ITextureMgr;

            foreach( IPane pane in paneSet )
            {
                if (PaneHelper.IsTextureCapturedPane(pane))
                {
                    ITextureImage tex = TextureMgrHelper.FindCaptureTextureByName(mgr, pane.PaneName);
                    ICaptureTexture captureTexture = tex.ICaptureTexture;
                    if (tex != null)
                    {
                        CaptureTextureResource res = new CaptureTextureResource();
                        res.Name = pane.PaneName;
                        res.FrameBufferCaptureEnabled = captureTexture.FrameBufferCaptureEnabled;
                        res.CaptureOnlyFirstFrame = captureTexture.CaptureOnlyFirstFrame;
                        res.ClearColor = captureTexture.ClearColor;
                        res.Format = tex.PixelFmt;
                        res.TextureScale = captureTexture.TextureScale;
                        res.IsHidingFromList = tex.IsHidingFromList;
                        res.Usage = captureTexture.Usage;
                        captureTextureResourceSet.Add(res);
                    }
                }
            }

            return captureTextureResourceSet.ToArray();
        }

        /// <summary>
        /// コンストラクタ
        /// 公開
        /// </summary>
        public ClipBoardPane( IPane[] paneSet, PaneSetDupulicator.Option option )
        {
            _paneSet = paneSet;
            _option = option;

            _textureResourcePathSet = InitialzeTextureResourcePathSet_( paneSet );
            _fontResourceSet = InitialzeFontResourcePathSet_( paneSet );
            _captureTextureResourceSet = InitialzeCaptureTextureResourceSet_(paneSet);

            // フォーマットマップ
            this.TextureFormatMap = new Dictionary<string, TexImagePixelFmt>();
            foreach(var texPath in _textureResourcePathSet)
            {
                if(this.TextureFormatMap.ContainsKey(texPath))
                {
                    Debug.Assert(false);
                    continue;
                }

                ITextureMgr texMgr = paneSet[0].OwnerSubScene.ITextureMgr;
                ITextureImage texImg = texMgr.FindITextureImageByName(Path.GetFileNameWithoutExtension(texPath));
                if (texImg != null)
                {
                    this.TextureFormatMap.Add(texPath, texImg.PixelFmt);
                }
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public ClipBoardPane(
            IPane[] paneSet,
            PaneSetDupulicator.Option option,
            string[] textureResourcePathSet,
            FontResource[] fontResourceSet,
            Dictionary<string, TexImagePixelFmt> fmtMap,
            CaptureTextureResource[] captureTextureResourceSet)
        {
            _paneSet = paneSet;
            _option = option;

            _textureResourcePathSet = textureResourcePathSet;
            _fontResourceSet = fontResourceSet;
            _captureTextureResourceSet = captureTextureResourceSet;
            this.TextureFormatMap = new Dictionary<string, TexImagePixelFmt>(fmtMap);
        }
    }
}
