﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Imaging;
using System.Diagnostics;
using System.IO;


namespace LECore.Structures
{
    using LECoreInterface;
    using LECore.Structures.Core;
    using System.Drawing.Drawing2D;

    /// <summary>
    /// Pane からサイズ情報を取得して Capture されたテクスチャを表すクラスです。
    /// </summary>
    internal class CaptureTexture
        : ICaptureTexture
        , ITextureImage
    {
        #region Field
        /// <summary>
        /// このキャプチャテクスチャのサイズを決定している参照ペインです。
        /// </summary>
        IPane _refPane;

        /// <summary>
        /// テクスチャのカラービットマップデータです。
        /// </summary>
        Bitmap _colorBitmap = null;

        /// <summary>
        /// テクスチャのアルファビットマップデータです。
        /// </summary>
        Bitmap _alphaBitmap = null;

        /// <summary>
        /// テクスチャビットマップを最後に更新した時刻です。この値を元に D3D 側のテクスチャを更新します。
        /// </summary>
        DateTime _lastUpdateTime = DateTime.MinValue;

        /// <summary>
        /// キャプチャテクスチャの使われ方。名前のサフィックスを作成するために使用されます。
        /// </summary>
        CaptureTextureUsage _usage = CaptureTextureUsage.Normal;

        /// <summary>
        /// 最後に参照していたペインの名前です。
        /// 基本的に参照ペインを設定された時に保存された名前で、参照元ペインが存在しなくなったときに参照の復帰に使用されます。
        /// </summary>
        string _lastReferencedPaneName = "";

        /// <summary>
        /// テクスチャのピクセルフォーマットです。
        /// </summary>
        TexImagePixelFmt _format = TexImagePixelFmt.RGBA8;

        /// <summary>
        /// テクスチャのフォーマットが固定されているかどうかのフラグです。
        /// </summary>
        bool _pixelFmtIsFixed = false;

        /// <summary>
        /// テクスチャ一覧に表示するかどうかのフラグです。
        /// </summary>
        bool _isHidingFromList = false;

        /// <summary>
        /// フレームバッファをキャプチャするかどうかの設定です。
        /// </summary>
        bool _frameBufferCaptureEnabled = true;

        /// <summary>
        /// 初回フレームのみテクスチャをキャプチャするかどうかのフラグです。
        /// </summary>
        bool _captureOnlyFirstFrame = false;

        /// <summary>
        /// テクスチャ解像度にかけるスケールです。
        /// </summary>
        float _textureScale = 1.0f;

        /// <summary>
        /// キャプチャテクスチャのクリアカラーです。
        /// </summary>
        FloatColor _clearColor = FloatColor.Black;

        /// <summary>
        /// 前回更新した際のキャプチャ領域です。変更があった場合のみ更新されます。
        /// </summary>
        RectangleF _prevCapturedRect = new RectangleF();
        #endregion

        /// <summary>
        /// コンストラクタ。Usage はデフォルトの Normal が使われます。
        /// </summary>
        /// <param name="pane">参照するペイン</param>
        /// <param name="isHidingFromList">テクスチャ一覧に表示するかどうかのフラグ</param>
        public CaptureTexture(IPane pane, bool isHidingFromList)
        {
            SetSourcePane(pane);
            _isHidingFromList = isHidingFromList;
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="pane">参照するペイン</param>
        /// <param name="isHidingFromList">テクスチャ一覧に表示するかどうかのフラグ</param>
        /// <param name="usage">キャプチャテクスチャの使われ方</param>
        public CaptureTexture(IPane pane, bool isHidingFromList, CaptureTextureUsage usage)
        {
            // SetSourcePane 内で参照されるため先に設定しておく
            _usage = usage;
            SetSourcePane(pane);

            _isHidingFromList = isHidingFromList;
        }


        /// <summary>
        /// 初期サイズをペインからではなく、固定サイズで作成するコンストラクタです。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="width"></param>
        /// <param name="height"></param>
        /// <param name="isHidingFromList"></param>
        public CaptureTexture(IPane pane, int width, int height, bool isHidingFromList)
        {
            SetSourcePane(pane, width, height);
            _isHidingFromList = isHidingFromList;
        }

        /// <summary>
        /// キャプチャペインで設定されるパラメータをコピーします。
        /// </summary>
        /// <param name="src"></param>
        public void CopyCaptureParams(ICaptureTexture src)
        {
            if (src != null)
            {
                _frameBufferCaptureEnabled = src.FrameBufferCaptureEnabled;
                _captureOnlyFirstFrame = src.CaptureOnlyFirstFrame;
                _clearColor = src.ClearColor;
                _textureScale = src.TextureScale;
            }
        }

        /// <summary>
        ///
        /// </summary>
        private void DoRender_(
            IRenderer renderer,
            ITextureImage bgImage,
            float viewportScaleH,
            float viewportScaleV)
        {
            DrawableOption drawableOption = new DrawableOption();
            drawableOption.EnableFlag(
                DrawableOptionFlag.IgnoreBVDrawing |
                DrawableOptionFlag.IgnoreDummyObjectDrawing);

            RectangleF rect = PaneHelper.GetBoundingVolumeInWorldWithPartsHierarchy(_refPane, true);

            int width = (int)Math.Round(rect.Width);
            int height = (int)Math.Round(rect.Height);

            // 描画するペインサイズがレンダーターゲットのサイズよりも大きい場合、veiwportScale に縦横それぞれが収まるスケールが設定されます。
            // 元のペインサイズがぴったり収まるようなビューポートが設定されていると想定して、ビューポートサイズに依存しているパラメータを計算します。
            int viewportWidth = (int)(renderer.ViewportSize.Width / viewportScaleH);
            int viewportHeight = (int)(renderer.ViewportSize.Height / viewportScaleV);

            renderer.SetViewTransform(viewportScaleH, viewportScaleV, 0.0f, 0.0f, -5.0f, 0.0f);

            renderer.Color = ColorHelper.ConvertToLinerGamma(
                                Color.FromArgb(
                                    (int)(ClearColor.A * 255.0f),
                                    (int)(ClearColor.R * 255.0f),
                                    (int)(ClearColor.G * 255.0f),
                                    (int)(ClearColor.B * 255.0f)));
            renderer.FillRectangle(-viewportWidth / 2, -viewportHeight / 2, 0.0f, viewportWidth, viewportHeight);

            // 背景画像が設定されていれば、描画します。
            if (bgImage != null)
            {
                if (FrameBufferCaptureEnabled)
                {
                    // 状態リセット
                    renderer.SetMatrialColorBlend(FloatColor.White, FloatColor.Transparence, false);
                    renderer.SetTextureMtx(0, FVec2.Empty, FVec2.One, 0);

                    int offsetX = bgImage.GDIBitmap.Width / 2 + (int)rect.X;
                    int offsetY = bgImage.GDIBitmap.Height / 2 - height - (int)rect.Y;

                    FVec2 imgSize = new FVec2(bgImage.Size);
                    FVec3 centerPos = new FVec3(-viewportWidth * 0.5f - offsetX, -viewportHeight * 0.5f - offsetY, 0.0f);

                    renderer.PushMtx();
                    {
                        renderer.Scale(1.0f, -1.0f);

                        // デフォルトだとsRGB テクスチャとして扱われないためサンプラの設定する。
                        renderer.SetTextureSamplingState(
                            0,
                            LECore.Structures.Nsrif.Attributes.AttrTexWrap.Clamp,
                            LECore.Structures.Nsrif.Attributes.AttrTexWrap.Clamp,
                            LECore.Structures.Nsrif.Attributes.AttrTexFilterMin.Linear,
                            LECore.Structures.Nsrif.Attributes.AttrTexFilterMag.Linear, true);

                        renderer.DrawImage(bgImage.GDIBitmap, centerPos, imgSize, new TexCoord4[] { new TexCoord4() });
                    }
                    renderer.PopMtx();
                }
            }

            // 親にターゲットのキャプチャペインが存在する場合のみ描画するようにします。
            if (_refPane.PaneKind == PaneKind.Capture)
            {
                SubScene scene = _refPane.OwnerSubScene as SubScene;
                if (scene != null)
                {
                    IPane refPane = _refPane;
                    int currentTime = GlobalTime.Inst.Time;

                    // クローンしたシーンを 0 フレーム目で再評価する
                    if (CaptureOnlyFirstFrame)
                    {
                        refPane = scene.FindPaneByName(_refPane.PaneName);

                        // EvaluateAnimation への置き換えの方がより適切か？
                        {
                            scene.EvaluateMultiAnimation = true; // イベントを抑制する
                            SubSceneHelper.EvaluateAnimationForce(scene, 0);
                            scene.EvaluateMultiAnimation = false;
                        }
                    }

                    RectangleF subSceneLocalRect = PaneHelper.GetBoundingVolumeInWorld(_refPane, true);

                    // 描画するペインサイズがレンダーターゲットのサイズよりも大きい場合はスケールをかけて現在のレンダーターゲットに収まるようにレンダリングします。
                    renderer.SetViewTransform(viewportScaleH, viewportScaleV, viewportWidth / 2 + subSceneLocalRect.X, -viewportHeight / 2 + (int)Math.Round(subSceneLocalRect.Height) + subSceneLocalRect.Y, -5.0f, 0.0f);

                    scene.Draw(renderer, drawableOption,
                        (pane) =>
                        {
                            bool result = false;
                            IHierarchyNode checkPane = pane;
                            while (checkPane != null)
                            {
                                if (checkPane == refPane)
                                {
                                    result = true;
                                    break;
                                }
                                checkPane = checkPane.Parent;
                            }

                            return result;
                        });

                    // 初期フレームキャプチャの場合は元のフレームへ戻す。
                    if (CaptureOnlyFirstFrame)
                    {
                        scene.EvaluateMultiAnimation = true; // イベントを抑制する
                        SubSceneHelper.EvaluateAnimationForce(scene, currentTime);
                        scene.EvaluateMultiAnimation = false;
                    }
                }
            }
        }

        /// <summary>
        /// キャプチャテクスチャの内容を更新します。
        /// </summary>
        /// <param name="renderer"></param>
        /// <param name="bgImage"></param>
        public void RenderCaptureTexture(IRenderer renderer, ITextureImage bgImage)
        {
            if (_refPane != null)
            {
                RectangleF rect = PaneHelper.GetBoundingVolumeInWorldWithPartsHierarchy(_refPane, true);
                int width = (int)Math.Round(rect.Width);
                int height = (int)Math.Round(rect.Height);

                // レンダーターゲットのサイズに収まる最大サイズでキャプチャするように調整する
                // レンダリング対象のペインがレンダーターゲットよりも大きい場合は縦横どちらも収まるようにスケールをかけて調整する。
                float fitScaleH = 1.0f;
                float fitScaleV = 1.0f;

                if (width > renderer.ViewportSize.Width)
                {
                    fitScaleH = renderer.ViewportSize.Width / (float)width;
                    width = (int)renderer.ViewportSize.Width;
                }
                if (height > renderer.ViewportSize.Height)
                {
                    fitScaleV = renderer.ViewportSize.Height / (float)height;
                    height = (int)renderer.ViewportSize.Height;
                }

                DrawerForCapture drawerForCapture = (gc) =>
                {
                    DoRender_(renderer, bgImage, fitScaleH, fitScaleV);
                };

                // キャプチャ時に描画内容を画面に表示しないようにする。
                renderer.NeedPresent = false;
                Bitmap newCaptureImage = renderer.Capture(drawerForCapture, new Rectangle(0, 0, width, height), true);
                renderer.NeedPresent = true;

                if (newCaptureImage != null)
                {
                    // キャプチャ領域に更新があった場合のみ TextureMgrWindow に変更を通知する。
                    bool bNotify = false;
                    if (rect.X != _prevCapturedRect.X ||
                        rect.Y != _prevCapturedRect.Y ||
                        rect.Width != _prevCapturedRect.Width ||
                        rect.Height != _prevCapturedRect.Height)
                    {
                        bNotify = true;
                    }
                    _prevCapturedRect = rect;

                    // レンダラ管理のテクスチャを破棄する
                    renderer.TryRemoveTextureFromBitmap(GDIBitmap, RendererTextrureFormat.ARGB);

                    int scaledWidth = Math.Max((int)(rect.Width * TextureScale), 1);
                    int scaledHeight = Math.Max((int)(rect.Height * TextureScale), 1);
                    Bitmap scaledBitmap = new Bitmap(scaledWidth, scaledHeight);
                    Graphics g = Graphics.FromImage(scaledBitmap);

                    g.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.NearestNeighbor;
                    g.DrawImage(newCaptureImage, 0, 0, scaledWidth, scaledHeight);
                    newCaptureImage.Dispose();
                    g.Dispose();

                    Bitmap alpha = new Bitmap(scaledWidth, scaledHeight);
                    g = Graphics.FromImage(alpha);

                    if (FrameBufferCaptureEnabled)
                    {
                        g.FillRectangle(Brushes.White, 0, 0, scaledWidth, scaledHeight);
                    }
                    else
                    {
                        Brush b = new SolidBrush(Color.FromArgb(255, (int)(ClearColor.A * 255.0f), (int)(ClearColor.A * 255.0f), (int)(ClearColor.A * 255.0f)));
                        g.FillRectangle(b, 0, 0, scaledWidth, scaledHeight);
                        b.Dispose();
                    }

                    UpdateBitmap(scaledBitmap, alpha, bNotify);
                }
            }
        }

        #region ICaptureTexture メンバー

        /// <summary>
        /// このキャプチャテクスチャのもとになっているペイン
        /// </summary>
        public IPane OwnerPane
        {
            get { return _refPane; }
            set
            {
                _refPane = value;

                SetSourcePane(_refPane);
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// 参照しているペイン名を取得します。
        /// </summary>
        public string ReferencedPaneName
        {
            get
            {
                if (_refPane != null)
                {
                    return PaneHelper.GetFullyQualifiedCaptureTextureName(_refPane);
                }
                else
                {
                    return _lastReferencedPaneName;
                }
            }
        }

        /// <summary>
        /// フレームバッファキャプチャを行うかどうか
        /// </summary>
        public bool FrameBufferCaptureEnabled
        {
            get { return _frameBufferCaptureEnabled; }
            set
            {
                _frameBufferCaptureEnabled = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// フレームバッファキャプチャの際に初回だけキャプチャするかどうか
        /// </summary>
        public bool CaptureOnlyFirstFrame
        {
            get { return _captureOnlyFirstFrame; }
            set
            {
                _captureOnlyFirstFrame = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// キャプチャ画像のクリアカラー
        /// </summary>
        public FloatColor ClearColor
        {
            get { return _clearColor; }
            set
            {
                _clearColor = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// レンダーターゲットテクスチャのフォーマット
        /// </summary>
        public TexImagePixelFmt Format
        {
            get { return PixelFmt; }
            set
            {
                PixelFmt = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// テクスチャの解像度のかけるスケール
        /// </summary>
        public float TextureScale
        {
            get { return _textureScale; }
            set
            {
                _textureScale = value;
                NotifyChangeToScene_();
            }
        }

        /// <summary>
        /// テクスチャの使われ方
        /// </summary>
        public CaptureTextureUsage Usage
        {
            get { return _usage; }
            set
            {
                _usage = value;
            }
        }

        #endregion ICaptureTexture メンバ

        #region イベント通知
        /// <summary>
        /// 変更をオーナーペイン経由で通知します。
        /// </summary>
        void NotifyChangeToScene_()
        {
            if (OwnerPane != null)
            {
                var cap = OwnerPane.ICapture as Capture;
                if (cap != null)
                {
                    cap.NotifyModifyEvent(cap, (int)SceneModifyEventArgs.Kind.PaneModify);
                    cap.NotifyModifyEvent(cap, (int)SceneModifyEventArgs.Kind.TextureManager);
                }
            }
        }
        #endregion イベント通知

        #region ITextureImage メンバ

        public TexImagePixelFmt PixelFmt
        {
            get { return _format; }
            set { _format = value; }
        }

        public TextureSourceType SourceType
        {
            get { return TextureSourceType.Dynamic; }
        }

        public string Name
        {
            get
            {
                if (_refPane != null)
                {
                    return PaneHelper.GetFullyQualifiedCaptureTextureName(_refPane) + GetNameSuffix(Usage);
                }
                else
                {
                    return _lastReferencedPaneName + GetNameSuffix(Usage);
                }
            }
        }
        public string FilePath
        {
            get { return string.Empty; }
        }
        public IVec3 Size
        {
            get { return _colorBitmap != null ? new IVec3(_colorBitmap.Width, _colorBitmap.Height, 0) : new IVec3(1, 1, 0); }
        }
        public IVec3 ActualSize
        {
            get { return Size; }
        }
        public Bitmap GDIBitmap
        {
            get { return _colorBitmap; }
        }
        public Bitmap ColorGDIBitmap
        {
            get { return _colorBitmap; }
        }
        public Bitmap AlphaGDIBitmap
        {
            get { return _alphaBitmap; }
        }
        public bool SrcImageHasAlphaBit
        {
            get { return true; }
        }
        public bool SrcImageIsGrayScale
        {
            get { return false; }
        }

        public bool PixelFmtIsFixed { get { return _pixelFmtIsFixed; } internal set { _pixelFmtIsFixed = value; } }
        public bool Valid
        {
            get { return true; }
        }
        public int TextureDataSize
        {
            get { return _colorBitmap != null ? 4 * _colorBitmap.Width * _colorBitmap.Height : 0; }
        }
        public int NativeTextureDataSize
        {
            get { return TextureDataSize; }
        }
        public int NativeTextureDataSizeNX
        {
            get { return TextureDataSize; }
        }
        public DateTime UpdateTime
        {
            get { return _lastUpdateTime; }
        }
        public string LastReferencedPaneName
        {
            get { return _lastReferencedPaneName; }
            set { _lastReferencedPaneName = value; }
        }

        public bool IsHidingFromList
        {
            get { return _isHidingFromList; }
        }

        public ICaptureTexture ICaptureTexture
        {
            get { return this as ICaptureTexture; }
        }


        /// <summary>
        /// 変更時イベント：現在はTexturemMgrクラスへの通知にためにだけ使用されています。
        /// </summary>
        public event EventHandler OnChanged;
        #endregion

        /// <summary>
        /// 変更イベントを通知します。
        /// </summary>
        void NotifyChanged_()
        {
            if (OnChanged != null)
            {
                OnChanged(this, null);
            }
        }

        /// <summary>
        /// このテクスチャのサイズ計算元となるペインを設定し、ビットマップなど内部状態を初期化します。
        /// </summary>
        /// <param name="pane"></param>
        public void SetSourcePane(IPane pane)
        {
            if (pane != null)
            {
                SetSourcePane(pane, (int)Math.Round(pane.Size.X), (int)Math.Round(pane.Size.Y));
            }
            else
            {
                SetSourcePane(pane, 1, 1);
            }
        }

        /// <summary>
        /// このテクスチャのサイズ計算元となるペインを設定し、ビットマップなど内部状態を初期化します。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="width"></param>
        /// <param name="height"></param>
        public void SetSourcePane(IPane pane, int width, int height)
        {
            if (pane != null)
            {
                _refPane = pane;

                // 幅、高さどちらかに 0 が設定された場合ビットマップの作成ができないため、仮のサイズでインスタンスを作成する。
                if (width <= 0 ||
                    height <= 0)
                {
                    width = 1;
                    height = 1;
                }

                UpdateBitmap(new Bitmap(width, height), new Bitmap(width, height), true);
            }
            else
            {
                if (_refPane != null)
                {
                    _lastReferencedPaneName = PaneHelper.GetFullyQualifiedCaptureTextureName(_refPane);
                }
                _refPane = null;
                Bitmap invalidBmp = LEImageResMgr.GetManifestResourcePng("Invalid_Image.png");
                UpdateBitmap(invalidBmp, invalidBmp, true);
            }
        }

        /// <summary>
        /// 保持しているテクスチャビットマップを更新する
        /// </summary>
        public void UpdateBitmap(Bitmap color, Bitmap alpha, bool bNotifyChange)
        {
            if (_colorBitmap != null)
            {
                _colorBitmap.Dispose();
            }

            _colorBitmap = color;

            if (_alphaBitmap != null)
            {
                _alphaBitmap.Dispose();
            }

            _alphaBitmap = alpha;
            _lastUpdateTime = DateTime.Now;

            if (bNotifyChange)
            {
                NotifyChanged_();
            }
        }

        /// <summary>
        /// 名前に付加されるサフィックスを取得します。
        /// </summary>
        /// <param name="usage"></param>
        /// <returns></returns>
        static public string GetNameSuffix(CaptureTextureUsage usage)
        {
            string suffix = "";
            switch (usage)
            {
                case CaptureTextureUsage.Normal:
                    break;
                case CaptureTextureUsage.Mask:
                    suffix = "_M";
                    break;
                case CaptureTextureUsage.DropShadow:
                    suffix = "_D";
                    break;
                default:
                    Debug.Assert(false);
                    break;
            }

            return suffix;
        }
    }
}
