﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using System.Text.RegularExpressions;

namespace LECore.Structures
{
    using LECoreInterface;
    using LECore.Manipulator;
    using LECore.Structures.Core;
    #region IAnimFrameSectionBinarySettings

    /// <summary>
    /// アニメーションフレーム区間バイナリ設定公開インタフェース
    /// </summary>
    public interface IAnimFrameSectionBinarySettings
    {
        bool Active { get; }
        string FileName { get; }
        AnimationLoopType LoopType { get; }
        ParamaterKind TargetAnimKind { get; }
    }

    /// <summary>
    /// アニメーションフレーム区間バイナリ設定
    /// </summary>
    internal class AnimFrameSectionBinarySettings
        : IAnimFrameSectionBinarySettings
    {

        // バイナリ出力するかどうか
        bool _active = true;

        // バイナリファイル名
        string _fileName = string.Empty;

        // ループの種類
        AnimationLoopType _loopType = AnimationLoopType.OneTime;

        // バイナリ出力するファイルの種類
        ParamaterKind _targetAnimKind = ParamaterKind.Animation_All;

        #region IAnimFrameSectionBinarySettings
        public bool Active { get { return _active; } }
        public string FileName { get { return _fileName; } }
        public AnimationLoopType LoopType { get { return _loopType; } }
        public ParamaterKind TargetAnimKind { get { return _targetAnimKind; } }
        #endregion IAnimFrameSectionBinarySettings

        /// <summary>
        /// 設定
        /// </summary>
        public void Set(
            bool active,
            string outBinaryName,
            AnimationLoopType loopType,
            ParamaterKind targetAnimKind)
        {
            _active = active;
            _fileName = outBinaryName;
            _loopType = loopType;
            _targetAnimKind = targetAnimKind;
        }
    }

    #endregion IAnimFrameSectionBinarySettings


    /// <summary>
    /// アニメーションフレーム区間公開インタフェース
    /// </summary>
    public interface IAnimFrameSection
    {
        string Name { get; }
        string Comment { get; }
        int StartFrame { get; }
        int EndFrame { get; }
        int Length { get; }
        ILEGroup[] TargetGroup { get; }
        bool BindAnimationRecursively { get; }
        bool IsReadOnlyLocked { get; }
        IAnimFrameSectionSet Owner { get; }
        IAnimFrameSectionBinarySettings BinarySettings { get; }
        IUserDataHolder UserDataHolder { get; }
    }

    /// <summary>
    /// アニメーションフレーム区間
    /// </summary>
    internal class AnimFrameSection : IAnimFrameSection
    {
        public event OnAnimFrameSectionSetUpdateHandler OnAnimSectionTagChanged;

        string _name;
        string _comment;
        int _startFrame;
        int _endFrame;
        List<ILEGroup> _tagetGroup = new List<ILEGroup>();
        bool _bBindAnimationRecursively = false;
        IAnimFrameSectionSet _owner = null;

        readonly AnimFrameSectionBinarySettings _binarySettings = new AnimFrameSectionBinarySettings();
        readonly UserDataHolder _userDataHolder = new UserDataHolder();

        #region IAnimFrameSection
        public string Name { get { return _name; } }
        public string Comment { get { return _comment; } }
        public int StartFrame { get { return _startFrame; } }
        public int EndFrame { get { return _endFrame; } }
        public int Length { get { return _endFrame - _startFrame; } }
        public ILEGroup[] TargetGroup { get { return _tagetGroup.ToArray(); } }
        public bool BindAnimationRecursively { get { return _bBindAnimationRecursively; } }
        public bool IsReadOnlyLocked { get { return this._owner != null ? _owner.CheckReadOnlyByName(this.Name) : false; } }
        public IAnimFrameSectionSet Owner { get { return this._owner; } }

        public IAnimFrameSectionBinarySettings BinarySettings { get { return _binarySettings; } }
        public IUserDataHolder UserDataHolder { get { return _userDataHolder; } }
        #endregion IAnimFrameSection

        /// <summary>
        /// 初期化、一度だけ実行されます。
        /// </summary>
        public void Initialzie(IAnimFrameSectionSet owner)
        {
            Ensure.Argument.NotNull(owner);
            _owner = owner;
            _userDataHolder.OnChanged += () => NotifyAnimSectionTagChanged();
        }

        /// <summary>
        /// 値の設定
        /// 値の妥当性の検証は、ユーザコードで行ってください。
        /// </summary>
        public void Set(string name, string comment, int startFrame, int endFrame)
        {
            Debug.Assert(startFrame <= endFrame);

            _name = name;
            // バイナリファイル名も同時に変更します。
            _binarySettings.Set(
                _binarySettings.Active,
                name,
                _binarySettings.LoopType,
                _binarySettings.TargetAnimKind);

            _comment = comment;
            _startFrame = startFrame;
            _endFrame = endFrame;
        }

        /// <summary>
        /// 対象グループを設定します。
        /// </summary>
        public void SetTargetGroup(ILEGroup[] group, bool bBindAnimationRecursively)
        {
            _tagetGroup.Clear();
            _tagetGroup.AddRange(group);
            _bBindAnimationRecursively = bBindAnimationRecursively;
        }

        /// <summary>
        /// 変更をマネージャにタグセットに通知します。
        /// </summary>
        public void NotifyAnimSectionTagChanged()
        {
            if (OnAnimSectionTagChanged != null)
            {
                OnAnimSectionTagChanged(SceneModifyEventArgs.Kind.AnimSectionTagSet);
            }
        }
    }

    internal delegate void OnAnimFrameSectionSetUpdateHandler(SceneModifyEventArgs.Kind kind);

    /// <summary>
    /// アニメーションフレーム区間セット
    /// 公開インタフェース
    /// </summary>
    public interface IAnimFrameSectionSet
    {
        IEnumerable<IAnimFrameSection> IAnimFrameSectionSet { get; }
        IAnimFrameSection TargetIAnimFrameSection { get; }

        string GetUniqueFrameSectionName(string origName);


        bool CheckExistsByName(string name);
        bool CheckExistsByOutputBinaryName(string outputBinaryName);
        bool CheckReadOnlyByName(string name);

        bool ModeReadOnly { get; }
    }

    /// <summary>
    /// ヘルパ
    /// </summary>
    public static class AnimFrameSectionHelper
    {
        public static bool ExistTargetByName(this IAnimFrameSection animFrameSection, string name )
        {
            return animFrameSection.TargetGroup.FirstOrDefault((g) => g.GrouprName == name) != null;
        }
    }

    /// <summary>
    /// AnimFrameSectionSetヘルパ
    /// </summary>
    public static class AnimFrameSectionSetHelper
    {
        public const string DefaultFrameSectionName = "Tag_00";

        /// <summary>
        /// バイナリ設定
        /// </summary>
        public static bool IsAnimFrameSectionBinarySettingsSame(
            IAnimFrameSectionBinarySettings bs1,
            IAnimFrameSectionBinarySettings bs2)
        {
            return (bs1.Active == bs2.Active &&
                bs1.FileName == bs2.FileName &&
                bs1.LoopType == bs2.LoopType &&
                bs1.TargetAnimKind == bs2.TargetAnimKind);
        }

        /// <summary>
        /// アニメーションフレーム区間が同一か判定します。
        /// </summary>
        public static bool IsAnimFrameSectionSame(IAnimFrameSection afs1, IAnimFrameSection afs2)
        {
            if (afs1 == null)
            {
                return afs2 == null;
            }

            if (afs2 == null)
            {
                return afs1 == null;
            }

            // 基本的なパラメータについて
            if (afs1.Name != afs2.Name ||
                afs1.Comment != afs2.Comment ||
                afs1.StartFrame != afs2.StartFrame ||
                afs1.BindAnimationRecursively != afs2.BindAnimationRecursively ||
                afs1.EndFrame != afs2.EndFrame)
            {
                return false;
            }

            // TargetGroupについて
            if (afs1.TargetGroup.Length != afs2.TargetGroup.Length)
            {
                return false;
            }
            else
            {
                foreach (ILEGroup g in afs1.TargetGroup)
                {
                    if (Array.IndexOf(afs2.TargetGroup, g) == -1)
                    {
                        return false;
                    }
                }
            }

            // バイナリ出力設定について
            return IsAnimFrameSectionBinarySettingsSame(afs1.BinarySettings, afs2.BinarySettings);
        }

        /// <summary>
        /// アニメーションフレーム区間セットが同一か判定します。
        /// </summary>
        public static bool IsAnimFrameSectionSetSame(IAnimFrameSectionSet afs1, IAnimFrameSectionSet afs2)
        {
            if(afs1.IAnimFrameSectionSet.Count() != afs2.IAnimFrameSectionSet.Count() || afs1.ModeReadOnly != afs2.ModeReadOnly)
            {
                return false;
            }

            for(int i=0;i < afs1.IAnimFrameSectionSet.Count();i++)
            {
                if (!IsAnimFrameSectionSame(afs1.IAnimFrameSectionSet.ElementAt(i), afs2.IAnimFrameSectionSet.ElementAt(i)))
                {
                    return false;
                }
            }

            return true;
        }

        /// <summary>
        /// 名前で、アニメーションフレーム区間を検索します。
        /// </summary>
        static public IAnimFrameSection FindAnimFrameSectionByName(this IAnimFrameSectionSet set, string name)
        {
            return set.IAnimFrameSectionSet.FirstOrDefault((section) => section.Name == name);
        }

        /// <summary>
        /// フレーム範囲が妥当か調査します。
        /// </summary>
        static public bool CheckFrameRangeValid(AnimationLoopType loopType, int startFrame, int endFrame)
        {
            if (loopType == AnimationLoopType.Wiat)
            {
                return startFrame == endFrame;
            }
            else
            {
                return startFrame < endFrame;
            }
        }

        /// <summary>
        /// 名前文字列が妥当かチェックします。
        /// </summary>
        static public bool CheckFrameSectionNameValid(string name)
        {
            // 名前文字列が不正ではないか？
            if (name == null || (name == string.Empty))
            {
                return false;
            }

            if (AppConstants.RegexInvalidCharForObjName.IsMatch(name))
            {
                return false;
            }

            if (AppConstants.MaxAnimFrameSectionTagNameLength < name.Length)
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// フレーム区間の調整
        /// </summary>
        static public void AdjustFrameRange(AnimationLoopType loopType, ref int startFrame, ref int endFrame, bool adjustEnd)
        {
            if (loopType == AnimationLoopType.Wiat)
            {
                if (adjustEnd)
                {
                    endFrame = startFrame;
                }
                else
                {
                    startFrame = endFrame;
                }
            }
            else
            {
                if (endFrame <= startFrame)
                {
                    if (adjustEnd)
                    {
                        endFrame = startFrame + 1;
                    }
                    else
                    {
                        startFrame = endFrame - 1;
                    }
                }
            }
        }

        /// <summary>
        /// 存在しないグループをすべて削除します。
        /// </summary>
        static public bool RemoveInvalidTargetGroupAll(this IAnimFrameSectionSet animTagSet, ILEGroupMgr groupMgr)
        {
            AnimFrameSectionSetManipulator animTagSetMnp = new AnimFrameSectionSetManipulator();
            animTagSetMnp.BindTarget(animTagSet);

            bool isModified = false;
            List<ILEGroup> newGroup = new List<ILEGroup>();
            foreach (IAnimFrameSection section in animTagSet.IAnimFrameSectionSet)
            {
                newGroup.Clear();
                newGroup.AddRange(section.TargetGroup);
                newGroup.RemoveAll((group) => groupMgr.FindGroupByName(group.GrouprName) == null);

                if (section.TargetGroup.Length != newGroup.Count)
                {
                    animTagSetMnp.SetTagTargetGroup(section, newGroup.ToArray(), section.BindAnimationRecursively);
                    isModified = true;
                }
            }

            return isModified;
        }

        /// <summary>
        /// 対象ペインが持つキーから自動設定ボタンハンドラ
        /// </summary>
        static public ParamaterKind SearchOutBinayParamaterKindFromKeys(
            ISubScene subScene, IAnimFrameSection targetFrameSection, bool bOnlyCheckEnableKind = false)
        {
            if (subScene == null || targetFrameSection == null || targetFrameSection.Owner == null)
            {
                return ParamaterKind.None;
            }

            // 全てのアニメーション種類をチェックします。
            ParamaterKind paramFlags = ParamaterKind.None;
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                // 指定されている項目だけ調査する場合
                if (bOnlyCheckEnableKind)
                {
                    if ((targetFrameSection.BinarySettings.TargetAnimKind & kind) == 0)
                    {
                        continue;
                    }
                }

                // タグが対象としているペインについて...
                foreach (IPane pane in EnumerateTargetPanes_(subScene, targetFrameSection))
                {
                    var animAtters = pane.GetAnimationTargetAttributeSet(kind);
                    if (animAtters.Any((attr) => attr.CheckHaveKeyRecursiveInTargetTag(
                        targetFrameSection.StartFrame,
                        targetFrameSection.EndFrame,
                        targetFrameSection.Name,
                        subScene.IsAnimEditSeparateMode())))
                    {
                        paramFlags |= kind;
                        break;
                    }
                }
            }

            return paramFlags;
        }

        /// <summary>
        /// ペインが対象タグ区間にキーを持つパラメータを取得
        /// </summary>
        static public ParamaterKind SearchOutBinayParamaterKindFromKeysInTargetTag(
            ISubScene subScene, IAnimFrameSection targetFrameSection, bool bOnlyCheckEnableKind = false)
        {
            if (subScene == null || targetFrameSection == null || targetFrameSection.Owner == null)
            {
                return ParamaterKind.None;
            }

            // 全てのアニメーション種類をチェックします。
            ParamaterKind paramFlags = ParamaterKind.None;
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                // 指定されている項目だけ調査する場合
                if (bOnlyCheckEnableKind)
                {
                    if ((targetFrameSection.BinarySettings.TargetAnimKind & kind) == 0)
                    {
                        continue;
                    }
                }

                // タグが対象としているペインについて...
                foreach (IPane pane in EnumerateTargetPanes_(subScene, targetFrameSection))
                {
                    var animAtters = pane.GetAnimationTargetAttributeSet(kind);
                    if (animAtters.Any((attr) => attr.CheckHaveKeyRecursiveInTargetTag(
                        targetFrameSection.StartFrame, targetFrameSection.EndFrame, targetFrameSection.Name, subScene.IsAnimEditSeparateMode())))
                    {
                        paramFlags |= kind;
                        break;
                    }
                }
            }

            return paramFlags;
        }

        /// <summary>
        /// キーから自動設定ボタンハンドラ
        /// </summary>
        static public void SetOutBinayParamaterKindFromKeys(
            ISubScene subScene, IAnimFrameSection targetFrameSection)
        {
            if (subScene == null || targetFrameSection == null || targetFrameSection.Owner == null)
            {
                return;
            }

            // 全てのアニメーション種類をチェックします。
            subScene.BeginMassiveModify();

            ParamaterKind paramFlags = SearchOutBinayParamaterKindFromKeys(subScene, targetFrameSection);

            AnimFrameSectionSetManipulator mnp = new AnimFrameSectionSetManipulator();
            mnp.BindTarget(targetFrameSection.Owner);
            mnp.SetOutBinayParamaterKind(targetFrameSection, paramFlags);

            subScene.EndMassiveModify();
        }

        /// <summary>
        /// タグ区間の対象ペインを列挙します。
        /// グループ指定が無い場合は、ペイン全体を列挙します。
        /// </summary>
        static private IEnumerable<IPane> EnumerateTargetPanes_(ISubScene subScene, IAnimFrameSection targetFrameSection)
        {
            if (targetFrameSection == null)
            {
                yield break;
            }

            if (targetFrameSection.TargetGroup.Length <= 0)
            {
                foreach (IPane pane in subScene.IPaneArray)
                {
                    yield return pane;
                }
            }
            else
            {
                foreach (ILEGroup group in targetFrameSection.TargetGroup)
                {
                    foreach (IPane pane in group.Member)
                    {
                        yield return pane;
                    }
                }
            }
        }

        /// <summary>
        /// アニメーションが出力されないアニメーション区間タグ設定があるかどうか
        /// </summary>
        public static bool CheckNotExportAnimationTag(ISubScene subScene, string animTagName)
        {
            IAnimFrameSection tag = FindAnimFrameSectionByName(subScene.IAnimFrameSectionSet, animTagName);
            if (tag == null)
            {
                return false;
            }

            ParamaterKind paramKind = AnimFrameSectionSetHelper.SearchOutBinayParamaterKindFromKeys(subScene, tag, true);
            return (paramKind == ParamaterKind.None);
        }
    }

    /// <summary>
    /// アニメーションフレーム区間セット
    /// </summary>
    internal class AnimFrameSectionSet : IAnimFrameSectionSet
    {
        readonly List<IAnimFrameSection> _animFrameSectionSet = new List<IAnimFrameSection>();
        readonly List<IAnimFrameSection> _readnOnlyAnimFrameSectionSet = new List<IAnimFrameSection>();
        IAnimFrameSection _targetIAnimFrameSection = null;
        // 変更イベント
        internal event OnAnimFrameSectionSetUpdateHandler OnAnimSectionTagChanged;
        OnAnimFrameSectionSetUpdateHandler _animFrameSectionSetUpdateHandler;

        /// <summary>
        /// アニメーションフレーム区間配列を取得します。
        /// </summary>
        public IEnumerable<IAnimFrameSection> IAnimFrameSectionSet
        {
            get { return _animFrameSectionSet; }
        }

        /// <summary>
        /// 操作対象のアニメーションフレーム区間配列を取得します。
        /// </summary>
        public IAnimFrameSection TargetIAnimFrameSection
        {
            get
            {
                return _targetIAnimFrameSection;
            }

            internal set
            {
                if (_targetIAnimFrameSection != value)
                {
                    _targetIAnimFrameSection = value;
                    NotifyAnimSectionTagChanged_(SceneModifyEventArgs.Kind.AnimSectionTagTarget);
                }
            }
        }

        /// <summary>
        /// 分割/結合ボタンの有効無効。
        /// </summary>
        public bool ModeReadOnly { get; set; } = false;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnimFrameSectionSet()
        {
            _animFrameSectionSetUpdateHandler = new OnAnimFrameSectionSetUpdateHandler(NotifyAnimSectionTagChanged_);
        }

        #region 登録、削除
        /// <summary>
        /// 新しいフレーム範囲の登録
        /// </summary>
        public IAnimFrameSection AddNewFrameSection(
            string name,
            string comment,
            int startFrame,
            int endFrame)
        {
            AnimFrameSection newSection = null;

            // 名前で検索して発見されなければ...
            if (!CheckExistsByName(name))
            {
                // 新規作成後、新規登録
                newSection = new AnimFrameSection();
                newSection.Set(name, comment, startFrame, endFrame);

                DoRegister_(newSection);
            }

            return newSection;
        }

        /// <summary>
        /// フレーム範囲の登録
        /// </summary>
        public void AddFrameSection(IAnimFrameSection animFrameSection)
        {
            if (CheckExistsByName(animFrameSection.Name) != false)
            {
                return;
            }

            AnimFrameSection fs = animFrameSection as AnimFrameSection;
            DoRegister_(fs);
        }

        /// <summary>
        /// 実際の登録
        /// </summary>
        private void DoRegister_(AnimFrameSection animFrameSection)
        {
            animFrameSection.Initialzie(this);
            animFrameSection.OnAnimSectionTagChanged += _animFrameSectionSetUpdateHandler;
            _animFrameSectionSet.Add(animFrameSection);

            NotifyAnimSectionTagChanged_();
        }

        /// <summary>
        /// フレーム範囲の削除
        /// </summary>
        public void Remove(IAnimFrameSection animFrameSection)
        {
            AnimFrameSection fs = animFrameSection as AnimFrameSection;
            if (_animFrameSectionSet.Contains(fs))
            {
                SetReadOnlyState(fs, false);
                _animFrameSectionSet.Remove(fs);

                fs.OnAnimSectionTagChanged -= _animFrameSectionSetUpdateHandler;

                NotifyAnimSectionTagChanged_();
            }
        }

        /// <summary>
        /// 読みとり専用状態を設定します。
        /// <remarks>
        /// ユーザのUI 操作によって実行されないメソッドなの
        /// で、UIへの変更通知をしていません。
        /// </remarks>
        /// </summary>
        public void SetReadOnlyState(AnimFrameSection frameSection, bool isReadonly)
        {
            if (!this._animFrameSectionSet.Contains(frameSection))
            {
                return;
            }

            if (isReadonly)
            {
                if (!_readnOnlyAnimFrameSectionSet.Contains(frameSection))
                {
                    _readnOnlyAnimFrameSectionSet.Add(frameSection);
                }
            }
            else
            {
                _readnOnlyAnimFrameSectionSet.Remove(frameSection);
            }
        }

        public bool CheckReadOnlyByName(string name)
        {
            var frameSection = FindFrameSectionByName_(name);
            return frameSection != null ? _readnOnlyAnimFrameSectionSet.Contains(frameSection) : false;
        }

        #endregion 登録、削除

        #region 妥当性のチェック
        /// <summary>
        /// 同名の IAnimFrameSection が存在するか調査します。
        /// </summary>
        public bool CheckExistsByName(string name)
        {
            return _animFrameSectionSet.Exists(
                delegate(IAnimFrameSection af) { return af.Name == name; });
        }

        /// <summary>
        /// 指定したバイナリ出力名を持つ、IAnimFrameSection が存在するか調査します。
        /// </summary>
        public bool CheckExistsByOutputBinaryName(string outputBinaryName)
        {
            return _animFrameSectionSet.Exists(
                delegate(IAnimFrameSection af)
                {
                    return af.BinarySettings.FileName == outputBinaryName;
                });
        }

        /// <summary>
        /// タグの番号を取得します。
        /// </summary>
        public int GetTagIndex(IAnimFrameSection target)
        {
            return _animFrameSectionSet.IndexOf(target);
        }

        /// <summary>
        /// 名前でアニメーション範囲を検索します。
        /// </summary>
        IAnimFrameSection FindFrameSectionByName_(string name)
        {
            return _animFrameSectionSet.Find(
                delegate(IAnimFrameSection af) { return af.Name == name; });
        }

        /// <summary>
        /// ユニークなアニメーション範囲名を取得します。
        /// </summary>
        public string GetUniqueFrameSectionName(string origName)
        {
            return SubSceneHelper.GetValidName(FindFrameSectionByName_, AppConstants.MaxAnimFrameSectionTagNameLength, origName);
        }

        #endregion 妥当性のチェック

        #region 値の設定

        /// <summary>
        /// タグの順番を変更します。
        /// </summary>
        public bool SetTagIndex(IAnimFrameSection target, int newIndex)
        {
            // 存在するか
            if (!_animFrameSectionSet.Contains(target))
            {
                return false;
            }

            // 番号は適切か
            if (newIndex < 0 || newIndex > _animFrameSectionSet.Count)
            {
                return false;
            }

            // 挿入します。
            int oldIndex = _animFrameSectionSet.IndexOf(target);
            _animFrameSectionSet.Insert(newIndex, target);


            // 以前の実態を消去します。
            if (newIndex < oldIndex)
            {
                oldIndex++;
            }
            _animFrameSectionSet.RemoveAt(oldIndex);

            NotifyAnimSectionTagChanged_();
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool SetTagBasic(IAnimFrameSection target, string name, string comment, int start, int end)
        {
            // 存在するか
            if (!_animFrameSectionSet.Contains(target))
            {
                return false;
            }

            // 名前文字列が不正ではないか？
            if (!AnimFrameSectionSetHelper.CheckFrameSectionNameValid(name))
            {
                return false;
            }

            // 名前変更が指定された場合、
            // すでに存在す名前を指定されていないか？
            if (target.Name != name)
            {
                if (CheckExistsByName(name))
                {
                    return false;
                }
            }

            // フレーム範囲
            if (!AnimFrameSectionSetHelper.CheckFrameRangeValid(target.BinarySettings.LoopType, start, end))
            {
                return false;
            }

            AnimFrameSection fs = target as AnimFrameSection;
            fs.Set(name, comment, start, end);

            NotifyAnimSectionTagChanged_();
            return true;
        }


        /// <summary>
        ///
        /// </summary>
        public bool SetOutBinayActive(IAnimFrameSection target, bool outBinaryActive)
        {
            // 存在するか
            if (!_animFrameSectionSet.Contains(target))
            {
                return false;
            }

            AnimFrameSection fs = target as AnimFrameSection;
            AnimFrameSectionBinarySettings binSetting =
                                fs.BinarySettings as AnimFrameSectionBinarySettings;

            if (outBinaryActive == binSetting.Active)
            {
                return true;
            }

            binSetting.Set(
                outBinaryActive,
                binSetting.FileName,
                binSetting.LoopType,
                binSetting.TargetAnimKind);

            NotifyAnimSectionTagChanged_();
            return true;

        }

        /// <summary>
        ///
        /// </summary>
        public bool SetOutBinayFileName(IAnimFrameSection target, string binName)
        {
            // 存在するか
            if (!_animFrameSectionSet.Contains(target))
            {
                return false;
            }

            // 名前文字列が不正ではないか？
            if ((binName == string.Empty) ||
                 AppConstants.RegexInvalidCharForObjName.IsMatch(binName))
            {
                return false;
            }

            // 出力ファイル名
            if ((target.BinarySettings.FileName != binName) &&
                CheckExistsByOutputBinaryName(binName))
            {
                return false;
            }

            AnimFrameSection fs = target as AnimFrameSection;
            AnimFrameSectionBinarySettings binSetting =
                                fs.BinarySettings as AnimFrameSectionBinarySettings;

            if (binName == binSetting.FileName)
            {
                return true;
            }

            binSetting.Set(
                binSetting.Active,
                binName,
                binSetting.LoopType,
                binSetting.TargetAnimKind);

            NotifyAnimSectionTagChanged_();
            return true;
        }

        /// <summary>
        ///
        /// </summary>
        public bool SetOutBinayLoopType(IAnimFrameSection target, AnimationLoopType loopType)
        {
            // 存在するか
            if (!_animFrameSectionSet.Contains(target))
            {
                return false;
            }


            AnimFrameSection fs = target as AnimFrameSection;
            AnimFrameSectionBinarySettings binSetting =
                                fs.BinarySettings as AnimFrameSectionBinarySettings;

            if (loopType == binSetting.LoopType)
            {
                return true;
            }

            binSetting.Set(
                binSetting.Active,
                binSetting.FileName,
                loopType,
                binSetting.TargetAnimKind);

            // アニメーション区間の調整をします。
            int startFrame = target.StartFrame;
            int endFrame = target.EndFrame;
            AnimFrameSectionSetHelper.AdjustFrameRange(target.BinarySettings.LoopType, ref startFrame, ref endFrame, true);
            this.SetTagBasic(target, target.Name, target.Comment, startFrame, endFrame);

            NotifyAnimSectionTagChanged_();
            return true;
        }


        /// <summary>
        ///
        /// </summary>
        public bool SetOutBinayParamaterKind(IAnimFrameSection target, ParamaterKind targetAnimKind)
        {
            // 存在するか
            if (!_animFrameSectionSet.Contains(target))
            {
                return false;
            }


            AnimFrameSection fs = target as AnimFrameSection;
            AnimFrameSectionBinarySettings binSetting =
                                fs.BinarySettings as AnimFrameSectionBinarySettings;

            if (targetAnimKind == binSetting.TargetAnimKind)
            {
                return true;
            }

            binSetting.Set(
                binSetting.Active,
                binSetting.FileName,
                binSetting.LoopType,
                targetAnimKind);

            NotifyAnimSectionTagChanged_();
            return true;
        }

        /// <summary>
        /// タグの対象グループを設定します。
        /// </summary>
        public bool SetTagTargetGroup(
            IAnimFrameSection target,
            ILEGroup[] tagetGroupSet,
            bool bindAnimationRecursively)
        {
            AnimFrameSection fs = target as AnimFrameSection;
            fs.SetTargetGroup(tagetGroupSet, bindAnimationRecursively);

            NotifyAnimSectionTagChanged_();
            return true;
        }

        #endregion 値の設定

        /// <summary>
        /// 変更イベントの通知
        /// </summary>
        void NotifyAnimSectionTagChanged_(SceneModifyEventArgs.Kind kind = SceneModifyEventArgs.Kind.AnimSectionTagSet)
        {
            if (OnAnimSectionTagChanged != null)
            {
                OnAnimSectionTagChanged(kind);
            }
        }
    }
}
