﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.Web;
using System.IO;



namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Rlyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Rlyt;


    /// <summary>
    /// アプリケーション内部のデータ構造クラスを、
    /// XMLシリアライズ可能な、クラスに変換するクラスです。
    /// </summary>
    public class RlytConverter
    {
        SerializableConverterUtil    _serializableConverterUtil = null;

        #region 外部非公開メソッド（その他）
        string _VersionString
        {
            get{ return _serializableConverterUtil.VersionString;}
        }


        #endregion 外部非公開メソッド（その他）

        string                      _rootPath     = string.Empty;
        readonly LEMsgReporter      _msgReporter  = null;

        /// <summary>
        /// 警告表示を行います。
        /// </summary>
        void ReportWarningMsg_( string msg )
        {
            if( _msgReporter != null )
            {
                _msgReporter.ReportWarning( StringResMgr.Get( "LECORE_DLG_TITLE_LOADRLYT" ), msg );
            }
        }

        /// <summary>
        /// ログ出力します。
        /// </summary>
        void OutLog_( string msgID )
        {
            if( _msgReporter != null )
            {
                _msgReporter.OutLogByID( msgID );
            }
        }

        /// <summary>
        /// 相対パスを取得します。
        /// </summary>
        string GetRelativePath_( string path )
        {
            if( _rootPath == string.Empty )
            {
                return path;
            }

            Uri   root       = new Uri( _rootPath );
            Uri   srcUri     = new Uri( root, path );

            // 相対パスを取得します。
            string relativePathEscaped = root.MakeRelativeUri( srcUri ).ToString();

            // Uri クラス が +  エンコードしてくれない文字があるので、自分で置換します。
            // relativePathEscaped = HttpUtility.UrlEncode( relativePathEscaped );
            relativePathEscaped = relativePathEscaped.Replace( "+", "%2b" );

            // エスケープされていない文字列に変換します
            string resultPath = HttpUtility.UrlDecode( relativePathEscaped );
            // Uriクラスによって、置換された\マークをもとに戻します。
            resultPath = resultPath.Replace( '/', '\\' );


            return resultPath;
        }

        /// <summary>
        /// 絶対パスを取得する
        /// </summary>
        string GetAbsolutePath_( string path )
        {
            if( _rootPath == string.Empty )
            {
                return path;
            }

            Uri   root       = new Uri( _rootPath );

            return new Uri( root, path ).LocalPath;
        }




        /// <summary>
        /// コンストラクタ
        /// </summary>
        public RlytConverter( string rootPath, LEMsgReporter msgReporter )
        {
            // 指定された場合は、絶対パスであること。
            Debug.Assert( rootPath == string.Empty || Path.IsPathRooted( rootPath ) );

            _rootPath    = rootPath;
            _msgReporter = msgReporter;
        }

        /// <summary>
        /// シーンクラスを  FileFmt.Document に変換します。
        /// LECore内部クラス => シリアライズ可能オブジェクト への変換です。
        /// </summary>
        /// <returns></returns>
        public FileFmt.Document Convert_SODocument( ISubScene scene )
        {
            _serializableConverterUtil = new SerializableConverterUtil();

            FileFmt.Document  result = new FileFmt.Document();

            // ヘッダ情報関連
            result.head = Convert_head_( null );
            result.body = Convert_SODocumentBody_( scene );

            result.version = _VersionString;

            _serializableConverterUtil = null;

            return result;
        }


        #region クリップボードから利用される・シリアライズオブジェクト変換関数

        #region ペイン
        /// <summary>
        /// ペインをシリアライズ可能なペインクラスに変換します。
        /// </summary>
        internal FileFmt.Pane ConvertPaneToSerializable( AppData.IPane srcPane )
        {
            return Convert_SOPane( srcPane );
        }

        /// <summary>
        /// シリアライズ可能なペインクラスをペインに変換します。
        /// </summary>
        internal AppData.IPane ConvertPaneToInternal( FileFmt.Pane srcPane )
        {
            // AppData.TextureMgr.Inst.StartUpdate();
            AppData.IPane result = Convert_Pane_( srcPane );
            // AppData.TextureMgr.Inst.EndUpdate();

            return result;
        }
        #endregion ペイン

        #region 親子階層

        /// <summary>
        /// シリアライズ可能なペイン階層情報に変換します。
        /// </summary>
        public PaneHierarchy
            ConvertPaneHierarchyToSerializable(
               AppData.IPane pane )
        {
            PaneHierarchy    dstHierarchy = new PaneHierarchy();
            PaneTree         dstPaneTree  = new PaneTree();

            Convert_PaneTree_SubTree_( pane, dstPaneTree );

            dstHierarchy.paneTree = dstPaneTree;

            return dstHierarchy;
        }

        #endregion 親子階層


        #endregion

        #region シリアライズ可能オブジェクト => LECore内部クラス への変換

        #region 基本的な型の変換
        /// <summary>
        /// Vec2 => FVec2
        /// </summary>
        FVec2 Convert_FVec2_( Vec2 srcVec )
        {
            FVec2 result = new FVec2();

            result.X = srcVec.x;
            result.Y = srcVec.y;

            return result;
        }

        /// <summary>
        /// TexVec2 => FVec2
        /// </summary>
        FVec2 Convert_FVec2_( TexVec2 srcVec )
        {
            FVec2 result = new FVec2();

            result.X = srcVec.s;
            result.Y = srcVec.t;

            return result;
        }

        /// <summary>
        /// TexCoord => TexCoord4
        /// </summary>
        TexCoord4 Convert_TexCoord4_( TexCoord srcCoord )
        {
            TexCoord4 dstCoord = new TexCoord4();

            dstCoord.LT = Convert_FVec2_( srcCoord.texLT );
            dstCoord.RT = Convert_FVec2_( srcCoord.texRT );
            dstCoord.LD = Convert_FVec2_( srcCoord.texLB );
            dstCoord.RD = Convert_FVec2_( srcCoord.texRB );

            return dstCoord;
        }

        /// <summary>
        /// FVec3 => Vec2
        /// </summary>
        FVec3 Convert_FVec3_( Vec2 srcVec )
        {
            FVec3 result = new FVec3();

            result.X = (float)srcVec.x;
            result.Y = (float)srcVec.y;
            result.Z = 0.0f;

            return result;
        }

        /// <summary>
        /// Vec3 => FVec3
        /// </summary>
        FVec3  Convert_FVec3_( Vec3 srcVec )
        {
            FVec3 result = new FVec3();

            result.X = (float)srcVec.x;
            result.Y = (float)srcVec.y;
            result.Z = (float)srcVec.z;

            return result;
        }


        /// <summary>
        /// Color4  => RGBAColor
        /// </summary>
        RGBAColor Convert_RGBAColor_( Color4  srcColor )
        {
            RGBAColor result = new RGBAColor();

            result.R = (byte)srcColor.r;
            result.G = (byte)srcColor.g;
            result.B = (byte)srcColor.b;
            result.A = (byte)srcColor.a;

            return result;
        }
        // 注意：
        // スキーマの生成コードが派生クラスの規定値設定をうまく取り扱ってくれないので
        // 別クラスとしています。
        /// <summary>
        /// BlackColor  => RGBAColor
        /// </summary>
        RGBAColor Convert_RGBAColor_( BlackColor  srcColor )
        {
            RGBAColor result = new RGBAColor();

            result.R = (byte)srcColor.r;
            result.G = (byte)srcColor.g;
            result.B = (byte)srcColor.b;
            result.A = (byte)srcColor.a;

            return result;
        }

        /// <summary>
        /// WhiteColor  => RGBAColor
        /// </summary>
        RGBAColor Convert_RGBAColor_( WhiteColor  srcColor )
        {
            RGBAColor result = new RGBAColor();

            result.R = (byte)srcColor.r;
            result.G = (byte)srcColor.g;
            result.B = (byte)srcColor.b;
            result.A = (byte)srcColor.a;

            return result;
        }


        /// <summary>
        /// ColorS10_4 => RGBAColor
        /// </summary>
        RGBAColor Convert_RGBAColor_( ColorS10_4 srcColor )
        {
            RGBAColor result = new RGBAColor();

            result.R = Convert.ToByte( srcColor.r );
            result.G = Convert.ToByte( srcColor.g );
            result.B = Convert.ToByte( srcColor.b );
            result.A = Convert.ToByte( srcColor.a );

            return result;
        }

        /// <summary>
        /// アフィン変換中心(水平)
        /// </summary>
        HorizontalLocation  Convert_HorizontalLocation_( Position position )
        {
            // 水平
            switch( position.x )
            {
                case HorizontalPosition.Left:      return HorizontalLocation.Left;
                case HorizontalPosition.Center:    return HorizontalLocation.Center;
                case HorizontalPosition.Right:     return HorizontalLocation.Right;
                default:Debug.Assert( false );  return HorizontalLocation.Left;
            }
        }

        /// <summary>
        /// アフィン変換中心(垂直)
        /// </summary>
        VerticalLocation  Convert_VerticalLocation_( Position position )
        {
            // 垂直
            switch( position.y )
            {
                case VerticalPosition.Top:       return VerticalLocation.Top;
                case VerticalPosition.Center:    return VerticalLocation.Center;
                case VerticalPosition.Bottom:    return VerticalLocation.Bottom;
                default:Debug.Assert( false );  return VerticalLocation.Top;
            }
        }


        /// <summary>
        /// FileFmt.PaneKind => PaneModify.Kind
        /// </summary>
        AppData.PaneKind Convert_SOPaneKind_( FileFmt.PaneKind kind )
        {
            switch( kind )
            {
                case FileFmt.PaneKind.Null: return AppData.PaneKind.Null;
                case FileFmt.PaneKind.Picture: return AppData.PaneKind.Picture;
                case FileFmt.PaneKind.TextBox: return AppData.PaneKind.Letter;
                default:Debug.Assert( false );return AppData.PaneKind.Null;
            }
        }

        /// <summary>
        /// FileFmt.TexelFormat => AppData.TexImagePixelFmt
        /// </summary>
        TexImagePixelFmt
            Convert_TexelFormat_( FileFmt.TexelFormat srcTexelFmt )
        {
            switch( srcTexelFmt )
            {
                case TexelFormat.I4: return TexImagePixelFmt.I4;
                case TexelFormat.I8: return TexImagePixelFmt.I8;
                case TexelFormat.IA4: return TexImagePixelFmt.IA4;
                case TexelFormat.IA8: return TexImagePixelFmt.IA8;
                case TexelFormat.RGB565: return TexImagePixelFmt.RGB565;
                case TexelFormat.RGB5A3: return TexImagePixelFmt.RGB5A3;
                case TexelFormat.RGBA8: return TexImagePixelFmt.RGBA8;
                case TexelFormat.CMPR: return TexImagePixelFmt.CMPR;
                default: Debug.Assert( false ); return TexImagePixelFmt.RGB565;
            }
        }


        /// <summary>
        /// SByte の 行列インデックスを AttrTexCoordMtx 型を に変換します。
        /// -1が Identity  に変換されます。
        /// </summary>
        AttrTexCoordMtx
            Convert_AttrTexCoordMtx_( SByte matrix )
        {
            if( matrix == -1 )
            {
                return AttrTexCoordMtx.Identity;
            }
            else
            {
                return (AttrTexCoordMtx)matrix;
            }
        }

        /// <summary>
        /// TexGenSrc => LETexGenMethod 変換
        /// </summary>
        LETexGenMethod Convert_LETexGenMethod_( TexGenSrc srcParam )
        {
            switch( srcParam )
            {
                case TexGenSrc.Tex0: return LETexGenMethod.UV0;
                case TexGenSrc.Tex1: return LETexGenMethod.UV1;
                case TexGenSrc.Tex2: return LETexGenMethod.UV2;
                case TexGenSrc.Tex3: return LETexGenMethod.UV3;
                case TexGenSrc.Tex4: return LETexGenMethod.UV4;
                case TexGenSrc.Tex5: return LETexGenMethod.UV5;
                case TexGenSrc.Tex6: return LETexGenMethod.UV6;
                case TexGenSrc.Tex7: return LETexGenMethod.UV7;
                default: Debug.Assert( false ); return LETexGenMethod.UV0;
            }
        }

        #endregion 基本的な型の変換

        #region ペイン関連
        #region マテリアル関連
        /// <summary>
        /// FileFmt.Material => AppData.Material
        /// </summary>
        void Convert_Pane_Material_( AppData.Material dstMat, FileFmt.Material srcMat )
        {
            Debug.Assert( dstMat != null );
            Debug.Assert( srcMat != null );

            dstMat.BlackColor = Convert_RGBAColor_( srcMat.blackColor );
            dstMat.WhiteColor = Convert_RGBAColor_( srcMat.whiteColor );

            // マテリアルがテクスチャを持っている場合はテクスチャを読み込みます。
            // (TextBoxペインは、テクスチャを持たない場合が多いはずです。)
            if( srcMat.texMap != null )
            {
                Debug.Assert( srcMat.texBlendRatio.Length == srcMat.texMap.Length );

                int numTexMap = srcMat.texMap.Length;
                for( int i = 0; i < numTexMap; i++ )
                {
                    MaterialTexMap dstMatTex = dstMat.GetMaterialTexMapByIndex( i );

                    // テクスチャ番号と同一番号のTexCoordGenを使用する仕様になっています。
                    Debug.Assert( srcMat.texMap.Length == srcMat.texCoordGen.Length );
                    FileFmt.TexMap        srcTexMap        = srcMat.texMap[i];
                    FileFmt.TexCoordGen  srcTexCoordGen   = srcMat.texCoordGen[i];


                    // テクスチャリソースの読み込みを行います。
                    // AppData.TextureMgr.Inst.RegisterTextureImageFromFile( srcTexMap.imageName );

                    if( dstMatTex != null )
                    {
                        dstMatTex = dstMat.RegisterMatTexture( srcTexMap.imageName, i );
                    }
                    Debug.Assert( dstMatTex != null );

                    // MaterialTexMap への値の設定
                    // TODO:enum 型については、強制的にキャストしています。
                    dstMatTex.PaletteName   = srcTexMap.paletteName;
                    dstMatTex.MagFilter     = (AttrTexFilterMag)srcTexMap.filterMax;
                    dstMatTex.MinFilter     = (AttrTexFilterMin)srcTexMap.filterMin;
                    dstMatTex.WrapS         = (AttrTexWrap)srcTexMap.wrap_s;
                    dstMatTex.WrapT         = (AttrTexWrap)srcTexMap.wrap_t;
                    dstMatTex.TexGen.Method = Convert_LETexGenMethod_( srcTexCoordGen.srcParam );



                    FileFmt.TexBlendRatio srcBlendRatio = srcMat.texBlendRatio[i];
                    dstMatTex.ColorBlendWeight = srcBlendRatio.color;
                }

                // TexMatrixの初期化
                Debug.Assert( dstMat.TexMtxSet.Length >= srcMat.texMatrix.Length );
                for( int i = 0; i < srcMat.texMatrix.Length; i++ )
                {
                    TexMatrix   srcMtx = srcMat.texMatrix[i];

                    dstMat.TexMtxSet[i].Trans = new FVec2( srcMtx.translate.x, srcMtx.translate.y );
                    dstMat.TexMtxSet[i].Scale = new FVec2( srcMtx.scale.x, srcMtx.scale.y );
                    dstMat.TexMtxSet[i].Rotate = srcMtx.rotate;
                }
            }
        }


        #region ----------- Material_Revo 関連 -----------

        #region 詳細なマテリアル設定

        #region TEV関連列挙型 変換
        /// <summary>
        ///
        /// </summary>
        AttrTevArg
            Convert_AttrTevArg_( FileFmt.TevColorArg src )
        {
            switch( src )
            {
                case TevColorArg.A0: return AttrTevArg.A0;
                case TevColorArg.A1: return AttrTevArg.A1;
                case TevColorArg.A2: return AttrTevArg.A2;
                case TevColorArg.APrev: return AttrTevArg.APrev;
                case TevColorArg.C0: return AttrTevArg.C0;
                case TevColorArg.C1: return AttrTevArg.C1;
                case TevColorArg.C2: return AttrTevArg.C2;
                case TevColorArg.CPrev: return AttrTevArg.CPrev;
                case TevColorArg.V0_5: return AttrTevArg.Half;
                case TevColorArg.Konst: return AttrTevArg.Konst;
                case TevColorArg.V1_0: return AttrTevArg.One;
                case TevColorArg.RasA: return AttrTevArg.RasA;
                case TevColorArg.RasC: return AttrTevArg.RasC;
                case TevColorArg.TexA: return AttrTevArg.TexA;
                case TevColorArg.TexC: return AttrTevArg.TexC;
                case TevColorArg.V0: return AttrTevArg.Zero;
                default:Debug.Assert( false ); return AttrTevArg.Zero;
            }
        }

        /// <summary>
        ///
        /// </summary>
        AttrTevArg
            Convert_TevAlphaArg_( FileFmt.TevAlphaArg src )
        {
            switch( src )
            {
                case TevAlphaArg.A0: return AttrTevArg.A0;
                case TevAlphaArg.A1: return AttrTevArg.A1;
                case TevAlphaArg.A2: return AttrTevArg.A2;
                case TevAlphaArg.APrev: return AttrTevArg.APrev;
                case TevAlphaArg.Konst: return AttrTevArg.Konst;
                case TevAlphaArg.RasA: return AttrTevArg.RasA;
                case TevAlphaArg.TexA: return AttrTevArg.TexA;
                case TevAlphaArg.V0: return AttrTevArg.Zero;
                default:Debug.Assert( false ); return AttrTevArg.Zero;
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="src"></param>
        /// <returns></returns>
        AttrTevBias
            Convert_AttrTevBias_( FileFmt.TevBias src )
        {
            switch( src )
            {
                case TevBias.M0_5: return AttrTevBias.SubHalf;
                case TevBias.P0_5: return AttrTevBias.AddHalf;
                case TevBias.V0: return AttrTevBias.Zero;

                default:Debug.Assert( false ); return AttrTevBias.Zero;
            }
        }

        /// <summary>
        ///
        /// </summary>
        AttrTevScale
            Convert_AttrTevScale_( FileFmt.TevScale src )
        {
            switch( src )
            {
                case TevScale.V1_2: return AttrTevScale.Divide2;
                case TevScale.V1: return AttrTevScale.Scale1;
                case TevScale.V2: return AttrTevScale.Scale2;
                case TevScale.V4: return AttrTevScale.Scale4;
                default:Debug.Assert( false ); return AttrTevScale.Scale1;
            }
        }


        /// <summary>
        ///
        /// </summary>
        AttrTevKSel
            Convert_AttrTevKSel_( FileFmt.TevKColorSel  src )
        {
            switch( src )
            {
                case TevKColorSel.V1_8: return AttrTevKSel.C1_8;
                case TevKColorSel.V2_8: return AttrTevKSel.C2_8;
                case TevKColorSel.V3_8: return AttrTevKSel.C3_8;
                case TevKColorSel.V4_8: return AttrTevKSel.C4_8;
                case TevKColorSel.V5_8: return AttrTevKSel.C5_8;
                case TevKColorSel.V6_8: return AttrTevKSel.C6_8;
                case TevKColorSel.V7_8: return AttrTevKSel.C7_8;
                case TevKColorSel.V8_8: return AttrTevKSel.C8_8;


                case TevKColorSel.K0: return AttrTevKSel.K0;
                case TevKColorSel.K0_a: return AttrTevKSel.K0A;
                case TevKColorSel.K0_b: return AttrTevKSel.K0B;
                case TevKColorSel.K0_g: return AttrTevKSel.K0G;
                case TevKColorSel.K0_r: return AttrTevKSel.K0R;

                case TevKColorSel.K1: return AttrTevKSel.K1;
                case TevKColorSel.K1_a: return AttrTevKSel.K1A;
                case TevKColorSel.K1_b: return AttrTevKSel.K1B;
                case TevKColorSel.K1_g: return AttrTevKSel.K1G;
                case TevKColorSel.K1_r: return AttrTevKSel.K1R;

                case TevKColorSel.K2: return AttrTevKSel.K2;
                case TevKColorSel.K2_a: return AttrTevKSel.K2A;
                case TevKColorSel.K2_b: return AttrTevKSel.K2B;
                case TevKColorSel.K2_g: return AttrTevKSel.K2G;
                case TevKColorSel.K2_r: return AttrTevKSel.K2R;

                case TevKColorSel.K3: return AttrTevKSel.K3;
                case TevKColorSel.K3_a: return AttrTevKSel.K3A;
                case TevKColorSel.K3_b: return AttrTevKSel.K3B;
                case TevKColorSel.K3_g: return AttrTevKSel.K3G;
                case TevKColorSel.K3_r: return AttrTevKSel.K3R;


                default:Debug.Assert( false ); return AttrTevKSel.C1_8;
            }
        }
        /// <summary>
        ///
        /// </summary>
        AttrTevKSel
            Convert_AttrTevKSel_( FileFmt.TevKAlphaSel src )
        {
            switch( src )
            {
                case TevKAlphaSel.V1_8: return AttrTevKSel.C1_8;
                case TevKAlphaSel.V2_8: return AttrTevKSel.C2_8;
                case TevKAlphaSel.V3_8: return AttrTevKSel.C3_8;
                case TevKAlphaSel.V4_8: return AttrTevKSel.C4_8;
                case TevKAlphaSel.V5_8: return AttrTevKSel.C5_8;
                case TevKAlphaSel.V6_8: return AttrTevKSel.C6_8;
                case TevKAlphaSel.V7_8: return AttrTevKSel.C7_8;
                case TevKAlphaSel.V8_8: return AttrTevKSel.C8_8;


                case TevKAlphaSel.K0_a: return AttrTevKSel.K0A;
                case TevKAlphaSel.K0_b: return AttrTevKSel.K0B;
                case TevKAlphaSel.K0_g: return AttrTevKSel.K0G;
                case TevKAlphaSel.K0_r: return AttrTevKSel.K0R;

                case TevKAlphaSel.K1_a: return AttrTevKSel.K1A;
                case TevKAlphaSel.K1_b: return AttrTevKSel.K1B;
                case TevKAlphaSel.K1_g: return AttrTevKSel.K1G;
                case TevKAlphaSel.K1_r: return AttrTevKSel.K1R;


                case TevKAlphaSel.K2_a: return AttrTevKSel.K2A;
                case TevKAlphaSel.K2_b: return AttrTevKSel.K2B;
                case TevKAlphaSel.K2_g: return AttrTevKSel.K2G;
                case TevKAlphaSel.K2_r: return AttrTevKSel.K2R;


                case TevKAlphaSel.K3_a: return AttrTevKSel.K3A;
                case TevKAlphaSel.K3_b: return AttrTevKSel.K3B;
                case TevKAlphaSel.K3_g: return AttrTevKSel.K3G;
                case TevKAlphaSel.K3_r: return AttrTevKSel.K3R;


                default:Debug.Assert( false ); return AttrTevKSel.C1_8;
            }
        }
        /// <summary>
        ///
        /// </summary>
        AttrTevOp
            Convert_AttrTevOp_( FileFmt.TevOpC src )
        {
            switch( src )
            {
                case TevOpC.Add: return AttrTevOp.Add;
                case TevOpC.Comp_bgr24_eq: return AttrTevOp.CompBgr24Eq;
                case TevOpC.Comp_bgr24_gt: return AttrTevOp.CompBgr24Gt;
                case TevOpC.Comp_gr16_eq: return AttrTevOp.CompGr16Eq;
                case TevOpC.Comp_gr16_gt: return AttrTevOp.CompGr16Gt;
                case TevOpC.Comp_r8_eq: return AttrTevOp.CompR8Eq;
                case TevOpC.Comp_r8_gt: return AttrTevOp.CompR8Gt;
                case TevOpC.Comp_rgb8_eq: return AttrTevOp.CompRgb8Eq;
                case TevOpC.Comp_rgb8_gt: return AttrTevOp.CompRgb8Gt;
                case TevOpC.Sub: return AttrTevOp.Sub;

                default:Debug.Assert( false ); return AttrTevOp.CompBgr24Eq;
            }
        }
        /// <summary>
        ///
        /// </summary>
        AttrTevOp
            Convert_AttrTevOp_( FileFmt.TevOpA src )
        {
            switch( src )
            {
                case TevOpA.Add: return AttrTevOp.Add;
                case TevOpA.Comp_a8_eq: return AttrTevOp.CompA8Eq;
                case TevOpA.Comp_a8_gt: return AttrTevOp.CompA8Gt;
                case TevOpA.Comp_bgr24_eq: return AttrTevOp.CompBgr24Eq;
                case TevOpA.Comp_bgr24_gt: return AttrTevOp.CompBgr24Gt;
                case TevOpA.Comp_gr16_eq: return AttrTevOp.CompGr16Eq;
                case TevOpA.Comp_gr16_gt: return AttrTevOp.CompGr16Gt;
                case TevOpA.Comp_r8_eq: return AttrTevOp.CompR8Eq;
                case TevOpA.Comp_r8_gt: return AttrTevOp.CompR8Gt;
                case TevOpA.Sub: return AttrTevOp.Sub;
                default:Debug.Assert( false ); return AttrTevOp.CompBgr24Eq;
            }
        }
        /// <summary>
        ///
        /// </summary>
        AttrTevOutReg
            Convert_AttrTevOutReg_( FileFmt.TevRegID  src )
        {
            switch( src )
            {
                case TevRegID.Prev: return AttrTevOutReg.Prev;
                case TevRegID.Reg0: return AttrTevOutReg.Reg0;
                case TevRegID.Reg1: return AttrTevOutReg.Reg1;
                case TevRegID.Reg2: return AttrTevOutReg.Reg2;
                default:Debug.Assert( false ); return AttrTevOutReg.Prev;
            }
        }

        /// <summary>
        /// FileFmt.BlendFactorDst => AppData.AttrBlendFactor
        /// </summary>
        AttrBlendFactor
            Convert_AttrBlendFactor_( FileFmt.BlendFactorDst srcFactor )
        {
            switch( srcFactor )
            {
                case FileFmt.BlendFactorDst.V1_0: return AttrBlendFactor.One;
                case FileFmt.BlendFactorDst.V0: return AttrBlendFactor.Zero;

                case FileFmt.BlendFactorDst.SrcClr: return AttrBlendFactor.SrcColor;
                case FileFmt.BlendFactorDst.SrcAlpha: return AttrBlendFactor.SrcAlpha;
                case FileFmt.BlendFactorDst.DstAlpha: return AttrBlendFactor.DstAlpha;
                case FileFmt.BlendFactorDst.InvSrcClr: return AttrBlendFactor.InvSrcColor;
                case FileFmt.BlendFactorDst.InvSrcAlpha: return AttrBlendFactor.InvSrcAlpha;
                case FileFmt.BlendFactorDst.InvDstAlpha: return AttrBlendFactor.InvDstAlpha;

                default: Debug.Assert( false ); return AttrBlendFactor.Zero;
            }
        }

        /// <summary>
        /// FileFmt.BlendFactorSrc => AttrBlendFactor.IPEData
        /// </summary>
        AttrBlendFactor
            Convert_AttrBlendFactor_( FileFmt.BlendFactorSrc srcFactor )
        {
            switch( srcFactor )
            {
                case FileFmt.BlendFactorSrc.V1_0: return AttrBlendFactor.One;
                case FileFmt.BlendFactorSrc.V0: return AttrBlendFactor.Zero;
                case FileFmt.BlendFactorSrc.SrcAlpha: return AttrBlendFactor.SrcAlpha;
                case FileFmt.BlendFactorSrc.DstClr: return AttrBlendFactor.DstColor;
                case FileFmt.BlendFactorSrc.DstAlpha: return AttrBlendFactor.DstAlpha;
                case FileFmt.BlendFactorSrc.InvSrcAlpha: return AttrBlendFactor.InvSrcAlpha;
                case FileFmt.BlendFactorSrc.InvDstClr: return AttrBlendFactor.InvDstColor;
                case FileFmt.BlendFactorSrc.InvDstAlpha: return AttrBlendFactor.InvDstAlpha;
                default:  Debug.Assert( false ); return AttrBlendFactor.Zero;
            }
        }

        #endregion TEV関連列挙型 変換

        /// <summary>
        ///
        /// </summary>
        void Convert_ToTevDirectStage(
            AppData.TevDirectStage             dstStage,
            FileFmt.Material_RevoTevStageColor srcStage )
        {
            dstStage.A = Convert_AttrTevArg_( srcStage.a );
            dstStage.B = Convert_AttrTevArg_( srcStage.b );
            dstStage.C = Convert_AttrTevArg_( srcStage.c );
            dstStage.D = Convert_AttrTevArg_( srcStage.d );

            dstStage.Bias = Convert_AttrTevBias_( srcStage.bias );
            dstStage.Clamp = srcStage.clamp;

            dstStage.KSel = Convert_AttrTevKSel_( srcStage.konst );
            dstStage.Op   = Convert_AttrTevOp_( srcStage.op );
            dstStage.Scale = Convert_AttrTevScale_( srcStage.scale );

            dstStage.OutReg = Convert_AttrTevOutReg_( srcStage.outReg );
        }

        /// <summary>
        ///
        /// </summary>
        void Convert_ToTevDirectAlpha(
            AppData.TevDirectStage             dstStage,
            FileFmt.Material_RevoTevStageAlpha srcStage )
        {
            dstStage.A = Convert_TevAlphaArg_( srcStage.a );
            dstStage.B = Convert_TevAlphaArg_( srcStage.b );
            dstStage.C = Convert_TevAlphaArg_( srcStage.c );
            dstStage.D = Convert_TevAlphaArg_( srcStage.d );

            dstStage.Bias = Convert_AttrTevBias_( srcStage.bias );
            dstStage.Clamp = srcStage.clamp;

            dstStage.KSel = Convert_AttrTevKSel_( srcStage.konst );
            dstStage.Op   = Convert_AttrTevOp_( srcStage.op );
            dstStage.Scale = Convert_AttrTevScale_( srcStage.scale );

            dstStage.OutReg = Convert_AttrTevOutReg_( srcStage.outReg );
        }


        /// <summary>
        /// FileFmt.Material_RevoTevStageIndirect => AppData.TevIndirectStage
        /// </summary>
        void Convert_Material_TevIndirectStage_(
            AppData.TevIndirectStage  dstStage,
            FileFmt.Material_RevoTevStageIndirect srcStage )
        {


            dstStage.AddPrev = srcStage.addPrev;
            dstStage.AlphaSel = AttrTevIndAlpha.ItbaOff;// OK ?
            dstStage.BiasSel = (AttrTevIndBias)srcStage.bias;
            dstStage.Format = (AttrTevIndFormat)srcStage.format;
            dstStage.IndStage = srcStage.indStage;
            dstStage.MatrixSel = (AttrTevIndMatrix)srcStage.matrix;
            srcStage.utcLod = dstStage.UtcLod;
            dstStage.WrapS = (AttrTevIndWrap)srcStage.wrap_s;
            dstStage.WrapT = (AttrTevIndWrap)srcStage.wrap_t;
        }

        AttrRasOrder
            Convert_AttrRasOrder_( FileFmt.TevChannelID srcRasOrder )
        {
            switch( srcRasOrder )
            {
                case TevChannelID.ColorNull: return AttrRasOrder.ColorNull;
                case TevChannelID.Color0a0: return AttrRasOrder.Color0A0;
                case TevChannelID.ColorZero: return AttrRasOrder.ColorZero;
                // 以下は想定しない、不正な入力
                default: Debug.Assert( false );return AttrRasOrder.ColorZero;
            }
        }

        /// <summary>
        /// FileFmt.Material_RevoTevStage => AppData.TevStage
        /// </summary>
        void Convert_ToTevStage(
            AppData.TevStage               dstStage,
            FileFmt.Material_RevoTevStage  srcStage )
        {
            Convert_ToTevDirectStage( dstStage.ColorStage, srcStage.color );
            Convert_ToTevDirectAlpha( dstStage.AlphaStage, srcStage.alpha );

            // インダイレクトステージ
            if( srcStage.indirect != null )
            {
                Convert_Material_TevIndirectStage_( dstStage.IndirectStage, srcStage.indirect );
            }


            dstStage.RasOrder = Convert_AttrRasOrder_( srcStage.colorChannel );

            // TexMap, TexCoordGen
            dstStage.TexMapOrder    = srcStage.texMap;
            dstStage.TexCoordOrder  = srcStage.texCoordGen;

            // swap table
            dstStage.RasSwap = srcStage.rasColSwap;
            // swap table
            dstStage.TexMapSwap = srcStage.texColSwap;
        }

        /// <summary>
        /// AppData.ITexIndirectStage => FileFmt.Material_RevoIndirectStage
        /// </summary>
        void Convert_ToIndirectStage(
            AppData.TexIndirectStage            dstIndStage,
            int                                 index,
            FileFmt.Material_RevoIndirectStage  srcStage )
        {
            dstIndStage.Index = index;
            dstIndStage.TexMapOrder = srcStage.texMap;
            dstIndStage.TexCoordOrder = srcStage.texCoordGen;

            dstIndStage.TexCoordScaleS = (AttrIndScale)srcStage.scale_s;
            dstIndStage.TexCoordScaleT = (AttrIndScale)srcStage.scale_t;
        }







        #endregion 詳細なマテリアル設定

        /// <summary>
        /// FileFmt.ColorSource => AttrColorSource
        /// </summary>
        AttrColorSource
            Convert_ToAppColorSource_( FileFmt.ColorSource colSrc )
        {
            return (AttrColorSource)colSrc;
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoAlphaCompare
        /// </summary>

        void Convert_ToAppRevoAlphaCompare_( AppData.PEACompare dstACmp, FileFmt.Material_RevoAlphaCompare srcACmp )
        {
            dstACmp.Comp0 = (AttrCompareFunc)ConvertEnumTypesFromRToL_( dstACmp.Comp0, srcACmp.comp0 );
            dstACmp.Comp1 = (AttrCompareFunc)ConvertEnumTypesFromRToL_( dstACmp.Comp1, srcACmp.comp1 );
            dstACmp.Op    = (AttrAlphaOp)ConvertEnumTypesFromRToL_( dstACmp.Op, srcACmp.op );
            dstACmp.Ref0  = (int)srcACmp.ref0;
            dstACmp.Ref1  = (int)srcACmp.ref1;
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoBlendMode
        /// </summary>
        void Convert_ToFileRevoBlendMode_( AppData.PEBlend dstBlend, FileFmt.Material_RevoBlendMode srcBlend )
        {
            dstBlend.DstFactor = Convert_AttrBlendFactor_( srcBlend.dstFactor );
            dstBlend.LogicOp   = (AttrLogicOp)ConvertEnumTypesFromRToL_( dstBlend.LogicOp, srcBlend.op );
            dstBlend.SrcFactor = Convert_AttrBlendFactor_( srcBlend.srcFactor );
            dstBlend.Type      = (AttrBlendType)ConvertEnumTypesFromRToL_( dstBlend.Type, srcBlend.type );
        }

        /// <summary>
        /// AppData.ITevColorSwapTable => FileFmt.Material_RevoSwapTable
        /// </summary>
        void Convert_ToFileRevoSwapTable_( AppData.TevColorSwapTable dstTbl, FileFmt.Material_RevoSwapTable srcTbl  )
        {
            dstTbl.R = (AttrSwapArg)srcTbl.r;
            dstTbl.G = (AttrSwapArg)srcTbl.g;
            dstTbl.B = (AttrSwapArg)srcTbl.b;
            dstTbl.A = (AttrSwapArg)srcTbl.a;
        }

        /// <summary>
        /// FileFmt.Material_Revo => AppData.RevHWMaterial
        /// </summary>
        void Convert_Pane_RevMaterial_( AppData.RevHWMaterial dstMat, FileFmt.Material_Revo srcMat )
        {
            // ChannelControl 関連
            dstMat.ChannelData.DiffuseCtrlColorSrc =
                Convert_ToAppColorSource_( srcMat.channelControl[0].materialSource );
            dstMat.ChannelData.AlphaCtrlColorSrc   =
                Convert_ToAppColorSource_( srcMat.channelControl[1].materialSource );

            // PE - AlphaCompare
            Convert_ToAppRevoAlphaCompare_( dstMat.PEData.ACompare, srcMat.alphaCompare );
            Convert_ToFileRevoBlendMode_( dstMat.PEData.Blend, srcMat.blendMode );

            dstMat.ChannelData.MatColReg0 = Convert_RGBAColor_( srcMat.matColReg );
            dstMat.MaterialName = srcMat.name;

            // Tev - Colorレジスタ
            for( int i = 0; i < 3; i++ )
            {
                dstMat.TevData.SetTevColors( Convert_RGBAColor_( srcMat.tevColReg[i] ), i );
            }

            // Tev - KColorレジスタ
            for( int i = 0; i < 4; i++ )
            {
                dstMat.TevData.SetTevKColors( Convert_RGBAColor_( srcMat.tevConstReg[i] ), i );
            }

            // Tev - SwapTable
            for( int i = 0; i < 4; i++ )
            {
                Convert_ToFileRevoSwapTable_( dstMat.TevData.SwapTables[i], srcMat.swapTable[i] );
            }

            // Tev - TevStage
            if( srcMat.tevStage != null )
            {
                dstMat.TevData.TevStages.NumStages = srcMat.tevStageNum;
                // 現在の仕様では、必ず等しくなるはずです。
                Debug.Assert( srcMat.tevStage.Length == srcMat.tevStageNum );

                for( int i = 0; i < srcMat.tevStage.Length; i++ )
                {
                    Convert_ToTevStage( dstMat.TevData.TevStages[i], srcMat.tevStage[i] );
                }
            }

            // インダイレクト行列
            if( srcMat.indirectMatrix != null )
            {
                for( int i = 0 ; i < srcMat.indirectMatrix.Length ; i++ )
                {
                    TexMatrix srcIndMtx = srcMat.indirectMatrix[i];
                    TexMtx dstIndMtx = dstMat.TexIndirectMtxSet[i] as TexMtx;

                    dstIndMtx.Trans = Convert_FVec2_( srcIndMtx.translate );
                    dstIndMtx.Scale = Convert_FVec2_( srcIndMtx.scale );
                    dstIndMtx.Rotate = srcIndMtx.rotate;
                }
            }

            // インダイレクトステージ
            if( srcMat.indirectStage != null )
            {
                dstMat.TexIndirectStageSet.NumStageUsed = srcMat.indirectStageNum;
                for( int i = 0; i < srcMat.indirectStage.Length; i++ )
                {
                    Convert_ToIndirectStage(
                        dstMat.TexIndirectStageSet[i] as TexIndirectStage,
                        i,
                        srcMat.indirectStage[i] );
                }
            }




            // Textrue に関する情報
            // 通常のマテリアルのロードで初期化は完了しているはずなので行わない。
        }
        #endregion ----------- Material_Revo 関連 -----------
        #endregion マテリアル関連

        #region 各種ペインに固有なパラメータの読み込み


        /// <summary>
        /// Nullペインを TextBox へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_TextBox_( AppData.Pane dstPane, FileFmt.Pane srcPane )
        {
            FileFmt.TextBox srcTexBoxData = srcPane.Item as  FileFmt.TextBox;
            if( srcTexBoxData != null )
            {

                string contents = SerializableConverterUtil.ReplaceLineFeedCode( srcTexBoxData.text );
                AppData.TextBox textBox = new AppData.TextBox( dstPane, contents, srcTexBoxData.font );

                // AppData.TextBoxのコンストラクタで
                // サイズが自動計算されてしまうので、再度ペインサイズを設定します。
                dstPane.Width = srcPane.size.x;
                dstPane.Height = srcPane.size.y;

                textBox.FontSize =
                    new FVec2( srcTexBoxData.fontSize.x, srcTexBoxData.fontSize.y );

                textBox.CharSpace = srcTexBoxData.charSpace;
                textBox.LineSpace = srcTexBoxData.lineSpace;

                textBox.AllocStringLength = srcTexBoxData.allocateStringLength;
                textBox.SpecifiedStringLength = srcTexBoxData.allocateStringLengthSpecified;

                textBox.BasePosTypeH =
                    Convert_HorizontalLocation_( srcTexBoxData.positionType );
                textBox.BasePosTypeV =
                    Convert_VerticalLocation_( srcTexBoxData.positionType );

                textBox.TopColor = Convert_RGBAColor_( srcTexBoxData.topColor );
                textBox.BottomColor = Convert_RGBAColor_( srcTexBoxData.bottomColor );

                // マテリアルの読み込み
                Convert_Pane_Material_( textBox.Material, srcTexBoxData.material );
                Convert_Pane_RevMaterial_( textBox.RevHWMaterial, srcTexBoxData.materialRevo );
            }
        }

        /// <summary>
        /// テクスチャ座標を読み込みます。
        /// </summary>
        void Convert_Pane_Picture_TexCoords_(
            AppData.Picture  dstPicture,
            FileFmt.Picture  srcPicture )
        {
            int NumTexCoord = srcPicture.texCoord.Length;
            dstPicture.NumTexCoord = NumTexCoord;
            for( int i = 0; i < NumTexCoord; i++ )
            {
                TexCoord  srcCoord = srcPicture.texCoord[i];
                TexCoord4 dstCoord = Convert_TexCoord4_( srcCoord );

                dstPicture.SetTexCoord4( dstCoord, i );
            }

        }

        /// <summary>
        /// Nullペインを Picture へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_Picture_( AppData.Pane dstPane, FileFmt.Pane srcPane )
        {
            AppData.Picture  dstPicture = new AppData.Picture( dstPane );

            FileFmt.Picture  srcPicture = srcPane.Item as FileFmt.Picture;
            Debug.Assert( srcPicture != null );

            dstPicture.VtxCol_LT = Convert_RGBAColor_( srcPicture.vtxColLT );
            dstPicture.VtxCol_RT = Convert_RGBAColor_( srcPicture.vtxColRT );
            dstPicture.VtxCol_LD = Convert_RGBAColor_( srcPicture.vtxColLB );
            dstPicture.VtxCol_RD = Convert_RGBAColor_( srcPicture.vtxColRB );

            // テクスチャ座標情報の読み込み
            Convert_Pane_Picture_TexCoords_( dstPicture, srcPicture );

            // 詳細なマテリアル設定を行うか？
            dstPicture.UseDetailedMaterial = srcPicture.detailSetting;

            // マテリアルの読み込み
            Convert_Pane_Material_( dstPicture.Material, srcPicture.material );
            Convert_Pane_RevMaterial_( dstPicture.RevHWMaterial[0], srcPicture.materialRevo );
        }


        #region Windowペイン固有
        /// <summary>
        /// ウインドウコンテンツ領域情報を読み込みます。
        /// </summary>
        void Convert_Pane_WindowContent_(
            AppData.LEWindow       dstWnd,
            FileFmt.WindowContent  srcContet )
        {
            // 頂点カラー
            dstWnd.SetContentsConerColorByIndex( 0, Convert_RGBAColor_( srcContet.vtxColLT ) );
            dstWnd.SetContentsConerColorByIndex( 1, Convert_RGBAColor_( srcContet.vtxColRT ) );
            dstWnd.SetContentsConerColorByIndex( 2, Convert_RGBAColor_( srcContet.vtxColRB ) );
            dstWnd.SetContentsConerColorByIndex( 3, Convert_RGBAColor_( srcContet.vtxColLB ) );


            // マテリアル情報
            LEWindowPartsID  partsID = LEWindowPartsID.Content;

            // 詳細なマテリアル設定を行うか？
            dstWnd.SetUseDetailedMaterialFlag( partsID, srcContet.detailSetting );

            AppData.Material      dstMtl    = dstWnd.GetMaterial( partsID ) as AppData.Material;
            AppData.RevHWMaterial dstRevMtl = dstWnd.IRevHWMaterial[ dstWnd.PartsIDToInt( partsID )] as RevHWMaterial;

            Convert_Pane_Material_( dstMtl, srcContet.material );
            Convert_Pane_RevMaterial_( dstRevMtl, srcContet.materialRevo );

            // テクスチャ座標
            // ( 注意：テクスチャ情報を含む、マテリアルのロード後に行う必要がある。)
            if( srcContet.texCoord != null )
            {
                // テクスチャ座標数
                dstWnd.NumContentsTexCoord = srcContet.texCoord.Length;
                // テクスチャ座標
                for( int i = 0; i < srcContet.texCoord.Length; i++ )
                {
                    TexCoord4 dstCoord = Convert_TexCoord4_( srcContet.texCoord[i] );
                    dstWnd.SetContentsTexCood4( i, dstCoord );
                }
            }
        }

        /// <summary>
        /// コンテンツ膨張量を読み込みます。
        /// </summary>
        AppData.ILEWindowContentsInflation
            Convert_Pane_Window_InflationRect_( FileFmt.InflationRect srcInflation )
        {
            return new AppData.ILEWindowContentsInflation(
                srcInflation.t,
                srcInflation.b,
                srcInflation.l,
                srcInflation.r );
        }


        /// <summary>
        /// Windowペインのフレーム種類型を変換します。
        /// </summary>
        AppData.LEWindowPartsID
            Convert_Pane_WindowFrameType_( WindowFrameType frameType )
        {
            switch( frameType )
            {
                case WindowFrameType.CornerLT: return AppData.LEWindowPartsID.CornerLT;
                case WindowFrameType.CornerRT: return AppData.LEWindowPartsID.CornerRT;
                case WindowFrameType.CornerLB: return AppData.LEWindowPartsID.CornerLB;
                case WindowFrameType.CornerRB: return AppData.LEWindowPartsID.CornerRB;
                case WindowFrameType.FrameT: return AppData.LEWindowPartsID.FrameT;
                case WindowFrameType.FrameB: return AppData.LEWindowPartsID.FrameB;
                case WindowFrameType.FrameL: return AppData.LEWindowPartsID.FrameL;
                case WindowFrameType.FrameR: return AppData.LEWindowPartsID.FrameR;
                default:Debug.Assert( false );return AppData.LEWindowPartsID.CornerLT;
            }
        }

        /// <summary>
        /// Nullペインを Window へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_Window_( AppData.Pane dstPane, FileFmt.Pane srcPane )
        {
            AppData.LEWindow  dstWnd = new AppData.LEWindow( dstPane );

            FileFmt.Window  srcWnd = srcPane.Item as FileFmt.Window;
            Debug.Assert( srcWnd != null );

            //---------- コンテンツ領域
            Convert_Pane_WindowContent_( dstWnd, srcWnd.content );
            dstWnd.ContentsInflation = Convert_Pane_Window_InflationRect_( srcWnd.contentInflation );


            //---------- フレーム領域
            dstWnd.NumTexture = srcWnd.frame.Length+1;
            for( int i = 0; i < srcWnd.frame.Length; i++ )
            {
                FileFmt.WindowFrame srcFrame = srcWnd.frame[i];
                AppData.LEWindowPartsID partsID = Convert_Pane_WindowFrameType_( srcFrame.frameType );

                // マテリアル情報

                // 詳細なマテリアル設定をおこなうか？
                dstWnd.SetUseDetailedMaterialFlag( partsID,  srcFrame.detailSetting );

                AppData.Material      dstMtl      = dstWnd.GetMaterial( partsID ) as AppData.Material;
                int                   partsIndex  = dstWnd.PartsIDToInt( partsID );
                AppData.RevHWMaterial dstRevMtl   = dstWnd.IRevHWMaterial[ partsIndex ] as AppData.RevHWMaterial;

                Convert_Pane_Material_( dstMtl, srcFrame.material);
                Convert_Pane_RevMaterial_( dstRevMtl, srcFrame.materialRevo );

                // フリップ情報設定
                dstWnd.SetMaterialFlipState( partsID,
                    (LEWindowFrameFlipRotate)srcFrame.textureFlip[0] );
            }

            // マテリアル名を更新します。
            // (過去のバージョンのデータを正しく読み込むために必要です。)
            dstWnd.UpdateMatarialName();
        }
        #endregion Windowペイン固有

        /// <summary>
        /// Nullペインを Window へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_Bounding_( AppData.Pane dstPane, FileFmt.Pane srcPane )
        {
            AppData.Bounding  dstBounding = new AppData.Bounding( dstPane );

            FileFmt.Bounding  srcBounding = srcPane.Item as FileFmt.Bounding;
            Debug.Assert( srcBounding != null );
            // 現在のところ、何もパラメータが存在しません。
        }

        #endregion 各種ペインに固有なパラメータの読み込み

        /// <summary>
        /// シーンからペインを検索して取得します。
        /// ペインが発見されなかったときは、デバックコンソールに警告を表示します。
        /// </summary>
        /// <param name="subScene">シーン</param>
        /// <param name="paneName">ペイン名</param>
        /// <returns>ペイン</returns>
        AppData.IPane SearchPaneFromScene_( ISubScene subScene, string paneName )
        {
            AppData.IPane pane = SubSceneHelper.FindPaneByName( subScene, paneName );

            if( pane == null )
            {
                string errMsg
                    = string.Format( "WARNING:Can't find PaneModify( name = {0}) ", paneName );
                DbgConsole.WriteLine( errMsg );
            }
            return pane;
        }

        /// <summary>
        /// 階層構造を構築します。
        /// 再帰的呼び出しをおこないます。
        /// </summary>
        /// <param name="subScene">シーン</param>
        /// <param name="parentPane">親ペイン</param>
        /// <param name="tree">親ペインの接続情報木</param>
        void BuildHierarchyConnection_
            (
            ISubScene         subScene,
            AppData.IPane     parentPane,
            PaneTree          parentTree
            )
        {
            Debug.Assert( parentPane != null );
            Debug.Assert( parentTree != null );

            // すべての子ツリーに対して...
            if( parentTree.paneTree != null )
            {
                foreach( PaneTree subTree in parentTree.paneTree )
                {
                    // 子供ペインをシーンから検索して、子供ペインを登録します。
                    AppData.IPane childPane = SearchPaneFromScene_( subScene, subTree.name );
                    if( childPane != null )
                    {
                        childPane.DestroyHierarchyConnection();
                        parentPane.AddChildNode( childPane );
                        // サブツリーに対して再帰的に処理をおこないます。
                        BuildHierarchyConnection_( subScene, childPane, subTree );
                    }
                }
            }
        }

        /// <summary>
        /// ユーザ情報文字列を取得します。
        /// ユーザ情報が正式に実装されるまでの、暫定的な実装です。
        /// </summary>
        string HACK_GetUserDataString_( FileFmt.Pane srcPane )
        {
            string result = string.Empty;
            if( srcPane.userData != null && srcPane.userData.Length != 0 )
            {
                // 暫定的な仕様として、唯一の、ユーザデータが、
                // 文字列であると仮定しています。
                UserDataString uds = srcPane.userData[0] as UserDataString;
                if( uds.Value != null )
                {
                    result = uds.Value;
                }
            }
            return result;
        }

        /// <summary>
        /// 基本パラメータのコピー
        /// </summary>
        /// <param name="dstPane"></param>
        /// <param name="srcPane"></param>
        void SetBasicPaneParams_( AppData.Pane dstPane, FileFmt.Pane srcPane )
        {
            dstPane.Size       = Convert_FVec3_( srcPane.size );

            dstPane.Trans = Convert_FVec3_( srcPane.translate );
            dstPane.Scale = Convert_FVec3_( srcPane.scale );
            dstPane.RotAng = Convert_FVec3_( srcPane.rotate );

            dstPane.Transparency = srcPane.alpha;

            dstPane.BasePosTypeH = Convert_HorizontalLocation_( srcPane.basePositionType );
            dstPane.BasePosTypeV = Convert_VerticalLocation_( srcPane.basePositionType );


            dstPane.Locked = srcPane.locked;
            dstPane.Hidden = srcPane.hidden;

            dstPane.UserCommentString = srcPane.comment;
            dstPane.PaneName = srcPane.name;
            dstPane.UserData = HACK_GetUserDataString_( srcPane );

            dstPane.Visible = srcPane.visible;
            dstPane.InfluenceChildrenTransparency = srcPane.influencedAlpha;
            dstPane.LocationAdjust = srcPane.locationAdjust;

        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="srcPane"></param>
        /// <returns></returns>
        AppData.Pane Convert_Pane_( FileFmt.Pane srcPane )
        {
            AppData.Pane result = new AppData.Pane();

            // 標準的なパラメータの設定(Nullペイン)
            SetBasicPaneParams_( result, srcPane );

            // 各種ペインに固有な情報の設定
            switch( srcPane.kind )
            {
                case FileFmt.PaneKind.Null: /* DO NOTHING */ break;
                case FileFmt.PaneKind.TextBox: Convert_Pane_TextBox_( result, srcPane ); break;
                case FileFmt.PaneKind.Picture: Convert_Pane_Picture_( result, srcPane ); break;
                case FileFmt.PaneKind.Window : Convert_Pane_Window_( result, srcPane ); break;
                case FileFmt.PaneKind.Bounding: Convert_Pane_Bounding_( result, srcPane ); break;
                default:Debug.Assert( false );break;
            }

            return result;
        }

        /// <summary>
        /// ペイン情報をシーンに読み込みます。
        /// </summary>
        void Convert_LayoutData_PaneSet_( FileFmt.RLYT rlyt, ISubScene subScene )
        {
            if( rlyt.paneSet != null )
            {
                OutLog_( "LECORE_RLYTLOAD_PANE" );
                FileFmt.Pane[] paneSet   = rlyt.paneSet;
                Group    rlytRootGroup   = rlyt.groupSet.group;
                foreach( FileFmt.Pane srcPane in paneSet )
                {
                    try
                    {
                        // コンバートして、シーン に登録していきます。
                        // 同一名称ペインの登録は行われません。
                        AppData.Pane pane = Convert_Pane_( srcPane );
                        if( SubSceneHelper.FindPaneByName( subScene, pane.PaneName ) == null )
                        {
                            (subScene as SubScene).AddPane( pane );
                        }
                    }
                    catch
                    {
                        // エラーが起こっても処理は続行されます。
                        // "ペイン読み込み中にエラーが検出されました。ペイン [{0}] の読み込みに失敗しました。
                        string msg = StringResMgr.Get( "LECORE_SYS_ERROR_LOAD_PANE", srcPane.name );
                        ReportWarningMsg_( msg );
                    }
                }
            }
        }

        #endregion ペイン関連

        #region 階層・グループ関連
        /// <summary>
        /// 階層構造をシーンに読み込みます。
        /// </summary>
        void Convert_LayoutData_Herarchy_( FileFmt.RLYT rlyt, ISubScene subScene )
        {
            if( rlyt.paneHierarchy != null )
            {
                try
                {
                    OutLog_( "LECORE_RLYTLOAD_HIER" );
                    PaneTree rootTree = rlyt.paneHierarchy.paneTree;
                    Debug.Assert( rootTree != null );
                    AppData.IPane rootPane  = SearchPaneFromScene_( subScene, rootTree.name );
                    Debug.Assert( rootPane != null );

                    BuildHierarchyConnection_( subScene, rootPane, rootTree );
                }
                catch
                {

                    string msg = StringResMgr.Get( "LECORE_SYS_ERROR_LOAD_HIERARCHY" );
                    ReportWarningMsg_( msg );
                }
            }
        }

        /// <summary>
        /// グループの新規作成
        ///
        /// 名前の重複するグループの登録が起こった場合は、
        /// 何も登録を行いません。
        /// </summary>
        LEGroup MakeNewGroup_(
            LEGroupMgr       groupMgr,
            FileFmt.Group    srcGroup,
            IPane[]          groupMemberSet )
        {
            // すでに登録されていなれば...
            LEGroup          newGroup = groupMgr.FindGroupByName( srcGroup.name ) as LEGroup;
            if( newGroup == null )
            {
                // ペインの配列からグループを作成する。
                newGroup = groupMgr.MakeNewGroup( srcGroup.name, groupMemberSet );
                // グループのユーザコメントを設定します。
                newGroup.UserCommentString = srcGroup.comment;
            }
            return newGroup;
        }

        /// <summary>
        /// ペインをメンバーとして、グループをシーンにロードします。
        /// </summary>
        LEGroup Convert_LayoutData_Group_( FileFmt.Group srcGroup, ISubScene subScene )
        {
            Debug.Assert( srcGroup.paneRef != null && srcGroup.group == null );

            LEGroupMgr       groupMgr = (subScene as SubScene).LEGroupMgr;
            LEGroup          newGroup = null;

            if( srcGroup.paneRef != null )
            {
                // ペインをグループ登録します。
                if( srcGroup.paneRef.Length != 0 )
                {
                    // グループのメンバとなるペインの配列を作成する
                    IPane[]      memberPaneSet = new IPane[srcGroup.paneRef.Length];
                    for( int i = 0; i < srcGroup.paneRef.Length; i++ )
                    {
                        GroupPaneRef memberPaneRef = srcGroup.paneRef[i];
                        memberPaneSet[i] =  SearchPaneFromScene_( subScene, memberPaneRef.name );
                        Debug.Assert( memberPaneSet[i] != null );
                    }
                    // 登録する。
                    newGroup = MakeNewGroup_( groupMgr, srcGroup, memberPaneSet );
                }
            }
            return newGroup;
        }

        /// <summary>
        /// グループセットをシーンにロードします。
        /// </summary>
        void Convert_LayoutData_GroupSet_( FileFmt.RLYT rlyt, ISubScene subScene )
        {
            if( rlyt.groupSet != null &&
                rlyt.groupSet.group != null &&
                rlyt.groupSet.group.group != null )
            {
                // 全てのグループについて...
                OutLog_( "LECORE_RLYTLOAD_GROUP" );
                foreach( FileFmt.Group group in rlyt.groupSet.group.group )
                {
                    try
                    {
                        Convert_LayoutData_Group_( group, subScene );
                    }
                    catch
                    {
                        string msg = StringResMgr.Get( "LECORE_SYS_ERROR_LOAD_GROUP", group.name );
                        ReportWarningMsg_( msg );
                    }
                }
            }
        }

        #endregion 階層・グループ関連

        #region テクスチャ・フォントリソース関連
        /// <summary>
        /// すべてのテクスチャを読み込み、マネージャに登録します。
        /// </summary>
        void Convert_TextureFile_( TextureMgr textureMgr, TextureFile[] textureFileSet )
        {
            if( textureFileSet == null )
            {
                return;
            }

            // すべてのテクスチャを読み込み、マネージャに登録します。
            OutLog_( "LECORE_RLYTLOAD_TEX" );
            textureMgr.StartUpdate();
            foreach( TextureFile  textureFile in textureFileSet )
            {
                string textureName = Path.GetFileName( textureFile.imagePath );
                try
                {
                    string completePath = GetAbsolutePath_( textureFile.imagePath );
                    AppData.TextureImage texImg =
                        textureMgr.RegisterTextureImageFromFile( completePath );

                    // テクセルフォーマットを設定。
                    if( !texImg.PixelFmtIsFixed )
                    {
                        texImg.PixelFmt = Convert_TexelFormat_( textureFile.format );
                    }
                }
                catch
                {

                    string msg = StringResMgr.Get( "LECORE_SYS_ERROR_LOAD_TEXTURE", textureName );
                    ReportWarningMsg_( msg );

                    string completePath = GetAbsolutePath_( textureFile.imagePath );
                    // 処理は続行されます。
                }
            }
            textureMgr.EndUpdate();
        }

        /// <summary>
        /// すべてのフォントを読み込み、マネージャに登録します。
        /// </summary>
        void Convert_FontFile_( FontManager fontMgr, FontFile[] fontFileSet )
        {
            if( fontFileSet == null )
            {
                return;
            }

            // すべてのフォントを読み込み、マネージャに登録します。
            OutLog_( "LECORE_RLYTLOAD_FONT" );
            foreach( FontFile  fontFile in fontFileSet )
            {
                string fontName = Path.GetFileName( fontFile.path );
                try
                {
                    string completePath = GetAbsolutePath_( fontFile.path );
                    fontMgr.CreateILEFontFromBRFNT( completePath );
                }
                catch( Exception e )
                {

                    string msg = StringResMgr.Get( "LECORE_SYS_ERROR_LOAD_FONT", fontName );
                    ReportWarningMsg_( msg + "\n" + e.Message );
                    // 処理は続行されます。
                }
            }
        }

        #endregion テクスチャ・フォントリソース関連

        /// <summary>
        /// 背景・スクリーン情報を構築します。
        /// </summary>
        void Convert_ScreenSetting_( FileFmt.ScreenSetting srcSetting, ISubScene subScene )
        {
            try
            {
                // 背景・スクリーン情報を構築します。
                if( srcSetting != null )
                {
                    OutLog_( "LECORE_RLYTLOAD_RLYTSETTINGS" );
                    if( srcSetting.backGround != null )
                    {
                        ScreenSettingBackGround  srcBackGround = srcSetting.backGround;

                        subScene.BackGround.BackGroundColor =
                            Convert_RGBAColor_( srcBackGround.color ).AsColor;

                        if( srcBackGround.imageFile != null && srcBackGround.imageFile != string.Empty )
                        {
                            string absolutePath = GetAbsolutePath_( srcBackGround.imageFile );
                            subScene.BackGround.BackgroundImage =
                                TextureImageFactory.BuildTextureImageFromTga( absolutePath );
                        }
                    }

                    // グリッド関連
                    if( srcSetting.grid != null )
                    {
                        ScreenSettingGrid srcGrid = srcSetting.grid;
                        subScene.BackGround.GridColor = Convert_RGBAColor_( srcGrid.color ).AsColor;
                        subScene.BackGround.GridSize = srcGrid.thickLineInterval;
                        subScene.BackGround.GridVisible = srcGrid.visible;
                        subScene.BackGround.GridSnapEnalbed = srcGrid.moveMethod != GridMoveMethod.None;
                    }

                    // レイアウトサイズ
                    subScene.BackGround.ScreenSize =
                        new FVec2( srcSetting.layoutSize.x, srcSetting.layoutSize.y );
                }
            }
            catch
            {
                string msg = StringResMgr.Get( "LECORE_SYS_ERROR_LOAD_LAYOUTSETTING" );
                ReportWarningMsg_( msg );
            }
        }


        /// <summary>
        /// FileFmt.Document を シーンクラス に変換します。
        /// </summary>
        /// <returns></returns>
        public void Convert_LayoutData( ISubScene subScene, FileFmt.Document soDocument )
        {
            FileFmt.RLYT            rlyt = soDocument.body.rlyt;

            TextureMgr texMgr = subScene.ITextureMgr as TextureMgr;
            FontManager fontMgr = subScene.ILEFontManager as FontManager;

            subScene.BeginMassiveModify();

            // テクスチャをすべて読み込みます。
            Convert_TextureFile_( texMgr, rlyt.textureFile );

            // フォントをすべて読み込みます
            Convert_FontFile_( fontMgr, rlyt.fontFile );

            // ペイン情報をシーンに読み込みます。
            Convert_LayoutData_PaneSet_( rlyt, subScene );

            // 階層構造を構築します。
            Convert_LayoutData_Herarchy_( rlyt, subScene );
            // グループを構築します。
            Convert_LayoutData_GroupSet_( rlyt, subScene );

            // 背景・スクリーン情報を構築します。
            Convert_ScreenSetting_( rlyt.screenSetting, subScene );

            subScene.EndMassiveModify();
        }



        #endregion シリアライズ可能オブジェクト => LECore内部クラス への変換

        #region LECore内部クラス => シリアライズ可能オブジェクト への変換

        #region 外部非公開メソッド(オブジェクト生成関連)

        #region FileFmt.Head ヘッダ情報関連
        /// <summary>
        /// FileFmt.HeadCreate へと変換します。
        /// </summary>
        FileFmt.HeadCreate Convert_headCreate_( object srcData )
        {
            FileFmt.HeadCreate result = new FileFmt.HeadCreate();

            result.date          = DateTime.Now;

            result.host          = Environment.MachineName;
            result.user          = Environment.UserName;

            result.source        = string.Empty;

            return result;
        }

        /// <summary>
        /// FileFmt.HeadGenerator へと変換します。
        /// </summary>
        FileFmt.HeadGenerator Convert_headGenerator_( object srcData )
        {
            FileFmt.HeadGenerator rusult = new FileFmt.HeadGenerator();

            rusult.name     = LayoutEditorCore.AppName;
            rusult.version = LayoutEditorCore.Version;

            return rusult;
        }

        /// <summary>
        /// FileFmt.Head へと変換します。
        /// </summary>
        FileFmt.Head Convert_head_( object srcData )
        {
            FileFmt.Head     result = new FileFmt.Head();

            result.comment   = string.Empty;
            result.title     = string.Empty;
            result.create    = Convert_headCreate_( null );
            result.generator = Convert_headGenerator_( null );

            return result;
        }

        #endregion FileFmt.Head ヘッダ情報関連

        #region FileFmt.DocumentBody 以下

        #region 基本的な型の変換
        /// <summary>
        /// FVec3 => Vec2
        /// </summary>
        Vec2 Convert_Vec2_( FVec3 srcVec )
        {
            Vec2 result = new Vec2();

            result.x = srcVec.X;
            result.y = srcVec.Y;

            return result;
        }

        Vec2 Convert_Vec2_( FVec2 srcVec )
        {
            Vec2 result = new Vec2();

            result.x = srcVec.X;
            result.y = srcVec.Y;

            return result;
        }

        /// <summary>
        /// FVec3 => Vec3
        /// </summary>
        Vec3 Convert_Vec3_( FVec3 srcVec )
        {
            Vec3 result =new Vec3();

            result.x = srcVec.X;
            result.y = srcVec.Y;
            result.z = srcVec.Z;

            return result;
        }

        /// <summary>
        /// FVec2 => TexVec2
        /// </summary>
        TexVec2 Convert_TexVec2_( FVec2 srcVec )
        {
            TexVec2 result =new TexVec2();

            result.s = srcVec.X;
            result.t = srcVec.Y;

            return result;
        }

        /// <summary>
        /// TexCoord4 => TexCoord
        /// </summary>
        TexCoord Convert_TexCoord_( TexCoord4 srcTexCoord4 )
        {
            TexCoord dstTexCoords        = new TexCoord();

            dstTexCoords.texLT = Convert_TexVec2_( srcTexCoord4.LT );
            dstTexCoords.texRT = Convert_TexVec2_( srcTexCoord4.RT );
            dstTexCoords.texLB = Convert_TexVec2_( srcTexCoord4.LD );
            dstTexCoords.texRB = Convert_TexVec2_( srcTexCoord4.RD );

            return dstTexCoords;
        }


        /// <summary>
        /// RGBAColor => Color4
        /// </summary>
        Color4 Convert_Color4_( RGBAColor srcColor )
        {
            Color4 result = new Color4();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;

            return result;
        }

        /// <summary>
        /// RGBAColor => Color4
        /// </summary>
        BlackColor Convert_BlackColor_( RGBAColor srcColor )
        {
            BlackColor result = new BlackColor();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;

            return result;
        }

        /// <summary>
        /// RGBAColor => Color4
        /// </summary>
        WhiteColor Convert_WhiteColor_( RGBAColor srcColor )
        {
            WhiteColor result = new WhiteColor();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;

            return result;
        }

        /// <summary>
        /// RGBAColor => FileFmt.ColorS10_4
        /// </summary>
        ColorS10_4 Convert_ColorS10_4_( RGBAColor srcColor )
        {
            ColorS10_4 result = new ColorS10_4();

            result.r = Convert.ToInt16( srcColor.R );
            result.g = Convert.ToInt16( srcColor.G );
            result.b = Convert.ToInt16( srcColor.B );
            result.a = Convert.ToInt16( srcColor.A );

            return result;
        }

        /// <summary>
        /// HorizontalLocation, VerticalLocation => Position
        /// </summary>
        Position Convert_Position_( HorizontalLocation baseH, VerticalLocation baseV )
        {
            Position result = new Position();

            // 水平
            switch( baseH )
            {
                case HorizontalLocation.Left:
                    result.x = HorizontalPosition.Left;break;
                case HorizontalLocation.Center:
                    result.x = HorizontalPosition.Center;break;
                case HorizontalLocation.Right:
                    result.x = HorizontalPosition.Right;break;
                default:
                    Debug.Assert( false );break;
            }

            // 垂直
            switch( baseV )
            {
                case VerticalLocation.Top:
                    result.y = VerticalPosition.Top;break;
                case VerticalLocation.Center:
                    result.y = VerticalPosition.Center;break;
                case VerticalLocation.Bottom:
                    result.y = VerticalPosition.Bottom;break;
                default:
                    Debug.Assert( false );break;
            }

            return result;
        }

        /// <summary>
        /// AppData.TexImagePixelFmt => FileFmt.TexelFormat
        /// </summary>
        FileFmt.TexelFormat
            Convert_TexImagePixelFmt_( AppData.LECoreInterface.TexImagePixelFmt srcTexelFmt )
        {
            try
            {
                FileFmt.TexelFormat texelFmt =
                    (FileFmt.TexelFormat)Enum.Parse( typeof( FileFmt.TexelFormat ), srcTexelFmt.ToString() );
                return texelFmt;
            }
            catch
            {
                Debug.Assert( false );
                // 変換に失敗したので、適当な値を返します。
                return FileFmt.TexelFormat.RGB565;
            }
        }

        /// <summary>
        /// ScreenCoordinate => FileFmt.ScreenOriginType (座標系種類)
        /// </summary>
        FileFmt.ScreenOriginType
            Convert_ScreenCoordinate_( AppData.LEScreenCoordinate screenCoordinate )
        {
//            if( screenCoordinate.Origin == FVec2.Empty )
//            {
//                return FileFmt.ScreenOriginType.Classic;
//            }
//            else
//            {
//                return FileFmt.ScreenOriginType.Normal;
//            }
            return FileFmt.ScreenOriginType.Normal;
        }

        #endregion 基本的な型の変換

        #region ペイン関連

        #region マテリアル関連

        #region マテリアル(基本)
        /// <summary>
        /// AppData.IMaterial => FileFmt.Material
        /// </summary>
        FileFmt.Material Convert_ToFileMaterial( AppData.IMaterial appMat )
        {
            FileFmt.Material     fileMat = null;
            if( appMat != null )
            {
                fileMat = new FileFmt.Material();

                fileMat.blackColor =  Convert_BlackColor_( appMat.BlackColor );
                fileMat.whiteColor =  Convert_WhiteColor_( appMat.WhiteColor );
                fileMat.name       = appMat.MaterialName;


                IMaterialTexMap[] texMapSet = appMat.IMaterialTexMapSet;
                int               numTexMap = texMapSet.Length;

                fileMat.texMap          = new FileFmt.TexMap[numTexMap];
                fileMat.texBlendRatio   = new FileFmt.TexBlendRatio[numTexMap];
                fileMat.textureStage    = new FileFmt.MaterialTextureStage[numTexMap];

                fileMat.texMatrix       = new FileFmt.TexMatrix[numTexMap];
                fileMat.texCoordGen     = new FileFmt.TexCoordGen[numTexMap];

                // すべてのテクスチャについて....
                for( int i = 0; i < texMapSet.Length; i++ )
                {
                    IMaterialTexMap        srcTex = texMapSet[i];
                    if( srcTex != null )
                    {
                        // ----------- FileFmt.MaterialTextureStage について
                        fileMat.textureStage[i] = Convert_ToFileTexStage_( srcTex );

                        // ----------- FileFmt.TexMap について
                        // enum については、全く同じ内容なので、強制キャストしています。
                        fileMat.texMap[i] = Convert_ToFileTexMap_( srcTex );

                        // ----------- FileFmt.TexBlendRatio について
                        fileMat.texBlendRatio[i]   = new FileFmt.TexBlendRatio();
                        fileMat.texBlendRatio[i].color = srcTex.ColorBlendWeight;

                        // ----------- テクスチャ座標生成関連パラメータについて
                        fileMat.texMatrix[i]    = Convert_ToFileTexMtx_( srcTex.ITexGen.ITexMtx );
                        fileMat.texCoordGen[i]  = Convert_ToFileTexCoordGen_( srcTex.ITexGen );
                    }
                }
            }

            return fileMat;
        }

        /// <summary>
        /// AppData.AttrColorSource => FileFmt.ColorSource
        /// </summary>
        FileFmt.ColorSource Convert_ToFileColorSource_( AttrColorSource appColorSrc )
        {
            return (FileFmt.ColorSource)appColorSrc;
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoAlphaCompare
        /// </summary>
        FileFmt.Material_RevoAlphaCompare
            Convert_ToFileRevoAlphaCompare_( AppData.IPEData appPE )
        {
            FileFmt.Material_RevoAlphaCompare   dstAlphaCompare = new FileFmt.Material_RevoAlphaCompare();

            dstAlphaCompare.comp0 = (FileFmt.Compare)ConvertEnumTypesFromRToL_( dstAlphaCompare.comp0, appPE.ACompare.Comp0 );
            dstAlphaCompare.comp1 = (FileFmt.Compare)ConvertEnumTypesFromRToL_( dstAlphaCompare.comp1, appPE.ACompare.Comp1 );
            dstAlphaCompare.op    = (FileFmt.AlphaOp)ConvertEnumTypesFromRToL_( dstAlphaCompare.op, appPE.ACompare.Op );
            dstAlphaCompare.ref0  = (byte)appPE.ACompare.Ref0;
            dstAlphaCompare.ref1  = (byte)appPE.ACompare.Ref1;
            return dstAlphaCompare;
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoBlendMode
        /// </summary>
        FileFmt.Material_RevoBlendMode
            Convert_ToFileRevoBlendMode_( AppData.IPEData appPE )
        {
            FileFmt.Material_RevoBlendMode   dstBlendMode = new FileFmt.Material_RevoBlendMode();

            dstBlendMode.dstFactor  = (FileFmt.BlendFactorDst)Convert_ToBlendFactorDst_( appPE.Blend.DstFactor );
            dstBlendMode.op         = (FileFmt.LogicOp)ConvertEnumTypesFromRToL_( dstBlendMode.op, appPE.Blend.LogicOp );
            dstBlendMode.srcFactor  = (FileFmt.BlendFactorSrc)Convert_ToBlendFactorSrc_( appPE.Blend.SrcFactor );
            dstBlendMode.type       = (FileFmt.BlendMode)ConvertEnumTypesFromRToL_( dstBlendMode.type, appPE.Blend.Type );

            return dstBlendMode;
        }

        /// <summary>
        /// AppData.ITevColorSwapTable => FileFmt.Material_RevoSwapTable
        /// </summary>
        FileFmt.Material_RevoSwapTable
            Convert_ToFileRevoSwapTable_( AppData.ITevColorSwapTable appTbl )
        {
            FileFmt.Material_RevoSwapTable dstTbl = new Material_RevoSwapTable();

            dstTbl.r = (TevColorChannel)appTbl.R;
            dstTbl.g = (TevColorChannel)appTbl.G;
            dstTbl.b = (TevColorChannel)appTbl.B;
            dstTbl.a = (TevColorChannel)appTbl.A;

            return dstTbl;
        }

        /// <summary>
        /// AppData.IMaterialTexMap => FileFmt.TexCoord
        /// </summary>
        FileFmt.TexCoord
            Convert_ToFileTexCoord_( AppData.TexCoord4 appTexCoord )
        {
            FileFmt.TexCoord dstTexCoord = new FileFmt.TexCoord();

            dstTexCoord.texLT = Convert_TexVec2_( appTexCoord.LT );
            dstTexCoord.texRT = Convert_TexVec2_( appTexCoord.RT );
            dstTexCoord.texLB = Convert_TexVec2_( appTexCoord.LD );
            dstTexCoord.texRB = Convert_TexVec2_( appTexCoord.RD );

            return dstTexCoord;
        }

        /// <summary>
        /// AppData.IMaterialTexMap => FileFmt.TexMap
        /// </summary>
        FileFmt.MaterialTextureStage
            Convert_ToFileTexStage_( AppData.IMaterialTexMap appTex )
        {
            FileFmt.MaterialTextureStage dstTexStage = new MaterialTextureStage();

            dstTexStage.texCoordGen = (SByte)appTex.SlotIdx;
            dstTexStage.texMap      = (SByte)appTex.SlotIdx;

            return dstTexStage;
        }

        /// <summary>
        /// AppData.IMaterialTexMap => FileFmt.TexMap
        /// </summary>
        FileFmt.TexMap
            Convert_ToFileTexMap_( AppData.IMaterialTexMap appTex )
        {
            FileFmt.TexMap dstTexMap = new FileFmt.TexMap();

            dstTexMap.wrap_s = (FileFmt.TexWrapMode)appTex.WrapS;
            dstTexMap.wrap_t = (FileFmt.TexWrapMode)appTex.WrapT;
            dstTexMap.imageName   = appTex.TexImgName;
            dstTexMap.paletteName = null; // TODO:
            dstTexMap.filterMin = (TexFilter)appTex.MinFilter;
            dstTexMap.filterMax = (TexFilter)appTex.MagFilter;

            return dstTexMap;
        }

        /// <summary>
        /// AttrTexCoordMtx 型を SByte に変換します。
        /// Identity が -1 に変換されます。
        /// </summary>
        SByte
            Convert_ToFileTexCoordGenMtxID_( AttrTexCoordMtx mtxID )
        {
            if( mtxID == AttrTexCoordMtx.Identity )
            {
                return -1;
            }
            else
            {
                return (SByte)mtxID;
            }
        }

        /// <summary>
        /// AppData.ITexGen => FileFmt.TexCoordGen
        /// FIXME:enum 型等は 強引にキャストしています。
        /// </summary>
        FileFmt.TexCoordGen
            Convert_ToFileTexCoordGen_( AppData.ITexGen appTexGen )
        {

            FileFmt.TexCoordGen dstGen = new TexCoordGen();

            dstGen.func     = TexGenType.Mtx2x4;
            dstGen.srcParam = (FileFmt.TexGenSrc)appTexGen.Method;
            dstGen.matrix   = Convert_ToFileTexCoordGenMtxID_( appTexGen.MtxID );

            return dstGen;
        }

        /// <summary>
        /// AppData.ITexMtx => FileFmt.TexCoordGen
        /// FIXME:enum 型等は 強引にキャストしています。
        /// </summary>
        FileFmt.TexMatrix
            Convert_ToFileTexMtx_( AppData.ITexMtx appMtx )
        {

            FileFmt.TexMatrix dstMtx = new TexMatrix();

            dstMtx.translate = Convert_Vec2_( appMtx.Trans );
            dstMtx.scale     = Convert_Vec2_( appMtx.Scale );
            dstMtx.rotate    = appMtx.Rotate;

            return dstMtx;
        }


        #endregion マテリアル(基本)

        #region 詳細なマテリアル設定


        #region TEV関連列挙型 変換

        /// <summary>
        /// AppData.AttrRasOrder => FileFmt.TevChannelID
        /// </summary>
        FileFmt.TevChannelID
            Convert_TevChannelID_( AttrRasOrder srcRasOrder )
        {
            switch( srcRasOrder )
            {
                case AttrRasOrder.ColorNull: return TevChannelID.ColorNull;
                case AttrRasOrder.Color0A0:  return TevChannelID.Color0a0;
                case AttrRasOrder.ColorZero: return TevChannelID.ColorZero;
                    // 以下は想定しない、不正な入力
                default: Debug.Assert( false );return TevChannelID.ColorNull;
            }
        }

        /// <summary>
        /// AppData.AttrBlendFactor => FileFmt.BlendFactorDst
        /// </summary>
        FileFmt.BlendFactorDst
            Convert_ToBlendFactorDst_( AttrBlendFactor srcFactor )
        {
            switch( srcFactor )
            {
                case AttrBlendFactor.One: return FileFmt.BlendFactorDst.V1_0;
                case AttrBlendFactor.Zero: return FileFmt.BlendFactorDst.V0;

                case AttrBlendFactor.SrcColor: return FileFmt.BlendFactorDst.SrcClr;
                case AttrBlendFactor.SrcAlpha: return FileFmt.BlendFactorDst.SrcAlpha;
                case AttrBlendFactor.DstAlpha: return FileFmt.BlendFactorDst.DstAlpha;
                case AttrBlendFactor.InvSrcColor: return FileFmt.BlendFactorDst.InvSrcClr;
                case AttrBlendFactor.InvSrcAlpha: return FileFmt.BlendFactorDst.InvSrcAlpha;
                case AttrBlendFactor.InvDstAlpha: return FileFmt.BlendFactorDst.InvDstAlpha;

                default: Debug.Assert( false ); return FileFmt.BlendFactorDst.V0;
            }
        }

        /// <summary>
        /// AttrBlendFactor.IPEData => FileFmt.BlendFactorSrc
        /// </summary>
        FileFmt.BlendFactorSrc
            Convert_ToBlendFactorSrc_( AttrBlendFactor srcFactor )
        {
            switch( srcFactor )
            {
                case AttrBlendFactor.One: return FileFmt.BlendFactorSrc.V1_0;
                case AttrBlendFactor.Zero: return FileFmt.BlendFactorSrc.V0;
                case AttrBlendFactor.SrcAlpha: return FileFmt.BlendFactorSrc.SrcAlpha;
                case AttrBlendFactor.DstColor: return FileFmt.BlendFactorSrc.DstClr;
                case AttrBlendFactor.DstAlpha: return FileFmt.BlendFactorSrc.DstAlpha;
                case AttrBlendFactor.InvSrcAlpha: return FileFmt.BlendFactorSrc.InvSrcAlpha;
                case AttrBlendFactor.InvDstColor: return FileFmt.BlendFactorSrc.InvDstClr;
                case AttrBlendFactor.InvDstAlpha: return FileFmt.BlendFactorSrc.InvDstAlpha;
                default:  Debug.Assert( false ); return FileFmt.BlendFactorSrc.V0;
            }
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.TevColorArg
            Convert_TevColorArg_( AttrTevArg src )
        {
            switch( src )
            {
                case AttrTevArg.A0: return TevColorArg.A0;
                case AttrTevArg.A1: return TevColorArg.A1;
                case AttrTevArg.A2: return TevColorArg.A2;
                case AttrTevArg.APrev: return TevColorArg.APrev;
                case AttrTevArg.C0: return TevColorArg.C0;
                case AttrTevArg.C1: return TevColorArg.C1;
                case AttrTevArg.C2: return TevColorArg.C2;
                case AttrTevArg.CPrev: return TevColorArg.CPrev;
                case AttrTevArg.Half: return TevColorArg.V0_5;
                case AttrTevArg.Konst: return TevColorArg.Konst;
                case AttrTevArg.One: return TevColorArg.V1_0;
                case AttrTevArg.RasA: return TevColorArg.RasA;
                case AttrTevArg.RasC: return TevColorArg.RasC;
                case AttrTevArg.TexA: return TevColorArg.TexA;
                case AttrTevArg.TexC: return TevColorArg.TexC;
                case AttrTevArg.Zero: return TevColorArg.V0;
                default:Debug.Assert( false ); return TevColorArg.C0;
            }
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.TevAlphaArg
            Convert_TevAlphaArg_( AttrTevArg src )
        {
            switch( src )
            {
                case AttrTevArg.A0: return TevAlphaArg.A0;
                case AttrTevArg.A1: return TevAlphaArg.A1;
                case AttrTevArg.A2: return TevAlphaArg.A2;
                case AttrTevArg.APrev: return TevAlphaArg.APrev;
                case AttrTevArg.Konst: return TevAlphaArg.Konst;
                case AttrTevArg.RasA: return TevAlphaArg.RasA;
                case AttrTevArg.TexA: return TevAlphaArg.TexA;
                case AttrTevArg.Zero: return TevAlphaArg.V0;
                default:Debug.Assert( false ); return TevAlphaArg.V0;
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="src"></param>
        /// <returns></returns>
         FileFmt.TevBias
             Convert_TevBias_( AttrTevBias src )
         {
             switch( src )
             {
                 case AttrTevBias.SubHalf: return TevBias.M0_5;
                 case AttrTevBias.AddHalf: return TevBias.P0_5;
                 case AttrTevBias.Zero: return TevBias.V0;
                 default:Debug.Assert( false ); return TevBias.V0;
             }
         }

        /// <summary>
        ///
        /// </summary>
        FileFmt.TevScale
            Convert_TevScale_( AttrTevScale src )
        {
            switch( src )
            {
                case AttrTevScale.Divide2: return TevScale.V1_2;
                case AttrTevScale.Scale1: return TevScale.V1;
                case AttrTevScale.Scale2: return TevScale.V2;
                case AttrTevScale.Scale4: return TevScale.V4;
                default:Debug.Assert( false ); return TevScale.V1;
            }
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.TevKColorSel
            Convert_TevKColorSel_( AttrTevKSel src )
        {
            switch( src )
            {
                case AttrTevKSel.C1_8: return TevKColorSel.V1_8;
                case AttrTevKSel.C2_8: return TevKColorSel.V2_8;
                case AttrTevKSel.C3_8: return TevKColorSel.V3_8;
                case AttrTevKSel.C4_8: return TevKColorSel.V4_8;
                case AttrTevKSel.C5_8: return TevKColorSel.V5_8;
                case AttrTevKSel.C6_8: return TevKColorSel.V6_8;
                case AttrTevKSel.C7_8: return TevKColorSel.V7_8;
                case AttrTevKSel.C8_8: return TevKColorSel.V8_8;


                case AttrTevKSel.K0: return TevKColorSel.K0;
                case AttrTevKSel.K0A: return TevKColorSel.K0_a;
                case AttrTevKSel.K0B: return TevKColorSel.K0_b;
                case AttrTevKSel.K0G: return TevKColorSel.K0_g;
                case AttrTevKSel.K0R: return TevKColorSel.K0_r;

                case AttrTevKSel.K1: return TevKColorSel.K1;
                case AttrTevKSel.K1A: return TevKColorSel.K1_a;
                case AttrTevKSel.K1B: return TevKColorSel.K1_b;
                case AttrTevKSel.K1G: return TevKColorSel.K1_g;
                case AttrTevKSel.K1R: return TevKColorSel.K1_r;

                case AttrTevKSel.K2: return TevKColorSel.K2;
                case AttrTevKSel.K2A: return TevKColorSel.K2_a;
                case AttrTevKSel.K2B: return TevKColorSel.K2_b;
                case AttrTevKSel.K2G: return TevKColorSel.K2_g;
                case AttrTevKSel.K2R: return TevKColorSel.K2_r;

                case AttrTevKSel.K3: return TevKColorSel.K3;
                case AttrTevKSel.K3A: return TevKColorSel.K3_a;
                case AttrTevKSel.K3B: return TevKColorSel.K3_b;
                case AttrTevKSel.K3G: return TevKColorSel.K3_g;
                case AttrTevKSel.K3R: return TevKColorSel.K3_r;


                default:Debug.Assert( false ); return TevKColorSel.V1_8;
            }
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.TevKAlphaSel
            Convert_TevKAlphaSel_( AttrTevKSel src )
        {
            switch( src )
            {
                case AttrTevKSel.C1_8: return TevKAlphaSel.V1_8;
                case AttrTevKSel.C2_8: return TevKAlphaSel.V2_8;
                case AttrTevKSel.C3_8: return TevKAlphaSel.V3_8;
                case AttrTevKSel.C4_8: return TevKAlphaSel.V4_8;
                case AttrTevKSel.C5_8: return TevKAlphaSel.V5_8;
                case AttrTevKSel.C6_8: return TevKAlphaSel.V6_8;
                case AttrTevKSel.C7_8: return TevKAlphaSel.V7_8;
                case AttrTevKSel.C8_8: return TevKAlphaSel.V8_8;

                case AttrTevKSel.K0A: return TevKAlphaSel.K0_a;
                case AttrTevKSel.K0B: return TevKAlphaSel.K0_b;
                case AttrTevKSel.K0G: return TevKAlphaSel.K0_g;
                case AttrTevKSel.K0R: return TevKAlphaSel.K0_r;

                case AttrTevKSel.K1A: return TevKAlphaSel.K1_a;
                case AttrTevKSel.K1B: return TevKAlphaSel.K1_b;
                case AttrTevKSel.K1G: return TevKAlphaSel.K1_g;
                case AttrTevKSel.K1R: return TevKAlphaSel.K1_r;


                case AttrTevKSel.K2A: return TevKAlphaSel.K2_a;
                case AttrTevKSel.K2B: return TevKAlphaSel.K2_b;
                case AttrTevKSel.K2G: return TevKAlphaSel.K2_g;
                case AttrTevKSel.K2R: return TevKAlphaSel.K2_r;


                case AttrTevKSel.K3A: return TevKAlphaSel.K3_a;
                case AttrTevKSel.K3B: return TevKAlphaSel.K3_b;
                case AttrTevKSel.K3G: return TevKAlphaSel.K3_g;
                case AttrTevKSel.K3R: return TevKAlphaSel.K3_r;

                case AttrTevKSel.K0:
                case AttrTevKSel.K1:
                case AttrTevKSel.K2:
                case AttrTevKSel.K3:
                default:Debug.Assert( false ); return TevKAlphaSel.V1_8;
            }
        }


        /// <summary>
        ///
        /// </summary>
        FileFmt.TevOpC
            Convert_TevOpC_( AttrTevOp src )
        {
            switch( src )
            {
                case AttrTevOp.Add: return TevOpC.Add;
                case AttrTevOp.CompBgr24Eq: return TevOpC.Comp_bgr24_eq;
                case AttrTevOp.CompBgr24Gt: return TevOpC.Comp_bgr24_gt;
                case AttrTevOp.CompGr16Eq: return TevOpC.Comp_gr16_eq;
                case AttrTevOp.CompGr16Gt: return TevOpC.Comp_gr16_gt;
                case AttrTevOp.CompR8Eq: return TevOpC.Comp_r8_eq;
                case AttrTevOp.CompR8Gt: return TevOpC.Comp_r8_gt;
                case AttrTevOp.CompRgb8Eq: return TevOpC.Comp_rgb8_eq;
                case AttrTevOp.CompRgb8Gt: return TevOpC.Comp_rgb8_gt;
                case AttrTevOp.Sub: return TevOpC.Sub;

                default:Debug.Assert( false ); return TevOpC.Comp_bgr24_eq;
            }
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.TevOpA
            Convert_TevOpA_( AttrTevOp src )
        {
            switch( src )
            {
                case AttrTevOp.Add: return TevOpA.Add;
                case AttrTevOp.CompA8Eq: return TevOpA.Comp_a8_eq;
                case AttrTevOp.CompA8Gt: return TevOpA.Comp_a8_gt;
                case AttrTevOp.CompBgr24Eq: return TevOpA.Comp_bgr24_eq;
                case AttrTevOp.CompBgr24Gt: return TevOpA.Comp_bgr24_gt;
                case AttrTevOp.CompGr16Eq: return TevOpA.Comp_gr16_eq;
                case AttrTevOp.CompGr16Gt: return TevOpA.Comp_gr16_gt;
                case AttrTevOp.CompR8Eq: return TevOpA.Comp_r8_eq;
                case AttrTevOp.CompR8Gt: return TevOpA.Comp_r8_gt;
                case AttrTevOp.Sub: return TevOpA.Sub;
                default:Debug.Assert( false ); return TevOpA.Comp_bgr24_eq;
            }
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.TevRegID
            Convert_TevRegID_( AttrTevOutReg src )
        {
            switch( src )
            {
                case AttrTevOutReg.Prev: return TevRegID.Prev;
                case AttrTevOutReg.Reg0: return TevRegID.Reg0;
                case AttrTevOutReg.Reg1: return TevRegID.Reg1;
                case AttrTevOutReg.Reg2: return TevRegID.Reg2;
                default:Debug.Assert( false ); return TevRegID.Prev;
            }
        }

        /// <summary>
        /// 右の型から左の型への変換
        /// 文字列が同一である型の変換に利用します(大文字、小文字を区別しません)。
        /// </summary>
        object ConvertEnumTypesFromRToL_( Enum type0, Enum type1 )
        {
             return Enum.Parse( type0.GetType(), type1.ToString(), true );
        }

        #endregion TEV関連列挙型 変換

        /// <summary>
        /// AppData.TevDirectStage => FileFmt.Material_RevoTevStageColor
        /// </summary>
        FileFmt.Material_RevoTevStageColor
            Convert_Material_RevoTevStageColor_( AppData.TevDirectStage srcStage )
        {
            Material_RevoTevStageColor dstStage
                = new Material_RevoTevStageColor();

            dstStage.a = Convert_TevColorArg_( srcStage.A );
            dstStage.b = Convert_TevColorArg_( srcStage.B );
            dstStage.c = Convert_TevColorArg_( srcStage.C );
            dstStage.d = Convert_TevColorArg_( srcStage.D );

            dstStage.bias  = Convert_TevBias_( srcStage.Bias );
            dstStage.clamp = srcStage.Clamp;

            dstStage.konst = Convert_TevKColorSel_( srcStage.KSel );
            dstStage.op    = Convert_TevOpC_( srcStage.Op );
            dstStage.scale = Convert_TevScale_( srcStage.Scale );

            dstStage.outReg = Convert_TevRegID_( srcStage.OutReg );

            return dstStage;
        }

        /// <summary>
        /// AppData.TevDirectStage => FileFmt.Material_RevoTevStageAlpha
        /// </summary>
        FileFmt.Material_RevoTevStageAlpha
            Convert_Material_RevoTevStageAlpha_( AppData.TevDirectStage srcStage )
        {
            Material_RevoTevStageAlpha dstStage
                = new Material_RevoTevStageAlpha();


            dstStage.a = Convert_TevAlphaArg_( srcStage.A);
            dstStage.b = Convert_TevAlphaArg_( srcStage.B);
            dstStage.c = Convert_TevAlphaArg_( srcStage.C);
            dstStage.d = Convert_TevAlphaArg_( srcStage.D);

            dstStage.bias  = Convert_TevBias_( srcStage.Bias );
            dstStage.clamp = srcStage.Clamp;

            dstStage.konst = Convert_TevKAlphaSel_( srcStage.KSel );
            dstStage.op    = Convert_TevOpA_( srcStage.Op );
            dstStage.scale = Convert_TevScale_( srcStage.Scale );

            dstStage.outReg = Convert_TevRegID_( srcStage.OutReg );

            return dstStage;
        }

        /// <summary>
        /// AppData.TevIndirectStage => FileFmt.Material_RevoTevStageIndirect
        /// </summary>
        FileFmt.Material_RevoTevStageIndirect
            Convert_Material_RevoTevStageIndirect_( AppData.TevIndirectStage srcStage )
        {
            Material_RevoTevStageIndirect dstStage = new Material_RevoTevStageIndirect();

            dstStage.addPrev = srcStage.AddPrev;
            dstStage.alpha = IndTexAlphaSel.Off;// srcStage.AlphaSel; OK ?
            dstStage.bias = (IndTexBiasSel)srcStage.BiasSel;
            dstStage.format = (IndTexFormat)srcStage.Format;
            dstStage.indStage = (Byte)srcStage.IndStage;
            dstStage.matrix = (IndTexMtxID)srcStage.MatrixSel;
            dstStage.utcLod = srcStage.UtcLod;
            dstStage.wrap_s = (IndTexWrap)srcStage.WrapS;
            dstStage.wrap_t = (IndTexWrap)srcStage.WrapT;

            return dstStage;
        }

        /// <summary>
        /// AppData.TevStage => FileFmt.Material_RevoTevStage
        /// </summary>
        FileFmt.Material_RevoTevStage
            Convert_RevoTevStage_( AppData.ITevStage srcStage )
        {
            Material_RevoTevStage dstStage =
                new Material_RevoTevStage();

            // カラーステージ
            dstStage.color = Convert_Material_RevoTevStageColor_( srcStage.ColorStage );
            // アルファステージ
            dstStage.alpha = Convert_Material_RevoTevStageAlpha_( srcStage.AlphaStage );
            // インダイレクトステージ
            dstStage.indirect = Convert_Material_RevoTevStageIndirect_( srcStage.IndirectStage );

            // カラーオーダー
            dstStage.colorChannel = Convert_TevChannelID_( srcStage.RasOrder );

            // TexMap, TexCoordGen
            dstStage.texMap      = Convert.ToSByte( srcStage.TexMapOrder );
            dstStage.texCoordGen = Convert.ToSByte( srcStage.TexCoordOrder );

            // swap table
            dstStage.rasColSwap = Convert.ToSByte( srcStage.RasSwap );
            // swap table
            dstStage.texColSwap = Convert.ToSByte( srcStage.TexMapSwap );

            return dstStage;
        }

        /// <summary>
        /// AppData.ITexMtx => FileFmt.Material_RevoIndirectMatrix
        /// </summary>
        FileFmt.TexMatrix
            Convert_IndirectMatrix_( AppData.ITexMtx  srcIndMtx )
        {
            FileFmt.TexMatrix dstIndMtx =
                new FileFmt.TexMatrix();

            dstIndMtx.rotate = srcIndMtx.Rotate;

            dstIndMtx.scale       = Convert_Vec2_( srcIndMtx.Scale );
            dstIndMtx.translate   = Convert_Vec2_( srcIndMtx.Trans );

            return dstIndMtx;
        }

        /// <summary>
        /// AppData.ITexIndirectStage => FileFmt.Material_RevoIndirectStage
        /// </summary>
        FileFmt.Material_RevoIndirectStage
            Convert_IndirectStage_( AppData.ITexIndirectStage  srcIndStage )
        {
            Material_RevoIndirectStage dstIndStage =
                new Material_RevoIndirectStage();

            dstIndStage.texMap      = Convert.ToByte( srcIndStage.TexMapOrder );
            dstIndStage.texCoordGen = Convert.ToByte( srcIndStage.TexCoordOrder );

            dstIndStage.scale_s = (IndTexScale)srcIndStage.TexCoordScaleS;
            dstIndStage.scale_t = (IndTexScale)srcIndStage.TexCoordScaleT;

            return dstIndStage;
        }


        /// <summary>
        /// AppData.IRevHWMaterial => FileFmt.Material_Revo
        /// </summary>
        FileFmt.Material_Revo Convert_ToFileMaterial_Revo( AppData.IRevHWMaterial appMat )
        {
            FileFmt.Material_Revo    dstMat = new Material_Revo();

            // ChannelControl 関連
            dstMat.channelControl = new FileFmt.Material_RevoChannelControl[2];
            dstMat.channelControl[0] = new FileFmt.Material_RevoChannelControl();
            dstMat.channelControl[0].channel          = ChannelID.Color0;
            dstMat.channelControl[0].materialSource   = Convert_ToFileColorSource_( appMat.IChannelData.DiffuseCtrlColorSrc );

            dstMat.channelControl[1] = new FileFmt.Material_RevoChannelControl();
            dstMat.channelControl[1].channel          = ChannelID.Alpha0;
            dstMat.channelControl[1].materialSource   = Convert_ToFileColorSource_( appMat.IChannelData.AlphaCtrlColorSrc );

            dstMat.matColReg      = Convert_Color4_( appMat.IChannelData.MatColReg0 );
            dstMat.name           = appMat.MaterialName;



            // PE - AlphaCompare
            dstMat.alphaCompare = Convert_ToFileRevoAlphaCompare_( appMat.IPEData );
            dstMat.blendMode    = Convert_ToFileRevoBlendMode_( appMat.IPEData );

            // Tev - Colorレジスタ
            dstMat.tevColReg = new FileFmt.ColorS10_4[3];
            for( int i = 0; i < 3; i++ )
            {
                dstMat.tevColReg[i] = Convert_ColorS10_4_( appMat.ITevData.GetTevColors( i ) );
            }

            // Tev - コンスタント レジスタ
            dstMat.tevConstReg    = new FileFmt.Color4[4];
            for( int i = 0; i < 4; i++ )
            {
                dstMat.tevConstReg[i] = Convert_Color4_( appMat.ITevData.GetTevKColors( i ) );
            }

            // Tev - SwapTable
            dstMat.swapTable = new Material_RevoSwapTable[4];
            for( int i = 0; i < 4; i++ )
            {
                dstMat.swapTable[i] = Convert_ToFileRevoSwapTable_( appMat.ITevData.ISwapTables[i] );
            }

            // Tev - TevStage
            // 現在は、使用される情報のみ編集可能となっているので、
            // 保存すべきステージの数 = 使用されるステージの数です。
            int numTevStage = appMat.ITevData.NumStages;
            dstMat.tevStage = new FileFmt.Material_RevoTevStage[numTevStage];
            dstMat.tevStageNum = (byte)numTevStage;
            for( int i = 0; i < dstMat.tevStage.Length; i++ )
            {
                dstMat.tevStage[i] = Convert_RevoTevStage_( appMat.ITevData.GetITevStage( i ) );
            }


            // Tev - インダイレクト行列
            dstMat.indirectMatrix = new FileFmt.TexMatrix[appMat.IIndirectTexMtxSet.NumTexIndirectMtx];
            for( int i = 0 ; i < dstMat.indirectMatrix.Length ; i++ )
            {
                dstMat.indirectMatrix[i] = Convert_IndirectMatrix_( appMat.IIndirectTexMtxSet[i] );
            }

            // インダイレクトステージ
            dstMat.indirectStage  = new FileFmt.Material_RevoIndirectStage[appMat.ITexIndirectStageSet.NumTexIndirectStages];
            dstMat.indirectStageNum = (byte)appMat.ITexIndirectStageSet.NumStageUsed;
            for( int i = 0; i < dstMat.indirectStage.Length; i++ )
            {
                dstMat.indirectStage[i] = Convert_IndirectStage_( appMat.ITexIndirectStageSet[ i ] );
            }




            // Textrue に関する情報
            AppData.IMaterialTexMap[] texMaps = appMat.IMaterialTexMapSet;

            dstMat.texMap    = new FileFmt.TexMap[texMaps.Length];
            dstMat.texMatrix = new FileFmt.TexMatrix[texMaps.Length];
            dstMat.texCoordGen = new FileFmt.TexCoordGen[texMaps.Length];

            for( int i = 0;i < texMaps.Length; i++ )
            {
                if(  texMaps[i] != null )
                {
                    dstMat.texMap[i]       = Convert_ToFileTexMap_( texMaps[i] );
                    dstMat.texMatrix[i]    = Convert_ToFileTexMtx_( texMaps[i].ITexGen.ITexMtx );
                    dstMat.texCoordGen[i]  = Convert_ToFileTexCoordGen_( texMaps[i].ITexGen );
                }
            }

            return dstMat;
        }
        #endregion 詳細なマテリアル設定

        #endregion マテリアル関連

        #region 各種ペイン固有パラメータの書き出し

        /// <summary>
        /// TextBoxペイン固有情報の書き出し
        /// </summary>
        FileFmt.TextBox  Convert_Item_TextBox_( AppData.IPane pane )
        {
            AppData.ITextBox  srcTextBox = pane.ITextBox;
            Debug.Assert( srcTextBox != null );

            // フォントが正しく設定されていない場合は保存を中断します。
            if( srcTextBox.ILEFont == null )
            {
                string message = String.Format( "Invalid Font [Pane = {0}]", pane.PaneName );
                throw new Exception( message );
            }


            FileFmt.TextBox dstTexBox = new FileFmt.TextBox();
            dstTexBox.font          = srcTextBox.ILEFont.FontName;

            if( srcTextBox.SpecifiedStringLength )
            {
                dstTexBox.allocateStringLength = Convert.ToUInt32( srcTextBox.AllocStringLength );
                dstTexBox.allocateStringLengthSpecified = true;
            }
            else
            {
                dstTexBox.allocateStringLength = 0;
                dstTexBox.allocateStringLengthSpecified = false;
            }

            dstTexBox.charSpace  = srcTextBox.CharSpace;
            dstTexBox.lineSpace  = srcTextBox.LineSpace;
            dstTexBox.text       = srcTextBox.ContentsText;
            dstTexBox.fontSize   = new Vec2();
            dstTexBox.fontSize.x = srcTextBox.FontSize.X;
            dstTexBox.fontSize.y = srcTextBox.FontSize.Y;

            dstTexBox.bottomColor = Convert_Color4_( srcTextBox.BottomColor );
            dstTexBox.topColor    = Convert_Color4_( srcTextBox.TopColor );

            dstTexBox.positionType =
                Convert_Position_( srcTextBox.BasePosTypeH, srcTextBox.BasePosTypeV );

            // マテリアル情報の書き出し。

            dstTexBox.material = Convert_ToFileMaterial( srcTextBox.IMaterial );

            // Revolution マテリアル情報の書き出し。
            if( srcTextBox is IRevHWMaterialHolder )
            {
                IRevHWMaterialHolder revHWMaterialHolder =
                    srcTextBox as IRevHWMaterialHolder;

                Debug.Assert( revHWMaterialHolder.IRevHWMaterial.Length == 1 );
                IRevHWMaterial revMat = revHWMaterialHolder.IRevHWMaterial[0];
                dstTexBox.materialRevo = Convert_ToFileMaterial_Revo( revMat );
            }

            return  dstTexBox;
        }

        /// <summary>
        /// テクスチャ座標情報の書き出し
        /// </summary>
        FileFmt.TexCoord[] Convert_TexCoord_( AppData.IPicture   srcPicture )
        {
            TexCoord[] dstTexCoords = new TexCoord[srcPicture.NumTexCoord];

            for( int i = 0; i < dstTexCoords.Length; i++ )
            {
                TexCoord4 srcTexCoord4 = srcPicture.GetTexCoord4( i );
                dstTexCoords[i] = Convert_TexCoord_( srcTexCoord4 );
            }

            return dstTexCoords;
        }

        /// <summary>
        /// Pictureペイン固有情報の書き出し
        /// </summary>
        FileFmt.Picture Convert_Item_Picture_( AppData.IPane pane )
        {
            FileFmt.Picture      dstPicture = new FileFmt.Picture();
            AppData.IPicture     picture    = pane.IPicture;

            dstPicture.vtxColLT = Convert_Color4_( picture.VtxCol_LT );
            dstPicture.vtxColRT = Convert_Color4_( picture.VtxCol_RT );
            dstPicture.vtxColLB = Convert_Color4_( picture.VtxCol_LD );
            dstPicture.vtxColRB = Convert_Color4_( picture.VtxCol_RD );

            // テクスチャ座標情報の書き出し
            dstPicture.texCoord = Convert_TexCoord_( picture );

            // 詳細なマテリアル設定を行うか？
            dstPicture.detailSetting = picture.UseDetailedMaterial;

            // マテリアル情報の書き出し。
            dstPicture.material = Convert_ToFileMaterial( picture.IMaterial );

            // Revolution マテリアル情報の書き出し。
            IRevHWMaterialHolder    revPicture = picture as IRevHWMaterialHolder;
            dstPicture.materialRevo = Convert_ToFileMaterial_Revo( revPicture.IRevHWMaterial[0] );

            return dstPicture;
        }


        #region Windowペイン関連
        /// <summary>
        /// LEWindowFrameFlipRotate => FileFmt.TextureFlip
        /// 強引にキャストしています。
        /// </summary>
        FileFmt.TextureFlip
            Convert_Item_TextureFlip_( LEWindowFrameFlipRotate srcFlipRotState )
        {
            return (FileFmt.TextureFlip)srcFlipRotState;
        }


        /// <summary>
        /// Windowペインのフレーム種類型を変換します。
        /// </summary>
        WindowFrameType
            Convert_Item_LEWindowPartsID_( AppData.LEWindowPartsID  frameType )
        {
            switch( frameType )
            {
                case AppData.LEWindowPartsID.CornerLT: return WindowFrameType.CornerLT;
                case AppData.LEWindowPartsID.CornerRT: return WindowFrameType.CornerRT;
                case AppData.LEWindowPartsID.CornerLB: return WindowFrameType.CornerLB;
                case AppData.LEWindowPartsID.CornerRB: return WindowFrameType.CornerRB;
                case AppData.LEWindowPartsID.FrameT: return WindowFrameType.FrameT;
                case AppData.LEWindowPartsID.FrameB: return WindowFrameType.FrameB;
                case AppData.LEWindowPartsID.FrameL: return WindowFrameType.FrameL;
                case AppData.LEWindowPartsID.FrameR: return WindowFrameType.FrameR;
                default:Debug.Assert( false );return WindowFrameType.CornerLT;
            }
        }

        /// <summary>
        /// ウインドウフレーム情報配列を書き出します。
        /// </summary>
        FileFmt.WindowFrame[]
            Convert_Item_WindowFrameSet_( AppData.ILEWindow srcWindow )
        {

            int numFrameTextures = srcWindow.NumTexture - 1;
            FileFmt.WindowFrame[]   dstWindowFrameSet = new FileFmt.WindowFrame[numFrameTextures];

            IRevHWMaterialHolder   srcRevWindow = srcWindow as IRevHWMaterialHolder;


            for( int i = 0; i < dstWindowFrameSet.Length; i++ )
            {
                int frameIndex = i + 1;

                dstWindowFrameSet[i] = new FileFmt.WindowFrame();

                LEWindowPartsID partsID = srcWindow.IntToPartsID( frameIndex );

                dstWindowFrameSet[i].frameType = Convert_Item_LEWindowPartsID_( partsID );

                // 詳細なマテリアル設定を行うか？
                dstWindowFrameSet[i].detailSetting = srcWindow.GetUseDetailedMaterialFlag( partsID );

                dstWindowFrameSet[i].material = Convert_ToFileMaterial( srcWindow.GetMaterial( partsID ) );
                dstWindowFrameSet[i].materialRevo = Convert_ToFileMaterial_Revo( srcRevWindow.IRevHWMaterial[frameIndex] );

                dstWindowFrameSet[i].textureFlip = new FileFmt.TextureFlip[1];
                dstWindowFrameSet[i].textureFlip[0] = Convert_Item_TextureFlip_( srcWindow.GetMaterialFlipRotateState( partsID ) );
            }

            return dstWindowFrameSet;
        }
        /// <summary>
        /// Windowペインコンテンツ領域：拡大量情報の書き出し。
        /// </summary>
        FileFmt.InflationRect
            Convert_Item_InflationRect_( AppData.ILEWindowContentsInflation srcInflation )
        {
            FileFmt.InflationRect dstInflation = new FileFmt.InflationRect();

            dstInflation.t = srcInflation.Top;
            dstInflation.b = srcInflation.Bottom;
            dstInflation.l = srcInflation.Left;
            dstInflation.r = srcInflation.Right;

            return dstInflation;
        }

        /// <summary>
        /// Windowペインコンテンツ領域情報の書き出し。
        /// </summary>
        FileFmt.WindowContent
            Convert_Item_WindowContent_( AppData.ILEWindow srcWindow )
        {
            FileFmt.WindowContent       dstContent = new FileFmt.WindowContent();

            dstContent.vtxColLT = Convert_Color4_( srcWindow.GetContentsConerColorByIndex( 0 ) );
            dstContent.vtxColRT = Convert_Color4_( srcWindow.GetContentsConerColorByIndex( 1 ) );
            dstContent.vtxColRB = Convert_Color4_( srcWindow.GetContentsConerColorByIndex( 2 ) );
            dstContent.vtxColLB = Convert_Color4_( srcWindow.GetContentsConerColorByIndex( 3 ) );

            // 詳細なマテリアル設定を行うか？
            dstContent.detailSetting = srcWindow.GetUseDetailedMaterialFlag( LEWindowPartsID.Content );

            // マテリアル情報を書き出します。
            IRevHWMaterialHolder   srcRevWindow = srcWindow as IRevHWMaterialHolder;

            dstContent.material = Convert_ToFileMaterial( srcWindow.GetMaterial( LEWindowPartsID.Content ) );
            dstContent.materialRevo = Convert_ToFileMaterial_Revo( srcRevWindow.IRevHWMaterial[ srcWindow.PartsIDToInt( LEWindowPartsID.Content )] );

            // テクスチャ座標を書き出します。
            // テクスチャが張られていない場合は、nullが代入されます。
            dstContent.texCoord = new TexCoord[ srcWindow.NumContentsTexCoord];
            for( int i = 0; i < dstContent.texCoord.Length; i++ )
            {
                TexCoord4 srcTexCoord4 = srcWindow.GetContentsTexCood4( i );
                dstContent.texCoord[i] = Convert_TexCoord_( srcTexCoord4 );
            }

            return dstContent;
        }

        /// <summary>
        /// Windowペイン固有情報の書き出し
        /// </summary>
        FileFmt.Window Convert_Item_Window_( AppData.IPane pane )
        {
            FileFmt.Window      dstWindow = new FileFmt.Window();

            ILEWindow srcWindow = pane.ILEWindow;

            dstWindow.content = Convert_Item_WindowContent_( srcWindow );
            dstWindow.contentInflation = Convert_Item_InflationRect_( srcWindow.ContentsInflation );
            dstWindow.frame = Convert_Item_WindowFrameSet_( srcWindow );

            return dstWindow;
        }
        #endregion Windowペイン関連

        /// <summary>
        /// Boundingペイン固有情報の書き出し
        /// </summary>
        FileFmt.Bounding Convert_Item_Bounding_( AppData.IPane pane )
        {
            FileFmt.Bounding      dstBounding = new FileFmt.Bounding();
            // TODO:
            return dstBounding;
        }

        /// <summary>
        /// 各種ペイン(ピクチャ、レターペインなど)に固有なパラメータを格納するクラスへと変換します。
        /// TODO:
        /// </summary>
        object Convert_Item_( AppData.IPane srcPane )
        {
            // 各種ペインに固有な情報の設定
            switch( srcPane.PaneKind )
            {
                case AppData.PaneKind.Null: /* DO NOTHING */ return null;
                case AppData.PaneKind.Letter: return Convert_Item_TextBox_( srcPane );
                case AppData.PaneKind.Picture: return Convert_Item_Picture_( srcPane );
                case AppData.PaneKind.Window: return Convert_Item_Window_( srcPane );
                case AppData.PaneKind.Bounding: return Convert_Item_Bounding_( srcPane );
                default:Debug.Assert( false ); return null;
            }
        }
        #endregion 各種ペイン固有パラメータの書き出し

        /// <summary>
        /// PaneModify.Kind => FileFmt.PaneKind
        /// </summary>
        FileFmt.PaneKind Convert_SOPaneKind_( AppData.PaneKind kind )
        {
            switch( kind )
            {
                case AppData.PaneKind.Null:return FileFmt.PaneKind.Null;
                case AppData.PaneKind.Picture:return FileFmt.PaneKind.Picture;
                case AppData.PaneKind.Letter:return FileFmt.PaneKind.TextBox;
                case AppData.PaneKind.Window:return FileFmt.PaneKind.Window;
                case AppData.PaneKind.Bounding:return FileFmt.PaneKind.Bounding;
                default:Debug.Assert( false );return FileFmt.PaneKind.Null;
            }
        }

        /// <summary>
        /// ユーザデータ配列を作成します。
        /// 暫定的な実装です。
        /// </summary>
        object[] HACK_Convert_PaneUserData( AppData.IPane pane )
        {
            UserDataString       userDataStr = new UserDataString();
            userDataStr.Value = pane.UserData;

            return new object[] { userDataStr };
        }

        /// <summary>
        /// FileFmt.PaneModify へと変換します。
        /// </summary>
        FileFmt.Pane     Convert_SOPane( AppData.IPane pane )
        {
            FileFmt.Pane result = new FileFmt.Pane();

            result.translate  = Convert_Vec3_( pane.Trans );
            result.rotate     = Convert_Vec3_( pane.RotAng );

            result.scale      = Convert_Vec2_( pane.Scale );
            result.size       = Convert_Vec2_( pane.Size );

            result.alpha      = pane.Transparency;

            result.kind       = Convert_SOPaneKind_( pane.PaneKind );

            result.basePositionType = Convert_Position_( pane.BasePosTypeH, pane.BasePosTypeV );

            result.name   = pane.PaneName;

            result.comment    = pane.UserCommentString;
            result.userData   = HACK_Convert_PaneUserData( pane );
            result.visible    = pane.Visible;

            result.locked     = pane.Locked;
            result.hidden     = pane.Hidden;
            result.influencedAlpha = pane.InfluenceChildrenTransparency;
            result.locationAdjust = pane.LocationAdjust;


            // TODO:ピクチャ、レターペインの固有情報
            try
            {
                result.Item = Convert_Item_( pane );
            }
            catch( Exception e )
            {
                // Nullペインとして保存します。
                result.kind = FileFmt.PaneKind.Null;

                string title = StringResMgr.Get( "LECORE_RLYTSAVE_PANE" );
                string msg0 = StringResMgr.Get( "LECORE_RLYTSAVE_CVTTONULL" );
                string message = msg0 + " : " + e.Message;
                _msgReporter.ReportError( title, message );
            }

            return result;
        }

        /// <summary>
        /// FileFmt.PaneModify[] へと変換します。
        /// </summary>
        FileFmt.Pane[]   Convert_SOPaneArray( ISubScene scene )
        {
            OutLog_( "LECORE_RLYTSAVE_PANE" );

            AppData.IPane[]        paneSet = scene.IPaneArray;
            List<FileFmt.Pane>      result = new List<FileFmt.Pane>();

            foreach( AppData.IPane srcPane in paneSet )
            {
                try
                {
                    result.Add( Convert_SOPane( srcPane ) );
                }
                catch( Exception e )
                {
                    string title = StringResMgr.Get( "LECORE_RLYTSAVE_PANE" );
                    _msgReporter.ReportError( title, e.Message );
                }
            }
            return result.ToArray();
        }

        #endregion ペイン関連

        #region グループ 階層関連

        #region グループ
        /// <summary>
        /// PaneModify から GroupPaneRef へと変換します。
        /// </summary>
        GroupPaneRef Convert_GroupPaneRef_( AppData.Pane pane )
        {
            GroupPaneRef result = new GroupPaneRef();

            result.name = pane.PaneName;

            return result;
        }

        /// <summary>
        /// GroupModify へと変換します。
        /// </summary>
        FileFmt.Group Convert_Group_( ILEGroup srcGroup )
        {
            FileFmt.Group    dstGroup       = new Group();
            ArrayList        memberGroupSet = new ArrayList();
            ArrayList        memberPaneSet  = new ArrayList();

            // グループのメンバ情報をコレクションに格納します。
            if( srcGroup.HasMember )
            {
                IPane[]      members = srcGroup.Member;
                foreach( IPane member in  members )
                {
                    // メンバとして登録します。
                    // 種類によって別のコレクションに格納されます。
                    // 現在はルートグループ以外は、必ず、ペインとして出力されます。
                    GroupPaneRef       memberPane = new GroupPaneRef();
                    memberPane.name = member.PaneName;
                    memberPaneSet.Add( memberPane );
                }
            }

            dstGroup.comment = srcGroup.UserCommentString;
            dstGroup.name    = srcGroup.GrouprName;
            dstGroup.group   = memberGroupSet.ToArray( typeof( FileFmt.Group ) ) as FileFmt.Group[];
            dstGroup.paneRef = memberPaneSet.ToArray( typeof( FileFmt.GroupPaneRef ) ) as FileFmt.GroupPaneRef[];

            return dstGroup;
        }

        /// <summary>
        /// FileFmt.GroupSet へと変換します。
        /// </summary>
        FileFmt.GroupSet Convert_SOGroupSet_( ISubScene scene )
        {
            ILEGroupMgr            groupMgr    =  scene.ILEGroupMgr;
            List<FileFmt.Group>   dstGroupSet =  new List<FileFmt.Group>();

            // 全てのグループについて...
            OutLog_( "LECORE_RLYTLOAD_GROUP" );
            foreach( ILEGroup srcGroup in groupMgr.GroupSet )
            {
                try
                {
                    FileFmt.Group dstGroup = Convert_Group_( srcGroup );
                    if( dstGroup != null )
                    {
                        dstGroupSet.Add( dstGroup );
                    }
                    else
                    {
                        Debug.Assert( false );
                    }
                }
                catch( Exception e )
                {
                    _msgReporter.ReportError( StringResMgr.Get( "LECORE_RLYTLOAD_GROUP" ), e.Message );
                }
            }


            FileFmt.GroupSet result = new FileFmt.GroupSet();
            result.group = new FileFmt.Group();
            result.group.name    = groupMgr.RootNodeName;
            result.group.group = dstGroupSet.ToArray();
            result.group.comment = "";
            return result;
        }
        #endregion グループ

        #region 親子階層
        void Convert_PaneTree_SubTree_( AppData.IPane pane, PaneTree paneTree )
        {
            IHierarchyNode[] chilren = pane.Children;

            paneTree.name     = pane.PaneName;
            paneTree.paneTree = new PaneTree[chilren.Length];

            for( int i = 0; i < chilren.Length; i++ )
            {
                AppData.IPane   child = chilren[i] as AppData.IPane;
                PaneTree  subTree    = new PaneTree();

                // 子供について再帰的に処理をおこないます。
                Convert_PaneTree_SubTree_( child, subTree );

                paneTree.paneTree[i] = subTree;
            }
        }

        /// <summary>
        /// PaneTree へと 変換します。
        /// </summary>
        PaneTree Convert_PaneTree_( ISubScene scene )
        {
            PaneTree      result   = new PaneTree();

            Convert_PaneTree_SubTree_( scene.RootIPane, result );

            return result;
        }

        /// <summary>
        /// FileFmt.PaneHierarchy へと変換します。
        /// </summary>
        FileFmt.PaneHierarchy Convert_SOPaneHierarchy_( ISubScene scene )
        {
            OutLog_( "LECORE_RLYTLOAD_HIER" );
            try
            {
                FileFmt.PaneHierarchy result = new FileFmt.PaneHierarchy();
                result.paneTree = Convert_PaneTree_( scene );
                return result;
            }
            catch( Exception e )
            {
                string title = StringResMgr.Get( "LECORE_RLYTLOAD_HIER" );
                _msgReporter.ReportError( title, e.Message );
                return null;
            }
        }
        #endregion 親子階層

        #endregion グループ 階層関連

        #region テクスチャ、フォントリソース関連
        /// <summary>
        /// テクスチャリストを構築します。
        ///
        /// 現在は、シーンのテクスチャリストを出力しています。
        /// レイアウトから使用されない画像も含まれます。
        /// </summary>
        FileFmt.TextureFile[]    Convert_SOTextureFiles_( ISubScene scene )
        {
            List<FileFmt.TextureFile> texImgs   = new List<TextureFile>();
            ITextureImage[]            textures  = scene.ITextureMgr.ITextureImageSet;

            OutLog_( "LECORE_RLYTSAVE_TEX" );
            foreach( ITextureImage texImg in textures )
            {
                try
                {
                    FileFmt.TextureFile texFile = new TextureFile();
                    texFile.format = texImg.PixelFmtIsFixed ?
                        FileFmt.TexelFormat.NW4R_TGA : Convert_TexImagePixelFmt_( texImg.PixelFmt );
                    texFile.imagePath = GetRelativePath_( texImg.FilePath );

                    texImgs.Add( texFile );
                }
                catch( Exception e )
                {
                    _msgReporter.ReportError( StringResMgr.Get( "LECORE_RLYTSAVE_TEX" ), e.Message );
                }
            }


            return texImgs.ToArray();
        }

        /// <summary>
        /// フォントリストを構築します。
        ///
        /// 現在は、シーンのテクスチャリストを出力しています。
        /// レイアウトから使用されない画像も含まれます。
        /// </summary>
        FileFmt.FontFile[]       Convert_SOFontFiles_( ISubScene scene )
        {
            OutLog_( "LECORE_RLYTSAVE_FONT" );

            List<FileFmt.FontFile> fontFileSet = new List<FileFmt.FontFile>();
            ILEFont[]               leFonts      = scene.ILEFontManager.ILEFontSet;
            foreach( ILEFont leFont in leFonts )
            {
                try
                {
                    FileFmt.FontFile fontFile = new FileFmt.FontFile();
                    fontFile.path = GetRelativePath_( leFont.FontPath );

                    fontFileSet.Add( fontFile );
                }
                catch( Exception e )
                {
                    _msgReporter.ReportError( StringResMgr.Get( "LECORE_RLYTSAVE_FONT" ), e.Message );
                }
            }
            return fontFileSet.ToArray();
        }

        #endregion テクスチャ、フォントリソース関連

        /// <summary>
        /// スクリーン設定を変換します。
        /// </summary>
        FileFmt.ScreenSetting Convert_SOScreenSetting_( ISubScene scene )
        {

            AppData.BackGround          backGround = scene.BackGround;
            FileFmt.ScreenSetting       screenSetting = new ScreenSetting();

            //-------------- backGround 関連
            screenSetting.backGround = new ScreenSettingBackGround();
            screenSetting.backGround.color =
                Convert_Color4_( new RGBAColor( backGround.BackGroundColor ) );

            // 背景画像が存在する場合は...
            if( scene.BackGround.BackgroundImage != null )
            {
                screenSetting.backGround.imageFile =
                    GetRelativePath_( backGround.BackgroundImage.FilePath );
            }

            //-------------- layoutSize 関連
            screenSetting.layoutSize = new Vec2();
            screenSetting.layoutSize.x = backGround.ScreenSize.X;
            screenSetting.layoutSize.y = backGround.ScreenSize.Y;

            screenSetting.grid = new ScreenSettingGrid();
            screenSetting.grid.color = Convert_Color4_( new RGBAColor( backGround.GridColor ) );
            screenSetting.grid.thickLineInterval = backGround.GridSize;
            screenSetting.grid.thinDivisionNum = backGround.GridDivisionNum;
            screenSetting.grid.visible         = backGround.GridVisible;
            screenSetting.grid.moveMethod      = ( backGround.GridSnapEnalbed ) ? GridMoveMethod.Grid : GridMoveMethod.None;


            //-------------- 座標系関連
            screenSetting.origin = Convert_ScreenCoordinate_( LayoutEditorCore.Config.ScreenCoordinate );

            return screenSetting ;
        }

        /// <summary>
        /// rlyt へと変換します。
        /// </summary>
        FileFmt.RLYT Convert_SORLYT_( ISubScene scene )
        {
            FileFmt.RLYT result = new FileFmt.RLYT();

            result.paneSet          = Convert_SOPaneArray( scene );

            result.groupSet         = Convert_SOGroupSet_( scene );
            result.paneHierarchy    = Convert_SOPaneHierarchy_( scene );
            result.textureFile      = Convert_SOTextureFiles_( scene );
            result.fontFile         = Convert_SOFontFiles_( scene );
            result.screenSetting    = Convert_SOScreenSetting_( scene );


            return result;
        }

        /// <summary>
        /// FileFmt.DocumentBody へと変換します。
        /// </summary>
        FileFmt.DocumentBody Convert_SODocumentBody_( ISubScene scene )
        {
            FileFmt.DocumentBody result = new FileFmt.DocumentBody();

            result.rlyt = Convert_SORLYT_( scene  );

            return result;
        }


        #endregion FileFmt.DocumentBody 以下

        #endregion 外部非公開メソッド(オブジェクト生成関連)

        #endregion LECore内部クラス => シリアライズ可能オブジェクト への変換
    }
}
