﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Linq;
using LECore.Structures.SerializableObject;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;
    using System.IO;

    using LECore.Manipulator;

    /// <summary>
    /// ヘルパクラス
    /// </summary>
    internal static class RlytConverterHepler
    {
        /// <summary>
        /// オプションフラグがあるかどうか？
        /// </summary>
        internal static bool ContainsUsageOption_(this FileFmt.PartsPropertyUsageOptions[] usageOptions, FileFmt.PartsPropertyUsageOptions option)
        {
            return usageOptions != null && usageOptions.Contains(option);
        }
    }

    public partial class RlytConverter
    {
        /// <summary>
        /// 標準ユーザデータの要素名
        /// </summary>
        const string _BasicUserDataStringName = "__BasicUserDataString";

        #region App => File

        #region ペイン関連

        #region マテリアル関連





        #endregion マテリアル関連

        #region 各種ペイン固有パラメータの書き出し

        #region TextBox
        /// <summary>
        /// TextBox の変換
        /// </summary>
        FileFmt.TextBox Convert_Item_TextBoxSelf_( AppData.ITextBox srcTextBox, bool forceOutShadowParamater = false)
        {
            Debug.Assert( srcTextBox != null );

            FileFmt.TextBox dstTexBox = new FileFmt.TextBox();
            dstTexBox.font = srcTextBox.FontName;

            if( srcTextBox.SpecifiedStringLength )
            {
                dstTexBox.allocateStringLength = Convert.ToUInt32( srcTextBox.AllocStringLength );
                dstTexBox.allocateStringLengthSpecified = true;
            }
            else
            {
                dstTexBox.allocateStringLength = 0;
                dstTexBox.allocateStringLengthSpecified = false;
            }

            dstTexBox.charSpace = srcTextBox.CharSpace;
            dstTexBox.lineSpace = srcTextBox.LineSpace;
            dstTexBox.text = srcTextBox.ContentsText;
            dstTexBox.textID = srcTextBox.TextID;
            dstTexBox.fontSize = new Vec2();
            dstTexBox.fontSize.x = srcTextBox.FontSize.X;
            dstTexBox.fontSize.y = srcTextBox.FontSize.Y;

            bool isFontValid = srcTextBox.ILEFont != null && !srcTextBox.ILEFont.IsFileNotFound();
            dstTexBox.fontSizeOriginal = new Vec2();
            dstTexBox.fontSizeOriginal.x = isFontValid ? srcTextBox.ILEFont.Width : srcTextBox.FontSizeOriginal.X;
            dstTexBox.fontSizeOriginal.y = isFontValid ? srcTextBox.ILEFont.Height : srcTextBox.FontSizeOriginal.Y;

            dstTexBox.bottomColor = Convert_Color4_(srcTextBox.BottomColor);
            dstTexBox.topColor = Convert_Color4_( srcTextBox.TopColor );

            dstTexBox.positionType =
                Convert_Position_( srcTextBox.BasePosTypeH, srcTextBox.BasePosTypeV );

            dstTexBox.textAlignment = Convert_TextAlignment_( srcTextBox.LineAlignment );

            dstTexBox.centerCeilingEnabled = srcTextBox.CenterCeilingEnabled;
            dstTexBox.drawFromRightToLeft = srcTextBox.DrawFromRightToLeft;

            // 行ごとの幅とオフセット
            dstTexBox.lineWidthOffsetEnabled = srcTextBox.LineWidthOffsetEnabled;
            if (!srcTextBox.LineWidthOffsetEnabled)
            {
                dstTexBox.lineWidthOffset = null;
            }
            else
            {
                dstTexBox.lineWidthOffset = new LineWidthOffset[ITextBoxHelper.LineOffsetWidthCountMax];
                for (int i = 0; i < ITextBoxHelper.LineOffsetWidthCountMax; i++)
                {
                    dstTexBox.lineWidthOffset[i] = new LineWidthOffset();
                    dstTexBox.lineWidthOffset[i].lineOffset = srcTextBox.LineOffset[i];
                    dstTexBox.lineWidthOffset[i].lineWidth = srcTextBox.LineWidth[i];
                }
            }

            // 斜体
            dstTexBox.italicFactor = srcTextBox.ItalicFactor;

            // 影
            dstTexBox.shadowEnabled = srcTextBox.ShadowEnabled;
            dstTexBox.shadowOffset = Convert_Vec2_(srcTextBox.ShadowOffset);
            dstTexBox.shadowScale = Convert_Vec2_(srcTextBox.ShadowScale);
            dstTexBox.shadowItalicFactor = srcTextBox.ShadowItalicFactor;
            dstTexBox.shadowTopColor = Convert_Color4_(srcTextBox.ShadowTopColor);
            dstTexBox.shadowBottomColor = Convert_Color4_(srcTextBox.ShadowBottomColor);

            // 文字変化
            dstTexBox.perCharacterTransformEvalTimeOffset = srcTextBox.PerCharacterTransformTimeOffset;
            dstTexBox.perCharacterTransformEvalTimeWidth = srcTextBox.PerCharacterTransformTimeWidth;
            dstTexBox.perCharacterTransformEnabled = srcTextBox.IsPerCharTransformEnabled;
            dstTexBox.perCharacterTransformLoopType = Convert_PerCharTransformLoopType_(srcTextBox.PerCharacterTransformLoopType);
            dstTexBox.perCharacterTransformOriginV = Convert_VerticalPosition_(srcTextBox.PerCharacterTransformOriginV);
            dstTexBox.perCharacterTransformOriginVOffset = srcTextBox.PerCharacterTransformOriginVOffset;
            dstTexBox.perCharacterTransformSplitByCharWidth = srcTextBox.PerCharacterTransformSplitByCharWidth;
            dstTexBox.perCharacterTransformAutoShadowAlpha = srcTextBox.PerCharacterTransformAutoShadowAlpha;
            dstTexBox.perCharacterTransformOriginToCenter = srcTextBox.PerCharacterTransformOriginToCenter;

            dstTexBox.perCharacterTransformFixSpaceEnabled = srcTextBox.IsPerCharTransformFixSpaceEnabled;
            dstTexBox.perCharacterTransformFixSpaceWidth = srcTextBox.PerCharTransformFixSpaceWidth;
            dstTexBox.perCharacterTransformFixSpaceOrigin = Convert_HorizontalPosition_(srcTextBox.PerCharTransformFixSpaceOrigin);

            dstTexBox.perCharacterTransformEasyCircleBeginAngle = srcTextBox.PerCharTransformEasyCircleBeginAngle;
            dstTexBox.perCharacterTransformEasyCircleEndAngle = srcTextBox.PerCharTransformEasyCircleEndAngle;
            dstTexBox.perCharacterTransformEasyCircleRadiusX = srcTextBox.PerCharTransformEasyCircleRadiusX;
            dstTexBox.perCharacterTransformEasyCircleRadiusY = srcTextBox.PerCharTransformEasyCircleRadiusY;
            dstTexBox.perCharacterTransformEasyCircleDivisionNumber = srcTextBox.PerCharTransformEasyCircleDivisionNumber;
            dstTexBox.perCharacterTransformEasyCircleCharAngle = srcTextBox.PerCharTransformEasyCircleCharAngle;

            // 文字毎の変換の簡単設定
            if (srcTextBox.EasySettingOfPerCharTransform == null || srcTextBox.EasySettingOfPerCharTransform.Length == 0)
            {
                dstTexBox.easySettingOfPerCharTransform = null;
            }
            else
            {
                int length = srcTextBox.EasySettingOfPerCharTransform.Length;
                dstTexBox.easySettingOfPerCharTransform = new FileFmt.EasySettingOfPerCharTransform[length];
                for (int i = 0; i < length; i++)
                {
                    dstTexBox.easySettingOfPerCharTransform[i] = new LECore.Structures.SerializableObject.Lyt.EasySettingOfPerCharTransform();
                    dstTexBox.easySettingOfPerCharTransform[i].easySettingMoveFromLeft = srcTextBox.EasySettingOfPerCharTransform[i].easySettingMoveFromLeft;
                    dstTexBox.easySettingOfPerCharTransform[i].easySettingCalcEndFrame = srcTextBox.EasySettingOfPerCharTransform[i].easySettingCalcEndFrame;
                    dstTexBox.easySettingOfPerCharTransform[i].easySettingBeginFrame = srcTextBox.EasySettingOfPerCharTransform[i].easySettingBeginFrame;
                    dstTexBox.easySettingOfPerCharTransform[i].easySettingEndFrame = srcTextBox.EasySettingOfPerCharTransform[i].easySettingEndFrame;
                    dstTexBox.easySettingOfPerCharTransform[i].easySettingStopFrameOfFirstChar = srcTextBox.EasySettingOfPerCharTransform[i].easySettingStopFrameOfFirstChar;
                    dstTexBox.easySettingOfPerCharTransform[i].easySettingStartFrameOfLastChar = srcTextBox.EasySettingOfPerCharTransform[i].easySettingStartFrameOfLastChar;
                }
            }

            dstTexBox.detailSetting = true; // 必ず有効

            // 縁どりOFF
            dstTexBox.invisibleBorderEnabled = srcTextBox.InvisibleBorderEnabled;
            dstTexBox.doubleDrawnBorderEnabled = srcTextBox.DoubleDrawnBorderEnabled;

            if (forceOutShadowParamater || srcTextBox.NeedToOutShadowParamaers())
            {
                dstTexBox.shadowBlackColor = Convert_Color4_(srcTextBox.ShadowBlackColor);
                dstTexBox.shadowWhiteColor = Convert_Color4_(srcTextBox.ShadowWhiteColor);
            }

            // 拡張タグの解釈
            dstTexBox.extendedTagEnabled = srcTextBox.IsExtendedTagEnabled;

            return dstTexBox;
        }

        /// <summary>
        /// TextBoxペイン固有情報の書き出し
        /// </summary>
        FileFmt.TextBox Convert_Item_TextBoxImpl_(AppData.ITextBox srcTextBox, bool forceOutShadowParamater)
        {
            //-----------------------------------------
            // TextBox 自身の変換
            FileFmt.TextBox dstTextBox = Convert_Item_TextBoxSelf_(srcTextBox, forceOutShadowParamater);

            //-----------------------------------------
            // マテリアル情報の書き出し。
            dstTextBox.material = Convert_ToFileMaterial_( srcTextBox.IMaterial );

            //-----------------------------------------
            // Revolution マテリアル情報の書き出し。
            if( srcTextBox is IRevHWMaterialHolder )
            {
                IRevHWMaterialHolder revHWMaterialHolder =
                    srcTextBox as IRevHWMaterialHolder;

                Debug.Assert( revHWMaterialHolder.IRevHWMaterial.Length == 1 );
                IRevHWMaterial revMat = revHWMaterialHolder.IRevHWMaterial[0];
                dstTextBox.materialCtr = Convert_ToFileMaterial_Revo_( revMat );
            }

            return dstTextBox;
        }

        /// <summary>
        /// TextBoxペイン固有情報の書き出し
        /// </summary>
        FileFmt.TextBox Convert_Item_TextBox_( AppData.IPane pane )
        {
            AppData.ITextBox srcTextBox = pane.ITextBox;

            // 指定があれば、フォントが正しく設定されていない場合は保存を中断します。
            if (this.SaveOptions.ConvertInvalidTextboxToNull)
            {
                if (!LEFontHelper.IsComplexFont(srcTextBox.ILEFont) && LEFontHelper.IsFileNotFound(srcTextBox.ILEFont))
                {
                    string message = String.Format("Invalid Font [Pane = {0}]", pane.PaneName);
                    throw new Exception(message);
                }
            }

            return this.Convert_Item_TextBoxImpl_(srcTextBox, false);
        }


        #endregion TextBox

        #region Picture
        /// <summary>
        /// テクスチャ座標情報の書き出し
        /// </summary>
        FileFmt.TexCoord[] Convert_TexCoord_( AppData.IPicture srcPicture )
        {
            TexCoord[] dstTexCoords = new TexCoord[srcPicture.NumTexCoord];

            for( int i = 0 ; i < dstTexCoords.Length ; i++ )
            {
                TexCoord4 srcTexCoord4 = srcPicture.GetTexCoord4( i );
                dstTexCoords[i] = Convert_TexCoord_( srcTexCoord4 );
            }

            return dstTexCoords;
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.Picture Convert_Item_PictureSelf_( AppData.IPicture srcPicture )
        {
            Debug.Assert( srcPicture != null );

            FileFmt.Picture dstPicture = new FileFmt.Picture();

            dstPicture.vtxColLT = Convert_Color4_( srcPicture.VtxCol_LT );
            dstPicture.vtxColRT = Convert_Color4_( srcPicture.VtxCol_RT );
            dstPicture.vtxColLB = Convert_Color4_( srcPicture.VtxCol_LB );
            dstPicture.vtxColRB = Convert_Color4_( srcPicture.VtxCol_RB );

            // テクスチャ座標情報の書き出し
            dstPicture.texCoord = Convert_TexCoord_( srcPicture );

            // 詳細なマテリアル設定を行うか？
            // フラグは廃止され、必ず true にして出力するように変更になりました。
            dstPicture.detailSetting = true;

            switch (srcPicture.ShapeType)
            {
                case ShapeType.NormalQuad:
                    break;
                case ShapeType.GfxPrimitiveCircle:
                case ShapeType.GfxPrimitiveRoundRect:
                    dstPicture.shapeType = srcPicture.ShapeType;
                    dstPicture.shapeParam = new ShapeParam();
                    dstPicture.shapeParam.circleSlice = srcPicture.CircleSlice;
                    dstPicture.shapeParam.sphereSlice = srcPicture.SphereSlice;
                    dstPicture.shapeParam.sphereStack = srcPicture.SphereStack;
                    dstPicture.shapeParam.roundRadius = srcPicture.RoundRadius;
                    dstPicture.shapeParam.roundSlice = srcPicture.RoundSlice;
                    break;
                default:
                    Debug.Assert(false);
                    break;
            }

            return dstPicture;
        }

        /// <summary>
        /// Pictureペイン固有情報の書き出し
        /// </summary>
        FileFmt.Picture Convert_Item_PictureImpl_( AppData.IPicture srcPicture )
        {
            FileFmt.Picture dstPicture = Convert_Item_PictureSelf_( srcPicture );

            // マテリアル
            dstPicture.material = Convert_ToFileMaterial_( srcPicture.IMaterial );

            // Revolutionマテリアル
            IRevHWMaterialHolder revPicture = srcPicture as IRevHWMaterialHolder;
            dstPicture.materialCtr = Convert_ToFileMaterial_Revo_( revPicture.IRevHWMaterial[0] );

            // プロシージャル形状情報
            dstPicture.proceduralShapeParam = Convert_ToFile_ProceduralShape_(srcPicture.IProceduralShape);

            return dstPicture;
        }

        /// <summary>
        /// Pictureペイン固有情報の書き出し
        /// </summary>
        FileFmt.Picture Convert_Item_Picture_( AppData.IPane pane )
        {
            return Convert_Item_PictureImpl_(pane.IPicture);
        }

        #endregion

        #region Windowペイン関連
        /// <summary>
        /// LEWindowFrameFlipRotate => FileFmt.TextureFlip
        /// 強引にキャストしています。
        /// </summary>
        FileFmt.TextureFlip
            Convert_Item_TextureFlip_( LEWindowFrameFlipRotate srcFlipRotState )
        {
            return (FileFmt.TextureFlip)srcFlipRotState;
        }

        /// <summary>
        /// Windowペインのフレーム種類型を変換します。
        /// </summary>
        WindowFrameType
            Convert_Item_LEWindowPartsID_( AppData.LEWindowPartsID frameType )
        {
            switch( frameType )
            {
                case AppData.LEWindowPartsID.CornerLT: return WindowFrameType.CornerLT;
                case AppData.LEWindowPartsID.CornerRT: return WindowFrameType.CornerRT;
                case AppData.LEWindowPartsID.CornerLB: return WindowFrameType.CornerLB;
                case AppData.LEWindowPartsID.CornerRB: return WindowFrameType.CornerRB;
                case AppData.LEWindowPartsID.FrameT: return WindowFrameType.FrameT;
                case AppData.LEWindowPartsID.FrameB: return WindowFrameType.FrameB;
                case AppData.LEWindowPartsID.FrameL: return WindowFrameType.FrameL;
                case AppData.LEWindowPartsID.FrameR: return WindowFrameType.FrameR;
                default: Debug.Assert( false ); return WindowFrameType.CornerLT;
            }
        }

        /// <summary>
        /// ウインドウフレーム情報配列を書き出します。
        /// </summary>
        FileFmt.WindowFrame[]
            Convert_Item_WindowFrameSet_( AppData.ILEWindow srcWindow )
        {

            int numFrameTextures = srcWindow.NumTexture - 1;
            FileFmt.WindowFrame[] dstWindowFrameSet = new FileFmt.WindowFrame[numFrameTextures];

            for( int i = 0 ; i < dstWindowFrameSet.Length ; i++ )
            {
                int frameIndex = i + 1;

                dstWindowFrameSet[i] = new FileFmt.WindowFrame();

                LEWindowPartsID partsID = srcWindow.IntToPartsID( frameIndex );

                dstWindowFrameSet[i].frameType = Convert_Item_LEWindowPartsID_( partsID );

                // 詳細なマテリアル設定を行うか？
                // フラグは廃止され、必ず true にして出力するように変更になりました。
                dstWindowFrameSet[i].detailSetting = true;

                dstWindowFrameSet[i].textureFlip = new FileFmt.TextureFlip[1];
                dstWindowFrameSet[i].textureFlip[0] = Convert_Item_TextureFlip_( srcWindow.GetMaterialFlipRotateState( partsID ) );
            }
            return dstWindowFrameSet;
        }

        /// <summary>
        /// Windowペインコンテンツ領域：拡大量情報の書き出し。
        /// </summary>
        FileFmt.InflationRect
            Convert_Item_InflationRect_( AppData.ILEWindowContentsInflation srcInflation )
        {
            FileFmt.InflationRect dstInflation = new FileFmt.InflationRect();

            dstInflation.t = srcInflation.Top;
            dstInflation.b = srcInflation.Bottom;
            dstInflation.l = srcInflation.Left;
            dstInflation.r = srcInflation.Right;

            return dstInflation;
        }

        /// <summary>
        /// Windowペインコンテンツ領域：拡大量情報の書き出し。
        /// </summary>
        FileFmt.WindowFrameSize
            Convert_Item_InflationRect_(AppData.LEWindowFrameSize frameSize)
        {
            FileFmt.WindowFrameSize dst = new FileFmt.WindowFrameSize();

            dst.t = frameSize.Top;
            dst.b = frameSize.Bottom;
            dst.l = frameSize.Left;
            dst.r = frameSize.Right;

            return dst;
        }

        /// <summary>
        /// Windowペインコンテンツ領域情報の書き出し。
        /// </summary>
        FileFmt.WindowContent
            Convert_Item_WindowContent_( AppData.ILEWindow srcWindow )
        {
            FileFmt.WindowContent dstContent = new FileFmt.WindowContent();

            dstContent.vtxColLT = Convert_Color4_(srcWindow.VtxCol_LT);
            dstContent.vtxColRT = Convert_Color4_(srcWindow.VtxCol_RT);
            dstContent.vtxColLB = Convert_Color4_(srcWindow.VtxCol_LB);
            dstContent.vtxColRB = Convert_Color4_(srcWindow.VtxCol_RB);

            // 詳細なマテリアル設定を行うか？
            // フラグは廃止され、必ず true にして出力するように変更になりました。
            dstContent.detailSetting = true;

            // マテリアル情報を書き出します。
            dstContent.material = Convert_ToFileMaterial_( srcWindow.GetMaterial( LEWindowPartsID.Content ) );

            // テクスチャ座標を書き出します。
            // テクスチャが張られていない場合は、nullが代入されます。
            dstContent.texCoord = new TexCoord[srcWindow.NumContentsTexCoord];
            for( int i = 0 ; i < dstContent.texCoord.Length ; i++ )
            {
                TexCoord4 srcTexCoord4 = srcWindow.GetContentsTexCoord4( i );
                dstContent.texCoord[i] = Convert_TexCoord_( srcTexCoord4 );
            }

            return dstContent;
        }

        /// <summary>
        /// Windowペイン固有情報の書き出し
        /// </summary>
        FileFmt.Window Convert_Item_WindowSelf_( ILEWindow srcWindow )
        {
            FileFmt.Window dstWindow = new FileFmt.Window();

            dstWindow.content = Convert_Item_WindowContent_( srcWindow );
            dstWindow.contentInflation = Convert_Item_InflationRect_( srcWindow.ContentsInflation );
            dstWindow.frame = Convert_Item_WindowFrameSet_( srcWindow );
            dstWindow.kind = srcWindow.WindowKind;
            dstWindow.useVtxColorForAllWindow = srcWindow.UseVtxColorForAllWindow;

            FileFmt.UseLTMaterial useLTMaterial = new FileFmt.UseLTMaterial();
            useLTMaterial.value = srcWindow.UseLTMaterial;
            dstWindow.useLTMaterial = useLTMaterial;

            dstWindow.frameSize = srcWindow.IsFrameSizeOrFrameSizeCurveSpecified ? Convert_Item_InflationRect_(srcWindow.FrameSize) : null;

            dstWindow.notDrawContent = srcWindow.NotDrawContent;

            return dstWindow;
        }

        /// <summary>
        /// Windowペイン固有情報の書き出し
        /// </summary>
        FileFmt.Window Convert_Item_Window_( AppData.IPane pane )
        {
            ILEWindow			srcWindow = pane.ILEWindow;
            FileFmt.Window		dstWindow = Convert_Item_WindowSelf_( srcWindow );

            //--------------------------------------------
            // マテリアル 関連の処理。
            // コンテンツ領域のマテリアル
            // (Texture座標数を読み込む都合から、Convert_Item_WindowSelf_()で読み込みます。)
            // dstWindow.content.material = Convert_ToFileMaterial( srcWindow.GetMaterial( LEWindowPartsID.Content ) );

            // コンテンツ領域の詳細マテリアル
            IRevHWMaterialHolder srcRevWindow = srcWindow as IRevHWMaterialHolder;
            dstWindow.content.materialCtr = Convert_ToFileMaterial_Revo_( srcRevWindow.IRevHWMaterial[srcWindow.PartsIDToInt( LEWindowPartsID.Content )] );

            // フレーム領域のマテリアル
            for( int i = 0 ; i < dstWindow.frame.Length ; i++ )
            {
                int frameIndex = i + 1;
                LEWindowPartsID partsID = srcWindow.IntToPartsID( frameIndex );
                // マテリアル関連
                dstWindow.frame[i].material = Convert_ToFileMaterial_( srcWindow.GetMaterial( partsID ) );
                dstWindow.frame[i].materialCtr = Convert_ToFileMaterial_Revo_( srcRevWindow.IRevHWMaterial[frameIndex] );
            }

            return dstWindow;
        }

        #endregion Windowペイン関連

        #region Boundingペイン関連
        /// <summary>
        /// Boundingペイン固有情報の書き出し
        /// </summary>
        FileFmt.Bounding Convert_Item_Bounding_( AppData.IPane pane )
        {
            FileFmt.Bounding dstBounding = new FileFmt.Bounding();
            // TODO:
            return dstBounding;
        }
        #endregion Boundingペイン関連

        #region Partsペイン関連
        /// <summary>
        /// Partsペイン固有情報の書き出し
        /// </summary>
        private FileFmt.Parts Convert_Item_Parts_( AppData.IPane pane )
        {
            Ensure.Argument.NotNull(pane.IPartsLayout);
            return Convert_Item_Parts_Impl_(pane.IPartsLayout);
        }

        /// <summary>
        /// Partsペインプロパティの書き出し
        /// </summary>
        private FileFmt.Property [] Convert_Item_PartsProperties_(
            AppData.IPartsLayout partsLayout,
            IEnumerable<IPartsPropaerty> srcPartsPropaerties)
        {
            List<FileFmt.Property> dstPoperties = new List<Property>();
            foreach (var partsPropaerties in srcPartsPropaerties)
            {
                FileFmt.Property property = new FileFmt.Property();
                property.kind = Convert_SOPaneKind_(partsPropaerties.PaneKind);
                if (partsPropaerties.Paramater != null)
                {
                    try
                    {
                        if (partsPropaerties.Paramater.OwnerPane != null)
                        {
                            // pos
                            if ((partsPropaerties.Option & AppData.PartsPropertyUsageOptions.UsePanePosition) != 0)
                            {
                                property.translate = Convert_Vec3_(partsPropaerties.Paramater.OwnerPane.Trans);
                            }

                            // size
                            if ((partsPropaerties.Option & AppData.PartsPropertyUsageOptions.UsePaneSize) != 0)
                            {
                                property.size = Convert_Vec2_(partsPropaerties.Paramater.OwnerPane.Size);
                            }

                            // scale
                            if ((partsPropaerties.Option & AppData.PartsPropertyUsageOptions.UsePaneScale) != 0)
                            {
                                property.scale = Convert_Vec2_(partsPropaerties.Paramater.OwnerPane.Scale);
                            }

                            // rot
                            if ((partsPropaerties.Option & AppData.PartsPropertyUsageOptions.UsePaneRotate) != 0)
                            {
                                property.rotate = Convert_Vec3_(partsPropaerties.Paramater.OwnerPane.RotAng);
                            }

                            // 透明度
                            if ((partsPropaerties.Option & AppData.PartsPropertyUsageOptions.UsePaneTransparency) != 0)
                            {
                                property.alpha = partsPropaerties.Paramater.OwnerPane.Transparency;
                            }

                            // 表示・非表示
                            if ((partsPropaerties.Option & AppData.PartsPropertyUsageOptions.UsePaneVisibility) != 0)
                            {
                                property.visible = partsPropaerties.Paramater.OwnerPane.Visible;
                            }

                            // 基本ユーザデータ
                            if ((partsPropaerties.Option & AppData.PartsPropertyUsageOptions.UsePaneBasicUserData) != 0)
                            {
                                property.basicUserData = partsPropaerties.Paramater.OwnerPane.UserData;
                            }

                            // 拡張ユーザデータ
                            if ((partsPropaerties.Option & AppData.PartsPropertyUsageOptions.UsePaneExUserData) != 0)
                            {
                                var userData = Convert_PaneUserData_(partsPropaerties.Paramater.OwnerPane);
                                int numDstLength = userData.Length - 1;
                                Debug.Assert(numDstLength >= 0);

                                property.userData = new object[numDstLength];
                                Array.Copy(userData, 1, property.userData, 0, numDstLength);
                            }
                        }

                        if (PartsPropertyUsageOptionsHelper.IsPaneExDataOverwrote(partsPropaerties.Option))
                        {
                            switch (partsPropaerties.PaneKind)
                            {
                                case LECore.Structures.PaneKind.Picture:
                                    property.Item = Convert_Item_PictureImpl_(partsPropaerties.Paramater as IPicture);
                                    break;
                                case LECore.Structures.PaneKind.Textbox:

                                    IPane originalPane = partsLayout.GetOriginalPane(partsPropaerties.TargetName);
                                    bool forceOutShadowParamater =
                                        originalPane != null &&
                                        originalPane.ITextBox != null &&
                                        originalPane.ITextBox.HasShadowParamaerAnimation();

                                    property.Item = Convert_Item_TextBoxImpl_(partsPropaerties.Paramater as ITextBox, forceOutShadowParamater);
                                    break;
                                case LECore.Structures.PaneKind.Window:
                                    property.Item = Convert_Item_Window_(partsPropaerties.Paramater.OwnerPane);
                                    break;
                                case LECore.Structures.PaneKind.Capture:
                                    property.Item = Convert_SOCaptureTexture_((partsPropaerties.Paramater as AppData.ICapture).OwnTexture);
                                    break;
                                case LECore.Structures.PaneKind.Parts:
                                    // 再帰的に書き出します。
                                    property.Item = Convert_Item_Parts_Impl_(partsPropaerties.Paramater as AppData.IPartsLayout);
                                    break;
                                default:
                                    Debug.Assert(false);
                                    break;
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        string title = string.Format("{0}：{1} = {2} ",
                            LECoreStringResMgr.Get("LECORE_ERROR_PARTSPANESAVING_PARAM"),
                            LECoreStringResMgr.Get("LECORE_TARGETPANE"),
                            partsPropaerties.TargetName);
                        string message = LECoreStringResMgr.Get("LECORE_ERROR_PARTSPANESAVING_PARAM_ABORTED");

                        _msgReporter.ReportError(title, message, e.Message);

                        // 環境変数が定義されていれば詳細をレポートする。
                        if (!string.IsNullOrEmpty(Environment.GetEnvironmentVariable("NW_LYT_REPORT_EXCEPTION_DETAIL")))
                        {
                            ReportErrorMsg_(e.ToString());
                        }
                    }
                }

                property.target = partsPropaerties.TargetName;
                property.usageOptions = Convert_PartsPropertyUsageOptions_(partsPropaerties.Option);

                dstPoperties.Add(property);
            }

            return dstPoperties.ToArray();
        }

        /// <summary>
        /// Partsペイン固有情報の書き出し
        /// </summary>
        private FileFmt.Parts Convert_Item_Parts_Impl_(AppData.IPartsLayout partsLayout)
        {
            FileFmt.Parts dstParts = new FileFmt.Parts();

            dstParts.path = partsLayout.PartsLayoutName;
            dstParts.property = Convert_Item_PartsProperties_(partsLayout, partsLayout.PartsPropaerties);

            // GetAdjustmentMagnify
            dstParts.magnify = new Vec2();
            dstParts.rawMagnify = new Vec2();
            dstParts.sizeConstraint = new Vec2();

            var adjustmentMagnify = (partsLayout as PartsLayout).GetAdjustmentMagnifyForFileSave();

            dstParts.magnify.x = adjustmentMagnify.X;
            dstParts.magnify.y = adjustmentMagnify.Y;
            dstParts.rawMagnify.x = partsLayout.Magnify.X;
            dstParts.rawMagnify.y = partsLayout.Magnify.Y;
            dstParts.sizeConstraint.x = partsLayout.SizeConstraint.X;
            dstParts.sizeConstraint.y = partsLayout.SizeConstraint.Y;

            return dstParts;
        }

        #endregion Parts ペイン関連

        #region Captureペイン関連
        /// <summary>
        ///
        /// </summary>
        FileFmt.Capture Convert_Item_CaptureSelf_(AppData.ICapture srcCapture)
        {
            Debug.Assert(srcCapture != null);

            FileFmt.Capture dstCapture = new FileFmt.Capture();

            return dstCapture;
        }



        /// <summary>
        /// Capture ペイン固有情報の書き出し
        /// </summary>
        FileFmt.Capture Convert_Item_CaptureImpl_(AppData.ICapture srcCapture)
        {
            FileFmt.Capture dstCapture = Convert_Item_CaptureSelf_(srcCapture);

            return dstCapture;
        }

        /// <summary>
        /// Capture ペイン固有情報の書き出し
        /// </summary>
        FileFmt.Capture Convert_Item_Capture_(AppData.IPane pane)
        {
            return Convert_Item_CaptureImpl_(pane.ICapture);
        }

        #endregion Captureペイン関連

        #region Alignment ペイン関連
        /// <summary>
        ///
        /// </summary>
        FileFmt.Alignment Convert_Item_AlignmentSelf_(AppData.IAlignment srcAlignment)
        {
            Debug.Assert(srcAlignment != null);

            FileFmt.Alignment dst = new FileFmt.Alignment();

            dst.horizontal = (FileFmt.HorizontalPosition)srcAlignment.Horizontal;
            dst.vertical = (FileFmt.VerticalPosition)srcAlignment.Vertical;
            dst.defaultMargin = srcAlignment.DefaultHorizontalMargin;
            dst.defaultVerticalMargin = srcAlignment.DefaultVerticalMargin;
            dst.isExtendEdgeEnabled = srcAlignment.IsExtendEdgeHorizontalEnabled;
            dst.isExtendEdgeVerticalEnabled = srcAlignment.IsExtendEdgeVerticalEnabled;
            dst.isAlignmentHorizontalEnabled = srcAlignment.IsAlignmentHorizontalEnabled;
            dst.isAlignmentVerticalEnabled = srcAlignment.IsAlignmentVerticalEnabled;
            return dst;
        }

        /// <summary>
        /// Capture ペイン固有情報の書き出し
        /// </summary>
        FileFmt.Alignment Convert_Item_AlignmentImpl_(AppData.IAlignment srcAlignment)
        {
            FileFmt.Alignment dstAlignment = Convert_Item_AlignmentSelf_(srcAlignment);

            return dstAlignment;
        }

        FileFmt.Alignment Convert_Item_Alignment_(AppData.IPane pane)
        {
            return Convert_Item_AlignmentImpl_(pane.IAlignment);
        }
        #endregion

        #region Scissor ペイン関連
        /// <summary>
        ///
        /// </summary>
        FileFmt.Scissor Convert_Item_ScissorSelf_(AppData.IScissor srcScissor)
        {
            Debug.Assert(srcScissor != null);

            FileFmt.Scissor dst = new FileFmt.Scissor();
            return dst;
        }

        /// <summary>
        /// Capture ペイン固有情報の書き出し
        /// </summary>
        FileFmt.Scissor Convert_Item_ScissorImpl_(AppData.IScissor srcScissor)
        {
            FileFmt.Scissor dstScissor = Convert_Item_ScissorSelf_(srcScissor);

            return dstScissor;
        }

        FileFmt.Scissor Convert_Item_Scissor_(AppData.IPane pane)
        {
            return Convert_Item_ScissorImpl_(pane.IScissor);
        }
        #endregion

        #region AnimationShare ペイン関連


        #endregion AnimationShare ペイン関連

        /// <summary>
        /// 各種ペイン(ピクチャ、レターペインなど)に固有なパラメータを
        /// 格納するクラスへと変換します。
        /// </summary>
        object Convert_Item_( AppData.IPane srcPane )
        {
            // 各種ペインに固有な情報の設定
            switch( srcPane.PaneKind )
            {
                case AppData.PaneKind.Null: /* DO NOTHING */ return null;
                case AppData.PaneKind.Textbox: return Convert_Item_TextBox_( srcPane );
                case AppData.PaneKind.Picture: return Convert_Item_Picture_( srcPane );
                case AppData.PaneKind.Window: return Convert_Item_Window_( srcPane );
                case AppData.PaneKind.Bounding: return Convert_Item_Bounding_( srcPane );
                case AppData.PaneKind.Parts: return Convert_Item_Parts_( srcPane );
                case AppData.PaneKind.Capture: return Convert_Item_Capture_( srcPane );
                case AppData.PaneKind.Alignment: return Convert_Item_Alignment_(srcPane);
                case AppData.PaneKind.Scissor: return Convert_Item_Scissor_(srcPane);
                default: Debug.Assert( false ); return null;
            }
        }
        #endregion 各種ペイン固有パラメータの書き出し

        /// <summary>
        /// ユーザデータ配列を作成します。
        /// </summary>
        object[] Convert_PaneUserData_( AppData.IPane pane )
        {
            List<object> userDataSet = new List<object>();

            //--------------------------------
            // 標準の要素を抽出
            // ランタイムで固定長サイズ確保される概念なので空でも
            // 必ず出力するようにします。
            UserDataString basicUserDataStr = new UserDataString();
            basicUserDataStr.Value = pane.UserData;
            basicUserDataStr.name = _BasicUserDataStringName;
            userDataSet.Add(basicUserDataStr);

            userDataSet.AddRange(Convert_UserDataEx_(pane.IUserDataHolder.UserDataElementSet));

            return userDataSet.ToArray();
        }

        static object[] Convert_UserDataEx_(AppData.IUserDataElement[] srcUserDataSet)
        {
            return Convert_PaneUserDataEx_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(srcUserDataSet);
        }

        /// <summary>
        /// ユーザデータ配列を作成します。
        /// </summary>
        internal static object[] Convert_PaneUserDataEx_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(AppData.IUserDataElement[] srcUserDataSet)
            where UserDataFloatList : IUserDataFloatList<UserDataElementType>, new()
            where UserDataIntList : IUserDataIntList<UserDataElementType>, new()
            where UserDataNone : IUserDataNone<UserDataElementType>, new()
            where UserDataString : IUserDataString<UserDataElementType>, new()
        {
            List<object> userDataSet = new List<object>();

            //--------------------------------
            // 拡張ユーザデータを抽出
            foreach (IUserDataElement userData in srcUserDataSet)
            {
                switch (userData.UserDataKind)
                {
                    case UserDataKind.None:
                        {
                            UserDataNone udn = new UserDataNone();
                            udn.name = userData.Name;
                            udn.Value = null;
                            udn.kind = (UserDataElementType)Enum.Parse(typeof(UserDataElementType), userData.UserDataKind.ToString());
                            udn.kindSpecified = true;
                            udn.overwrite = userData.Overwrite;
                            userDataSet.Add(udn);
                            break;
                        }
                    case UserDataKind.String:
                        {
                            UserDataString uds = new UserDataString();
                            uds.name = userData.Name;
                            uds.Value = UserDataElementHelper.ToString(userData);
                            uds.kind = (UserDataElementType)Enum.Parse(typeof(UserDataElementType), userData.UserDataKind.ToString());
                            uds.kindSpecified = true;
                            uds.overwrite = userData.Overwrite;
                            userDataSet.Add(uds);
                            break;
                        }
                    case UserDataKind.Float:
                    case UserDataKind.AnmFloat:
                    case UserDataKind.AnmFloatVec2:
                    case UserDataKind.AnmFloatVec3:
                        {
                            UserDataFloatList udf = new UserDataFloatList();
                            udf.name = userData.Name;
                            udf.Value = UserDataElementHelper.ToString(userData);
                            udf.kind = (UserDataElementType)Enum.Parse(typeof(UserDataElementType), userData.UserDataKind.ToString());
                            udf.kindSpecified = true;
                            udf.overwrite = userData.Overwrite;
                            userDataSet.Add(udf);
                            break;
                        }
                    case UserDataKind.Int:
                    case UserDataKind.AnmInt:
                    case UserDataKind.AnmIntVec2:
                    case UserDataKind.AnmByteRGBA4:
                    case UserDataKind.AnmByteRGBA4Degamma:
                        {
                            UserDataIntList udi = new UserDataIntList();
                            udi.name = userData.Name;
                            udi.Value = UserDataElementHelper.ToString(userData);
                            udi.kind = (UserDataElementType)Enum.Parse(typeof(UserDataElementType), userData.UserDataKind.ToString());
                            udi.kindSpecified = true;
                            udi.overwrite = userData.Overwrite;
                            userDataSet.Add(udi);
                            break;
                        }
                    default:
                        {
                            Debug.Assert(false);
                            break;
                        }
                }
            }

            return userDataSet.ToArray();
        }

        /// <summary>
        /// FileFmt.PaneModify へと変換します。
        /// </summary>
        FileFmt.Pane Convert_SOPane( AppData.IPane pane )
        {
            FileFmt.Pane result = new FileFmt.Pane();

            result.translate = Convert_Vec3_( pane.Trans );
            result.rotate = Convert_Vec3_( pane.RotAng );

            result.scale = Convert_Vec2_( pane.Scale );
            result.size = Convert_Vec2_( pane.Size );

            result.alpha = pane.Transparency;

            result.kind = Convert_SOPaneKind_( pane.PaneKind );

            result.basePositionType = Convert_Position_( pane.BasePosTypeH, pane.BasePosTypeV );
            result.parentRelativePositionType = Convert_Position_(pane.ParentBasePosTypeH, pane.ParentBasePosTypeV);

            result.name = pane.PaneName;

            result.comment = pane.UserCommentString;
            result.userData = Convert_PaneUserData_( pane );
            result.visible = pane.Visible;

            result.locked = pane.Locked;
            result.hidden = pane.Hidden;
            result.influencedAlpha = pane.InfluenceChildrenTransparency;
            result.locationAdjust = pane.LocationAdjust;

            result.alignmentMarginEnabled = pane.IsAlignmentHorizontalMarginEnabled;
            result.alignmentMargin = pane.AlignmentHorizontalMargin;
            result.alignmentVerticalMargin = pane.AlignmentVerticalMargin;
            result.alignmentIgnore = pane.IsAlignmentIgnore;
            result.alignmentVerticalMarginEnabled = pane.IsAlignmentVerticalMarginEnabled;
            result.alignmentVerticalMargin = pane.AlignmentVerticalMargin;

            result.partsMagnifyInfluence = Convert_PartsMagnifyInfluence_(pane.PartsMagnifyInfluence);
            result.ignorePartsMagnify = false;
            result.readonlyLocked = pane.IsReadOnlyLocked;
            result.avoidPaneTreeCompression = pane.AvoidPaneTreeCompression;
            result.extUserDataAnimEnabled = pane.ExtUserDataAnimEnabled;

            result.mask = Convert_ToFile_Mask_(pane.IMask);
            result.dropShadow = Convert_ToFile_DropShadow_(pane.IDropShadow);

            // TODO:ピクチャ、レターペインの固有情報
            try
            {
                result.Item = Convert_Item_( pane );
            }
            catch( Exception e )
            {
                // Nullペインとして保存します。
                result.kind = FileFmt.PaneKind.Null;

                string title = LECoreStringResMgr.Get( "LECORE_RLYTSAVE_PANE" );
                string message = LECoreStringResMgr.Get( "LECORE_RLYTSAVE_CVTTONULL" ) + " : {0}";

                _msgReporter.ReportError( title, message, e.Message );

                // 環境変数が定義されていれば詳細をレポートする。
                if (!string.IsNullOrEmpty(Environment.GetEnvironmentVariable("NW_LYT_REPORT_EXCEPTION_DETAIL")))
                {
                    ReportErrorMsg_(e.ToString());
                }
            }

            return result;
        }

        /// <summary>
        /// FileFmt.PaneModify[] へと変換します。
        /// </summary>
        FileFmt.Pane[] Convert_SOPaneArray( ISubScene scene )
        {
            OutLog_( "LECORE_RLYTSAVE_PANE" );

            List<FileFmt.Pane> result = new List<FileFmt.Pane>();

            foreach( AppData.IPane srcPane in scene.IPaneArray )
            {
                try
                {
                    _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_PANE") + string.Format(" [{0}]", srcPane.PaneName));
                    result.Add(Convert_SOPane(srcPane));
                }
                catch (Exception e)
                {
                    string title = LECoreStringResMgr.Get("LECORE_RLYTSAVE_PANE");
                    _msgReporter.ReportError(title, e.Message);
                }
                finally
                {
                    _msgReporter.EndPacking();
                }
            }

            return result.ToArray();
        }

        #endregion ペイン関連

        #endregion

        #region File => App

        #region ペイン関連

        #region 各種ペインに固有なパラメータの読み込み

        /// <summary>
        /// AppData.TextBox に変換します。
        /// </summary>
        AppData.TextBox Convert_Pane_TextBoxSelf_(AppData.Pane dstPane, FileFmt.TextBox srcTextBox)
        {
            string contents = SerializableConverterUtil.ReplaceLineFeedCode(srcTextBox.text);
            AppData.TextBox dstTextBox = new AppData.TextBox(dstPane, contents, srcTextBox.font);

            dstTextBox.FontSize = new FVec2(srcTextBox.fontSize.x, srcTextBox.fontSize.y);
            dstTextBox.FontSizeOriginal = (srcTextBox.fontSizeOriginal != null) ? new FVec2(srcTextBox.fontSizeOriginal.x, srcTextBox.fontSizeOriginal.y) : FVec2.Empty;
            dstTextBox.TextID = srcTextBox.textID;

            dstTextBox.CharSpace = srcTextBox.charSpace;
            dstTextBox.LineSpace = srcTextBox.lineSpace;

            dstTextBox.AllocStringLength = srcTextBox.allocateStringLength;
            dstTextBox.SpecifiedStringLength = srcTextBox.allocateStringLengthSpecified;

            dstTextBox.BasePosTypeH = Convert_HorizontalLocation_(srcTextBox.positionType);
            dstTextBox.BasePosTypeV = Convert_VerticalLocation_(srcTextBox.positionType);
            dstTextBox.LineAlignment = Convert_LineAlignment_(srcTextBox.textAlignment);
            dstTextBox.CenterCeilingEnabled = srcTextBox.centerCeilingEnabled;
            dstTextBox.DrawFromRightToLeft = srcTextBox.drawFromRightToLeft;

            dstTextBox.LineWidthOffsetEnabled = srcTextBox.lineWidthOffsetEnabled;
            for (int i = 0; i < ITextBoxHelper.LineOffsetWidthCountMax; i++)
            {
                if (srcTextBox.lineWidthOffset == null || i >= srcTextBox.lineWidthOffset.Length)
                {
                    dstTextBox.LineOffset[i] = 0.0f;
                    dstTextBox.LineWidth[i] = 0.0f;
                }
                else
                {
                    dstTextBox.LineOffset[i] = srcTextBox.lineWidthOffset[i].lineOffset;
                    dstTextBox.LineWidth[i] = srcTextBox.lineWidthOffset[i].lineWidth;
                }
            }

            dstTextBox.TopColor = Convert_RGBAColor_(srcTextBox.topColor);
            dstTextBox.BottomColor = Convert_RGBAColor_(srcTextBox.bottomColor);

            // 斜体
            dstTextBox.ItalicFactor = srcTextBox.italicFactor;

            // 影
            dstTextBox.ShadowEnabled = srcTextBox.shadowEnabled;

            dstTextBox.ShadowOffset = SetFVec2IfExist_(srcTextBox.shadowOffset, dstTextBox.ShadowOffset);
            dstTextBox.ShadowScale = SetFVec2IfExist_(srcTextBox.shadowScale, dstTextBox.ShadowScale);

            dstTextBox.ShadowItalicFactor = srcTextBox.shadowItalicFactor;
            dstTextBox.ShadowTopColor = SetRGBAColorIfExsit_(srcTextBox.shadowTopColor, dstTextBox.ShadowTopColor);
            dstTextBox.ShadowBottomColor = SetRGBAColorIfExsit_(srcTextBox.shadowBottomColor, dstTextBox.ShadowBottomColor);
            dstTextBox.PerCharacterTransformTimeOffset = srcTextBox.perCharacterTransformEvalTimeOffset;
            dstTextBox.PerCharacterTransformTimeWidth = srcTextBox.perCharacterTransformEvalTimeWidth;
            dstTextBox.PerCharacterTransformLoopType = Convert_PerCharTransformLoopType_(srcTextBox.perCharacterTransformLoopType);
            dstTextBox.PerCharacterTransformOriginV = Convert_VerticalLocation_(srcTextBox.perCharacterTransformOriginV);
            dstTextBox.PerCharacterTransformOriginVOffset = srcTextBox.perCharacterTransformOriginVOffset;
            dstTextBox.PerCharacterTransformSplitByCharWidth = srcTextBox.perCharacterTransformSplitByCharWidth;
            dstTextBox.PerCharacterTransformAutoShadowAlpha= srcTextBox.perCharacterTransformAutoShadowAlpha;
            dstTextBox.PerCharacterTransformOriginToCenter = srcTextBox.perCharacterTransformOriginToCenter;
            dstTextBox.IsPerCharTransformEnabled = srcTextBox.perCharacterTransformEnabled;

            dstTextBox.IsPerCharTransformFixSpaceEnabled = srcTextBox.perCharacterTransformFixSpaceEnabled;
            dstTextBox.PerCharTransformFixSpaceWidth = srcTextBox.perCharacterTransformFixSpaceWidth;
            dstTextBox.PerCharTransformFixSpaceOrigin = Convert_HorizontalLocation_(srcTextBox.perCharacterTransformFixSpaceOrigin);

            dstTextBox.PerCharTransformEasyCircleBeginAngle = srcTextBox.perCharacterTransformEasyCircleBeginAngle;
            dstTextBox.PerCharTransformEasyCircleEndAngle = srcTextBox.perCharacterTransformEasyCircleEndAngle;
            dstTextBox.PerCharTransformEasyCircleRadiusX = srcTextBox.perCharacterTransformEasyCircleRadiusX;
            dstTextBox.PerCharTransformEasyCircleRadiusY = srcTextBox.perCharacterTransformEasyCircleRadiusY;
            dstTextBox.PerCharTransformEasyCircleDivisionNumber = srcTextBox.perCharacterTransformEasyCircleDivisionNumber;
            dstTextBox.PerCharTransformEasyCircleCharAngle = srcTextBox.perCharacterTransformEasyCircleCharAngle;

            // 文字毎の変換の簡単設定
            if (srcTextBox.easySettingOfPerCharTransform == null || srcTextBox.easySettingOfPerCharTransform.Length == 0)
            {
                dstTextBox.EasySettingOfPerCharTransform = null;
            }
            else
            {
                int length = srcTextBox.easySettingOfPerCharTransform.Length;
                dstTextBox.EasySettingOfPerCharTransform = new AppData.EasySettingOfPerCharTransform[length];
                for (int i = 0; i < length; i++)
                {
                    dstTextBox.EasySettingOfPerCharTransform[i].easySettingMoveFromLeft = srcTextBox.easySettingOfPerCharTransform[i].easySettingMoveFromLeft;
                    dstTextBox.EasySettingOfPerCharTransform[i].easySettingCalcEndFrame = srcTextBox.easySettingOfPerCharTransform[i].easySettingCalcEndFrame;
                    dstTextBox.EasySettingOfPerCharTransform[i].easySettingBeginFrame = srcTextBox.easySettingOfPerCharTransform[i].easySettingBeginFrame;
                    dstTextBox.EasySettingOfPerCharTransform[i].easySettingEndFrame = srcTextBox.easySettingOfPerCharTransform[i].easySettingEndFrame;
                    dstTextBox.EasySettingOfPerCharTransform[i].easySettingStopFrameOfFirstChar = srcTextBox.easySettingOfPerCharTransform[i].easySettingStopFrameOfFirstChar;
                    dstTextBox.EasySettingOfPerCharTransform[i].easySettingStartFrameOfLastChar = srcTextBox.easySettingOfPerCharTransform[i].easySettingStartFrameOfLastChar;
                }
            }

            dstTextBox.ShadowBlackColor = SetRGBAColorIfExsit_(srcTextBox.shadowBlackColor, dstTextBox.ShadowBlackColor);
            dstTextBox.ShadowWhiteColor = SetRGBAColorIfExsit_(srcTextBox.shadowWhiteColor, dstTextBox.ShadowWhiteColor);

            // 縁取りOFF
            dstTextBox.InvisibleBorderEnabled = srcTextBox.invisibleBorderEnabled;

            // 2度書き縁取り
            dstTextBox.DoubleDrawnBorderEnabled = srcTextBox.doubleDrawnBorderEnabled;

            // 拡張タグの解釈
            dstTextBox.IsExtendedTagEnabled = LayoutEditorCore.ExtendedTagEnabled;

            return dstTextBox;
        }

        /// <summary>
        /// パラメータを部分的に読み込みます。
        /// </summary>
        void Convert_Pane_TextBoxTextAndID_(AppData.Pane dstPane, FileFmt.TextBox srcTexBoxData)
        {
            if (dstPane.TextBox != null)
            {
                dstPane.TextBox.ContentsText = SerializableConverterUtil.ReplaceLineFeedCode(srcTexBoxData.text);
                dstPane.TextBox.TextID = srcTexBoxData.textID;
            }
        }

        /// <summary>
        /// Nullペインを TextBox へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_TextBoxImpl_(AppData.Pane dstPane, FileFmt.TextBox srcTexBoxData)
        {
            AppData.TextBox textBox = Convert_Pane_TextBoxSelf_(dstPane, srcTexBoxData);

            // マテリアルの読み込み
            Convert_Pane_RevMaterial_(textBox.RevHWMaterial, srcTexBoxData.materialCtr);
            Convert_Pane_Material_(textBox.Material, srcTexBoxData.material);
        }

        /// <summary>
        /// Nullペインを TextBox へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_TextBox_(AppData.Pane dstPane, FileFmt.TextBox srcTexBoxData)
        {
            if( srcTexBoxData != null )
            {
                Convert_Pane_TextBoxImpl_(dstPane, srcTexBoxData);
            }
        }

        /// <summary>
        /// テクスチャ座標を読み込みます。
        /// </summary>
        void Convert_Pane_Picture_TexCoords_(
            AppData.Picture dstPicture,
            FileFmt.Picture srcPicture )
        {
            int NumTexCoord = srcPicture.texCoord.Length;
            dstPicture.NumTexCoord = NumTexCoord;
            for( int i = 0 ; i < NumTexCoord ; i++ )
            {
                TexCoord srcCoord = srcPicture.texCoord[i];
                TexCoord4 dstCoord = Convert_TexCoord4_( srcCoord );

                dstPicture.SetTexCoord4( dstCoord, i );
            }

        }

        /// <summary>
        /// Nullペインを Picture へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        AppData.Picture Convert_Pane_PictureSelf_( AppData.Pane dstPane, FileFmt.Picture srcPicture )
        {
            AppData.Picture dstPicture = new AppData.Picture( dstPane );

            dstPicture.VtxCol_LT = Convert_RGBAColor_( srcPicture.vtxColLT );
            dstPicture.VtxCol_RT = Convert_RGBAColor_( srcPicture.vtxColRT );
            dstPicture.VtxCol_LB = Convert_RGBAColor_( srcPicture.vtxColLB );
            dstPicture.VtxCol_RB = Convert_RGBAColor_( srcPicture.vtxColRB );

            // テクスチャ座標情報の読み込み
            Convert_Pane_Picture_TexCoords_( dstPicture, srcPicture );

            // 詳細なマテリアル設定を行うか？
            dstPicture.UseDetailedMaterial = srcPicture.detailSetting;

            dstPicture.ShapeType = srcPicture.shapeType;
            if (srcPicture.shapeParam != null)
            {
                dstPicture.CircleSlice = srcPicture.shapeParam.circleSlice;
                dstPicture.SphereSlice = srcPicture.shapeParam.sphereSlice;
                dstPicture.SphereStack = srcPicture.shapeParam.sphereSlice;
                dstPicture.RoundRadius = srcPicture.shapeParam.roundRadius;
                dstPicture.RoundSlice = srcPicture.shapeParam.roundSlice;
            }

            return dstPicture;
        }

        /// <summary>
        /// Nullペインを Picture へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_PictureImpl_(AppData.Pane dstPane, FileFmt.Picture srcPicture)
        {
            AppData.Picture dstPicture = Convert_Pane_PictureSelf_(dstPane, srcPicture);

            // マテリアルの読み込み
            if (srcPicture.detailSetting)
            {
                Convert_Pane_RevMaterial_(dstPicture.RevHWMaterial[0], srcPicture.materialCtr);
            }

            Convert_Pane_Material_(dstPicture.Material, srcPicture.material);

            // プロシージャル形状
            Convert_Pane_ProceduralShape_(dstPicture.IProceduralShape as ProceduralShape, srcPicture.proceduralShapeParam);
        }

        /// <summary>
        /// Nullペインを Picture へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_Picture_(AppData.Pane dstPane, FileFmt.Picture srcPicture)
        {
            Debug.Assert( srcPicture != null );
            Convert_Pane_PictureImpl_(dstPane, srcPicture);
        }

        #region Windowペイン固有
        /// <summary>
        /// ウインドウコンテンツ領域情報を読み込みます。
        /// </summary>
        void Convert_Pane_WindowContent_(
            AppData.LEWindow dstWnd,
            FileFmt.WindowContent srcContet )
        {
            // 頂点カラー
            dstWnd.VtxCol_LT = Convert_RGBAColor_(srcContet.vtxColLT);
            dstWnd.VtxCol_RT = Convert_RGBAColor_(srcContet.vtxColRT);
            dstWnd.VtxCol_LB = Convert_RGBAColor_(srcContet.vtxColLB);
            dstWnd.VtxCol_RB = Convert_RGBAColor_(srcContet.vtxColRB);


            // マテリアル情報
            LEWindowPartsID partsID = LEWindowPartsID.Content;

            // 詳細なマテリアル設定を行うか？
            dstWnd.SetUseDetailedMaterialFlag( partsID, srcContet.detailSetting );

            AppData.Material dstMtl = dstWnd.GetMaterial( partsID ) as AppData.Material;
            //AppData.RevHWMaterial dstRevMtl = dstWnd.IRevHWMaterial[dstWnd.PartsIDToInt( partsID )] as RevHWMaterial;

            Convert_Pane_Material_( dstMtl, srcContet.material );
            //Convert_Pane_RevMaterial_( dstRevMtl, srcContet.materialRevo );

            // テクスチャ座標
            // ( 注意：テクスチャ情報を含む、マテリアルのロード後に行う必要がある。)
            if( srcContet.texCoord != null )
            {
                // テクスチャ座標数
                dstWnd.NumContentsTexCoord = srcContet.texCoord.Length;
                // テクスチャ座標
                for( int i = 0 ; i < srcContet.texCoord.Length ; i++ )
                {
                    TexCoord4 dstCoord = Convert_TexCoord4_( srcContet.texCoord[i] );
                    dstWnd.SetContentsTexCood4( i, dstCoord );
                }
            }
        }

        /// <summary>
        /// コンテンツ膨張量を読み込みます。
        /// </summary>
        AppData.ILEWindowContentsInflation
            Convert_Pane_Window_InflationRect_( FileFmt.InflationRect srcInflation )
        {
            return new AppData.ILEWindowContentsInflation(
                srcInflation.t,
                srcInflation.b,
                srcInflation.l,
                srcInflation.r );
        }


        /// <summary>
        /// Windowペインのフレーム種類型を変換します。
        /// </summary>
        AppData.LEWindowPartsID
            Convert_Pane_WindowFrameType_( WindowFrameType frameType )
        {
            switch( frameType )
            {
                case WindowFrameType.CornerLT: return AppData.LEWindowPartsID.CornerLT;
                case WindowFrameType.CornerRT: return AppData.LEWindowPartsID.CornerRT;
                case WindowFrameType.CornerLB: return AppData.LEWindowPartsID.CornerLB;
                case WindowFrameType.CornerRB: return AppData.LEWindowPartsID.CornerRB;
                case WindowFrameType.FrameT: return AppData.LEWindowPartsID.FrameT;
                case WindowFrameType.FrameB: return AppData.LEWindowPartsID.FrameB;
                case WindowFrameType.FrameL: return AppData.LEWindowPartsID.FrameL;
                case WindowFrameType.FrameR: return AppData.LEWindowPartsID.FrameR;
                default: Debug.Assert( false ); return AppData.LEWindowPartsID.CornerLT;
            }
        }

        AppData.LEWindow Convert_Pane_WindowSelf_( AppData.Pane dstPane, FileFmt.Window srcWnd )
        {
            AppData.LEWindow dstWnd = new AppData.LEWindow( dstPane );

            //---------- コンテンツ領域
            Convert_Pane_WindowContent_( dstWnd, srcWnd.content );
            dstWnd.ContentsInflation = Convert_Pane_Window_InflationRect_( srcWnd.contentInflation );

            dstWnd.UseLTMaterial = srcWnd.useLTMaterial != null ? srcWnd.useLTMaterial.value : false;

            dstWnd.FrameSize = srcWnd.frameSize != null ? new AppData.LEWindowFrameSize(srcWnd.frameSize.t, srcWnd.frameSize.b, srcWnd.frameSize.l, srcWnd.frameSize.r, true) : AppData.LEWindowFrameSize.Empty;

            dstWnd.NotDrawContent = srcWnd.notDrawContent;

            dstWnd.WindowKind = srcWnd.kind;
            dstWnd.UseVtxColorForAllWindow = srcWnd.useVtxColorForAllWindow;
            dstWnd.NumTexture = srcWnd.frame.Length + 1;

            //---------- フレーム領域
            for( int i = 0 ; i < srcWnd.frame.Length ; i++ )
            {
                FileFmt.WindowFrame srcFrame = srcWnd.frame[i];
                AppData.LEWindowPartsID partsID = Convert_Pane_WindowFrameType_( srcFrame.frameType );

                // マテリアル情報

                // 詳細なマテリアル設定をおこなうか？
                dstWnd.SetUseDetailedMaterialFlag( partsID, srcFrame.detailSetting );

                // フリップ情報設定
                dstWnd.SetMaterialFlipState( partsID,
                    (LEWindowFrameFlipRotate)srcFrame.textureFlip[0] );
            }

            // マテリアル名を更新します。
            // (過去のバージョンのデータを正しく読み込むために必要です。)
            dstWnd.UpdateMatarialName();

            return dstWnd;
        }

        /// <summary>
        /// Nullペインを Window へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_Window_(AppData.Pane dstPane, FileFmt.Window srcWnd)
        {
            AppData.LEWindow dstWnd = Convert_Pane_WindowSelf_( dstPane, srcWnd );

            //---------------------------------------------
            // マテリアル関連の処理
            // コンテンツ
            {
                AppData.LEWindowPartsID partsID = LEWindowPartsID.Content;
                AppData.RevHWMaterial dstRevMtl = dstWnd.IRevHWMaterial[dstWnd.PartsIDToInt(partsID)] as RevHWMaterial;
                if (srcWnd.content.detailSetting)
                {
                    Convert_Pane_RevMaterial_(dstRevMtl, srcWnd.content.materialCtr);
                }

                AppData.Material dstMtl = dstWnd.GetMaterial(partsID) as AppData.Material;
                Convert_Pane_Material_(dstMtl, srcWnd.content.material);
            }

            // フレーム
            for( int i = 0 ; i < srcWnd.frame.Length ; i++ )
            {
                FileFmt.WindowFrame srcFrame = srcWnd.frame[i];
                AppData.LEWindowPartsID partsID = Convert_Pane_WindowFrameType_( srcFrame.frameType );

                // 詳細マテリアル
                int partsIndex = dstWnd.PartsIDToInt( partsID );
                AppData.RevHWMaterial dstRevMtl = dstWnd.IRevHWMaterial[partsIndex] as AppData.RevHWMaterial;
                if (srcFrame.detailSetting)
                {
                    Convert_Pane_RevMaterial_(dstRevMtl, srcFrame.materialCtr);
                }

                // マテリアル
                AppData.Material dstMtl = dstWnd.GetMaterial(partsID) as AppData.Material;
                Convert_Pane_Material_(dstMtl, srcFrame.material);
            }

            dstWnd.OnWindowKindChangeUpdate();
        }
        #endregion Windowペイン固有

        /// <summary>
        /// Nullペインを Window へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_Bounding_(AppData.Pane dstPane, FileFmt.Bounding srcBounding)
        {
            AppData.Bounding dstBounding = new AppData.Bounding( dstPane );
            Debug.Assert( srcBounding != null );
            // 現在のところ、何もパラメータが存在しません。
        }

        /// <summary>
        /// Nullペインを Capture へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        AppData.Capture Convert_Pane_CaptureSelf_(AppData.Pane dstPane, FileFmt.Capture srcCapture)
        {
            AppData.Capture dstCapture = new AppData.Capture(dstPane);

            return dstCapture;
        }

        /// <summary>
        /// Nullペインを Capture へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_CaptureImpl_(AppData.Pane dstPane, FileFmt.Capture srcCapture)
        {
            Convert_Pane_CaptureSelf_(dstPane, srcCapture);
        }

        /// <summary>
        /// Nullペインを Capture へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_Capture_(AppData.Pane dstPane, FileFmt.Capture srcCapture)
        {
            Debug.Assert( srcCapture != null );
            Convert_Pane_CaptureImpl_(dstPane, srcCapture);
        }

        /// <summary>
        /// Nullペインを Alignment へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        AppData.Alignment Convert_Pane_AlignmentSelf_(AppData.Pane dstPane, FileFmt.Alignment src)
        {
            AppData.Alignment dst = new AppData.Alignment(dstPane);

            dst.Horizontal = (AppData.HorizontalLocation)src.horizontal;
            dst.Vertical = (AppData.VerticalLocation)src.vertical;
            dst.DefaultHorizontalMargin = src.defaultMargin;
            dst.DefaultVerticalMargin = src.defaultVerticalMargin;
            dst.IsExtendEdgeHorizontalEnabled = src.isExtendEdgeEnabled;
            dst.IsExtendEdgeVerticalEnabled = src.isExtendEdgeVerticalEnabled;
            dst.IsAlignmentHorizontalEnabled = src.isAlignmentHorizontalEnabled;
            dst.IsAlignmentVerticalEnabled = src.isAlignmentVerticalEnabled;

            return dst;
        }

        /// <summary>
        /// Nullペインを Capture へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        void Convert_Pane_AlignmentImpl_(AppData.Pane dstPane, FileFmt.Alignment src)
        {
            Convert_Pane_AlignmentSelf_(dstPane, src);
        }

        void Convert_Pane_Alignment_(AppData.Pane dstPane, FileFmt.Alignment src)
        {
            Convert_Pane_AlignmentImpl_(dstPane, src);
        }

        /// <summary>
        /// Nullペインを Scissor へと初期化し、
        /// パラメータを読み込みます。
        /// </summary>
        AppData.Scissor Convert_Pane_ScissorSelf_(AppData.Pane dstPane, FileFmt.Scissor src)
        {
            AppData.Scissor dst = new AppData.Scissor(dstPane);
            return dst;
        }

        void Convert_Pane_ScissorImpl_(AppData.Pane dstPane, FileFmt.Scissor src)
        {
            Convert_Pane_ScissorSelf_(dstPane, src);
        }

        void Convert_Pane_Scissor_(AppData.Pane dstPane, FileFmt.Scissor src)
        {
            Convert_Pane_ScissorImpl_(dstPane, src);
        }

        /// <summary>
        /// 読み込み対象とする部品レイアウトを取得します。
        /// </summary>
        private ISubScene GetTargetSubScene(string sourceLayoutParts)
        {
            var partsSubScenes = LECore.LayoutEditorCore.Scene.PartsSubScenes;
            var partsScenePair = partsSubScenes.FirstOrDefault(
                (partsSubScene) => Path.GetFileName(partsSubScene.FilePath) == sourceLayoutParts);

            if (!partsScenePair.IsLoaded)
            {
                // 部品レイアウトの初期化
                _msgReporter.ReportError(
                    LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSPANELOADING"), // "部品ペイン読み込み"
                    LECoreStringResMgr.Get("LECORE_ERROR_PARTSPANELOADING_CANTFINDLYT"), // 部品レイアウト[{0}]がみつかりません。部品ルートの設定を確認し、実際にファイルが存在するか確認してください。
                    sourceLayoutParts);
                return Scene.Instance.CreateSubSceneInstance();
            }
            else
            {
                return partsScenePair.SubScene;
            }
        }

        /// <summary>
        /// 上書き対象を探します。
        /// </summary>
        private AppData.Pane FindPropertyReadTraget_(
            ISubScene dstPartsSubScene, FileFmt.Property property)
        {
            AppData.Pane targetPane = dstPartsSubScene.FindPaneByName(property.target) as AppData.Pane;
            if (targetPane == null)
            {
                var newTargets = dstPartsSubScene.GetPaneSet(
                    (pane) => pane.PaneKind == Convert_SOPaneKind_(property.kind));
                if (newTargets.Count() == 1)
                {
                    targetPane = newTargets.First() as AppData.Pane;

                    // 部品レイアウトの初期化
                    _msgReporter.ReportError(
                        LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSPANELOADING"), // "部品ペイン読み込み"
                        LECoreStringResMgr.Get("LECORE_CATEGORY_REPORT_AUTOREPAIR"), // "上書き対象ペイン[{0}]がみつかりません。一時的に同一種類ペイン[{1}]に自動修復しました。自動修復は保存されないので、部品レイアウトを保存しなおして下さい。"
                        property.target,
                        targetPane.PaneName);

                    property.target = targetPane.PaneName;
                }
                else
                {
                    _msgReporter.ReportError(
                        LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSPANELOADING"), // "部品ペイン読み込み"
                        LECoreStringResMgr.Get("LECORE_CATEGORY_REPORT_CANTFINDTARGET"), // "上書き対象ペイン[{0}]がみつかりません。部品レイアウトでのペイン名変更が影響している可能性があります。"
                        property.target);

                    targetPane = new AppData.Pane();
                }
            }

            return targetPane;
        }

        /// <summary>
        /// 部品ペインの読み込み。
        /// </summary>
        private void Convert_Pane_Parts_(
            AppData.Pane dstPane, FileFmt.Parts srcParts)
        {
            Ensure.Argument.NotNull(srcParts);

            try
            {
                _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSPANELOADING"));

                ISubScene partsSubScene = GetTargetSubScene(srcParts.path);
                AppData.PartsLayout dstParts = new AppData.PartsLayout(dstPane, partsSubScene, srcParts.path);

                ISubScene dstPartsSubScene = dstParts.PartsSubScene;
                if (srcParts.property != null)
                {
                    try
                    {
                        // "部品ペイン-上書きペイン読み込み"
                        _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSPROPLOADING"));
                        var dstProperties = Convert_Pane_PartsProperties_(
                            srcParts.property, dstPartsSubScene);

                        // 上書き設定を探してロード結果で置き換えます。
                        foreach (var dstProperty in dstProperties)
                        {
                            var sameTarget = dstParts.RawPartsPropaerties.Find(
                                (item) =>item.TargetName == dstProperty.TargetName);
                            if (sameTarget != null)
                            {
                                int index = dstParts.RawPartsPropaerties.IndexOf(sameTarget);
                                dstParts.RawPartsPropaerties[index] = dstProperty;
                            }
                            else
                            {
                                // 設定が無かった。警告を表示してデータを破棄します。
                                _msgReporter.ReportError(
                                    LECoreStringResMgr.Get("LECORE_CATEGORY_PARTSPANELOADING"),
                                    LECoreStringResMgr.Get("LECORE_CATEGORY_REPORT_CANTFINDOVERRIDE_PARAM"),
                                dstProperty.TargetName,
                                Path.GetFileName(this._inputFilePath),
                                Path.GetFileName(srcParts.path));
                            }
                        }

                        // 変更通知イベントを起こす。
                        dstParts.NotifyPartsPropertyChanged();
                        (dstParts.PartsSubScene as SubScene).PrepareTextboxScFontUpdate();

                        foreach (var property in dstProperties.Where((prop) => (prop.Option & Structures.PartsPropertyUsageOptions.UsePaneExParameterTexture) != 0))
                        {
                            dstParts.ModifyOverwriteParamaters(property, property.Option);
                        }

                    }
                    finally
                    {
                        _msgReporter.EndPacking();
                    }
                }

                // rawMagnify の方が優先して読み込まれます。
                if (srcParts.rawMagnify != null)
                {
                    dstParts.Magnify = new FVec3(srcParts.rawMagnify.x, srcParts.rawMagnify.y, 0.0f);
                }
                else
                {
                    if (srcParts.magnify != null)
                    {
                        dstParts.Magnify = new FVec3(srcParts.magnify.x, srcParts.magnify.y, 0.0f);
                    }
                }

                if (srcParts.sizeConstraint != null)
                {
                    dstParts.SizeConstraint = new FVec3(srcParts.sizeConstraint.x, srcParts.sizeConstraint.y, 0.0f);
                }
            }
            finally
            {
                _msgReporter.EndPacking();
            }
        }

        /// <summary>
        /// 部品ペインプロパティの読み込み。
        /// </summary>
        private List<IPartsPropaerty> Convert_Pane_PartsProperties_(
            Property[] srcProperties, ISubScene dstPartsSubScene)
        {
            var dstProperties = new List<IPartsPropaerty>();

            if (srcProperties != null)
            {
                foreach (var property in srcProperties)
                {
                    IPaneExParamater paramater = Convert_Pane_PartsProperty_(dstPartsSubScene, property);


                    // ペイン基本情報の上書き読み込み。
                    AppData.Pane dstPane = paramater != null ?
                        paramater.OwnerPane as AppData.Pane : FindPropertyReadTraget_(dstPartsSubScene, property);

                    Convert_Pane_PartsProperties_BasicPane_(property, dstPane);

                    AppData.PartsPropaerty dstProperty = new PartsPropaerty(
                        Convert_SOPaneKind_(property.kind),
                        property.target,
                        paramater,
                        GetPartsPropertyUsageOptions_(property));

                    dstProperties.Add(dstProperty);
                }
            }

            return dstProperties;
        }

        /// <summary>
        /// 「ペインパラメータを上書き」が有効になるかどうか判定します。
        /// </summary>
        bool IsPaneExParamaterEnabled_(Property property, AppData.PartsPropertyUsageOptions propertyUsage)
        {
            // 実際にデータが書きだされていれば有効
            return (property.Item != null);
        }

        /// <summary>
        /// PartsPropertyUsageOptions
        /// </summary>
        AppData.PartsPropertyUsageOptions GetPartsPropertyUsageOptions_(FileFmt.Property property)
        {
            AppData.PartsPropertyUsageOptions result = Convert_PartsPropertyUsageOptions_(property.usageOptions);

            if (IsPaneExParamaterEnabled_(property, result)) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePaneExParamater; }

            if (property.userData != null) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePaneExUserData; }
            if (property.alpha.HasValue) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePaneTransparency; }
            if (property.visible.HasValue) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePaneVisibility; }

            if (property.translate != null) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePanePosition; }
            if (property.size != null) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePaneSize; }
            if (property.scale != null) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePaneScale; }
            if (property.rotate != null) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePaneRotate; }

            if (property.basicUserData != null) { result |= LECore.Structures.PartsPropertyUsageOptions.UsePaneBasicUserData; }

            return result;
        }

        /// <summary>
        /// ペイン基本情報の読み込み
        /// </summary>
        private void Convert_Pane_PartsProperties_BasicPane_(FileFmt.Property property, AppData.Pane dstPane)
        {
            if (property != null && dstPane != null)
            {
                // 位置
                if (property.translate != null)
                {
                    dstPane.Trans = Convert_FVec3_(property.translate);
                }

                // サイズ
                if (property.size != null)
                {
                    dstPane.Size = Convert_FVec3_(property.size);
                }

                // スケール
                if (property.scale != null)
                {
                    dstPane.Scale = Convert_FVec3_(property.scale);
                }

                // 回転
                if (property.rotate != null)
                {
                    dstPane.RotAng = Convert_FVec3_(property.rotate);
                }

                // 透明度
                if (property.alpha.HasValue)
                {
                    dstPane.Transparency = property.alpha.Value;
                }

                // 表示・非表示
                if (property.visible.HasValue)
                {
                    dstPane.Visible = property.visible.Value;
                }

                // 基本ユーザ情報
                if (property.basicUserData != null)
                {
                    dstPane.UserData = property.basicUserData;
                }

                // ユーザデータの読み込み
                if (property.userData != null)
                {
                    // 元を削除して置き換えます。
                    dstPane.UserDataHolder.RemoveUserDataElementAll();

                    // 通常の拡張ユーザデータと異なり基本ユーザデータが内包されていない点に注意。
                    foreach (UserDataElement ud in property.userData.ToList().ConvertAll((item) => GetUserDataElement_(item)))
                    {
                        if (ud == null) { continue; }
                        dstPane.UserDataHolder.AddUserDataElement(ud.Name, ud.UserDataKind, ud.Overwrite, ud.Value);
                    }
                }
            }
        }

        /// <summary>
        /// 部品プロパティの読み込み。
        /// </summary>
        private IPaneExParamater Convert_Pane_PartsProperty_(ISubScene dstPartsSubScene, Property property)
        {
            IPaneExParamater paramater = null;
            AppData.Pane targetPane = FindPropertyReadTraget_(dstPartsSubScene, property);
            if (property.Item != null)
            {
                if (property.usageOptions.ContainsUsageOption_(FileFmt.PartsPropertyUsageOptions.UseMaterialColorBlend) ||
                    property.usageOptions.ContainsUsageOption_(FileFmt.PartsPropertyUsageOptions.UseTextBoxText) ||
                    property.usageOptions.ContainsUsageOption_(FileFmt.PartsPropertyUsageOptions.UseMaterialTexture))
                {
                    // カラー補間だけ上書き
                    if (property.usageOptions.ContainsUsageOption_(FileFmt.PartsPropertyUsageOptions.UseMaterialColorBlend))
                    {
                        // カラー補間の部分上書き
                        switch (property.kind)
                        {
                            case FileFmt.PaneKind.TextBox: Convert_Pane_Material_ColorBlend_(targetPane.IMaterial.First() as AppData.Material, (property.Item as FileFmt.TextBox).material); break;
                            case FileFmt.PaneKind.Picture: Convert_Pane_Material_ColorBlend_(targetPane.IMaterial.First() as AppData.Material, (property.Item as FileFmt.Picture).material); break;
                            case FileFmt.PaneKind.Parts: Convert_Pane_Parts_(targetPane, property.Item as FileFmt.Parts); break;
                            case FileFmt.PaneKind.Window:
                                {
                                    var dstWnd = targetPane.ILEWindow;
                                    var srcWnd = property.Item as FileFmt.Window;
                                    if (dstWnd != null)
                                    {
                                        // コンテンツ
                                        {
                                            AppData.LEWindowPartsID partsID = LEWindowPartsID.Content;
                                            AppData.Material dstMtl = dstWnd.GetMaterial(partsID) as AppData.Material;
                                            Convert_Pane_Material_ColorBlend_(dstMtl, srcWnd.content.material);
                                        }

                                        // フレーム
                                        for (int i = 0; i < srcWnd.frame.Length; i++)
                                        {
                                            FileFmt.WindowFrame srcFrame = srcWnd.frame[i];
                                            AppData.LEWindowPartsID partsID = Convert_Pane_WindowFrameType_(srcFrame.frameType);

                                            AppData.Material dstMtl = dstWnd.GetMaterial(partsID) as AppData.Material;
                                            Convert_Pane_Material_ColorBlend_(dstMtl, srcFrame.material);
                                        }
                                    }
                                    break;
                                }
                            default: Debug.Assert(false, "UnExpected case."); break;
                        }
                    }

                    // テクスチャだけ上書き
                    if (property.usageOptions.ContainsUsageOption_(FileFmt.PartsPropertyUsageOptions.UseMaterialTexture))
                    {
                        switch (property.kind)
                        {
                            case FileFmt.PaneKind.Picture: Convert_Pane_Material_Texture_(targetPane.IMaterial.First() as AppData.Material, (property.Item as FileFmt.Picture).material); break;
                            case FileFmt.PaneKind.Window:
                                {
                                    var dstWnd = targetPane.ILEWindow;
                                    var srcWnd = property.Item as FileFmt.Window;
                                    if (dstWnd != null)
                                    {
                                        // コンテンツ
                                        {
                                            AppData.LEWindowPartsID partsID = LEWindowPartsID.Content;
                                            AppData.Material dstMtl = dstWnd.GetMaterial(partsID) as AppData.Material;
                                            Convert_Pane_Material_Texture_(dstMtl, srcWnd.content.material);
                                        }

                                        // フレーム
                                        for (int i = 0; i < srcWnd.frame.Length; i++)
                                        {
                                            FileFmt.WindowFrame srcFrame = srcWnd.frame[i];
                                            AppData.LEWindowPartsID partsID = Convert_Pane_WindowFrameType_(srcFrame.frameType);

                                            AppData.Material dstMtl = dstWnd.GetMaterial(partsID) as AppData.Material;
                                            Convert_Pane_Material_Texture_(dstMtl, srcFrame.material);
                                        }
                                    }
                                    break;
                                }
                            default: Debug.Assert(false, "UnExpected case."); break;
                        }
                    }

                    // テキストだけ上書き
                    if (property.usageOptions.ContainsUsageOption_(FileFmt.PartsPropertyUsageOptions.UseTextBoxText))
                    {
                        // テキストボックスの部分上書き
                        Debug.Assert(property.kind == FileFmt.PaneKind.TextBox);
                        Convert_Pane_TextBoxTextAndID_(targetPane, property.Item as FileFmt.TextBox);
                    }
                }
                else
                {
                    // 全体を上書きする。
                    switch (property.kind)
                    {
                        case FileFmt.PaneKind.TextBox: Convert_Pane_TextBoxImpl_(targetPane, property.Item as FileFmt.TextBox); break;
                        case FileFmt.PaneKind.Picture: Convert_Pane_PictureImpl_(targetPane, property.Item as FileFmt.Picture); break;
                        case FileFmt.PaneKind.Window: Convert_Pane_Window_(targetPane, property.Item as FileFmt.Window); break;
                        case FileFmt.PaneKind.Capture:
                            AppData.Capture dstCapture = new AppData.Capture(targetPane);

                            FileFmt.CaptureTexture srcCapture = property.Item as FileFmt.CaptureTexture;

                            AppData.CaptureTexture dstCaptureTexture = TextureMgrHelper.FindCaptureTextureByName(dstPartsSubScene.ITextureMgr, srcCapture.paneName) as AppData.CaptureTexture;

                            if (dstCaptureTexture != null)
                            {
                                dstCaptureTexture.CaptureOnlyFirstFrame = srcCapture.captureOnlyFirstFrame;
                                dstCaptureTexture.FrameBufferCaptureEnabled = srcCapture.framebufferCaptureEnabled;
                                dstCaptureTexture.ClearColor = Convert_Color4f_(srcCapture.clearColor);
                                dstCaptureTexture.Format = Convert_TexelFormat_(srcCapture.format);
                                dstCaptureTexture.TextureScale = srcCapture.filters[0].scale;
                            }
                            break;
                        case FileFmt.PaneKind.Parts: Convert_Pane_Parts_(targetPane, property.Item as FileFmt.Parts); break;
                        default: Debug.Assert(false, "UnExpected case."); break;
                    }
                }
            }

            paramater = targetPane.IPaneExParamater;

            return paramater;
        }

#endregion 各種ペインに固有なパラメータの読み込み

        /// <summary>
        /// シーンからペインを検索して取得します。
        /// ペインが発見されなかったときは、デバックコンソールに警告を表示します。
        /// </summary>
        /// <param name="subScene">シーン</param>
        /// <param name="paneName">ペイン名</param>
        /// <returns>ペイン</returns>
        AppData.IPane SearchPaneFromScene_( ISubScene subScene, string paneName )
        {
            AppData.IPane pane = SubSceneHelper.FindPaneByName( subScene, paneName );

            if( pane == null )
            {
                string errMsg
                    = string.Format( "WARNING:Can't find PaneModify( name = {0}) ", paneName );
                DbgConsole.WriteLine( errMsg );
            }
            return pane;
        }

        /// <summary>
        /// 階層構造を構築します。
        /// 再帰的呼び出しをおこないます。
        /// </summary>
        /// <param name="subScene">シーン</param>
        /// <param name="parentPane">親ペイン</param>
        /// <param name="tree">親ペインの接続情報木</param>
        void BuildHierarchyConnection_
            (
            ISubScene subScene,
            AppData.IPane parentPane,
            PaneTree parentTree
            )
        {
            Debug.Assert( parentPane != null );
            Debug.Assert( parentTree != null );

            // すべての子ツリーに対して...
            if( parentTree.paneTree != null )
            {
                foreach( PaneTree subTree in parentTree.paneTree )
                {
                    // 子供ペインをシーンから検索して、子供ペインを登録します。
                    AppData.IPane childPane = SearchPaneFromScene_( subScene, subTree.name );
                    if( childPane != null )
                    {
                        // 読み取り専用ロック指定時は、階層が構築済みで変更の必要はないはず。
                        // 再構築は行わないようにする。
                        // (派生部品の設定で再構築してしまい、派生元の変更が反映されない不具合があった。)
                        if (!childPane.IsReadOnlyLocked)
                        {
                            childPane.DestroyHierarchyConnection();
                            parentPane.AddChildNodeByLoaclCoordinate(childPane);
                        }

                        // サブツリーに対して再帰的に処理をおこないます。
                        BuildHierarchyConnection_( subScene, childPane, subTree );
                    }
                }
            }
        }

#region UserData関連
        /// <summary>
        /// ユーザ情報文字列(基本)を取得します。
        /// ユーザ情報文字列(基本)は、特別な名前で保存されています。
        /// </summary>
        string GetUserDataString_( FileFmt.Pane srcPane )
        {
            if( srcPane.userData != null && srcPane.userData.Length > 0 )
            {
                if( srcPane.userData.Length > 0 )
                {
                    UserDataString uds = srcPane.userData[0] as UserDataString;
                    if( uds != null && uds.Value != null )
                    {
                        return uds.Value;
                    }
                }
            }
            // 発見されなかった。
            return string.Empty;
        }

        internal static UserDataElement GetUserDataElement_(object userData)
        {
            return GetUserDataElement_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(userData);
        }

        /// <summary>
        ///
        /// </summary>
        static UserDataElement GetUserDataElement_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(object userData)
            where UserDataFloatList : class, IUserDataFloatList<UserDataElementType>, new()
            where UserDataIntList : class, IUserDataIntList<UserDataElementType>, new()
            where UserDataNone : class, IUserDataNone<UserDataElementType>, new()
            where UserDataString : class, IUserDataString<UserDataElementType>, new()
        {
            if( userData is UserDataNone )
            {
                UserDataNone udn = userData as UserDataNone;
                return new UserDataElement( udn.name, UserDataKind.None, udn.overwrite, udn.Value );
            }
            if( userData is UserDataString )
            {
                UserDataString uds = userData as UserDataString;
                if( uds.name != _BasicUserDataStringName )
                {
                    return new UserDataElement( uds.name, UserDataKind.String, uds.overwrite, uds.Value );
                }
                else
                {
                    return null;
                }
            }
            else if( userData is UserDataIntList )
            {
                UserDataIntList udi = userData as UserDataIntList;

                UserDataKind kind = UserDataKind.Int;
                if( udi.kindSpecified )
                {
                    Enum.TryParse<UserDataKind>( udi.kind.ToString(), out kind );
                }

                return new UserDataElement(
                    udi.name,
                    kind,
                    udi.overwrite,
                    UserDataElementHelper.ParseString(
                    kind,
                    udi.Value,
                    UserDataElementHelper.BlankSeparatorStr));
            }
            else if( userData is UserDataFloatList )
            {
                UserDataFloatList udf = userData as UserDataFloatList;

                UserDataKind kind = UserDataKind.Float;
                if( udf.kindSpecified )
                {
                    Enum.TryParse<UserDataKind>( udf.kind.ToString(), out kind );
                }

                return new UserDataElement(
                    udf.name,
                    kind,
                    udf.overwrite,
                    UserDataElementHelper.ParseString(
                        kind,
                        udf.Value,
                        UserDataElementHelper.BlankSeparatorStr ) );
            }
            else
            {
                Debug.Assert( false );
                return null;
            }
        }

        internal static UserDataElement[] GetUserDataElementSetImpl_(object[] userData)
        {
            return GetUserDataElementSetImpl_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(userData);
        }

        /// <summary>
        ///
        /// </summary>
        internal static UserDataElement[] GetUserDataElementSetImpl_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(object[] userData)
            where UserDataFloatList : class, IUserDataFloatList<UserDataElementType>, new()
            where UserDataIntList : class, IUserDataIntList<UserDataElementType>, new()
            where UserDataNone : class, IUserDataNone<UserDataElementType>, new()
            where UserDataString : class, IUserDataString<UserDataElementType>, new()
        {
            List<UserDataElement> userDataElementSet = new List<UserDataElement>();

            if (userData != null)
            {
                for (int i = 0; i < userData.Length; i++)
                {
                    UserDataElement data = GetUserDataElement_<UserDataFloatList, UserDataIntList, UserDataNone, UserDataString, UserDataElementType>(userData[i]);
                    if (data != null)
                    {
                        userDataElementSet.Add(data);
                    }
                }
            }

            return userDataElementSet.ToArray();
        }

        /// <summary>
        ///
        /// </summary>
        UserDataElement[] GetUserDataElementSet_(object[] userData)
        {
            if (userData == null)
            {
                return new UserDataElement[0];
            }

            // 先頭のひとつは、(拡張ではない)ユーザデータなので無視
            return GetUserDataElementSetImpl_(userData.Skip(0).ToArray());
        }
#endregion UserData関連

        /// <summary>
        /// 基本パラメータのコピー
        /// </summary>
        /// <param name="dstPane"></param>
        /// <param name="srcPane"></param>
        void SetBasicPaneParams_( AppData.Pane dstPane, FileFmt.Pane srcPane )
        {
            dstPane.Size = Convert_FVec3_( srcPane.size );

            dstPane.Trans = Convert_FVec3_( srcPane.translate );
            dstPane.Scale = Convert_FVec3_( srcPane.scale );
            dstPane.RotAng = Convert_FVec3_( srcPane.rotate );

            dstPane.Transparency = srcPane.alpha;

            dstPane.BasePosTypeH = Convert_HorizontalLocation_( srcPane.basePositionType );
            dstPane.BasePosTypeV = Convert_VerticalLocation_( srcPane.basePositionType );
            dstPane.ParentBasePosTypeH = Convert_HorizontalLocation_(srcPane.parentRelativePositionType);
            dstPane.ParentBasePosTypeV = Convert_VerticalLocation_(srcPane.parentRelativePositionType);


            dstPane.Locked = srcPane.locked;
            dstPane.Hidden = srcPane.hidden;

            dstPane.UserCommentString = srcPane.comment;
            dstPane.PaneName = srcPane.name;
            dstPane.UserData = GetUserDataString_( srcPane );

            dstPane.Visible = srcPane.visible;
            dstPane.InfluenceChildrenTransparency = srcPane.influencedAlpha;
            dstPane.LocationAdjust = srcPane.locationAdjust;

            dstPane.IsAlignmentHorizontalMarginEnabled = srcPane.alignmentMarginEnabled;
            dstPane.AlignmentHorizontalMargin = srcPane.alignmentMargin;
            dstPane.AlignmentVerticalMargin = srcPane.alignmentVerticalMargin;
            dstPane.IsAlignmentIgnore = srcPane.alignmentIgnore;
            dstPane.IsAlignmentVerticalMarginEnabled = srcPane.alignmentVerticalMarginEnabled;
            dstPane.AlignmentVerticalMargin = srcPane.alignmentVerticalMargin;

            // 古いデータで ignorePartsMagnify が指定されていれば優先する。
            // 新しいデータでは強制的に ignorePartsMagnify を false で出力している。
            dstPane.PartsMagnifyInfluence = srcPane.ignorePartsMagnify ?
                AppData.PartsMagnifyInfluence.None : Convert_PartsMagnifyInfluence_(srcPane.partsMagnifyInfluence);

            // ファイル保存時の読み込み専用状態を復元
            dstPane.IsReadOnlyLocked = srcPane.readonlyLocked;

            dstPane.AvoidPaneTreeCompression = srcPane.avoidPaneTreeCompression;
            dstPane.ExtUserDataAnimEnabled = srcPane.extUserDataAnimEnabled;

            // ユーザデータの読み込み
            UserDataElement[] userDataSet = GetUserDataElementSet_( srcPane.userData );
            foreach( UserDataElement ud in userDataSet )
            {
                dstPane.UserDataHolder.AddUserDataElement( ud.Name, ud.UserDataKind, ud.Overwrite, ud.Value );
            }
        }

        /// <summary>
        ///
        /// </summary>
        /// <param name="srcPane"></param>
        /// <returns></returns>
        AppData.Pane Convert_Pane_( FileFmt.Pane srcPane )
        {
            AppData.Pane result = new AppData.Pane();

            // 標準的なパラメータの設定(Nullペイン)
            SetBasicPaneParams_( result, srcPane );

            // マスクの設定
            if (srcPane.mask != null)
            {
                Convert_Pane_Mask_(result.IMask as Structures.Mask, srcPane.mask);
            }

            // ドロップシャドウ効果の設定
            if (srcPane.dropShadow != null)
            {
                Convert_Pane_DropShadow_(result.IDropShadow as Structures.DropShadow, srcPane.dropShadow);
            }

            // 各種ペインに固有な情報の設定
            switch ( srcPane.kind )
            {
                case FileFmt.PaneKind.Null: /* DO NOTHING */ break;
                case FileFmt.PaneKind.TextBox:
                    {
                        Convert_Pane_TextBox_(result, srcPane.Item as FileFmt.TextBox);
                        // AppData.TextBoxのコンストラクタで
                        // サイズが自動計算されてしまうので、再度ペインサイズを設定します。
                        result.Width = srcPane.size.x;
                        result.Height = srcPane.size.y;
                        break;
                    }
                case FileFmt.PaneKind.Picture: Convert_Pane_Picture_( result, srcPane.Item as FileFmt.Picture ); break;
                case FileFmt.PaneKind.Window: Convert_Pane_Window_( result, srcPane.Item as FileFmt.Window ); break;
                case FileFmt.PaneKind.Bounding: Convert_Pane_Bounding_( result, srcPane.Item as FileFmt.Bounding ); break;
                case FileFmt.PaneKind.Capture: Convert_Pane_Capture_( result, srcPane.Item as FileFmt.Capture ); break;
                case FileFmt.PaneKind.Alignment: Convert_Pane_Alignment_(result, srcPane.Item as FileFmt.Alignment); break;
                case FileFmt.PaneKind.Scissor: Convert_Pane_Scissor_(result, srcPane.Item as FileFmt.Scissor); break;
                case FileFmt.PaneKind.Parts: Convert_Pane_Parts_(result, srcPane.Item as FileFmt.Parts); break;
                default: Debug.Assert( false ); break;
            }

            // baseValueの設定
            PaneManipulator paneMnp = new PaneManipulator();
            paneMnp.BindTarget(result);
            result.StoreCurrentValueToBaseValue();

            return result;
        }

        /// <summary>
        /// ペイン情報をシーンに読み込みます。
        /// </summary>
        void Convert_LayoutData_PaneSet_(FileFmt.LYT rlyt, ISubScene subScene)
        {
            if (rlyt.paneSet == null)
            {
                return;
            }

            OutLog_("LECORE_RLYTLOAD_PANE");
            FileFmt.Pane[] paneSet = rlyt.paneSet;
            Group rlytRootGroup = rlyt.groupSet.group;

            // インポートするデータに、読み取り専用属性が true のペインがあれば派生部品なのでチェックを行う。
            bool isSrcDerivativePartsAndAbleToValidate = paneSet.Any((srcPane) => srcPane.readonlyLocked);
            foreach (FileFmt.Pane srcPane in paneSet)
            {
                try
                {
                    // コンバートして、シーン に登録していきます。
                    _msgReporter.BeginPacking(LECoreStringResMgr.Get("LECORE_PANE") + string.Format(" [{0}]", srcPane.name));
                    AppData.Pane pane = Convert_Pane_(srcPane);
                    IPane exsitPane = SubSceneHelper.FindPaneByName(subScene, pane.PaneName);

                    // 派生部品で、有効性チェックができるならチェックをおこないます。
                    if (isSrcDerivativePartsAndAbleToValidate)
                    {
                        if (exsitPane != null)
                        {
                            // NG : 読み込もうとした、通常ペインと同じ名前の、読み取り専用ペインがすでに存在する。
                            // 派生元部品は、読み取り専用ペインとして中間ファイルに記録されているはずなのでおかしい。
                            if (exsitPane.IsReadOnlyLocked && !pane.IsReadOnlyLocked)
                            {
                                // 同名ペイン[{0}]が存在するため、情報を破棄しました。おそらく派生元部品へのペイン追加が原因です。
                                ReportErrorMsg_(LECoreStringResMgr.Get("LECORE_ERROR_DERIVPARTS_NAMECONFLICT", pane.PaneName));
                            }
                        }
                        else
                        {
                            // 事前に読み込まれているはずの、読み取り専用ペインが発見されなかった。
                            if (pane.IsReadOnlyLocked)
                            {
                                // 派生元でペインの削除があった。ファイルのペインは不要なので読み込まない。
                                continue;
                            }
                        }
                    }

                    // 同一名称ペインの登録は行われません。
                    if (exsitPane == null)
                    {
                        string originalName = pane.PaneName;

                        (subScene as SubScene).AddPane(pane);

                        // 名前が変更された場合、変更を元に戻して警告を表示する。
                        if (originalName != pane.PaneName)
                        {
                            pane.PaneName = originalName;
                            ReportErrorMsg_(LECoreStringResMgr.Get("LECORE_WARN_NAMECONFLICT"));
                        }
                        else
                        {
                            // 部品ペインの名前と、その部品ペインが参照する部品レイアウト内のペインの名前が重複しているかチェック。
                            string uniqueName = subScene.GetUniqueNewPaneName(pane, pane.PaneName, pane.PaneName);
                            if (originalName != uniqueName)
                            {
                                ReportErrorMsg_(LECoreStringResMgr.Get("LECORE_WARN_NAMECONFLICT"));
                            }
                        }
                    }
                }
                catch (Exception e)
                {
                    // エラーが起こっても処理は続行されます。
                    // "ペイン読み込み中にエラーが検出されました。ペイン [{0}] の読み込みに失敗しました。
                    string formatedMsg = String.Format(LECoreStringResMgr.Get("LECORE_SYS_ERROR_LOAD_PANE"), srcPane.name);
                    LayoutEditorCore.MsgReporter.ReportError(this, new ErrorLoadPaneMessageArgs(){ msg = formatedMsg });

                    // 環境変数が定義されていれば詳細をレポートする。
                    if (!string.IsNullOrEmpty(Environment.GetEnvironmentVariable("NW_LYT_REPORT_EXCEPTION_DETAIL")))
                    {
                        ReportErrorMsg_(e.ToString());
                    }
                }
                finally
                {
                    _msgReporter.EndPacking();
                }
            }
        }

        /// <summary>
        /// ペインの読み込み中にエラーが発生した場合に使用されます。
        /// ペインの読み込みに失敗したかどうかを判定するために別クラスとして定義しています。
        /// </summary>
        public class ErrorLoadPaneMessageArgs : MessageArgs
        {

        }

#endregion ペイン関連

#endregion
    }
}
