﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Rlyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Rlyt;

    /// <summary>
    /// XMLシリアライズ可能なクラスへの変換。
    /// (詳細なMaterial(RVL)に関する部分)
    /// </summary>
    public partial class RlytConverter
    {
        #region File => App

        #region 型変換関連

        #region TEV関連列挙型 変換

        /// <summary>
        ///
        /// </summary>
    private AttrCombineMode
    Convert_AttrCtrCombineMode_( FileFmt.TevMode mode )
        {
        switch( mode )
        {
        case TevMode.Replace:     return AttrCombineMode.Replace;
        case TevMode.Modulate:    return AttrCombineMode.Modulate;
        case TevMode.Add:         return AttrCombineMode.Add;
        case TevMode.AddSigned:   return AttrCombineMode.AddSigned;
        case TevMode.Interpolate: return AttrCombineMode.Interpolate;
        case TevMode.Subtract:    return AttrCombineMode.Subtract;
        case TevMode.AddMult:     return AttrCombineMode.AddMultDMP;
        case TevMode.MultAdd:     return AttrCombineMode.MultAddDMP;
        default: Debug.Assert( false ); return AttrCombineMode._Unknown;
        }
        }

        /// <summary>
        ///
        /// </summary>
        AttrTevScale
            Convert_AttrTevScale_( FileFmt.TevScale src )
        {
        switch( src )
        {
                case TevScale.V1: return AttrTevScale.Scale1;
                case TevScale.V2: return AttrTevScale.Scale2;
                case TevScale.V4: return AttrTevScale.Scale4;
                default: Debug.Assert( false ); return AttrTevScale.Scale1;
        }
        }

        /// <summary>
        /// FileFmt.BlendFactorDst => AppData.AttrBlendFactor
        /// </summary>
        AttrBlendFactor
            Convert_AttrBlendFactor_( FileFmt.BlendFactorDst srcFactor )
        {
        switch( srcFactor )
        {
                case FileFmt.BlendFactorDst.V1_0: return AttrBlendFactor.One;
                case FileFmt.BlendFactorDst.V0: return AttrBlendFactor.Zero;
                case FileFmt.BlendFactorDst.SrcClr: return AttrBlendFactor.SrcColor;
                case FileFmt.BlendFactorDst.SrcAlpha: return AttrBlendFactor.SrcAlpha;
                case FileFmt.BlendFactorDst.DstAlpha: return AttrBlendFactor.DstAlpha;
                case FileFmt.BlendFactorDst.InvSrcClr: return AttrBlendFactor.InvSrcColor;
                case FileFmt.BlendFactorDst.InvSrcAlpha: return AttrBlendFactor.InvSrcAlpha;
                case FileFmt.BlendFactorDst.InvDstAlpha: return AttrBlendFactor.InvDstAlpha;
                default: Debug.Assert( false ); return AttrBlendFactor.Zero;
        }
        }

        /// <summary>
        /// FileFmt.BlendFactorSrc => AttrBlendFactor.IPEData
        /// </summary>
        AttrBlendFactor
            Convert_AttrBlendFactor_( FileFmt.BlendFactorSrc srcFactor )
        {
        switch( srcFactor )
        {
                case FileFmt.BlendFactorSrc.V1_0: return AttrBlendFactor.One;
                case FileFmt.BlendFactorSrc.V0: return AttrBlendFactor.Zero;
                case FileFmt.BlendFactorSrc.SrcAlpha: return AttrBlendFactor.SrcAlpha;
                case FileFmt.BlendFactorSrc.DstClr: return AttrBlendFactor.DstColor;
                case FileFmt.BlendFactorSrc.DstAlpha: return AttrBlendFactor.DstAlpha;
                case FileFmt.BlendFactorSrc.InvSrcAlpha: return AttrBlendFactor.InvSrcAlpha;
                case FileFmt.BlendFactorSrc.InvDstClr: return AttrBlendFactor.InvDstColor;
                case FileFmt.BlendFactorSrc.InvDstAlpha: return AttrBlendFactor.InvDstAlpha;
                default: Debug.Assert( false ); return AttrBlendFactor.Zero;
        }
        }

        #endregion TEV関連列挙型 変換

        /// <summary>
        ///
        /// </summary>
    void Convert_ToTevDirectStage(
      AppData.TevDirectStage dstStage,
      FileFmt.Material_CTRTevStageRgb srcStage)
    {
        dstStage.CombineMode = Convert_AttrCtrCombineMode_( srcStage.mode);
        dstStage.Scale = Convert_AttrTevScale_(srcStage.scale);

        ///@@@@
        //srcStage
        for( int i = 0; i < 3; i++ ) {
        //  dstStage.TevArgs[i] = Convert_TevArgRgb_();


            //dstMat.TevData.SetTevKColors( Convert_RGBAColor_( srcMat.tevConstReg[ i ] ), i );
        }
    }

        /// <summary>
        ///
        /// </summary>
    void Convert_ToTevDirectAlpha(
      AppData.TevDirectStage dstStage,
      FileFmt.Material_CTRTevStageAlpha srcStage)
    {
      //dstStage.A = Convert_TevAlphaArg_(srcStage.a);
      //dstStage.B = Convert_TevAlphaArg_(srcStage.b);
      //dstStage.C = Convert_TevAlphaArg_(srcStage.c);
      //dstStage.D = Convert_TevAlphaArg_(srcStage.d);

      //dstStage.Bias = Convert_AttrTevBias_(srcStage.bias);
      //dstStage.Clamp = srcStage.clamp;

      //dstStage.KSel = Convert_AttrTevKSel_(srcStage.konst);
      //dstStage.Op = Convert_AttrTevOp_(srcStage.op);
      dstStage.Scale = Convert_AttrTevScale_(srcStage.scale);

      //dstStage.OutReg = Convert_AttrTevOutReg_(srcStage.outReg);
    }

        /// <summary>
        /// FileFmt.Material_RevoTevStage => AppData.TevStage
        /// </summary>
        void Convert_ToTevStage(
            AppData.TevStage dstStage,
            FileFmt.Material_CTRTevStage srcStage )
        {
        Convert_ToTevDirectStage( dstStage.ColorStage, srcStage.rgb );
        Convert_ToTevDirectAlpha( dstStage.AlphaStage, srcStage.alpha );
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoAlphaCompare
        /// </summary>
        void Convert_ToAppRevoAlphaCompare_( AppData.PEACompare dstACmp, FileFmt.Material_CTRAlphaCompare srcACmp )
        {
        dstACmp.Comp = (AttrCompareFunc)ConvertEnumTypesFromRToL_( dstACmp.Comp, srcACmp.comp );
        dstACmp.Ref = srcACmp.@ref;
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoBlendMode
        /// </summary>
        void Convert_ToFileRevoBlendMode_( AppData.PEBlend dstBlend, FileFmt.Material_CTRBlendMode srcBlend )
        {
            dstBlend.DstFactor = Convert_AttrBlendFactor_( srcBlend.dstFactor );
            dstBlend.SrcFactor = Convert_AttrBlendFactor_( srcBlend.srcFactor );
      dstBlend.BlendOperation = (AttrBlendOperation)ConvertEnumTypesFromRToL_( dstBlend.BlendOperation, srcBlend.blendOp );
            dstBlend.Type = (AttrBlendType)ConvertEnumTypesFromRToL_( dstBlend.Type, srcBlend.type );
        }

        #endregion

        /// <summary>
        /// FileFmt.Material_Revo => AppData.RevHWMaterial
        /// </summary>
        void Convert_Pane_RevMaterial_( AppData.RevHWMaterial dstMat, FileFmt.Material_CTR srcMat )
        {
            // ChannelControl 関連
      //dstMat.ChannelData.DiffuseCtrlColorSrc =
      //  Convert_ToAppColorSource_( srcMat.channelControl[0].materialSource );
      //dstMat.ChannelData.AlphaCtrlColorSrc =
      //  Convert_ToAppColorSource_( srcMat.channelControl[1].materialSource );

            // PE - AlphaCompare
            Convert_ToAppRevoAlphaCompare_( dstMat.PEData.ACompare, srcMat.alphaCompare );
            Convert_ToFileRevoBlendMode_( dstMat.PEData.Blend, srcMat.blendMode );

            //dstMat.ChannelData.MatColReg0 = Convert_RGBAColor_( srcMat.matColReg );
            dstMat.MaterialName = srcMat.name;

#if false

#else
      dstMat.TevData.SetTevColors( Convert_RGBAColor_( srcMat.tevColReg), 0 );

      for( int i = 0; i < 4; i++ ) {
          dstMat.TevData.SetTevKColors( Convert_RGBAColor_( srcMat.tevConstReg[ i ] ), i );
      }
#endif

#if false

#endif

            // Tev - TevStage
            if( srcMat.tevStage != null )
            {
                dstMat.TevData.TevStages.NumStages = srcMat.tevStageNum;
                // 現在の仕様では、必ず等しくなるはずです。
                Debug.Assert( srcMat.tevStage.Length == srcMat.tevStageNum );

                for( int i = 0 ; i < srcMat.tevStage.Length ; i++ )
                {
                    Convert_ToTevStage( dstMat.TevData.TevStages[i], srcMat.tevStage[i] );
                }
            }

#if false

#endif

            // Textrue に関する情報
            // 通常のマテリアルのロードで初期化は完了しているはずなので行わない。
        }


        #endregion

        #region App => File

        #region TEV関連列挙型 変換

        /// <summary>
        /// AppData.AttrBlendFactor => FileFmt.BlendFactorDst
        /// </summary>
        FileFmt.BlendFactorDst
            Convert_ToBlendFactorDst_( AttrBlendFactor srcFactor )
        {
            switch( srcFactor )
            {
                case AttrBlendFactor.One: return FileFmt.BlendFactorDst.V1_0;
                case AttrBlendFactor.Zero: return FileFmt.BlendFactorDst.V0;

                case AttrBlendFactor.SrcColor: return FileFmt.BlendFactorDst.SrcClr;
                case AttrBlendFactor.SrcAlpha: return FileFmt.BlendFactorDst.SrcAlpha;
                case AttrBlendFactor.DstAlpha: return FileFmt.BlendFactorDst.DstAlpha;
                case AttrBlendFactor.InvSrcColor: return FileFmt.BlendFactorDst.InvSrcClr;
                case AttrBlendFactor.InvSrcAlpha: return FileFmt.BlendFactorDst.InvSrcAlpha;
                case AttrBlendFactor.InvDstAlpha: return FileFmt.BlendFactorDst.InvDstAlpha;

                default: Debug.Assert( false ); return FileFmt.BlendFactorDst.V0;
            }
        }

        /// <summary>
        /// AttrBlendFactor.IPEData => FileFmt.BlendFactorSrc
        /// </summary>
        FileFmt.BlendFactorSrc
            Convert_ToBlendFactorSrc_( AttrBlendFactor srcFactor )
        {
            switch( srcFactor )
            {
                case AttrBlendFactor.One: return FileFmt.BlendFactorSrc.V1_0;
                case AttrBlendFactor.Zero: return FileFmt.BlendFactorSrc.V0;
                case AttrBlendFactor.SrcAlpha: return FileFmt.BlendFactorSrc.SrcAlpha;
                case AttrBlendFactor.DstColor: return FileFmt.BlendFactorSrc.DstClr;
                case AttrBlendFactor.DstAlpha: return FileFmt.BlendFactorSrc.DstAlpha;
                case AttrBlendFactor.InvSrcAlpha: return FileFmt.BlendFactorSrc.InvSrcAlpha;
                case AttrBlendFactor.InvDstColor: return FileFmt.BlendFactorSrc.InvDstClr;
                case AttrBlendFactor.InvDstAlpha: return FileFmt.BlendFactorSrc.InvDstAlpha;
                default: Debug.Assert( false ); return FileFmt.BlendFactorSrc.V0;
            }
        }

        /// <summary>
        ///
        /// </summary>
    private FileFmt.TevMode
            Convert_TevMode_( AttrCombineMode src )
        {
        switch( src )
        {
        case AttrCombineMode.Replace:     return TevMode.Replace;
        case AttrCombineMode.Modulate:    return TevMode.Modulate;
        case AttrCombineMode.Add:         return TevMode.Add;
        case AttrCombineMode.AddSigned:   return TevMode.AddSigned;
        case AttrCombineMode.Interpolate: return TevMode.Interpolate;
        case AttrCombineMode.Subtract:    return TevMode.Subtract;
        case AttrCombineMode.AddMultDMP:  return TevMode.AddMult;
        case AttrCombineMode.MultAddDMP:  return TevMode.MultAdd;
                default: Debug.Assert( false ); return TevMode.Replace;
        }
        }

        /// <summary>
        ///
        /// </summary>
    private FileFmt.TevSource
            Convert_TevSource_( AttrTevSource src )
        {
        switch( src )
        {
        case AttrTevSource.Texture0: return FileFmt.TevSource.Texture0;
        case AttrTevSource.Texture1: return FileFmt.TevSource.Texture1;
        case AttrTevSource.Texture2: return FileFmt.TevSource.Texture2;
        case AttrTevSource.Texture3: return FileFmt.TevSource.Texture01;
        case AttrTevSource.Constant: return FileFmt.TevSource.Constant;
        case AttrTevSource.Primary:  return FileFmt.TevSource.Primary;
        case AttrTevSource.Previous: return FileFmt.TevSource.Previous;
        case AttrTevSource.Register: return FileFmt.TevSource.Register;
                default: Debug.Assert( false ); return FileFmt.TevSource.Texture0;
        }
    }

        /// <summary>
        ///
        /// </summary>
    private FileFmt.TevOpRgb
            Convert_TevOpRgb_( AttrTevOp src )
        {
        switch( src )
        {
        case AttrTevOp.Rgb:      return FileFmt.TevOpRgb.Rgb;
        case AttrTevOp.InvRgb:   return FileFmt.TevOpRgb.InvRgb;
        case AttrTevOp.Alpha:    return FileFmt.TevOpRgb.Alpha;
        case AttrTevOp.InvAlpha: return FileFmt.TevOpRgb.InvAlpha;
        case AttrTevOp.Rrr:      return FileFmt.TevOpRgb.Rrr;
        case AttrTevOp.InvRrr:   return FileFmt.TevOpRgb.InvRrr;
        case AttrTevOp.Ggg:      return FileFmt.TevOpRgb.Ggg;
        case AttrTevOp.InvGgg:   return FileFmt.TevOpRgb.InvGgg;
        case AttrTevOp.Bbb:      return FileFmt.TevOpRgb.Bbb;
        case AttrTevOp.InvBbb:   return FileFmt.TevOpRgb.InvBbb;
                default: Debug.Assert( false ); return FileFmt.TevOpRgb.Rgb;
        }
    }

        /// <summary>
        ///
        /// </summary>
    private FileFmt.TevOpAlp
            Convert_TevOpAlpha_( AttrTevOp src )
        {
        switch( src )
        {
        case AttrTevOp.Alpha:    return FileFmt.TevOpAlp.Alpha;
        case AttrTevOp.InvAlpha: return FileFmt.TevOpAlp.InvAlpha;
        case AttrTevOp.R:        return FileFmt.TevOpAlp.R;
        case AttrTevOp.InvR:     return FileFmt.TevOpAlp.InvR;
        case AttrTevOp.G:        return FileFmt.TevOpAlp.G;
        case AttrTevOp.InvG:     return FileFmt.TevOpAlp.InvG;
        case AttrTevOp.B:        return FileFmt.TevOpAlp.B;
        case AttrTevOp.InvB:     return FileFmt.TevOpAlp.InvB;
                default: Debug.Assert( false ); return FileFmt.TevOpAlp.Alpha;
        }
    }

        /// <summary>
        ///
        /// </summary>
    private FileFmt.TevArgRgb
            Convert_TevArgRgb_( DirectStageTevArg src )
        {
        FileFmt.TevArgRgb       arg = new FileFmt.TevArgRgb();

        arg.source = Convert_TevSource_( src.Source);
        arg.op     = Convert_TevOpRgb_( src.Op);

        return arg;
    }

    /// <summary>
        ///
        /// </summary>
        FileFmt.TevScale
            Convert_TevScale_( AttrTevScale src )
        {
            switch( src )
            {
                case AttrTevScale.Scale1: return TevScale.V1;
                case AttrTevScale.Scale2: return TevScale.V2;
                case AttrTevScale.Scale4: return TevScale.V4;
                default: Debug.Assert( false ); return TevScale.V1;
            }
        }

        /// <summary>
        /// 右の型から左の型への変換
        /// 文字列が同一である型の変換に利用します(大文字、小文字を区別しません)。
        /// </summary>
        object ConvertEnumTypesFromRToL_( Enum type0, Enum type1 )
        {
            return Enum.Parse( type0.GetType(), type1.ToString(), true );
        }

        #endregion TEV関連列挙型 変換

        #region 型変換

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoAlphaCompare
        /// </summary>
        FileFmt.Material_CTRAlphaCompare
            Convert_ToFileRevoAlphaCompare_( AppData.IPEData appPE )
        {
            FileFmt.Material_CTRAlphaCompare dstAlphaCompare = new FileFmt.Material_CTRAlphaCompare();
            dstAlphaCompare.comp = (FileFmt.Compare)ConvertEnumTypesFromRToL_( dstAlphaCompare.comp, appPE.ACompare.Comp );
            dstAlphaCompare.@ref = appPE.ACompare.Ref;
            return dstAlphaCompare;
        }

        /// <summary>
        /// AppData.IPEData => FileFmt.Material_RevoBlendMode
        /// </summary>
        FileFmt.Material_CTRBlendMode
            Convert_ToFileRevoBlendMode_( AppData.IPEData appPE )
        {
            FileFmt.Material_CTRBlendMode dstBlendMode = new FileFmt.Material_CTRBlendMode();

            dstBlendMode.dstFactor = (FileFmt.BlendFactorDst)Convert_ToBlendFactorDst_( appPE.Blend.DstFactor );
            dstBlendMode.srcFactor = (FileFmt.BlendFactorSrc)Convert_ToBlendFactorSrc_( appPE.Blend.SrcFactor );
            dstBlendMode.type = (FileFmt.BlendMode)ConvertEnumTypesFromRToL_( dstBlendMode.type, appPE.Blend.Type );

      //
            dstBlendMode.blendOp = (FileFmt.BlendOp)ConvertEnumTypesFromRToL_( dstBlendMode.blendOp, appPE.Blend.BlendOperation );

            return dstBlendMode;
        }

        /// <summary>
        /// AppData.TevDirectStage => FileFmt.Material_RevoTevStageColor
        /// </summary>
    FileFmt.Material_CTRTevStageRgb
      Convert_Material_RevoTevStageColor_(AppData.TevDirectStage srcStage)
    {
        Material_CTRTevStageRgb dstStage = new Material_CTRTevStageRgb();

        dstStage.mode  = Convert_TevMode_( srcStage.CombineMode);
        dstStage.scale = Convert_TevScale_( srcStage.Scale);

        //@@@
        //int numTevStage = appMat.ITevData.NumStages;
        dstStage.arg = new TevArgRgb[ 3 ];
        for( int i = 0; i < 3; i++ )
        {
            dstStage.arg[ i ] = Convert_TevArgRgb_( srcStage.TevArgs[ i ]);
        }

        return dstStage;
    }

        /// <summary>
        /// AppData.TevDirectStage => FileFmt.Material_RevoTevStageAlpha
        /// </summary>
    FileFmt.Material_CTRTevStageAlpha
      Convert_Material_RevoTevStageAlpha_(AppData.TevDirectStage srcStage)
    {
        Material_CTRTevStageAlpha dstStage = new Material_CTRTevStageAlpha();

        //@@
      //dstStage.a = Convert_TevAlphaArg_(srcStage.A);
      //dstStage.b = Convert_TevAlphaArg_(srcStage.B);
      //dstStage.c = Convert_TevAlphaArg_(srcStage.C);
      //dstStage.d = Convert_TevAlphaArg_(srcStage.D);

      //dstStage.bias = Convert_TevBias_(srcStage.Bias);
      //dstStage.clamp = srcStage.Clamp;

      //dstStage.konst = Convert_TevKAlphaSel_(srcStage.KSel);
      //dstStage.op = Convert_TevOpA_(srcStage.Op);
      dstStage.scale = Convert_TevScale_(srcStage.Scale);

      //dstStage.outReg = Convert_TevRegID_(srcStage.OutReg);

      return dstStage;
    }

        /// <summary>
        /// AppData.TevIndirectStage => FileFmt.Material_RevoTevStageIndirect
        /// </summary>
    //FileFmt.Material_RevoTevStageIndirect
    //  Convert_Material_RevoTevStageIndirect_( AppData.TevIndirectStage srcStage )
    //{
    //  Material_RevoTevStageIndirect dstStage = new Material_RevoTevStageIndirect();

    //  dstStage.addPrev = srcStage.AddPrev;
    //  dstStage.alpha = IndTexAlphaSel.Off;// srcStage.AlphaSel; OK ?
    //  dstStage.bias = (IndTexBiasSel)srcStage.BiasSel;
    //  dstStage.format = (IndTexFormat)srcStage.Format;
    //  dstStage.indStage = (Byte)srcStage.IndStage;
    //  dstStage.matrix = (IndTexMtxID)srcStage.MatrixSel;
    //  dstStage.utcLod = srcStage.UtcLod;
    //  dstStage.wrap_s = (IndTexWrap)srcStage.WrapS;
    //  dstStage.wrap_t = (IndTexWrap)srcStage.WrapT;

    //  return dstStage;
    //}

        /// <summary>
        /// AppData.TevStage => FileFmt.Material_RevoTevStage
        /// </summary>
        FileFmt.Material_CTRTevStage
            Convert_RevoTevStage_( AppData.ITevStage srcStage )
        {
        Material_CTRTevStage dstStage = new Material_CTRTevStage();

        // カラーステージ
        dstStage.rgb = Convert_Material_RevoTevStageColor_( srcStage.ColorStage );
        //// アルファステージ
        dstStage.alpha = Convert_Material_RevoTevStageAlpha_( srcStage.AlphaStage );
        return dstStage;
        }

        /// <summary>
        /// AppData.ITexMtx => FileFmt.Material_RevoIndirectMatrix
        /// </summary>
        FileFmt.TexMatrix
            Convert_IndirectMatrix_( AppData.ITexMtx srcIndMtx )
        {
            FileFmt.TexMatrix dstIndMtx =
                new FileFmt.TexMatrix();

            dstIndMtx.rotate = srcIndMtx.Rotate;

            dstIndMtx.scale = Convert_Vec2_( srcIndMtx.Scale );
            dstIndMtx.translate = Convert_Vec2_( srcIndMtx.Trans );

            return dstIndMtx;
        }

        #endregion

        /// <summary>
        /// AppData.IRevHWMaterial => FileFmt.Material_Revo
        /// </summary>
        FileFmt.Material_CTR Convert_ToFileMaterial_Revo_( AppData.IRevHWMaterial appMat )
        {
            FileFmt.Material_CTR dstMat = new Material_CTR();

            // ChannelControl 関連
      //dstMat.channelControl = new FileFmt.Material_RevoChannelControl[2];
      //dstMat.channelControl[0] = new FileFmt.Material_RevoChannelControl();
      //dstMat.channelControl[0].channel = ChannelID.Color0;
      //dstMat.channelControl[0].materialSource = Convert_ToFileColorSource_( appMat.IChannelData.DiffuseCtrlColorSrc );

      //dstMat.channelControl[1] = new FileFmt.Material_RevoChannelControl();
      //dstMat.channelControl[1].channel = ChannelID.Alpha0;
      //dstMat.channelControl[1].materialSource = Convert_ToFileColorSource_( appMat.IChannelData.AlphaCtrlColorSrc );

            //dstMat.matColReg = Convert_Color4_( appMat.IChannelData.MatColReg0 );
            dstMat.name = appMat.MaterialName;



            // PE - AlphaCompare
            dstMat.alphaCompare = Convert_ToFileRevoAlphaCompare_( appMat.IPEData );
            dstMat.blendMode = Convert_ToFileRevoBlendMode_( appMat.IPEData );

#if false
            // Tev - Colorレジスタ
            dstMat.tevColReg = new FileFmt.ColorS10_4[3];
            for( int i = 0 ; i < 3 ; i++ )
            {
                dstMat.tevColReg[i] = Convert_ColorS10_4_( appMat.ITevData.GetTevColors( i ) );
            }

            // Tev - コンスタント レジスタ
            dstMat.tevConstReg = new FileFmt.Color4[4];
            for( int i = 0 ; i < 4 ; i++ )
            {
                dstMat.tevConstReg[i] = Convert_Color4_( appMat.ITevData.GetTevKColors( i ) );
            }

            // Tev - SwapTable
            dstMat.swapTable = new Material_RevoSwapTable[4];
            for( int i = 0 ; i < 4 ; i++ )
            {
                dstMat.swapTable[i] = Convert_ToFileRevoSwapTable_( appMat.ITevData.ISwapTables[i] );
            }
#else
      dstMat.tevColReg = Convert_Color4_( appMat.ITevData.GetTevColors( 0 ) );

      dstMat.tevConstReg = new Color4[ 4 ];
      for( int i = 0; i < 4; i++ ) {
          dstMat.tevConstReg[ i ] = Convert_Color4_( appMat.ITevData.GetTevKColors( i ) );
      }
#endif

            // Tev - TevStage
            // 現在は、使用される情報のみ編集可能となっているので、
            // 保存すべきステージの数 = 使用されるステージの数です。
            int numTevStage = appMat.ITevData.NumStages;
            dstMat.tevStage = new FileFmt.Material_CTRTevStage[numTevStage];
            dstMat.tevStageNum = (byte)numTevStage;
            for( int i = 0 ; i < dstMat.tevStage.Length ; i++ )
            {
                dstMat.tevStage[i] = Convert_RevoTevStage_( appMat.ITevData.GetITevStage( i ) );
            }

#if false
            // Tev - インダイレクト行列
            dstMat.indirectMatrix = new FileFmt.TexMatrix[appMat.IIndirectTexMtxSet.NumTexIndirectMtx];
            for( int i = 0 ; i < dstMat.indirectMatrix.Length ; i++ )
            {
                dstMat.indirectMatrix[i] = Convert_IndirectMatrix_( appMat.IIndirectTexMtxSet[i] );
            }

            // インダイレクトステージ
            dstMat.indirectStage = new FileFmt.Material_RevoIndirectStage[appMat.ITexIndirectStageSet.NumTexIndirectStages];
            dstMat.indirectStageNum = (byte)appMat.ITexIndirectStageSet.NumStageUsed;
            for( int i = 0 ; i < dstMat.indirectStage.Length ; i++ )
            {
                dstMat.indirectStage[i] = Convert_IndirectStage_( appMat.ITexIndirectStageSet[i] );
            }
#endif



            // Textrue に関する情報
            AppData.IMaterialTexMap[] texMaps = appMat.IMaterialTexMapSet;

            dstMat.texMap = new FileFmt.TexMap[texMaps.Length];
            dstMat.texMatrix = new FileFmt.TexMatrix[texMaps.Length];
            dstMat.texCoordGen = new FileFmt.TexCoordGen[texMaps.Length];

            for( int i = 0 ; i < texMaps.Length ; i++ )
            {
                if( texMaps[i] != null )
                {
                    dstMat.texMap[i] = Convert_ToFileTexMap_( texMaps[i] );
                    dstMat.texMatrix[i] = Convert_ToFileTexMtx_( texMaps[i].ITexGen.ITexMtx );
                    dstMat.texCoordGen[i] = Convert_ToFileTexCoordGen_( texMaps[i].ITexGen );
                }
            }

            return dstMat;
        }


        #endregion
    }
}
