﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Linq;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    /// <summary>
    /// XMLシリアライズ可能なクラスへの変換。
    /// (Materialに関する部分)
    /// </summary>
    public partial class RlytConverter
    {
        #region File => App

        /// <summary>
        /// テクスチャの読み込み
        /// </summary>
        void Convert_Pane_MaterialTexture_(
            AppData.MaterialTexMap	dstMatTex,
            FileFmt.TexMap				srcTexMap,
            FileFmt.TexBlendRatio		srcBlendRatio,
            FileFmt.TexCoordGen		srcTexCoordGen )
        {
            Debug.Assert( dstMatTex != null );
            Debug.Assert( srcTexCoordGen != null );

            // MaterialTexMap への値の設定
            // TODO:enum 型については、強制的にキャストしています。
            dstMatTex.MagFilter = (AttrTexFilterMag)srcTexMap.magFilter;
            dstMatTex.MinFilter = (AttrTexFilterMin)srcTexMap.minFilter;
            dstMatTex.WrapS = (AttrTexWrap)srcTexMap.wrap_s;
            dstMatTex.WrapT = (AttrTexWrap)srcTexMap.wrap_t;
            dstMatTex.ResourceType = (AttrTextureResourceType)srcTexMap.textureResourceType;

            dstMatTex.TexGen.Method = Convert_LETexGenMethod_( srcTexCoordGen.srcParam );
            dstMatTex.TexGen.ProjectionScale = SetFVec2IfExist_(srcTexCoordGen.projectionScale, dstMatTex.TexGen.ProjectionScale);
            dstMatTex.TexGen.ProjectionOffset = SetFVec2IfExist_(srcTexCoordGen.projectionTrans, dstMatTex.TexGen.ProjectionOffset);
            dstMatTex.TexGen.IsFittingLayoutSizeEnabled = srcTexCoordGen.fittingLayoutSizeEnabled;
            dstMatTex.TexGen.IsFittingPaneSizeEnabled = srcTexCoordGen.fittingPaneSizeEnabled;
            dstMatTex.TexGen.IsAdjustProjectionPaneSREnabled = srcTexCoordGen.adjustProjectionSREnabled;
        }

        /// <summary>
        /// テクスチャ行列の読み込み
        /// </summary>
        void Convert_Pane_MaterialTexMtx_(
            AppData.TexMtx dstMtx,
            FileFmt.TexMatrix srcMtx )
        {
            dstMtx.Trans = new FVec2( srcMtx.translate.x, srcMtx.translate.y );
            dstMtx.Scale = new FVec2( srcMtx.scale.x, srcMtx.scale.y );
            dstMtx.Rotate = srcMtx.rotate;
        }

        /// <summary>
        /// FileFmt.Material => AppData.Material
        /// </summary>
        void Convert_Pane_Material_ColorBlend_(AppData.Material dstMat, FileFmt.Material srcMat)
        {
            Debug.Assert(dstMat != null);
            Debug.Assert(srcMat != null);

            if (srcMat.blackColorFloat != null)
            {
                dstMat.BlackColor = Convert_FloatColor_(srcMat.blackColorFloat);
            }
            else
            {
                dstMat.BlackColor = Convert_FloatColor_(srcMat.blackColor);
            }

            if (srcMat.whiteColorFloat != null)
            {
                dstMat.WhiteColor = Convert_FloatColor_(srcMat.whiteColorFloat);
            }
            else
            {
                dstMat.WhiteColor = Convert_FloatColor_(srcMat.whiteColor);
            }

            dstMat.IsThresholdingAlphaInterpolationEnabled = srcMat.isThresholdingAlphaInterpolationEnabled;
        }

        /// <summary>
        /// FileFmt.Material => AppData.Material
        /// </summary>
        void Convert_Pane_Material_Texture_(AppData.Material dstMat, FileFmt.Material srcMat)
        {
            Debug.Assert(dstMat != null);
            Debug.Assert(srcMat != null);

            if (srcMat.hsbAdjustment != null)
            {
                dstMat.HOffset = srcMat.hsbAdjustment.hOffset;
                dstMat.SScale = srcMat.hsbAdjustment.sScale;
                dstMat.BScale = srcMat.hsbAdjustment.bScale;
            }

            // マテリアルがテクスチャを持っている場合はテクスチャを読み込みます。
            // (TextBoxペインは、テクスチャを持たない場合が多いはずです。)
            if (srcMat.texMap != null)
            {
                Debug.Assert(srcMat.texBlendRatio == null || srcMat.texBlendRatio.Length == srcMat.texMap.Length);

                //-------------------------------------
                // マテリアルテクスチャの読み込み
                dstMat.RemoveMatTextureAllNoEvent();

                int numTexMap = srcMat.texMap.Length;
                for (int i = 0; i < numTexMap; i++)
                {
                    MaterialTexMap dstMatTex = dstMat.GetMaterialTexMapByIndex(i);
                    // テクスチャ番号と同一番号のTexCoordGenを使用する仕様になっています。
                    Debug.Assert(srcMat.texMap.Length == srcMat.texCoordGen.Length);

                    if (dstMatTex != null)
                    {
                        FileFmt.TexMap srcTexMap = srcMat.texMap[i];
                        dstMatTex = dstMat.RegisterMatTexture(srcTexMap.imageName, i, (AttrTextureResourceType)srcTexMap.textureResourceType);
                    }
                    Debug.Assert(dstMatTex != null);

                    Convert_Pane_MaterialTexture_(
                        dstMatTex,
                        srcMat.texMap[i],
                        srcMat.texBlendRatio != null ? srcMat.texBlendRatio[i] : null,
                        srcMat.texCoordGen[i]);
                }

                //-------------------------------------
                // TexMatrixの読み込み
                Debug.Assert(dstMat.TexMtxSet.Length >= srcMat.texMatrix.Length);
                for (int i = 0; i < srcMat.texMatrix.Length; i++)
                {
                    Convert_Pane_MaterialTexMtx_(dstMat.TexMtxSet[i], srcMat.texMatrix[i]);
                }
            }
        }

        /// <summary>
        /// FileFmt.Material => AppData.Material
        /// </summary>
        void Convert_Pane_Material_( AppData.Material dstMat, FileFmt.Material srcMat )
        {
            Debug.Assert( dstMat != null );
            Debug.Assert( srcMat != null );

            Convert_Pane_Material_ColorBlend_(dstMat, srcMat);

            Convert_Pane_Material_Texture_(dstMat, srcMat);
        }

        #endregion

        #region App => File

        void Convert_ToFileMaterialColorBlend_(AppData.IColorBlendHolder appMat, FileFmt.Material dstMat)
        {
            // 実際にデータに FloatColor が設定されていたら FloatColor 用のタグを出力する。
            // FloatColor が有効な状態でもなるべく古いタグの状態にしておくため。
            if (!appMat.BlackColor.ByteConvert || !appMat.WhiteColor.ByteConvert)
            {
                dstMat.blackColorFloat = Convert_BlackColor_(appMat.BlackColor);
                dstMat.whiteColorFloat = Convert_WhiteColor_(appMat.WhiteColor);
                dstMat.blackColor = null;
                dstMat.whiteColor = null;
            }
            else
            {
                dstMat.blackColor = Convert_BlackColor_(appMat.BlackColor.ToSystemColor());
                dstMat.whiteColor = Convert_WhiteColor_(appMat.WhiteColor.ToSystemColor());
                dstMat.blackColorFloat = null;
                dstMat.whiteColorFloat = null;
            }
            dstMat.isThresholdingAlphaInterpolationEnabled = appMat.IsThresholdingAlphaInterpolationEnabled;
        }

        /// <summary>
        /// AppData.IMaterial => FileFmt.Material
        /// </summary>

        FileFmt.Material Convert_ToFileMaterial_( AppData.IMaterial appMat )
        {
            FileFmt.Material fileMat = null;
            if( appMat != null )
            {
                fileMat = new FileFmt.Material();

                Convert_ToFileMaterialColorBlend_(appMat, fileMat);

                fileMat.name = appMat.MaterialName;

                fileMat.hsbAdjustment = new MaterialHSBAdjustment();
                fileMat.hsbAdjustment.hOffset = appMat.HOffset;
                fileMat.hsbAdjustment.sScale = appMat.SScale;
                fileMat.hsbAdjustment.bScale = appMat.BScale;

                IMaterialTexMap[] texMapSet = appMat.IMaterialTexMapSet;
                int numTexMap = texMapSet.Length;

                fileMat.texMap = new FileFmt.TexMap[numTexMap];
                fileMat.texBlendRatio = null;
                fileMat.textureStage = new FileFmt.MaterialTextureStage[numTexMap];

                fileMat.texMatrix = new FileFmt.TexMatrix[numTexMap];
                fileMat.texCoordGen = new FileFmt.TexCoordGen[numTexMap];

                // すべてのテクスチャについて....
                for( int i = 0 ; i < texMapSet.Length ; i++ )
                {
                    IMaterialTexMap srcTex = texMapSet[i];
                    if( srcTex != null )
                    {
                        // ----------- FileFmt.MaterialTextureStage について
                        fileMat.textureStage[i] = Convert_ToFileTexStage_( srcTex );

                        // ----------- FileFmt.TexMap について
                        // enum については、全く同じ内容なので、強制キャストしています。
                        fileMat.texMap[i] = Convert_ToFileTexMap_( srcTex );

                        // ----------- テクスチャ座標生成関連パラメータについて
                        fileMat.texMatrix[i] = Convert_ToFileTexMtx_( srcTex.ITexGen.ITexMtx );
                        fileMat.texCoordGen[i] = Convert_ToFileTexCoordGen_( srcTex.ITexGen );
                    }
                }
            }

            return fileMat;
        }

#region 型変換
        /// <summary>
        /// AppData.IMaterialTexMap => FileFmt.TexCoord
        /// </summary>
        FileFmt.TexCoord
            Convert_ToFileTexCoord_( AppData.TexCoord4 appTexCoord )
        {
            FileFmt.TexCoord dstTexCoord = new FileFmt.TexCoord();

            dstTexCoord.texLT = Convert_TexVec2_( appTexCoord.LT );
            dstTexCoord.texRT = Convert_TexVec2_( appTexCoord.RT );
            dstTexCoord.texLB = Convert_TexVec2_( appTexCoord.LB );
            dstTexCoord.texRB = Convert_TexVec2_( appTexCoord.RB );

            return dstTexCoord;
        }

        /// <summary>
        /// AppData.IMaterialTexMap => FileFmt.TexMap
        /// </summary>
        FileFmt.MaterialTextureStage
            Convert_ToFileTexStage_( AppData.IMaterialTexMap appTex )
        {
            FileFmt.MaterialTextureStage dstTexStage = new MaterialTextureStage();

      //dstTexStage.texCoordGen = (SByte)appTex.SlotIdx;
            dstTexStage.texMap = (SByte)appTex.SlotIdx;

            return dstTexStage;
        }

        /// <summary>
        /// AppData.IMaterialTexMap => FileFmt.TexMap
        /// </summary>
        FileFmt.TexMap
            Convert_ToFileTexMap_( AppData.IMaterialTexMap appTex )
        {
            FileFmt.TexMap dstTexMap = new FileFmt.TexMap();

            dstTexMap.wrap_s = (FileFmt.TexWrapMode)appTex.WrapS;
            dstTexMap.wrap_t = (FileFmt.TexWrapMode)appTex.WrapT;
            dstTexMap.imageName = appTex.TexImgName;
      //dstTexMap.paletteName = null; // TODO:
            dstTexMap.minFilter = (TexFilter)appTex.MinFilter;
            dstTexMap.magFilter = (TexFilter)appTex.MagFilter;
            dstTexMap.textureResourceType = (FileFmt.TextureResourceType)appTex.ResourceType;

            return dstTexMap;
        }

        /// <summary>
        /// AttrTexCoordMtx 型を SByte に変換します。
        /// Identity が -1 に変換されます。
        /// </summary>
        SByte
            Convert_ToFileTexCoordGenMtxID_( AttrTexCoordMtx mtxID )
        {
            if( mtxID == AttrTexCoordMtx.Identity )
            {
                return -1;
            }
            else
            {
                return (SByte)mtxID;
            }
        }

        /// <summary>
        /// AppData.ITexGen => FileFmt.TexCoordGen
        /// FIXME:enum 型等は 強引にキャストしています。
        /// </summary>
        FileFmt.TexCoordGen
            Convert_ToFileTexCoordGen_( AppData.ITexGen appTexGen )
        {

            FileFmt.TexCoordGen dstGen = new TexCoordGen();
            dstGen.srcParam = (FileFmt.TexGenSrc)appTexGen.Method;
            dstGen.projectionScale = Convert_Vec2_(appTexGen.ProjectionScale);
            dstGen.projectionTrans = Convert_Vec2_(appTexGen.ProjectionOffset);
            dstGen.fittingLayoutSizeEnabled = appTexGen.IsFittingLayoutSizeEnabled;
            dstGen.fittingPaneSizeEnabled = appTexGen.IsFittingPaneSizeEnabled;
            dstGen.adjustProjectionSREnabled = appTexGen.IsAdjustProjectionPaneSREnabled;

            return dstGen;
        }

        /// <summary>
        /// AppData.ITexMtx => FileFmt.TexCoordGen
        /// FIXME:enum 型等は 強引にキャストしています。
        /// </summary>
        FileFmt.TexMatrix
            Convert_ToFileTexMtx_( AppData.ITexMtx appMtx )
        {

            FileFmt.TexMatrix dstMtx = new TexMatrix();

            dstMtx.translate = Convert_Vec2_( appMtx.Trans );
            dstMtx.scale = Convert_Vec2_( appMtx.Scale );
            dstMtx.rotate = appMtx.Rotate;

            return dstMtx;
        }
#endregion 型変換

#endregion
    }
}
