﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    /// <summary>
    /// XMLシリアライズ可能なクラスへの変換。
    /// </summary>
    public partial class RlytConverter
    {
        #region App => File

        #region グループ 階層関連

        #region グループ
        /// <summary>
        /// PaneModify から GroupPaneRef へと変換します。
        /// </summary>
        GroupPaneRef Convert_GroupPaneRef_( AppData.Pane pane )
        {
            GroupPaneRef result = new GroupPaneRef();

            result.name = pane.PaneName;

            return result;
        }

        /// <summary>
        /// GroupModify へと変換します。
        /// </summary>
        FileFmt.Group Convert_Group_( ILEGroup srcGroup )
        {
            FileFmt.Group dstGroup = new Group();
            List<FileFmt.Group> memberGroupSet = new List<FileFmt.Group>();
            List<FileFmt.GroupPaneRef> memberPaneSet = new List<FileFmt.GroupPaneRef>();

            // グループのメンバ情報をコレクションに格納します。
            if( srcGroup.HasMember )
            {
                IPane[] members = srcGroup.Member;
                foreach( IPane member in members )
                {
                    // メンバとして登録します。
                    // 種類によって別のコレクションに格納されます。
                    // 現在はルートグループ以外は、必ず、ペインとして出力されます。
                    GroupPaneRef memberPane = new GroupPaneRef();
                    memberPane.name = member.PaneName;
                    memberPaneSet.Add( memberPane );
                }
            }

            dstGroup.comment = srcGroup.UserCommentString;
            dstGroup.name = srcGroup.GrouprName;
            dstGroup.group = memberGroupSet.ToArray();
            dstGroup.paneRef = memberPaneSet.ToArray();

            return dstGroup;
        }

        /// <summary>
        /// FileFmt.GroupSet へと変換します。
        /// </summary>
        FileFmt.GroupSet Convert_SOGroupSet_( ISubScene scene )
        {
            ILEGroupMgr groupMgr = scene.ILEGroupMgr;
            List<FileFmt.Group> dstGroupSet = new List<FileFmt.Group>();

            // 全てのグループについて...
            OutLog_( "LECORE_RLYTLOAD_GROUP" );
            foreach( ILEGroup srcGroup in groupMgr.GroupSet )
            {
                try
                {
                    FileFmt.Group dstGroup = Convert_Group_( srcGroup );
                    if( dstGroup != null )
                    {
                        dstGroupSet.Add( dstGroup );
                    }
                    else
                    {
                        Debug.Assert( false );
                    }
                }
                catch( Exception e )
                {
                    _msgReporter.ReportError( LECoreStringResMgr.Get( "LECORE_RLYTLOAD_GROUP" ), e.Message );
                }
            }


            FileFmt.GroupSet result = new FileFmt.GroupSet();
            result.group = new FileFmt.Group();
            result.group.name = groupMgr.RootNodeName;
            result.group.group = dstGroupSet.ToArray();
            result.group.comment = "";
            return result;
        }
        #endregion グループ

        #region 親子階層
        void Convert_PaneTree_SubTree_( AppData.IPane pane, PaneTree paneTree )
        {
            IHierarchyNode[] chilren = pane.Children;

            paneTree.name = pane.PaneName;
            paneTree.paneTree = new PaneTree[chilren.Length];

            for( int i = 0 ; i < chilren.Length ; i++ )
            {
                AppData.IPane child = chilren[i] as AppData.IPane;
                PaneTree subTree = new PaneTree();

                // 子供について再帰的に処理をおこないます。
                Convert_PaneTree_SubTree_( child, subTree );

                paneTree.paneTree[i] = subTree;
            }
        }

        /// <summary>
        /// PaneTree へと 変換します。
        /// </summary>
        PaneTree Convert_PaneTree_( ISubScene scene )
        {
            PaneTree result = new PaneTree();

            Convert_PaneTree_SubTree_( scene.RootIPane, result );

            return result;
        }

        /// <summary>
        /// FileFmt.PaneHierarchy へと変換します。
        /// </summary>
        FileFmt.PaneHierarchy Convert_SOPaneHierarchy_( ISubScene scene )
        {
            OutLog_( "LECORE_RLYTLOAD_HIER" );
            try
            {
                FileFmt.PaneHierarchy result = new FileFmt.PaneHierarchy();
                result.paneTree = Convert_PaneTree_( scene );
                return result;
            }
            catch( Exception e )
            {
                string title = LECoreStringResMgr.Get( "LECORE_RLYTLOAD_HIER" );
                _msgReporter.ReportError( title, e.Message );
                return null;
            }
        }
        #endregion 親子階層

        #endregion グループ 階層関連

        #endregion

        #region File => App

        #region 階層・グループ関連
        /// <summary>
        /// 階層構造をシーンに読み込みます。
        /// </summary>
        void Convert_LayoutData_Herarchy_( FileFmt.LYT rlyt, ISubScene subScene )
        {
            if( rlyt.paneHierarchy != null )
            {
                try
                {
                    OutLog_( "LECORE_RLYTLOAD_HIER" );
                    PaneTree rootTree = rlyt.paneHierarchy.paneTree;
                    Debug.Assert( rootTree != null );
                    AppData.IPane rootPane = SearchPaneFromScene_( subScene, rootTree.name );
                    Debug.Assert( rootPane != null );

                    BuildHierarchyConnection_( subScene, rootPane, rootTree );
                }
                catch
                {

                    string msg = LECoreStringResMgr.Get( "LECORE_SYS_ERROR_LOAD_HIERARCHY" );
                    ReportErrorMsg_( msg );
                }
            }
        }

        /// <summary>
        /// グループの新規作成
        ///
        /// 名前の重複するグループの登録が起こった場合は、
        /// 何も登録を行いません。
        /// </summary>
        LEGroup MakeNewGroup_(
            LEGroupMgr groupMgr,
            FileFmt.Group srcGroup,
            IPane[] groupMemberSet )
        {
            // すでに登録されていなれば...
            LEGroup newGroup = groupMgr.FindGroupByName( srcGroup.name ) as LEGroup;
            if( newGroup == null )
            {
                // ペインの配列からグループを作成する。
                newGroup = groupMgr.MakeNewGroup( srcGroup.name, groupMemberSet );

                // グループのユーザコメントを設定します。
                newGroup.UserCommentString = srcGroup.comment;
            }
            return newGroup;
        }

        /// <summary>
        /// ペインをメンバーとして、グループをシーンにロードします。
        /// </summary>
        LEGroup Convert_LayoutData_Group_( FileFmt.Group srcGroup, ISubScene subScene )
        {
#if false
            Debug.Assert( srcGroup.paneRef != null && srcGroup.group == null );

            LEGroupMgr groupMgr = ( subScene as SubScene ).LEGroupMgr;
            LEGroup newGroup = null;

            if( srcGroup.paneRef != null )
            {
                // ペインをグループ登録します。
                if( srcGroup.paneRef.Length != 0 )
                {
                    // グループのメンバとなるペインの配列を作成する
                    IPane[] memberPaneSet = new IPane[srcGroup.paneRef.Length];
                    for( int i = 0 ; i < srcGroup.paneRef.Length ; i++ )
                    {
                        GroupPaneRef memberPaneRef = srcGroup.paneRef[i];
                        memberPaneSet[i] = SearchPaneFromScene_( subScene, memberPaneRef.name );
                        Debug.Assert( memberPaneSet[i] != null );
                    }
                    // 登録する。
                    newGroup = MakeNewGroup_( groupMgr, srcGroup, memberPaneSet );
                }
            }
#else

            LEGroupMgr groupMgr = ( subScene as SubScene ).LEGroupMgr;
            LEGroup newGroup = null;
      IPane[] memberPaneSet = null;

            if( srcGroup.paneRef != null )
            {
          // ペインをグループ登録します。
          if( srcGroup.paneRef.Length != 0 )
          {
              // グループのメンバとなるペインの配列を作成する
              memberPaneSet = new IPane[srcGroup.paneRef.Length];
              for( int i = 0 ; i < srcGroup.paneRef.Length ; i++ )
              {
                  GroupPaneRef memberPaneRef = srcGroup.paneRef[i];
                  memberPaneSet[i] = SearchPaneFromScene_( subScene, memberPaneRef.name );
                  Debug.Assert( memberPaneSet[i] != null );
              }
          }
            }

      // 登録する。
      newGroup = MakeNewGroup_( groupMgr, srcGroup, memberPaneSet );
#endif

            return newGroup;
        }

        /// <summary>
        /// グループセットをシーンにロードします。
        /// </summary>
        void Convert_LayoutData_GroupSet_( FileFmt.LYT rlyt, ISubScene subScene )
        {
            if( rlyt.groupSet != null &&
                rlyt.groupSet.group != null &&
                rlyt.groupSet.group.group != null )
            {
                // 全てのグループについて...
                OutLog_( "LECORE_RLYTLOAD_GROUP" );
                foreach( FileFmt.Group group in rlyt.groupSet.group.group )
                {
                    try
                    {
                        Convert_LayoutData_Group_( group, subScene );
                    }
                    catch
                    {
                        string msg = LECoreStringResMgr.Get( "LECORE_SYS_ERROR_LOAD_GROUP" );
                        ReportErrorMsg_( msg, group.name );
                    }
                }
            }
        }

        #endregion 階層・グループ関連

        #endregion
    }
}
