﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Text;
using System.Linq;

namespace LECore.Save_Load
{
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Structures.LECoreInterface;

    using LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Nsrif.Attributes;

    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lyt;

    /// <summary>
    /// XMLシリアライズ可能なクラスへの変換。
    /// (基本的なデータ構造に関する部分)
    /// </summary>
    public partial class RlytConverter
    {
        #region File => LECore内部クラス への変換
        #region 基本的な型の変換
        /// <summary>
        /// Vec2 => FVec2
        /// </summary>
        FVec2 Convert_FVec2_( Vec2 srcVec )
        {
            FVec2 result = new FVec2();

            if (srcVec != null)
            {
                result.X = srcVec.x;
                result.Y = srcVec.y;
            }

            return result;
        }

        /// <summary>
        /// Vec2 => FVec2
        /// </summary>
        FVec2 SetFVec2IfExist_(Vec2 srcVec, FVec2 dst)
        {
            if (srcVec != null)
            {
                return new FVec2(srcVec.x, srcVec.y);
            }
            else
            {
                return dst;
            }
        }

        /// <summary>
        /// TexVec2 => FVec2
        /// </summary>
        FVec2 Convert_FVec2_( TexVec2 srcVec )
        {
            FVec2 result = new FVec2();

            if (srcVec != null)
            {
                result.X = srcVec.s;
                result.Y = srcVec.t;
            }

            return result;
        }

        /// <summary>
        /// TexCoord => TexCoord4
        /// </summary>
        TexCoord4 Convert_TexCoord4_( TexCoord srcCoord )
        {
            TexCoord4 dstCoord = new TexCoord4();

            if (srcCoord != null)
            {
                dstCoord.LT = Convert_FVec2_(srcCoord.texLT);
                dstCoord.RT = Convert_FVec2_(srcCoord.texRT);
                dstCoord.LB = Convert_FVec2_(srcCoord.texLB);
                dstCoord.RB = Convert_FVec2_(srcCoord.texRB);
            }

            return dstCoord;
        }

        /// <summary>
        /// FVec3 => Vec2
        /// </summary>
        FVec3 Convert_FVec3_( Vec2 srcVec )
        {
            FVec3 result = new FVec3();

            if (srcVec != null)
            {
                result.X = (float)srcVec.x;
                result.Y = (float)srcVec.y;
                result.Z = 0.0f;
            }

            return result;
        }

        /// <summary>
        /// Vec3 => FVec3
        /// </summary>
        FVec3 SetFVec3IfExist_(Vec3 srcVec, FVec3 dst)
        {
            if (srcVec != null)
            {
                return new FVec3(srcVec.x, srcVec.y, srcVec.z);
            }
            else
            {
                return dst;
            }
        }

        /// <summary>
        /// Vec3 => FVec3
        /// </summary>
        FVec3 Convert_FVec3_( Vec3 srcVec )
        {
            FVec3 result = new FVec3();

            if (srcVec != null)
            {
                result.X = (float)srcVec.x;
                result.Y = (float)srcVec.y;
                result.Z = (float)srcVec.z;
            }

            return result;
        }


        /// <summary>
        /// Color4  => RGBAColor
        /// </summary>
        RGBAColor Convert_RGBAColor_( Color4 srcColor )
        {
            RGBAColor result = new RGBAColor();
            if (srcColor != null)
            {
                result.R = (byte)srcColor.r;
                result.G = (byte)srcColor.g;
                result.B = (byte)srcColor.b;
                result.A = (byte)srcColor.a;
            }

            return result;
        }

        /// <summary>
        /// Color4  => RGBAColor
        /// </summary>
        RGBAColor SetRGBAColorIfExsit_(Color4 srcColor, RGBAColor dst)
        {
            if (srcColor != null)
            {
                return new RGBAColor((byte)srcColor.r, (byte)srcColor.g, (byte)srcColor.b, (byte)srcColor.a);
            }
            else
            {
                return dst;
            }
        }

        // 注意：
        // スキーマの生成コードが派生クラスの規定値設定をうまく取り扱ってくれないので
        // 別クラスとしています。
        /// <summary>
        /// BlackColor  => RGBAColor
        /// </summary>
        RGBAColor Convert_RGBAColor_( BlackColor srcColor )
        {
            RGBAColor result = new RGBAColor();

            if (srcColor != null)
            {
                result.R = (byte)srcColor.r;
                result.G = (byte)srcColor.g;
                result.B = (byte)srcColor.b;
                result.A = (byte)srcColor.a;
            }

            return result;
        }

        /// <summary>
        /// WhiteColor  => RGBAColor
        /// </summary>
        RGBAColor Convert_RGBAColor_( WhiteColor srcColor )
        {
            RGBAColor result = new RGBAColor();

            if (srcColor != null)
            {
                result.R = (byte)srcColor.r;
                result.G = (byte)srcColor.g;
                result.B = (byte)srcColor.b;
                result.A = (byte)srcColor.a;
            }

            return result;
        }

        /// <summary>
        /// BlackColor  => FloatColor
        /// </summary>
        FloatColor Convert_FloatColor_( BlackColor srcColor )
        {
            FloatColor color = new FloatColor(System.Drawing.Color.FromArgb(srcColor.a, srcColor.r, srcColor.g, srcColor.b));
            color.ByteConvert = true;

            return color;
        }

        /// <summary>
        /// WhiteColor  => FloatColor
        /// </summary>
        FloatColor Convert_FloatColor_( WhiteColor srcColor )
        {
            FloatColor color = new FloatColor(System.Drawing.Color.FromArgb(srcColor.a, srcColor.r, srcColor.g, srcColor.b));
            color.ByteConvert = true;

            return color;
        }

        /// <summary>
        /// BlackColorFloat  => FloatColor
        /// </summary>
        FloatColor Convert_FloatColor_( BlackColorFloat srcColor )
        {
            FloatColor result = new FloatColor();

            if (srcColor != null)
            {
                result.R = srcColor.r;
                result.G = srcColor.g;
                result.B = srcColor.b;
                result.A = srcColor.a;
                result.ByteConvert = srcColor.byte_convert;
            }

            return result;
        }

        /// <summary>
        /// WhiteColorFloat  => FloatColor
        /// </summary>
        FloatColor Convert_FloatColor_( WhiteColorFloat srcColor )
        {
            FloatColor result = new FloatColor();

            if (srcColor != null)
            {
                result.R = srcColor.r;
                result.G = srcColor.g;
                result.B = srcColor.b;
                result.A = srcColor.a;
                result.ByteConvert = srcColor.byte_convert;
            }

            return result;
        }

        /// <summary>
        /// アフィン変換中心(水平)
        /// </summary>
        HorizontalLocation Convert_HorizontalLocation_( Position position )
        {
            if (position == null)
            {
                return HorizontalLocation.Center;
            }

            // 水平
            switch( position.x )
            {
                case HorizontalPosition.Left: return HorizontalLocation.Left;
                case HorizontalPosition.Center: return HorizontalLocation.Center;
                case HorizontalPosition.Right: return HorizontalLocation.Right;
                default: Debug.Assert( false ); return HorizontalLocation.Left;
            }
        }

        /// <summary>
        /// アフィン変換中心(水平)
        /// </summary>
        HorizontalLocation Convert_HorizontalLocation_(HorizontalPosition position)
        {
            // 水平
            switch (position)
            {
                case HorizontalPosition.Left: return HorizontalLocation.Left;
                case HorizontalPosition.Center: return HorizontalLocation.Center;
                case HorizontalPosition.Right: return HorizontalLocation.Right;
                default: Debug.Assert(false); return HorizontalLocation.Left;
            }
        }

        /// <summary>
        /// アフィン変換中心(垂直)
        /// </summary>
        VerticalLocation Convert_VerticalLocation_( Position position )
        {
            if (position == null)
            {
                return VerticalLocation.Center;
            }

            // 垂直
            return Convert_VerticalLocation_(position.y);
        }

        /// <summary>
        /// アフィン変換中心(垂直)
        /// </summary>
        VerticalLocation Convert_VerticalLocation_(VerticalPosition vpos)
        {
            switch (vpos)
            {
                case VerticalPosition.Top: return VerticalLocation.Top;
                case VerticalPosition.Center: return VerticalLocation.Center;
                case VerticalPosition.Bottom: return VerticalLocation.Bottom;
                default: Debug.Assert( false ); return VerticalLocation.Top;
            }
        }

        /// <summary>
        /// 部品倍率モード
        /// </summary>
        AppData.PartsMagnifyInfluence Convert_PartsMagnifyInfluence_(FileFmt.PartsMagnifyInfluence src)
        {
            // 垂直
            switch (src)
            {
                case FileFmt.PartsMagnifyInfluence.None: return AppData.PartsMagnifyInfluence.None;
                case FileFmt.PartsMagnifyInfluence.ScaleMagnify: return AppData.PartsMagnifyInfluence.ScaleMagnify;
                case FileFmt.PartsMagnifyInfluence.AdjustToPartsBound: return AppData.PartsMagnifyInfluence.AdjustToPartsBound;
                default: Debug.Assert(false); return AppData.PartsMagnifyInfluence.ScaleMagnify;
            }
        }

        /// <summary>
        /// 行そろえ位置
        /// </summary>
        LineAlignment Convert_LineAlignment_( TextAlignment textAlignment )
        {
            switch( textAlignment )
            {
                case TextAlignment.Synchronous: return LineAlignment.NotSpecified;
                case TextAlignment.Left: return LineAlignment.Left;
                case TextAlignment.Center: return LineAlignment.Center;
                case TextAlignment.Right: return LineAlignment.Right;
                default: Debug.Assert( false ); return LineAlignment.NotSpecified;
            }
        }

        /// <summary>
        ///
        /// </summary>
        PerCharTransformLoopType Convert_PerCharTransformLoopType_(FileFmt.PerCharacterTransformLoopType loopType)
        {
            switch (loopType)
            {
                case FileFmt.PerCharacterTransformLoopType.OneTime: return PerCharTransformLoopType.OneTime;
                case FileFmt.PerCharacterTransformLoopType.Loop: return PerCharTransformLoopType.Loop;
                default: Debug.Assert(false); return PerCharTransformLoopType.Loop;
            }
        }

        /// <summary>
        /// FileFmt.PaneKind => PaneModify.Kind
        /// </summary>
        AppData.PaneKind Convert_SOPaneKind_( FileFmt.PaneKind kind )
        {
            switch( kind )
            {
                case FileFmt.PaneKind.Null: return AppData.PaneKind.Null;
                case FileFmt.PaneKind.Bounding: return AppData.PaneKind.Bounding;
                case FileFmt.PaneKind.Picture: return AppData.PaneKind.Picture;
                case FileFmt.PaneKind.TextBox: return AppData.PaneKind.Textbox;
                case FileFmt.PaneKind.Window: return AppData.PaneKind.Window;
                case FileFmt.PaneKind.Parts: return AppData.PaneKind.Parts;
                case FileFmt.PaneKind.Capture: return AppData.PaneKind.Capture;
                case FileFmt.PaneKind.Alignment: return AppData.PaneKind.Alignment;
                case FileFmt.PaneKind.Scissor: return AppData.PaneKind.Scissor;
                default: Debug.Assert( false ); return AppData.PaneKind.Null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        AppData.PartsPropertyUsageOptions Convert_PartsPropertyUsageOptions_(FileFmt.PartsPropertyUsageOptions[] option)
        {
            AppData.PartsPropertyUsageOptions dst = AppData.PartsPropertyUsageOptions.None;

            if (option != null)
            {
                if (option.Contains(FileFmt.PartsPropertyUsageOptions.UseTextBoxText)) { dst |= AppData.PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText; }
                if (option.Contains(FileFmt.PartsPropertyUsageOptions.UseMaterialColorBlend)) { dst |= AppData.PartsPropertyUsageOptions.UsePaneExParamaterColorBlend; }
                if (option.Contains(FileFmt.PartsPropertyUsageOptions.UseMaterialTexture)) { dst |= AppData.PartsPropertyUsageOptions.UsePaneExParameterTexture; }
            }

            return dst;
        }

        /// <summary>
        /// FileFmt.TexelFormat => AppData.TexImagePixelFmt
        /// </summary>
        TexImagePixelFmt
            Convert_TexelFormat_(FileFmt.TexelFormat srcTexelFmt)
        {
            switch (srcTexelFmt)
            {
                case TexelFormat.L4: return TexImagePixelFmt.L4;
                case TexelFormat.A4: return TexImagePixelFmt.A4;

                case TexelFormat.L8: return TexImagePixelFmt.L8;
                case TexelFormat.A8: return TexImagePixelFmt.A8;
                case TexelFormat.LA4: return TexImagePixelFmt.LA4;
                case TexelFormat.LA8: return TexImagePixelFmt.LA8;
                case TexelFormat.HILO8: return TexImagePixelFmt.HILO8;
                case TexelFormat.RGB565: return TexImagePixelFmt.RGB565;
                case TexelFormat.RGB8: return TexImagePixelFmt.RGB8;
                case TexelFormat.RGB5A1: return TexImagePixelFmt.RGB5A1;
                case TexelFormat.RGBA4: return TexImagePixelFmt.RGBA4;
                case TexelFormat.RGBA8: return TexImagePixelFmt.RGBA8;
                case TexelFormat.R10G10B10A2: return TexImagePixelFmt.R10G10B10A2;
                case TexelFormat.ETC1: return TexImagePixelFmt.ETC1;
                case TexelFormat.ETC1A4: return TexImagePixelFmt.ETC1A4;

                case TexelFormat.BC1: return TexImagePixelFmt.BC1;
                case TexelFormat.BC2: return TexImagePixelFmt.BC2;
                case TexelFormat.BC3: return TexImagePixelFmt.BC3;
                case TexelFormat.BC7: return TexImagePixelFmt.BC7;
                case TexelFormat.BC4A: return TexImagePixelFmt.BC4A;
                case TexelFormat.BC4L: return TexImagePixelFmt.BC4L;
                case TexelFormat.BC5: return TexImagePixelFmt.BC5;
                case TexelFormat.RGB565_INDIRECT: return TexImagePixelFmt.RGB565_INDIRECT;

                case TexelFormat.ASTC4x4: return TexImagePixelFmt.ASTC4x4;
                case TexelFormat.ASTC5x4: return TexImagePixelFmt.ASTC5x4;
                case TexelFormat.ASTC5x5: return TexImagePixelFmt.ASTC5x5;
                case TexelFormat.ASTC6x5: return TexImagePixelFmt.ASTC6x5;
                case TexelFormat.ASTC6x6: return TexImagePixelFmt.ASTC6x6;
                case TexelFormat.ASTC8x5: return TexImagePixelFmt.ASTC8x5;
                case TexelFormat.ASTC8x6: return TexImagePixelFmt.ASTC8x6;
                case TexelFormat.ASTC8x8: return TexImagePixelFmt.ASTC8x8;
                case TexelFormat.ASTC10x5: return TexImagePixelFmt.ASTC10x5;
                case TexelFormat.ASTC10x6: return TexImagePixelFmt.ASTC10x6;
                case TexelFormat.ASTC10x8: return TexImagePixelFmt.ASTC10x8;
                case TexelFormat.ASTC10x10: return TexImagePixelFmt.ASTC10x10;
                case TexelFormat.ASTC12x10: return TexImagePixelFmt.ASTC12x10;
                case TexelFormat.ASTC12x12: return TexImagePixelFmt.ASTC12x12;

                default: Debug.Assert(false); return TexImagePixelFmt.RGB565;
            }
        }

        /// <summary>
        /// SByte の 行列インデックスを AttrTexCoordMtx 型を に変換します。
        /// -1が Identity  に変換されます。
        /// </summary>
        AttrTexCoordMtx
            Convert_AttrTexCoordMtx_( SByte matrix )
        {
            if( matrix == -1 )
            {
                return AttrTexCoordMtx.Identity;
            }
            else
            {
                return (AttrTexCoordMtx)matrix;
            }
        }

        /// <summary>
        /// TexGenSrc => LETexGenMethod 変換
        /// </summary>
        LETexGenMethod Convert_LETexGenMethod_( TexGenSrc srcParam )
        {
            switch( srcParam )
            {
                case TexGenSrc.Tex0: return LETexGenMethod.UV0;
                case TexGenSrc.Tex1: return LETexGenMethod.UV1;
                case TexGenSrc.Tex2: return LETexGenMethod.UV2;
                case TexGenSrc.OrthogonalProjection: return LETexGenMethod.OrthogonalProjection;
                case TexGenSrc.PaneBasedProjection: return LETexGenMethod.PaneBasedProjection;
                case TexGenSrc.PerspectiveProjection: return LETexGenMethod.PerspectiveProjection;
                case TexGenSrc.PaneBasedPerspectiveProjection: return LETexGenMethod.PaneBasedPerspectiveProjection;
        		default: Debug.Assert( false ); return LETexGenMethod.UV0;
            }
        }

#endregion 基本的な型の変換
#endregion

#region LECore内部クラス => File への変換

#region 基本的な型の変換
        /// <summary>
        /// FVec3 => Vec2
        /// </summary>
        Vec2 Convert_Vec2_( FVec3 srcVec )
        {
            Vec2 result = new Vec2();

            result.x = srcVec.X;
            result.y = srcVec.Y;

            return result;
        }

        Vec2 Convert_Vec2_( FVec2 srcVec )
        {
            Vec2 result = new Vec2();

            result.x = srcVec.X;
            result.y = srcVec.Y;

            return result;
        }

        /// <summary>
        /// FVec3 => Vec3
        /// </summary>
        Vec3 Convert_Vec3_( FVec3 srcVec )
        {
            Vec3 result = new Vec3();

            result.x = srcVec.X;
            result.y = srcVec.Y;
            result.z = srcVec.Z;

            return result;
        }

        /// <summary>
        /// FVec2 => TexVec2
        /// </summary>
        TexVec2 Convert_TexVec2_( FVec2 srcVec )
        {
            TexVec2 result = new TexVec2();

            result.s = srcVec.X;
            result.t = srcVec.Y;

            return result;
        }

        /// <summary>
        /// TexCoord4 => TexCoord
        /// </summary>
        TexCoord Convert_TexCoord_( TexCoord4 srcTexCoord4 )
        {
            TexCoord dstTexCoords = new TexCoord();

            dstTexCoords.texLT = Convert_TexVec2_( srcTexCoord4.LT );
            dstTexCoords.texRT = Convert_TexVec2_( srcTexCoord4.RT );
            dstTexCoords.texLB = Convert_TexVec2_( srcTexCoord4.LB );
            dstTexCoords.texRB = Convert_TexVec2_( srcTexCoord4.RB );

            return dstTexCoords;
        }


        /// <summary>
        /// RGBAColor => Color4
        /// </summary>
        Color4 Convert_Color4_( RGBAColor srcColor )
        {
            Color4 result = new Color4();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;

            return result;
        }

        /// <summary>
        /// LECore.FloatColor => Serializable.FloatColor
        /// </summary>
        Color4f Convert_FloatColor_(FloatColor srcColor)
        {
            Color4f result = new Color4f();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;
            result.byte_convert = srcColor.ByteConvert;

            return result;
        }

        /// <summary>
        /// Serializable.FloatColor => LECore.FloatColor
        /// </summary>
        FloatColor Convert_Color4f_(Color4f srcColor)
        {
            FloatColor result = new FloatColor();

            result.R = srcColor.r;
            result.G = srcColor.g;
            result.B = srcColor.b;
            result.A = srcColor.a;
            result.ByteConvert = srcColor.byte_convert;

            return result;
        }

        /// <summary>
        /// RGBAColor => Color4
        /// </summary>
        BlackColor Convert_BlackColor_( System.Drawing.Color srcColor )
        {
            BlackColor result = new BlackColor();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;

            return result;
        }

        /// <summary>
        /// RGBAColor => Color4
        /// </summary>
        WhiteColor Convert_WhiteColor_( System.Drawing.Color srcColor )
        {
            WhiteColor result = new WhiteColor();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;

            return result;
        }

        /// <summary>
        /// FloatColor => BlackColorFloat
        /// </summary>
        BlackColorFloat Convert_BlackColor_( FloatColor srcColor )
        {
            BlackColorFloat result = new BlackColorFloat();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;
            result.byte_convert = srcColor.ByteConvert;

            return result;
        }

        /// <summary>
        /// FloatColor => WhiteColorFloat
        /// </summary>
        WhiteColorFloat Convert_WhiteColor_( FloatColor srcColor )
        {
            WhiteColorFloat result = new WhiteColorFloat();

            result.r = srcColor.R;
            result.g = srcColor.G;
            result.b = srcColor.B;
            result.a = srcColor.A;
            result.byte_convert = srcColor.ByteConvert;

            return result;
        }

        /// <summary>
        /// HorizontalLocation, VerticalLocation => Position
        /// </summary>
        Position Convert_Position_( HorizontalLocation baseH, VerticalLocation baseV )
        {
            Position result = new Position();

            result.x = Convert_HorizontalPosition_(baseH);
            result.y = Convert_VerticalPosition_(baseV);

            return result;
        }

        HorizontalPosition Convert_HorizontalPosition_(HorizontalLocation baseH)
        {
            // 垂直
            switch (baseH)
            {
                case HorizontalLocation.Left: return HorizontalPosition.Left;
                case HorizontalLocation.Center: return HorizontalPosition.Center;
                case HorizontalLocation.Right: return HorizontalPosition.Right;
                default: Debug.Assert(false); return HorizontalPosition.Center;
            }
        }

        VerticalPosition Convert_VerticalPosition_(VerticalLocation baseV)
        {
            // 垂直
            switch (baseV)
            {
                case VerticalLocation.Top: return VerticalPosition.Top;
                case VerticalLocation.Center: return VerticalPosition.Center;
                case VerticalLocation.Bottom: return VerticalPosition.Bottom;
                default: Debug.Assert(false); return VerticalPosition.Center;
            }
        }

        /// <summary>
        /// 部品倍率モード
        /// </summary>
        FileFmt.PartsMagnifyInfluence Convert_PartsMagnifyInfluence_(AppData.PartsMagnifyInfluence src)
        {
            // 垂直
            switch (src)
            {
                case AppData.PartsMagnifyInfluence.None: return FileFmt.PartsMagnifyInfluence.None;
                case AppData.PartsMagnifyInfluence.ScaleMagnify: return FileFmt.PartsMagnifyInfluence.ScaleMagnify;
                case AppData.PartsMagnifyInfluence.AdjustToPartsBound: return FileFmt.PartsMagnifyInfluence.AdjustToPartsBound;
                default: Debug.Assert(false); return FileFmt.PartsMagnifyInfluence.ScaleMagnify;
            }
        }

        /// <summary>
        /// LineAlignment => TextAlignment
        /// </summary>
        TextAlignment Convert_TextAlignment_( LineAlignment lineAlignment )
        {
            switch( lineAlignment )
            {
                case LineAlignment.NotSpecified: return TextAlignment.Synchronous;
                case LineAlignment.Left: return TextAlignment.Left;
                case LineAlignment.Center: return TextAlignment.Center;
                case LineAlignment.Right: return TextAlignment.Right;
                default: Debug.Assert( false ); return TextAlignment.Synchronous;
            }
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.PerCharacterTransformLoopType Convert_PerCharTransformLoopType_(PerCharTransformLoopType loopType)
        {
            switch (loopType)
            {
                case PerCharTransformLoopType.OneTime: return FileFmt.PerCharacterTransformLoopType.OneTime;
                case PerCharTransformLoopType.Loop: return FileFmt.PerCharacterTransformLoopType.Loop;
                default: Debug.Assert(false); return FileFmt.PerCharacterTransformLoopType.Loop;
            }
        }

        /// <summary>
        /// PaneModify.Kind => FileFmt.PaneKind
        /// </summary>
        FileFmt.PaneKind Convert_SOPaneKind_(AppData.PaneKind kind)
        {
            switch (kind)
            {
                case AppData.PaneKind.Null: return FileFmt.PaneKind.Null;
                case AppData.PaneKind.Picture: return FileFmt.PaneKind.Picture;
                case AppData.PaneKind.Textbox: return FileFmt.PaneKind.TextBox;
                case AppData.PaneKind.Window: return FileFmt.PaneKind.Window;
                case AppData.PaneKind.Bounding: return FileFmt.PaneKind.Bounding;
                case AppData.PaneKind.Parts: return FileFmt.PaneKind.Parts;
                case AppData.PaneKind.Capture: return FileFmt.PaneKind.Capture;
                case AppData.PaneKind.Alignment: return FileFmt.PaneKind.Alignment;
                case AppData.PaneKind.Scissor: return FileFmt.PaneKind.Scissor;
                default: Debug.Assert(false); return FileFmt.PaneKind.Null;
            }
        }

        /// <summary>
        ///
        /// </summary>
        FileFmt.PartsPropertyUsageOptions[] Convert_PartsPropertyUsageOptions_(AppData.PartsPropertyUsageOptions option)
        {
            List<FileFmt.PartsPropertyUsageOptions> dst = new List<FileFmt.PartsPropertyUsageOptions>();

            if (option.HasFlag(AppData.PartsPropertyUsageOptions.UsePaneExParamaterUseTextBoxText)) { dst.Add(FileFmt.PartsPropertyUsageOptions.UseTextBoxText); }
            if (option.HasFlag(AppData.PartsPropertyUsageOptions.UsePaneExParamaterColorBlend)) { dst.Add(FileFmt.PartsPropertyUsageOptions.UseMaterialColorBlend); }
            if (option.HasFlag(AppData.PartsPropertyUsageOptions.UsePaneExParameterTexture)) { dst.Add(FileFmt.PartsPropertyUsageOptions.UseMaterialTexture); }

            return dst.Count == 0 ? null : dst.ToArray();
        }

        /// <summary>
        /// AppData.TexImagePixelFmt => FileFmt.TexelFormat
        /// </summary>
        FileFmt.TexelFormat
            Convert_TexImagePixelFmt_( AppData.LECoreInterface.TexImagePixelFmt srcTexelFmt )
        {
            try
            {
                FileFmt.TexelFormat texelFmt =
                    (FileFmt.TexelFormat)Enum.Parse( typeof( FileFmt.TexelFormat ), srcTexelFmt.ToString() );
                return texelFmt;
            }
            catch
            {
                Debug.Assert( false );
                // 変換に失敗したので、適当な値を返します。
                return FileFmt.TexelFormat.RGB565;
            }
        }

        /// <summary>
        /// ScreenCoordinate => FileFmt.ScreenOriginType (座標系種類)
        /// </summary>
        FileFmt.ScreenOriginType
            Convert_ScreenCoordinate_( AppData.LEScreenCoordinate screenCoordinate )
        {
            return FileFmt.ScreenOriginType.Normal;
        }

#endregion 基本的な型の変換

#endregion
    }
}
