﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;
using System.Diagnostics;

namespace LECore.Save_Load
{
    using Structures;
    using Structures.Core;
    using Structures.SerializableObject.Lan;
    using System.Linq;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;
    using static RlanHelper;

    class RlvcConverter
    {
        #region --------------- Rlvc
        /// <summary>
        /// キーフレームを読み込むべきサブアトリビュートを取得します。
        /// TextBoxペイン用
        /// </summary>
        static AnmAttribute GetTextBoxSubAnmAttr_( TextBox textBox, AnimTargetType type )
        {
            switch( type )
            {
                case AnimTargetType.LT_r: return textBox.TopColorAnmAttr.FindAttributeByIdx( 0 );
                case AnimTargetType.LT_g: return textBox.TopColorAnmAttr.FindAttributeByIdx( 1 );
                case AnimTargetType.LT_b: return textBox.TopColorAnmAttr.FindAttributeByIdx( 2 );
                case AnimTargetType.LT_a: return textBox.TopColorAnmAttr.FindAttributeByIdx( 3 );

                case AnimTargetType.LB_r: return textBox.BottomColorAnmAttr.FindAttributeByIdx( 0 );
                case AnimTargetType.LB_g: return textBox.BottomColorAnmAttr.FindAttributeByIdx( 1 );
                case AnimTargetType.LB_b: return textBox.BottomColorAnmAttr.FindAttributeByIdx( 2 );
                case AnimTargetType.LB_a: return textBox.BottomColorAnmAttr.FindAttributeByIdx( 3 );

                default: Debug.Assert( false ); return null;
            }
        }

        /// <summary>
        /// キーフレームを読み込むべきサブアトリビュートを取得します。
        /// Pictureペイン用
        /// </summary>
        static AnmAttribute GetVtxCol4SubAnmAttr_( IVertexColor4Holder holder, AnimTargetType type )
        {
            AnmAttribute anmAttrLT = holder.VtxCol_LTIAnmAttr as AnmAttribute;
            AnmAttribute anmAttrRT = holder.VtxCol_RTIAnmAttr as AnmAttribute;
            AnmAttribute anmAttrLB = holder.VtxCol_LBIAnmAttr as AnmAttribute;
            AnmAttribute anmAttrRB = holder.VtxCol_RBIAnmAttr as AnmAttribute;

            switch( type )
            {
                case AnimTargetType.LT_r: return anmAttrLT.FindAttributeByIdx( 0 );
                case AnimTargetType.LT_g: return anmAttrLT.FindAttributeByIdx( 1 );
                case AnimTargetType.LT_b: return anmAttrLT.FindAttributeByIdx( 2 );
                case AnimTargetType.LT_a: return anmAttrLT.FindAttributeByIdx( 3 );

                case AnimTargetType.RT_r: return anmAttrRT.FindAttributeByIdx( 0 );
                case AnimTargetType.RT_g: return anmAttrRT.FindAttributeByIdx( 1 );
                case AnimTargetType.RT_b: return anmAttrRT.FindAttributeByIdx( 2 );
                case AnimTargetType.RT_a: return anmAttrRT.FindAttributeByIdx( 3 );

                case AnimTargetType.LB_r: return anmAttrLB.FindAttributeByIdx( 0 );
                case AnimTargetType.LB_g: return anmAttrLB.FindAttributeByIdx( 1 );
                case AnimTargetType.LB_b: return anmAttrLB.FindAttributeByIdx( 2 );
                case AnimTargetType.LB_a: return anmAttrLB.FindAttributeByIdx( 3 );

                case AnimTargetType.RB_r: return anmAttrRB.FindAttributeByIdx( 0 );
                case AnimTargetType.RB_g: return anmAttrRB.FindAttributeByIdx( 1 );
                case AnimTargetType.RB_b: return anmAttrRB.FindAttributeByIdx( 2 );
                case AnimTargetType.RB_a: return anmAttrRB.FindAttributeByIdx( 3 );

                default: Debug.Assert( false ); return null;
            }
        }

        /// <summary>
        /// キーフレームの読み込みを行います
        /// </summary>
        static void SetKeyFrame_( AppData.Pane dstPane, AnimTarget animTarget, string tag)
        {
            AnmAttribute subAnmAttr = null;

            // ペイン透明度
            if( animTarget.target == AnimTargetType.PaneAlpha )
            {
                RlanHelper.SetHermiteKeyFrame_( dstPane.PaneAttrRef.TransparencyAttr, animTarget, tag);
            }
            else
            {
                // ペイン頂点カラー
                // ペインの種類に応じて...
                switch( dstPane.PaneKind )
                {
                    case PaneKind.Textbox:
                    subAnmAttr = GetTextBoxSubAnmAttr_( dstPane.TextBox, animTarget.target );
                    RlanHelper.SetHermiteKeyFrame_( subAnmAttr, animTarget, tag);
                    break;
                    case PaneKind.Picture:
                    subAnmAttr = GetVtxCol4SubAnmAttr_( dstPane.Picture, animTarget.target );
                    RlanHelper.SetHermiteKeyFrame_( subAnmAttr, animTarget, tag);
                    break;
                    case PaneKind.Window:
                    subAnmAttr = GetVtxCol4SubAnmAttr_( dstPane.ILEWindow, animTarget.target );
                    RlanHelper.SetHermiteKeyFrame_( subAnmAttr, animTarget, tag);
                    break;
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        static public void LoadRlvcToPane( object dstObject, FileFmt.AnimContent srcAnimContent, string tag)
        {
            Pane dstPane = dstObject as Pane;
            if( dstPane == null )
            {
                return;
            }

            if( dstPane.PaneName != srcAnimContent.name )
            {
                return;
            }

            foreach( FileFmt.AnimTarget animTarget in srcAnimContent.Items )
            {
                SetKeyFrame_( dstPane, animTarget, tag);
            }
        }

        #endregion ------------- Rlvc

        #region ------------- Rlvc
        public static AnimContent[] SaveVertexColorAnim(IPane pane, string tag)
        {
            return ConvertToAnimContents<IPane, AnimVertexColorTarget>(
                tag,
                ParamaterKind.Animation_VertexColors,
                pane,
                x => new[] { new AnimContentSource<IPane>(x.PaneName, x) },
                x  => {
                    // 透明度アトリビュート
                    var transparency = new[] { new AnimTargetSource(x.IPaneAttribute.TransparencyIAttr, AnimTargetType.PaneAlpha) };
                    switch (x.PaneKind)
                    {
                        case PaneKind.Textbox:
                            return transparency
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.ITextBox.TopColorIAnmAttr, new[] { AnimTargetType.LT_r, AnimTargetType.LT_g, AnimTargetType.LT_b, AnimTargetType.LT_a }))
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.ITextBox.BottomColorIAnmAttr, new[] { AnimTargetType.LB_r, AnimTargetType.LB_g, AnimTargetType.LB_b, AnimTargetType.LB_a }));
                        case PaneKind.Picture:
                            return transparency
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.IPicture.VtxCol_LTIAnmAttr, new[] { AnimTargetType.LT_r, AnimTargetType.LT_g, AnimTargetType.LT_b, AnimTargetType.LT_a }))
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.IPicture.VtxCol_RTIAnmAttr, new[] { AnimTargetType.RT_r, AnimTargetType.RT_g, AnimTargetType.RT_b, AnimTargetType.RT_a }))
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.IPicture.VtxCol_LBIAnmAttr, new[] { AnimTargetType.LB_r, AnimTargetType.LB_g, AnimTargetType.LB_b, AnimTargetType.LB_a }))
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.IPicture.VtxCol_RBIAnmAttr, new[] { AnimTargetType.RB_r, AnimTargetType.RB_g, AnimTargetType.RB_b, AnimTargetType.RB_a }));
                        case PaneKind.Window:
                            return transparency
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.ILEWindow.VtxCol_LTIAnmAttr, new[] { AnimTargetType.LT_r, AnimTargetType.LT_g, AnimTargetType.LT_b, AnimTargetType.LT_a }))
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.ILEWindow.VtxCol_RTIAnmAttr, new[] { AnimTargetType.RT_r, AnimTargetType.RT_g, AnimTargetType.RT_b, AnimTargetType.RT_a }))
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.ILEWindow.VtxCol_LBIAnmAttr, new[] { AnimTargetType.LB_r, AnimTargetType.LB_g, AnimTargetType.LB_b, AnimTargetType.LB_a }))
                            .Concat(GetAnimTargetSourcesFromSubAttributes(x.ILEWindow.VtxCol_RBIAnmAttr, new[] { AnimTargetType.RB_r, AnimTargetType.RB_g, AnimTargetType.RB_b, AnimTargetType.RB_a }));
                    }
                    return transparency;
                }).ToArray();
        }


        #endregion
    }
}
