﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;

using System.Collections;
using System.Diagnostics;

namespace LECore.Save_Load
{
    using Structures;
    using Structures.Core;
    using Structures.SerializableObject.Lan;
    using static RlanHelper;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    class RlmcConverter
    {
        #region ------------ Rlmc 書き出し

        #region ------- private

        #region 詳細マテリアル関連
        /// <summary>
        /// アニメーションターゲット列挙型を文字列から生成します。
        /// </summary>
        static FileFmt.AnimTargetType GetTevColorAnimTragetType_( int regIdx, string enumBaseStr )
        {
            string enumStr = string.Format( enumBaseStr, regIdx.ToString() );
            return (FileFmt.AnimTargetType)Enum.Parse( typeof( FileFmt.AnimTargetType ), enumStr );
        }
        #endregion 詳細マテリアル関連

        #endregion ------- private

        /// <summary>
        /// Rlmc の エクスポート
        /// </summary>
        public static FileFmt.AnimContent[] SaveMaterialAnimation(AppData.IPane srcPane, string tag)
        {
            var materialContents = ConvertToAnimContents<IMaterial, AnimMaterialColorTarget>(
                    tag,
                    ParamaterKind.Animation_MaterialColors,
                    srcPane,
                    x => x.UsedMaterials.Select(y => new AnimContentSource<IMaterial>(y.MaterialName, y)),
                    x => GetAnimTargetSourcesFromSubAttributes(x.BlackColorIAnmAttr,
                        x.BlackColor.ByteConvert ?
                        new[] { AnimTargetType.BlackColor_r, AnimTargetType.BlackColor_g, AnimTargetType.BlackColor_b, AnimTargetType.BlackColor_a } :
                        new[] { AnimTargetType.BlackColorFloat_r, AnimTargetType.BlackColorFloat_g, AnimTargetType.BlackColorFloat_b, AnimTargetType.BlackColorFloat_a })
                    .Concat(GetAnimTargetSourcesFromSubAttributes(x.WhiteColorIAnmAttr,
                        x.WhiteColor.ByteConvert ?
                        new[] { AnimTargetType.WhiteColor_r, AnimTargetType.WhiteColor_g, AnimTargetType.WhiteColor_b, AnimTargetType.WhiteColor_a } :
                        new[] { AnimTargetType.WhiteColorFloat_r, AnimTargetType.WhiteColorFloat_g, AnimTargetType.WhiteColorFloat_b, AnimTargetType.WhiteColorFloat_a })));

            var revMaterialContentss = ConvertToAnimContents<RevHWMaterial, AnimMaterialColorTarget>(
                    tag,
                    ParamaterKind.Animation_MaterialColors,
                    srcPane,
                    x => (x.IPaneExParamater is IRevHWMaterialHolder ? ((IRevHWMaterialHolder)x.IPaneExParamater).IRevHWMaterial : Enumerable.Empty<IRevHWMaterial>())
                        .Where(y => y.LowLevelCombinerSettingsEnabled || y.CombinerUserShaderSettingsEnabled)
                        .Select(y => new AnimContentSource<RevHWMaterial>(y.MaterialName, (RevHWMaterial)y)),
                    x => Enumerable.Range(0, TevColorsConstants.NumTevKColorRegs)
                        .SelectMany(y => GetAnimTargetSourcesFromSubAttributes(x.TevData.GetTevColorAttr(y),
                            new[] { "TevConstColor{0}_r", "TevConstColor{0}_g", "TevConstColor{0}_b", "TevConstColor{0}_a" }.Select(z => GetTevColorAnimTragetType_(y, z)))));

            return materialContents.Concat(revMaterialContentss).ToArray();
        }

        #endregion -------------- Rlmc


        #region ----------- Rlmc 読み込み

        /// <summary>
        /// 混合比率対象アトリビュートを取得します。
        ///
        ///
        /// NOTICE：
        /// アトリビュート列挙子の文字列表現を解析することによって、挿入すべきアトリビュートを判定します。
        /// 列挙子名が変更された場合に、正しく動作しなくなる可能性があります。
        /// </summary>
        static AnmAttribute
            GetMaterialTexAnmAttribute_
            (
            AppData.Material dstMtl,
            FileFmt.AnimTargetType targetType
            )
        {
            string typeStr = targetType.ToString();


            int index = typeStr.IndexOf( "_a" );
            if( index == -1 )
            {
                // カラー混合比の場合
                index = typeStr.IndexOf( "_c" );
                if( index != -1 )
                {
                    // カラーブレンドウェイトは廃止されました。
                    Debug.Assert(false);
                }
            }
            else
            {
                // アルファブレンドウェイトは廃止されました。
                Debug.Assert( false);
            }

            // 不明なアトリビュート
            return null;
        }

        /// <summary>
        /// キーフレームを挿入するアトリビュートを取得します。
        /// </summary>
        static AnmAttribute
            GetTargetAnmAttr_
            (
            AppData.Material dstMtl,
            FileFmt.AnimTargetType targetType
            )
        {
            switch( targetType )
            {
                // 黒カラー
                case FileFmt.AnimTargetType.BlackColor_r: return dstMtl.BlackColorAnmAttr.FindAttributeByIdx( 0 );
                case FileFmt.AnimTargetType.BlackColor_g: return dstMtl.BlackColorAnmAttr.FindAttributeByIdx( 1 );
                case FileFmt.AnimTargetType.BlackColor_b: return dstMtl.BlackColorAnmAttr.FindAttributeByIdx( 2 );
                case FileFmt.AnimTargetType.BlackColor_a: return dstMtl.BlackColorAnmAttr.FindAttributeByIdx( 3 );

                // 白カラー
                case FileFmt.AnimTargetType.WhiteColor_r: return dstMtl.WhiteColorAnmAttr.FindAttributeByIdx( 0 );
                case FileFmt.AnimTargetType.WhiteColor_g: return dstMtl.WhiteColorAnmAttr.FindAttributeByIdx( 1 );
                case FileFmt.AnimTargetType.WhiteColor_b: return dstMtl.WhiteColorAnmAttr.FindAttributeByIdx( 2 );
                case FileFmt.AnimTargetType.WhiteColor_a: return dstMtl.WhiteColorAnmAttr.FindAttributeByIdx( 3 );

                // 黒カラー(Float)
                case FileFmt.AnimTargetType.BlackColorFloat_r: return dstMtl.BlackColorAnmAttr.FindAttributeByIdx( 0 );
                case FileFmt.AnimTargetType.BlackColorFloat_g: return dstMtl.BlackColorAnmAttr.FindAttributeByIdx( 1 );
                case FileFmt.AnimTargetType.BlackColorFloat_b: return dstMtl.BlackColorAnmAttr.FindAttributeByIdx( 2 );
                case FileFmt.AnimTargetType.BlackColorFloat_a: return dstMtl.BlackColorAnmAttr.FindAttributeByIdx( 3 );

                // 白カラー(Float)
                case FileFmt.AnimTargetType.WhiteColorFloat_r: return dstMtl.WhiteColorAnmAttr.FindAttributeByIdx( 0 );
                case FileFmt.AnimTargetType.WhiteColorFloat_g: return dstMtl.WhiteColorAnmAttr.FindAttributeByIdx( 1 );
                case FileFmt.AnimTargetType.WhiteColorFloat_b: return dstMtl.WhiteColorAnmAttr.FindAttributeByIdx( 2 );
                case FileFmt.AnimTargetType.WhiteColorFloat_a: return dstMtl.WhiteColorAnmAttr.FindAttributeByIdx( 3 );

                // ブレンド比率
                case FileFmt.AnimTargetType.TexBlendRatio0_c:
                case FileFmt.AnimTargetType.TexBlendRatio1_c:
                case FileFmt.AnimTargetType.TexBlendRatio2_c:
                AnmAttribute anmAttr = GetMaterialTexAnmAttribute_( dstMtl, targetType );

                Debug.Assert( anmAttr != null );
                return anmAttr;

                default: return null;
            }
        }

        /// <summary>
        /// キーフレームを挿入するアトリビュートを取得します。
        /// </summary>
        static AnmAttribute
        GetTargetAnmAttrRevHW_
            (
            AppData.RevHWMaterial dstMtl,
            FileFmt.AnimTargetType targetType
            )
        {
            switch (targetType)
            {
                // tev定数レジスタ0
                case FileFmt.AnimTargetType.TevConstColor0_r: return dstMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor0_g: return dstMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor0_b: return dstMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor0_a: return dstMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(3);
                // tev定数レジスタ1
                case FileFmt.AnimTargetType.TevConstColor1_r: return dstMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor1_g: return dstMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor1_b: return dstMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor1_a: return dstMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(3);
                // tev定数レジスタ2
                case FileFmt.AnimTargetType.TevConstColor2_r: return dstMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor2_g: return dstMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor2_b: return dstMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor2_a: return dstMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(3);
                // tev定数レジスタ3
                case FileFmt.AnimTargetType.TevConstColor3_r: return dstMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor3_g: return dstMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor3_b: return dstMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor3_a: return dstMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(3);

                // tev定数レジスタ4
                case FileFmt.AnimTargetType.TevConstColor4_r: return dstMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor4_g: return dstMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor4_b: return dstMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor4_a: return dstMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(3);

                // tev定数レジスタ0(Float)
                case FileFmt.AnimTargetType.TevConstColor0Float_r: return dstMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor0Float_g: return dstMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor0Float_b: return dstMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor0Float_a: return dstMtl.TevData.GetTevColorAttr(0).FindAttributeByIdx(3);
                // tev定数レジスタ1(Float)
                case FileFmt.AnimTargetType.TevConstColor1Float_r: return dstMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor1Float_g: return dstMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor1Float_b: return dstMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor1Float_a: return dstMtl.TevData.GetTevColorAttr(1).FindAttributeByIdx(3);
                // tev定数レジスタ2(Float)
                case FileFmt.AnimTargetType.TevConstColor2Float_r: return dstMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor2Float_g: return dstMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor2Float_b: return dstMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor2Float_a: return dstMtl.TevData.GetTevColorAttr(2).FindAttributeByIdx(3);
                // tev定数レジスタ3(Float)
                case FileFmt.AnimTargetType.TevConstColor3Float_r: return dstMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor3Float_g: return dstMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor3Float_b: return dstMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor3Float_a: return dstMtl.TevData.GetTevColorAttr(3).FindAttributeByIdx(3);

                // tev定数レジスタ4(Float)
                case FileFmt.AnimTargetType.TevConstColor4Float_r: return dstMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(0);
                case FileFmt.AnimTargetType.TevConstColor4Float_g: return dstMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(1);
                case FileFmt.AnimTargetType.TevConstColor4Float_b: return dstMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(2);
                case FileFmt.AnimTargetType.TevConstColor4Float_a: return dstMtl.TevData.GetTevColorAttr(4).FindAttributeByIdx(3);

                default: return null;
            }
        }

        /// <summary>
        /// マテリアルカラー
        /// </summary>
        static public void LoadRlmcToMaterial( object dstObject, FileFmt.AnimContent animContent, string tag)
        {
            Pane pane = dstObject as Pane;
            if( pane != null )
            {
                // 名前が一致すれば
                AppData.Material dstMaterial = pane.FindMaterialByName( animContent.name );
                if( dstMaterial != null )
                {
                    foreach( FileFmt.AnimTarget animTarget in animContent.Items )
                    {
                        AnmAttribute targetAttr = GetTargetAnmAttr_( dstMaterial, animTarget.target );
                        if( targetAttr != null )
                        {
                            RlanHelper.SetHermiteKeyFrame_( targetAttr, animTarget, tag);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// マテリアルカラー（ハード固有表現）
        /// </summary>
        static public void LoadRlmcToMaterialRevHW(object dstObject, FileFmt.AnimContent animContent, string tag)
        {
            IPane pane = dstObject as IPane;
            if (pane == null)
            {
                return;
            }

            IRevHWMaterialHolder matHolder = pane.IPaneExParamater as IRevHWMaterialHolder;
            if (matHolder == null)
            {
                return;
            }

            foreach (RevHWMaterial dstMaterial in matHolder.IRevHWMaterial)
            {
                if (dstMaterial.MaterialName == animContent.name)
                {
                    foreach (FileFmt.AnimTarget animTarget in animContent.Items)
                    {
                        AnmAttribute targetAttr = GetTargetAnmAttrRevHW_(dstMaterial, animTarget.target);
                        if (targetAttr != null)
                        {
                            RlanHelper.SetHermiteKeyFrame_(targetAttr, animTarget, tag);
                        }
                    }
                }
            }
        }

        #endregion --------- Rlmc
    }
}
