﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Collections;
using System.Diagnostics;
using System.Text;
using LECore.Structures;

namespace LECore.Save_Load
{
    using Structures;
    using Structures.Core;
    using Structures.SerializableObject.Lan;
    using System.Linq;
    using AppData = LECore.Structures;
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    /// <summary>
    /// アニメーションヘルパクラスです。
    /// </summary>
    static class RlanHelper
    {
        public delegate void AnimLoadFunction( object target, FileFmt.AnimContent srcAnimContent, string tag);

        /// <summary>
        /// アニメーションロード関数
        /// </summary>
        static readonly AnimLoadFunction[] _loadFuncRlts = { RltsConverter.LoadRltsToMaterial };
        static readonly AnimLoadFunction[] _loadFuncRlis = { RltsConverter.LoadRltsToIndirectMaterial };
        static readonly AnimLoadFunction[] _loadFuncRltp = { RltpConverter.LoadRltpToMaterial };
        static readonly AnimLoadFunction[] _loadFuncRlmc = { RlmcConverter.LoadRlmcToMaterial, RlmcConverter.LoadRlmcToMaterialRevHW};
        static readonly AnimLoadFunction[] _loadFuncRlpa = { RlpaConverter.LoadRlpaToPane };
        static readonly AnimLoadFunction[] _loadFuncRlvc = { RlvcConverter.LoadRlvcToPane };
        static readonly AnimLoadFunction[] _loadFuncRlvi = { RlviConverter.LoadRlviToPane };
        static readonly AnimLoadFunction[] _loadFuncRlac = { RlacConverter.LoadRlmcToMaterial };
        static readonly AnimLoadFunction[] _loadFuncRlfs = { RlfsConverter.Load };
        static readonly AnimLoadFunction[] _loadFuncRlto = { RltoConverter.Load };
        static readonly AnimLoadFunction[] _loadFuncRltc = { RltoConverter.LoadRltcToPane };
        static readonly AnimLoadFunction[] _loadFuncRlwn = { RlwnConverter.LoadRlwnToPane };
        static readonly AnimLoadFunction[] _loadFuncRlud = { RludConverter.LoadRludToExtUserData };
        static readonly AnimLoadFunction[] _loadFuncRlma = { RlmaConverter.LoadRlmaToMask };
        static readonly AnimLoadFunction[] _loadFuncRlds = { RldsConverter.LoadRldsToDropShadow };
        static readonly AnimLoadFunction[] _loadFuncRlps = { RlpsConverter.LoadRlpsToProceduralShape };

        /// <summary>
        /// 型変換(InfinityType)
        /// </summary>
        static public FileFmt.InfinityType
            ConvertToInfinityType( AnmCurveInfinityType infinityType )
        {
            switch( infinityType )
            {
                case AnmCurveInfinityType.Constant: return FileFmt.InfinityType.Constant;
                case AnmCurveInfinityType.Cycle: return FileFmt.InfinityType.Cycle;
                default: Debug.Assert( false ); return FileFmt.InfinityType.Constant;
            }
        }

        /// <summary>
        /// 型変換(SlopeType)
        /// </summary>
        static public FileFmt.SlopeType
            ConvertToSlopeType( InterporationType type )
        {
            switch( type )
            {
                case InterporationType.Fixed: return SlopeType.Fixed;
                case InterporationType.Zero: return SlopeType.Flat;
                case InterporationType.Liner: return SlopeType.Linear;
                case InterporationType.Spline: return SlopeType.Smooth;
                case InterporationType.Clamped: return SlopeType.Clamped;
                case InterporationType.Step: return SlopeType.Step;
                default:
                Debug.Assert( false ); return SlopeType.Fixed;
            }
        }

        /// <summary>
        /// 型変換(InfinityType)
        /// </summary>
        static public AnmCurveInfinityType
            ConvertToInfinityType( FileFmt.InfinityType infinityType )
        {
            switch( infinityType )
            {
                case InfinityType.Constant: return AnmCurveInfinityType.Constant;
                case InfinityType.Cycle: return AnmCurveInfinityType.Cycle;
                default: Debug.Assert( false ); return AnmCurveInfinityType.Constant;
            }
        }

        /// <summary>
        /// 型変換(SlopeType)
        /// </summary>
        static public InterporationType
            ConvertToInterporationType( FileFmt.SlopeType type )
        {
            switch( type )
            {
                case SlopeType.Fixed: return InterporationType.Fixed;
                case SlopeType.Flat: return InterporationType.Zero;
                case SlopeType.Linear: return InterporationType.Liner;
                case SlopeType.Smooth: return InterporationType.Spline;
                case SlopeType.Clamped: return InterporationType.Clamped;
                case SlopeType.Step: return InterporationType.Step;
                default:
                Debug.Assert( false ); return InterporationType.Fixed;
            }
        }

        /// <summary>
        /// 変換
        /// </summary>
        static public FileFmt.AnimationType ParamaterKindToAnimationType(ParamaterKind animationFileType)
        {
            switch (animationFileType)
            {
                case ParamaterKind.Animation_PaneSRT: return AnimationType.PaneSRT;
                case ParamaterKind.Animation_VertexColors: return AnimationType.VertexColor;
                case ParamaterKind.Animation_Visivility: return AnimationType.Visibility;
                case ParamaterKind.Animation_TextureSRT: return AnimationType.TextureSRT;
                case ParamaterKind.Animation_TexturePattern: return AnimationType.TexturePattern;
                case ParamaterKind.Animation_MaterialColors: return AnimationType.MaterialColor;
                case ParamaterKind.Animation_IndirectTextureSRT: return AnimationType.IndTextureSRT;
                case ParamaterKind.Animation_AlphaTest: return AnimationType.AlphaTest;
                case ParamaterKind.Animation_FontShadowColor: return AnimationType.FontShadow;
                case ParamaterKind.Animation_PerCharacterTransform: return AnimationType.PerCharacterTransform;
                case ParamaterKind.Animation_PerCharacterTransformCurve: return AnimationType.PerCharacterTransformCurve;
                case ParamaterKind.Animation_Window: return AnimationType.Window;
                case ParamaterKind.Animation_ExtUserData: return AnimationType.ExtUserData;
                case ParamaterKind.Animation_MaskTextureSRT: return AnimationType.MaskTextureSRT;
                case ParamaterKind.Animation_DropShadow: return AnimationType.DropShadow;
                case ParamaterKind.Animation_ProceduralShape: return AnimationType.ProceduralShape;
                default: Debug.Assert(false); return AnimationType.PaneSRT;
            }
        }

        /// <summary>
        /// 変換
        /// </summary>
        static public ParamaterKind AnimationTypeToParamaterKind(FileFmt.AnimationType animationFileType)
        {
            switch (animationFileType)
            {
                case AnimationType.PaneSRT: return ParamaterKind.Animation_PaneSRT;
                case AnimationType.VertexColor: return ParamaterKind.Animation_VertexColors;
                case AnimationType.Visibility: return ParamaterKind.Animation_Visivility;
                case AnimationType.TextureSRT: return ParamaterKind.Animation_TextureSRT;
                case AnimationType.TexturePattern: return ParamaterKind.Animation_TexturePattern;
                case AnimationType.MaterialColor: return ParamaterKind.Animation_MaterialColors;
                case AnimationType.IndTextureSRT: return ParamaterKind.Animation_IndirectTextureSRT;
                case AnimationType.AlphaTest: return ParamaterKind.Animation_AlphaTest;
                case AnimationType.PerCharacterTransform: return ParamaterKind.Animation_PerCharacterTransform;
                case AnimationType.PerCharacterTransformCurve: return ParamaterKind.Animation_PerCharacterTransformCurve;
                case AnimationType.Window: return ParamaterKind.Animation_Window;
                case AnimationType.ExtUserData: return ParamaterKind.Animation_ExtUserData;
                case AnimationType.MaskTextureSRT: return ParamaterKind.Animation_MaskTextureSRT;
                case AnimationType.DropShadow: return ParamaterKind.Animation_DropShadow;
                case AnimationType.ProceduralShape: return ParamaterKind.Animation_ProceduralShape;
                default: Debug.Assert(false); return ParamaterKind.Animation_PaneSRT;
            }
        }

        /// <summary>
        /// 出力ファイル形式のキーフレームインスタンスを取得します。
        /// </summary>
        static public FileFmt.Hermite GetFileFmtKeyInstance_(ParamaterKind animationFileType, FileFmt.AnimTargetType type)
        {
            switch( animationFileType )
            {
                case ParamaterKind.Animation_TexturePattern: return new FileFmt.StepU16();
                case ParamaterKind.Animation_IndirectTextureSRT:
                case ParamaterKind.Animation_TextureSRT: return new FileFmt.Hermite();
                case ParamaterKind.Animation_Visivility: return new FileFmt.StepBool();
                case ParamaterKind.Animation_PaneSRT: return new FileFmt.Hermite();
                case ParamaterKind.Animation_VertexColors: return new FileFmt.HermiteU8();
                case ParamaterKind.Animation_MaterialColors:
                    {
                        if (type >= FileFmt.AnimTargetType.BlackColorFloat_r &&
                            type <= FileFmt.AnimTargetType.TevConstColor4Float_a)
                        {
                            return new FileFmt.Hermite();
                        }
                        else
                        {
                            return new FileFmt.HermiteU8();
                        }
                    }
                case ParamaterKind.Animation_AlphaTest: return new FileFmt.Hermite();
                case ParamaterKind.Animation_PerCharacterTransform: return new FileFmt.Hermite();
                case ParamaterKind.Animation_PerCharacterTransformCurve: return new FileFmt.Hermite();
                case ParamaterKind.Animation_Window: return new FileFmt.Hermite();
                case ParamaterKind.Animation_ExtUserData: return new FileFmt.Hermite();
                case ParamaterKind.Animation_MaskTextureSRT: return new FileFmt.Hermite();
                case ParamaterKind.Animation_DropShadow: return new FileFmt.Hermite();
                case ParamaterKind.Animation_ProceduralShape: return new FileFmt.Hermite();
                default: Debug.Assert( false ); return null;
            }
        }

        /// <summary>
        /// AnimTag のフラグ情報を ParamaterKind へ変換します。
        /// </summary>
        public static ParamaterKind ConvertAnimTagFlagsToParamaterKind(AnimTag animTag)
        {
            ParamaterKind result = ParamaterKind.None;

            if (animTag.outputPaneSRT) { result |= ParamaterKind.Animation_PaneSRT; }
            if (animTag.outputVertexColor) { result |= ParamaterKind.Animation_VertexColors; }
            if (animTag.outputVisibility) { result |= ParamaterKind.Animation_Visivility; }
            if (animTag.outputMaterialColor) { result |= ParamaterKind.Animation_MaterialColors; }
            if (animTag.outputTexturePattern) { result |= ParamaterKind.Animation_TexturePattern; }
            if (animTag.outputTextureSRT) { result |= ParamaterKind.Animation_TextureSRT; }
            if (animTag.outputIndTextureSRT) { result |= ParamaterKind.Animation_IndirectTextureSRT; }
            if (animTag.outputAlphaTest) { result |= ParamaterKind.Animation_AlphaTest; }
            if (animTag.outputPerCharacterTransform) { result |= ParamaterKind.Animation_PerCharacterTransform; }
            if (animTag.outputWindow) { result |= ParamaterKind.Animation_Window; }
            if (animTag.outputExtUserData) { result |= ParamaterKind.Animation_ExtUserData; }
            if (animTag.outputMaskTextureSRT) { result |= ParamaterKind.Animation_MaskTextureSRT; }
            if (animTag.outputDropShadow) { result |= ParamaterKind.Animation_DropShadow; }
            if (animTag.outputProceduralShape) { result |= ParamaterKind.Animation_ProceduralShape; }

            // 文字毎の変換を表現するカーブ。
            // 時間軸にそったアニメーションではないので区間タグでは管理されない。
            // ここでは、常にバイナリ出力するようにフラグを立てておく。
            result |= ParamaterKind.Animation_PerCharacterTransformCurve;

            return result;
        }

        /// <summary>
        /// キーフレーム配列を変換生成します。
        /// </summary>
        static public FileFmt.Hermite[]
           Convert_KeyFrams(
           AppData.Core.IAnmAttribute anmAttribute,
           ParamaterKind animationFileType,
           FileFmt.AnimTargetType type,
           string tag)
        {
            return Convert_KeyFrams_( anmAttribute, animationFileType, type, tag);
        }

        /// <summary>
        /// 複数キーで記録する必要のある種類か
        /// </summary>
        static bool KeyMustBeRecordedAsMultiKeys_( Hermite dstKeyFrm )
        {
            return !( dstKeyFrm is StepBool ) && !( dstKeyFrm is StepU16 );
        }

        /// <summary>
        /// キーフレーム配列を変換生成します。
        /// </summary>
        /// <param name="anmAttribute"></param>
        /// <returns></returns>
        static FileFmt.Hermite[]
            Convert_KeyFrams_(
            AppData.Core.IAnmAttribute anmAttribute,
            ParamaterKind animationFileType,
            FileFmt.AnimTargetType type,
            string tag
            )
        {
            IAnmCurve anmCurve = anmAttribute.GetAnmCurveRelationTag(tag);
            if (anmCurve == null)
            {
                return new Hermite[0];
            }

            // すべてのキーフレームについて...
            ArrayList dstKeyFrmSet = new ArrayList();
            foreach( IAnmKeyFrame srcKeyFrm in anmCurve.IKeyFrameSet )
            {
                Hermite dstKeyFrm = GetFileFmtKeyInstance_(animationFileType, type);
                dstKeyFrm.frame = srcKeyFrm.Time;
                dstKeyFrm.value = srcKeyFrm.ValueAsFloat;
                dstKeyFrm.slope = srcKeyFrm.InTangent;
                dstKeyFrm.slopeType = ConvertToSlopeType( srcKeyFrm.InterporationType );

                // キーを登録
                dstKeyFrmSet.Add( dstKeyFrm );

                // ステップキー
                // 次のキーの位置にキーを挿入します。
                if( srcKeyFrm.InterporationType == InterporationType.Step &&
                    KeyMustBeRecordedAsMultiKeys_( dstKeyFrm ) )
                {
                    IAnmKeyFrame nextKey = srcKeyFrm.OwnerIAnmCurve.GetNextKeyFrame( srcKeyFrm );
                    Hermite dstKeyFrm_Right = GetFileFmtKeyInstance_(animationFileType, type);

                    dstKeyFrm_Right.frame = ( nextKey != null ) ? nextKey.Time : srcKeyFrm.Time;
                    dstKeyFrm_Right.value = srcKeyFrm.ValueAsFloat;
                    dstKeyFrm_Right.slope = srcKeyFrm.OutTangent;
                    dstKeyFrm_Right.slopeType = dstKeyFrm.slopeType;
                    // OutTangent用（右スロープ）のキーフレームを登録
                    dstKeyFrmSet.Add( dstKeyFrm_Right );
                }
                else
                {
                    // ファイルフォーマットの都合から
                    // In Out のスロープ値（
                    // srcKeyFrm.InTangent と srcKeyFrm.OutTangent）が異なる場合には、
                    // 同一時間にキーを複製し新規登録する必要があります。
                    if( srcKeyFrm.InTangent != srcKeyFrm.OutTangent )
                    {
                        Hermite dstKeyFrm_Right = GetFileFmtKeyInstance_(animationFileType, type);
                        // オフセット分ずらした位置として変換します。
                        dstKeyFrm_Right.frame = dstKeyFrm.frame;
                        dstKeyFrm_Right.value = dstKeyFrm.value;
                        dstKeyFrm_Right.slope = srcKeyFrm.OutTangent; // NOTICE !
                        dstKeyFrm_Right.slopeType = dstKeyFrm.slopeType;

                        // OutTangent用（右スロープ）のキーフレームを登録
                        dstKeyFrmSet.Add( dstKeyFrm_Right );
                    }
                }
            }

            return dstKeyFrmSet.ToArray( typeof( Hermite ) ) as Hermite[];
        }

        /// <summary>
        /// アニメーションアトリビュートにキーフレーム情報を設定します。
        /// </summary>
        /// <param name="dstAttribute"></param>
        /// <param name="srcAnimTarget"></param>
        static public void SetHermiteKeyFrame_
            (
            AnmAttribute dstAttribute,
            FileFmt.AnimTarget srcAnimTarget,
            string tag
            )
        {
            AnmCurve anmCurve = dstAttribute.GetAnmCurveRelationTag(tag) as AnmCurve;

            // カーブが無い(非アクティブ状態)
            if (anmCurve == null)
            {
                return;
            }

            // 読み込み専用なら、設定をキャンセルします。
            if (anmCurve.IsReadOnlyLocked)
            {
                return;
            }

            // 読み取り専用ではない
            // ファイルに読み取り専用と記録されたキーを入力しようとしている。
            // この場合、派生元でキー削除があったとみなし入力をキャンセルする。
            if (srcAnimTarget.readonlyLocked)
            {
                return;
            }

            // Pre Post Infinity 情報をロードします。
            RlanHelper.SetPrePostInfinityToAnmCurve( anmCurve, srcAnimTarget );

            // パラメタライズドアニメーションカーブを設定する
            if (srcAnimTarget.parameterizedAnimParameter != null && srcAnimTarget.parameterizedAnimParameter.Length > 0)
            {
                anmCurve.SetParameterizedAnimCurve(srcAnimTarget.parameterizedAnimParameter);
            }

            Hermite[] srcKeySet = srcAnimTarget.key;

            if(srcKeySet == null)
            {
                return;
            }

            // 全てのキーについて
            int keyIdx0 = 0;
            float scaleConversion = 1.0f;
            while( keyIdx0 < srcKeySet.Length )
            {
                // 浮動小数点カラーになる可能性のあるカーブは内部的に float で扱うため
                // 読み込んだ byte のデータを 0.0f - 1.0f に変換する。
                if ((srcAnimTarget.target >= AnimTargetType.BlackColor_r &&
                     srcAnimTarget.target <= AnimTargetType.WhiteColor_a) ||
                    (srcAnimTarget.target >= AnimTargetType.TevColor0_r &&
                     srcAnimTarget.target <= AnimTargetType.TevConstColor4_a))
                {
                    // キーを作成する際に byte の値域でスケールをかけて 0.0 - 1.0 に変換する。
                    scaleConversion = 255.0f;
                }

                InterporationType interpTypeKey0 =
                    ConvertToInterporationType( srcKeySet[keyIdx0].slopeType );

                // 次のキーが存在する
                int keyIdx1 = keyIdx0 + 1;
                if( keyIdx1 < srcKeySet.Length )
                {
                    InterporationType interpTypeKey1 =
                        ConvertToInterporationType( srcKeySet[keyIdx1].slopeType );
                    // ステップキーか判定します。
                    if( KeyMustBeRecordedAsMultiKeys_( srcKeySet[keyIdx0] ) &&
                        interpTypeKey0 == InterporationType.Step &&
                        interpTypeKey1 == InterporationType.Step )
                    {
                        int keyIdx2 = keyIdx1 + 1;
                        if( keyIdx2 < srcKeySet.Length )
                        {
                            if( srcKeySet[keyIdx1].frame == srcKeySet[keyIdx2].frame )
                            {
                                // 2つのキーを１つの
                                // ステップキーとして登録(余分なキーはスキップします)
                                // キーフレームとして 登録
                                anmCurve.AddNewKeyFrame(
                                    (int)srcKeySet[keyIdx0].frame,
                                    srcKeySet[keyIdx0].value,
                                    true,
                                    srcKeySet[keyIdx0].slope,
                                    srcKeySet[keyIdx0].slope,
                                    interpTypeKey0,
                                    scaleConversion);

                                // 次の次のキーへ
                                keyIdx0 = keyIdx1 + 1;
                                continue;
                            }
                        }
                        else if( srcKeySet[keyIdx0].frame == srcKeySet[keyIdx1].frame )
                        {
                            // 終端キーステップキーは同一フレームに2つのキー
                            anmCurve.AddNewKeyFrame(
                                    (int)srcKeySet[keyIdx0].frame,
                                    srcKeySet[keyIdx0].value,
                                    true,
                                    srcKeySet[keyIdx0].slope,
                                    srcKeySet[keyIdx0].slope,
                                    interpTypeKey0,
                                    scaleConversion);

                            // 次の次のキーへ
                            keyIdx0 = keyIdx1 + 1;
                            continue;
                        }
                    }
                    // 同一時間か判定します
                    else if( srcKeySet[keyIdx0].frame == srcKeySet[keyIdx1].frame )
                    {
                        // 同一時間に存在する最後のキーを探します。
                        int sameTimeKeyIdxEnd = keyIdx1;
                        while( sameTimeKeyIdxEnd + 1 < srcKeySet.Length &&
                                srcKeySet[keyIdx1].frame == srcKeySet[sameTimeKeyIdxEnd + 1].frame )
                        {
                            sameTimeKeyIdxEnd++;
                        }

                        // 複数のキーを１つの
                        // 左右別スロープのキーとして登録する
                        // 両端以外のキー情報は破棄される。
                        anmCurve.AddNewKeyFrame(
                                (int)srcKeySet[keyIdx0].frame,
                                srcKeySet[keyIdx0].value,
                                srcKeySet[keyIdx0].slope == srcKeySet[sameTimeKeyIdxEnd].slope,
                                srcKeySet[keyIdx0].slope,
                                srcKeySet[sameTimeKeyIdxEnd].slope, // 最後のキーのスロープを使用します。
                                interpTypeKey0,
                                scaleConversion);

                        // 同一時間中、最後のキーの次のキーへ
                        keyIdx0 = sameTimeKeyIdxEnd + 1;
                        continue;
                    }
                }
                // 通常のキーとして登録
                anmCurve.AddNewKeyFrame(
                        (int)srcKeySet[keyIdx0].frame,
                        srcKeySet[keyIdx0].value,
                        true,
                        srcKeySet[keyIdx0].slope,
                        srcKeySet[keyIdx0].slope,
                        interpTypeKey0,
                        scaleConversion);
                // 次のキーへ
                keyIdx0 = keyIdx0 + 1;
            }
        }

        /// <summary>
        /// アニメーションカーブにキーフレームを設定します。
        /// </summary>
        public static void SetBoolKeyFrame_( AnmAttribute dstAttribute, FileFmt.AnimTarget srcTarget, string tag )
        {
            SetHermiteKeyFrame_( dstAttribute, srcTarget, tag );
        }

        /// <summary>
        /// アニメーションカーブに Pre Post Infinity 設定を行います。
        /// </summary>
        public static void SetPrePostInfinityToAnmCurve(
            AnmCurve anmCurve,
            FileFmt.AnimTarget srcAnimTarget )
        {
            anmCurve.PreInfinityType = RlanHelper.ConvertToInfinityType( srcAnimTarget.preInfinityType );
            anmCurve.PostInfinityType = RlanHelper.ConvertToInfinityType( srcAnimTarget.postInfinityType );
        }

        /// <summary>
        /// アニメーションロード関数を取得します。???
        /// </summary>
        public static AnimLoadFunction[] GetAnimLoadFunctionSet( ParamaterKind animKind )
        {
            switch( animKind )
            {
                case ParamaterKind.Animation_TextureSRT:return _loadFuncRlts;
                case ParamaterKind.Animation_IndirectTextureSRT:return _loadFuncRlis;
                case ParamaterKind.Animation_TexturePattern: return _loadFuncRltp;
                case ParamaterKind.Animation_MaterialColors: return _loadFuncRlmc;
                case ParamaterKind.Animation_PaneSRT: return _loadFuncRlpa;
                case ParamaterKind.Animation_VertexColors: return _loadFuncRlvc;
                case ParamaterKind.Animation_Visivility: return _loadFuncRlvi;
                case ParamaterKind.Animation_AlphaTest: return _loadFuncRlac;
                case ParamaterKind.Animation_FontShadowColor: return _loadFuncRlfs;
                case ParamaterKind.Animation_PerCharacterTransform: return _loadFuncRlto;
                case ParamaterKind.Animation_PerCharacterTransformCurve: return _loadFuncRltc;
                case ParamaterKind.Animation_Window: return _loadFuncRlwn;
                case ParamaterKind.Animation_ExtUserData: return _loadFuncRlud;
                case ParamaterKind.Animation_MaskTextureSRT: return _loadFuncRlma;
                case ParamaterKind.Animation_DropShadow: return _loadFuncRlds;
                case ParamaterKind.Animation_ProceduralShape: return _loadFuncRlps;

                default:
                {
                    Debug.Assert( false );
                    return new AnimLoadFunction[0];
                }
            }
        }


        /// <summary>
        /// ファイルロードを実行します。
        /// </summary>
        public static void DoLoad(
            AnimLoadFunction loadFunction,
            IEnumerable<AppData.IPane> dstPaneArray,
            FileFmt.AnimContent[] animContentSet,
            string tag)
        {
            foreach( FileFmt.AnimContent srcAnimContent in animContentSet )
            {
                foreach( IPane dstPane in dstPaneArray )
                {
                    loadFunction( dstPane, srcAnimContent, tag);
                }
            }
        }

        //----------------------------------------------------------
        // 保存関係
        //----------------------------------------------------------

        /// <summary>
        /// すべてのペインのアニメーションを保存形式に変換
        /// </summary>
        static public FileFmt.AnimContent[]
            SaveAllPaneAnim(ParamaterKind kind, AppData.ISubScene srcSubScene, string tag)
        {
            return srcSubScene.IPaneArray.SelectMany(x => SavePaneAnim(kind, x, tag)).ToArray();
        }

        /// <summary>
        /// ペインのアニメーションを保存形式に変換
        /// </summary>
        static public FileFmt.AnimContent[] SavePaneAnim(ParamaterKind kind, AppData.IPane pane, string tag)
        {
            switch (kind)
            {
                case ParamaterKind.Animation_PaneSRT: return RlpaConverter.SavePaneSRTAnim(pane, tag);
                case ParamaterKind.Animation_Visivility: return RlviConverter.SaveVisibilityAnim(pane, tag);
                case ParamaterKind.Animation_VertexColors: return RlvcConverter.SaveVertexColorAnim(pane, tag);
                case ParamaterKind.Animation_MaterialColors: return RlmcConverter.SaveMaterialAnimation(pane, tag);
                case ParamaterKind.Animation_TexturePattern: return RltpConverter.SaveTexturePatternAnim(pane, tag);
                case ParamaterKind.Animation_TextureSRT: return RltsConverter.SaveTextureSRTAnim(pane, tag);
                case ParamaterKind.Animation_IndirectTextureSRT: return RltsConverter.SaveIndirectTextureSRTAnim(pane, tag);
                case ParamaterKind.Animation_AlphaTest: return RlacConverter.SaveAnimation(pane, tag);
                case ParamaterKind.Animation_FontShadowColor: return RlfsConverter.SaveAnimation(pane, tag);
                case ParamaterKind.Animation_PerCharacterTransform: return RltoConverter.SaveAnimation(pane, tag);
                case ParamaterKind.Animation_PerCharacterTransformCurve: return RltoConverter.SavePerCharTransCurveAnim(pane, tag);
                case ParamaterKind.Animation_Window: return RlwnConverter.SavePerCharTransCurveAnim(pane, tag);
                case ParamaterKind.Animation_ExtUserData: return RludConverter.SaveExtUserDataAnim(pane, tag);
                case ParamaterKind.Animation_MaskTextureSRT: return RlmaConverter.SaveMaskAnim(pane, tag);
                case ParamaterKind.Animation_DropShadow: return RldsConverter.SaveDropShadowAnim(pane, tag);
                case ParamaterKind.Animation_ProceduralShape: return RlpsConverter.SaveProceduralShapeAnim(pane, tag);

                default: Debug.Assert(false); return new FileFmt.AnimContent[0];
            }
        }

        /// <summary>
        /// AppData.Material => FileFmt.AnimTarget[]
        /// </summary>
        static public IEnumerable<TType> ExtractAnimTargetIfValid<TType>(
            IAnmAttribute srcAttr, ParamaterKind kind, FileFmt.AnimTargetType type, byte id, string tag, bool exportEmpty)
            where TType : AnimTarget, new()
        {
            var dstTarget = new TType()
            {
                id = id,
            };

            // パラメタライズドアニメーションの出力
            IAnmCurve anmCurve = tag == null ? srcAttr.ICurrentAnimationCurve: srcAttr.GetAnmCurveRelationTag(tag);
            if (anmCurve != null && anmCurve.IsParameterizedAnim)
            {
                dstTarget.parameterizedAnimParameter = anmCurve.ParameterizedAnimParameters;
                dstTarget.target = type;
                yield return dstTarget;
            }

            FileFmt.Hermite[] dstKeys = RlanHelper.Convert_KeyFrams(srcAttr, kind, type, tag);
            if (exportEmpty || dstKeys.Length != 0)
            {
                dstTarget.key = dstKeys;
                dstTarget.target = type;

                IAnmCurve curve = tag == null ? srcAttr.ICurrentAnimationCurve : srcAttr.GetAnmCurveRelationTag(tag);
                dstTarget.preInfinityType = RlanHelper.ConvertToInfinityType(curve.PreInfinityType);
                dstTarget.postInfinityType = RlanHelper.ConvertToInfinityType(curve.PostInfinityType);
                dstTarget.readonlyLocked = curve.IsReadOnlyLocked;

                yield return dstTarget;
            }
        }

        #region ConvertToAnimContents

        /// <summary>
        /// ConvertToAnimContents を使って FileFmt.AnimContent を生成するための情報
        /// </summary>
        internal class AnimContentSource<S>
        {
            public AnimContentSource(string name, S source)
            {
                this.name = name;
                this.source = source;
            }

            public AnimContentSource(string name, string extUserDataTargetName, S source)
            {
                this.name = name;
                this.extUserDataTargetName = extUserDataTargetName;
                this.source = source;
            }

            public string name { get; set; }
            public string extUserDataTargetName { get; set; }
            public S source { get; set; }
        }

        /// <summary>
        /// ConvertToAnimContents を使って AnimTarget を生成するための情報
        /// </summary>
        internal class AnimTargetSource
        {
            public AnimTargetSource(IAnmAttribute attribute, AnimTargetType type, byte id = 0)
            {
                this.type = type;
                this.attribute = attribute;
                this.id = id;
            }

            public AnimTargetType type { get; set; }
            public IAnmAttribute attribute { get; set; }
            public byte id { get; set; }
        }

        /// <summary>
        /// attribute のサブアトリビュートに対応する AnimTargetSource を列挙する
        /// </summary>
        internal static IEnumerable<AnimTargetSource> GetAnimTargetSourcesFromSubAttributes(IAnmAttribute attribute, IEnumerable<AnimTargetType> types, byte id = 0)
        {
            return types.Select((x, i) => new AnimTargetSource(attribute.FindSubAttributeByIdx(i), x, id));
        }

        /// <summary>
        /// AnimContent を生成して列挙します。
        /// </summary>
        /// <typeparam name="S">S のインスタンスごとに AnimContent を生成します。</typeparam>
        /// <typeparam name="T">AnimTarget の型</typeparam>
        /// <param name="tag">指定されたタグ名のカーブを出力します。</param>
        /// <param name="kind">パラメータの種類</param>
        /// <param name="pane">出力対象のペイン</param>
        /// <param name="getAnimContentSource">AnimContent を生成するのに必要な情報を取得します。</param>
        /// <param name="getAnimTargetSources">AnimTarget を生成するのに必要な情報を取得します。</param>
        /// <returns></returns>
        internal static IEnumerable<FileFmt.AnimContent> ConvertToAnimContents<S, T>(
            string tag,
            ParamaterKind kind,
            AppData.IPane pane,
            Func<IPane, IEnumerable<AnimContentSource<S>>> getAnimContentSource,
            Func<S, IEnumerable<AnimTargetSource>> getAnimTargetSources) where T : AnimTarget, new()
        {
            return ConvertToAnimContents<S, T>(
                tag,
                pane,
                getAnimContentSource,
                (t, s) => getAnimTargetSources(s).SelectMany(info =>
                    RlanHelper.ExtractAnimTargetIfValid<T>(info.attribute, kind, info.type, info.id, t, false)));
        }

        /// <summary>
        /// AnimContent を生成して列挙します。
        /// </summary>
        /// <typeparam name="S">S のインスタンスごとに AnimContent を生成します。</typeparam>
        /// <typeparam name="T">AnimTarget の型</typeparam>
        /// <param name="tag">指定されたタグ名のカーブを出力します。</param>
        /// <param name="pane">出力対象のペイン</param>
        /// <param name="getAnimContentSource">AnimContent を生成するのに必要な情報を取得します。</param>
        /// <param name="getAnimTargets">AnimContent 内の AnimTargets を生成します。</param>
        /// <returns></returns>
        internal static IEnumerable<FileFmt.AnimContent> ConvertToAnimContents<S, T>(
            string tag,
            AppData.IPane pane,
            Func<IPane, IEnumerable<AnimContentSource<S>>> getAnimContentSource,
            Func<string, S, IEnumerable<T>> getAnimTargets) where T : AnimTarget
        {
            return getAnimContentSource(pane).Select(s =>
            {
                var targets = getAnimTargets(tag, s.source).ToArray();

                if (targets.Any())
                {
                    return new AnimContent()
                    {
                        name = s.name,
                        extUserDataTargetName = s.extUserDataTargetName,
                        Items = targets.ToArray<AnimTarget>(),
                    };
                }
                else
                {
                    return null;
                }
            }).Where(x => x != null);
        }
        #endregion
    }
}
