﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Structures.Core.Command;
    using LECore.Structures.Nsrif.Attributes;
    using System.Diagnostics;
    using StateMachineCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IStateMachine>;

    public class StateMachineManipulator : BaseManipulator
    {
        StateMachineData _target = null;
        StateMachineCommandFactory _commandFactory;

        //----------------------------------------------------------

        public StateMachineManipulator()
        {
            _commandFactory = new StateMachineCommandFactory(delegate (IStateMachine mementoSrc) { return new StateMachineMemento(mementoSrc); });
        }

        public void BindTarget(IStateMachine target)
        {
            _target = target as StateMachineData;
            System.Diagnostics.Debug.Assert(_target != null);
        }

        //----------------------------------------------------------


        public IStateMachine IStateMachine
        {
            get
            {
                return _target;
            }
        }

        public bool IsEnabled
        {
            set
            {
                if (_target.IsEnabled != value)
                {
                    _commandFactory.Modify(_target, delegate ()
                    {
                        _target.IsEnabled = value;
                    });
                }
            }
        }

        //----------------------------------------------------------

        public void InitialSetup()
        {
            if (!_target.Layers.Any())
            {
                _commandFactory.Modify(_target, delegate ()
                {
                    var layer = _target.AddLayer("Layer1");
                });
                SetLayerSelected("Layer1");
            }

            // Base を 選択状態にしておく
            foreach (var stateLayer in _target.Layers.Where(l => l.GetBaseState() == null))
            {
                SetLayerEditItemSelected(stateLayer, "Base");
            }
        }

        public static void SetStateMachineRunModeRecursively(IStateMachine stateMachine, StateMachineRunMode runMode)
        {
            var stateMachineData = stateMachine as StateMachineData;

            if(stateMachineData.RunMode != runMode)
            {
                stateMachineData.RunMode = runMode;
                // なにかの初期化処理
                stateMachineData.SetRunModeInitialState();
            }

            var partsPanes = stateMachine.OwnerSubScene.FindPanesByKind(PaneKind.Parts);
            foreach (var partsStateMachine in stateMachine.EnumrateChildStateMachine())
            {
                SetStateMachineRunModeRecursively(partsStateMachine, runMode);
            }
        }


        public void SetStateMachineRunMode(StateMachineRunMode runMode)
        {
            if(_target.RunMode != runMode)
            {
                SetStateMachineRunModeRecursively(_target, runMode);
            }
        }

        public void RunUpdate(float step, FVec2 mousePos, bool isLButtonClick)
        {
            _target.OwnerSubScene.BeginMassiveModify();
            _target.RunUpdate(step, mousePos, isLButtonClick);
            // 時間変更イベントによって再描画がトリガーされるので、変更を適用した際のイベント通知を行わない。
            _target.OwnerSubScene.EndMassiveModify(true, true);
        }

        //----------------------------------------------------------

        public void AddStateToLayerByCloneSource(IStateLayer layer, string newStateName, string sourceStateName)
        {
            if (layer.FindStateByName(newStateName) != null)
            {
                return;
            }

            var sourceState = layer.FindStateByName(sourceStateName);
            if (sourceState == null)
            {
                return;
            }

            var addedState = _target.AddStateToLayer(layer.Name, newStateName);
            addedState.CloneAllParamaters(sourceState);
            SetLayerEditItemSelected(layer, newStateName);
        }

        public void RemoveEditItemFromLayer(IStateLayer layer, IStateLayerEditItem editItem)
        {
            if(!editItem.IsDeletable())
            {
                return;
            }

            if (editItem is IState)
            {
                RemoveStateFromLayer(layer, (editItem as IState).Name);
            }
            else if (editItem is IStateTransition)
            {
                RemoveTranstionFromLayer(layer, (editItem as IStateTransition).StartStateName, (editItem as IStateTransition).EndStateName);
            }
            else
            {
                Debug.Assert(false);
            }
        }

        public void RemoveStateFromLayer(IStateLayer layer, string stateName)
        {
            var state = layer.FindStateByName(stateName);
            if (state == null)
            {
                return;
            }

            // Base ステートは削除禁止。
            if(state.IsBaseState())
            {
                return;
            }

            _target.RemoveStateFromLayer(layer.Name, stateName);
        }

        //----------------------------------------------------------

        public void AddTranstionToLayer(IStateLayer layer, string startStateName, string endStateName)
        {
            if (layer.FindTransitionByName(startStateName, endStateName) != null)
            {
                return;
            }

            var transition = _target.AddTransitionToLayer(layer.Name, startStateName, endStateName);
            SetLayerEditItemSelected(layer, transition.EditName);
        }

        public void RemoveTranstionFromLayer(IStateLayer layer, string startStateName, string endStateName)
        {
            if (layer.FindTransitionByName(startStateName, endStateName) == null)
            {
                return;
            }

            _target.RemoveTransitionFromLayer(layer.Name, startStateName, endStateName);
        }

        public void EditTranstion(IStateLayer layer, string startStateName, string endStateName, float totalDuration, IStateEasing stateEasing, bool isCalcelable, bool isLoop)
        {
            if (layer.FindTransitionByName(startStateName, endStateName) == null)
            {
                return;
            }

            _target.SetTransition(layer.Name, startStateName, endStateName, (transition) =>
             {
                 bool isModified = false;

                 if (transition.TotalDuration != totalDuration)
                 {
                     float scaling = totalDuration / transition.TotalDuration;

                     transition.TotalDuration = totalDuration;
                     isModified |= true;

                     // スケールを子要素に適用する
                     (transition as StateTransitionData).Scale(0.0f, scaling, false);
                 }

                 if (!transition.StateEasing.IsSame(stateEasing))
                 {
                     (transition.StateEasing as StateEasing).EasingType = stateEasing.EasingType;
                     (transition.StateEasing as StateEasing).EasingExtraParamater = stateEasing.EasingExtraParamater;
                     isModified |= true;
                 }

                 if (transition.IsCancelable != isCalcelable)
                 {
                     transition.IsCancelable = isCalcelable;
                     isModified |= true;
                 }

                 if (transition.IsLoop != isLoop)
                 {
                     transition.IsLoop = isLoop;
                     isModified |= true;
                 }

                 return isModified;
             });
        }

        public void EditTranstionTrigger(IStateLayer layer, string startStateName, string endStateName, TriggerKind triggerKind, StateMachineOperator op, string variableName, float variableValue)
        {
            if (layer.FindTransitionByName(startStateName, endStateName) == null)
            {
                return;
            }

            _target.SetTransition(layer.Name, startStateName, endStateName, (transition) =>
            {
                var trigger = transition.Trigger as StateTransitionTriggerData;
                Debug.Assert(trigger != null);

                bool isModified = false;
                if (trigger.TriggerKind != triggerKind)
                {
                    trigger.TriggerKind = triggerKind;
                    isModified |= true;
                }

                if (trigger.StateMachineOperator != op)
                {
                    trigger.StateMachineOperator = op;
                    isModified |= true;
                }

                // TODO : variableName, variableValue

                return isModified;
            });
        }

        public void EditTransitionTrack(IStateLayer layer, IStateTransition transition, string name, FeatureParamaterKind kind,
            float offset, float duration, IStateEasing stateEasing)
        {
            if (layer.FindTransitionByName(transition.StartStateName, transition.EndStateName) == null)
            {
                return;
            }

            _target.SetTransition(layer.Name, transition.StartStateName, transition.EndStateName, (t) =>
            {
                var track = t.FindIStateTransitionTrackDataByName(name, kind);
                if (!track.IsSame(offset, duration, stateEasing))
                {
                    t.SetStateTransitionTrackDataByName(name, kind, offset, duration, stateEasing);
                    return true;
                }

                return false;
            });
        }

        //----------------------------------------------------------

        public void AddTransitionTrackKey(IStateLayer layer, IStateTransition transition, string name, FeatureParamaterKind kind,
            float time,
            IStateEasing stateEasing,
            float x, float y, float z, float w)
        {
            if (layer.FindTransitionByName(transition.StartStateName, transition.EndStateName) == null)
            {
                return;
            }

            _target.SetTransition(layer.Name, transition.StartStateName, transition.EndStateName, (t) =>
            {
                StateTransitionTrackData track = t.FindStateTransitionTrackDataByName(name, kind);
                bool isModified = false;

                // track 区間をはみ出している場合は拡張する。
                {
                    float offset = track.Offset;
                    float duration = track.Duration;

                    bool needToAdjustTrack = false;
                    if (track.Offset > time)
                    {
                        offset = time;
                        needToAdjustTrack |= true;
                    }
                    else if (track.Offset + track.Duration < time)
                    {
                        duration = time - track.Offset + 1;
                        needToAdjustTrack |= true;
                    }

                    if (needToAdjustTrack)
                    {
                        t.SetStateTransitionTrackDataByName(name, kind, offset, duration, track.StateEasing);
                        isModified |= needToAdjustTrack;
                    }
                }

                if (t.SetStateTransitionTrackKey(name, kind, time, stateEasing, x, y, z, w))
                {
                    isModified |= true;
                }

                return isModified;
            });
        }

        public void RemoveTransitionTrackKey(IStateLayer layer, IStateTransition transition, string name, FeatureParamaterKind kind, float time)
        {
            if (layer.FindTransitionByName(transition.StartStateName, transition.EndStateName) == null)
            {
                return;
            }

            _target.SetTransition(layer.Name, transition.StartStateName, transition.EndStateName, (t) =>
            {
                var track = t.FindIStateTransitionTrackDataByName(name, kind) as StateTransitionTrackData;

                // 同じ時間に、キーが存在するなら更新する
                KeyParamater keyParamater = track.KeyParamaters.FirstOrDefault(key => key.Time == time);
                if (keyParamater != null)
                {
                    track.KeyParamaterList.Remove(keyParamater);
                    return true;
                }

                return false;
            });
        }

        //----------------------------------------------------------

        /// <summary>
        ///
        /// </summary>
        public void AddTransitionTrackEvent(IStateLayer layer, IStateTransition transition, string name,
            float time,
            StateTrackKeyEventKind keyEventKind,
            StateMachineEventKind eventkind, object param1, object param2, uint delayFrames)
        {
            FeatureParamaterKind kind = FeatureParamaterKind.StateMachineEvent;

            if (layer.FindTransitionByName(transition.StartStateName, transition.EndStateName) == null)
            {
                return;
            }

            _target.SetTransition(layer.Name, transition.StartStateName, transition.EndStateName, (t) =>
            {
                StateTransitionTrackData track = t.FindStateTransitionTrackDataByName(name, kind);
                if(track == null)
                {
                    Debug.Assert(false);
                    return false;
                }

                bool isModified = false;

                // track 区間をはみ出している場合は拡張する。（TODO：メソッド化）
                {
                    float offset = track.Offset;
                    float duration = track.Duration;

                    bool needToAdjustTrack = false;
                    if (track.Offset > time)
                    {
                        offset = time;
                        needToAdjustTrack |= true;
                    }
                    else if (track.Offset + track.Duration < time)
                    {
                        duration = time - track.Offset + 1;
                        needToAdjustTrack |= true;
                    }

                    if (needToAdjustTrack)
                    {
                        t.SetStateTransitionTrackDataByName(name, kind, offset, duration, track.StateEasing);
                        isModified |= needToAdjustTrack;
                    }
                }

                if (t.SetStateTransitionTrackEvent(name, time, eventkind, param1, param2, delayFrames))
                {
                    isModified |= true;
                }

                return isModified;
            });
        }

        //----------------------------------------------------------

        public void SetLayerSelected(string name)
        {
            foreach (var layer in _target.Layers)
            {
                bool selectFlag = layer.Name == name;

                _target.SetLayer(layer.Name, l =>
                {
                    if (l.IsSelected != selectFlag)
                    {
                        l.IsSelected = selectFlag;
                        return true;
                    }

                    return false;
                });
            }
        }

        //----------------------------------------------------------

        public void SetLayerEditItemSelected(IStateLayer layer, string editName)
        {
            _target.OwnerSubScene.BeginMassiveModify();

            foreach (var prop in layer.EnumrateEditItems())
            {
                bool selectFlag = prop.EditName == editName;
                _target.SetPropertySelection(layer.Name, prop.EditName, selectFlag);
            }

            _target.OwnerSubScene.EndMassiveModify();
        }

        //----------------------------------------------------------

        public void AddParamatersToLayer(IStateLayer layer, IEnumerable<IPane> panes, FeatureParamaterKind flag)
        {
            _target.OwnerSubScene.BeginMassiveModify();

            foreach (var pane in panes)
            {
                foreach (FeatureParamaterKind kind in Enum.GetValues(typeof(FeatureParamaterKind)))
                {
                    if (flag.HasFlag(kind))
                    {
                        _target.AddNewFeatureParamaterToLayer(layer.Name, pane.PaneName, kind);
                    }
                }
            }

            _target.OwnerSubScene.EndMassiveModify();
        }

        //----------------------------------------------------------

        public void EditParamatersToLayer(IStateLayer layer, IEnumerable<IPane> panes, FeatureParamaterKind flag)
        {
            _target.OwnerSubScene.BeginMassiveModify();

            foreach (var pane in panes)
            {
                foreach (FeatureParamaterKind kind in Enum.GetValues(typeof(FeatureParamaterKind)))
                {
                    if (flag.HasFlag(kind))
                    {
                        _target.AddNewFeatureParamaterToLayer(layer.Name, pane.PaneName, kind);
                    }
                    else
                    {
                        _target.RemoveFeatureParamaterToLayer(layer.Name, pane.PaneName, kind);
                    }
                }
            }

            _target.OwnerSubScene.EndMassiveModify();
        }

        public void StoreParamatersFromCurrent(IStateLayer layer, IStateLayerEditItem editItem)
        {
            if(editItem is IState)
            {
                IState state = editItem as IState;


                IEnumerable<IFeatureParamater> paramaters = editItem.GetFreatureParamaters();
                _target.OwnerSubScene.BeginMassiveModify();
                foreach (var param in paramaters)
                {
                    _target.SetFeatureParamater(param, (p) =>
                    {
                        p.StoreFeatureParamaterValueFromScene(_target.OwnerSubScene, param.Name, param.Kind);
                        return true;
                    });
                }
                _target.OwnerSubScene.EndMassiveModify();
            }
            else if(editItem is IStateTransition)
            {
                IStateTransition transition = editItem as IStateTransition;


                IEnumerable<IFeatureParamater> paramaters = editItem.GetFreatureParamaters();

                _target.OwnerSubScene.BeginMassiveModify();
                foreach (var param in paramaters)
                {
                    _target.SetFeatureParamater(param, (p) =>
                    {

                        // Transition は現在の計算結果のテンポラリインスタンスが返ってくる。まずそこに、Store して
                        p.StoreFeatureParamaterValueFromScene(_target.OwnerSubScene, param.Name, param.Kind);
                        // Storeした結果をキーとして追加を試みる
                        AddTransitionTrackKey(layer, transition, param.Name, param.Kind, transition.LocalTime.Time, transition.StateEasing, p.X, p.Y, p.Z, p.W);

                        return true;
                    });
                }
                _target.OwnerSubScene.EndMassiveModify();
            }
            else
            {
                Debug.Assert(false);
            }
        }

        public void EditStateFeatureParamater(IStateLayer layer, IState state, string name, FeatureParamaterKind kind, float x, float y, float z, float w)
        {
            _target.SetStateFeatureParamater(layer.Name, state.Name, name, kind, (param) =>
            {
                bool isModified = false;

                if (param.X != x)
                {
                    param.X = x;
                    isModified |= true;
                }

                if (param.Y != y)
                {
                    param.Y = y;
                    isModified |= true;
                }

                if (param.Z != z)
                {
                    param.Z = z;
                    isModified |= true;
                }

                if (param.W != w)
                {
                    param.W = w;
                    isModified |= true;
                }

                return isModified;
            });
        }
    }
}
