﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using LECore.Structures.Core;

namespace LECore.Manipulator
{
    using System.IO;
    using LECore.Save_Load;
    using LECore.Structures;
    using LECore.Structures.Core.Command;
    using ParamaterAnimaitonCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IParamaterAnimaiton>;
    using ParamaterPaneCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IParamaterPane>;
    using PartsSettingsCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.PartsSettings>;
    using ControlSettingsCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.ControlSettings>;
    using ControlSettingsSetCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.ControlSettingsSet>;

    using System.Drawing;

    public class ControlSettingsSetManipulator : BaseManipulator
    {
        ControlSettingsSet _targetCtrlSet = null;

        private readonly ControlSettingsSetCommandCreater _ControlSettingsSetCommandCreater;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ControlSettingsSetManipulator()
        {
            _ControlSettingsSetCommandCreater = new ControlSettingsSetCommandCreater(
                (src) => new ControlSettingsSetMemento(src));
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        public void BindTarget(IControlSettingsSet targetCtrlSet)
        {
            Ensure.Argument.True(targetCtrlSet is ControlSettingsSet);
            _targetCtrlSet = targetCtrlSet as ControlSettingsSet;
        }

        /// <summary>
        /// 追加
        /// </summary>
        public void Add(IPartsControlSetting contorlSetting, string descriptionName, string description)
        {
            ControlSettings newCtrl = new ControlSettings(_targetCtrlSet);
            ControlSettingsManipulator.DoInitialize(newCtrl, contorlSetting, descriptionName, description);

            _ControlSettingsSetCommandCreater.Modify(_targetCtrlSet, () => _targetCtrlSet.CloneAdd(newCtrl));
        }

        /// <summary>
        /// 削除
        /// </summary>
        public void Remove(IControlSettings ctrlSettings)
        {
            _ControlSettingsSetCommandCreater.Modify(_targetCtrlSet, () => _targetCtrlSet.Remove(ctrlSettings));
        }
    }

    /// <summary>
    /// PartsSetting の編集クラスです。
    /// </summary>
    public class ControlSettingsManipulator : BaseManipulator
    {
        ControlSettings _targetCtrl = null;

        private readonly ControlSettingsCommandCreater _ControlSettingsCommandCreater;

        private readonly ParamaterPaneCommandCreater _ParamaterPaneCommandCreater;
        private readonly ParamaterAnimaitonCommandCreater _ParamaterAnimaitonCommandCreater;

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public ControlSettingsManipulator()
        {
            _ControlSettingsCommandCreater = new ControlSettingsCommandCreater(
                (src) => new ControlSettingsMemento(src));
            _ParamaterPaneCommandCreater = new ParamaterPaneCommandCreater(
                (src) => new ParamaterPaneMemento(src));
            _ParamaterAnimaitonCommandCreater = new ParamaterAnimaitonCommandCreater(
                (src) => new ParamaterAnimationMemento(src));
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        public void BindTarget(IControlSettings targetCtrl)
        {
            Ensure.Argument.True(targetCtrl is ControlSettings);
            _targetCtrl = targetCtrl as ControlSettings;
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <param name="tempalte">コントロール定義です。</param>
        public void InitialzeControl(IPartsControlSetting contorlSetting, string descriptionName, string description)
        {
            Ensure.Argument.NotNull(contorlSetting);

            if (_targetCtrl.Name != contorlSetting.Name ||
                _targetCtrl.DescriptionName != descriptionName ||
                _targetCtrl.Description != description)
            {
                _ControlSettingsCommandCreater.Modify(_targetCtrl, () => DoInitialize(_targetCtrl, contorlSetting, descriptionName, description));
            }
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        public static void DoInitialize(IControlSettings dstSettings, IPartsControlSetting srcSetting, string descriptionName, string description)
        {
            List<IParamaterPane> paramPanes = new List<IParamaterPane>();
            foreach (var pane in srcSetting.PaneNames)
            {
                paramPanes.Add(new ParamaterPane(pane.Name, string.Empty, pane.IsRequired));
            }

            List<IParamaterAnimaiton> paramAnims = new List<IParamaterAnimaiton>();
            foreach (var anim in srcSetting.AnimationNames)
            {
                paramAnims.Add(new ParamaterAnimaiton(anim.Name, string.Empty, anim.IsRequired));
            }

            (dstSettings as ControlSettings).Initialize(srcSetting.Name, srcSetting.UIName, descriptionName, description, paramPanes, paramAnims);
        }

        /// <summary>
        /// ペインの変更をします。
        /// </summary>
        public void ModifyParamaterPane(
            IParamaterPane paramaterPane, string paneName)
        {
            if (_targetCtrl.Panes.Contains(paramaterPane))
            {
                if (paramaterPane.PaneName != paneName)
                {
                    _ParamaterPaneCommandCreater.Modify(
                        paramaterPane, () => paramaterPane.PaneName = paneName);
                }
            }
        }

        /// <summary>
        /// ペインの変更をします。
        /// </summary>
        public void ModifyParamaterAnimation(
            IParamaterAnimaiton paramaterAnimaiton, string tagName)
        {
            if (_targetCtrl.Animations.Contains(paramaterAnimaiton))
            {
                if (paramaterAnimaiton.TagName != tagName)
                {
                    _ParamaterAnimaitonCommandCreater.Modify(
                        paramaterAnimaiton, () => paramaterAnimaiton.TagName = tagName);
                }
            }
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <param name="tempalte">部品設定のひな形です。</param>
        public void SetDescriptions(string descriptionName, string description)
        {
            if (_targetCtrl.DescriptionName != descriptionName ||
                _targetCtrl.Description != description)
            {
                _ControlSettingsCommandCreater.Modify(
                    _targetCtrl, () => _targetCtrl.SetDescriptions(descriptionName, description));
            }
        }
    }

    /// <summary>
    /// PartsSetting の編集クラスです。
    /// </summary>
    public class PartsSettingsManipulator : BaseManipulator
    {
        PartsSettings _target = null;

        private readonly PartsSettingsCommandCreater _PartsSettingsCommandCreater;

        //-----------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsSettingsManipulator()
        {
            _PartsSettingsCommandCreater = new PartsSettingsCommandCreater(
                (src) => new PartsSettingsMemento(src));
        }

        //-----------------------------------------

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        public void BindTarget(IPartsSettings target)
        {
            Ensure.Argument.True(target is PartsSettings);
            _target = target as PartsSettings;
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <param name="tempalte">コントロール定義です。</param>
        public void Initialze(IPartsControlSetting contorlSetting, string descriptionName, string description)
        {
            Ensure.Argument.NotNull(contorlSetting);

            if (_target.Name != contorlSetting.Name ||
                _target.DescriptionName != descriptionName ||
                _target.Description != description)
            {
                _PartsSettingsCommandCreater.Modify(_target, () =>
                    _target.Initialize(contorlSetting.Name, descriptionName, description, FVec3.Empty));
            }
        }

        /// <summary>
        /// 初期化します。
        /// </summary>
        /// <param name="tempalte">部品設定のひな形です。</param>
        public void SetDescriptions(string descriptionName, string description)
        {
            if (_target.DescriptionName != descriptionName ||
                _target.Description != description)
            {
                _PartsSettingsCommandCreater.Modify(
                    _target, () => _target.SetDescriptions(descriptionName, description));
            }
        }

        /// <summary>
        /// 上書きするペインを設定します。
        /// リセットする場合は、空のペイン列を指定します。
        /// </summary>
        public void SetOverrideProperties(IEnumerable<PartsPropaertySettingsSource> overridePanes)
        {
            _PartsSettingsCommandCreater.Modify(
                    _target, () => _target.SetOverrideProperties(overridePanes));
        }

        /// <summary>
        /// リセットします。
        /// </summary>
        public void Reset()
        {
            _PartsSettingsCommandCreater.Modify(_target, () => _target.Reset());
        }

        //----------------------------------------------------------

        private static void ClonePanes_(SubScene subScene, IPane pane)
        {
            // アニメーション付でクローンして登録。
            subScene.AddPane(pane.Clone());

            foreach(IPane child in pane.Children)
            {
                ClonePanes_(subScene, child);
            }
        }

        /// <summary>
        /// 選択ペインセットを部品レイアウトとして切り出します。
        /// </summary>
        public static void CreatePartsBySelectedNull(
            IPane nullPane, string filePath, string name, string description, IPartsControlSetting controlSetting)
        {
            // ペインを複製して登録。
            SubScene newScene = Scene.Instance.CreateSubSceneInstance();

            // フォントやテクスチャ類
            // フォントやテクスチャ類をコピーする
            SubSceneHelper.CopySubSceneTextureAndFont(nullPane.OwnerSubScene, newScene);

            foreach (IPane pane in nullPane.Children)
            {
                ClonePanes_(newScene, pane);
            }

            // キャプチャテクスチャのコピー
            SubSceneHelper.CopySubSceneCaptureTexture(nullPane.OwnerSubScene, newScene);

            // シーンを作り、ペインと関連リソースを複製する。
            // シーンを部品レイアウトとして部品ルート以下に保存する。
            // ツール内の部品レイアウト情報を更新する(これで部品として利用できる状態に)。
            var partsSettingsMnp = new PartsSettingsManipulator();
            partsSettingsMnp.BindTarget(newScene.IPartsSettings);
            partsSettingsMnp.Initialze(controlSetting, name, description);

            // 新規コントロールを追加
            var ctrlSettingsSetMnp = new ControlSettingsSetManipulator();
            ctrlSettingsSetMnp.BindTarget(newScene.IControlSettingsSet);
            ctrlSettingsSetMnp.Add(controlSetting, name, description);

            ExportOption exportOption = new ExportOption() {
                UseBaseValue = true,
                Frame = GlobalTime.Inst.Time
            };
            bool bSaveResult = LayoutEditorCore.ExportToFileAll(newScene, filePath, exportOption);
            string partsFileName = Path.GetFileName(filePath);
            var targetPartsSubScene = LayoutEditorCore.Scene.FindPartsSubSceneByFileName(partsFileName);
            if (targetPartsSubScene == null)
            {
                targetPartsSubScene = LayoutEditorCore.Scene.AddPartsSubScene(filePath);
            }

            (LayoutEditorCore.Scene as Scene).ReloadOnePartsSubScene(partsFileName);
        }
    }
}
