﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System.Linq;

namespace LECore.Manipulator
{
    using System;
    using LECore.Structures;
    using LECore.Structures.Core.Command;
    using PartsLayoutCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.PartsLayout>;
    using PropPaneCommandCreater = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.IPane>;

    /// <summary>
    /// PartsSetting の編集クラスです。
    /// </summary>
    public class PartsLayoutManipulator : BaseManipulator
    {
        PartsLayout _target = null;
        private readonly PartsLayoutCommandCreater _PartsLayoutCommandCreater;
        private readonly PropPaneCommandCreater _PropPaneCommandCreater;

        //-----------------------------------------

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public PartsLayoutManipulator()
        {
            _PartsLayoutCommandCreater = new PartsLayoutCommandCreater(
                (src) => new PartsLayoutMemento(src));

            _PropPaneCommandCreater = new MementoCommandFactory<IPane>(
                (src) => new PaneMemento(src));
        }

        //-----------------------------------------

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        public void BindTarget(IPartsLayout target)
        {
            Ensure.Argument.True(target is IPartsLayout);
            _target = target as PartsLayout;
        }

        /// <summary>
        /// 操作対象を取得します。
        /// </summary>
        public IPartsLayout IPartsLayout
        {
            get { return _target; }
        }

        /// <summary>
        /// レイアウト名設定(TODO:Undo)
        /// </summary>
        public string PartsLayoutName
        {
            set{ _target.PartsLayoutName = value;}
        }

        /// <summary>
        /// レイアウト名設定(TODO:Undo)
        /// </summary>
        public FVec3 Magnify
        {
            set
            {
                if (_target.Magnify != value)
                {
                    _PartsLayoutCommandCreater.Modify(_target, () => _target.Magnify = value);
                }
            }
        }

        /// <summary>
        /// 指定のサイズです。
        /// </summary>
        public FVec3 SizeConstraint
        {
            set
            {
                if (_target.SizeConstraint != value)
                {
                    _PartsLayoutCommandCreater.Modify(_target, () => _target.SizeConstraint = value);
                }
            }
        }

        //-----------------------------------------

        /// <summary>
        /// 自身が持つ部品ペインを上書きします。
        /// <remarks>
        /// 関連シーンは読み込み済みと想定しています。
        /// </remarks>
        /// </summary>
        public void OverrideChildePartsPane(string paneName, string newPartsLayoutName)
        {
            ISubScene parentSubScene = (_target as IPaneExParamater).OwnerPane.OwnerSubScene;
            parentSubScene.BeginMassiveModify();

            _PartsLayoutCommandCreater.Modify(
                _target,
                () => DoOverrideChildePartsPane(paneName, newPartsLayoutName));
            parentSubScene.EndMassiveModify();
        }

        /// <summary>
        /// 上書きプロパティを無効化します。
        /// 内部で、複数のコマンドを作るので、BeginMassiveModify で囲っています。
        /// </summary>
        public void ModifyOverwriteParamaters(string targetName, PaneKind paneKind, PartsPropertyUsageOptions option)
        {
            IPane targetPane = _target.PartsSubScene.FindPaneByName(targetName);
            if (targetPane == null)
            {
                return;
            }

            ISubScene parentSubScene = (_target as IPaneExParamater).OwnerPane.OwnerSubScene;
            parentSubScene.BeginMassiveModify();

            _PartsLayoutCommandCreater.Modify(
                _target,
                () =>
                    _PropPaneCommandCreater.Modify(
                    targetPane,
                    () => _target.ModifyOverwriteParamaters(_target.FindPartsPropertyByName(targetName), option)));

            parentSubScene.EndMassiveModify();
        }

        /// <summary>
        /// ペインにプロパティを上書きします
        /// </summary>
        public void SetPropertyToPane(IPartsLayout parentPartsLayout, string targetName, PartsPropertyUsageOptions option)
        {
            IPane targetPane = _target.PartsSubScene.FindPaneByName(targetName);
            if (targetPane == null)
            {
                return;
            }

            _PartsLayoutCommandCreater.Modify(
                _target,
                () =>
                    _PropPaneCommandCreater.Modify(
                    targetPane,
                    () => _target.SetPropertyToPane(
                        targetPane, _target.GetParentOverwriteProperty(parentPartsLayout, targetName), option)));
        }

        /// <summary>
        /// 部品ペインのリロード
        /// </summary>
        static public void ReloadPartsPane(IPane partsPane)
        {
            var targetSubScene = partsPane.OwnerSubScene as SubScene;
            try
            {
                targetSubScene.BeginMassiveModify();

                string paneName = partsPane.PaneName;

                var newPartsSubScene = LECore.LayoutEditorCore.Scene.FindPartsSubSceneByFileName(partsPane.IPartsLayout.PartsLayoutName);

                if (newPartsSubScene == null || !newPartsSubScene.IsLoaded)
                {
                    // エラーが起こっても処理は続行されます。
                    // 部品ペイン [{0}] の再読み込みに失敗しました。
                    LayoutEditorCore.MsgReporter.ReportError(
                        "",
                        LECoreStringResMgr.Get("ERROR_FAILED_TO_RELOAD_PARTSPANE"),
                        partsPane.IPartsLayout.PartsLayoutName);


                    return;
                }

                Ensure.Operation.True(newPartsSubScene.IsLoaded);

                // 新規部品ペインを登録する。
                SubSceneManipulator targetSubSceneMnp = new SubSceneManipulator();
                targetSubSceneMnp.BindTarget(targetSubScene);

                Pane newPartsPane = targetSubSceneMnp.AddPartsPaneWithoutUndo(
                    "TempParts", partsPane.IPartsLayout.PartsLayoutName, newPartsSubScene.SubScene) as Pane;

                FVec3 sizeConstraint = (partsPane.IPartsLayout as PartsLayout).SizeConstraint;
                FVec3 magnify = (partsPane.IPartsLayout as PartsLayout).Magnify;

                (partsPane.IPartsLayout as PartsLayout).Restore(newPartsPane.IPartsLayout as PartsLayout);

                (partsPane.IPartsLayout as PartsLayout).SizeConstraint = sizeConstraint;
                (partsPane.IPartsLayout as PartsLayout).Magnify = magnify;

                (partsPane.IPartsLayout as PartsLayout).CheckPartsPaneNameValidity();

                targetSubScene.RemovePain(newPartsPane);

                // 名前と階層構造が確定したのちにリロードしたパーツペインのキャプチャテクスチャ参照情報を更新します。
                SubSceneHelper.ResolvePartsLayoutCaptureTextureReference(partsPane);

            }
            finally
            {
                // Undoコマンドは破棄する。
                targetSubScene.EndMassiveModify(true, false);
            }
        }

        /// <summary>
        /// リロード：部品レイアウト変更
        /// </summary>
        public void ChangePartsPaneLayout(string newPartsLayoutName)
        {
            _PartsLayoutCommandCreater.Modify(
                _target, () => DoReloadPartsPane(_target.OwnerPane, newPartsLayoutName));
        }

        //----------------------------------------------------------

        /// <summary>
        /// 自身が持つ部品ペインを上書きします。
        /// </summary>
        private void DoOverrideChildePartsPane(string paneName, string newPartsLayoutName)
        {
            var partsProp = _target.RawPartsPropaerties.Find(
                (prop) =>
                    prop.PaneKind == PaneKind.Parts &&
                    prop.TargetName == paneName);
            if (partsProp == null)
            {
                return;
            }

            // 対象ペインを取る
            // 対象部品の兄弟に、新規に指定された部品ペインを作る。
            var paneParamater = partsProp.Paramater as IPaneExParamater;
            var partsPane = paneParamater.OwnerPane;

            IPane newPartsPane = DoReloadPartsPane(partsPane, newPartsLayoutName);
            if (newPartsPane != null)
            {
                _target.OverrideChildePartsPane(paneName, newPartsPane.IPaneExParamater);
            }
        }

        /// <summary>
        /// 自身が持つ部品ペインを上書きします。
        /// </summary>
        static private IPane DoReloadPartsPane(IPane partsPane, string newPartsLayoutName)
        {
            string paneName = partsPane.PaneName;
            var newPartsSubScene = LECore.LayoutEditorCore.Scene.FindPartsSubSceneByFileName(newPartsLayoutName);
            if (newPartsSubScene == null || !newPartsSubScene.IsLoaded)
            {
                LayoutEditorCore.MsgReporter.ReportError(
                    "",
                    LECoreStringResMgr.Get("ERROR_PARTS_NOT_LOADED", newPartsLayoutName));
                return null;
            }

            Ensure.Operation.True(newPartsSubScene.IsLoaded);

            // 新規部品ペインを登録する。
            var targetSubScene = partsPane.OwnerSubScene;
            SubSceneManipulator targetSubSceneMnp = new SubSceneManipulator();
            targetSubSceneMnp.BindTarget(targetSubScene);
            IPane newPartsPane = targetSubSceneMnp.AddPartsPane("TempParts", newPartsLayoutName, newPartsSubScene.SubScene);

            // 上書きプロパティのコピー
            var partsLayout = (newPartsPane.IPartsLayout as PartsLayout);
            if (partsLayout != null)
            {
                partsLayout.SetEnabledProperties(partsPane.IPartsLayout.PartsPropaerties, false);
            }

            // 階層
            if (partsPane.Parent != null)
            {
                HierarchyManipulator.SetHierarchyToPanePair(
                    partsPane.Parent as IPane, newPartsPane);
                HierarchyManipulator.ChangeChildOrder(
                    partsPane.Parent as IPane, newPartsPane, Array.IndexOf(partsPane.Parent.Children, partsPane));
            }

            // グループ
            if (partsPane.OwnerSubScene != null)
            {
                GroupMgrManipulator groupMgrMnp = new GroupMgrManipulator();
                groupMgrMnp.BindTarget(partsPane.OwnerSubScene.ILEGroupMgr);
                groupMgrMnp.BeginEditGroup();
                var groups = partsPane.OwnerSubScene.ILEGroupMgr.GroupSet.Where((grp) => grp.ContainsPaneName(partsPane.PaneName));
                foreach (var group in groups)
                {
                    int currentIndex = Array.IndexOf(group.Member, partsPane);
                    groupMgrMnp.AddMember(group, newPartsPane);
                    groupMgrMnp.ChangeChildOrder(group, newPartsPane, currentIndex);
                }

                groupMgrMnp.EndEditGroup();
            }

            // アニメーションなど情報をコピーします。
            PaneParamaterCopyOption option = new PaneParamaterCopyOption();
            option.DisableOption(ParamaterKind.All);
            option.EnableOption(
                ParamaterKind.PaneBasicFlags |
                ParamaterKind.SRTAll |
                ParamaterKind.Transparency |
                ParamaterKind.UserDataAll |
                ParamaterKind.Animation_All);
            PaneParamaterPaster.PasteParamaters(option, newPartsPane, partsPane);

            // もとのペイン名に戻す
            {
                // 古い部品ペインを削除する。
                var selectedPanes = targetSubScene.ISelectedSet.IPaneArray;
                targetSubSceneMnp.ResetSelectedSet();
                targetSubSceneMnp.SelectPanesByPaneRef(partsPane);
                targetSubSceneMnp.DeleteSelectedPanes();

                PaneManipulator paneMnp = new PaneManipulator();
                paneMnp.BindTarget(newPartsPane);
                paneMnp.PaneName = paneName;

                // 選択の復元
                selectedPanes.ToList().ForEach(
                    (selectedPane) => targetSubSceneMnp.SelectPanesByPaneName(selectedPane.PaneName));

                // 名前と階層構造が確定したのちにリロードしたパーツペインのキャプチャテクスチャ参照情報を更新します。
                SubSceneHelper.ResolvePartsLayoutCaptureTextureReference(newPartsPane);
            }

            return newPartsPane;
        }
    }
}
