﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Linq;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Util;
    using LECore.Structures;
    using LECore.Structures.Core;
    using LECore.Save_Load;

    using FileFmt = LECore.Structures.SerializableObject.Lyt;
    using LECore.Structures.Core.Command;

    /// <summary>
    /// Paneに対する、値の”設定”APIを外部に公開するためのクラスです。
    /// 本クラスが Undoコマンドの生成に関する責任を負うこととします。
    ///
    /// 下位のPaneクラスはUndoのことを気にしません。
    /// クライアントコードは、値を参照するだけであれば、本クラスを意識する必要が無く、
    /// ただ IPane の メソッドを使用すればいいはずです。
    ///
    /// 本クラスでは、ペインから、アトリビュートを取得し、アトリビュートから、
    /// サブアトリビュートを取り出し、そのアトリビュートに対して、操作を行います。
    /// </summary>
    public class PaneManipulator : BaseManipulator
    {
        #region フィールド

        Pane _targetPane = null;
        AnmAttributeManipulator _anmAttrMnp = null;
        bool _transAffectsChildren = true;


        #endregion フィールド

        #region プロパティ

        /// <summary>
        /// ペインの参照
        /// </summary>
        public IPane IPane
        {
            get { return _targetPane; }
        }

        /// <summary>
        /// 子供に影響を与えないか判定します。
        /// </summary>
        public bool TransAffectsChildren
        {
            get { return _transAffectsChildren; }
            set { _transAffectsChildren = value; }
        }

        #endregion プロパティ

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="targetPane"></param>
        public PaneManipulator()
        {
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetPane"></param>
        public void BindTarget( IPane targetPane )
        {
            _targetPane = targetPane as Pane;
            Debug.Assert( _targetPane != null );

            if( _anmAttrMnp == null )
            {
                _anmAttrMnp = new AnmAttributeManipulator();
            }

            // アトリビュート操作クラスの初期化を行います。
            _anmAttrMnp.BindTarget( _targetPane.IAttributeRef );
        }

        #region 基本的なパラメータの設定

        /// <summary>
        /// ペインのアトリビュートを変更します。
        ///
        /// 変更するアトリビュートに対して、
        /// 適切な型のパラメータを渡す必要があります。
        /// </summary>
        /// <param name="attribute"></param>
        /// <param name="newValue"></param>
        void ModifyAttr_( AnmAttribute attribute, object newValue)
        {
            Debug.Assert(_targetPane != null);

            ISubScene subScene = _targetPane.OwnerSubScene;
            bool modifyBase = subScene.IsBaseValueEditable();
            ManipulatorUtility.ModifyAttrWithCreatingCmd(attribute, newValue, modifyBase);
        }

        /// <summary>
        /// ペインのプロパティを変更します。
        ///
        /// 変更するプロパティに対して、
        /// 適切な型のパラメータを渡す必要があります。
        /// </summary>
        /// <param name="pane"></param>
        /// <param name="newValue"></param>
        /// <param name="getter"></param>
        /// <param name="setter"></param>
        private void ModifyPaneProp_( Pane pane, object newValue, PanePropertyAccessor.Getter getter, PanePropertyAccessor.Setter setter)
        {
            Debug.Assert( _targetPane != null );
            PanePropertyAccessor accessor = new PanePropertyAccessor( pane, getter, setter);
            ManipulatorUtility.ModifyPropWithCreatingCmd( accessor, newValue );
        }

        /// <summary>
        /// 子供を固定したまま、親を操作するため、
        /// 親に与えたSRT変換の逆変換を、子供に適用します。
        ///
        /// MEMO：
        /// 現状では、Tの変更時のみ、本処理を行うことにします。
        /// SRが複合的に使用された、親空間での、
        /// 逆変換をSRTから生成できないためです。（Sのみ、Rのみなら可能）
        /// </summary>
        void AdjustChildrenToFixT_( FVec3 dT )
        {
            dT.Scale( -1.0f );

            Matrix m = _targetPane.LocalMtx;
            m.Invert();

            PointF childDT = dT.FVec2.AsPointF;
            childDT = MathUtil.MtxTransformVec( m, childDT );

            PaneManipulator childPaneMnp = new PaneManipulator();
            foreach( IPane pane in _targetPane.Children )
            {
                childPaneMnp.BindTarget( pane );
                childPaneMnp.Trans = pane.Trans + new FVec3( childDT );
            }
        }

        public FVec3 Size
        {
            set
            {
                Debug.Assert(_targetPane != null);

                if (_transAffectsChildren)
                {
                    if (_targetPane.Size != value)
                    {
                        ModifyAttr_(_targetPane.PaneAttrRef.SizeAttr, new FVec2(value.X, value.Y));
                    }
                }
                else
                {
                    ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
                    subScene.BeginMassiveModify();
                    {
                        var preWorldT = new FVec3(_targetPane.XInWorld, _targetPane.YInWorld, _targetPane.ZInWorld4x4);
                        if (_targetPane.Size != value)
                        {
                            ModifyAttr_(_targetPane.PaneAttrRef.SizeAttr, new FVec2(value.X, value.Y));
                        }

                        // 子階層に、サイズ変更によっておこった親の移動を打ち消す変更を適用します。
                        var afterWorldT = new FVec3(_targetPane.XInWorld, _targetPane.YInWorld, _targetPane.ZInWorld4x4);
                        FVec3 diffT = afterWorldT - preWorldT;
                        AdjustChildrenToFixT_(diffT);
                    }
                    subScene.EndMassiveModify();
                }
            }
        }

        public FVec3 RotAng
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.RotAng != value )
                {
                    ModifyAttr_( _targetPane.PaneAttrRef.RotAngAttr, value );
                }
            }
        }

        public FVec3 Scale
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( value.X != 0.0f && value.Y != 0.0f )
                {
                    if( _targetPane.Scale != value )
                    {
                        ModifyAttr_( _targetPane.PaneAttrRef.ScaleAttr,
                                     new FVec2( value.X, value.Y ) );
                    }
                }
            }
        }

        /// <summary>
        /// 平行移動値設定
        /// </summary>
        FVec3 _Trans
        {
            set
            {
                Debug.Assert( _targetPane != null );
                Debug.Assert( _targetPane.Trans != value );
                if( _targetPane.Trans != value )
                {
                    ModifyAttr_( _targetPane.PaneAttrRef.TransAttr, value );
                }
            }
        }

        /// <summary>
        /// 平行移動値設定：子供の位置調整込み
        /// </summary>
        FVec3 _TransAvoidAffectingChildren
        {
            set
            {
                Debug.Assert( _targetPane != null );
                Debug.Assert( !_transAffectsChildren );
                Debug.Assert( _targetPane.Trans != value );

                ISubScene subScene = LECore.LayoutEditorCore.Scene.CurrentISubScene;
                subScene.BeginMassiveModify();

                FVec3 diffT = value - _targetPane.Trans;
                this._Trans = value;
                AdjustChildrenToFixT_( diffT );

                subScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// 平行移動値設定（外部に公開されるプロパティ）
        /// </summary>
        public FVec3 Trans
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.Trans != value )
                {
                    if( _transAffectsChildren )
                    {
                        this._Trans = value;
                    }
                    else
                    {
                        this._TransAvoidAffectingChildren = value;
                    }
                }
            }
        }

        /// <summary>
        /// Undoコマンドの生成を行わずにTransを更新します。
        /// </summary>
        public FVec3 TransWithoutCmdMaking
        {
            set
            {
                Debug.Assert( _targetPane != null );
                _targetPane.PaneAttrRef.TransAttr.SetValue( value );
                _targetPane.StoreCurrentValueToBaseValue();
            }
        }

        public float Z
        {
            set
            {
                Debug.Assert( _targetPane != null );
                FVec3 oldVal = _targetPane.Trans;
                this.Trans = new FVec3( oldVal.X, oldVal.Y, value );
            }
        }

        public float Width
        {
            set
            {
                Debug.Assert(_targetPane != null);
                FVec3 oldVal = _targetPane.Size;
                this.Size = new FVec3(value, oldVal.Y, oldVal.Z);
            }
        }

        public float Height
        {
            set
            {
                Debug.Assert( _targetPane != null );
                FVec3 oldVal = _targetPane.Size;
                this.Size = new FVec3( oldVal.X, value, oldVal.Z );
            }
        }

        public float RotAngleZ
        {
            set
            {
                Debug.Assert( _targetPane != null );
                FVec3 oldVal = _targetPane.RotAng;
                this.RotAng = new FVec3( oldVal.X, oldVal.Y, value );
            }
        }

        public bool InfluenceChildrenTransparencyWithoutCmdMaking
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.InfluenceChildrenTransparency != value )
                {
                    _targetPane.InfluenceChildrenTransparency = value;
                }
            }
        }

        public byte Transparency
        {
            set
            {
                Debug.Assert( _targetPane != null );

                if( _targetPane.Transparency != value )
                {
                    ModifyAttr_( _targetPane.PaneAttrRef.TransparencyAttr, value );
                }
            }
        }

        public bool Visible
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.Visible != value )
                {
                    ModifyAttr_( _targetPane.PaneAttrRef.VisibleAttr, value );
                }
            }
        }

        public bool InfluenceChildrenTransparency
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.InfluenceChildrenTransparency != value )
                {
                    ModifyPaneProp_( _targetPane, value,
                                     delegate( Pane pane) {
                                         return pane.InfluenceChildrenTransparency; },
                                     delegate( Pane pane, object param) {
                                         pane.InfluenceChildrenTransparency = (bool)param; });
                }
            }
        }

        public bool LocationAdjust
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.LocationAdjust != value )
                {
                    ModifyPaneProp_( _targetPane, value,
                                     delegate( Pane pane) { return pane.LocationAdjust; },
                                     delegate( Pane pane, object param) {
                                         pane.LocationAdjust = (bool)param; });
                }
            }
        }

        public bool IsAlignmentHorizontalMarginEnabled
        {
            set
            {
                Debug.Assert(_targetPane != null);
                if (_targetPane.IsAlignmentHorizontalMarginEnabled != value)
                {
                    ModifyPaneProp_(_targetPane, value,
                                     delegate (Pane pane) { return pane.IsAlignmentHorizontalMarginEnabled; },
                                     delegate (Pane pane, object param)
                                     {
                                         pane.IsAlignmentHorizontalMarginEnabled = (bool)param;
                                     });
                }
            }
        }

        public bool IsAlignmentVerticalMarginEnabled
        {
            set
            {
                Debug.Assert(_targetPane != null);
                if (_targetPane.IsAlignmentVerticalMarginEnabled != value)
                {
                    ModifyPaneProp_(_targetPane, value,
                                     delegate (Pane pane) { return pane.IsAlignmentVerticalMarginEnabled; },
                                     delegate (Pane pane, object param)
                                     {
                                         pane.IsAlignmentVerticalMarginEnabled = (bool)param;
                                     });
                }
            }
        }

        public float AlignmentHorizontalMargin
        {
            set
            {
                Debug.Assert(_targetPane != null);
                if (_targetPane.AlignmentHorizontalMargin != value)
                {
                    ModifyPaneProp_(_targetPane, value,
                                     delegate (Pane pane) { return pane.AlignmentHorizontalMargin; },
                                     delegate (Pane pane, object param)
                                     {
                                         pane.AlignmentHorizontalMargin = (float)param;
                                     });
                }
            }
        }

        public float AlignmentVerticalMargin
        {
            set
            {
                Debug.Assert(_targetPane != null);
                if (_targetPane.AlignmentVerticalMargin != value)
                {
                    ModifyPaneProp_(_targetPane, value,
                                     delegate (Pane pane) { return pane.AlignmentVerticalMargin; },
                                     delegate (Pane pane, object param)
                                     {
                                         pane.AlignmentVerticalMargin = (float)param;
                                     });
                }
            }
        }

        public bool IsAlignmentIgnore
        {
            set
            {
                Debug.Assert(_targetPane != null);
                if (_targetPane.IsAlignmentIgnore != value)
                {
                    ModifyPaneProp_(_targetPane, value,
                                     delegate (Pane pane) { return pane.IsAlignmentIgnore; },
                                     delegate (Pane pane, object param)
                                     {
                                         pane.IsAlignmentIgnore = (bool)param;
                                     });
                }
            }
        }

        public PartsMagnifyInfluence PartsMagnifyInfluence
        {
            set
            {
                Debug.Assert(_targetPane != null);
                if (_targetPane.PartsMagnifyInfluence != value)
                {
                    ModifyPaneProp_(_targetPane, value,
                                     delegate(Pane pane)
                                     {
                                         return pane.PartsMagnifyInfluence;
                                     },
                                     delegate(Pane pane, object param)
                                     {
                                         pane.PartsMagnifyInfluence = (PartsMagnifyInfluence)param;
                                     });
                }
            }
        }

        public HorizontalLocation BasePosTypeH
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.BasePosTypeH != value )
                {
                    ModifyPaneProp_( _targetPane, value,
                                     delegate( Pane pane) { return pane.BasePosTypeH; },
                                     delegate( Pane pane, object param) {
                                         pane.BasePosTypeH = (HorizontalLocation)param; });
                }
            }
        }

        public VerticalLocation BasePosTypeV
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.BasePosTypeV != value )
                {
                    ModifyPaneProp_( _targetPane, value,
                                     delegate( Pane pane) { return pane.BasePosTypeV; },
                                     delegate( Pane pane, object param) {
                                         pane.BasePosTypeV = (VerticalLocation)param; });
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public HorizontalLocation ParentBasePosTypeH
        {
            set
            {
                Debug.Assert(_targetPane != null);
                if (_targetPane.ParentBasePosTypeH != value)
                {
                    ModifyPaneProp_(_targetPane, value,
                                     delegate(Pane pane) { return pane.ParentBasePosTypeH; },
                                     delegate(Pane pane, object param)
                                     {
                                         pane.ParentBasePosTypeH = (HorizontalLocation)param;
                                     });
                }
            }
        }

        /// <summary>
        ///
        /// </summary>
        public VerticalLocation ParentBasePosTypeV
        {
            set
            {
                Debug.Assert(_targetPane != null);
                if (_targetPane.ParentBasePosTypeV != value)
                {
                    ModifyPaneProp_(_targetPane, value,
                                     delegate(Pane pane) { return pane.ParentBasePosTypeV; },
                                     delegate(Pane pane, object param)
                                     {
                                         pane.ParentBasePosTypeV = (VerticalLocation)param;
                                     });
                }
            }
        }

        /// <summary>
        /// 編集禁止状態フラグを木構造に沿って、再帰的に設定します。
        ///
        /// MEMO:
        /// 木構造に沿っての値の更新を ペインクラス内部で行わないのは、
        /// Undo/Redoコマンド生成を単純化する狙いがあるためです。
        /// </summary>
        static void SetPaneLockedRecursive_( Pane pane, bool boolValue )
        {
            if( pane.Locked != boolValue )
            {
                pane.Locked = boolValue;
                // TODO:Undo
            }
            foreach( Pane childPane in pane.Children )
            {
                SetPaneLockedRecursive_( childPane, boolValue );
            }
        }

        /// <summary>
        /// 編集禁止状態フラグを設定します。
        /// </summary>
        public bool Locked
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.Locked != value )
                {
                    _targetPane.Locked = value;
                    // TODO:Undo
                }
                // SetPaneLockedRecursive_( _targetPane, value );
            }
        }

        /// <summary>
        /// 隠された状態フラグを木構造に沿って、再帰的に設定します。
        /// </summary>
        static void SetPaneHiddenRecursive_( Pane pane, bool boolValue )
        {
            if( pane.Hidden != boolValue )
            {
                pane.Hidden = boolValue;
                // TODO:Undo
            }
            foreach( Pane childPane in pane.Children )
            {
                SetPaneHiddenRecursive_( childPane, boolValue );
            }
        }

        /// <summary>
        /// 隠された状態フラグを設定します。
        /// </summary>
        public bool Hidden
        {
            set
            {
                Debug.Assert( _targetPane != null );
                if( _targetPane.Hidden != value )
                {
                    _targetPane.Hidden = value;
                    // TODO:Undo
                }
                // SetPaneHiddenRecursive_( _targetPane, value );
            }
        }

        /// <summary>
        /// ノード圧縮禁止か。
        /// </summary>
        public bool AvoidPaneTreeCompression
        {
            set
            {
                ModifyPaneProp_(_targetPane, value,
                    delegate(Pane pane) { return pane.AvoidPaneTreeCompression; },
                    delegate(Pane pane, object param) { pane.AvoidPaneTreeCompression = (bool)param; });
            }
        }

        /// <summary>
        /// 拡張ユーザー情報のアニメーションを有効にするか。
        /// </summary>
        public bool ExtUserDataAnimEnabled
        {
            set
            {
                ModifyPaneProp_(_targetPane, value,
                    delegate(Pane pane) { return pane.ExtUserDataAnimEnabled; },
                    delegate(Pane pane, object param) { pane.ExtUserDataAnimEnabled = (bool)param; });
            }
        }

        #endregion 基本的なパラメータの設定

        #region 文字列情報
        /// <summary>
        /// ペイン名
        /// </summary>
        public string PaneName
        {
            set
            {
                Debug.Assert( _targetPane != null );

                _targetPane.OwnerSubScene.BeginMassiveModify();
                //ModifyPaneProp_( _targetPane, "PaneName", value);

                // 各種ペインの名前変更の追従
                SubSceneHelper.UpdateTexMapReferencedCaptureTexture(_targetPane.OwnerSubScene, _targetPane, _targetPane.PaneName, value);


                // 実際の名称変更
                ModifyPaneProp_( _targetPane, value,
                                 delegate( Pane pane) { return pane.PaneName; },
                                 delegate( Pane pane, object param) { pane.PaneName = (string)param; });

                // 名前を変更した結果、リンク切れのキャプチャテクスチャのソースになれないかチェック
                PaneHelper.ReconnectInvalidCaptureTexture(_targetPane);

                _targetPane.OwnerSubScene.EndMassiveModify();
            }
        }

        /// <summary>
        /// コメント
        /// </summary>
        public string UserCommentString
        {
            set
            {
                Debug.Assert( _targetPane != null );
                //ModifyPaneProp_( _targetPane, "UserCommentString", value);
                ModifyPaneProp_( _targetPane, value,
                                 delegate( Pane pane) { return pane.UserCommentString; },
                                 delegate( Pane pane, object param) { pane.UserCommentString = (string)param; });
            }
        }

        /// <summary>
        /// ユーザデータ
        /// </summary>
        public string UserData
        {
            set
            {
                Debug.Assert( _targetPane != null );
                //ModifyPaneProp_( _targetPane, "UserData", value);
                ModifyPaneProp_( _targetPane, value,
                                 delegate( Pane pane) { return pane.UserData; },
                                 delegate( Pane pane, object param) { pane.UserData = (string)param; });
            }
        }
        #endregion 文字列情報

        #region ユーザ情報



        #endregion ユーザ情報

        #region その他の操作

        /// <summary>
        /// 現在の値、時間でキーフレームを生成します。
        /// </summary>
        public void MakeKeyFrameByCurrentParams()
        {
            _anmAttrMnp.MakeKeyFrameByCurrentParams();
        }

        /// <summary>
        /// ワールド系ベクタでオフセットを加算します。
        /// ( 正確には、スクリーン座標系？ )
        /// </summary>
        /// <param name="vOffset"></param>
        public void TranslateInWorld( FVec3 vOffsetWld )
        {
            if( vOffsetWld == FVec3.Empty )
            {
                return;
            }

            // 座標系変換のための対処
            // vOffsetWld.Y = vOffsetWld.Y * PaneHelper.YUpValueInClientCoordinate;
            // vOffsetWld.Y = vOffsetWld.Y * -1.0f; // 必要！ -> 不要　// ParentMtxの変更により不要になった。

            // 親ペインローカル系のベクトルに変換
            Matrix mtxInvPane = _targetPane.ParentMtx;
            FVec3 vOffset = vOffsetWld;
            if( mtxInvPane.IsInvertible )
            {
                mtxInvPane.Invert();
                vOffset = MathUtil.MtxTransformVec( mtxInvPane, vOffset );
            }

            // オフセットとして加算
            FVec3 newTrans = _targetPane.Trans + vOffset;
            this.Trans = newTrans;
        }
        #endregion その他の操作

        #region 沢山のパラメータの一斉コピー
        public void Copy( IPane srcPane, PaneParamaterCopyOption option )
        {
            // 変更前の状態を複製します。
            IPane[] paneSet = { _targetPane };
            IPane[] beforeCopy = PaneSetDupulicator.ClonePaneSet( paneSet );

            // コピー処理のアンドゥで拡張ユーザー情報が正しく復元できるように復元ポイントを作成
            CreateUserDataRestorePoint(false);

            // コピーします。
            PaneParamaterPaster.PasteParamaters( option, _targetPane, srcPane );

            // 変更後の状態を複製します。
            IPane[] afterCopy = PaneSetDupulicator.ClonePaneSet( paneSet );

            // コマンドをつくり登録します。
            _CommandFactory.MakePaneCopyCmd( _targetPane, beforeCopy[0], afterCopy[0] );

            // コピー処理のリドゥで拡張ユーザー情報が正しく復元できるように復元ポイントを作成
            CreateUserDataRestorePoint(true);
        }

        public void CopyMaterial(
            IPane srcPane,
            int srcMaterialIdx,
            int dstMaterialIdx,
            PaneParamaterCopyOption option)
        {
            // 変更前の状態を複製します。
            IPane[] paneSet = { _targetPane };
            IPane[] beforeCopy = PaneSetDupulicator.ClonePaneSet( paneSet );

            // コピーします。
            // マテリアル
            Debug.Assert( _targetPane.IMaterial.Length > dstMaterialIdx && srcPane.IMaterial.Length > srcMaterialIdx );
            MaterialParamaterPaster.CopyMaterialParamater(
                                                          option,
                                                          _targetPane.IMaterial[dstMaterialIdx],
                                                          srcPane.IMaterial[srcMaterialIdx]);

            // 詳細なマテリアル設定を行うフラグのコピー
            LEWindow dstWindow = _targetPane.ILEWindow as LEWindow;
            LEWindow srcWindow = srcPane.ILEWindow as LEWindow;

            LEWindowPartsID srcId = srcWindow.IntToPartsID( srcMaterialIdx);
            LEWindowPartsID dstId = srcWindow.IntToPartsID( dstMaterialIdx);
            bool detailed = srcWindow.GetUseDetailedMaterialFlag( srcId);
            dstWindow.SetUseDetailedMaterialFlag( dstId, detailed);

            // 詳細マテリアル
            IRevHWMaterial[] dstRevMtlSet = PaneHelper.GetRevHWMatFromPane( _targetPane );
            IRevHWMaterial[] srcRevMtlSet = PaneHelper.GetRevHWMatFromPane( srcPane );

            Debug.Assert( dstRevMtlSet.Length > dstMaterialIdx && srcRevMtlSet.Length > srcMaterialIdx );
            MaterialParamaterPaster.CopyMaterialDetailParamaters(
                                                                 option,
                                                                 dstRevMtlSet[dstMaterialIdx],
                                                                 srcRevMtlSet[srcMaterialIdx],
                                                                 true);

            // 変更後の状態を複製します。
            IPane[] afterCopy = PaneSetDupulicator.ClonePaneSet( paneSet );

            // コマンドをつくり登録します。
            _CommandFactory.MakePaneCopyCmd( _targetPane, beforeCopy[0], afterCopy[0] );
        }

        /// <summary>
        /// ペインのコピーをアンドゥ・リドゥしたときに参照リストを復元するための復元ポイントを作成します。
        /// </summary>
        /// <param name="redo">リドゥしたときに復元するか</param>
        public void CreateUserDataRestorePoint(bool redo)
        {
            CreateUserDataRestorePoint(_targetPane, redo);
        }

        /// <summary>
        /// ペインのコピーをアンドゥ・リドゥしたときに参照リストを復元するための復元ポイントを作成します。
        /// </summary>
        /// <param name="pane">対象ペイン</param>
        /// <param name="redo">リドゥしたときに復元するか</param>
        private void CreateUserDataRestorePoint(IPane pane, bool redo)
        {
            if (pane != null)
            {
                // 現在の参照リストをコピー
                IUserDataElement[] userDataElementSet = pane.IUserDataHolder.UserDataElementSet;
                UserDataElement[] elements = new UserDataElement[pane.IUserDataHolder.UserDataElementSet.Length];
                userDataElementSet.CopyTo(elements, 0);

                // 復元ポイントを作成
                if (!redo)
                {
                    _CommandFactory.MakeUserDataRestoreOnUndoCmd((UserDataHolder)pane.IUserDataHolder, elements);
                }
                else
                {
                    _CommandFactory.MakeUserDataRestoreOnRedoCmd((UserDataHolder)pane.IUserDataHolder, elements);
                }

                // 部品ペインの上書きデータについて処理
                IPartsLayout partsLayout = pane.IPartsLayout;
                if (partsLayout != null)
                {
                    ISubScene subScene = partsLayout.PartsSubScene;
                    if (subScene != null)
                    {
                        foreach (IPane childPane in subScene.IPaneArray)
                        {
                            CreateUserDataRestorePoint(childPane, redo);
                        }
                    }
                }
            }
        }

        #endregion 沢山のパラメータの一斉コピー
    }

    /// <summary>
    /// 拡張ユーザデータ操作クラス
    /// </summary>
    public class UserDataHolderManipulator : BaseManipulator
    {
        UserDataHolder _targetHolder = null;

        public IUserDataHolder IUserDataHolder { get { return _targetHolder; } }

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserDataHolderManipulator()
        {
        }

        /// <summary>
        ///
        /// </summary>
        public void BindTarget(IUserDataHolder holder)
        {
            _targetHolder = holder as UserDataHolder;
        }

        /// <summary>
        /// ユーザ情報項目を追加します。
        /// </summary>
        public void AddUserDataElement(string name, UserDataKind kind, object value)
        {
            if (UserDataElementHelper.CheckParamatersValid(name, kind, value))
            {
                Pane ownerNode = _targetHolder.OwnerNode as Pane;
                bool overwrite = _targetHolder.Overwrite;

                UserDataElement data = UserDataElementHelper.FindByName(name, _targetHolder.UserDataElementSet) as UserDataElement;

                SubScene subScene = Scene.CurrentSubScene;

                if (data != null && overwrite)
                {
                    // 差し替え前の項目番号を取得
                    int elementIndex = Array.IndexOf(_targetHolder.UserDataElementSet, data);
                    int attributeIndex = _targetHolder.SubAttrArray.IndexOf(data.IAnmAttribute);

                    // ユーザー情報項目を差し替え
                    _targetHolder.RemoveUserDataElement(name);
                    _targetHolder.InsertUserDataElement(name, kind, overwrite, value, elementIndex, attributeIndex);

                    UserDataElement element = (UserDataElement)_targetHolder.UserDataElementSet[elementIndex];

                    // 差し替えコマンドを作成
                    CommandFactory attributeCommandFactory = subScene.CommandFactory;
                    attributeCommandFactory.MakeUserDataElementReplaceCmd(_targetHolder, data, element, elementIndex, attributeIndex );
                }
                else if (data == null)
                {
                    // ユーザ情報項目を追加
                    _targetHolder.AddUserDataElement(name, kind, overwrite, value);

                    data = (UserDataElement)_targetHolder.UserDataElementSet[_targetHolder.UserDataElementSet.Length - 1];

                    // 追加コマンドを作成
                    CommandFactory attributeCommandFactory = subScene.CommandFactory;
                    attributeCommandFactory.MakeUserDataElementAddCmd(_targetHolder, data);
                }
            }
        }

        /// <summary>
        /// ユーザ情報項目を削除します。
        /// </summary>
        public void RemoveUserDataElement(string name)
        {
            Pane ownerNode = _targetHolder.OwnerNode as Pane;
            bool overwrite = _targetHolder.Overwrite;

            UserDataElement data = UserDataElementHelper.FindByName(name, _targetHolder.UserDataElementSet) as UserDataElement;

            if (data != null)
            {
                if (overwrite)
                {
                    // 上書きのときはNoneデータに差し替え
                    AddUserDataElement(name, UserDataKind.None, null);
                }
                else
                {
                    // 削除前の項目番号を取得
                    int elementIndex = Array.IndexOf(_targetHolder.UserDataElementSet, data);
                    int attributeIndex = _targetHolder.SubAttrArray.IndexOf(data.IAnmAttribute);

                    // ユーザ情報項目を削除
                    _targetHolder.RemoveUserDataElement(name);

                    // 削除コマンドを作成
                    CommandFactory attributeCommandFactory = Scene.CurrentSubScene.CommandFactory;
                    attributeCommandFactory.MakeUserDataElementRemoveCmd(_targetHolder, data, elementIndex, attributeIndex);
                }
            }
        }

        /// <summary>
        /// ユーザ情報項目をリセットします。
        /// </summary>
        public void ResetUserDataElement(string name)
        {
            // TODO: アニメーション対応
            UserDataElement data = UserDataElementHelper.FindByName(name, _targetHolder.UserDataElementSet) as UserDataElement;

            SubScene subScene = Scene.CurrentSubScene;

            if (data != null && data.Overwrite)
            {
                // 上書き前のデータを取得
                UserDataElement originalData = _targetHolder.FindOriginalData(name) as UserDataElement;

                if (originalData == null)
                {
                    // 上書き前のデータがなければ削除
                    // 削除前の項目番号を取得
                    int elementIndex = Array.IndexOf(_targetHolder.UserDataElementSet, data);
                    int attributeIndex = _targetHolder.SubAttrArray.IndexOf(data.IAnmAttribute);

                    // ユーザ情報項目を削除
                    _targetHolder.RemoveUserDataElement(name);

                    // 削除コマンドを作成
                    CommandFactory attributeCommandFactory = Scene.CurrentSubScene.CommandFactory;
                    attributeCommandFactory.MakeUserDataElementRemoveCmd(_targetHolder, data, elementIndex, attributeIndex);
                }
                else
                {
                    // 差し替え前の項目番号を取得
                    int elementIndex = Array.IndexOf(_targetHolder.UserDataElementSet, data);
                    int attributeIndex = _targetHolder.SubAttrArray.IndexOf(data.IAnmAttribute);

                    // ユーザー情報項目を差し替え
                    _targetHolder.RemoveUserDataElement(name);
                    _targetHolder.InsertUserDataElement(name, originalData.UserDataKind, false, originalData.Value, elementIndex, attributeIndex);

                    UserDataElement element = (UserDataElement)_targetHolder.UserDataElementSet[elementIndex];

                    // 差し替えコマンドを作成
                    CommandFactory attributeCommandFactory = subScene.CommandFactory;
                    attributeCommandFactory.MakeUserDataElementReplaceCmd(_targetHolder, data, element, elementIndex, attributeIndex );
                }
            }
        }

        /// <summary>
        /// 全てのアニメーション形式のユーザ情報項目を削除します。
        /// </summary>
        public void RemoveAllAnimElement()
        {
            SubScene subScene = Scene.CurrentSubScene;

            subScene.BeginMassiveModify();

            IUserDataElement[] elementSet = _targetHolder.UserDataElementSet;

            foreach (IUserDataElement element in elementSet)
            {
                UserDataElement data = (UserDataElement)element;

                // アニメーションならユーザ情報項目を削除
                if (data.IAnmAttribute != null)
                {
                    // 削除前の項目番号を取得
                    int elementIndex = Array.IndexOf(_targetHolder.UserDataElementSet, data);
                    int attributeIndex = _targetHolder.SubAttrArray.IndexOf(data.IAnmAttribute);

                    // ユーザ情報項目を削除
                    _targetHolder.RemoveUserDataElement(data.Name);

                    // 削除コマンドを作成
                    CommandFactory attributeCommandFactory = subScene.CommandFactory;
                    attributeCommandFactory.MakeUserDataElementRemoveCmd(_targetHolder, data, elementIndex, attributeIndex);
                }
            }

            subScene.EndMassiveModify();
        }

        /// <summary>
        /// ユーザ情報項目の番号を変更します。
        /// </summary>
        public void ChangeUserDataIndex(string name, int newIndex)
        {
            UserDataElement data = UserDataElementHelper.FindByName(name, _targetHolder.UserDataElementSet) as UserDataElement;
            if (data != null)
            {
                int beforeElementIndex = Array.IndexOf(_targetHolder.UserDataElementSet, data);
                int afterElementIndex = newIndex;

                int beforeAttributeIndex = 0;
                int afterAttributeIndex = 0;

                // アトリビュート番号を調べる
                if (data.UserDataAttribute != null)
                {
                    beforeAttributeIndex = _targetHolder.SubAttrArray.IndexOf(data.UserDataAttribute);

                    IUserDataElement[] elementSet = _targetHolder.UserDataElementSet;

                    // 変更後のアトリビュート番号を計算
                    if (afterElementIndex >= beforeElementIndex)
                    {
                        for (int i = 0; i <= afterElementIndex; ++i)
                        {
                            if (elementSet[i].IAnmAttribute != null)
                            {
                                ++afterAttributeIndex;
                            }
                        }

                        --afterAttributeIndex;  // 自分の分のカウントを減らす
                    }
                    else
                    {
                        for (int i = 0; i < afterElementIndex; ++i)
                        {
                            if (elementSet[i].IAnmAttribute != null)
                            {
                                ++afterAttributeIndex;
                            }
                        }
                    }
                }

                _targetHolder.RemoveUserDataElement(data);
                _targetHolder.InsertUserDataElement(data, afterElementIndex, afterAttributeIndex);

                // 番号変更コマンドを作成
                CommandFactory attributeCommandFactory = Scene.CurrentSubScene.CommandFactory;
                attributeCommandFactory.MakeUserDataElementIndexChangeCmd(_targetHolder, data, beforeElementIndex, afterElementIndex, beforeAttributeIndex, afterAttributeIndex);
            }
        }

        /// <summary>
        /// ユーザ情報項目のパラメータを設定します。
        /// </summary>
        public void SetUserDataPamaters(string name, string newName, UserDataKind kind, object value)
        {
            if (UserDataElementHelper.CheckParamatersValid(newName, kind, value))
            {
                Pane ownerNode = _targetHolder.OwnerNode as Pane;
                bool overwrite = false;

                // 部品ペインの上書きデータかどうか判定
                if (ownerNode != null && ownerNode.OwnerSubScene.IPartsLayout != null)
                {
                    overwrite = true;
                }

                IUserDataElement[] userDataElementSet = _targetHolder.UserDataElementSet;
                UserDataElement data = UserDataElementHelper.FindByName(name, userDataElementSet) as UserDataElement;

                if (data != null)
                {
                    // アニメーション型への変更。もしくは、アニメーション型からの変更なら非サポートなので処理しません
                    //（ツールが正しく処理ができないため）。
                    if (UserDataElementHelper.IsInvalidModificationOfKind(data.UserDataKind, kind))
                    {
                        Debug.Assert(false);
                        return;
                    }

                    if (newName == name || UserDataElementHelper.FindByName(newName, userDataElementSet) == null)
                    {
                        // ユーザ情報項目の値をセット
                        bool beforeOverwrite = data.Overwrite;
                        UserDataKind beforeKind = data.UserDataKind;
                        object beforeVal = data.Value;
                        object beforeBaseVal = null;
                        object afterBaseVal = null;
                        if (data.IAnmAttribute is AnmAttribute)
                        {
                            ((AnmAttribute)data.IAnmAttribute).GetBaseValue(out beforeBaseVal);
                            afterBaseVal = ManipulatorUtility.CanModifyBase() ? value : beforeBaseVal;
                        }

                        _targetHolder.SetUserDataPamaters(name, newName, overwrite, kind, value, afterBaseVal);
                        object afterVal = data.Value;

                        // コマンドを作成
                        CommandFactory attributeCommandFactory = Scene.CurrentSubScene.CommandFactory;
                        attributeCommandFactory.MakeUserDataElementSetCmd(_targetHolder, data, name, newName, beforeOverwrite, overwrite, beforeKind, kind, beforeVal,afterVal, beforeBaseVal, afterBaseVal);
                    }
                }
            }
        }
    }

    /// <summary>
    /// ペインのプロパティを変更するクラス
    /// </summary>
    internal class PanePropertyAccessor : IValueAccessor
    {
        public delegate object Getter( LECore.Structures.Pane pane);
        public delegate void Setter( LECore.Structures.Pane pane, object value);

        private Pane    _pane = null;
        private Getter _getter = null;
        private Setter _setter = null;

        ///
        public PanePropertyAccessor( LECore.Structures.Pane pane, Getter getter, Setter setter)
        {
            _pane = pane;
            _getter = getter;
            _setter = setter;
        }

        ///
        void IValueAccessor.GetValue( out object value)
        {
            value = _getter( _pane);
        }

        ///
        void IValueAccessor.SetValue( object value)
        {
            _setter( _pane, value);
        }
    }
}
