﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;
using System.Collections;
using System.Drawing;
using System.Drawing.Drawing2D;

namespace LECore.Manipulator
{
    using Structures;
    using Structures.Core;
    using Structures.Core.Command;
    using Util;

    /// <summary>
    /// マウス操作によって、階層構造を構築するクラスです。
    /// </summary>
    public class HierarchyManipulator :
        BaseManipulator,
        IDrawable
    {
        // 動作モード
        enum Mode
        {
            None,            // 待機状態
            MakeHierarchy,   // 階層作成中
            BreakeHierarchy  // 階層削除中
        }


        // 親ノード
        Pane                 _parentPane              = null;
        // 子ノード候補
        Pane                 _childCandidatePane      = null;

        // 開始点
        Point                _beginPos = Point.Empty;
        // 現在の位置
        Point                _currentPos = Point.Empty;
        // 動作モード
        Mode                 _mode = Mode.None;

        /// <summary>
        /// 構築中かどうか判定します。
        /// </summary>
        public bool IsBuilding
        {
            get{ return _mode == Mode.MakeHierarchy;}
        }

        /// <summary>
        /// 構築中かどうか判定します。
        /// </summary>
        public bool IsBreaking
        {
            get{ return _mode == Mode.BreakeHierarchy;}
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public HierarchyManipulator(){}

        /// <summary>
        /// シーンからペインを選択します。
        /// </summary>
        /// <param name="posInScene"></param>
        /// <returns></returns>
        Pane SelectPaneFromScene_( Point posInScene )
        {
            return SubSceneHelper.GetPanesByPoint( Scene.CurrentSubScene, new FVec2( posInScene ) ) as Pane;
        }

        /// <summary>
        ///
        /// </summary>
        void ResetState_()
        {
            // リセットします。
            _parentPane          = null;
            _childCandidatePane  = null;
            _mode = Mode.None;
        }

        /// <summary>
        /// 階層作成を開始します。
        /// </summary>
        /// <param name="parentNode"></param>
        public bool BeginMakingHierarchy( Point posInScene )
        {
            Pane selPane = SelectPaneFromScene_( posInScene );
            if( selPane != null )
            {
                _beginPos    = posInScene;
                _currentPos  = posInScene;

                _parentPane  = selPane;

                _mode = Mode.MakeHierarchy;
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// 生成を完了し、実際に親子構造を作成します。
        /// </summary>
        public void EndMakingHierarchy()
        {
            Debug.Assert( _parentPane != null );
            Debug.Assert( _mode == Mode.MakeHierarchy );

            if( _childCandidatePane != null )
            {
                SetHierarchyToPanePair( _parentPane, _childCandidatePane );
            }

            // リセットします。
            ResetState_();
        }

         /// <summary>
         /// 2つのペインから、親子階層を設定します。
         /// </summary>
         /// <param name="parent"></param>
         /// <param name="child"></param>
         /// <returns></returns>
        static public void SetHierarchyToPanePair( IPane parent, IPane child )
        {
            Pane parentPane = parent as Pane;
            Pane childPane = child  as Pane;

            Debug.Assert( parentPane != null && childPane != null );
            Debug.Assert( childPane.IsValidForChildOf( parentPane ) );

            // Undoコマンドの登録
            _CommandFactory.MakeMakeHierarchyCmd(
                Scene.CurrentSubScene,
                parentPane,
                childPane );

            // 階層設定
            parentPane.AddChildPane( childPane );
        }

        /// <summary>
        ///
        /// </summary>
        static public int ChangeChildOrder( IPane parent, IPane child, int newIndex )
        {
            Debug.Assert( parent != null && child != null );
            Debug.Assert( newIndex > -1 );

            Pane parentPane = parent as Pane;
            Pane childPane = child  as Pane;

            _CommandFactory.MakeEditHierarchyOrderCmd( Scene.CurrentSubScene, parentPane, childPane, newIndex );

            return parentPane.ChangeChildOrder( childPane, newIndex );

        }


        /// <summary>
        /// 階層構造をリセットします。
        /// 内部では、RootNodeに対する階層設定を行っています。
        /// </summary>
        /// <param name="child"></param>
        static public void ResetHierarchy( IPane child )
        {
            // 自分の所属するルートペインを求めます。
            IPane rootPane = child;
            while( rootPane.Parent != null )
            {
                rootPane = rootPane.Parent as IPane;
            }
            // シーンに所属しているペインなら必ずrootPaneにたどり着くはず。
            Debug.Assert( rootPane != child );

            SetHierarchyToPanePair( rootPane, child );
        }


        /// <summary>
        /// 階層解除を開始します。
        /// </summary>
        /// <param name="parentNode"></param>
        public bool BeginBreakingHierarchy( Point posInScene )
        {
            Pane selPane = SelectPaneFromScene_( posInScene );
            if( selPane != null )
            {
                _beginPos    = posInScene;
                _currentPos  = posInScene;

                _parentPane  = selPane;

                _mode = Mode.BreakeHierarchy;
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// 解除を完了し、実際に親子構造を解除します。
        ///
        /// 階層解除は、
        /// RootNodeに対する子ノード登録と考えます。
        /// </summary>
        public void EndBreakingHierarchy()
        {
            Debug.Assert( _parentPane != null );
            Debug.Assert( _mode == Mode.BreakeHierarchy );

            if( _childCandidatePane != null )
            {
                ResetHierarchy( _childCandidatePane );
            }

            ResetState_();
        }

        /// <summary>
        /// 現在のマウス位置を設定します。
        /// </summary>
        /// <param name="posInScene"></param>
        public void SetCurrentMousePos( Point posInScene )
        {
            Debug.Assert( _parentPane != null );

            _currentPos   = posInScene;
            Pane selPane  = SelectPaneFromScene_( posInScene );

            // 子供ノード候補を一旦リセットします。
            _childCandidatePane = null;

            if( selPane != null )
            {
                // 作成モードである
                if( _mode == Mode.MakeHierarchy )
                {
                    if( selPane.IsValidForChildOf( _parentPane ) )
                    {
                        // 新規候補としてふさわしい。
                        _childCandidatePane = selPane;
                    }
                    else
                    {
                        // 新規候補としてふさわしくない。
                        _childCandidatePane = null;

                    }
                }
                // 削除モードである
                else
                {
                    if( selPane.IsChildrenOf( _parentPane ) )
                    {
                        // 新規候補としてふさわしい。
                        _childCandidatePane = selPane;
                    }
                    else
                    {
                        // 新規候補としてふさわしくない。
                        _childCandidatePane = null;

                    }
                }
            }
        }



        #region IDrawable メンバ
        void FillPaneRect_( IRenderer renderer, Pane pane )
        {
            // 境界線の内側を塗りつぶす
            float lineW = renderer.LineWidth;

            RectangleF bv = pane.BoundingVolumeInWorld;
            renderer.FillRectangle( bv.X + lineW,
                                    bv.Y + lineW,
                                    0,
                                    bv.Width - lineW,
                                    bv.Height - lineW );
        }

        /// <summary>
        /// ペインの中心位置を取得します。
        /// </summary>
        /// <param name="pane"></param>
        /// <returns></returns>
        PointF GetPaneCenter_( Pane pane )
        {
            PointF posPane = pane.BoundingVolumeInWorld.Location;
            return MathUtil.AddVec( posPane, new PointF( pane.Width / 2, pane.Height / 2 ) );
        }

        /// <summary>
        /// 描画します
        /// </summary>
        /// <param name="renderer"></param>
        public void Draw( IRenderer renderer, DrawableOption option )
        {
            if( _parentPane != null )
            {

                PointF posParent = GetPaneCenter_( _parentPane );
                PointF posChild  = new PointF( _currentPos.X, _currentPos.Y );

                Color oldColor = renderer.Color;

                renderer.Color = _childCandidatePane != null ? Color.Red : Color.Black ;
                renderer.Color = Color.FromArgb( 64, renderer.Color );

                renderer.DrawLine( posParent, posChild );


                FillPaneRect_( renderer, _parentPane );

                if( _childCandidatePane != null )
                {
                    FillPaneRect_( renderer, _childCandidatePane );
                }

                renderer.Color = oldColor;
            }
        }

        #endregion IDrawable メンバ

    }
}
