﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Save_Load;
    using LECore.Structures.Core.Command;
    using LECore.Structures.LECoreInterface;

    using CaptureTextureCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.ICaptureTexture>;

    /// <summary>
    /// CaptureTextureManipulator の概要の説明です。
    /// </summary>
    public class CaptureTextureManipulator : BaseManipulator
    {
        // 操作対象
        CaptureTexture  _target = null;

        /// <summary>
        /// Undoコマンド生成クラス。
        /// </summary>
        CaptureTextureCommandFactory _commandFactory;

        /// <summary>
        /// 変更コマンドを登録しつつ、パラメータを更新します。
        /// 任意のValueChangeFunctionでパラメータを更新してください。
        /// </summary>
        void ValueChangeWithCommandRegister_( CaptureTextureCommandFactory.InstanceModifyFunction valueChangeFunction )
        {
            _commandFactory.Modify( _target, valueChangeFunction );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CaptureTextureManipulator()
        {
            _commandFactory = new CaptureTextureCommandFactory
                (
                 delegate ( ICaptureTexture mementoSrc ){return new CaptureTextureMemento( mementoSrc, Scene.CurrentSubScene);}
                 );
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetCaptureTexture"></param>
        public void BindTarget( ICaptureTexture targetCaptureTexture)
        {
            _target = targetCaptureTexture as CaptureTexture;
            Debug.Assert( _target != null );
        }

        /// <summary>
        /// 操作対象
        /// </summary>
        public ICaptureTexture ICaptureTexture
        {
            get{ return _target;}
        }

        /// <summary>
        /// オーナーペインの設定
        /// </summary>
        public IPane OwnerPane
        {
            set{
                ValueChangeWithCommandRegister_( delegate()
                {
                    _target.OwnerPane = value;
                } );
            }
        }

        /// <summary>
        /// フレームバッファキャプチャの有効設定
        /// </summary>
        public bool FrameBufferCaptureEnabled
        {
            set{
                ValueChangeWithCommandRegister_( delegate()
                {
                    _target.FrameBufferCaptureEnabled = value;
                } );
            }
        }

        /// <summary>
        /// フレームバッファキャプチャの際に初回のみキャプチャするかどうか
        /// </summary>
        public bool CaptureOnlyFirstFrame
        {
            set{
                ValueChangeWithCommandRegister_( delegate()
                {
                    _target.CaptureOnlyFirstFrame = value;
                } );
            }
        }


        /// <summary>
        /// クリアカラーの変更
        /// </summary>
        public FloatColor ClearColor
        {
            set
            {
                ValueChangeWithCommandRegister_( delegate()
                {
                    _target.ClearColor = value;
                } );
            }
        }

        /// <summary>
        /// テクスチャフォーマットの変更
        /// </summary>
        public TexImagePixelFmt Format
        {
            set
            {
                ValueChangeWithCommandRegister_(delegate ()
                {
                    _target.Format = value;
                });
            }
        }

        /// <summary>
        /// テクスチャ解像度のスケール
        /// </summary>
        public float TextureScale
        {
            set
            {
                ValueChangeWithCommandRegister_(delegate ()
                {
                    _target.TextureScale = value;
                });
            }
        }

    }
}
