﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using LECore.Structures;
    using LECore.Save_Load;
    using LECore.Structures.Core.Command;
    using LECore.Structures.LECoreInterface;

    using RlytConverter = LECore.Save_Load.RlytConverter;
    using CaptureCommandFactory = LECore.Structures.Core.Command.MementoCommandFactory<LECore.Structures.ICapture>;

    /// <summary>
    /// TextBoxManipulator の概要の説明です。
    /// </summary>
    public class CaptureManipulator : BaseManipulator
    {
        /// <summary>
        ///
        /// </summary>
        static readonly RlytConverter _RlytConverter =
        new RlytConverter( string.Empty, LECore.LayoutEditorCore.MsgReporter );

        // 操作対象
        Capture     _targetCapture = null;

        /// <summary>
        /// Undoコマンド生成クラス。
        /// </summary>
        CaptureCommandFactory _commandFactory;

        /// <summary>
        /// 変更コマンドを登録しつつ、パラメータを更新します。
        /// 任意のValueChangeFunctionでパラメータを更新してください。
        /// </summary>
        void ValueChangeWithCommandRegister_( CaptureCommandFactory.InstanceModifyFunction valueChangeFunction )
        {
            _commandFactory.Modify( _targetCapture, valueChangeFunction );
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public CaptureManipulator()
        {
            _commandFactory = new CaptureCommandFactory
                (
                 delegate ( ICapture mementoSrc ){return new CaptureMemento( mementoSrc );}
                 );
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        /// <param name="targetCapture"></param>
        public void BindTarget( ICapture targetCapture)
        {
            _targetCapture = targetCapture as Capture;
            Debug.Assert( _targetCapture!= null );
        }

        /// <summary>
        /// 操作対象
        /// </summary>
        public ICapture ICapture
        {
            get{ return _targetCapture;}
        }

        /// <summary>
        /// フレームバッファキャプチャの有効設定
        /// </summary>
        public bool FrameBufferCaptureEnabled
        {
            set{
                ICaptureTexture tex = _targetCapture.OwnTexture;

                if (tex != null)
                {
                    CaptureTextureManipulator mnp = new CaptureTextureManipulator();
                    mnp.BindTarget(tex);

                    mnp.FrameBufferCaptureEnabled = value;
                }
            }
        }

        /// <summary>
        /// フレームバッファキャプチャの際に初回のみキャプチャするかどうか
        /// </summary>
        public bool CaptureOnlyFirstFrame
        {
            set{
                ICaptureTexture tex = _targetCapture.OwnTexture;

                if (tex != null)
                {
                    CaptureTextureManipulator mnp = new CaptureTextureManipulator();
                    mnp.BindTarget(tex);

                    mnp.CaptureOnlyFirstFrame = value;
                }
            }
        }


        /// <summary>
        /// クリアカラーの変更
        /// </summary>
        public FloatColor ClearColor
        {
            set
            {
                ICaptureTexture tex = _targetCapture.OwnTexture;

                if (tex != null)
                {
                    CaptureTextureManipulator mnp = new CaptureTextureManipulator();
                    mnp.BindTarget(tex);

                    mnp.ClearColor = value;
                }
            }
        }

        /// <summary>
        /// レンダーターゲットテクスチャのフォーマット
        /// </summary>
        public TexImagePixelFmt Format
        {
            set
            {
                ICaptureTexture tex = _targetCapture.OwnTexture;

                if (tex != null)
                {
                    CaptureTextureManipulator mnp = new CaptureTextureManipulator();
                    mnp.BindTarget(tex);

                    mnp.Format = value;
                }
            }
        }

        /// <summary>
        /// テクスチャの解像度のかけるスケール
        /// </summary>
        public float TextureScale
        {
            set
            {
                ICaptureTexture tex = _targetCapture.OwnTexture;
                if (tex != null)
                {
                    CaptureTextureManipulator mnp = new CaptureTextureManipulator();
                    mnp.BindTarget(tex);

                    mnp.TextureScale = value;
                }
            }
        }
    }
}
