﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Diagnostics;

namespace LECore.Manipulator
{
    using Structures.Core;
    using Structures.Core.Command;

    /// <summary>
    /// KeyFrameManipulator の概要の説明です。
    /// </summary>
    public class AnmKeyFrameManipulator :
        BaseManipulator
    {
        AnmKeyFrame           _target        = null;
        AnmCurveManipulator   _ownerCurveMnp = null;


        #region ------------ private ---------------
        /// <summary>
        /// パラメータが対象キーと完全に同一か？
        /// </summary>
        bool IsChangingTraget_( float val, float time )
        {
            return (  _target.ValueAsFloat    != val ||
                      _target.Time            != time );
        }

        #endregion ------------ private ---------------

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public AnmKeyFrameManipulator()
        {
        }

        /// <summary>
        /// 操作対象を設定します。
        /// </summary>
        public void BindTarget( IAnmKeyFrame target )
        {
            _target = target as AnmKeyFrame;
            Debug.Assert( _target != null );

            if( _ownerCurveMnp == null )
            {
                _ownerCurveMnp = new AnmCurveManipulator();
            }
        }

        /// <summary>
        /// 重複を確認してキーの値の変更(Undo/Redoが適用されます。)
        /// </summary>
        public void Modify( float val, float time )
        {
            Debug.Assert( _target != null );

            IAnmKeyFrame key = _target.OwnerAnmCurve.FindKeyExistAt( time );

            if( key == null || key == _target )
            {
                ModifyForce(val, time);
            }
        }

        /// <summary>
        /// キーの値の変更(Undo/Redoが適用されます。)
        /// </summary>
        public void ModifyForce(float val, float time)
        {
            Debug.Assert(_target != null);

            if (IsChangingTraget_(val, time))
            {
                AnmKeyFrame beforeKey = _target.Clone() as AnmKeyFrame;

                // 値の変更
                this.ValueAsFloatWithoutCommand = val;
                this.TimeWithoutCommand = time;

                AnmKeyFrame afterKey = _target.Clone() as AnmKeyFrame;

                // Undoコマンドの生成。
                _CommandFactory.MakeAnmKeyModCmd(_target, beforeKey, afterKey);
            }
        }

        /// <summary>
        /// キーの値、時間、スロープの変更(Undo/Redoが適用されます。)
        /// </summary>
        public void ModifyForce(float val, float time, float inSlope, float outSlope)
        {
            Debug.Assert(_target != null);

            if (IsChangingTraget_(val, time) || _target.InTangent != inSlope || _target.OutTangent != outSlope)
            {
                AnmKeyFrame beforeKey = _target.Clone() as AnmKeyFrame;

                // 値の変更
                this.ValueAsFloatWithoutCommand = val;
                this.TimeWithoutCommand = time;
                this.InTangentWithoutCommand = inSlope;
                this.OutTangentWithoutCommand = outSlope;

                AnmKeyFrame afterKey = _target.Clone() as AnmKeyFrame;

                // Undoコマンドの生成。
                _CommandFactory.MakeAnmKeyModCmd(_target, beforeKey, afterKey);
            }
        }

        /// <summary>
        /// スロープを再計算します。
        /// </summary>
        public void UpdateSlopeWithoutCommand()
        {
            _target.UpdateSlope(false);
        }

        /// <summary>
        /// スロープを再計算します。
        /// </summary>
        public void UpdateSlope()
        {
            // 更新前後の値を計算する。
            var beforeInSlope = _target.InTangent;
            var beforeOutSlope = _target.OutTangent;
            _target.UpdateSlope(false);
            var afterInSlope = _target.InTangent;
            var afterOutSlope = _target.OutTangent;

            // 一旦元に戻す。
            _target.InTangent = beforeInSlope;
            _target.OutTangent = beforeOutSlope;

            // Undo/Redo 対象として変更する。
            ModifyForce(_target.ValueAsFloat, _target.Time, afterInSlope, afterOutSlope);
        }

        /// <summary>
        /// 左Tangent（スロープ）
        /// </summary>
        public float InTangent
        {
            set
            {
                if( _target.InTangent == value )
                {
                    return;
                }

                AnmKeyFrame beforeKey = _target.Clone() as AnmKeyFrame;

                // コマンド生成無しでパラメータを操作します。
                this.InTangentWithoutCommand = value;
                _target.InterporationType = InterporationType.Fixed;

                AnmKeyFrame afterKey = _target.Clone() as AnmKeyFrame;
                // Undoコマンドの生成。
                _CommandFactory.MakeAnmKeyModCmd( _target, beforeKey, afterKey );
            }
        }

        /// <summary>
        /// 右Tangent（スロープ）
        /// </summary>
        public float OutTangent
        {
            set
            {
                if( _target.OutTangent == value )
                {
                    return;
                }

                AnmKeyFrame beforeKey = _target.Clone() as AnmKeyFrame;

                // コマンド生成無しでパラメータを操作します。
                this.OutTangentWithoutCommand = value;
                _target.InterporationType = InterporationType.Fixed;

                AnmKeyFrame afterKey = _target.Clone() as AnmKeyFrame;
                // Undoコマンドの生成。
                _CommandFactory.MakeAnmKeyModCmd( _target, beforeKey, afterKey );
            }
        }

        /// <summary>
        /// 補間種類を設定します。
        /// </summary>
        public InterporationType InterporationType
        {
            set
            {
                if( _target.InterporationType == value )
                {
                    return;
                }

                AnmKeyFrame beforeKey = _target.Clone() as AnmKeyFrame;
                this.InterporationTypeWithoutCommand = value;
                AnmKeyFrame afterKey = _target.Clone() as AnmKeyFrame;
                // Undoコマンドの生成。
                _CommandFactory.MakeAnmKeyModCmd( _target, beforeKey, afterKey );

            }
        }

        /// <summary>
        /// スロープを同期編集するか設定します。
        /// </summary>
        public bool UnifyTangents
        {
            set
            {
                // 更新が無ければ何もしません。
                if( _target.UnifyTangents == value )
                {
                    return;
                }

                AnmKeyFrame beforeKey = _target.Clone() as AnmKeyFrame;
                _target.UnifyTangents = value;
                AnmKeyFrame afterKey = _target.Clone() as AnmKeyFrame;
                // Undoコマンドの生成。
                _CommandFactory.MakeAnmKeyModCmd( _target, beforeKey, afterKey );
            }
        }

        #region Undoコマンドの生成をしないAPI
        public float TimeWithoutCommand
        {
            set
            {
                // カーブに変更設定。
                if( _target.Time != value )
                {
                    _target.OwnerAnmCurve.SetCurveStateDirty();
                    _target.Time = value;
                }

            }
        }

        public float ValueAsFloatWithoutCommand
        {
            set
            {
                Trace.Assert( !float.IsNaN( value ) );
                if( _target.ValueAsFloat       != value )
                {
                    _target.ValueAsFloat       = value;
                }
            }
        }

        /// <summary>
        /// 左Tangent（スロープ）Undoコマンド生成なし
        /// </summary>
        public float InTangentWithoutCommand
        {
            set
            {
                Trace.Assert( !float.IsNaN( value ) );
                if( _target.InTangent       != value )
                {
                    _target.InTangent       = value;
                }
            }
        }

        /// <summary>
        /// 右Tangent（スロープ）Undoコマンド生成なし
        /// </summary>
        public float OutTangentWithoutCommand
        {
            set
            {
                Trace.Assert( value != float.NaN );
                if( _target.OutTangent       != value )
                {
                    _target.OutTangent       = value;
                }
            }
        }

        /// <summary>
        /// 補間種類を設定します。Undoコマンド生成なし
        /// </summary>
        public InterporationType InterporationTypeWithoutCommand
        {
            set
            {
                if( _target.InterporationType == value )
                {
                    return;
                }
                _target.InterporationType = value;
            }
        }
        #endregion Undoコマンドの生成をしないAPI



        ///// <summary>
        ///// キーが所属するアニメーションカーブから自らを削除します。
        ///// </summary>
        public void RemoveFromOwner( bool bEvaluate )
        {
            Debug.Assert( _target != null );

            _ownerCurveMnp.BindTarget( _target.OwnerAnmCurve );
            _ownerCurveMnp.RemoveKeyFrame( _target as IAnmKeyFrame, bEvaluate );
        }
    }
}
