﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Windows.Forms;
using System.Threading.Tasks;

namespace LECore
{
    /// <summary>
    /// エラー表示処理
    /// </summary>
    /// <param name="title">タイトル文字列</param>
    /// <param name="msg">メッセージ本体</param>
    /// <param name="messageID">メッセージ識別番号(同一メッセージの表示抑止などに利用)</param>
    public delegate void LEReportMsgHandler(object sender, MessageArgs args);
    public delegate DialogResult LEReportMsgBoxHandler(string title, string msg, MessageBoxButtons messageBoxButtons);
    public delegate void LEReportMsgBoxAsyncHandler(string title, string msg);

    public class MessageArgs : EventArgs
    {
        public string title;
        public string msg;
        public DialogResult dialogResult;
    }

    /// <summary>
    /// ILEErrorReporter の概要の説明です。
    /// </summary>
    public class LEMsgReporter
    {
        public event LEReportMsgHandler PreReportMsg = null;
        public event LEReportMsgHandler OnReportMsg = null;
        public event LEReportMsgHandler OnReportLog = null;
        public event LEReportMsgBoxHandler OnShowDialog = null;

        /// <summary>
        /// 非同期のダイアログ表示イベント
        /// </summary>
        public event LEReportMsgBoxAsyncHandler ShowMessageDialogAsync
        {
            add
            {
                lock (ShowMessageDialogAsyncLocker)
                {
                    _ShowMessageDialogAsync += value;
                }
            }
            remove
            {
                lock (ShowMessageDialogAsyncLocker)
                {
                    _ShowMessageDialogAsync -= value;
                }
            }
        }

        private object ShowMessageDialogAsyncLocker = new object();
        private event LEReportMsgBoxAsyncHandler _ShowMessageDialogAsync;

        public event Action<string> OnBeginPacking = null;
        public event Action<bool> OnEndPacking = null;

        /// <summary>
        /// 強制的にエラー報告を停止する。
        /// </summary>
        public bool ForceStopErrorReport { get; set; }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public LEMsgReporter()
        {
        }

        /// <summary>
        /// メッセージの集約を開始します。
        /// </summary>
        public void BeginPacking(string categoryName)
        {
            if (OnBeginPacking != null)
            {
                OnBeginPacking(categoryName);
            }
        }

        /// <summary>
        /// メッセージの集約を終了します。
        /// </summary>
        public void EndPacking()
        {
            this.EndPacking(false);
        }

        /// <summary>
        /// メッセージの集約を終了します。
        /// </summary>
        public void EndPacking(bool doReportMessage)
        {
            if (OnEndPacking != null)
            {
                OnEndPacking(doReportMessage);
            }
        }

        /// <summary>
        /// エラーを報告する。書式指定
        /// </summary>
        public void ReportError( string title, string msg, params object[] parameters )
        {
            if (PreReportMsg != null)
            {
                string formatedMsg = String.Format( msg, parameters );
                // 第三引数は、msg　から取得する点に注意してください。
                PreReportMsg(null, new MessageArgs(){ title = title, msg = formatedMsg });
            }

            if (this.ForceStopErrorReport)
            {
                return;
            }

            if (OnReportMsg != null)
            {
                string formatedMsg = String.Format( msg, parameters );
                // 第三引数は、msg　から取得する点に注意してください。
                OnReportMsg(null, new MessageArgs(){ title = title, msg = formatedMsg });
            }
        }

        /// <summary>
        /// エラーを報告する。
        /// </summary>
        public void ReportError(object sender, MessageArgs args)
        {
            if (PreReportMsg != null)
            {
                PreReportMsg(sender, args);
            }

            if (this.ForceStopErrorReport)
            {
                return;
            }

            if (OnReportMsg != null)
            {
                OnReportMsg(sender, args);
            }
        }

        /// <summary>
        /// ログ出力。
        /// ユーザに確認(＋そのためのインタラクション)を求めません。
        /// </summary>
        public void OutLog( string title, string msg )
        {
            if ( PreReportMsg != null ) { PreReportMsg(this, new MessageArgs(){ title = title, msg = msg }); }
            if( OnReportLog != null ) { OnReportLog(this, new MessageArgs(){ title = title, msg = msg }); }
        }

        /// <summary>
        /// ダイアログを表示します。表示できなかった場合は、null を返します。
        /// バックグラウンド処理などであっても、強制的にダイアログが表示されることに注意してください。
        /// </summary>
        public DialogResult? ShowDialog(string title, string msg, MessageBoxButtons messageBoxButtons)
        {
            if (OnShowDialog != null) { return OnShowDialog(title, msg, messageBoxButtons); }

            return null;
        }

        /// <summary>
        /// 非同期にダイアログ表示
        /// </summary>
        public void OnShowMessageDialogAsync(string title, string msg)
        {
            lock (ShowMessageDialogAsyncLocker)
            {
                _ShowMessageDialogAsync?.Invoke(title, msg);
            }
        }
    }
}
