﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Management;
using System.Net;
using System.Net.Sockets;
using System.Reflection;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Nintendo.InGameEditing.Communication;

namespace LECore
{
    /// <summary>
    /// プレビュー用アーカイブの作成と通信によるコマンド転送を行います。
    /// Htcsを介するGfxMode時専用の実装です。
    /// </summary>
    public static class HtcsPreviewStarter
    {
        // 非同期受信用のイベント引数
        private static SocketAsyncEventArgs CurrentSocketAsyncEventArgs { get; set; } = null;

        public static bool IsWaitingToReceiveResponse { get { return CurrentSocketAsyncEventArgs != null; } }

        // 非同期受信でエラーが起きたかどうか
        public static bool ErrorAync { get; set; } = false;

        private static IPEndPoint lastTargetPoint = null;
        private static TcpClient connector = null;
        private const int ackByteSize = 1;

        /// <summary>
        /// プレビューをスタートします。
        /// </summary>
        public static int Start(ConnectionManager portMapper, string protocol, string peerType, string[] messageArgs, int retryDuration)
        {
            int errorLevel = 1;
            const int interval = 20;
            int retryCount = retryDuration/interval;
            while (retryCount >= 0)
            {
                var portInfo = portMapper.TargetInfos
                    .Where(t => (protocol == "HTCS" && peerType == t.PeerType)// HTCS 用
                        || t.PeerType == protocol) // HIO 用
                    .SelectMany(tt => tt.PortInfos)
                    .FirstOrDefault(p => p.PortName.Contains("NW4FLytViewer"));
                if (portInfo != null)
                {
                    errorLevel = SendLayoutViewerCommand(portInfo.EndPoint, string.Join(" ", messageArgs))
                        ? 0 : 4;
                    break;
                }

                --retryCount;
                Thread.Sleep(interval);
            }

            return errorLevel;
        }

        /// <summary>
        /// コマンドを転送します。
        /// </summary>
        private static bool SendLayoutViewerCommand(IPEndPoint targetPoint, string commandStr)
        {
            if (connector != null && (!targetPoint.Equals(lastTargetPoint) || !connector.Connected))
            {
                connector.Dispose();
                connector = null;
            }

            lastTargetPoint = null;

            if (connector == null)
            {
                connector = new TcpClient();
                connector.Connect(targetPoint);
            }

            {
                if (!connector.Connected)
                {
                    connector.Dispose();
                    connector = null;
                    return false;
                }

                // 接続できるかの確認だけ行う
                if (string.IsNullOrEmpty(commandStr))
                {
                    return true;
                }


                var content = Encoding.GetEncoding("shift_jis").GetBytes(commandStr);
                var size = toBytesLittleEndian((int)content.Length);
                var stream = connector.GetStream();
                try
                {
                    stream.Write(size, 0 , 4);
                    stream.Write(content, 0, content.Length);
                    stream.Flush();

                    // ビューアの処理が追いつかなくなることを防ぐために、受信確認をする
                    byte[] buffer = new byte[ackByteSize];
                    var socketAsyncEventArgs = new SocketAsyncEventArgs();
                    socketAsyncEventArgs.SetBuffer(buffer, 0, ackByteSize);
                    socketAsyncEventArgs.Completed += SocketAsyncEventArgs_Completed;
                    lock (ViewerExecuter.viewerThreadLocker)
                    {
                        CurrentSocketAsyncEventArgs = socketAsyncEventArgs;
                    }

                    var result = connector.Client.ReceiveAsync(socketAsyncEventArgs);
                    if (!result)
                    {
                        lock (ViewerExecuter.viewerThreadLocker)
                        {
                            CurrentSocketAsyncEventArgs.Dispose();
                            CurrentSocketAsyncEventArgs = null;
                        }
                    }
                }
                catch (Exception)
                {
                    return false;
                }

                lastTargetPoint = targetPoint;
            }

            return true;
        }

        // 整数をリトルエンディアンのバイト列に変換
        private static byte[] toBytesLittleEndian(int value)
        {
            return new byte[] { (byte)value, (byte)(value >> 8), (byte)(value >> 16), (byte)(value >> 24) };
        }

        // 受信完了
        private static void SocketAsyncEventArgs_Completed(object sender, SocketAsyncEventArgs e)
        {
            lock (ViewerExecuter.viewerThreadLocker)
            {
                if (e == CurrentSocketAsyncEventArgs)
                {
                    // 切断しても Success になることがあるので受信サイズも比較する
                    if (e.SocketError != SocketError.Success || e.BytesTransferred != ackByteSize)
                    {
                        ErrorAync = true;
                    }

                    CurrentSocketAsyncEventArgs = null;
                    Monitor.Pulse(ViewerExecuter.viewerThreadLocker);
                }
            }

            e.Dispose();
        }
    }
}
