﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Text.RegularExpressions;
using System.Drawing;
using System.Reflection;
using System.Diagnostics;
using System.Linq;
using System.Collections.Generic;
using LECore.Structures;

namespace LECore
{
    /// <summary>
    /// アプリケーション・コアで使用する定数値を管理するクラス。
    /// </summary>
    public class AppConstants
    {
        public const string AppName = "LayoutEditor";
        public const string FullAppName = "NW LayoutEditor";
        public static readonly string MainWindowTitle = FullAppName + (!Environment.Is64BitProcess ? " [x86]":"");
        public const string InitialFileName = "Untitled.";

        public const string ErrorLogPrefix = " ** ERROR ** ";
        public const string WarnigLogPrefix = " ** WARNING ** ";

        public const int  MaxUserdataStringLength  = 8;
        public const int  MaxObjectNameLength      = 24;
        public const int  MaxPaneNameLength        = MaxObjectNameLength;
        public const int  MaxControlNameLength     = 24;
        public const int  MaxGroupNameLength       = 32;
        public const int MaxAnimFrameSectionTagNameLength = 64;

        public const string RegexStrValidCharForObjectName = @"[0-9a-zA-Z_.-]";
        public const string RegexStrInvalidCharForObjectName = @"[^0-9a-zA-Z_.-]";
        public const string RegexStrInvalidCharForUserDataStr = @"[^ -~]";
        public const string RegexStrInvalidCharForUserDataExName = @"[^0-9A-Za-z\-\._]+";
        public const string RegexStrInvalidCharForUserDataExString = @"[^\u0000-\u007F]+"; // アスキー以外は認めない

        // ペイン名として不正な文字列をあらわす、正規表現
        static public readonly Regex      RegexInvalidCharForObjName
            = new Regex( LECore.AppConstants.RegexStrInvalidCharForObjectName );

        // ユーザデータ文字列として不正な文字列をあらわす、正規表現
        static public readonly Regex      RegexInvalidCharForUserDataStr
            = new Regex( LECore.AppConstants.RegexStrInvalidCharForUserDataStr );

        // ユーザ拡張データとして不正な文字列をあらわす、正規表現
        static public readonly Regex      RegexInvalidCharForUserDataExName
            = new Regex( LECore.AppConstants.RegexStrInvalidCharForUserDataExName );

        // ユーザ拡張データ文字列として不正な文字列をあらわす、正規表現
        static public readonly Regex		RegexInvalidCharForUserDataExString
            = new Regex( LECore.AppConstants.RegexStrInvalidCharForUserDataExString );

        static private readonly string _NwToolsRootPath;
        static private readonly string _NwFontToolsRootPath;

        public const string LayoutFileExtNoDot = "flyt";
        public const string LayoutFileExt = "." + LayoutFileExtNoDot;
        public const string LayoutBinaryExt = ".b" + LayoutFileExtNoDot;
        public const string LayoutFileExtPattern = "*." + LayoutFileExtNoDot;
        public const string LayoutFileDesc = "layout data";

        public const string AnimationFileExtNoDot = "flan";
        public const string AnimationFileExt = "." + AnimationFileExtNoDot;
        public const string AnimationBinaryExt = ".b" + AnimationFileExtNoDot;
        public const string AnimationFileExtPattern = "*." + AnimationFileExtNoDot;
        public const string AnimationFileDesc = "layout-animation data";

        public const string ContorlFileExtNoDot = "flct";
        public const string ContorlFileExt = "." + ContorlFileExtNoDot;
        public const string ContorlFileExtPattern = "*." + ContorlFileExtNoDot;
        public const string ContorlFileDesc = "lyout control data";
        public const string DefaultContorlFileName = "NW_DefaultControl" + ContorlFileExt;

        public const string ProjectFileExtNoDot = "flpj";
        public const string ProjectFileExt = "." + ProjectFileExtNoDot;
        public const string ProjectFileExtPattern = "*." + ProjectFileExtNoDot;
        public const string ProjectFileDesc = "project data";
        public const string ProjectFileName = "NW_LytProject" + ProjectFileExt;

        public const string UserDataPresetFileExtNoDot = "flup";
        public const string UserDataPresetFileExt = "." + UserDataPresetFileExtNoDot;
        public const string UserDataPresetFileExtPattern = "*." + UserDataPresetFileExtNoDot;

        public const string TTFFileExtNoDot = "ttf";
        public const string TTFFileExtExt = "." + TTFFileExtNoDot;
        public const string TTFFileExtExtPattern = "*." + TTFFileExtNoDot;

        public const string OTFFileExtNoDot = "otf";
        public const string OTFFileExtExt = "." + OTFFileExtNoDot;
        public const string OTFFileExtExtPattern = "*." + OTFFileExtNoDot;

        public const string BFTTFFileExtNoDot = "bfttf";
        public const string BFTTFFileExt = "." + BFTTFFileExtNoDot;
        public const string BFTTFFileExtPattern = "*." + BFTTFFileExtNoDot;

        public const string BFOTFFileExtNoDot = "bfotf";
        public const string BFOTFFileExtExt = "." + BFOTFFileExtNoDot;
        public const string BFOTFFileExtExtPattern = "*." + BFOTFFileExtNoDot;

        public const string FCPXFileExtNoDot = "fcpx";
        public const string FCPXFileExt = "." + FCPXFileExtNoDot;
        public const string FCPXFileExtPattern = "*." + FCPXFileExtNoDot;

        public const string FFNTFileExtNoDot = "ffnt";
        public const string FFNTFileExt = "." + FFNTFileExtNoDot;
        public const string FFNTFileExtPattern = "*." + FFNTFileExtNoDot;

        public const string AllFileDesc = "layout and all animation data";

        public const string BFFNTFileExtNoDot = "bffnt";
        public const string BFFNTFileExt = "." + BFFNTFileExtNoDot;
        public const string BFFNTFileExtPattern = "*." + BFFNTFileExtNoDot;
        public const string FontFileDesc = "font data";
        public const string FontFileAutoFilteredSuffix = "_auto_filtered";


        public const string CurveEditorFilterFileExt = ".xml";
        public const string CurveEditorFilterFileDesc = "CurveEditor Filter";

        public const string ThumbnailExt = ".png";
        public const string ThumbnailDirectory = ".NwThumbs";

        public const string TgaExt = ".tga";

        private static string _InternalModeName = "";

        public static readonly PointF DefaultScreenSize = new PointF(1280.0f, 720.0f);

        // コンバイナファイル定義です。
        public const string ECMBFileExtNoDot = "ecmb";
        public const string ECMBFileExt = "." + ECMBFileExtNoDot;
        public const string ECMBFileExtPattern = "*." + ECMBFileExtNoDot;
        public const string FCMBFileExtNoDot = "fcmb";
        public const string FCMBFileExt = "." + FCMBFileExtNoDot;
        public const string FCMBFileExtPattern = "*." + FCMBFileExtNoDot;


        public const int    CMBFileNameLength = 96;


        /// <summary>
        /// Layout で指定可能なテクスチャファイルかどうか
        /// </summary>
        /// <param name="filePath"></param>
        public static bool IsTextureFile(string filePath)
        {
            try
            {
                switch (Path.GetExtension(filePath).ToLower())
                {
                    case TgaExt:
                        return true;
                }
            }
            catch
            {
            }

            return false;
        }


        /// <summary>
        ///
        /// </summary>
        public static string GetDeprecatedAutoFilteredFontDir()
        {
            return Path.Combine(Path.GetTempPath(), "NW_Lyt_AutoFilteredFont");
        }

        public static string GetDeprecatedAutoFilteredLinearFontDir()
        {
            return Path.Combine(Path.GetTempPath(), "NW_Lyt_AutoFilteredLinearFont");
        }

        public static string GetDeprecatedTempImageFontDir()
        {
            return Path.Combine(Path.GetTempPath(), "NW_Lyt_TempImageFont");
        }

        public static string GetDeprecatedFtxbCacheDir()
        {
            return Path.Combine(Path.GetTempPath(), "NW_Lyt_FtxbCache");
        }

        /// <summary>
        /// フィルタフォントの削除
        /// </summary>
        static public void DeleteDeprecatedDirectoriesAll()
        {
            try
            {
                string autoGenFontDir = AppConstants.GetDeprecatedAutoFilteredFontDir();
                if (Directory.Exists(autoGenFontDir))
                {
                    Directory.Delete(autoGenFontDir, true);
                }

                string autoGenLinearFontDir = AppConstants.GetDeprecatedAutoFilteredLinearFontDir();
                if (Directory.Exists(autoGenLinearFontDir))
                {
                    Directory.Delete(autoGenLinearFontDir, true);
                }

                string tempImageFontDir = AppConstants.GetDeprecatedTempImageFontDir();
                if (Directory.Exists(tempImageFontDir))
                {
                    Directory.Delete(tempImageFontDir, true);
                }

                var ftxbCacheDir = AppConstants.GetDeprecatedFtxbCacheDir();
                if (Directory.Exists(ftxbCacheDir))
                {
                    Directory.Delete(ftxbCacheDir, true);
                }
            }
            catch (Exception e)
            {
                Debug.Assert(false, e.ToString());
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        static AppConstants()
        {
            if (LayoutEditorCore.PlatformDetail != null)
            {
                _NwToolsRootPath = LayoutEditorCore.PlatformDetail.NwToolsRootPath;
                _NwFontToolsRootPath = LayoutEditorCore.PlatformDetail.NwFontToolsRootPath;
            }
        }

        /// <summary>
        /// NitendoWareツールルートパスを取得します。
        /// </summary>
        public static string NwToolsRootPath
        {
            get { return _NwToolsRootPath; }
        }

        /// <summary>
        /// 環境が正しい設定か？
        /// </summary>
        public static bool IsToolEnviromentValid
        {
            get { return !string.IsNullOrEmpty(AppConstants.NwToolsRootPath) && Directory.Exists(AppConstants.NwToolsRootPath); }
        }

        /// <summary>
        /// 実機通信に必要な環境変数設定が正しい設定か？
        /// </summary>
        public static bool CheckTargetComEnviromentValid()
        {
            bool isInvalid = GetTargetComEnvList().Any(
                (env) =>
                {
                    string envVal = Environment.GetEnvironmentVariable(env);
                    return string.IsNullOrEmpty(envVal) || !Directory.Exists(envVal);
                });

            return !isInvalid;
        }

        /// <summary>
        /// Gfxモードで動作しているかどうか
        /// </summary>
        public static bool IsGfxMode
        {
            get
            {
                // プラットフォーム設定をxmlから読み込んでいるプラグインならばGfxModeとして動作します。
                var prefOwner = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
                if (prefOwner == null || prefOwner.PlatformPreference == null)
                {
                    Debug.Assert(false);
                    return false;
                }

                // NW4F モードなら false
                if (prefOwner.PlatformPreference.IsNW4FMode)
                {
                    Debug.Assert(false);
                    return false;
                }

                return true;
            }
        }

        /// <summary>
        /// 実機通信に必要な環境変数設定
        /// </summary>
        public static IEnumerable<string> GetTargetComEnvList()
        {
            return LayoutEditorCore.PlatformDetail.GetTargetComEnvList();
        }

        /// <summary>
        /// NitendoWareツールルートパスを取得します。
        /// </summary>
        public static string LayoutToolsPath
        {
            get { return LayoutEditorCore.PlatformDetail.LayoutToolsRootPath; }
        }

        /// <summary>
        /// PC ビューアの実行ファイル名
        /// </summary>
        public static string PCViewerFileName
        {
            get { return IsGfxMode ? "lytviewerGfx.exe" : "lytviewer.exe"; }
        }

        /// <summary>
        /// ビューアフォルダ名
        /// </summary>
        public static string ViewerSubFolderName
        {
            get { return IsGfxMode ? "LayoutViewer" : "LayoutViewer-Nw4f"; }
        }

        /// <summary>
        /// バイナリコンバータのパス
        /// </summary>
        public static string LayoutConverterPath
        {
            get { return LayoutEditorCore.PlatformDetail.LayoutConverterPath; }
        }

        /// <summary>
        /// CtrLytPCプレビュー用バイナリコンバータのパス
        /// </summary>
        public static string LayoutConverterForPCViewerPath
        {
            get { return LayoutEditorCore.PlatformDetail.LayoutConverterForPCViewerPath; }
        }

        /// <summary>
        /// フォントコンバータのパス（実機やビューアーが参照するフォントを作るためのコンバーター）
        /// </summary>
        public static string FontConverterConsolePath
        {
            get
            {
                // TODO: プラットフォームに依存するものは AppConstants から外す
                // 現在はプラットフォームに依存していないので動作に問題はない。

                // プラットフォーム設定をxmlから読み込んでいるプラグインならばGfxModeとして動作します。
                IPlatformPreferenceOwner prefOwner = LayoutEditorCore.PlatformDetail as IPlatformPreferenceOwner;
                if (prefOwner != null && prefOwner.PlatformPreference != null)
                {
                    if(!String.IsNullOrEmpty( prefOwner.PlatformPreference.FontConverterConsolePath))
                    {
                        return Path.Combine(AppConstants.NwToolsRootPath, prefOwner.PlatformPreference.FontConverterConsolePath);
                    }
                }

                return LayoutEditorCore.PlatformDetail.FontConverterConsolePath;
            }
        }

        /// <summary>
        /// ツール内でのプレビュー用途に生成する一時フォントを生成するコンバーターのパス
        /// </summary>
        public static string FontConverterConsolePathForToolInternalPreview
        {
            get
            {
                return LayoutEditorCore.PlatformDetail.FontConverterConsolePath;
            }
        }

        /// <summary>
        /// 実機ビューア起動バッチのパス
        /// </summary>
        public static string RunCafeViewerBatPath
        {
            get { return LayoutEditorCore.PlatformDetail.RunCafeViewerBatPath; }
        }

        /// <summary>
        /// PCビューアのパス
        /// </summary>
        public static string PCLayoutViewerPath
        {
            get { return LayoutEditorCore.PlatformDetail.PCLayoutViewerPath; }
        }

        /// <summary>
        /// Cafeビューアスタータのパス
        /// </summary>
        public static string ViewerStarterPath
        {
            get { return LayoutEditorCore.PlatformDetail.ViewerStarterPath; }
        }

        /// <summary>
        /// プレビュー一時ディレクトリパスを取得します。
        /// </summary>
        public static string PreviewTemporaryPath
        {
            get { return LayoutEditorCore.PlatformDetail.PreviewTemporaryPath; }
        }

        /// <summary>
        /// LayoutEditorのフォルダ名
        /// </summary>
        public static string LayoutEditorFolderName
        {
            get { return "LayoutEditor"; }
        }

        /// <summary>
        /// プレビュー一時ファイルパスを取得します。
        /// </summary>
        public static string PlatformName
        {
            get
            {
                return (LayoutEditorCore.PlatformDetail != null) ? LayoutEditorCore.PlatformDetail.UIPlatformName : "";
            }
        }

        /// <summary>
        /// タイトルバーに表示するプラットフォーム名を取得します。
        /// </summary>
        public static string PlatformNameForTitleBar
        {
            get
            {
                if (IsGfxMode)
                {
                    // GFXモードなら設定を持っているのは確定なのでノーチェックでOK
                    var prefOwner = (IPlatformPreferenceOwner)LayoutEditorCore.PlatformDetail;
                    return prefOwner.PlatformPreference.UiPlatformName;
                }

                return PlatformName;
            }
        }

        /// <summary>
        /// サンプルデータパスです。
        /// </summary>
        public static string NwDefaultSampleRoot
        {
            get { return (LayoutEditorCore.PlatformDetail != null) ? LayoutEditorCore.PlatformDetail.NwDefaultSampleRoot : ""; }
        }

        /// <summary>
        /// 単体テストデータパスです。
        /// </summary>
        public static string NwUnitTestDataRoot
        {
            get
            {
                return Path.Combine(NwDefaultSampleRoot, "LayoutEditorUnitTest");
            }
        }

        /// <summary>
        /// 存在するディレクトリの取得を試行します。
        /// </summary>
        private static string TryGetExistedSamplePath_(string rootPath)
        {
            if (Directory.Exists(rootPath))
            {
                return rootPath;
            }
            else
            {
                // lyt のほうを探してみる
                string lytRootPath = rootPath.Replace("ui2d", "lyt");
                if (Directory.Exists(lytRootPath))
                {
                    return lytRootPath;
                }
                else if (rootPath.Contains("Nw4f\\"))
                {
                    // Nw4f フォルダ を削除して確認します（TODO:移行期間を過ぎたら削除します）
                    string resultPath = TryGetExistedSamplePath_(rootPath.Replace("Nw4f\\", ""));
                    if (!string.IsNullOrEmpty(resultPath))
                    {
                        return resultPath;
                    }
                }
            }

            return string.Empty;
        }

        /// <summary>
        /// NintendoSDK 環境での、デフォルトサンプルディレクトリを取得します。
        /// </summary>
        public static string GetNintendoSDKDefaultSamplePath(string sigloRootPath)
        {
            string result;
            string samplePathFromRoot = @"Samples\Sources\Resources\ui2d";

            // リポジトリ位置のフォルダを探します
            string repositorySampleRoot = Path.Combine(sigloRootPath, @"..\..\" + samplePathFromRoot);
            result = TryGetExistedSamplePath_(repositorySampleRoot);
            if (!string.IsNullOrEmpty(result))
            {
                return result;
            }

            // パッケージ位置を探します。
            string packageSampleRoot = Path.Combine(sigloRootPath, samplePathFromRoot);
            result = TryGetExistedSamplePath_(packageSampleRoot);
            if (!string.IsNullOrEmpty(result))
            {
                return result;
            }

            return string.Empty;
        }

        /// <summary>
        /// バージョン文字列を得る
        /// </summary>
        /// <returns></returns>
        public static string GetVersionString()
        {
            // 情報の取得
            Assembly assembly = Assembly.GetExecutingAssembly();
            FileVersionInfo info = FileVersionInfo.GetVersionInfo(assembly.Location);

            // ビルド日時の取得
            Version version = assembly.GetName().Version;
            DateTime buildDate = new DateTime(2000, 1, 1, 0, 0, 0);
            buildDate = buildDate.AddDays((double)version.Build);
            buildDate = buildDate.AddSeconds((double)version.Revision * 2.0);

            return string.Format("{0} [{1}-bit][{6}] {7} Version {2}.{3}.{4} Release {5} ({8} {9})",
                    FullAppName + ".exe",
                    IntPtr.Size == 8 ? "64" : "32",
                    info.ProductMajorPart,
                    info.ProductMinorPart,
                    info.ProductBuildPart,
                    info.ProductPrivatePart,
                    PlatformName,
                    _InternalModeName,
                    buildDate.ToShortDateString(),
                    buildDate.ToShortTimeString());
        }

        //----------------------------------------------------------

        public static readonly Color InactiveForeColor = Color.Gray;
        public static readonly Color InactiveBackColor = Color.FromArgb(252, 252, 255);

        public static readonly Color ActiveForeColor = Color.Black;
        public static readonly Color ActiveBackColor = Color.White;

        public static readonly Color InactiveCaptureTextureBackColor = Color.FromArgb(239, 237, 171);
        public static readonly Color ActiveCatpureTextureBackColor = Color.FromArgb(242, 240, 174);

        public static readonly Color SealedForeColor = Color.LightGray;
    }
}
