﻿/*--------------------------------------------------------------------------------*
Copyright (C)Nintendo All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*--------------------------------------------------------------------------------*/

// これは メイン DLL ファイルです。

#include "stdafx.h"

#include "GfxPrimitiveShape.h"

namespace Gfx
{

void PrimitiveShape::CreateShapeImpl([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, nn::gfx::util::PrimitiveShape* pShape)
{
    int vertexCount = pShape->GetVertexCount();
    int indexCount = pShape->GetIndexCount();

    size_t  vertexMemorySize = GetVertexStride() * vertexCount;
    void*   pVertexMemory = new char[vertexMemorySize];
    size_t  indexMemorySize = pShape->GetIndexBufferSize() * indexCount;
    void*   pIndexMemory = new char[indexMemorySize];

    pShape->Calculate(pVertexMemory, vertexMemorySize, pIndexMemory, indexMemorySize);

    float*  pVertexBufferFloat = static_cast<float*>(pVertexMemory);

    for (int i = 0; i < vertexCount; ++i)
    {
        PrimitiveShapeVertex^ vertex = gcnew PrimitiveShapeVertex();

        vertex->x = *pVertexBufferFloat;
        ++pVertexBufferFloat;
        vertex->y = *pVertexBufferFloat;
        ++pVertexBufferFloat;
        vertex->z = *pVertexBufferFloat;
        ++pVertexBufferFloat;
        vertex->u = *pVertexBufferFloat;
        ++pVertexBufferFloat;
        vertex->v = *pVertexBufferFloat;
        ++pVertexBufferFloat;

        vertexBuffer->Add(vertex);
    }

    nn::gfx::IndexFormat    indexFormat = pShape->GetIndexBufferFormat();

    switch (indexFormat)
    {
    case    nn::gfx::IndexFormat_Uint8:
    {
        unsigned char* pIndexBuffer = static_cast<unsigned char*>(pIndexMemory);
        for (int i = 0; i < indexCount; ++i)
        {
            indexBuffer->Add(*pIndexBuffer);
            pIndexBuffer++;
        }
    }
    break;
    case    nn::gfx::IndexFormat_Uint16:
    {
        unsigned short* pIndexBuffer = static_cast<unsigned short*>(pIndexMemory);
        for (int i = 0; i < indexCount; ++i)
        {
            indexBuffer->Add(*pIndexBuffer);
            pIndexBuffer++;
        }
    }
    break;
    case    nn::gfx::IndexFormat_Uint32:
    {
        unsigned int* pIndexBuffer = static_cast<unsigned int*>(pIndexMemory);
        for (int i = 0; i < indexCount; ++i)
        {
            indexBuffer->Add(*pIndexBuffer);
            pIndexBuffer++;
        }
    }
    break;
    }

    delete[] pVertexMemory;
    delete[] pIndexMemory;
}

void PrimitiveShape::CreateCube([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer)
{
    nn::gfx::util::CubeShape  cube(vertexFormat, nn::gfx::PrimitiveTopology_TriangleList);

    CreateShapeImpl(vertexBuffer, indexBuffer, &cube);
}

void PrimitiveShape::CreateSphere([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, int sliceCount, int stackCount)
{
    nn::gfx::util::SphereShape  sphere(vertexFormat, nn::gfx::PrimitiveTopology_TriangleList, sliceCount, stackCount);

    CreateShapeImpl(vertexBuffer, indexBuffer, &sphere);
}

void PrimitiveShape::CreateHemiSphere([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, int sliceCount)
{
    nn::gfx::util::HemiSphereShape  hemiSphere(vertexFormat, nn::gfx::PrimitiveTopology_TriangleList, sliceCount);

    CreateShapeImpl(vertexBuffer, indexBuffer, &hemiSphere);
}

void PrimitiveShape::CreateCylinder([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, int sliceCount)
{
    nn::gfx::util::CylinderShape    cylinder(vertexFormat, nn::gfx::PrimitiveTopology_TriangleList, sliceCount);

    CreateShapeImpl(vertexBuffer, indexBuffer, &cylinder);
}

void PrimitiveShape::CreateCircle([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, int sliceCount)
{
    nn::gfx::util::CircleShape  circle(vertexFormat, nn::gfx::PrimitiveTopology_TriangleList, sliceCount);

    CreateShapeImpl(vertexBuffer, indexBuffer, &circle);
}

void PrimitiveShape::CreateRoundRect([Runtime::InteropServices::Out] List<PrimitiveShapeVertex^>^ vertexBuffer, [Runtime::InteropServices::Out] List<int>^ indexBuffer, float width, float height, float cornerRadiusW, float cornerRadiusH, int sliceCount)
{
    float   halfWidth = width * 0.5f;
    float   halfHeight = height * 0.5f;
    float   cornerWidth = halfWidth < cornerRadiusW ? halfWidth : cornerRadiusW;
    float   cornerHeight = halfHeight < cornerRadiusH ? halfHeight : cornerRadiusH;

    float   quadrantSign[4][2] =
    {
        { 1.0f,  1.0f}, //  第一象限
        {-1.0f,  1.0f}, //  第二象限
        {-1.0f, -1.0f}, //  第三象限
        { 1.0f, -1.0f}, //  第四象限
    };

    int roundVertexCount = sliceCount + 1;

    // 象限分ループ
    for (int quadrantIndex = 0; quadrantIndex < 4; ++quadrantIndex)
    {
        // 角丸の中心座標
        {
            PrimitiveShapeVertex^ vertex = gcnew PrimitiveShapeVertex();

            vertex->x = (halfWidth - cornerWidth) * quadrantSign[quadrantIndex][0];
            vertex->y = (halfHeight - cornerHeight) * quadrantSign[quadrantIndex][1];
            vertex->z = 0.0f;

            vertex->u = CalculateTexMapU(vertex->x, halfWidth);
            vertex->v = CalculateTexMapV(vertex->y, halfHeight);

            vertexBuffer->Add(vertex);
        }

        float angle = 0.0f;
        //  円周側頂点
        // sin へ 2pi を渡すと精度が悪く 0 にならないため、そのまま計算すると頂点位置が微妙にずれて隙間が発生することがある。
        // 精度のよい第一象限の値と符号で生成する座標を作成するようにしています。
        for (int i = 0; i < roundVertexCount; ++i)
        {
            PrimitiveShapeVertex^ vertex = gcnew PrimitiveShapeVertex();

            float cosValue = cosf(angle);
            float sinValue = sinf(angle);
            vertex->x = (halfWidth - cornerWidth + cosValue * cornerWidth) * quadrantSign[quadrantIndex][0];
            vertex->y = (halfHeight - cornerHeight + sinValue * cornerHeight) * quadrantSign[quadrantIndex][1];
            vertex->z = 0.0f;

            vertex->u = CalculateTexMapU(vertex->x, halfWidth);
            vertex->v = CalculateTexMapV(vertex->y, halfHeight);

            vertexBuffer->Add(vertex);

            angle += ((nn::util::FloatPi * 0.5f) * (1.0f / (float)sliceCount));
            if (i > 0)
            {
                int indexOffsetBase = (roundVertexCount + 1) * quadrantIndex;

                indexBuffer->Add(indexOffsetBase);
                // ひとつ前の頂点のインデックスだが、原点頂点のオフセットが常に 1 加算されているため、オフセットがなしになる。
                indexBuffer->Add(indexOffsetBase + i);
                indexBuffer->Add(indexOffsetBase + i + 1);
            }
        }
    }

    int indexOffsetBase = (roundVertexCount + 1) * 4;

    // 左右の角間の矩形
    if (cornerWidth * 2.0f < width)
    {
        // 象限分ループ
        for (int quadrantIndex = 0; quadrantIndex < 4; ++quadrantIndex)
        {
            PrimitiveShapeVertex^ vertex = gcnew PrimitiveShapeVertex();

            vertex->x = (halfWidth - cornerWidth) * quadrantSign[quadrantIndex][0];
            vertex->y = halfHeight * quadrantSign[quadrantIndex][1];
            vertex->z = 0.0f;

            vertex->u = CalculateTexMapU(vertex->x, halfWidth);
            vertex->v = CalculateTexMapV(vertex->y, halfHeight);

            vertexBuffer->Add(vertex);

            vertex = gcnew PrimitiveShapeVertex();

            vertex->x = (halfWidth - cornerWidth) * quadrantSign[quadrantIndex][0];
            vertex->y = (halfHeight - cornerHeight) * quadrantSign[quadrantIndex][1];
            vertex->z = 0.0f;

            vertex->u = CalculateTexMapU(vertex->x, halfWidth);
            vertex->v = CalculateTexMapV(vertex->y, halfHeight);

            vertexBuffer->Add(vertex);
        }

        indexBuffer->Add(indexOffsetBase);
        indexBuffer->Add(indexOffsetBase + 2);
        indexBuffer->Add(indexOffsetBase + 3);

        indexBuffer->Add(indexOffsetBase);
        indexBuffer->Add(indexOffsetBase + 3);
        indexBuffer->Add(indexOffsetBase + 1);

        indexOffsetBase += 4;

        indexBuffer->Add(indexOffsetBase);
        indexBuffer->Add(indexOffsetBase + 2);
        indexBuffer->Add(indexOffsetBase + 3);

        indexBuffer->Add(indexOffsetBase);
        indexBuffer->Add(indexOffsetBase + 3);
        indexBuffer->Add(indexOffsetBase + 1);

        indexOffsetBase += 4;
    }

    // 上下の角間の矩形
    // 真ん中の領域もつなげて大きな矩形にする。
    if (cornerHeight * 2.0f < height)
    {
        // 象限分ループ
        for (int quadrantIndex = 0; quadrantIndex < 4; ++quadrantIndex)
        {
            PrimitiveShapeVertex^ vertex = gcnew PrimitiveShapeVertex();

            vertex->x = halfWidth * quadrantSign[quadrantIndex][0];
            vertex->y = (halfHeight - cornerHeight) * quadrantSign[quadrantIndex][1];
            vertex->z = 0.0f;

            vertex->u = CalculateTexMapU(vertex->x, halfWidth);
            vertex->v = CalculateTexMapV(vertex->y, halfHeight);

            vertexBuffer->Add(vertex);
        }

        for (int i = 0; i < 2; ++i)
        {
            indexBuffer->Add(indexOffsetBase);
            indexBuffer->Add(indexOffsetBase + i + 1);
            indexBuffer->Add(indexOffsetBase + i + 2);
        }
        indexOffsetBase += 4;
    }
}


}
