﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW4R_FROM_TOOL
#include <windows.h>

#include "binaryFileFormat.h"
#include "nw4r/misc.h"
#include "inlines.h"
#include "ByteOrderUtil.h"

namespace
{
#define NN_DETAIL_VERSION_MAJOR(version)        (((version) >> 24) & 0xff)
#define NN_DETAIL_VERSION_MINOR(version)        (((version) >> 16) & 0xff)
#define NN_DETAIL_VERSION_MICRO(version)        (((version) >>  8) & 0xff)
#define NN_DETAIL_VERSION_BINARYBUGFIX(version) (((version) >>  0) & 0xff)
}

namespace NW4R
{
namespace Font
{
namespace UnManaged
{
    //----------------------------------------------------------
    // エンディアンを転置する前の状態でファイルヘッダ内容を調査します。
    BinaryFileValidateCode ValidateBinaryFile(
        const BinaryFileHeader* header,
        u32                     signature,
        u32                     signature2,
        u32                     version,
        u16                     minBlocks)
    {
        NW4R_POINTER_ASSERT( header );
        if( header->signature != signature &&
            header->signature != signature2 )
        {
            NW4R_WARNING(FALSE, "Signature check failed ('%c%c%c%c' must be '%c%c%c%c').",
            BitExtract<u32>(header->signature, 24, 8),
            BitExtract<u32>(header->signature, 16, 8),
            BitExtract<u32>(header->signature,  8, 8),
            BitExtract<u32>(header->signature,  0, 8),
            BitExtract<u32>(signature, 24, 8),
            BitExtract<u32>(signature, 16, 8),
            BitExtract<u32>(signature,  8, 8),
            BitExtract<u32>(signature,  0, 8));
            return BinaryFileValidateCode::InvalidSignature;
        }

        bool isBigEndian = header->byteOrder == BYTE_ORDER_MARK;
        const u32 headerVersion = isBigEndian ? Swap4Byte(header->version) : header->version;
        if ( NN_DETAIL_VERSION_MAJOR(version) != NN_DETAIL_VERSION_MAJOR(headerVersion) ||
            NN_DETAIL_VERSION_MINOR(version) < NN_DETAIL_VERSION_MINOR(headerVersion)  ||
            NN_DETAIL_VERSION_BINARYBUGFIX(version) > NN_DETAIL_VERSION_BINARYBUGFIX(headerVersion) )
        {
            NW4R_WARNING(FALSE, "Version check faild ('%d.%d' must be '%d.%d').",
            BitExtract<u32>(header->version, 8, 8),
            BitExtract<u32>(header->version, 0, 8),
            BitExtract<u32>(version, 8, 8),
            BitExtract<u32>(version, 0, 8));
            return NW4R::Font::UnManaged::BinaryFileValidateCode::InvalidVersion;
        }

        const u32 fileSize = isBigEndian ? Swap4Byte(header->fileSize) : header->fileSize;
        if( fileSize < sizeof(BinaryFileHeader) + sizeof(BinaryBlockHeader) * minBlocks )
        {
            NW4R_WARNING(FALSE, "Too small file size(=%d).", fileSize );
            return BinaryFileValidateCode::InvalidFileSize;
        }

        const u16 dataBlocks = isBigEndian ? Swap2Byte(header->dataBlocks) : header->dataBlocks;
        if( dataBlocks < minBlocks )
        {
            NW4R_WARNING(FALSE, "Too small number of data blocks(=%d).", dataBlocks);
            return BinaryFileValidateCode::InvalidBlockSize;
        }

        return BinaryFileValidateCode::Valid;
    }


} /* namespace UnManaged */
} /* namespace ut */
} /* namespace nw4r */
