﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW4R_FROM_TOOL
#include <windows.h>
#include <malloc.h>

#include <cstdarg>
// #include <dolphin.h>
#include <cstdio>
#include "nw4r/misc.h"
#include "TextWriterBase.h"
#include "TagProcessorBase.h"
#include "inlines.h"

namespace NW4R
{
namespace Font
{
namespace UnManaged
{



//#define GET_FORMAT_BUFFER()                                 \
//    reinterpret_cast<CharType*>((mFormatBuffer != NULL) ?   \
//        mFormatBuffer:                                      \
//        __alloca(mFormatBufferSize))

#define GET_FORMAT_BUFFER()                                 \
    reinterpret_cast<CharType*>((mFormatBuffer != NULL) ?   \
        mFormatBuffer:                                      \
        _alloca(mFormatBufferSize))

/* ------------------------------------------------------------------------
        クラス変数定義
   ------------------------------------------------------------------------ */

template <typename CharType>
CharType* TextWriterBase<CharType>::mFormatBuffer = NULL;

template <typename CharType>
std::size_t TextWriterBase<CharType>::mFormatBufferSize = DEFAULT_FORMAT_BUFFER_SIZE;

template <typename CharType>
TagProcessorBase<CharType> TextWriterBase<CharType>::mDefaultTagProcessor;


/*!--------------------------------------------------------------------------*
  @brief        基点が中央であるときのカーソル位置補正の値を求めます。
                半分の値が浮動小数点にならないように整数値に切り上げます。

  @param[in]    value  考慮する幅/高さ

  @return       基点が中央であるときのカーソル位置補正の値を返します。
 *---------------------------------------------------------------------------*/
inline
f32
AdjustCenterValue(f32 value)
{
    return ::std::ceilf(value * 0.5f);
}

// 印字されない文字かどうかを判定する
bool IsPrintableChar(uint32_t code)
{
    // http://unicode.org/reports/tr9/ に記載されている双方向テキストは
    // 印字されない文字として扱う。

    // ALM はアラビア文字用で通常はテキストに入り込まないためチェックしない。
    // if (code == 0x061c /*ALM*/)
    // {
    //     return true;
    // }

    // 高速化のために 0x20XX かどうかで一旦ふるいにかける。
    if ((code & 0xffffff00) == 0x2000)
    {
        if ((0x200e <= code && code <= 0x200f) /*LRM,RLM*/ ||
            (0x202a <= code && code <= 0x202e) /*LRE,RLE,PDF,LRO,RLO*/ ||
            (0x2066 <= code && code <= 0x2069) /*LRI,RLI,FSI,PDI*/)
        {
            return false;
        }
    }

    return true;
}

/* =======================================================================
        public
   ======================================================================== */

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::TextWriterBase()

  Description:  コンストラクタ。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
TextWriterBase<CharType>::TextWriterBase()
: CharWriter(),
  mCharSpace(0),
  mLineSpace(0),
  mTabWidth (4),
  m_IsDrawFromRightToLeftEnabled(false),
  mDrawFlag( DEFAULT_DRAWFLAG ),
  mTagProcessor(&mDefaultTagProcessor)
{
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::~TextWriterBase()

  Description:  デストラクタ。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
TextWriterBase<CharType>::~TextWriterBase()
{
}






/* ------------------------------------------------------------------------
        タグ処理
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::SetTagProcessor( TagProcessorBase* )

  Description:  文字列描画中のタグを処理するオブジェクトを設定します。

  Arguments:    タグを処理するオブジェクトへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::SetTagProcessor(TagProcessorBase<CharType>* tagProcessor)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( tagProcessor );

    mTagProcessor = tagProcessor;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::ResetTagProcessor()

  Description:  文字列描画中のタグ処理を初期値に戻します。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::ResetTagProcessor()
{
    NW4R_POINTER_ASSERT( this );

    mTagProcessor = &mDefaultTagProcessor;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::GetTagProcessor()

  Description:  文字列描画中のタグを処理するオブジェクトを取得します。

  Arguments:    なし。

  Returns:      タグを処理するオブジェクトへのリファレンス。
 *---------------------------------------------------------------------------*/
template <typename CharType>
TagProcessorBase<CharType>&
TextWriterBase<CharType>::GetTagProcessor() const
{
    NW4R_POINTER_ASSERT( this );
    return *mTagProcessor;
}





/* ------------------------------------------------------------------------
        タグ処理
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::SetDrawFlag()

  Description:  文字列描画フラグを設定します。

  Arguments:    flags:  新しい描画フラグ

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::SetDrawFlag(u32 flags)
{
    NW4R_POINTER_ASSERT( this );
    mDrawFlag = flags;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::GetDrawFlag()

  Description:  文字列描画フラグを取得します。

  Arguments:    なし。

  Returns:      文字列描画フラグ。
 *---------------------------------------------------------------------------*/
template <typename CharType>
u32
TextWriterBase<CharType>::GetDrawFlag() const
{
    NW4R_POINTER_ASSERT( this );
    return mDrawFlag;
}








/* ------------------------------------------------------------------------
        行間/文字間/タブ幅
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::SetLineHeight( float )

  Description:  1行の高さを設定します。

  Arguments:    height: 新しい1行の高さ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::SetLineHeight(float height)
{
    NW4R_POINTER_ASSERT( this );
    const Font* font = GetFont();
    const int linefeed = (font != NULL) ? font->GetLineFeed(): 0;

    mLineSpace = height - linefeed * GetScaleV();
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::GetLineHeight()

  Description:  1行の高さを取得します。

  Arguments:    なし。

  Returns:      1行の高さ。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::GetLineHeight() const
{
    NW4R_POINTER_ASSERT( this );
    const Font* font = GetFont();
    const int linefeed = (font != NULL) ? font->GetLineFeed(): 0;

    return linefeed * GetScaleV() + mLineSpace;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::SetLineSpace( float )

  Description:  文字列描画時の行間を設定します。

  Arguments:    space:  新しい行間。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::SetLineSpace(float space)
{
    NW4R_POINTER_ASSERT( this );
    mLineSpace = space;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::SetCharSpace( float )

  Description:  文字列描画時の文字間を設定します。

  Arguments:    space:  新しい文字間。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::SetCharSpace(float space)
{
    NW4R_POINTER_ASSERT( this );
    mCharSpace = space;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::GetLineSpace()

  Description:  文字列描画時の行間を取得します。

  Arguments:    なし。

  Returns:      文字列描画時の行間。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::GetLineSpace() const
{
    NW4R_POINTER_ASSERT( this );
    return mLineSpace;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::GetCharSpace()

  Description:  文字列描画字の文字間を取得します。

  Arguments:    なし。

  Returns:      文字列描画時の文字間。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::GetCharSpace() const
{
    NW4R_POINTER_ASSERT( this );
    return mCharSpace;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::SetTabWidth( int )

  Description:  文字列描画字のタブ幅を設定します。

  Arguments:    tabWidth:   新しい文字幅。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::SetTabWidth( int tabWidth )
{
    NW4R_POINTER_ASSERT( this );
    mTabWidth = tabWidth;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::GetTabWidth()

  Description:  文字列描画字のタブ幅を取得します。

  Arguments:    なし。

  Returns:      文字列描画字のタブ幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
int
TextWriterBase<CharType>::GetTabWidth() const
{
    NW4R_POINTER_ASSERT( this );
    return mTabWidth;
}

/*---------------------------------------------------------------------------*
   右側から描画するかどうかを設定します。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void TextWriterBase<CharType>::SetDrawFromRightToLeftEnabled(bool isDrawFromRightToLeftEnabled)
{
    m_IsDrawFromRightToLeftEnabled = isDrawFromRightToLeftEnabled;
}

/*---------------------------------------------------------------------------*
   右側から描画するかどうかを取得します。
 *---------------------------------------------------------------------------*/
template <typename CharType>
bool TextWriterBase<CharType>::GetDrawFromRightToLeftEnabled() const
{
    return m_IsDrawFromRightToLeftEnabled;
}

/* ------------------------------------------------------------------------
        文字列サイズ計算
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcFormatStringWidth( StreamType, ... )

  Description:  文字列の幅を計算します。

  Arguments:    format: 書式文字列
                ...:    パラメータ

  Returns:      文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::CalcFormatStringWidth(
    StreamType  format,
    ...
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( format );
    Rect rect;
    va_list vargs;
    va_start(vargs, format);

    CalcVStringRect(&rect, format, vargs);

    va_end(vargs);
    return rect.GetWidth();
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcFormatStringHeight( StreamType, ... )

  Description:  文字列の高さを計算します。

  Arguments:    format: 書式文字列。
                ...:    パラメータ

  Returns:      文字列の高さ
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::CalcFormatStringHeight(
    StreamType  format,
    ...
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( format );
    Rect rect;
    va_list vargs;
    va_start(vargs, format);

    CalcVStringRect(&rect, format, vargs);

    va_end(vargs);
    return rect.GetHeight();
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcFormatStringRect( Rect*, StreamType, ... )

  Description:  文字列の描画矩形を取得します。

  Arguments:    pRect:  描画矩形を格納するバッファへのポインタ。
                format: 書式文字列。
                ...:    パラメータ

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::CalcFormatStringRect(
    Rect*      pRect,
    StreamType  format,
    ...
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pRect );
    NW4R_POINTER_ASSERT( format );
    va_list vargs;
    va_start(vargs, format);

    CalcVStringRect(pRect, format, vargs);

    va_end(vargs);
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcVStringRect( Rect*, StreamType, va_list )

  Description:  文字列の描画矩形を取得します。

  Arguments:    pRect:  描画矩形を格納するバッファへのポインタ。
                format: 書式文字列。
                args:   パラメータ

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::CalcVStringRect(
    Rect*           pRect,
    StreamType      format,
    va_list    args
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pRect );
    NW4R_POINTER_ASSERT( format );
    CharType* buffer = GET_FORMAT_BUFFER();

    int length = VSNPrintf(buffer, mFormatBufferSize, format, args);
    CalcStringRect(pRect, buffer, length);
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcStringWidth( StreamType, int )

  Description:  文字列の幅を取得します。

  Arguments:    str:    文字列
                length: 文字列長(sizeof(CharType)単位)

  Returns:      文字列の高さ。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::CalcStringWidth(
    StreamType  str,
    int         length
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( str );
    NW4R_MIN_ASSERT( length, 0 );
    Rect rect;

    CalcStringRect(&rect, str, length);

    return rect.GetWidth();
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcStringHeight( StreamType, int )

  Description:  文字列の高さを取得します。

  Arguments:    str:    文字列
                length: 文字列長(sizeof(CharType)単位)

  Returns:      文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::CalcStringHeight(
    StreamType  str,
    int         length
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( str );
    NW4R_MIN_ASSERT( length, 0 );
    Rect rect;

    CalcStringRect(&rect, str, length);

    return rect.GetHeight();
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcStringRect( Rect*, StreamType, int )

  Description:  文字列の描画矩形を取得します。

  Arguments:    pRect:  描画矩形を格納するバッファへのポインタ。
                str:    文字列
                length: 文字列長(sizeof(CharType)単位)

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::CalcStringRect(
    Rect*       pRect,
    StreamType  str,
    int         length
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pRect );
    NW4R_POINTER_ASSERT( str );
    NW4R_MIN_ASSERT( length, 0 );

    TextWriterBase<CharType> myCopy = *this;

    myCopy.CalcStringRectImpl(pRect, str, length);
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcStringWidth( StreamType )

  Description:  文字列の幅を取得します。

  Arguments:    str:    文字列

  Returns:      文字列の高さ。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::CalcStringWidth(
    StreamType  str
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( str );

    return CalcStringWidth(str, StrLen(str));
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcStringHeight( StreamType )

  Description:  文字列の幅を取得します。

  Arguments:    str:    文字列

  Returns:      文字列の高さ。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::CalcStringHeight(
    StreamType  str
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( str );

    return CalcStringHeight(str, StrLen(str));
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcStringRect( StreamType )

  Description:  文字列の幅を取得します。

  Arguments:    pRect:  描画矩形を格納するバッファへのポインタ。
                str:    文字列

  Returns:      文字列の高さ。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::CalcStringRect(
    Rect*       pRect,
    StreamType  str
) const
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pRect );
    NW4R_POINTER_ASSERT( str );

    CalcStringRect(pRect, str, StrLen(str));
}






/* ------------------------------------------------------------------------
        文字列描画
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::Printf( StreamType, ... )

  Description:  文字列を描画します。

  Arguments:    format: 書式文字列。
                ...:    パラメータ。

  Returns:      描画した文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::Printf(
    StreamType  format,
    ...
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( format );
    va_list vargs;
    va_start(vargs, format);

    float width = VPrintf(format, vargs);

    va_end(vargs);
    return width;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::VPrintf( StreamType, va_list )

  Description:  文字列を描画します。

  Arguments:    format: 書式文字列。
                args    パラメータ。

  Returns:      描画した文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::VPrintf(
    StreamType      format,
    va_list    args
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( format );
    CharType* buffer = GET_FORMAT_BUFFER();

    int length = VSNPrintf(buffer, mFormatBufferSize, format, args);
    float width = Print(buffer, length);

    return width;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::Print( StreamType, int )

  Description:  文字列を描画します。

  Arguments:    str:    文字列。
                length: 文字列長(sizeof(CharType)単位)

  Returns:      描画した文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::Print(
    StreamType  str,
    int         length
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( str );
    NW4R_MIN_ASSERT( length, 0 );

    return Print(str, length, 0, NULL);
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::Print( StreamType, int )

  Description:  文字列を描画します。

  Arguments:    str:    文字列。

  Returns:      描画した文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::Print(
    StreamType  str
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( str );

    return Print(str, StrLen(str));
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::Print( StreamType, int )

  Description:  文字列を描画します。

  Arguments:    str:    文字列。
                length: 文字列長(sizeof(CharType)単位)
                lineOffsetCount: 行ごとのオフセットの数
                pLineOffset: 行ごとのオフセット

  Returns:      描画した文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::Print(
    StreamType  str,
    int         length,
    int             lineOffsetCount,
    const float*    pLineOffset
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( str );
    NW4R_MIN_ASSERT( length, 0 );

    // 登録されているタスクを消去します。
    ClearRenderingTask();

    //TextWriterBase<CharType> myCopy = *this;

    float width = this->PrintImpl(str, length, lineOffsetCount, pLineOffset);

    SetCursor( this->GetCursorX(), this->GetCursorY() );

    return width;
}

#if 0
/* ------------------------------------------------------------------------
        カーソル移動付き文字列描画
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::Printf( float, float, StreamType, ... )

  Description:  座標(x,y)にカーソルを移動後、文字列を描画します。

  Arguments:    x:      新しいカーソルの座標 x
                y:      新しいカーソルの座標 y
                format: 書式文字列
                ...:    パラメータ

  Returns:      描画した文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::Printf(
    float       x,
    float       y,
    StreamType  format,
    ...
)
{
    va_list vargs;
    va_start(vargs, format);

    SetCursor(x, y);
    float width = VAPrintf(format, vargs);

    va_end(vargs);
    return width;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::VAPrintf( float, float, StreamType, va_list )

  Description:  座標(x,y)にカーソルを移動後、文字列を描画します。

  Arguments:    x:      新しいカーソルの座標 x
                y:      新しいカーソルの座標 y
                format: 書式文字列
                args    パラメータ

  Returns:      描画した文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::VAPrintf(
    float           x,
    float           y,
    StreamType      format,
    va_list    args
)
{
    SetCursor(x, y);
    return VAPrintf(format, args);
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::Print( float, float, StreamType, int )

  Description:  座標(x,y)にカーソルを移動後、文字列を描画します。

  Arguments:    x:      新しいカーソルの座標 x
                y:      新しいカーソルの座標 y
                str:    文字列
                length: 文字列長(sizeof(CharType)単位)

  Returns:      描画した文字列の幅。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::Print(
    float       x,
    float       y,
    StreamType  str,
    int         length
)
{
    SetCursor(x, y);
    return Print(str, length);
}
#endif





/* ------------------------------------------------------------------------
        sprintfバッファ
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::SetBuffer( CharType*, size_t )

  Description:  書式文字列の展開に使用するバッファを指定します。

  Arguments:    buffer: 設定するバッファへのポインタ。
                size:   バッファのサイズ(sizeof(CharType)単位)。

  Returns:      以前のバッファが存在する場合には以前のバッファへのポインタ。
                そうでなければ NULL。
 *---------------------------------------------------------------------------*/
template <typename CharType>
/* static */ void*
TextWriterBase<CharType>::SetBuffer(CharType* buffer, std::size_t size)
{
    NW4R_POINTER_ASSERT( buffer );
    NW4R_MIN_ASSERT(size, 1);
    void* oldBuffer = mFormatBuffer;

    mFormatBuffer       = buffer;
    mFormatBufferSize   = size;

    return oldBuffer;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::SetBuffer( size_t )

  Description:  書式文字列の展開に使用するバッファをスタックに確保するように
                設定します。

  Arguments:    size:   バッファのサイズ(sizeof(CharType)単位)。

  Returns:      以前のバッファが存在する場合には以前のバッファへのポインタ。
                そうでなければ NULL。
 *---------------------------------------------------------------------------*/
template <typename CharType>
/* static */ void*
TextWriterBase<CharType>::SetBuffer(std::size_t size)
{
    NW4R_MIN_ASSERT(size, 1);
    void* oldBuffer = mFormatBuffer;

    mFormatBuffer       = NULL;
    mFormatBufferSize   = size;

    return oldBuffer;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::GetBufferSize()

  Description:  書式文字列の展開に使用するバッファのサイズを取得します。

  Arguments:    なし。

  Returns:      書式文字列の展開に使用するバッファのサイズ(sizeof(CharType)単位)。
 *---------------------------------------------------------------------------*/
template <typename CharType>
/* static */ std::size_t
TextWriterBase<CharType>::GetBufferSize()
{
    return mFormatBufferSize;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::GetBuffer()

  Description:  書式文字列の展開に使用するバッファへのポインタを取得します。

  Arguments:    なし。

  Returns:      バッファが存在する場合にはバッファへのポインタ。
                そうでなければ NULL。
 *---------------------------------------------------------------------------*/
template <typename CharType>
/* static */ const void*
TextWriterBase<CharType>::GetBuffer()
{
    return mFormatBuffer;
}






/* =======================================================================
        private
   ======================================================================== */

/* ------------------------------------------------------------------------
        文字列処理
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcLineWidth( StreamType, int )

  Description:  1行の長さを計算します。

  Arguments:    str:        文字列
                length:     文字列長(sizeof(CharType)単位)

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::CalcLineWidth(
    StreamType              str,
    int                     length
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( str );
    NW4R_MIN_ASSERT( length, 0 );

    Rect rect;
    TextWriterBase<CharType> myCopy = *this;

    myCopy.SetCursor(0, 0);
    (void)myCopy.CalcLineRectImpl(&rect, &str, length);

    return rect.GetWidth();
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcLineRectImpl( StreamType, int )

  Description:  1行分の描画矩形を計算します。

  Arguments:    pRect:      描画矩形を格納するバッファへのポインタ。
                str:        文字列
                length:     文字列長(sizeof(CharType)単位)

  Returns:      読み進んだ文字列長(sizeof(CharType)単位)
 *---------------------------------------------------------------------------*/
template <typename CharType>
int
TextWriterBase<CharType>::CalcLineRectImpl(
    Rect*                   pRect,
    StreamType*             pStr,
    int                     length
)
{
    NW4R_POINTER_ASSERT(pRect);
    NW4R_POINTER_ASSERT(pStr);
    NW4R_POINTER_ASSERT(*pStr);
    NW4R_MIN_ASSERT(length, 0);

    const StreamType str            = *pStr;
    const StreamType end            = str + length;
    const bool bUseLimit            = m_WidthLimit < FLT_MAX;
    PrintContext<CharType> context = PrintContext<CharType>(this, str, 0, 0, 0, 0);
    f32 limitLeft                   = 0;
    f32 limitRight                  = 0;
    bool bCharSpace                 = false;
    bool bOverLimit                 = false;
    StreamType prevStreamPos        = NULL;
    Rect prevRect;

    /*
        文字列の途中でエンコーディングが変わる事は想定していない
    */
    NW4R_POINTER_ASSERT(GetFont());
    CharStrmReader reader = GetFont()->GetCharStrmReader();

    float fontHeight = context.writer->GetFontHeight();
    pRect->left     = 0;
    pRect->right    = 0;
    pRect->top      = Min(0.0f, fontHeight);
    pRect->bottom   = Max(0.0f, fontHeight);
    prevRect = *pRect;

    reader.Set(str);
    prevStreamPos = NULL;

    for (
        CharCode code = reader.Next();
        reinterpret_cast<StreamType>(reader.GetCurrentPos()) <= end;
    )
    {
        if (code < ' ')
        {
            //---- 制御文字(タグの開始)

            typename TagProcessor::Operation operation;

            if (code == '\n')
            {
                limitRight += static_cast<float>(GetFont()->GetKerning(context.prevCode, 0)) * GetScaleH();
                context.prevCode = 0;
            }

            Rect rect(limitRight, 0, 0, 0);

            context.str = reinterpret_cast<StreamType>(reader.GetCurrentPos());
            context.flags = 0;
            context.flags |= bCharSpace ? 0: CONTEXT_NO_CHAR_SPACE;
            SetCursorX(limitRight);

            //---- 折り返しの判定
            if ( bUseLimit
              && code != '\n'
              && prevStreamPos != NULL
            )
            {
                PrintContext<CharType> context2 = context;
                TextWriterBase<CharType> myCopy = *this;
                Rect rect2;

                context2.writer = &myCopy;
                operation = mTagProcessor->CalcRect(&rect2, code, &context2);

                if ( rect2.GetWidth() > 0.0f
                  && (myCopy.GetCursorX() - context.xOrigin > m_WidthLimit)
                )
                {
                    bOverLimit = true;
                    code       = '\n';
                    reader.Set(prevStreamPos);
                    continue;
                }
            }

            //---- タグ処理
            operation = mTagProcessor->CalcRect(&rect, code, &context);

            NW4R_POINTER_ASSERT(context.str);
            reader.Set(context.str);

            pRect->left     = Min(pRect->left,      rect.left);
            pRect->top      = Min(pRect->top,       rect.top);
            pRect->right    = Max(pRect->right,     rect.right);
            pRect->bottom   = Max(pRect->bottom,    rect.bottom);
            limitRight = GetCursorX();

            if (operation == TagProcessor::OPERATION_END_DRAW)
            {
                //---- 全部読み進んだ事にする
                *pStr += length;
                return false;
            }
            else if (operation == TagProcessor::OPERATION_NO_CHAR_SPACE)
            {
                bCharSpace = false;
            }
            else if (operation == TagProcessor::OPERATION_CHAR_SPACE)
            {
                bCharSpace = true;
            }
            else if (operation == TagProcessor::OPERATION_NEXT_LINE)
            {
                break;
            }
        }
        else if (IsPrintableChar(code))
        {
            //---- 通常の文字

            f32 crntRight = limitRight;

            if (bCharSpace)
            {
                crntRight += GetCharSpace();
            }

            if (IsWidthFixed())
            {
                crntRight += GetFixedWidth();
            }
            else
            {
                crntRight += GetFont()->GetCharWidth(code) * GetScaleH();
            }
            crntRight += static_cast<float>(GetFont()->GetKerning(context.prevCode, code) * GetScaleH());

            float kerningLast = static_cast<float>(GetFont()->GetKerning(code, 0) * GetScaleH());
            crntRight += kerningLast;

            //---- 折り返しの判定
            if (bUseLimit && prevStreamPos != NULL)
            {
                f32 width = crntRight - limitLeft;
                if (width > m_WidthLimit)
                {
                    bOverLimit = true;
                    code       = '\n';
                    reader.Set(prevStreamPos);
                    continue;
                }
            }

            limitRight = crntRight;
            pRect->left  = Min(pRect->left,  limitRight);
            pRect->right = Max(pRect->right, limitRight);
            limitRight -= kerningLast;
            bCharSpace = true;
            context.prevCode = code;
        }

        if (bUseLimit)
        {
            prevStreamPos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
        }

        code = reader.Next();
    }

    // 末尾の文字のカーニングの適用
    {
        limitRight += static_cast<float>(GetFont()->GetKerning(context.prevCode, 0)) * GetScaleH();
        pRect->left = Min(pRect->left, limitRight);
        pRect->right = Max(pRect->right, limitRight);
    }

    *pStr = reinterpret_cast<StreamType>(reader.GetCurrentPos());

    return bOverLimit;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::CalcStringRect( Rect*, StreamType, int, PrintContext* )

  Description:  文字列の描画矩形を計算します。

  Arguments:    pRect:      描画矩形を格納するバッファへのポインタ。
                str:        文字列
                length:     文字列長(sizeof(CharType)単位)
                context:    文字列描画コンテキスト

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
void
TextWriterBase<CharType>::CalcStringRectImpl(
    Rect*                   pRect,
    StreamType              str,
    int                     length
)
{
    NW4R_POINTER_ASSERT( this );
    NW4R_POINTER_ASSERT( pRect );
    NW4R_POINTER_ASSERT( str );
    NW4R_MIN_ASSERT( length, 0 );

    const StreamType end = str + length;
    int remain = length;

    StreamType pos = str;

    pRect->left     = 0;
    pRect->right    = 0;
    pRect->top      = 0;
    pRect->bottom   = 0;

    SetCursor(0, 0);

    do
    {
        Rect rect;
        CalcLineRectImpl(&rect, &pos, remain);

        // 理論上は、64-bit ビルドで問題が発生する可能性があるが、事実上問題がないと判断してキャストしています。
        remain   = static_cast<int>(end - pos);

        pRect->left     = Min(pRect->left,      rect.left);
        pRect->top      = Min(pRect->top,       rect.top);
        pRect->right    = Max(pRect->right,     rect.right);
        pRect->bottom   = Max(pRect->bottom,    rect.bottom);
    } while( remain > 0 );
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::PrintImpl( StreamType, int )

  Description:  文字列を描画します。

  Arguments:    str:        文字列
                length:     文字列長(sizeof(CharType)単位)

  Returns:      なし。
 *---------------------------------------------------------------------------*/
template <typename CharType>
float
TextWriterBase<CharType>::PrintImpl(
    StreamType              str,
    int                     length,
    int                     lineOffsetCount,
    const float*            pLineOffset
)
{
    NW4R_POINTER_ASSERT(str);
    NW4R_POINTER_ASSERT(GetFont());
    NW4R_MIN_ASSERT(length, 0);

    f32 xOrigin    = GetCursorX();
    f32 yOrigin    = GetCursorY();
    f32 limitLeft  = 0;
    f32 limitRight = 0;
    const bool bUseLimit = m_WidthLimit < FLT_MAX;
    const f32 orgCursorY = yOrigin;
    bool bCharSpace  = false;
    StreamType prevStreamPos = str;
    StreamType prevNewLinePos = str;
    int line = 0;

    f32 textWidth  = AdjustCursor(&xOrigin, &yOrigin, str, length);
    f32 yCursorAdj = orgCursorY - GetCursorY();

    PrintContext<CharType> context = PrintContext<CharType>(this, str, xOrigin, yOrigin, 0, 0);
    if (lineOffsetCount != 0)
    {
        float lineOffset = GetLineOffset(lineOffsetCount, pLineOffset, line);
        SetCursorX( lineOffset );
    }
    CharStrmReader reader = GetFont()->GetCharStrmReader();
    reader.Set(str);

    for (
        CharCode code = reader.Next();
        reinterpret_cast<StreamType>(reader.GetCurrentPos()) - str <= length;
    )
    {
        if (code < ' ')
        {
            //---- 制御文字(タグの開始)

            typename TagProcessor::Operation operation;
            context.str = reinterpret_cast<StreamType>(reader.GetCurrentPos());
            context.flags = 0;
            context.flags |= bCharSpace ? 0: CONTEXT_NO_CHAR_SPACE;

            //---- 折り返しの判定
            if ( bUseLimit
              && code != '\n'
              && prevStreamPos != prevNewLinePos
            )
            {
                PrintContext<CharType> context2 = context;
                TextWriterBase<CharType> myCopy = *this;
                Rect rect;

                context2.writer = &myCopy;
                operation = mTagProcessor->CalcRect(&rect, code, &context2);

                if ( rect.GetWidth() > 0.0f
                  && myCopy.GetCursorX() - context.xOrigin > m_WidthLimit
                )
                {
                    code = '\n';
                    context.prevCode = 0;
                    reader.Set(prevStreamPos);
                    continue;
                }
            }

            //---- タグ処理
            operation = mTagProcessor->Process(code, &context);
            limitRight = GetCursorX() - xOrigin;

            if (code == '\n')
            {
                context.prevCode = 0;
            }

            if (operation == TagProcessor::OPERATION_NEXT_LINE)
            {
                NW4R_POINTER_ASSERT(context.str);

                line++;
                float lineOffset = GetLineOffset(lineOffsetCount, pLineOffset, line);

                //---- 次行描画開始位置Xの補正
                if (IsDrawFlagSet(HORIZONTAL_ALIGN_MASK, HORIZONTAL_ALIGN_CENTER))
                {
                    // 理論上は、64-bit ビルドで問題が発生する可能性があるが、事実上問題がないと判断してキャストしています。
                    const int   remain  = static_cast<int>(length - (context.str - str));

                    const f32   width   = CalcLineWidth( context.str, remain );
                    const f32   offset  = AdjustCenterValue(textWidth) - AdjustCenterValue(width);
                    SetCursorX( context.xOrigin + offset + lineOffset );
                }
                else if (IsDrawFlagSet(HORIZONTAL_ALIGN_MASK, HORIZONTAL_ALIGN_RIGHT))
                {
                    // 理論上は、64-bit ビルドで問題が発生する可能性があるが、事実上問題がないと判断してキャストしています。
                    const int   remain  = static_cast<int>(length - (context.str - str));

                    const f32   width   = CalcLineWidth( context.str, remain );
                    const f32   offset  = textWidth - width;
                    SetCursorX( context.xOrigin + offset + lineOffset );
                }
                else
                {
                    //---- 最大幅の更新
                    const f32 width = GetCursorX() - context.xOrigin;
                    textWidth = Max(textWidth, width);

                    SetCursorX( context.xOrigin + lineOffset );
                }

                if (bUseLimit)
                {
                    prevNewLinePos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
                }
                bCharSpace = false;
            }
            else if (operation == TagProcessor::OPERATION_NO_CHAR_SPACE)
            {
                bCharSpace = false;
            }
            else if (operation == TagProcessor::OPERATION_CHAR_SPACE)
            {
                bCharSpace = true;
            }
            else if (operation == TagProcessor::OPERATION_END_DRAW)
            {
                break;
            }

            NW4R_POINTER_ASSERT(context.str);
            reader.Set(context.str);
        }
        else if (IsPrintableChar(code))
        {
            //---- 通常の文字

            const f32 baseY = GetCursorY();

            f32 crntRight = limitRight;
            f32 kerning = static_cast<float>(GetFont()->GetKerning(context.prevCode, code)) * GetScaleH();
            Glyph glyph;
            GetFont()->GetGlyph(&glyph, code);

            if (bCharSpace)
            {
                crntRight += GetCharSpace();
            }

            if (IsWidthFixed())
            {
                crntRight += GetFixedWidth();
            }
            else
            {
                crntRight += glyph.widths.charWidth * GetScaleH();
            }
            crntRight += kerning;

            f32 kerningLast = static_cast<float>(GetFont()->GetKerning(code, 0)) * GetScaleH();
            crntRight += kerningLast;

            //---- 折り返しの判定
            if (bUseLimit && prevStreamPos != prevNewLinePos)
            {
                f32 width = crntRight - limitLeft;
                if (width > m_WidthLimit)
                {
                    code = '\n';
                    context.prevCode = 0;
                    reader.Set(prevStreamPos);
                    continue;
                }
            }

            limitRight = crntRight;
            limitRight -= kerningLast;

            if (bCharSpace)
            {
                MoveCursorX( GetCharSpace() );
            }
            bCharSpace = true;

            MoveCursorX( kerning );

            //---- カーソルはベースラインにあるのでセル上端へ移動する
            {
                const Font* pFont = GetFont();
                const f32 adj = -(pFont->GetBaselinePos() + glyph.baselineDifference) * GetScaleV();
                MoveCursorY( adj );
            }

            //PrintGlyph(glyph);
            (void)CharWriter::Print(code);

            // 戻す
            SetCursorY( baseY );

            context.prevCode = code;
        }

        if (bUseLimit)
        {
            prevStreamPos = reinterpret_cast<StreamType>(reader.GetCurrentPos());
        }

        code = reader.Next();
    }

    // 末尾の文字のカーニングの適用
    {
        float kerning = static_cast<float>(GetFont()->GetKerning(context.prevCode, 0)) * GetScaleH();
        MoveCursorX( kerning );
    }

    //---- 最大幅の更新
    {
        const f32 width = GetCursorX() - context.xOrigin;
        textWidth = Max(textWidth, width);
    }

    //---- カーソル位置Yを描画フラグに従って移動
    // AdjustCursorでベースライン上に移動したカーソル位置を元に戻す
    if ( IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_MIDDLE)
      || IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_BOTTOM)
    )
    {
        SetCursorY(orgCursorY);
    }
    else
    {
        MoveCursorY(yCursorAdj);
    }

    return textWidth;
}

/*---------------------------------------------------------------------------*
  Name:         TextWriterBase::AdjustCursor( StreamType, int )

  Description:  文字列を描画フラグに従って描画するときの1文字目の
                描画開始位置にカーソルを移動します。

  Arguments:    xOrigin:    描画基準座標を格納したバッファへのポインタ。
                            補正後の描画基準座標が格納されて返る。
                yOrigin:    描画基準座標を格納したバッファへのポインタ。
                            補正後の描画基準座標が格納されて返る。
                str:        文字列
                length:     文字列長(sizeof(CharType)単位)

  Returns:      文字列の幅を返します。
 *---------------------------------------------------------------------------*/
template <typename CharType>
f32
TextWriterBase<CharType>::AdjustCursor(
    f32*                    xOrigin,
    f32*                    yOrigin,
    StreamType              str,
    int                     length
)
{
    float textWidth  = 0;
    float textHeight = 0;

    if(    ! IsDrawFlagSet( (HORIZONTAL_ALIGN_MASK | HORIZONTAL_ORIGIN_MASK | VERTICAL_ORIGIN_MASK),
                            (HORIZONTAL_ALIGN_LEFT | HORIZONTAL_ORIGIN_LEFT | VERTICAL_ORIGIN_BASELINE) )
        && ! IsDrawFlagSet( (HORIZONTAL_ALIGN_MASK | HORIZONTAL_ORIGIN_MASK | VERTICAL_ORIGIN_MASK),
                            (HORIZONTAL_ALIGN_LEFT | HORIZONTAL_ORIGIN_LEFT | VERTICAL_ORIGIN_TOP) ) )
    {
        Rect textRect;
        CalcStringRect(&textRect, str, length);
        textWidth  = textRect.left + textRect.right;
        textHeight = textRect.top  + textRect.bottom;
    }

    //---- 描画開始位置Xの補正
    // 各々の位置から 1 行目のアセンダラインに統一
    if( IsDrawFlagSet(HORIZONTAL_ORIGIN_MASK, HORIZONTAL_ORIGIN_CENTER) )
    {
        *xOrigin -= textWidth / 2;
    }
    else if( IsDrawFlagSet(HORIZONTAL_ORIGIN_MASK, HORIZONTAL_ORIGIN_RIGHT) )
    {
        *xOrigin -= textWidth;
    }

    //---- 描画開始位置Yの補正
    // 各々の位置から文字列の左端に統一
    if( IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_MIDDLE) )
    {
        *yOrigin -= textHeight / 2;
    }
    else if( IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_BOTTOM) )
    {
        *yOrigin -= textHeight;
    }

    //---- 1行目描画開始位置Xの補正
    // 文字列の左端からそれぞれの寄せ位置へカーソルを移動
    if( IsDrawFlagSet(HORIZONTAL_ALIGN_MASK, HORIZONTAL_ALIGN_CENTER) )
    {
        const float width = CalcLineWidth(str, length);
        const float offset = (textWidth - width) / 2;
        SetCursorX( *xOrigin + offset );
    }
    else if( IsDrawFlagSet(HORIZONTAL_ALIGN_MASK, HORIZONTAL_ALIGN_RIGHT) )
    {
        const float width = CalcLineWidth(str, length);
        const float offset = textWidth - width;
        SetCursorX( *xOrigin + offset );
    }
    else
    {
        SetCursorX( *xOrigin );
    }

    //---- 1行目描画開始位置Yの補正
    // 1 行目のベースライン位置にカーソルを移動
    if( IsDrawFlagSet(VERTICAL_ORIGIN_MASK, VERTICAL_ORIGIN_BASELINE) )
    {
        // yOrigin はベースラインであるのでそのままセット
        SetCursorY( *yOrigin );
    }
    else
    {
        // yOrigin はアセンダラインであるのでベースラインにカーソルを移動
        SetCursorY( *yOrigin + GetFontAscent() );
    }


    return textWidth;
}

/*---------------------------------------------------------------------------*/
template <typename CharType>
float TextWriterBase<CharType>::GetLineOffset(int lineOffsetCount, const float* pLineOffset, int line) const
{
    if (line >= lineOffsetCount)
    {
        return 0.0f;
    }
    return pLineOffset[line];
}

/* =======================================================================
        明示的実体化
   ======================================================================== */

// [CW3a2] template 関数内でauto変数としてクラスのインスタンスを作成すると
// Warning: result of function call is not used
#if defined(NW4R_CW3)
 #pragma warn_resultnotused off
#endif

template class TextWriterBase<char>;
template class TextWriterBase<wchar_t>;



} /* UnManaged */
} /* namespace ut */
} /* namespace nw4r */


// [CW3a2] namespace の閉じカッコの前で reset すると
// warn_resultnotused off が無効化される
#if defined(NW4R_CW3)
 #pragma warn_resultnotused reset
#endif


